/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#if (GS32F00xx == 0x3000)

#if defined(FDP_RT_FW)

#include "fdp_rt/fdp_rt.h"
#include "fdp_rt_defconfig.h"
#include "fdp_rt/fdp_rt_errno.h"
#include "fdp_rt/fdp_rt_uart_v3_0.h"

#include "boot_check_algorithm.h"
#include "boot_flash.h"
#include "boot_riscv_cache.h"

BOOT_FDP_RT_SEC static void fdp_rt_reset(fdp_rt_ops_t *ops)
{
	ops->rx_get_index = 0;
	ops->rx_put_index = 0;
	ops->cache_rxbuf_index = 0;
}

int fdp_rt_init(fdp_rt_ops_t *ops)
{
	boot_flash_addr_info_t flash_info;

	if (ops == NULL)
		return FDP_RT_FAIL;

	if (ops->rxbuf_size <= 64U)
		return FDP_RT_FAIL;

	fdp_rt_reset(ops);
	ops->phase = FDP_RT_CMD_STAGE;
	ops->cache_rxbuf_index = 0;

	boot_flash_get_addr_info(0x8000000, &flash_info);

	ops->flash_mode = flash_info.bank_mode;

	boot_flash_disable_register_write_protect(BOOT_FLASH_EFC0_ADDR);
	boot_flash_disable_register_write_protect(BOOT_FLASH_EFC1_ADDR);

	boot_flash_disable_fast_program_mode(BOOT_FLASH_EFC0_ADDR);
	boot_flash_disable_fast_program_mode(BOOT_FLASH_EFC1_ADDR);

	switch (ops->dev_id) {
		case FDP_RT_DEV_UART_V3 :
			fdp_rt_uart_init((fdp_rt_uart_dev_priv_t *)ops->dev_private_data,
							((fdp_rt_clk_priv_t *)ops->clk_private_data)->apb_freq);
			break;

		default :
			return FDP_RT_FAIL;
	}

	return FDP_RT_SUCCESS;
}

int fdp_rt_deinit(fdp_rt_ops_t *ops)
{
	if (ops == NULL)
		return FDP_RT_FAIL;

	switch (ops->dev_id) {
		case FDP_RT_DEV_UART_V3 :
			fdp_rt_uart_deinit((fdp_rt_uart_dev_priv_t *)ops->dev_private_data);
			break;

		default :
			return FDP_RT_FAIL;
	}

	return FDP_RT_SUCCESS;
}

int fdp_rt_read(fdp_rt_ops_t *ops)
{
	switch (ops->dev_id) {
		case FDP_RT_DEV_UART_V3 :
			return fdp_rt_uart_read((fdp_rt_uart_dev_priv_t *)ops->dev_private_data, ops->rxbuf,
									&ops->rx_put_index, &ops->rx_get_index, ops->rxbuf_size);

		default :
			return FDP_RT_FAIL;
	}

	return FDP_RT_SUCCESS;
}

int fdp_rt_write(fdp_rt_ops_t *ops)
{
	switch (ops->dev_id) {
		case FDP_RT_DEV_UART_V3 :
			return fdp_rt_uart_write((fdp_rt_uart_dev_priv_t *)ops->dev_private_data,
									ops->txbuf, ops->txbuf_size);
		default :
			return FDP_RT_FAIL;
	}
}

BOOT_FDP_RT_SEC static void fdp_rt_dev_detect(fdp_rt_ops_t *ops)
{
	ops->txbuf[0] = FDP_RT_ACK_DETECT;
	ops->txbuf[1] = ops->flash_mode;
	ops->txbuf_size = 2;
}

BOOT_FDP_RT_SEC static int fdp_rt_erase_addr(uintptr_t addr)
{
	boot_flash_addr_info_t addr_info;
	boot_flash_efc_status_t efc_info0;
	boot_flash_efc_status_t efc_info1;

	if (addr >= FDP_RT_CONFIG_FLASH_ADDR_MAX)
		return 	FDP_RT_FAIL;

	boot_flash_get_addr_info(addr, &addr_info);

	/* check if the flash controller is busy. */
	if (addr_info.bank_mode == 0) {
		boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &efc_info0);
		boot_flash_get_efc_status(BOOT_FLASH_EFC1_ADDR, &efc_info1);

		if (efc_info0.efc_busy || efc_info1.efc_busy)
			return FDP_RT_FLASH_BUSY;
	} else {
		boot_flash_get_efc_status(addr_info.flash_efc_addr, &efc_info0);

		if (efc_info0.efc_busy)
			return FDP_RT_FLASH_BUSY;
	}

	/* clear flash controller status. */
	boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);
	boot_flash_clear_efc_status(BOOT_FLASH_EFC1_ADDR, BOOT_FLASH_ALL_STATUS_MASK);

	/* erase flash sector base on address. */
	if (boot_flash_erase_main_flash_sector(addr) != BOOT_FLASH_STAT_SUCCESS)
		return FDP_RT_ERASE_ERR;

	return FDP_RT_SUCCESS;
}

BOOT_FDP_RT_SEC static int fdp_rt_write_addr_cmd(fdp_rt_ops_t *ops, uintptr_t addr)
{
	boot_flash_addr_info_t addr_info;
	boot_flash_efc_status_t efc_info;

	if (addr % FDP_RT_ADDR_ALINE || addr >= FDP_RT_CONFIG_FLASH_ADDR_MAX)
		return FDP_RT_FAIL;

	boot_flash_get_addr_info(addr, &addr_info);
	boot_flash_get_efc_status(addr_info.flash_efc_addr, &efc_info);

	if (efc_info.efc_busy)
		return FDP_RT_FLASH_BUSY;

	if (addr_info.bank_mode == 0)
		boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &efc_info);
	else
		boot_flash_clear_efc_status(addr_info.flash_efc_addr, BOOT_FLASH_ALL_STATUS_MASK);

	boot_machine_mode_invalid_DCache_line(addr);

	*(volatile uintptr_t *)addr;

	if (addr_info.bank_mode == 0)
		boot_flash_get_efc_status(BOOT_FLASH_EFC0_ADDR, &efc_info);
	else
		boot_flash_get_efc_status(addr_info.flash_efc_addr, &efc_info);

	if (!efc_info.sec_detected)
		return FDP_RT_FAIL;

	if (addr_info.bank_mode == 0)
		boot_flash_clear_efc_status(BOOT_FLASH_EFC0_ADDR, BOOT_FLASH_ALL_STATUS_MASK);
	else
		boot_flash_clear_efc_status(addr_info.flash_efc_addr, BOOT_FLASH_ALL_STATUS_MASK);

	ops->dest_address = addr;

	return FDP_RT_SUCCESS;
}

BOOT_FDP_RT_SEC static int fdp_rt_check_cmd(fdp_rt_ops_t *ops, uintptr_t addr)
{
	uint16_t crc16_val;

	boot_machine_mode_invalid_DCache_line(addr);

	crc16_val = boot_crc16_modbus((uint8_t *)addr, FDP_RT_DATA_FRAME_SIZE);

	ops->txbuf[0] = (crc16_val >> 8) & 0xff;
	ops->txbuf[1] = crc16_val & 0xff;
	ops->txbuf_size = 2;

	fdp_rt_write(ops);

	return FDP_RT_SUCCESS;
}

BOOT_FDP_RT_SEC static int fdp_rt_cmd_process(fdp_rt_ops_t *ops)
{
	uint8_t checksum;
	uint32_t address;

	checksum = boot_algorithm_checksum8(ops->cache_rxbuf, ops->cache_rxbuf_index - 1);

	if (checksum != ops->cache_rxbuf[ops->cache_rxbuf_index - 1])
		goto fdp_rt_err_dispose;

	if (ops->cache_rxbuf[1] != FDP_RT_HOST)
		goto fdp_rt_err_dispose;

	switch (ops->cache_rxbuf[0]) {
		case FDP_RT_CMD_DETECT_DEV :
			fdp_rt_dev_detect(ops);

			return fdp_rt_write(ops);

		case FDP_RT_CMD_ERASE :
			address = ops->cache_rxbuf[2];
			address |= ops->cache_rxbuf[3] << 8;
			address |= ops->cache_rxbuf[4] << 16;
			address |= ops->cache_rxbuf[5] << 24;

			if (fdp_rt_erase_addr(address) == FDP_RT_SUCCESS)
				goto fdp_rt_success;
			else
				goto fdp_rt_err_dispose;

		case FDP_RT_CMD_WRITE :
			address = ops->cache_rxbuf[2];
			address |= ops->cache_rxbuf[3] << 8;
			address |= ops->cache_rxbuf[4] << 16;
			address |= ops->cache_rxbuf[5] << 24;

			if (fdp_rt_write_addr_cmd(ops, address) != FDP_RT_SUCCESS)
				goto fdp_rt_err_dispose;
			else {
				ops->phase = FDP_RT_DATA_STAGE;
				goto fdp_rt_success;
			}

		case FDP_RT_CMD_CHECK :
			address = ops->cache_rxbuf[2];
			address |= ops->cache_rxbuf[3] << 8;
			address |= ops->cache_rxbuf[4] << 16;
			address |= ops->cache_rxbuf[5] << 24;

			return fdp_rt_check_cmd(ops, address);

		default :
			goto fdp_rt_err_dispose;
	}

fdp_rt_success :

	ops->txbuf[0] = (FDP_RT_ACK_OK >> 8) & 0xff;
	ops->txbuf[1] = FDP_RT_ACK_OK & 0xff;
	ops->txbuf_size = 2;

	fdp_rt_write(ops);

	return FDP_RT_SUCCESS;

fdp_rt_err_dispose :

	ops->txbuf[0] = (FDP_RT_ACK_ERROR >> 8) & 0xff;
	ops->txbuf[1] = FDP_RT_ACK_ERROR & 0xff;
	ops->txbuf_size = 2;

	fdp_rt_write(ops);

	ops->phase = FDP_RT_CMD_STAGE;

	return FDP_RT_FAIL;
}

BOOT_FDP_RT_SEC static int fdp_rt_write_addr(uintptr_t addr, uint8_t *buf)
{
	if (boot_flash_write_main_flash(addr, buf) != BOOT_FLASH_STAT_SUCCESS)
		return FDP_RT_FAIL;

	return FDP_RT_SUCCESS;
}

BOOT_FDP_RT_SEC static int fdp_rt_data_process(fdp_rt_ops_t *ops)
{
	if (ops->cache_rxbuf_index == FDP_RT_FRAME_SIZE) {
		ops->txbuf[0] = (FDP_RT_ACK_CONTINUE >> 8) & 0xff;
		ops->txbuf[1] = FDP_RT_ACK_CONTINUE & 0xff;
		ops->txbuf_size = 2;

		fdp_rt_write(ops);

		return FDP_RT_CONTINUE;
	}

	if (ops->cache_rxbuf_index < FDP_RT_DATA_FRAME_SIZE)
		return FDP_RT_CONTINUE;

	ops->phase = FDP_RT_CMD_STAGE;

	if (fdp_rt_write_addr(ops->dest_address,
			ops->cache_rxbuf) != FDP_RT_SUCCESS)
		return FDP_RT_FAIL;

	ops->txbuf[0] = (FDP_RT_ACK_OK >> 8) & 0xff;
	ops->txbuf[1] = FDP_RT_ACK_OK & 0xff;
	ops->txbuf_size = 2;

	fdp_rt_write(ops);

	return FDP_RT_SUCCESS;

fdp_rt_data_error :

	ops->txbuf[0] = (FDP_RT_ACK_ERROR >> 8) & 0xff;
	ops->txbuf[1] = FDP_RT_ACK_ERROR & 0xff;
	ops->txbuf_size = 2;

	fdp_rt_write(ops);

	ops->phase = FDP_RT_CMD_STAGE;

	return FDP_RT_FAIL;
}

int fdp_rt_schedule(fdp_rt_ops_t *ops)
{
	uint32_t len;
	uint32_t index;
	int ret = FDP_RT_SUCCESS;

	if (ops == NULL)
		return FDP_RT_FAIL;

#if (FDP_RT_CONFIG_INTERRUPT_ENABLE == 0U)

	ret = fdp_rt_read(ops);
	if (ret != FDP_RT_SUCCESS)
		goto fdp_rt_reset_init;

#endif

	if (ops->rx_put_index > ops->rx_get_index) {
		len = ops->rx_put_index - ops->rx_get_index;
		for (index = 0; index < len; index++) {
			ops->cache_rxbuf[ops->cache_rxbuf_index++] =
					ops->rxbuf[ops->rx_get_index++];
		}
	} else if (ops->rx_put_index < ops->rx_get_index) {
		/* step 1 : Copy the data from the current index to the end interval to the buffer */
		len = ops->rxbuf_size - ops->rx_get_index;
		for (index = 0; index < len; index++) {
			ops->cache_rxbuf[ops->cache_rxbuf_index++] =
					ops->rxbuf[ops->rx_get_index++];
		}

		/* step 2 : copy the data 0 indexed to the put index of rx cache buffer */
		ops->rx_get_index = 0;
		for (index = 0; index < ops->rx_put_index; index++) {
			ops->cache_rxbuf[ops->cache_rxbuf_index++] =
			ops->rxbuf[ops->rx_get_index++];
		}
	} else
		return FDP_RT_CONTINUE;

	if (ops->cache_rxbuf_index < FDP_RT_FRAME_SIZE)
		return FDP_RT_CONTINUE;

	if (ops->phase == FDP_RT_CMD_STAGE) {
		ret = fdp_rt_cmd_process(ops);

		if (ret != FDP_RT_SUCCESS)
			goto fdp_rt_reset_init;
	} else if (ops->phase == FDP_RT_DATA_STAGE) {
		ret = fdp_rt_data_process(ops);

		if (ret == FDP_RT_CONTINUE)
			return FDP_RT_CONTINUE;
	}

fdp_rt_reset_init :

	fdp_rt_reset(ops);

	return ret;
}

#endif

#endif
