/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdio.h>
#include <string.h>

#include "fdp_err.h"
#include "fdp_crc.h"
#include "fdp_defconfig.h"
#include "fdp_bswap.h"
#include "fdp_mem.h"
#include "fdp_storage_port.h"
#include "fdp_cache.h"

int fdp_ack_to_host(fdp_device_type_t type, bool ack)
{
	uint8_t buf[4];

	if (ack) {
		buf[0] = FDP_CMD_ACK_SUCCESS;
		buf[1] = FDP_DEVICE;
		buf[2] = 0x67;
		buf[3] = 0x8F;
		return fdp_transmit(buf, FDP_CMD_ACK_LENGTH, type);
	} else {
		buf[0] = FDP_CMD_ACK_ERROR;
		buf[1] = FDP_DEVICE;
		buf[2] = 0xF7;
		buf[3] = 0x8E;
		return fdp_transmit(buf, FDP_CMD_ACK_LENGTH, type);
	}
}

void fdp_reset_fdp(void)
{
	fdp_record_stat_t *fdp_status = fdp_get_current_stat();

	fdp_status->program_stage = FDP_PRG_CMD_PHASE;
	fdp_status->program_flow = FDP_BOOT_FLOW_WAIT_FOR_UPTATE;
	fdp_status->received_index = 0;
	fdp_status->rx_read = 0;
	fdp_status->rx_write = 0;
	fdp_status->frame_size = 0;
	fdp_status->max_frame_size = 0;
	fdp_status->received_data_size = 0;
	fdp_status->crc_val = 0;
	fdp_status->release_core = 0;
	fdp_status->watchdog_timeout = 0;
	fdp_status->mem_init_ready = false;
	fdp_status->transfer_done = false;
	fdp_status->download_ready = false;
}

/**
 * @brief Receive upper command.
 * 
 * @param buf Command data buffer.
 * @param len Received the length of Command data.
 * @return int If the return value is less than 0, it is error, else success.
 */
static int fdp_program_command_stage(uint8_t *buf, uint32_t len)
{
	uint8_t cmd = buf[0];
	uint8_t sendbuf[FDP_CMD_MAX_LENGTH] = { 0 };
	uint32_t crc_val;
	uint32_t vender_number;

	fdp_record_stat_t *fdp_status = fdp_get_current_stat();

	if (fdp_status->program_stage != FDP_PRG_CMD_PHASE)
		return -FDP_CHECKERR;

	/* Receive commands based on their length. */
	switch(cmd) {
		/* Reques start transfer command. */
		case FDP_CMD_START :
			if (len < FDP_CMD_START_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Check if the vender ID is correct. */
		case FDP_CMD_VENDER_CHECK :
			if (len < FDP_CMD_VENDER_CHECK_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Send the length of image to the upper layer. */
		case FDP_CMD_FRAME_SIZE :
			if (len < FDP_CMD_FRAME_SIZE_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Get the size of image. */
		case FDP_CMD_FILE_SIZE :
			if (len < FDP_CMD_FILE_SIZE_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Get image storage address. */
		case FDP_CMD_GET_ADDRESS :
			if (len < FDP_CMD_STORAGE_ADDR_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Initialize memory device. */
		case FDP_CMD_INIT_MEM :
			if (len < FDP_CMD_INIT_MEMORY_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Start transfer image command. */
		case FDP_CMD_TRANSFER_REQ :
			if (len < FDP_CMD_TRANSFER_REQ_LENGTH)
				return FDP_CONTINUE;
			break;

		/* Transfer done command. */
		case FDP_CMD_TRANSFER_END :
			if (len < FDP_CMD_TRANSFER_END_LENGTH)
				return FDP_CONTINUE;
			break;

		case FDP_CMD_CHECK_IMAGE :
			if (len < FDP_CMD_CHECK_IMAGE_LENGTH)
				return FDP_CONTINUE;
			break;

		case FDP_CMD_RELEASE_CORE :
			if (len < FDP_CMD_RELEASE_CORE_LENGTH)
				return FDP_CONTINUE;
			break;

		default :
			return -FDP_ERRRECV;
	}

	if (len > FDP_CMD_MAX_OF_LENGTH)
		return -FDP_ERRRECV;

	/* Check if the CRC is correct. */
	crc_val = fdp_crc16_modbus(buf, len - 2);

	if (((crc_val & 0xff00) >> 8) != buf[len - 2] &&
		(crc_val & 0x00ff) != buf[len - 1])
		return -FDP_ERRRECV;

	/* If we receive a command or data,
	we will process it and send an ACK after passing the check. */
	switch (cmd) {
		case FDP_CMD_START :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_WAIT_FOR_UPTATE)
				return -FDP_ERRRECV;
			fdp_reset_fdp();
			/* Start executing the update program. */
			fdp_status->program_flow = FDP_BOOT_FLOW_UPDATE;
			break;

		case FDP_CMD_VENDER_CHECK :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_UPDATE)
				return -FDP_ERRRECV;

			memcpy((void *)&vender_number, (void *)&buf[2], 4);

			if (vender_number != FDP_VENDER_NUMBER)
				return -FDP_ERRRECV;

			fdp_status->program_flow = FDP_BOOT_FLOW_CHECK_VENDER;

			break;

		case FDP_CMD_FRAME_SIZE :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_CHECK_VENDER)
				return -FDP_ERRRECV;

			sendbuf[0] = FDP_CMD_FRAME_SIZE;
			sendbuf[1] = FDP_DEVICE;
			/* Get user configuration of the frame size. */
			if (fdp_get_fram_size(&fdp_status->max_frame_size, fdp_status->dev_type))
				return -FDP_CHECKERR;

			/* If there is a minimum byte write requirement when writing memory,
			the macro can be opened and must be aligned with a multiple of
			the maximum number of bytes that can be supported for data packet transmission */
#if (FDP_MEM_LINE_SIZE)
			if (fdp_status->max_frame_size % FDP_MEM_LINE_SIZE)
				return -FDP_CHECKERR;
#endif

			memcpy((void *)&(sendbuf[2]), (void *)&fdp_status->max_frame_size, 4);
			crc_val = fdp_crc16_modbus(sendbuf, FDP_CMD_FRAME_SIZE_LENGTH - 2);
			sendbuf[FDP_CMD_FRAME_SIZE_LENGTH - 2] = (crc_val & 0xff00) >> 8;
			sendbuf[FDP_CMD_FRAME_SIZE_LENGTH - 1] = (crc_val & 0x00ff);

			if (fdp_transmit(sendbuf, FDP_CMD_FRAME_SIZE_LENGTH, fdp_status->dev_type))
				return -FDP_CHECKERR;

			fdp_status->program_flow = FDP_BOOT_FLOW_SEND_MAX_FRAME_SIZE;

			return FDP_SUCCESS;

		case FDP_CMD_FILE_SIZE :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_SEND_MAX_FRAME_SIZE)
				return -FDP_ERRRECV;
			/* We will save the size of image. */
			memcpy((void *)&fdp_status->file_size, (void *)&buf[2], 4);

			fdp_status->program_flow = FDP_BOOT_FLOW_GET_IMAGE_SIZE;
			break;

		case FDP_CMD_GET_ADDRESS :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_GET_IMAGE_SIZE)
				return -FDP_ERRRECV;

			/* Save the storage address. */
			memcpy((void *)&fdp_status->address, (void *)&buf[2], 4);

			if (FDP_SECURITY_MEMORY_CHECK(fdp_status->address, fdp_status->file_size))
				return -FDP_ERRRECV;

			fdp_status->program_flow = FDP_BOOT_FLOW_GET_STORAGE_ADDR;
			break;

		case FDP_CMD_INIT_MEM :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_GET_STORAGE_ADDR)
				return -FDP_ERRRECV;

			fdp_status->mem_init_ready = true;

			return FDP_SUCCESS;

		case FDP_CMD_TRANSFER_REQ :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_INIT_MEM)
				return -FDP_ERRRECV;

			memcpy((void *)&fdp_status->frame_size, (void *)&buf[2], 4);

			if (fdp_status->frame_size > fdp_status->max_frame_size)
				return -FDP_ERRRECV;

			fdp_status->program_stage = FDP_PRG_DATA_PHASE;

			break;

		case FDP_CMD_TRANSFER_END :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_DOWNLOAD_END)
				return -FDP_ERRRECV;

			/* Waitting for image download completed, User need to send ACK to the upper layer. */
			fdp_status->program_flow = FDP_BOOT_FLOW_UPDATE_END;

			break;

		case FDP_CMD_CHECK_IMAGE :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_UPDATE_END)
				return -FDP_ERRRECV;

#if FDP_IS_CHECK_IMAGE_CRC32
			/* Check if the image is correct. */
			fdp_dcache_invaild(fdp_status->address, fdp_status->file_size);
			fdp_status->crc_val = fdp_crc32((uint8_t *)fdp_status->address, fdp_status->file_size);

			memcpy((void *)&crc_val, (void *)&buf[2], 4);

			if (fdp_status->crc_val != crc_val)
				return -FDP_ERRRECV;
#endif

			fdp_status->image_ready = true;

			fdp_status->program_flow = FDP_BOOT_FLOW_CHECK_IMAGE;

			break;

		/* Check if necessary to release core. */
		case FDP_CMD_RELEASE_CORE :
			if (fdp_status->program_flow != FDP_BOOT_FLOW_CHECK_IMAGE)
				return -FDP_ERRRECV;

			memcpy((void *)&fdp_status->release_core, (void *)&buf[2], 4);

			fdp_status->program_flow = FDP_BOOT_FLOW_WAIT_FOR_UPTATE;

			break;

		default :
			return -FDP_ERRRECV;
	}

	fdp_ack_to_host(fdp_status->dev_type, true);

	return FDP_SUCCESS;
}

static int fdp_program_data_stage(uint8_t *buf, uint32_t len)
{
	fdp_record_stat_t *fdp_status = fdp_get_current_stat();

	if (buf == NULL || len == 0)
		return -FDP_ERRRECV;

	/* Only dispose data phase. */
	if (fdp_status->program_stage != FDP_PRG_DATA_PHASE)
		return -FDP_ERRINPARAM;

	if (fdp_status->program_flow != FDP_BOOT_FLOW_INIT_MEM)
		return -FDP_ERRRECV;

	if (len < fdp_status->frame_size)
		return FDP_CONTINUE;

	if (len != fdp_status->frame_size)
		return -FDP_ERRRECV;

#if (FDP_MEM_LINE_SIZE)
	if (fdp_status->frame_size % FDP_MEM_LINE_SIZE) {
		if ((fdp_status->received_data_size +
			fdp_status->frame_size) < fdp_status->file_size)
			return -FDP_ERRRECV;

		fdp_status->download_ready = true;
	}
#endif

	if ((fdp_status->received_data_size + fdp_status->frame_size) > fdp_status->file_size)
		return -FDP_ERRRECV;
	else if ((fdp_status->received_data_size + fdp_status->frame_size) == fdp_status->file_size)
		fdp_status->download_ready = true;

	fdp_status->crc_val = fdp_crc16_modbus(buf, len);

	fdp_status->program_flow = FDP_BOOT_FLOW_GET_IMAGE_DATA;
	fdp_status->program_stage = FDP_PRG_CHECK_PHASE;

	fdp_ack_to_host(fdp_status->dev_type, true);

	return FDP_SUCCESS;
}

static int fdp_program_check_stage(uint8_t *buf, uint32_t len)
{
	uint16_t crc_val;

	fdp_record_stat_t *fdp_status = fdp_get_current_stat();

	if (buf == NULL || len == 0)
		return -FDP_ERRRECV;

	/* Only dispose data phase. */
	if (fdp_status->program_stage != FDP_PRG_CHECK_PHASE)
		return -FDP_ERRINPARAM;

	/* Check if received data is correct. */
	if (fdp_status->program_flow != FDP_BOOT_FLOW_GET_IMAGE_DATA)
		return -FDP_ERRRECV;

	/* Check if received data is correct. */
	if (buf[0] != FDP_CMD_TRANSFER_CHECK)
		return -FDP_ERRRECV;

	if (len < FDP_CMD_TRANSFER_CHECK_LENGTH)
		return FDP_CONTINUE;

	if (len != FDP_CMD_TRANSFER_CHECK_LENGTH)
		return -FDP_ERRRECV;

	/* Check if the frame is correct. */
	crc_val = fdp_crc16_modbus(buf, len - 2);

	if (((crc_val & 0xff00) >> 8) != buf[len - 2] &&
		(crc_val & 0x00ff) != buf[len - 1])
		return -FDP_ERRRECV;

	/* Check if the data frame is correct. */
	memcpy((void *)&crc_val, (void *)&buf[2], 2);

	if (crc_val != fdp_status->crc_val)
		return -FDP_ERRRECV;

	fdp_status->transfer_done = true;

	return FDP_SUCCESS;
}

/**
 * @brief FDP dispose received data.
 * 
 * @param buf Received data buffer.
 * @return int Return is zero is success, else error.
 */
static int fdp_receive_data_stream(uint8_t *buf)
{
	int ret = 0;
	uint32_t len;
	bool wrap_around = false;

	fdp_record_stat_t *fdp_status = fdp_get_current_stat();
	uint8_t *fdp_fram_buf = fdp_get_frame_buffer();
	uint8_t *fdp_check_buf = fdp_get_check_buffer();

	if (buf == NULL)
		return -FDP_ERRINPARAM;

	if (fdp_status->rx_read < fdp_status->rx_write)
		len = fdp_status->rx_write - fdp_status->rx_read;
	else if (fdp_status->rx_read > fdp_status->rx_write)
		len = FDP_CONFIG_FRAME_BUFFER_SIZE - fdp_status->rx_read;
	else
		return FDP_CONTINUE;

	if (len == 0) {
		fdp_status->rx_read = 0;
		return FDP_CONTINUE;
	}

	if (fdp_status->program_stage != FDP_PRG_CHECK_PHASE)
		memcpy((void *)&(fdp_fram_buf[fdp_status->received_index]), (void *)&buf[fdp_status->rx_read], len);
	else
		memcpy((void *)&(fdp_check_buf[fdp_status->received_index]), (void *)&buf[fdp_status->rx_read], len);

	if ((fdp_status->rx_read + len) > FDP_CONFIG_FRAME_BUFFER_SIZE)
		fdp_status->rx_read = len - (FDP_CONFIG_FRAME_BUFFER_SIZE - fdp_status->rx_read - 1);
	else
		fdp_status->rx_read += len;

	switch (fdp_status->program_stage) {
		case FDP_PRG_CMD_PHASE :
			ret = fdp_program_command_stage(fdp_fram_buf,
											fdp_status->received_index + len);
			break;

		case FDP_PRG_DATA_PHASE :
			ret = fdp_program_data_stage(fdp_fram_buf,
										fdp_status->received_index + len);
			break;

		case FDP_PRG_CHECK_PHASE :
			ret = fdp_program_check_stage(fdp_check_buf,
										fdp_status->received_index + len);
			break;

		default :
			/* If the program stage is not valid, then reset the program stage. */
			ret = -FDP_ERRINPARAM;
			break;
	}

	if (ret < 0) {
		fdp_reset_fdp();
		fdp_ack_to_host(fdp_status->dev_type, false);
		return ret;
	}

	if (ret == FDP_CONTINUE)
		fdp_status->received_index += len;
	else
		fdp_status->received_index = 0;

	fdp_status->watchdog_timeout = 0;

	return FDP_SUCCESS;
}

fdp_record_stat_t *fdp_init(fdp_device_type_t type)
{
	fdp_record_stat_t *fdp_status = fdp_get_current_stat();

	memset((void *)fdp_status, 0, sizeof(fdp_record_stat_t));
	fdp_status->dev_type = type;
	fdp_status->program_flow = FDP_BOOT_FLOW_WAIT_FOR_UPTATE;

	if (FDP_MEM_LINE_SIZE > FDP_CONFIG_FRAME_BUFFER_SIZE)
		return NULL;

	if (fdp_device_init(type))
		return NULL;

	return fdp_status;
}

int fdp_receive_data_to_buffer(void)
{
	fdp_record_stat_t *fdp = fdp_get_current_stat();

	return fdp_receive(fdp_get_recv_buffer(), &fdp->rx_write, fdp->dev_type);
}

int fdp_download(fdp_device_type_t type)
{
	int ret = 0;

	fdp_record_stat_t *fdp = fdp_get_current_stat();

	ret = fdp_receive_data_stream(fdp_get_recv_buffer());
	if (ret)
		return ret;

	switch (fdp->program_flow) {
		case FDP_BOOT_FLOW_GET_STORAGE_ADDR :
			if (!fdp->mem_init_ready)
				break;

#if (FDP_MEM_ERASE_IS_DISABLE_INT)
#if (FDP_IS_USE_INTERRUPT)
			FDP_DISABLE_INTERRUPTS;
#endif
#endif
			/* Initialize memory. */
			ret = fdp_storage_dev_init(fdp->address, fdp->file_size);
			if (ret)
				goto fdp_down_err;

			/* If memory needs to be erased, execute; otherwise, return success. */
			ret = fdp_storage_dev_erase(fdp->address, fdp->file_size);
			if (ret)
				goto fdp_down_err;

#if (FDP_MEM_ERASE_IS_DISABLE_INT)
#if (FDP_IS_USE_INTERRUPT)
			FDP_ENABLE_INTERRUPTS;
#endif
#endif

			fdp->program_flow = FDP_BOOT_FLOW_INIT_MEM;
			fdp->mem_init_ready = false;

			fdp_ack_to_host(fdp->dev_type, true);

			break;

		case FDP_BOOT_FLOW_GET_IMAGE_DATA :

/* Check if the flash write is disable interrupt. */
#if (FDP_MEM_WRITE_IS_DISABLE_INT)
#if (FDP_IS_USE_INTERRUPT)
			FDP_DISABLE_INTERRUPTS;
#endif
#endif
			if (!fdp->transfer_done)
				break;

			/* Write data to memory. */
			ret = fdp_storage_dev_write(fdp_get_frame_buffer(),
										fdp->address + fdp->received_data_size,
										fdp->frame_size);

			if (ret)
				goto fdp_down_err;

			if (fdp->download_ready)
				fdp->program_flow = FDP_BOOT_FLOW_DOWNLOAD_END;
			else	/* Continue to transfer data. */
				fdp->program_flow = FDP_BOOT_FLOW_INIT_MEM;

			fdp->received_data_size += fdp->frame_size;
			fdp->program_stage = FDP_PRG_CMD_PHASE;
			fdp->transfer_done = false;

			fdp_ack_to_host(fdp->dev_type, true);

#if (FDP_MEM_WRITE_IS_DISABLE_INT)
#if (FDP_IS_USE_INTERRUPT)
			FDP_ENABLE_INTERRUPTS;
#endif
#endif

			break;
	}

	fdp->watchdog_timeout = 0;

	return FDP_SUCCESS;

fdp_down_err :

	fdp_reset_fdp();

	fdp_ack_to_host(fdp->dev_type, false);

#if (FDP_MEM_WRITE_IS_DISABLE_INT)
#if (FDP_IS_USE_INTERRUPT)
	FDP_ENABLE_INTERRUPTS;
#endif
#endif

	return ret;
}
