/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "fdp.h"
#include "fdp_err.h"
#include "fdp_jump.h"
#include "fdp_defconfig.h"

/**
 * @brief User define include header file.
 * 
 */
#include "driverlib.h"
#include "uart_dev.h"
#include "interrupt.h"
#include "gs32_flashAPI_lib.h"
#include "inc/hw_flash.h"

/* Define the device frame size and Can not more than 512 bytes. */
#define FDP_UART_FRAME_SIZE			512U
#define FDP_CAN_FRAME_SIZE			32U
#define FDP_I2C_FRAME_SIZE			32U
#define FDP_SPI_FRAME_SIZE			32U

#define FLASH_SECTOR_SIZE			FlashMainPageSize
#define BANK0_BASE_ADDR				0x08000000U
#define BANK1_BASE_ADDR				0x08080000U

int fdp_transmit(uint8_t *buf, uint32_t len, fdp_device_type_t type)
{
	switch (type) {
		case FDP_DEV_UART :
			if (Uart_transmit(buf, len))
				return -FDP_ERRTRAN;
			break;

		case FDP_DEV_CAN :
			break;

		case FDP_DEV_I2C :
			break;

		case FDP_DEV_SPI :
			break;

		default :
			return -FDP_ERRINPARAM;
	}

	return FDP_SUCCESS;
}

int fdp_receive(uint8_t *buf, uint32_t *index, fdp_device_type_t type)
{
	switch (type) {
		case FDP_DEV_UART :
			if (Uart_Receiver(buf, index, FDP_CONFIG_FRAME_BUFFER_SIZE))
				return -FDP_ERRTRAN;
			break;

		case FDP_DEV_CAN :
			break;

		case FDP_DEV_I2C :
			break;

		case FDP_DEV_SPI :
			break;

		default :
			return -FDP_ERRINPARAM;
	}

	return FDP_SUCCESS;
}

int fdp_device_init(fdp_device_type_t type)
{
	switch (type) {
		case FDP_DEV_UART :
			if (Uart_Init())
				return -FDP_ERRTRAN;
			break;

		case FDP_DEV_CAN :
			break;

		case FDP_DEV_I2C :
			break;

		case FDP_DEV_SPI :
			break;

		default :
			return -FDP_ERRINPARAM;
	}

	return FDP_SUCCESS;
}

int fdp_get_fram_size(uint32_t *size, fdp_device_type_t type)
{
	if (size == NULL)
		return -FDP_ERRINPARAM;

	switch (type) {
		case FDP_DEV_UART :
			*size = FDP_UART_FRAME_SIZE;
			break;

		case FDP_DEV_CAN :
			*size = FDP_CAN_FRAME_SIZE;
			break;

		case FDP_DEV_I2C :
			*size = FDP_I2C_FRAME_SIZE;
			break;

		case FDP_DEV_SPI :
			*size = FDP_SPI_FRAME_SIZE;
			break;

		default :
			return -FDP_ERRINPARAM;
	}

	return FDP_SUCCESS;
}

void fdp_globale_interrupt_enable(void)
{
	__enable_irq();
}

void fdp_globale_interrupt_disable(void)
{
	__enable_irq();
}

int fdp_boot_sel_mode_init(void)
{
	return FDP_SUCCESS;
}

int fdp_boot_mode_get(fdp_device_type_t *type)
{
	uint8_t pin_val = 1;

	if (type == NULL)
		return -FDP_ERRINPARAM;

	switch (pin_val) {
		case 0 :
			*type = FDP_MEM_FLASH;
			break;

		case 1 :
			*type = FDP_DEV_UART;
			break;

		case 2 :
			*type = FDP_DEV_CAN;
			break;

		case 3 :
			*type = FDP_DEV_I2C;
			break;

		case 4 :
			*type = FDP_DEV_SPI;
			break;

		default :
			return -FDP_ERRINPARAM;
	}

	return FDP_SUCCESS;
}

void fdp_boot_jump_mem(fdp_device_type_t type)
{
	switch (type) {
		case FDP_MEM_FLASH :
			fdp_jump_to_address_with_arg(0x08000000U, true);
			break;

		default :
			return ;
	}

	while(1);
}
