/*
 * Copyright (c) 2019 Nuclei Limited. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/******************************************************************************
 * @file     gcc_demosoc_ilm.ld
 * @brief    GNU Linker Script for Nuclei N/NX based device in ilm Download Mode
 * @version  V1.0.0
 * @date     17. Dec 2019
 ******************************************************************************/
OUTPUT_ARCH( "riscv" )

INCLUDE "../lds/memory_region.ld"

ENTRY( _start )

PROVIDE(__STACK_SIZE = 2K);

MEMORY
{
  app_flash			(rxa!w) : ORIGIN = FLASH_APP1_ADDR, LENGTH = FLASH_APP1_SIZE
  app_end_mark		(rxa!w) : ORIGIN = FLASH_ENDM1_ADDR, LENGTH = FLASH_ENDM1_SIZE 
  ilm      			(rxa!w) : ORIGIN = ILM_REGION_ADDR, LENGTH = ILM_REGION_SIZE
  dlm      			(wxa!r) : ORIGIN = DLM_REGION_ADDR, LENGTH = DLM_REGION_SIZE -__STACK_SIZE
  dlm_stk  			(wxa!r) : ORIGIN = DLM_REGION_ADDR + DLM_REGION_SIZE -__STACK_SIZE, LENGTH = __STACK_SIZE
  
  /* share memory only between bootloader and app */
  msgbootapp_ram 	(wxa!r) : ORIGIN = BOOTAPP_SHARE_ADDR, LENGTH = BOOTAPP_SHARE_SIZE
  /* shared memory used by CPU1 */
  msg_ram			(wxa!r) : ORIGIN = MSG_SHARE_ADDR, LENGTH = MSG_SHARE_SIZE

  /* single core mode only */
  flex_ram (wxa!r) : ORIGIN = 0x10220000, LENGTH = 128K
}

REGION_ALIAS("ROM", app_flash)
REGION_ALIAS("ILM", ilm)
REGION_ALIAS("RAM", dlm)
REGION_ALIAS("MSGBOOTAPP_RAM",msgbootapp_ram)
REGION_ALIAS("MSG_RAM", msg_ram)

SECTIONS
{
  /* To provide symbol __STACK_SIZE, __HEAP_SIZE and __SMP_CPU_CNT */
  PROVIDE(__HEAP_SIZE = 1K);
  PROVIDE(__SMP_CPU_CNT = 1);
  __TOT_STACK_SIZE = __STACK_SIZE * __SMP_CPU_CNT;

  .app_end_out		:ALIGN(16)
  {
  	KEEP(*(.app_end))
  	KEEP(*(.app_end.*))
  	. = ALIGN(16);
  } >app_end_mark
  
  .rela.init : { *(.rela.init) }
  
  .init           :
  {
    PROVIDE( __app_info_addr = . );
    KEEP(*(.fdp_app_info))
    . = ALIGN(8);
    PROVIDE( __app_start_addr = . );    
    *(.vtable)
    *(.vtable_s)
    KEEP (*(SORT_NONE(.init)))
    . = ALIGN(16);
  } >ROM AT>ROM
  
  .ilm_vector (NOLOAD)  :
  {
    . = ALIGN(512);
    __vector_remap_start__ = .;
    . += 260*4;
    __vector_remap_end__ = .;
  } >ILM

  PROVIDE( __vector_remap_start = __vector_remap_start__ );
  PROVIDE( __vector_remap_end = __vector_remap_end__ );
  PROVIDE( __image_start_offset = __app_start_addr - __app_info_addr);
  
  .rela.dyn : { *(.rela.dyn) }    /* add to avoid orphan sections warnings */
  .rela.text : { *(.rela.text*) } /* add to avoid orphan sections warnings */
  
  .text           :
  {
    *(.text.unlikely .text.unlikely.*)
    *(.text.startup .text.startup.*)
    *(.text .text.*)
    *(.gnu.linkonce.t.*)
    . = ALIGN(16);
  } >ILM AT>ROM

  .fini           :
  {
    KEEP (*(SORT_NONE(.fini)))
    . = ALIGN(4);
  } >ROM AT>ROM

  .preinit_array  :
  {
    PROVIDE_HIDDEN (__preinit_array_start = .);
    KEEP (*(.preinit_array))
    PROVIDE_HIDDEN (__preinit_array_end = .);
    . = ALIGN(4);
  } >ROM AT>ROM

  .init_array     :
  {
    PROVIDE_HIDDEN (__init_array_start = .);
    KEEP (*(SORT_BY_INIT_PRIORITY(.init_array.*) SORT_BY_INIT_PRIORITY(.ctors.*)))
    KEEP (*(.init_array EXCLUDE_FILE (*crtbegin.o *crtbegin?.o *crtend.o *crtend?.o ) .ctors))
    PROVIDE_HIDDEN (__init_array_end = .);
    . = ALIGN(4);
  } >ROM AT>ROM

  .fini_array     :
  {
    PROVIDE_HIDDEN (__fini_array_start = .);
    KEEP (*(SORT_BY_INIT_PRIORITY(.fini_array.*) SORT_BY_INIT_PRIORITY(.dtors.*)))
    KEEP (*(.fini_array EXCLUDE_FILE (*crtbegin.o *crtbegin?.o *crtend.o *crtend?.o ) .dtors))
    PROVIDE_HIDDEN (__fini_array_end = .);
    . = ALIGN(4);
  } >ROM AT>ROM

  .ctors          :
  {
    /* gcc uses crtbegin.o to find the start of
     * the constructors, so we make sure it is
     * first.  Because this is a wildcard, it
     * doesn't matter if the user does not
     * actually link against crtbegin.o; the
     * linker won't look for a file to match a
     * wildcard.  The wildcard also means that it
     * doesn't matter which directory crtbegin.o
     * is in.
     */
    KEEP (*crtbegin.o(.ctors))
    KEEP (*crtbegin?.o(.ctors))
    /* We don't want to include the .ctor section from
     * the crtend.o file until after the sorted ctors.
     * The .ctor section from the crtend file contains the
     * end of ctors marker and it must be last
     */
    KEEP (*(EXCLUDE_FILE (*crtend.o *crtend?.o ) .ctors))
    KEEP (*(SORT(.ctors.*)))
    KEEP (*(.ctors))
    . = ALIGN(4);
  } >ROM AT>ROM

  .dtors          :
  {
    KEEP (*crtbegin.o(.dtors))
    KEEP (*crtbegin?.o(.dtors))
    KEEP (*(EXCLUDE_FILE (*crtend.o *crtend?.o ) .dtors))
    KEEP (*(SORT(.dtors.*)))
    KEEP (*(.dtors))
    . = ALIGN(4);
  } >ROM AT>ROM

  .const.table  :
  {
    . = ALIGN(4);
    __dualcore_ack_callback_table_start__ = .;
    KEEP(*(.dualcore_ack))
    __dualcore_ack_callback_table_end__ = .;

    . = ALIGN(4);
    __dualcore_request_callback_table_start__ = .;
    KEEP(*(.dualcore_request))
    __dualcore_request_callback_table_end__ = .;

    . = ALIGN(4);
    __dualcore_notify_callback_table_start__ = .;
    KEEP(*(.dualcore_notify))
    __dualcore_notify_callback_table_end__ = .;

    . = ALIGN(4);
    __cli_cmds_table_start__ = .;
    KEEP(*(.cli_cmds))
    __cli_cmds_table_end__ = .;

    . = ALIGN(4);
    KEEP(*(.rom_data))
    KEEP(*(.rom_data.*))
    *(.rom_data_not_keep)
    *(.rom_data_not_keep.*)
    . = ALIGN(16);
  }>ROM AT>ROM

  .copy.table :
  {
    . = ALIGN(4);
    __copy_table_start__ = .;

    LONG (LOADADDR(.data))
    LONG (ADDR(.data))
    LONG (SIZEOF(.data) / 4)

    LONG (LOADADDR(.ilm))
    LONG (ADDR(.ilm))
    LONG (SIZEOF(.ilm) / 4)
    
    LONG (LOADADDR(.text))
    LONG (ADDR(.text))
    LONG (SIZEOF(.text) / 4)
    
    LONG (LOADADDR(.msgram_data))
    LONG (ADDR(.msgram_data))
    LONG (SIZEOF(.msgram_data) / 4)
    
    __copy_table_end__ = .;

    . = ALIGN(16);
  } > ROM

  PROVIDE( __copy_table_start = __copy_table_start__ );
  PROVIDE( __copy_table_end = __copy_table_end__ );

  .zero.table :
  {
    . = ALIGN(4);
    __zero_table_start__ = .;

    LONG (ADDR(.bss))
    LONG (SIZEOF(.bss) / 4)

    LONG (ADDR(.ilm_bss))
    LONG (SIZEOF(.ilm_bss) / 4)
    
    LONG (ADDR(.msgram_bss))
    LONG (SIZEOF(.msgram_bss) / 4)

    __zero_table_end__ = .;

    . = ALIGN(16);
  } > ROM
  
  .RomFunc :ALIGN(4)
  {
    *(.RomFunc)
    *(.RomFunc.*)
    . = ALIGN(16);
  } > ROM
  
  PROVIDE( __zero_table_start = __zero_table_start__ );
  PROVIDE( __zero_table_end = __zero_table_end__ );
  
  PROVIDE( _text_lma = LOADADDR(.text) );
  PROVIDE( _text = ADDR(.text) );
  PROVIDE (_etext = .);
  PROVIDE (__etext = .);
  PROVIDE (etext = .);
  
  .ilm   :ALIGN(4)
  {
    *(.RamFunc)
    *(.RamFunc.*)
    *(.IlmFunc)
    *(.IlmFunc.*)
    *(.IlmData)
    *(.IlmData.*)
    . = ALIGN(16);
  } >ILM  AT>ROM

  .ilm_bss (NOLOAD) : ALIGN(8)
  {
    *(.ilmbss)
    *(.ilmbss.*)
    . = ALIGN(4);
  } >ILM
  
  PROVIDE( _ilm_lma = LOADADDR(.ilm) );
  PROVIDE( _ilm = ADDR(.ilm) );
  PROVIDE( _eilm = . );
  PROVIDE( eilm = . );
  
  .msgram_bootapp_bss (NOLOAD) : ALIGN(4)
  {
    KEEP(*(.shared_bootapp_bss))
    KEEP(*(.shared_bootapp_bss.*))
    . = ALIGN(4);
  } > MSGBOOTAPP_RAM
  
  .rela.sdata : { *(.rela.sdata*) }  /* add to avoid orphan sections warnings */
  .rela.data : { *(.rela.data*) }    /* add to avoid orphan sections warnings */
  
  .data       : ALIGN(8)
  {
    KEEP(*(.data.ctest*))
    *(.data .data.*)
    *(.gnu.linkonce.d.*)
    . = ALIGN(8);
    PROVIDE( __gp_rel16$ = . + 0x8000 );
    PROVIDE( __global_pointer$ = . + 0x800 );
    *(.sdata .sdata.* .sdata*)
    *(.gnu.linkonce.s.*)
    /* readonly data placed in RAM for access speed */
    . = ALIGN(8);
    *(.srodata.cst16)
    *(.srodata.cst8)
    *(.srodata.cst4)
    *(.srodata.cst2)
    *(.srodata .srodata.*)
    *(.rdata)
    *(.rodata .rodata.*)
    *(.gnu.linkonce.r.*)
    /* below sections are used for rt-thread */
    . = ALIGN(4);
    __rt_init_start = .;
    KEEP(*(SORT(.rti_fn*)))
    __rt_init_end = .;
    . = ALIGN(4);
    __fsymtab_start = .;
    KEEP(*(FSymTab))
    __fsymtab_end = .;
    . = ALIGN(4);
    __vsymtab_start = .;
    KEEP(*(VSymTab))
    __vsymtab_end = .;
    . = ALIGN(16);
  } >RAM AT>ROM

  .tdata           : ALIGN(8)
  {
    PROVIDE( __tls_base = . );
    *(.tdata .tdata.* .gnu.linkonce.td.*)
    . = ALIGN(16);
  } >RAM AT>ROM

  PROVIDE( _data_lma = LOADADDR(.data) );
  PROVIDE( _data = ADDR(.data) );
  PROVIDE( _edata = . );
  PROVIDE( edata = . );

  PROVIDE( _fbss = . );
  PROVIDE( __bss_start = . );

  .tbss (NOLOAD)   : ALIGN(8)
  {
    *(.tbss .tbss.* .gnu.linkonce.tb.*)
    *(.tcommon)
    PROVIDE( __tls_end = . );
  } >RAM AT>RAM

  .tbss_space (NOLOAD) : ALIGN(8)
  {
    . = . + SIZEOF(.tbss);
  } >RAM AT>RAM

  .bss (NOLOAD)   : ALIGN(8)
  {
    __bss_start__ = .;
    *(.sbss*)
    *(.gnu.linkonce.sb.*)
    *(.bss .bss.*)
    *(.gnu.linkonce.b.*)
    *(COMMON)
    . = ALIGN(4);
    __bss_end__ = .;
  } >RAM AT>RAM

  PROVIDE( _end = . );
  PROVIDE( end = . );
  
  /* shared sections */
  .msgram_data : ALIGN(4)
  {
    KEEP(*(.shared_header))
    KEEP(*(.shared_data))
    KEEP(*(.shared_data.*))
    KEEP(*(.shared_cpu1tocpu2_data))
    KEEP(*(.shared_cpu1tocpu2_data.*))
    . = ALIGN(16);
  } > MSG_RAM AT>ROM
  
  .msgram_bss (NOLOAD) : ALIGN(4)
  {
    KEEP(*(.shared_bss))
    KEEP(*(.shared_bss.*))
    KEEP(*(.shared_cpu1tocpu2_bss))
    KEEP(*(.shared_cpu1tocpu2_bss.*))
    . = ALIGN(4);
  } > MSG_RAM
  
  .msgram_cpu2tocpu1_data (NOLOAD) : ALIGN(4)
  {
    KEEP(*(.shared_cpu2tocpu1_data))
    KEEP(*(.shared_cpu2tocpu1_data.*))
    . = ALIGN(4);
  } > MSG_RAM
  
  .msgram_cpu2tocpu1_bss (NOLOAD) : ALIGN(4)
  {
    KEEP(*(.shared_cpu2tocpu1_bss))
    KEEP(*(.shared_cpu2tocpu1_bss.*))
    . = ALIGN(4);
  } > MSG_RAM
  
  
  /* Nuclei C Runtime Library requirements:
   * 1. heap need to be align at 16 bytes
   * 2. __heap_start and __heap_end symbol need to be defined
   * 3. reserved at least __HEAP_SIZE space for heap
   */
  .heap (NOLOAD)   : ALIGN(16)
  {
    . = ALIGN(16);
    PROVIDE( __heap_start = . );
    . += __HEAP_SIZE;
    . = ALIGN(16);
    PROVIDE( __heap_limit = . );
  } >RAM AT>RAM

  .stack ORIGIN(dlm_stk) (NOLOAD) :
  {
    . = ALIGN(16);
    PROVIDE( _heap_end = . );
    PROVIDE( __heap_end = . );
    PROVIDE( __StackLimit = . );
    PROVIDE( __StackBottom = . );
    . += __TOT_STACK_SIZE;
    . = ALIGN(16);
    PROVIDE( __StackTop = . );
    PROVIDE( _sp = . );
  } >dlm_stk AT>dlm_stk

  .comment 0 : { *(.comment) }
  .debug_abbrev 0 : { *(.debug_abbrev) }
  .debug_aranges 0 : { *(.debug_aranges) }  
  .debug_frame 0 : { *(.debug_frame) }
  .debug_info 0 : { *(.debug_info) }
  .debug_line 0 : { *(.debug_line) }
  .debug_line_str 0 : { *(.debug_line_str) }
  .debug_loc 0 : { *(.debug_loc) }
  .debug_loclists 0 : { *(.debug_loclists) }
  .debug_macro 0 : { *(.debug_macro) }
  .debug_pubnames 0 : { *(.debug_pubnames) }
  .debug_pubtypes 0 : { *(.debug_pubtypes) }
  .debug_ranges 0 : { *(.debug_ranges) }
  .debug_rnglists 0 : { *(.debug_rnglists) }
  .debug_str 0 : { *(.debug_str) }
  .debug_str_offsets 0 : { *(.debug_str_offsets) }
  .riscv.attributes 0 : { *(.riscv.attributes) }
}
