/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

INCLUDE "../lds/memory_region.ld"

OUTPUT_ARCH( "riscv" )

ENTRY( _start )

MEMORY
{
	flash	(rxa!w) : ORIGIN = FLASH_BOOTLOAD_REGION_ADDR,	LENGTH = FLASH_BOOTLOAD_REGION_SIZE
	ilm		(rxa!w) : ORIGIN = ILM_REGION_ADDR,				LENGTH = ILM_REGION_SIZE
	dlm		(wxa!r) : ORIGIN = DLM_REGION_ADDR,				LENGTH = DLM_REGION_SIZE
	axi_ram	(wxa!r) : ORIGIN = AXI_RAM0_REGION_ADDR,		LENGTH = AXI_RAM0_REGION_SIZE
	s_msg	(wxa!r) : ORIGIN = MSG_SHARE_MEMORY_ADDR,		LENGTH = MSG_SHARE_MEMORY_SIZE
}

REGION_ALIAS("FLASH", flash)
REGION_ALIAS("ILM", ilm)
REGION_ALIAS("DLM", dlm)
REGION_ALIAS("RAM", axi_ram)
REGION_ALIAS("SHARE_RAM", s_msg)

SECTIONS
{
	PROVIDE(__HEAP_SIZE = 1K);
	PROVIDE(__STACK_SIZE = 2K);

	.init :
	{
		*(.vtable)
		*(.vtable_s)
		KEEP (*(SORT_NONE(.init)))
		. = ALIGN(16);
	} >FLASH AT>FLASH

	.text :
	{
		*(.text.unlikely .text.unlikely.*)
		*(.text.startup .text.startup.*)
		*(.text .text.*)
		*(.gnu.linkonce.t.*)
		*(.rom_data)
		. = ALIGN(16);
	} >FLASH AT>FLASH

	.copy.table :
	{
		. = ALIGN(4);
		__copy_table_start__ = .;

		LONG (LOADADDR(.data))
		LONG (ADDR(.data))
		LONG (SIZEOF(.data) / 4)

		LONG (LOADADDR(.tdata))
		LONG (ADDR(.tdata))
		LONG (SIZEOF(.tdata) / 4)

		LONG (LOADADDR(.ilm))
		LONG (ADDR(.ilm))
		LONG (SIZEOF(.ilm) / 4)

		__copy_table_end__ = .;

		. = ALIGN(16);
	} >FLASH AT>FLASH

	PROVIDE( __copy_table_start = __copy_table_start__ );
	PROVIDE( __copy_table_end = __copy_table_end__ );

	.zero.table :
	{
		. = ALIGN(4);
		__zero_table_start__ = .;

		LONG (ADDR(.bss))
		LONG (SIZEOF(.bss) / 4)

		LONG (ADDR(.tbss))
		LONG (SIZEOF(.tbss) / 4)

		LONG (ADDR(.msg))
		LONG (SIZEOF(.msg) / 4)

		__zero_table_end__ = .;

		. = ALIGN(16);
	} >FLASH AT>FLASH

	PROVIDE( __zero_table_start = __zero_table_start__ );
	PROVIDE( __zero_table_end = __zero_table_end__ );

	.ilm_vector (NOLOAD) :
	{
		. = ALIGN(512);
		__vector_remap_start__ = .;
		. += 260 *4;
		__vector_remap_end__ = .;
		. = ALIGN(16);
	} >ILM

	PROVIDE( __vector_remap_start = __vector_remap_start__ );
	PROVIDE( __vector_remap_end = __vector_remap_end__ );

	.ilm :
	{
		. = ALIGN(8);
		*(.RamFunc)
		*(.RamFunc.*)
		*(.IlmFunc)
		*(.IlmFunc.*)
		*(.IlmData)
		*(.IlmData.*)
		. = ALIGN(16);
	} >ILM AT>FLASH

	.data :
	{
		. = ALIGN(8);
		KEEP(*(.data.ctest*))
		*(.data .data.*)
		*(.gnu.linkonce.d.*)
		. = ALIGN(8);
		PROVIDE( __global_pointer$ = . + 0x800 );
		*(.sdata .sdata.* .sdata*)
		*(.gnu.linkonce.s.*)
		/* readonly data placed in RAM for access speed */
		. = ALIGN(8);
		*(.srodata.cst16)
		*(.srodata.cst8)
		*(.srodata.cst4)
		*(.srodata.cst2)
		*(.srodata .srodata.*)
		*(.rdata)
		*(.rodata .rodata.*)
		*(.gnu.linkonce.r.*)
		/* below sections are used for rt-thread */
		. = ALIGN(4);
		__rt_init_start = .;
		KEEP(*(SORT(.rti_fn*)))
		__rt_init_end = .;
		. = ALIGN(4);
		__fsymtab_start = .;
		KEEP(*(FSymTab))
		__fsymtab_end = .;
		. = ALIGN(4);
		__vsymtab_start = .;
		KEEP(*(VSymTab))
		__vsymtab_end = .;
		. = ALIGN(16);
	} >DLM AT>FLASH

	.tdata :
	{
		. = ALIGN(8);
		PROVIDE( __tls_base = . );
		*(.tdata .tdata.* .gnu.linkonce.td.*)
		. = ALIGN(16);
	} >DLM AT>FLASH

	.tbss (NOLOAD) : ALIGN(8)
	{
		. = ALIGN(8);
		*(.tbss .tbss.* .gnu.linkonce.tb.*)
		*(.tcommon)
		. = ALIGN(16);
		PROVIDE( __tls_end = . );
	} >DLM AT>DLM

	.bss (NOLOAD) : ALIGN(8)
	{
		. = ALIGN(8);
		__bss_start__ = .;
		*(.sbss*)
		*(.gnu.linkonce.sb.*)
		*(.bss .bss.*)
		*(.gnu.linkonce.b.*)
		*(COMMON)
		. = ALIGN(16);
		__bss_end__ = .;
	} >DLM AT>DLM

	.msg :
	{
		. = ALIGN(8);
		__share_msg_start__ = .;
		*(.share_msg .share_msg.*)
		. = ALIGN(16);
		__share_msg_end__ = .;
	} >SHARE_RAM AT> SHARE_RAM

	.heap (NOLOAD) : ALIGN(16)
	{
		. = ALIGN(16);
		PROVIDE( __heap_start = . );
		. += __HEAP_SIZE;
		. = ALIGN(16);
		PROVIDE( __heap_limit = . );
	} >DLM AT>DLM

	.stack (NOLOAD) : ALIGN(16)
	{
		. = ALIGN(16);
		PROVIDE( __stack_start = . );
		. += __STACK_SIZE;
		. = ALIGN(16);
		PROVIDE( __StackTop = . );
		PROVIDE( _sp = . );
	} >DLM AT>DLM

	.comment 0 : { *(.comment) }
	.debug_abbrev 0 : { *(.debug_abbrev) }
	.debug_aranges 0 : { *(.debug_aranges) }  
	.debug_frame 0 : { *(.debug_frame) }
	.debug_info 0 : { *(.debug_info) }
	.debug_line 0 : { *(.debug_line) }
	.debug_line_str 0 : { *(.debug_line_str) }
	.debug_loc 0 : { *(.debug_loc) }
	.debug_loclists 0 : { *(.debug_loclists) }
	.debug_macro 0 : { *(.debug_macro) }
	.debug_pubnames 0 : { *(.debug_pubnames) }
	.debug_pubtypes 0 : { *(.debug_pubtypes) }
	.debug_ranges 0 : { *(.debug_ranges) }
	.debug_rnglists 0 : { *(.debug_rnglists) }
	.debug_str 0 : { *(.debug_str) }
	.debug_str_offsets 0 : { *(.debug_str_offsets) }
	.riscv.attributes 0 : { *(.riscv.attributes) }
}
