/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdbool.h>

#include "sysctl.h"
#include "board_cfg.h"
#include "fdp_app_info.h"
#include "fdp_crc.h"
#include "fdp_err.h"
#include "fdp_storage_port.h"
#include "fdp_cache.h"

#if defined (MONITOR_PACKAGE)&& MONITOR_PACKAGE
volatile BOOTAPP_SHARE_BSS_T uint32_t pepare_cost = 0U;
volatile BOOTAPP_SHARE_BSS_T uint32_t package_cost = 0U;
volatile BOOTAPP_SHARE_BSS_T uint32_t ready_cost;
#endif
volatile BOOTAPP_SHARE_BSS_T uint32_t updateFlag;

int fdp_check_jtag_app(uintptr_t info_addr)
{
	volatile uint32_t *endmark = (volatile uint32_t *)&FLASH_ENDM1_ADDR;
	fdp_app_image_info_t *info = (fdp_app_image_info_t *)info_addr;

	if (!info)
		return -FDP_CHECKERR;

	/*Check KEY CRC, image CRC and image length  initial value */
	if ((info->key_check != 0xA5A5A5A5) ||
		(info->image_check != 0xA5A5A5A5)||
		(info->image_size != 0x0))
		return -FDP_CHECKERR;

	/* Check APP_START_MARK */
	if ((info->App_Start_Mark[0] != 0xAAAAAAAA)||
		(info->App_Start_Mark[1] != 0xBBBBBBBB)||
		(info->App_Start_Mark[2] != 0xCCCCCCCC)||
		(info->App_Start_Mark[3] != 0xDDDDDDDD))
		return -FDP_CHECKERR;

	/* check APP_END_MARK */
	if ((*(endmark + 0) != 0xAAAAAAAA)||
		(*(endmark + 1) != 0xBBBBBBBB)||
		(*(endmark + 2) != 0xCCCCCCCC)||
		(*(endmark + 3) != 0xDDDDDDDD))
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}
int fdp_check_image_info(uintptr_t info_addr)
{
	uint32_t crc32_val;

	fdp_app_image_info_t *info = (fdp_app_image_info_t *)info_addr;

	if (!info)
		return -FDP_CHECKERR;

	fdp_dcache_invaild(info_addr, 1);

	/* verify key value. */
	crc32_val = fdp_crc32((uint8_t *)&info->key, sizeof(info->key));
	if (crc32_val != info->key_check)
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}

int fdp_check_image_base_on_info(uintptr_t info_addr)
{
	volatile uint32_t crc32_val;
	fdp_app_image_info_t *info;

	if (fdp_check_image_info(info_addr) != FDP_SUCCESS)
		return -FDP_CHECKERR;

	info = (fdp_app_image_info_t *)info_addr;

	/* Check if the address is in the range of the flash memory. */
	if ((info->app_start_addr < BANK_MIN_ADDR) || (info->app_start_addr >= BANK_MAX_ADDR))
		return -FDP_CHECKERR;

	fdp_dcache_invaild((uintptr_t)(info->app_start_addr), info->image_size);
	crc32_val = fdp_crc32((uint8_t *)info->app_start_addr, info->image_size);

	if (crc32_val != info->image_check)
		return -FDP_CHECKERR;

	return FDP_SUCCESS;
}

bool fdp_is_entry_load_mode(void)
{
	if (updateFlag != FDP_BOOT_UPDATE_FLAG)
		return false;
	else
		return true;
}

void fdp_set_entry_load_mode(bool enable)
{
	if (enable)
		updateFlag = FDP_BOOT_UPDATE_FLAG;
	else
		updateFlag = FDP_BOOT_UPDATE_DISABLE;
}
