/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "device.h"
#include "log.h"
#include "board_cfg.h"
#include "app_info.h"
#include "hal_uart_transfer.h"
#include "hal_can_transfer.h"
#include "hal_spi_transfer.h"

/* Global variable */
volatile uint32_t Timer10msCNT = 0;
volatile uint32_t task1sFlag = 0;

volatile BOOTAPP_SHARE_BSS_T uint32_t updateFlag;

extern uint32_t __app_start_addr;
extern uint32_t jump_cmd_ready;
extern uint32_t __image_start_offset;
extern uint32_t FLASH_BOOT_ADDR;

#define FDP_BOOT_UPDATE_FLAG		0x9E8F6C2AU
#define FDP_BOOT_UPDATE_DISABLE		0x5A5A5A5AU

const APP_ENDM_T   uint32_t App_End_Mark[8] = {
		0xAAAAAAAA, 0xBBBBBBBB, 0xCCCCCCCC, 0xDDDDDDDD,
		0xAAAAAAAA, 0xBBBBBBBB, 0xCCCCCCCC, 0xDDDDDDDD,
};


FDP_APP_INFO fdp_app_image_info_t app0_info = {
	.app_start_addr = (uint32_t)&__app_start_addr,
	.offset = (uint32_t)&__image_start_offset,
	.image_check = 0xA5A5A5A5,
	.image_size = 0,
	.key = 0x9527,
	.key_check = 0xA5A5A5A5,
	.version = 1,
	.App_Start_Mark[0] = 0xAAAAAAAA,
	.App_Start_Mark[1] = 0xBBBBBBBB,
	.App_Start_Mark[2] = 0xCCCCCCCC,
	.App_Start_Mark[3] = 0xDDDDDDDD,
};

__INTERRUPT void APB_TIMER1_IRQ_HANDLER(void)
{
	Timer10msCNT ++;

	GPIO_togglePin(GPIO_PIN_LED1);

	if (Timer10msCNT >= 100)
	{
		task1sFlag = 1;
		Timer10msCNT = 0U;
	}

	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);

	__DSB();
}

int32_t Timer1_init(uint32_t period)
{
	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_init(CPUTIMER1_BASE, period -1U);

	Interrupt_register(INT_TIMER1, &APB_TIMER1_IRQ_HANDLER);
	Interrupt_enable(INT_TIMER1);

	return 0;
}

void fdp_set_entry_load_mode(bool enable)
{
	if (enable)
		updateFlag = FDP_BOOT_UPDATE_FLAG;
	else
		updateFlag = FDP_BOOT_UPDATE_DISABLE;
}

void fdp_jump_to_address(uintptr_t addr)
{
	asm volatile ("jr %0" :: "r"(addr));
}

void cUartPrint_init(uint32_t SCIBase, uint32_t baud)
{
    GPIO_setPinConfig(LOG_TX_PIN);
    GPIO_setPinConfig(LOG_RX_PIN);
    GPIO_setPadConfig(LOG_TX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
    GPIO_setPadConfig(LOG_RX_PIN_NUM, GPIO_PIN_TYPE_PULLUP);
	GPIO_setQualificationMode(LOG_TX_PIN_NUM, GPIO_QUAL_SYNC);
	GPIO_setQualificationMode(LOG_RX_PIN_NUM, GPIO_QUAL_SYNC);

	SCI_enableLoopback(SCIBase);

	SCI_disableFIFO(SCIBase);
	while (!SCI_getRxFIFOStatus(SCIBase) == SCI_FIFO_RX0) {
		SCI_readCharBlockingFIFO(SCIBase);
	}
	SCI_resetChannels(SCIBase);
	SCI_clearInterruptStatus(SCIBase, SCI_INT_RXFF | SCI_INT_TXFF | SCI_INT_TXRDY);

	SCI_setConfig(SCIBase, DEVICE_APBCLK_FREQ, baud,
				 (SCI_CONFIG_WLEN_8 | SCI_CONFIG_STOP_ONE | SCI_CONFIG_PAR_NONE));
	SCI_enableFIFO(SCIBase);
	SCI_setFIFOInterruptLevel(SCIBase, SCI_FIFO_TX0, SCI_FIFO_RX1);

	SCI_disableInterrupt(SCIBase, 0xff);
	SCI_disableLoopback(SCIBase);
}

void cInit_Leds(void)
{
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setPinConfig(GPIO_CFG_LED2);
    GPIO_setPadConfig(GPIO_PIN_LED1, GPIO_PIN_TYPE_STD);
    GPIO_setPadConfig(GPIO_PIN_LED2, GPIO_PIN_TYPE_STD);
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);
    GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);
    GPIO_writePin(GPIO_PIN_LED1, 0);
    GPIO_writePin(GPIO_PIN_LED2, 1);
}

void main(void)
{
	uint32_t tickCnt = 0;

	__disable_irq();

	Device_init();

    /* Initial Log SCI port  */
	cUartPrint_init(LOG_SCI_BASE, 115200);
	log_set_level(LOG_INFO);

	/* Initial Led */
	cInit_Leds();

	Uart_Init();

	can_ctrl_init();

	spi_init();

	Timer1_init(DEVICE_APBCLK_FREQ / 100);	//10ms

	__enable_irq();

	while (1) {

		if (jump_cmd_ready) {
			jump_cmd_ready = 0;
			__disable_irq();
			SCI_enableLoopback(UART_BASE_ADDRESS);
			CAN_initModule(CAN_BASE);
			fdp_set_entry_load_mode(true);
			fdp_jump_to_address((uintptr_t)&FLASH_BOOT_ADDR);
		}

		if (task1sFlag == 1U)
		{
			task1sFlag = 0;
			tickCnt += 100U;
			GPIO_togglePin(GPIO_PIN_LED2);
			log_info("ticks: %d\r\n", tickCnt);
		}
	}
}
