/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdlib.h>
#include "device.h"
#include "log.h"            //smp thread safe
#include "gs32_math.h"

#include "transform_functions.h"

#include "basic_math_functions.h"
#include "complex_math_functions.h"

#include "riscv_math_types.h"
#include "riscv_math_memory.h"

#include "none.h"
#include "utils.h"

#include "riscv_const_structs.h"
#include "riscv_common_tables.h"
#include "fast_math_functions.h"

#include "cfft_f32.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//#define LED_PIN             LED1_PIN

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

void report_fft_config_error(uint16_t fft_size) {
#ifdef DEBUG
	log_error("Error: Attempted to use unsupported or disabled FFT size: %u\n", fft_size);
	log_error("Please define INCLUDE_FFT_TABLES_%u to enable this size(16~4096).\n", fft_size);
#endif
}

// ȡӦ FFT С NMSIS ʵ
static const riscv_cfft_instance_f32* get_fft_instance(uint16_t fft_size) {
    switch (fft_size) {
		#if defined(INCLUDE_FFT_TABLES_16)
        	case 16:
            	return &riscv_cfft_sR_f32_len16;
		#endif
		#if defined(INCLUDE_FFT_TABLES_32)
        	case 32:
            	return &riscv_cfft_sR_f32_len32;
		#endif
		#if defined(INCLUDE_FFT_TABLES_64)
        	case 64:
            	return &riscv_cfft_sR_f32_len64;
		#endif
		#if defined(INCLUDE_FFT_TABLES_128)
        	case 128:
            	return &riscv_cfft_sR_f32_len128;
		#endif
		#if defined(INCLUDE_FFT_TABLES_256)
        	case 256:
            	return &riscv_cfft_sR_f32_len256;
		#endif
		#if defined(INCLUDE_FFT_TABLES_512)
        	case 512:
            	return &riscv_cfft_sR_f32_len512;
		#endif
		#if defined(INCLUDE_FFT_TABLES_1024)
        	case 1024:
            	return &riscv_cfft_sR_f32_len1024;
		#endif
		#if defined(INCLUDE_FFT_TABLES_2048)
        	case 2048:
            	return &riscv_cfft_sR_f32_len2048;
		#endif
		#if defined(INCLUDE_FFT_TABLES_4096)
        	case 4096:
            	return &riscv_cfft_sR_f32_len4096;
		#endif
        	default:
        		report_fft_config_error(fft_size); // 
        		return NULL; // ֵ֧ FFT С
    }
}

//  FFT ׶
static uint16_t calculate_fft_stages(uint16_t fft_size) {

    if (fft_size == 0) {
        return 0;
    }

    return 31 - __clz(fft_size);
}

// ʼ FFT ṹ壬ʹⲿṩĻ
CFFT_F32_STRUCT_Handle CFFT_f32_init(CFFT_F32_STRUCT *cfft_struct,
                                    uint16_t fft_size,
                                    float32_t *fft_in_buf) {
    if (cfft_struct == NULL || fft_in_buf == NULL ) {
        return NULL;
    }

    // ʼṹ
    cfft_struct->FFTSize = fft_size;
    cfft_struct->Stages = calculate_fft_stages(fft_size);
    cfft_struct->rvfft = get_fft_instance(fft_size);

    if (cfft_struct->rvfft == NULL) {
        return NULL; // ֵ֧ FFT С
    }

    // ûָ
    cfft_struct->InPtr = fft_in_buf;
    cfft_struct->OutPtr = fft_in_buf; // ʹͬһ
    cfft_struct->CurrentInPtr = fft_in_buf;

    return cfft_struct;
}


/*
 * InPtr is source data.
 * After calculation, results are stored in InPtr,
 * CurrentInPtr points to InPtr, CurrentOutPtr points to InPtr
 */
void CFFT_f32(CFFT_F32_STRUCT_Handle hndCFFT_F32) {
    if (hndCFFT_F32 == NULL || hndCFFT_F32->InPtr == NULL || hndCFFT_F32->rvfft == NULL) {
        return;
    }

    hndCFFT_F32->CurrentInPtr = hndCFFT_F32->InPtr;
    hndCFFT_F32->CurrentOutPtr = hndCFFT_F32->InPtr;

    //  NMSIS  FFT 
    // : FFT ʵ, ݻ, ifftFlag (0=FFT, 1=IFFT), bitReverseFlag (1=λת)
    riscv_cfft_f32(hndCFFT_F32->rvfft, hndCFFT_F32->InPtr, 0, 1);
}

// ִ Inverse FFT
void ICFFT_f32(CFFT_F32_STRUCT_Handle hndCFFT_F32) {
    if (hndCFFT_F32 == NULL || hndCFFT_F32->InPtr == NULL || hndCFFT_F32->rvfft == NULL) {
        return;
    }

    hndCFFT_F32->CurrentInPtr = hndCFFT_F32->InPtr;
    hndCFFT_F32->CurrentOutPtr = hndCFFT_F32->InPtr;

    //  NMSIS  FFT 
    // : FFT ʵ, ݻ, ifftFlag (0=FFT, 1=IFFT), bitReverseFlag (1=λת)
    riscv_cfft_f32(hndCFFT_F32->rvfft, hndCFFT_F32->InPtr, 1, 1);
}

//  FFT 
void CFFT_f32s_mag(CFFT_F32_STRUCT_Handle hndCFFT_F32) {
    if (hndCFFT_F32 == NULL || hndCFFT_F32->CurrentInPtr == NULL || hndCFFT_F32->CurrentOutPtr == NULL) {
        return;
    }

    hndCFFT_F32->CurrentInPtr = hndCFFT_F32->InPtr;
    hndCFFT_F32->CurrentOutPtr = hndCFFT_F32->OutPtr;

    //  NMSIS ķȼ㺯
    riscv_cmplx_mag_f32(hndCFFT_F32->CurrentInPtr, hndCFFT_F32->CurrentOutPtr, hndCFFT_F32->FFTSize);
}

//  FFT λ
void CFFT_f32_phase(CFFT_F32_STRUCT_Handle hndCFFT_F32) {
    if (hndCFFT_F32 == NULL || hndCFFT_F32->CurrentInPtr == NULL || hndCFFT_F32->CurrentOutPtr == NULL) {
        return;
    }

    hndCFFT_F32->CurrentInPtr = hndCFFT_F32->InPtr;

    // ֶλNMSIS ûֱṩλ㺯
    for (uint16_t i = 0; i < hndCFFT_F32->FFTSize; i++) {
        float32_t real = hndCFFT_F32->CurrentInPtr[2*i];
        float32_t imag = hndCFFT_F32->CurrentInPtr[2*i+1];
        hndCFFT_F32->CurrentOutPtr[i] = __atan2(imag, real);
    }
}

// Ԥתӱ NMSIS вҪΪѾԤ㣩
void CFFT_f32_sincostable(CFFT_F32_STRUCT_Handle cfft) {
    // NMSIS ѾԤתӣ˺ΪԶ
    // ҪԶתӼ
    (void)cfft; // δʹò
}


