/* ----------------------------------------------------------------------
 * Project:      NMSIS DSP Library
 * Title:        riscv_common_tables.c
 * Description:  common tables like fft twiddle factors, Bitreverse, reciprocal etc
 *
 * $Date:        23 April 2021
 * $Revision:    V1.9.0
 *
 * Target Processor: RISC-V Cores
 * -------------------------------------------------------------------- */
/*
 * Copyright (C) 2010-2021 ARM Limited or its affiliates. All rights reserved.
 * Copyright (c) 2019 Nuclei Limited. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include "riscv_math_types.h"
#include "riscv_common_tables.h"

/**
  @ingroup ComplexFFT
 */

/**
  @addtogroup CFFT_CIFFT Complex FFT Tables
  @{
 */


#if defined(INCLUDE_FFT_TABLES_16)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i < N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 16, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_16[32] = {
    1.000000000f,  0.000000000f,
    0.923879533f,  0.382683432f,
    0.707106781f,  0.707106781f,
    0.382683432f,  0.923879533f,
    0.000000000f,  1.000000000f,
   -0.382683432f,  0.923879533f,
   -0.707106781f,  0.707106781f,
   -0.923879533f,  0.382683432f,
   -1.000000000f,  0.000000000f,
   -0.923879533f, -0.382683432f,
   -0.707106781f, -0.707106781f,
   -0.382683432f, -0.923879533f,
   -0.000000000f, -1.000000000f,
    0.382683432f, -0.923879533f,
    0.707106781f, -0.707106781f,
    0.923879533f, -0.382683432f
};
#endif // INCLUDE_FFT_TABLES_16


#if defined(INCLUDE_FFT_TABLES_32)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 32, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_32[64] = {
    1.000000000f,  0.000000000f,
    0.980785280f,  0.195090322f,
    0.923879533f,  0.382683432f,
    0.831469612f,  0.555570233f,
    0.707106781f,  0.707106781f,
    0.555570233f,  0.831469612f,
    0.382683432f,  0.923879533f,
    0.195090322f,  0.980785280f,
    0.000000000f,  1.000000000f,
   -0.195090322f,  0.980785280f,
   -0.382683432f,  0.923879533f,
   -0.555570233f,  0.831469612f,
   -0.707106781f,  0.707106781f,
   -0.831469612f,  0.555570233f,
   -0.923879533f,  0.382683432f,
   -0.980785280f,  0.195090322f,
   -1.000000000f,  0.000000000f,
   -0.980785280f, -0.195090322f,
   -0.923879533f, -0.382683432f,
   -0.831469612f, -0.555570233f,
   -0.707106781f, -0.707106781f,
   -0.555570233f, -0.831469612f,
   -0.382683432f, -0.923879533f,
   -0.195090322f, -0.980785280f,
   -0.000000000f, -1.000000000f,
    0.195090322f, -0.980785280f,
    0.382683432f, -0.923879533f,
    0.555570233f, -0.831469612f,
    0.707106781f, -0.707106781f,
    0.831469612f, -0.555570233f,
    0.923879533f, -0.382683432f,
    0.980785280f, -0.195090322f
};
#endif // INCLUDE_FFT_TABLES_32


#if defined(INCLUDE_FFT_TABLES_64)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for(i = 0; i < N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 64, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_64[128] = {
    1.000000000f,  0.000000000f,
    0.995184727f,  0.098017140f,
    0.980785280f,  0.195090322f,
    0.956940336f,  0.290284677f,
    0.923879533f,  0.382683432f,
    0.881921264f,  0.471396737f,
    0.831469612f,  0.555570233f,
    0.773010453f,  0.634393284f,
    0.707106781f,  0.707106781f,
    0.634393284f,  0.773010453f,
    0.555570233f,  0.831469612f,
    0.471396737f,  0.881921264f,
    0.382683432f,  0.923879533f,
    0.290284677f,  0.956940336f,
    0.195090322f,  0.980785280f,
    0.098017140f,  0.995184727f,
    0.000000000f,  1.000000000f,
   -0.098017140f,  0.995184727f,
   -0.195090322f,  0.980785280f,
   -0.290284677f,  0.956940336f,
   -0.382683432f,  0.923879533f,
   -0.471396737f,  0.881921264f,
   -0.555570233f,  0.831469612f,
   -0.634393284f,  0.773010453f,
   -0.707106781f,  0.707106781f,
   -0.773010453f,  0.634393284f,
   -0.831469612f,  0.555570233f,
   -0.881921264f,  0.471396737f,
   -0.923879533f,  0.382683432f,
   -0.956940336f,  0.290284677f,
   -0.980785280f,  0.195090322f,
   -0.995184727f,  0.098017140f,
   -1.000000000f,  0.000000000f,
   -0.995184727f, -0.098017140f,
   -0.980785280f, -0.195090322f,
   -0.956940336f, -0.290284677f,
   -0.923879533f, -0.382683432f,
   -0.881921264f, -0.471396737f,
   -0.831469612f, -0.555570233f,
   -0.773010453f, -0.634393284f,
   -0.707106781f, -0.707106781f,
   -0.634393284f, -0.773010453f,
   -0.555570233f, -0.831469612f,
   -0.471396737f, -0.881921264f,
   -0.382683432f, -0.923879533f,
   -0.290284677f, -0.956940336f,
   -0.195090322f, -0.980785280f,
   -0.098017140f, -0.995184727f,
   -0.000000000f, -1.000000000f,
    0.098017140f, -0.995184727f,
    0.195090322f, -0.980785280f,
    0.290284677f, -0.956940336f,
    0.382683432f, -0.923879533f,
    0.471396737f, -0.881921264f,
    0.555570233f, -0.831469612f,
    0.634393284f, -0.773010453f,
    0.707106781f, -0.707106781f,
    0.773010453f, -0.634393284f,
    0.831469612f, -0.555570233f,
    0.881921264f, -0.471396737f,
    0.923879533f, -0.382683432f,
    0.956940336f, -0.290284677f,
    0.980785280f, -0.195090322f,
    0.995184727f, -0.098017140f
};
#endif // INCLUDE_FFT_TABLES_64


#if defined(INCLUDE_FFT_TABLES_128)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 128, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_128[256] = {
    1.000000000f,  0.000000000f,
    0.998795456f,  0.049067674f,
    0.995184727f,  0.098017140f,
    0.989176510f,  0.146730474f,
    0.980785280f,  0.195090322f,
    0.970031253f,  0.242980180f,
    0.956940336f,  0.290284677f,
    0.941544065f,  0.336889853f,
    0.923879533f,  0.382683432f,
    0.903989293f,  0.427555093f,
    0.881921264f,  0.471396737f,
    0.857728610f,  0.514102744f,
    0.831469612f,  0.555570233f,
    0.803207531f,  0.595699304f,
    0.773010453f,  0.634393284f,
    0.740951125f,  0.671558955f,
    0.707106781f,  0.707106781f,
    0.671558955f,  0.740951125f,
    0.634393284f,  0.773010453f,
    0.595699304f,  0.803207531f,
    0.555570233f,  0.831469612f,
    0.514102744f,  0.857728610f,
    0.471396737f,  0.881921264f,
    0.427555093f,  0.903989293f,
    0.382683432f,  0.923879533f,
    0.336889853f,  0.941544065f,
    0.290284677f,  0.956940336f,
    0.242980180f,  0.970031253f,
    0.195090322f,  0.980785280f,
    0.146730474f,  0.989176510f,
    0.098017140f,  0.995184727f,
    0.049067674f,  0.998795456f,
    0.000000000f,  1.000000000f,
   -0.049067674f,  0.998795456f,
   -0.098017140f,  0.995184727f,
   -0.146730474f,  0.989176510f,
   -0.195090322f,  0.980785280f,
   -0.242980180f,  0.970031253f,
   -0.290284677f,  0.956940336f,
   -0.336889853f,  0.941544065f,
   -0.382683432f,  0.923879533f,
   -0.427555093f,  0.903989293f,
   -0.471396737f,  0.881921264f,
   -0.514102744f,  0.857728610f,
   -0.555570233f,  0.831469612f,
   -0.595699304f,  0.803207531f,
   -0.634393284f,  0.773010453f,
   -0.671558955f,  0.740951125f,
   -0.707106781f,  0.707106781f,
   -0.740951125f,  0.671558955f,
   -0.773010453f,  0.634393284f,
   -0.803207531f,  0.595699304f,
   -0.831469612f,  0.555570233f,
   -0.857728610f,  0.514102744f,
   -0.881921264f,  0.471396737f,
   -0.903989293f,  0.427555093f,
   -0.923879533f,  0.382683432f,
   -0.941544065f,  0.336889853f,
   -0.956940336f,  0.290284677f,
   -0.970031253f,  0.242980180f,
   -0.980785280f,  0.195090322f,
   -0.989176510f,  0.146730474f,
   -0.995184727f,  0.098017140f,
   -0.998795456f,  0.049067674f,
   -1.000000000f,  0.000000000f,
   -0.998795456f, -0.049067674f,
   -0.995184727f, -0.098017140f,
   -0.989176510f, -0.146730474f,
   -0.980785280f, -0.195090322f,
   -0.970031253f, -0.242980180f,
   -0.956940336f, -0.290284677f,
   -0.941544065f, -0.336889853f,
   -0.923879533f, -0.382683432f,
   -0.903989293f, -0.427555093f,
   -0.881921264f, -0.471396737f,
   -0.857728610f, -0.514102744f,
   -0.831469612f, -0.555570233f,
   -0.803207531f, -0.595699304f,
   -0.773010453f, -0.634393284f,
   -0.740951125f, -0.671558955f,
   -0.707106781f, -0.707106781f,
   -0.671558955f, -0.740951125f,
   -0.634393284f, -0.773010453f,
   -0.595699304f, -0.803207531f,
   -0.555570233f, -0.831469612f,
   -0.514102744f, -0.857728610f,
   -0.471396737f, -0.881921264f,
   -0.427555093f, -0.903989293f,
   -0.382683432f, -0.923879533f,
   -0.336889853f, -0.941544065f,
   -0.290284677f, -0.956940336f,
   -0.242980180f, -0.970031253f,
   -0.195090322f, -0.980785280f,
   -0.146730474f, -0.989176510f,
   -0.098017140f, -0.995184727f,
   -0.049067674f, -0.998795456f,
   -0.000000000f, -1.000000000f,
    0.049067674f, -0.998795456f,
    0.098017140f, -0.995184727f,
    0.146730474f, -0.989176510f,
    0.195090322f, -0.980785280f,
    0.242980180f, -0.970031253f,
    0.290284677f, -0.956940336f,
    0.336889853f, -0.941544065f,
    0.382683432f, -0.923879533f,
    0.427555093f, -0.903989293f,
    0.471396737f, -0.881921264f,
    0.514102744f, -0.857728610f,
    0.555570233f, -0.831469612f,
    0.595699304f, -0.803207531f,
    0.634393284f, -0.773010453f,
    0.671558955f, -0.740951125f,
    0.707106781f, -0.707106781f,
    0.740951125f, -0.671558955f,
    0.773010453f, -0.634393284f,
    0.803207531f, -0.595699304f,
    0.831469612f, -0.555570233f,
    0.857728610f, -0.514102744f,
    0.881921264f, -0.471396737f,
    0.903989293f, -0.427555093f,
    0.923879533f, -0.382683432f,
    0.941544065f, -0.336889853f,
    0.956940336f, -0.290284677f,
    0.970031253f, -0.242980180f,
    0.980785280f, -0.195090322f,
    0.989176510f, -0.146730474f,
    0.995184727f, -0.098017140f,
    0.998795456f, -0.049067674f
};
#endif // INCLUDE_FFT_TABLES_128


#if defined(INCLUDE_FFT_TABLES_256)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for(i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 256, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_256[512] = {
    1.000000000f,  0.000000000f,
    0.999698819f,  0.024541229f,
    0.998795456f,  0.049067674f,
    0.997290457f,  0.073564564f,
    0.995184727f,  0.098017140f,
    0.992479535f,  0.122410675f,
    0.989176510f,  0.146730474f,
    0.985277642f,  0.170961889f,
    0.980785280f,  0.195090322f,
    0.975702130f,  0.219101240f,
    0.970031253f,  0.242980180f,
    0.963776066f,  0.266712757f,
    0.956940336f,  0.290284677f,
    0.949528181f,  0.313681740f,
    0.941544065f,  0.336889853f,
    0.932992799f,  0.359895037f,
    0.923879533f,  0.382683432f,
    0.914209756f,  0.405241314f,
    0.903989293f,  0.427555093f,
    0.893224301f,  0.449611330f,
    0.881921264f,  0.471396737f,
    0.870086991f,  0.492898192f,
    0.857728610f,  0.514102744f,
    0.844853565f,  0.534997620f,
    0.831469612f,  0.555570233f,
    0.817584813f,  0.575808191f,
    0.803207531f,  0.595699304f,
    0.788346428f,  0.615231591f,
    0.773010453f,  0.634393284f,
    0.757208847f,  0.653172843f,
    0.740951125f,  0.671558955f,
    0.724247083f,  0.689540545f,
    0.707106781f,  0.707106781f,
    0.689540545f,  0.724247083f,
    0.671558955f,  0.740951125f,
    0.653172843f,  0.757208847f,
    0.634393284f,  0.773010453f,
    0.615231591f,  0.788346428f,
    0.595699304f,  0.803207531f,
    0.575808191f,  0.817584813f,
    0.555570233f,  0.831469612f,
    0.534997620f,  0.844853565f,
    0.514102744f,  0.857728610f,
    0.492898192f,  0.870086991f,
    0.471396737f,  0.881921264f,
    0.449611330f,  0.893224301f,
    0.427555093f,  0.903989293f,
    0.405241314f,  0.914209756f,
    0.382683432f,  0.923879533f,
    0.359895037f,  0.932992799f,
    0.336889853f,  0.941544065f,
    0.313681740f,  0.949528181f,
    0.290284677f,  0.956940336f,
    0.266712757f,  0.963776066f,
    0.242980180f,  0.970031253f,
    0.219101240f,  0.975702130f,
    0.195090322f,  0.980785280f,
    0.170961889f,  0.985277642f,
    0.146730474f,  0.989176510f,
    0.122410675f,  0.992479535f,
    0.098017140f,  0.995184727f,
    0.073564564f,  0.997290457f,
    0.049067674f,  0.998795456f,
    0.024541229f,  0.999698819f,
    0.000000000f,  1.000000000f,
   -0.024541229f,  0.999698819f,
   -0.049067674f,  0.998795456f,
   -0.073564564f,  0.997290457f,
   -0.098017140f,  0.995184727f,
   -0.122410675f,  0.992479535f,
   -0.146730474f,  0.989176510f,
   -0.170961889f,  0.985277642f,
   -0.195090322f,  0.980785280f,
   -0.219101240f,  0.975702130f,
   -0.242980180f,  0.970031253f,
   -0.266712757f,  0.963776066f,
   -0.290284677f,  0.956940336f,
   -0.313681740f,  0.949528181f,
   -0.336889853f,  0.941544065f,
   -0.359895037f,  0.932992799f,
   -0.382683432f,  0.923879533f,
   -0.405241314f,  0.914209756f,
   -0.427555093f,  0.903989293f,
   -0.449611330f,  0.893224301f,
   -0.471396737f,  0.881921264f,
   -0.492898192f,  0.870086991f,
   -0.514102744f,  0.857728610f,
   -0.534997620f,  0.844853565f,
   -0.555570233f,  0.831469612f,
   -0.575808191f,  0.817584813f,
   -0.595699304f,  0.803207531f,
   -0.615231591f,  0.788346428f,
   -0.634393284f,  0.773010453f,
   -0.653172843f,  0.757208847f,
   -0.671558955f,  0.740951125f,
   -0.689540545f,  0.724247083f,
   -0.707106781f,  0.707106781f,
   -0.724247083f,  0.689540545f,
   -0.740951125f,  0.671558955f,
   -0.757208847f,  0.653172843f,
   -0.773010453f,  0.634393284f,
   -0.788346428f,  0.615231591f,
   -0.803207531f,  0.595699304f,
   -0.817584813f,  0.575808191f,
   -0.831469612f,  0.555570233f,
   -0.844853565f,  0.534997620f,
   -0.857728610f,  0.514102744f,
   -0.870086991f,  0.492898192f,
   -0.881921264f,  0.471396737f,
   -0.893224301f,  0.449611330f,
   -0.903989293f,  0.427555093f,
   -0.914209756f,  0.405241314f,
   -0.923879533f,  0.382683432f,
   -0.932992799f,  0.359895037f,
   -0.941544065f,  0.336889853f,
   -0.949528181f,  0.313681740f,
   -0.956940336f,  0.290284677f,
   -0.963776066f,  0.266712757f,
   -0.970031253f,  0.242980180f,
   -0.975702130f,  0.219101240f,
   -0.980785280f,  0.195090322f,
   -0.985277642f,  0.170961889f,
   -0.989176510f,  0.146730474f,
   -0.992479535f,  0.122410675f,
   -0.995184727f,  0.098017140f,
   -0.997290457f,  0.073564564f,
   -0.998795456f,  0.049067674f,
   -0.999698819f,  0.024541229f,
   -1.000000000f,  0.000000000f,
   -0.999698819f, -0.024541229f,
   -0.998795456f, -0.049067674f,
   -0.997290457f, -0.073564564f,
   -0.995184727f, -0.098017140f,
   -0.992479535f, -0.122410675f,
   -0.989176510f, -0.146730474f,
   -0.985277642f, -0.170961889f,
   -0.980785280f, -0.195090322f,
   -0.975702130f, -0.219101240f,
   -0.970031253f, -0.242980180f,
   -0.963776066f, -0.266712757f,
   -0.956940336f, -0.290284677f,
   -0.949528181f, -0.313681740f,
   -0.941544065f, -0.336889853f,
   -0.932992799f, -0.359895037f,
   -0.923879533f, -0.382683432f,
   -0.914209756f, -0.405241314f,
   -0.903989293f, -0.427555093f,
   -0.893224301f, -0.449611330f,
   -0.881921264f, -0.471396737f,
   -0.870086991f, -0.492898192f,
   -0.857728610f, -0.514102744f,
   -0.844853565f, -0.534997620f,
   -0.831469612f, -0.555570233f,
   -0.817584813f, -0.575808191f,
   -0.803207531f, -0.595699304f,
   -0.788346428f, -0.615231591f,
   -0.773010453f, -0.634393284f,
   -0.757208847f, -0.653172843f,
   -0.740951125f, -0.671558955f,
   -0.724247083f, -0.689540545f,
   -0.707106781f, -0.707106781f,
   -0.689540545f, -0.724247083f,
   -0.671558955f, -0.740951125f,
   -0.653172843f, -0.757208847f,
   -0.634393284f, -0.773010453f,
   -0.615231591f, -0.788346428f,
   -0.595699304f, -0.803207531f,
   -0.575808191f, -0.817584813f,
   -0.555570233f, -0.831469612f,
   -0.534997620f, -0.844853565f,
   -0.514102744f, -0.857728610f,
   -0.492898192f, -0.870086991f,
   -0.471396737f, -0.881921264f,
   -0.449611330f, -0.893224301f,
   -0.427555093f, -0.903989293f,
   -0.405241314f, -0.914209756f,
   -0.382683432f, -0.923879533f,
   -0.359895037f, -0.932992799f,
   -0.336889853f, -0.941544065f,
   -0.313681740f, -0.949528181f,
   -0.290284677f, -0.956940336f,
   -0.266712757f, -0.963776066f,
   -0.242980180f, -0.970031253f,
   -0.219101240f, -0.975702130f,
   -0.195090322f, -0.980785280f,
   -0.170961889f, -0.985277642f,
   -0.146730474f, -0.989176510f,
   -0.122410675f, -0.992479535f,
   -0.098017140f, -0.995184727f,
   -0.073564564f, -0.997290457f,
   -0.049067674f, -0.998795456f,
   -0.024541229f, -0.999698819f,
   -0.000000000f, -1.000000000f,
    0.024541229f, -0.999698819f,
    0.049067674f, -0.998795456f,
    0.073564564f, -0.997290457f,
    0.098017140f, -0.995184727f,
    0.122410675f, -0.992479535f,
    0.146730474f, -0.989176510f,
    0.170961889f, -0.985277642f,
    0.195090322f, -0.980785280f,
    0.219101240f, -0.975702130f,
    0.242980180f, -0.970031253f,
    0.266712757f, -0.963776066f,
    0.290284677f, -0.956940336f,
    0.313681740f, -0.949528181f,
    0.336889853f, -0.941544065f,
    0.359895037f, -0.932992799f,
    0.382683432f, -0.923879533f,
    0.405241314f, -0.914209756f,
    0.427555093f, -0.903989293f,
    0.449611330f, -0.893224301f,
    0.471396737f, -0.881921264f,
    0.492898192f, -0.870086991f,
    0.514102744f, -0.857728610f,
    0.534997620f, -0.844853565f,
    0.555570233f, -0.831469612f,
    0.575808191f, -0.817584813f,
    0.595699304f, -0.803207531f,
    0.615231591f, -0.788346428f,
    0.634393284f, -0.773010453f,
    0.653172843f, -0.757208847f,
    0.671558955f, -0.740951125f,
    0.689540545f, -0.724247083f,
    0.707106781f, -0.707106781f,
    0.724247083f, -0.689540545f,
    0.740951125f, -0.671558955f,
    0.757208847f, -0.653172843f,
    0.773010453f, -0.634393284f,
    0.788346428f, -0.615231591f,
    0.803207531f, -0.595699304f,
    0.817584813f, -0.575808191f,
    0.831469612f, -0.555570233f,
    0.844853565f, -0.534997620f,
    0.857728610f, -0.514102744f,
    0.870086991f, -0.492898192f,
    0.881921264f, -0.471396737f,
    0.893224301f, -0.449611330f,
    0.903989293f, -0.427555093f,
    0.914209756f, -0.405241314f,
    0.923879533f, -0.382683432f,
    0.932992799f, -0.359895037f,
    0.941544065f, -0.336889853f,
    0.949528181f, -0.313681740f,
    0.956940336f, -0.290284677f,
    0.963776066f, -0.266712757f,
    0.970031253f, -0.242980180f,
    0.975702130f, -0.219101240f,
    0.980785280f, -0.195090322f,
    0.985277642f, -0.170961889f,
    0.989176510f, -0.146730474f,
    0.992479535f, -0.122410675f,
    0.995184727f, -0.098017140f,
    0.997290457f, -0.073564564f,
    0.998795456f, -0.049067674f,
    0.999698819f, -0.024541229f
};
#endif // INCLUDE_FFT_TABLES_256


#if defined(INCLUDE_FFT_TABLES_512)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 512, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_512[1024] = {
    1.000000000f,  0.000000000f,
    0.999924702f,  0.012271538f,
    0.999698819f,  0.024541229f,
    0.999322385f,  0.036807223f,
    0.998795456f,  0.049067674f,
    0.998118113f,  0.061320736f,
    0.997290457f,  0.073564564f,
    0.996312612f,  0.085797312f,
    0.995184727f,  0.098017140f,
    0.993906970f,  0.110222207f,
    0.992479535f,  0.122410675f,
    0.990902635f,  0.134580709f,
    0.989176510f,  0.146730474f,
    0.987301418f,  0.158858143f,
    0.985277642f,  0.170961889f,
    0.983105487f,  0.183039888f,
    0.980785280f,  0.195090322f,
    0.978317371f,  0.207111376f,
    0.975702130f,  0.219101240f,
    0.972939952f,  0.231058108f,
    0.970031253f,  0.242980180f,
    0.966976471f,  0.254865660f,
    0.963776066f,  0.266712757f,
    0.960430519f,  0.278519689f,
    0.956940336f,  0.290284677f,
    0.953306040f,  0.302005949f,
    0.949528181f,  0.313681740f,
    0.945607325f,  0.325310292f,
    0.941544065f,  0.336889853f,
    0.937339012f,  0.348418680f,
    0.932992799f,  0.359895037f,
    0.928506080f,  0.371317194f,
    0.923879533f,  0.382683432f,
    0.919113852f,  0.393992040f,
    0.914209756f,  0.405241314f,
    0.909167983f,  0.416429560f,
    0.903989293f,  0.427555093f,
    0.898674466f,  0.438616239f,
    0.893224301f,  0.449611330f,
    0.887639620f,  0.460538711f,
    0.881921264f,  0.471396737f,
    0.876070094f,  0.482183772f,
    0.870086991f,  0.492898192f,
    0.863972856f,  0.503538384f,
    0.857728610f,  0.514102744f,
    0.851355193f,  0.524589683f,
    0.844853565f,  0.534997620f,
    0.838224706f,  0.545324988f,
    0.831469612f,  0.555570233f,
    0.824589303f,  0.565731811f,
    0.817584813f,  0.575808191f,
    0.810457198f,  0.585797857f,
    0.803207531f,  0.595699304f,
    0.795836905f,  0.605511041f,
    0.788346428f,  0.615231591f,
    0.780737229f,  0.624859488f,
    0.773010453f,  0.634393284f,
    0.765167266f,  0.643831543f,
    0.757208847f,  0.653172843f,
    0.749136395f,  0.662415778f,
    0.740951125f,  0.671558955f,
    0.732654272f,  0.680600998f,
    0.724247083f,  0.689540545f,
    0.715730825f,  0.698376249f,
    0.707106781f,  0.707106781f,
    0.698376249f,  0.715730825f,
    0.689540545f,  0.724247083f,
    0.680600998f,  0.732654272f,
    0.671558955f,  0.740951125f,
    0.662415778f,  0.749136395f,
    0.653172843f,  0.757208847f,
    0.643831543f,  0.765167266f,
    0.634393284f,  0.773010453f,
    0.624859488f,  0.780737229f,
    0.615231591f,  0.788346428f,
    0.605511041f,  0.795836905f,
    0.595699304f,  0.803207531f,
    0.585797857f,  0.810457198f,
    0.575808191f,  0.817584813f,
    0.565731811f,  0.824589303f,
    0.555570233f,  0.831469612f,
    0.545324988f,  0.838224706f,
    0.534997620f,  0.844853565f,
    0.524589683f,  0.851355193f,
    0.514102744f,  0.857728610f,
    0.503538384f,  0.863972856f,
    0.492898192f,  0.870086991f,
    0.482183772f,  0.876070094f,
    0.471396737f,  0.881921264f,
    0.460538711f,  0.887639620f,
    0.449611330f,  0.893224301f,
    0.438616239f,  0.898674466f,
    0.427555093f,  0.903989293f,
    0.416429560f,  0.909167983f,
    0.405241314f,  0.914209756f,
    0.393992040f,  0.919113852f,
    0.382683432f,  0.923879533f,
    0.371317194f,  0.928506080f,
    0.359895037f,  0.932992799f,
    0.348418680f,  0.937339012f,
    0.336889853f,  0.941544065f,
    0.325310292f,  0.945607325f,
    0.313681740f,  0.949528181f,
    0.302005949f,  0.953306040f,
    0.290284677f,  0.956940336f,
    0.278519689f,  0.960430519f,
    0.266712757f,  0.963776066f,
    0.254865660f,  0.966976471f,
    0.242980180f,  0.970031253f,
    0.231058108f,  0.972939952f,
    0.219101240f,  0.975702130f,
    0.207111376f,  0.978317371f,
    0.195090322f,  0.980785280f,
    0.183039888f,  0.983105487f,
    0.170961889f,  0.985277642f,
    0.158858143f,  0.987301418f,
    0.146730474f,  0.989176510f,
    0.134580709f,  0.990902635f,
    0.122410675f,  0.992479535f,
    0.110222207f,  0.993906970f,
    0.098017140f,  0.995184727f,
    0.085797312f,  0.996312612f,
    0.073564564f,  0.997290457f,
    0.061320736f,  0.998118113f,
    0.049067674f,  0.998795456f,
    0.036807223f,  0.999322385f,
    0.024541229f,  0.999698819f,
    0.012271538f,  0.999924702f,
    0.000000000f,  1.000000000f,
   -0.012271538f,  0.999924702f,
   -0.024541229f,  0.999698819f,
   -0.036807223f,  0.999322385f,
   -0.049067674f,  0.998795456f,
   -0.061320736f,  0.998118113f,
   -0.073564564f,  0.997290457f,
   -0.085797312f,  0.996312612f,
   -0.098017140f,  0.995184727f,
   -0.110222207f,  0.993906970f,
   -0.122410675f,  0.992479535f,
   -0.134580709f,  0.990902635f,
   -0.146730474f,  0.989176510f,
   -0.158858143f,  0.987301418f,
   -0.170961889f,  0.985277642f,
   -0.183039888f,  0.983105487f,
   -0.195090322f,  0.980785280f,
   -0.207111376f,  0.978317371f,
   -0.219101240f,  0.975702130f,
   -0.231058108f,  0.972939952f,
   -0.242980180f,  0.970031253f,
   -0.254865660f,  0.966976471f,
   -0.266712757f,  0.963776066f,
   -0.278519689f,  0.960430519f,
   -0.290284677f,  0.956940336f,
   -0.302005949f,  0.953306040f,
   -0.313681740f,  0.949528181f,
   -0.325310292f,  0.945607325f,
   -0.336889853f,  0.941544065f,
   -0.348418680f,  0.937339012f,
   -0.359895037f,  0.932992799f,
   -0.371317194f,  0.928506080f,
   -0.382683432f,  0.923879533f,
   -0.393992040f,  0.919113852f,
   -0.405241314f,  0.914209756f,
   -0.416429560f,  0.909167983f,
   -0.427555093f,  0.903989293f,
   -0.438616239f,  0.898674466f,
   -0.449611330f,  0.893224301f,
   -0.460538711f,  0.887639620f,
   -0.471396737f,  0.881921264f,
   -0.482183772f,  0.876070094f,
   -0.492898192f,  0.870086991f,
   -0.503538384f,  0.863972856f,
   -0.514102744f,  0.857728610f,
   -0.524589683f,  0.851355193f,
   -0.534997620f,  0.844853565f,
   -0.545324988f,  0.838224706f,
   -0.555570233f,  0.831469612f,
   -0.565731811f,  0.824589303f,
   -0.575808191f,  0.817584813f,
   -0.585797857f,  0.810457198f,
   -0.595699304f,  0.803207531f,
   -0.605511041f,  0.795836905f,
   -0.615231591f,  0.788346428f,
   -0.624859488f,  0.780737229f,
   -0.634393284f,  0.773010453f,
   -0.643831543f,  0.765167266f,
   -0.653172843f,  0.757208847f,
   -0.662415778f,  0.749136395f,
   -0.671558955f,  0.740951125f,
   -0.680600998f,  0.732654272f,
   -0.689540545f,  0.724247083f,
   -0.698376249f,  0.715730825f,
   -0.707106781f,  0.707106781f,
   -0.715730825f,  0.698376249f,
   -0.724247083f,  0.689540545f,
   -0.732654272f,  0.680600998f,
   -0.740951125f,  0.671558955f,
   -0.749136395f,  0.662415778f,
   -0.757208847f,  0.653172843f,
   -0.765167266f,  0.643831543f,
   -0.773010453f,  0.634393284f,
   -0.780737229f,  0.624859488f,
   -0.788346428f,  0.615231591f,
   -0.795836905f,  0.605511041f,
   -0.803207531f,  0.595699304f,
   -0.810457198f,  0.585797857f,
   -0.817584813f,  0.575808191f,
   -0.824589303f,  0.565731811f,
   -0.831469612f,  0.555570233f,
   -0.838224706f,  0.545324988f,
   -0.844853565f,  0.534997620f,
   -0.851355193f,  0.524589683f,
   -0.857728610f,  0.514102744f,
   -0.863972856f,  0.503538384f,
   -0.870086991f,  0.492898192f,
   -0.876070094f,  0.482183772f,
   -0.881921264f,  0.471396737f,
   -0.887639620f,  0.460538711f,
   -0.893224301f,  0.449611330f,
   -0.898674466f,  0.438616239f,
   -0.903989293f,  0.427555093f,
   -0.909167983f,  0.416429560f,
   -0.914209756f,  0.405241314f,
   -0.919113852f,  0.393992040f,
   -0.923879533f,  0.382683432f,
   -0.928506080f,  0.371317194f,
   -0.932992799f,  0.359895037f,
   -0.937339012f,  0.348418680f,
   -0.941544065f,  0.336889853f,
   -0.945607325f,  0.325310292f,
   -0.949528181f,  0.313681740f,
   -0.953306040f,  0.302005949f,
   -0.956940336f,  0.290284677f,
   -0.960430519f,  0.278519689f,
   -0.963776066f,  0.266712757f,
   -0.966976471f,  0.254865660f,
   -0.970031253f,  0.242980180f,
   -0.972939952f,  0.231058108f,
   -0.975702130f,  0.219101240f,
   -0.978317371f,  0.207111376f,
   -0.980785280f,  0.195090322f,
   -0.983105487f,  0.183039888f,
   -0.985277642f,  0.170961889f,
   -0.987301418f,  0.158858143f,
   -0.989176510f,  0.146730474f,
   -0.990902635f,  0.134580709f,
   -0.992479535f,  0.122410675f,
   -0.993906970f,  0.110222207f,
   -0.995184727f,  0.098017140f,
   -0.996312612f,  0.085797312f,
   -0.997290457f,  0.073564564f,
   -0.998118113f,  0.061320736f,
   -0.998795456f,  0.049067674f,
   -0.999322385f,  0.036807223f,
   -0.999698819f,  0.024541229f,
   -0.999924702f,  0.012271538f,
   -1.000000000f,  0.000000000f,
   -0.999924702f, -0.012271538f,
   -0.999698819f, -0.024541229f,
   -0.999322385f, -0.036807223f,
   -0.998795456f, -0.049067674f,
   -0.998118113f, -0.061320736f,
   -0.997290457f, -0.073564564f,
   -0.996312612f, -0.085797312f,
   -0.995184727f, -0.098017140f,
   -0.993906970f, -0.110222207f,
   -0.992479535f, -0.122410675f,
   -0.990902635f, -0.134580709f,
   -0.989176510f, -0.146730474f,
   -0.987301418f, -0.158858143f,
   -0.985277642f, -0.170961889f,
   -0.983105487f, -0.183039888f,
   -0.980785280f, -0.195090322f,
   -0.978317371f, -0.207111376f,
   -0.975702130f, -0.219101240f,
   -0.972939952f, -0.231058108f,
   -0.970031253f, -0.242980180f,
   -0.966976471f, -0.254865660f,
   -0.963776066f, -0.266712757f,
   -0.960430519f, -0.278519689f,
   -0.956940336f, -0.290284677f,
   -0.953306040f, -0.302005949f,
   -0.949528181f, -0.313681740f,
   -0.945607325f, -0.325310292f,
   -0.941544065f, -0.336889853f,
   -0.937339012f, -0.348418680f,
   -0.932992799f, -0.359895037f,
   -0.928506080f, -0.371317194f,
   -0.923879533f, -0.382683432f,
   -0.919113852f, -0.393992040f,
   -0.914209756f, -0.405241314f,
   -0.909167983f, -0.416429560f,
   -0.903989293f, -0.427555093f,
   -0.898674466f, -0.438616239f,
   -0.893224301f, -0.449611330f,
   -0.887639620f, -0.460538711f,
   -0.881921264f, -0.471396737f,
   -0.876070094f, -0.482183772f,
   -0.870086991f, -0.492898192f,
   -0.863972856f, -0.503538384f,
   -0.857728610f, -0.514102744f,
   -0.851355193f, -0.524589683f,
   -0.844853565f, -0.534997620f,
   -0.838224706f, -0.545324988f,
   -0.831469612f, -0.555570233f,
   -0.824589303f, -0.565731811f,
   -0.817584813f, -0.575808191f,
   -0.810457198f, -0.585797857f,
   -0.803207531f, -0.595699304f,
   -0.795836905f, -0.605511041f,
   -0.788346428f, -0.615231591f,
   -0.780737229f, -0.624859488f,
   -0.773010453f, -0.634393284f,
   -0.765167266f, -0.643831543f,
   -0.757208847f, -0.653172843f,
   -0.749136395f, -0.662415778f,
   -0.740951125f, -0.671558955f,
   -0.732654272f, -0.680600998f,
   -0.724247083f, -0.689540545f,
   -0.715730825f, -0.698376249f,
   -0.707106781f, -0.707106781f,
   -0.698376249f, -0.715730825f,
   -0.689540545f, -0.724247083f,
   -0.680600998f, -0.732654272f,
   -0.671558955f, -0.740951125f,
   -0.662415778f, -0.749136395f,
   -0.653172843f, -0.757208847f,
   -0.643831543f, -0.765167266f,
   -0.634393284f, -0.773010453f,
   -0.624859488f, -0.780737229f,
   -0.615231591f, -0.788346428f,
   -0.605511041f, -0.795836905f,
   -0.595699304f, -0.803207531f,
   -0.585797857f, -0.810457198f,
   -0.575808191f, -0.817584813f,
   -0.565731811f, -0.824589303f,
   -0.555570233f, -0.831469612f,
   -0.545324988f, -0.838224706f,
   -0.534997620f, -0.844853565f,
   -0.524589683f, -0.851355193f,
   -0.514102744f, -0.857728610f,
   -0.503538384f, -0.863972856f,
   -0.492898192f, -0.870086991f,
   -0.482183772f, -0.876070094f,
   -0.471396737f, -0.881921264f,
   -0.460538711f, -0.887639620f,
   -0.449611330f, -0.893224301f,
   -0.438616239f, -0.898674466f,
   -0.427555093f, -0.903989293f,
   -0.416429560f, -0.909167983f,
   -0.405241314f, -0.914209756f,
   -0.393992040f, -0.919113852f,
   -0.382683432f, -0.923879533f,
   -0.371317194f, -0.928506080f,
   -0.359895037f, -0.932992799f,
   -0.348418680f, -0.937339012f,
   -0.336889853f, -0.941544065f,
   -0.325310292f, -0.945607325f,
   -0.313681740f, -0.949528181f,
   -0.302005949f, -0.953306040f,
   -0.290284677f, -0.956940336f,
   -0.278519689f, -0.960430519f,
   -0.266712757f, -0.963776066f,
   -0.254865660f, -0.966976471f,
   -0.242980180f, -0.970031253f,
   -0.231058108f, -0.972939952f,
   -0.219101240f, -0.975702130f,
   -0.207111376f, -0.978317371f,
   -0.195090322f, -0.980785280f,
   -0.183039888f, -0.983105487f,
   -0.170961889f, -0.985277642f,
   -0.158858143f, -0.987301418f,
   -0.146730474f, -0.989176510f,
   -0.134580709f, -0.990902635f,
   -0.122410675f, -0.992479535f,
   -0.110222207f, -0.993906970f,
   -0.098017140f, -0.995184727f,
   -0.085797312f, -0.996312612f,
   -0.073564564f, -0.997290457f,
   -0.061320736f, -0.998118113f,
   -0.049067674f, -0.998795456f,
   -0.036807223f, -0.999322385f,
   -0.024541229f, -0.999698819f,
   -0.012271538f, -0.999924702f,
   -0.000000000f, -1.000000000f,
    0.012271538f, -0.999924702f,
    0.024541229f, -0.999698819f,
    0.036807223f, -0.999322385f,
    0.049067674f, -0.998795456f,
    0.061320736f, -0.998118113f,
    0.073564564f, -0.997290457f,
    0.085797312f, -0.996312612f,
    0.098017140f, -0.995184727f,
    0.110222207f, -0.993906970f,
    0.122410675f, -0.992479535f,
    0.134580709f, -0.990902635f,
    0.146730474f, -0.989176510f,
    0.158858143f, -0.987301418f,
    0.170961889f, -0.985277642f,
    0.183039888f, -0.983105487f,
    0.195090322f, -0.980785280f,
    0.207111376f, -0.978317371f,
    0.219101240f, -0.975702130f,
    0.231058108f, -0.972939952f,
    0.242980180f, -0.970031253f,
    0.254865660f, -0.966976471f,
    0.266712757f, -0.963776066f,
    0.278519689f, -0.960430519f,
    0.290284677f, -0.956940336f,
    0.302005949f, -0.953306040f,
    0.313681740f, -0.949528181f,
    0.325310292f, -0.945607325f,
    0.336889853f, -0.941544065f,
    0.348418680f, -0.937339012f,
    0.359895037f, -0.932992799f,
    0.371317194f, -0.928506080f,
    0.382683432f, -0.923879533f,
    0.393992040f, -0.919113852f,
    0.405241314f, -0.914209756f,
    0.416429560f, -0.909167983f,
    0.427555093f, -0.903989293f,
    0.438616239f, -0.898674466f,
    0.449611330f, -0.893224301f,
    0.460538711f, -0.887639620f,
    0.471396737f, -0.881921264f,
    0.482183772f, -0.876070094f,
    0.492898192f, -0.870086991f,
    0.503538384f, -0.863972856f,
    0.514102744f, -0.857728610f,
    0.524589683f, -0.851355193f,
    0.534997620f, -0.844853565f,
    0.545324988f, -0.838224706f,
    0.555570233f, -0.831469612f,
    0.565731811f, -0.824589303f,
    0.575808191f, -0.817584813f,
    0.585797857f, -0.810457198f,
    0.595699304f, -0.803207531f,
    0.605511041f, -0.795836905f,
    0.615231591f, -0.788346428f,
    0.624859488f, -0.780737229f,
    0.634393284f, -0.773010453f,
    0.643831543f, -0.765167266f,
    0.653172843f, -0.757208847f,
    0.662415778f, -0.749136395f,
    0.671558955f, -0.740951125f,
    0.680600998f, -0.732654272f,
    0.689540545f, -0.724247083f,
    0.698376249f, -0.715730825f,
    0.707106781f, -0.707106781f,
    0.715730825f, -0.698376249f,
    0.724247083f, -0.689540545f,
    0.732654272f, -0.680600998f,
    0.740951125f, -0.671558955f,
    0.749136395f, -0.662415778f,
    0.757208847f, -0.653172843f,
    0.765167266f, -0.643831543f,
    0.773010453f, -0.634393284f,
    0.780737229f, -0.624859488f,
    0.788346428f, -0.615231591f,
    0.795836905f, -0.605511041f,
    0.803207531f, -0.595699304f,
    0.810457198f, -0.585797857f,
    0.817584813f, -0.575808191f,
    0.824589303f, -0.565731811f,
    0.831469612f, -0.555570233f,
    0.838224706f, -0.545324988f,
    0.844853565f, -0.534997620f,
    0.851355193f, -0.524589683f,
    0.857728610f, -0.514102744f,
    0.863972856f, -0.503538384f,
    0.870086991f, -0.492898192f,
    0.876070094f, -0.482183772f,
    0.881921264f, -0.471396737f,
    0.887639620f, -0.460538711f,
    0.893224301f, -0.449611330f,
    0.898674466f, -0.438616239f,
    0.903989293f, -0.427555093f,
    0.909167983f, -0.416429560f,
    0.914209756f, -0.405241314f,
    0.919113852f, -0.393992040f,
    0.923879533f, -0.382683432f,
    0.928506080f, -0.371317194f,
    0.932992799f, -0.359895037f,
    0.937339012f, -0.348418680f,
    0.941544065f, -0.336889853f,
    0.945607325f, -0.325310292f,
    0.949528181f, -0.313681740f,
    0.953306040f, -0.302005949f,
    0.956940336f, -0.290284677f,
    0.960430519f, -0.278519689f,
    0.963776066f, -0.266712757f,
    0.966976471f, -0.254865660f,
    0.970031253f, -0.242980180f,
    0.972939952f, -0.231058108f,
    0.975702130f, -0.219101240f,
    0.978317371f, -0.207111376f,
    0.980785280f, -0.195090322f,
    0.983105487f, -0.183039888f,
    0.985277642f, -0.170961889f,
    0.987301418f, -0.158858143f,
    0.989176510f, -0.146730474f,
    0.990902635f, -0.134580709f,
    0.992479535f, -0.122410675f,
    0.993906970f, -0.110222207f,
    0.995184727f, -0.098017140f,
    0.996312612f, -0.085797312f,
    0.997290457f, -0.073564564f,
    0.998118113f, -0.061320736f,
    0.998795456f, -0.049067674f,
    0.999322385f, -0.036807223f,
    0.999698819f, -0.024541229f,
    0.999924702f, -0.012271538f
};
#endif // INCLUDE_FFT_TABLES_512


#if defined(INCLUDE_FFT_TABLES_1024)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 1024, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_1024[2048] = {
    1.000000000f,  0.000000000f,
    0.999981175f,  0.006135885f,
    0.999924702f,  0.012271538f,
    0.999830582f,  0.018406730f,
    0.999698819f,  0.024541229f,
    0.999529418f,  0.030674803f,
    0.999322385f,  0.036807223f,
    0.999077728f,  0.042938257f,
    0.998795456f,  0.049067674f,
    0.998475581f,  0.055195244f,
    0.998118113f,  0.061320736f,
    0.997723067f,  0.067443920f,
    0.997290457f,  0.073564564f,
    0.996820299f,  0.079682438f,
    0.996312612f,  0.085797312f,
    0.995767414f,  0.091908956f,
    0.995184727f,  0.098017140f,
    0.994564571f,  0.104121634f,
    0.993906970f,  0.110222207f,
    0.993211949f,  0.116318631f,
    0.992479535f,  0.122410675f,
    0.991709754f,  0.128498111f,
    0.990902635f,  0.134580709f,
    0.990058210f,  0.140658239f,
    0.989176510f,  0.146730474f,
    0.988257568f,  0.152797185f,
    0.987301418f,  0.158858143f,
    0.986308097f,  0.164913120f,
    0.985277642f,  0.170961889f,
    0.984210092f,  0.177004220f,
    0.983105487f,  0.183039888f,
    0.981963869f,  0.189068664f,
    0.980785280f,  0.195090322f,
    0.979569766f,  0.201104635f,
    0.978317371f,  0.207111376f,
    0.977028143f,  0.213110320f,
    0.975702130f,  0.219101240f,
    0.974339383f,  0.225083911f,
    0.972939952f,  0.231058108f,
    0.971503891f,  0.237023606f,
    0.970031253f,  0.242980180f,
    0.968522094f,  0.248927606f,
    0.966976471f,  0.254865660f,
    0.965394442f,  0.260794118f,
    0.963776066f,  0.266712757f,
    0.962121404f,  0.272621355f,
    0.960430519f,  0.278519689f,
    0.958703475f,  0.284407537f,
    0.956940336f,  0.290284677f,
    0.955141168f,  0.296150888f,
    0.953306040f,  0.302005949f,
    0.951435021f,  0.307849640f,
    0.949528181f,  0.313681740f,
    0.947585591f,  0.319502031f,
    0.945607325f,  0.325310292f,
    0.943593458f,  0.331106306f,
    0.941544065f,  0.336889853f,
    0.939459224f,  0.342660717f,
    0.937339012f,  0.348418680f,
    0.935183510f,  0.354163525f,
    0.932992799f,  0.359895037f,
    0.930766961f,  0.365612998f,
    0.928506080f,  0.371317194f,
    0.926210242f,  0.377007410f,
    0.923879533f,  0.382683432f,
    0.921514039f,  0.388345047f,
    0.919113852f,  0.393992040f,
    0.916679060f,  0.399624200f,
    0.914209756f,  0.405241314f,
    0.911706032f,  0.410843171f,
    0.909167983f,  0.416429560f,
    0.906595705f,  0.422000271f,
    0.903989293f,  0.427555093f,
    0.901348847f,  0.433093819f,
    0.898674466f,  0.438616239f,
    0.895966250f,  0.444122145f,
    0.893224301f,  0.449611330f,
    0.890448723f,  0.455083587f,
    0.887639620f,  0.460538711f,
    0.884797098f,  0.465976496f,
    0.881921264f,  0.471396737f,
    0.879012226f,  0.476799230f,
    0.876070094f,  0.482183772f,
    0.873094978f,  0.487550160f,
    0.870086991f,  0.492898192f,
    0.867046246f,  0.498227667f,
    0.863972856f,  0.503538384f,
    0.860866939f,  0.508830143f,
    0.857728610f,  0.514102744f,
    0.854557988f,  0.519355990f,
    0.851355193f,  0.524589683f,
    0.848120345f,  0.529803625f,
    0.844853565f,  0.534997620f,
    0.841554977f,  0.540171473f,
    0.838224706f,  0.545324988f,
    0.834862875f,  0.550457973f,
    0.831469612f,  0.555570233f,
    0.828045045f,  0.560661576f,
    0.824589303f,  0.565731811f,
    0.821102515f,  0.570780746f,
    0.817584813f,  0.575808191f,
    0.814036330f,  0.580813958f,
    0.810457198f,  0.585797857f,
    0.806847554f,  0.590759702f,
    0.803207531f,  0.595699304f,
    0.799537269f,  0.600616479f,
    0.795836905f,  0.605511041f,
    0.792106577f,  0.610382806f,
    0.788346428f,  0.615231591f,
    0.784556597f,  0.620057212f,
    0.780737229f,  0.624859488f,
    0.776888466f,  0.629638239f,
    0.773010453f,  0.634393284f,
    0.769103338f,  0.639124445f,
    0.765167266f,  0.643831543f,
    0.761202385f,  0.648514401f,
    0.757208847f,  0.653172843f,
    0.753186799f,  0.657806693f,
    0.749136395f,  0.662415778f,
    0.745057785f,  0.666999922f,
    0.740951125f,  0.671558955f,
    0.736816569f,  0.676092704f,
    0.732654272f,  0.680600998f,
    0.728464390f,  0.685083668f,
    0.724247083f,  0.689540545f,
    0.720002508f,  0.693971461f,
    0.715730825f,  0.698376249f,
    0.711432196f,  0.702754744f,
    0.707106781f,  0.707106781f,
    0.702754744f,  0.711432196f,
    0.698376249f,  0.715730825f,
    0.693971461f,  0.720002508f,
    0.689540545f,  0.724247083f,
    0.685083668f,  0.728464390f,
    0.680600998f,  0.732654272f,
    0.676092704f,  0.736816569f,
    0.671558955f,  0.740951125f,
    0.666999922f,  0.745057785f,
    0.662415778f,  0.749136395f,
    0.657806693f,  0.753186799f,
    0.653172843f,  0.757208847f,
    0.648514401f,  0.761202385f,
    0.643831543f,  0.765167266f,
    0.639124445f,  0.769103338f,
    0.634393284f,  0.773010453f,
    0.629638239f,  0.776888466f,
    0.624859488f,  0.780737229f,
    0.620057212f,  0.784556597f,
    0.615231591f,  0.788346428f,
    0.610382806f,  0.792106577f,
    0.605511041f,  0.795836905f,
    0.600616479f,  0.799537269f,
    0.595699304f,  0.803207531f,
    0.590759702f,  0.806847554f,
    0.585797857f,  0.810457198f,
    0.580813958f,  0.814036330f,
    0.575808191f,  0.817584813f,
    0.570780746f,  0.821102515f,
    0.565731811f,  0.824589303f,
    0.560661576f,  0.828045045f,
    0.555570233f,  0.831469612f,
    0.550457973f,  0.834862875f,
    0.545324988f,  0.838224706f,
    0.540171473f,  0.841554977f,
    0.534997620f,  0.844853565f,
    0.529803625f,  0.848120345f,
    0.524589683f,  0.851355193f,
    0.519355990f,  0.854557988f,
    0.514102744f,  0.857728610f,
    0.508830143f,  0.860866939f,
    0.503538384f,  0.863972856f,
    0.498227667f,  0.867046246f,
    0.492898192f,  0.870086991f,
    0.487550160f,  0.873094978f,
    0.482183772f,  0.876070094f,
    0.476799230f,  0.879012226f,
    0.471396737f,  0.881921264f,
    0.465976496f,  0.884797098f,
    0.460538711f,  0.887639620f,
    0.455083587f,  0.890448723f,
    0.449611330f,  0.893224301f,
    0.444122145f,  0.895966250f,
    0.438616239f,  0.898674466f,
    0.433093819f,  0.901348847f,
    0.427555093f,  0.903989293f,
    0.422000271f,  0.906595705f,
    0.416429560f,  0.909167983f,
    0.410843171f,  0.911706032f,
    0.405241314f,  0.914209756f,
    0.399624200f,  0.916679060f,
    0.393992040f,  0.919113852f,
    0.388345047f,  0.921514039f,
    0.382683432f,  0.923879533f,
    0.377007410f,  0.926210242f,
    0.371317194f,  0.928506080f,
    0.365612998f,  0.930766961f,
    0.359895037f,  0.932992799f,
    0.354163525f,  0.935183510f,
    0.348418680f,  0.937339012f,
    0.342660717f,  0.939459224f,
    0.336889853f,  0.941544065f,
    0.331106306f,  0.943593458f,
    0.325310292f,  0.945607325f,
    0.319502031f,  0.947585591f,
    0.313681740f,  0.949528181f,
    0.307849640f,  0.951435021f,
    0.302005949f,  0.953306040f,
    0.296150888f,  0.955141168f,
    0.290284677f,  0.956940336f,
    0.284407537f,  0.958703475f,
    0.278519689f,  0.960430519f,
    0.272621355f,  0.962121404f,
    0.266712757f,  0.963776066f,
    0.260794118f,  0.965394442f,
    0.254865660f,  0.966976471f,
    0.248927606f,  0.968522094f,
    0.242980180f,  0.970031253f,
    0.237023606f,  0.971503891f,
    0.231058108f,  0.972939952f,
    0.225083911f,  0.974339383f,
    0.219101240f,  0.975702130f,
    0.213110320f,  0.977028143f,
    0.207111376f,  0.978317371f,
    0.201104635f,  0.979569766f,
    0.195090322f,  0.980785280f,
    0.189068664f,  0.981963869f,
    0.183039888f,  0.983105487f,
    0.177004220f,  0.984210092f,
    0.170961889f,  0.985277642f,
    0.164913120f,  0.986308097f,
    0.158858143f,  0.987301418f,
    0.152797185f,  0.988257568f,
    0.146730474f,  0.989176510f,
    0.140658239f,  0.990058210f,
    0.134580709f,  0.990902635f,
    0.128498111f,  0.991709754f,
    0.122410675f,  0.992479535f,
    0.116318631f,  0.993211949f,
    0.110222207f,  0.993906970f,
    0.104121634f,  0.994564571f,
    0.098017140f,  0.995184727f,
    0.091908956f,  0.995767414f,
    0.085797312f,  0.996312612f,
    0.079682438f,  0.996820299f,
    0.073564564f,  0.997290457f,
    0.067443920f,  0.997723067f,
    0.061320736f,  0.998118113f,
    0.055195244f,  0.998475581f,
    0.049067674f,  0.998795456f,
    0.042938257f,  0.999077728f,
    0.036807223f,  0.999322385f,
    0.030674803f,  0.999529418f,
    0.024541229f,  0.999698819f,
    0.018406730f,  0.999830582f,
    0.012271538f,  0.999924702f,
    0.006135885f,  0.999981175f,
    0.000000000f,  1.000000000f,
   -0.006135885f,  0.999981175f,
   -0.012271538f,  0.999924702f,
   -0.018406730f,  0.999830582f,
   -0.024541229f,  0.999698819f,
   -0.030674803f,  0.999529418f,
   -0.036807223f,  0.999322385f,
   -0.042938257f,  0.999077728f,
   -0.049067674f,  0.998795456f,
   -0.055195244f,  0.998475581f,
   -0.061320736f,  0.998118113f,
   -0.067443920f,  0.997723067f,
   -0.073564564f,  0.997290457f,
   -0.079682438f,  0.996820299f,
   -0.085797312f,  0.996312612f,
   -0.091908956f,  0.995767414f,
   -0.098017140f,  0.995184727f,
   -0.104121634f,  0.994564571f,
   -0.110222207f,  0.993906970f,
   -0.116318631f,  0.993211949f,
   -0.122410675f,  0.992479535f,
   -0.128498111f,  0.991709754f,
   -0.134580709f,  0.990902635f,
   -0.140658239f,  0.990058210f,
   -0.146730474f,  0.989176510f,
   -0.152797185f,  0.988257568f,
   -0.158858143f,  0.987301418f,
   -0.164913120f,  0.986308097f,
   -0.170961889f,  0.985277642f,
   -0.177004220f,  0.984210092f,
   -0.183039888f,  0.983105487f,
   -0.189068664f,  0.981963869f,
   -0.195090322f,  0.980785280f,
   -0.201104635f,  0.979569766f,
   -0.207111376f,  0.978317371f,
   -0.213110320f,  0.977028143f,
   -0.219101240f,  0.975702130f,
   -0.225083911f,  0.974339383f,
   -0.231058108f,  0.972939952f,
   -0.237023606f,  0.971503891f,
   -0.242980180f,  0.970031253f,
   -0.248927606f,  0.968522094f,
   -0.254865660f,  0.966976471f,
   -0.260794118f,  0.965394442f,
   -0.266712757f,  0.963776066f,
   -0.272621355f,  0.962121404f,
   -0.278519689f,  0.960430519f,
   -0.284407537f,  0.958703475f,
   -0.290284677f,  0.956940336f,
   -0.296150888f,  0.955141168f,
   -0.302005949f,  0.953306040f,
   -0.307849640f,  0.951435021f,
   -0.313681740f,  0.949528181f,
   -0.319502031f,  0.947585591f,
   -0.325310292f,  0.945607325f,
   -0.331106306f,  0.943593458f,
   -0.336889853f,  0.941544065f,
   -0.342660717f,  0.939459224f,
   -0.348418680f,  0.937339012f,
   -0.354163525f,  0.935183510f,
   -0.359895037f,  0.932992799f,
   -0.365612998f,  0.930766961f,
   -0.371317194f,  0.928506080f,
   -0.377007410f,  0.926210242f,
   -0.382683432f,  0.923879533f,
   -0.388345047f,  0.921514039f,
   -0.393992040f,  0.919113852f,
   -0.399624200f,  0.916679060f,
   -0.405241314f,  0.914209756f,
   -0.410843171f,  0.911706032f,
   -0.416429560f,  0.909167983f,
   -0.422000271f,  0.906595705f,
   -0.427555093f,  0.903989293f,
   -0.433093819f,  0.901348847f,
   -0.438616239f,  0.898674466f,
   -0.444122145f,  0.895966250f,
   -0.449611330f,  0.893224301f,
   -0.455083587f,  0.890448723f,
   -0.460538711f,  0.887639620f,
   -0.465976496f,  0.884797098f,
   -0.471396737f,  0.881921264f,
   -0.476799230f,  0.879012226f,
   -0.482183772f,  0.876070094f,
   -0.487550160f,  0.873094978f,
   -0.492898192f,  0.870086991f,
   -0.498227667f,  0.867046246f,
   -0.503538384f,  0.863972856f,
   -0.508830143f,  0.860866939f,
   -0.514102744f,  0.857728610f,
   -0.519355990f,  0.854557988f,
   -0.524589683f,  0.851355193f,
   -0.529803625f,  0.848120345f,
   -0.534997620f,  0.844853565f,
   -0.540171473f,  0.841554977f,
   -0.545324988f,  0.838224706f,
   -0.550457973f,  0.834862875f,
   -0.555570233f,  0.831469612f,
   -0.560661576f,  0.828045045f,
   -0.565731811f,  0.824589303f,
   -0.570780746f,  0.821102515f,
   -0.575808191f,  0.817584813f,
   -0.580813958f,  0.814036330f,
   -0.585797857f,  0.810457198f,
   -0.590759702f,  0.806847554f,
   -0.595699304f,  0.803207531f,
   -0.600616479f,  0.799537269f,
   -0.605511041f,  0.795836905f,
   -0.610382806f,  0.792106577f,
   -0.615231591f,  0.788346428f,
   -0.620057212f,  0.784556597f,
   -0.624859488f,  0.780737229f,
   -0.629638239f,  0.776888466f,
   -0.634393284f,  0.773010453f,
   -0.639124445f,  0.769103338f,
   -0.643831543f,  0.765167266f,
   -0.648514401f,  0.761202385f,
   -0.653172843f,  0.757208847f,
   -0.657806693f,  0.753186799f,
   -0.662415778f,  0.749136395f,
   -0.666999922f,  0.745057785f,
   -0.671558955f,  0.740951125f,
   -0.676092704f,  0.736816569f,
   -0.680600998f,  0.732654272f,
   -0.685083668f,  0.728464390f,
   -0.689540545f,  0.724247083f,
   -0.693971461f,  0.720002508f,
   -0.698376249f,  0.715730825f,
   -0.702754744f,  0.711432196f,
   -0.707106781f,  0.707106781f,
   -0.711432196f,  0.702754744f,
   -0.715730825f,  0.698376249f,
   -0.720002508f,  0.693971461f,
   -0.724247083f,  0.689540545f,
   -0.728464390f,  0.685083668f,
   -0.732654272f,  0.680600998f,
   -0.736816569f,  0.676092704f,
   -0.740951125f,  0.671558955f,
   -0.745057785f,  0.666999922f,
   -0.749136395f,  0.662415778f,
   -0.753186799f,  0.657806693f,
   -0.757208847f,  0.653172843f,
   -0.761202385f,  0.648514401f,
   -0.765167266f,  0.643831543f,
   -0.769103338f,  0.639124445f,
   -0.773010453f,  0.634393284f,
   -0.776888466f,  0.629638239f,
   -0.780737229f,  0.624859488f,
   -0.784556597f,  0.620057212f,
   -0.788346428f,  0.615231591f,
   -0.792106577f,  0.610382806f,
   -0.795836905f,  0.605511041f,
   -0.799537269f,  0.600616479f,
   -0.803207531f,  0.595699304f,
   -0.806847554f,  0.590759702f,
   -0.810457198f,  0.585797857f,
   -0.814036330f,  0.580813958f,
   -0.817584813f,  0.575808191f,
   -0.821102515f,  0.570780746f,
   -0.824589303f,  0.565731811f,
   -0.828045045f,  0.560661576f,
   -0.831469612f,  0.555570233f,
   -0.834862875f,  0.550457973f,
   -0.838224706f,  0.545324988f,
   -0.841554977f,  0.540171473f,
   -0.844853565f,  0.534997620f,
   -0.848120345f,  0.529803625f,
   -0.851355193f,  0.524589683f,
   -0.854557988f,  0.519355990f,
   -0.857728610f,  0.514102744f,
   -0.860866939f,  0.508830143f,
   -0.863972856f,  0.503538384f,
   -0.867046246f,  0.498227667f,
   -0.870086991f,  0.492898192f,
   -0.873094978f,  0.487550160f,
   -0.876070094f,  0.482183772f,
   -0.879012226f,  0.476799230f,
   -0.881921264f,  0.471396737f,
   -0.884797098f,  0.465976496f,
   -0.887639620f,  0.460538711f,
   -0.890448723f,  0.455083587f,
   -0.893224301f,  0.449611330f,
   -0.895966250f,  0.444122145f,
   -0.898674466f,  0.438616239f,
   -0.901348847f,  0.433093819f,
   -0.903989293f,  0.427555093f,
   -0.906595705f,  0.422000271f,
   -0.909167983f,  0.416429560f,
   -0.911706032f,  0.410843171f,
   -0.914209756f,  0.405241314f,
   -0.916679060f,  0.399624200f,
   -0.919113852f,  0.393992040f,
   -0.921514039f,  0.388345047f,
   -0.923879533f,  0.382683432f,
   -0.926210242f,  0.377007410f,
   -0.928506080f,  0.371317194f,
   -0.930766961f,  0.365612998f,
   -0.932992799f,  0.359895037f,
   -0.935183510f,  0.354163525f,
   -0.937339012f,  0.348418680f,
   -0.939459224f,  0.342660717f,
   -0.941544065f,  0.336889853f,
   -0.943593458f,  0.331106306f,
   -0.945607325f,  0.325310292f,
   -0.947585591f,  0.319502031f,
   -0.949528181f,  0.313681740f,
   -0.951435021f,  0.307849640f,
   -0.953306040f,  0.302005949f,
   -0.955141168f,  0.296150888f,
   -0.956940336f,  0.290284677f,
   -0.958703475f,  0.284407537f,
   -0.960430519f,  0.278519689f,
   -0.962121404f,  0.272621355f,
   -0.963776066f,  0.266712757f,
   -0.965394442f,  0.260794118f,
   -0.966976471f,  0.254865660f,
   -0.968522094f,  0.248927606f,
   -0.970031253f,  0.242980180f,
   -0.971503891f,  0.237023606f,
   -0.972939952f,  0.231058108f,
   -0.974339383f,  0.225083911f,
   -0.975702130f,  0.219101240f,
   -0.977028143f,  0.213110320f,
   -0.978317371f,  0.207111376f,
   -0.979569766f,  0.201104635f,
   -0.980785280f,  0.195090322f,
   -0.981963869f,  0.189068664f,
   -0.983105487f,  0.183039888f,
   -0.984210092f,  0.177004220f,
   -0.985277642f,  0.170961889f,
   -0.986308097f,  0.164913120f,
   -0.987301418f,  0.158858143f,
   -0.988257568f,  0.152797185f,
   -0.989176510f,  0.146730474f,
   -0.990058210f,  0.140658239f,
   -0.990902635f,  0.134580709f,
   -0.991709754f,  0.128498111f,
   -0.992479535f,  0.122410675f,
   -0.993211949f,  0.116318631f,
   -0.993906970f,  0.110222207f,
   -0.994564571f,  0.104121634f,
   -0.995184727f,  0.098017140f,
   -0.995767414f,  0.091908956f,
   -0.996312612f,  0.085797312f,
   -0.996820299f,  0.079682438f,
   -0.997290457f,  0.073564564f,
   -0.997723067f,  0.067443920f,
   -0.998118113f,  0.061320736f,
   -0.998475581f,  0.055195244f,
   -0.998795456f,  0.049067674f,
   -0.999077728f,  0.042938257f,
   -0.999322385f,  0.036807223f,
   -0.999529418f,  0.030674803f,
   -0.999698819f,  0.024541229f,
   -0.999830582f,  0.018406730f,
   -0.999924702f,  0.012271538f,
   -0.999981175f,  0.006135885f,
   -1.000000000f,  0.000000000f,
   -0.999981175f, -0.006135885f,
   -0.999924702f, -0.012271538f,
   -0.999830582f, -0.018406730f,
   -0.999698819f, -0.024541229f,
   -0.999529418f, -0.030674803f,
   -0.999322385f, -0.036807223f,
   -0.999077728f, -0.042938257f,
   -0.998795456f, -0.049067674f,
   -0.998475581f, -0.055195244f,
   -0.998118113f, -0.061320736f,
   -0.997723067f, -0.067443920f,
   -0.997290457f, -0.073564564f,
   -0.996820299f, -0.079682438f,
   -0.996312612f, -0.085797312f,
   -0.995767414f, -0.091908956f,
   -0.995184727f, -0.098017140f,
   -0.994564571f, -0.104121634f,
   -0.993906970f, -0.110222207f,
   -0.993211949f, -0.116318631f,
   -0.992479535f, -0.122410675f,
   -0.991709754f, -0.128498111f,
   -0.990902635f, -0.134580709f,
   -0.990058210f, -0.140658239f,
   -0.989176510f, -0.146730474f,
   -0.988257568f, -0.152797185f,
   -0.987301418f, -0.158858143f,
   -0.986308097f, -0.164913120f,
   -0.985277642f, -0.170961889f,
   -0.984210092f, -0.177004220f,
   -0.983105487f, -0.183039888f,
   -0.981963869f, -0.189068664f,
   -0.980785280f, -0.195090322f,
   -0.979569766f, -0.201104635f,
   -0.978317371f, -0.207111376f,
   -0.977028143f, -0.213110320f,
   -0.975702130f, -0.219101240f,
   -0.974339383f, -0.225083911f,
   -0.972939952f, -0.231058108f,
   -0.971503891f, -0.237023606f,
   -0.970031253f, -0.242980180f,
   -0.968522094f, -0.248927606f,
   -0.966976471f, -0.254865660f,
   -0.965394442f, -0.260794118f,
   -0.963776066f, -0.266712757f,
   -0.962121404f, -0.272621355f,
   -0.960430519f, -0.278519689f,
   -0.958703475f, -0.284407537f,
   -0.956940336f, -0.290284677f,
   -0.955141168f, -0.296150888f,
   -0.953306040f, -0.302005949f,
   -0.951435021f, -0.307849640f,
   -0.949528181f, -0.313681740f,
   -0.947585591f, -0.319502031f,
   -0.945607325f, -0.325310292f,
   -0.943593458f, -0.331106306f,
   -0.941544065f, -0.336889853f,
   -0.939459224f, -0.342660717f,
   -0.937339012f, -0.348418680f,
   -0.935183510f, -0.354163525f,
   -0.932992799f, -0.359895037f,
   -0.930766961f, -0.365612998f,
   -0.928506080f, -0.371317194f,
   -0.926210242f, -0.377007410f,
   -0.923879533f, -0.382683432f,
   -0.921514039f, -0.388345047f,
   -0.919113852f, -0.393992040f,
   -0.916679060f, -0.399624200f,
   -0.914209756f, -0.405241314f,
   -0.911706032f, -0.410843171f,
   -0.909167983f, -0.416429560f,
   -0.906595705f, -0.422000271f,
   -0.903989293f, -0.427555093f,
   -0.901348847f, -0.433093819f,
   -0.898674466f, -0.438616239f,
   -0.895966250f, -0.444122145f,
   -0.893224301f, -0.449611330f,
   -0.890448723f, -0.455083587f,
   -0.887639620f, -0.460538711f,
   -0.884797098f, -0.465976496f,
   -0.881921264f, -0.471396737f,
   -0.879012226f, -0.476799230f,
   -0.876070094f, -0.482183772f,
   -0.873094978f, -0.487550160f,
   -0.870086991f, -0.492898192f,
   -0.867046246f, -0.498227667f,
   -0.863972856f, -0.503538384f,
   -0.860866939f, -0.508830143f,
   -0.857728610f, -0.514102744f,
   -0.854557988f, -0.519355990f,
   -0.851355193f, -0.524589683f,
   -0.848120345f, -0.529803625f,
   -0.844853565f, -0.534997620f,
   -0.841554977f, -0.540171473f,
   -0.838224706f, -0.545324988f,
   -0.834862875f, -0.550457973f,
   -0.831469612f, -0.555570233f,
   -0.828045045f, -0.560661576f,
   -0.824589303f, -0.565731811f,
   -0.821102515f, -0.570780746f,
   -0.817584813f, -0.575808191f,
   -0.814036330f, -0.580813958f,
   -0.810457198f, -0.585797857f,
   -0.806847554f, -0.590759702f,
   -0.803207531f, -0.595699304f,
   -0.799537269f, -0.600616479f,
   -0.795836905f, -0.605511041f,
   -0.792106577f, -0.610382806f,
   -0.788346428f, -0.615231591f,
   -0.784556597f, -0.620057212f,
   -0.780737229f, -0.624859488f,
   -0.776888466f, -0.629638239f,
   -0.773010453f, -0.634393284f,
   -0.769103338f, -0.639124445f,
   -0.765167266f, -0.643831543f,
   -0.761202385f, -0.648514401f,
   -0.757208847f, -0.653172843f,
   -0.753186799f, -0.657806693f,
   -0.749136395f, -0.662415778f,
   -0.745057785f, -0.666999922f,
   -0.740951125f, -0.671558955f,
   -0.736816569f, -0.676092704f,
   -0.732654272f, -0.680600998f,
   -0.728464390f, -0.685083668f,
   -0.724247083f, -0.689540545f,
   -0.720002508f, -0.693971461f,
   -0.715730825f, -0.698376249f,
   -0.711432196f, -0.702754744f,
   -0.707106781f, -0.707106781f,
   -0.702754744f, -0.711432196f,
   -0.698376249f, -0.715730825f,
   -0.693971461f, -0.720002508f,
   -0.689540545f, -0.724247083f,
   -0.685083668f, -0.728464390f,
   -0.680600998f, -0.732654272f,
   -0.676092704f, -0.736816569f,
   -0.671558955f, -0.740951125f,
   -0.666999922f, -0.745057785f,
   -0.662415778f, -0.749136395f,
   -0.657806693f, -0.753186799f,
   -0.653172843f, -0.757208847f,
   -0.648514401f, -0.761202385f,
   -0.643831543f, -0.765167266f,
   -0.639124445f, -0.769103338f,
   -0.634393284f, -0.773010453f,
   -0.629638239f, -0.776888466f,
   -0.624859488f, -0.780737229f,
   -0.620057212f, -0.784556597f,
   -0.615231591f, -0.788346428f,
   -0.610382806f, -0.792106577f,
   -0.605511041f, -0.795836905f,
   -0.600616479f, -0.799537269f,
   -0.595699304f, -0.803207531f,
   -0.590759702f, -0.806847554f,
   -0.585797857f, -0.810457198f,
   -0.580813958f, -0.814036330f,
   -0.575808191f, -0.817584813f,
   -0.570780746f, -0.821102515f,
   -0.565731811f, -0.824589303f,
   -0.560661576f, -0.828045045f,
   -0.555570233f, -0.831469612f,
   -0.550457973f, -0.834862875f,
   -0.545324988f, -0.838224706f,
   -0.540171473f, -0.841554977f,
   -0.534997620f, -0.844853565f,
   -0.529803625f, -0.848120345f,
   -0.524589683f, -0.851355193f,
   -0.519355990f, -0.854557988f,
   -0.514102744f, -0.857728610f,
   -0.508830143f, -0.860866939f,
   -0.503538384f, -0.863972856f,
   -0.498227667f, -0.867046246f,
   -0.492898192f, -0.870086991f,
   -0.487550160f, -0.873094978f,
   -0.482183772f, -0.876070094f,
   -0.476799230f, -0.879012226f,
   -0.471396737f, -0.881921264f,
   -0.465976496f, -0.884797098f,
   -0.460538711f, -0.887639620f,
   -0.455083587f, -0.890448723f,
   -0.449611330f, -0.893224301f,
   -0.444122145f, -0.895966250f,
   -0.438616239f, -0.898674466f,
   -0.433093819f, -0.901348847f,
   -0.427555093f, -0.903989293f,
   -0.422000271f, -0.906595705f,
   -0.416429560f, -0.909167983f,
   -0.410843171f, -0.911706032f,
   -0.405241314f, -0.914209756f,
   -0.399624200f, -0.916679060f,
   -0.393992040f, -0.919113852f,
   -0.388345047f, -0.921514039f,
   -0.382683432f, -0.923879533f,
   -0.377007410f, -0.926210242f,
   -0.371317194f, -0.928506080f,
   -0.365612998f, -0.930766961f,
   -0.359895037f, -0.932992799f,
   -0.354163525f, -0.935183510f,
   -0.348418680f, -0.937339012f,
   -0.342660717f, -0.939459224f,
   -0.336889853f, -0.941544065f,
   -0.331106306f, -0.943593458f,
   -0.325310292f, -0.945607325f,
   -0.319502031f, -0.947585591f,
   -0.313681740f, -0.949528181f,
   -0.307849640f, -0.951435021f,
   -0.302005949f, -0.953306040f,
   -0.296150888f, -0.955141168f,
   -0.290284677f, -0.956940336f,
   -0.284407537f, -0.958703475f,
   -0.278519689f, -0.960430519f,
   -0.272621355f, -0.962121404f,
   -0.266712757f, -0.963776066f,
   -0.260794118f, -0.965394442f,
   -0.254865660f, -0.966976471f,
   -0.248927606f, -0.968522094f,
   -0.242980180f, -0.970031253f,
   -0.237023606f, -0.971503891f,
   -0.231058108f, -0.972939952f,
   -0.225083911f, -0.974339383f,
   -0.219101240f, -0.975702130f,
   -0.213110320f, -0.977028143f,
   -0.207111376f, -0.978317371f,
   -0.201104635f, -0.979569766f,
   -0.195090322f, -0.980785280f,
   -0.189068664f, -0.981963869f,
   -0.183039888f, -0.983105487f,
   -0.177004220f, -0.984210092f,
   -0.170961889f, -0.985277642f,
   -0.164913120f, -0.986308097f,
   -0.158858143f, -0.987301418f,
   -0.152797185f, -0.988257568f,
   -0.146730474f, -0.989176510f,
   -0.140658239f, -0.990058210f,
   -0.134580709f, -0.990902635f,
   -0.128498111f, -0.991709754f,
   -0.122410675f, -0.992479535f,
   -0.116318631f, -0.993211949f,
   -0.110222207f, -0.993906970f,
   -0.104121634f, -0.994564571f,
   -0.098017140f, -0.995184727f,
   -0.091908956f, -0.995767414f,
   -0.085797312f, -0.996312612f,
   -0.079682438f, -0.996820299f,
   -0.073564564f, -0.997290457f,
   -0.067443920f, -0.997723067f,
   -0.061320736f, -0.998118113f,
   -0.055195244f, -0.998475581f,
   -0.049067674f, -0.998795456f,
   -0.042938257f, -0.999077728f,
   -0.036807223f, -0.999322385f,
   -0.030674803f, -0.999529418f,
   -0.024541229f, -0.999698819f,
   -0.018406730f, -0.999830582f,
   -0.012271538f, -0.999924702f,
   -0.006135885f, -0.999981175f,
   -0.000000000f, -1.000000000f,
    0.006135885f, -0.999981175f,
    0.012271538f, -0.999924702f,
    0.018406730f, -0.999830582f,
    0.024541229f, -0.999698819f,
    0.030674803f, -0.999529418f,
    0.036807223f, -0.999322385f,
    0.042938257f, -0.999077728f,
    0.049067674f, -0.998795456f,
    0.055195244f, -0.998475581f,
    0.061320736f, -0.998118113f,
    0.067443920f, -0.997723067f,
    0.073564564f, -0.997290457f,
    0.079682438f, -0.996820299f,
    0.085797312f, -0.996312612f,
    0.091908956f, -0.995767414f,
    0.098017140f, -0.995184727f,
    0.104121634f, -0.994564571f,
    0.110222207f, -0.993906970f,
    0.116318631f, -0.993211949f,
    0.122410675f, -0.992479535f,
    0.128498111f, -0.991709754f,
    0.134580709f, -0.990902635f,
    0.140658239f, -0.990058210f,
    0.146730474f, -0.989176510f,
    0.152797185f, -0.988257568f,
    0.158858143f, -0.987301418f,
    0.164913120f, -0.986308097f,
    0.170961889f, -0.985277642f,
    0.177004220f, -0.984210092f,
    0.183039888f, -0.983105487f,
    0.189068664f, -0.981963869f,
    0.195090322f, -0.980785280f,
    0.201104635f, -0.979569766f,
    0.207111376f, -0.978317371f,
    0.213110320f, -0.977028143f,
    0.219101240f, -0.975702130f,
    0.225083911f, -0.974339383f,
    0.231058108f, -0.972939952f,
    0.237023606f, -0.971503891f,
    0.242980180f, -0.970031253f,
    0.248927606f, -0.968522094f,
    0.254865660f, -0.966976471f,
    0.260794118f, -0.965394442f,
    0.266712757f, -0.963776066f,
    0.272621355f, -0.962121404f,
    0.278519689f, -0.960430519f,
    0.284407537f, -0.958703475f,
    0.290284677f, -0.956940336f,
    0.296150888f, -0.955141168f,
    0.302005949f, -0.953306040f,
    0.307849640f, -0.951435021f,
    0.313681740f, -0.949528181f,
    0.319502031f, -0.947585591f,
    0.325310292f, -0.945607325f,
    0.331106306f, -0.943593458f,
    0.336889853f, -0.941544065f,
    0.342660717f, -0.939459224f,
    0.348418680f, -0.937339012f,
    0.354163525f, -0.935183510f,
    0.359895037f, -0.932992799f,
    0.365612998f, -0.930766961f,
    0.371317194f, -0.928506080f,
    0.377007410f, -0.926210242f,
    0.382683432f, -0.923879533f,
    0.388345047f, -0.921514039f,
    0.393992040f, -0.919113852f,
    0.399624200f, -0.916679060f,
    0.405241314f, -0.914209756f,
    0.410843171f, -0.911706032f,
    0.416429560f, -0.909167983f,
    0.422000271f, -0.906595705f,
    0.427555093f, -0.903989293f,
    0.433093819f, -0.901348847f,
    0.438616239f, -0.898674466f,
    0.444122145f, -0.895966250f,
    0.449611330f, -0.893224301f,
    0.455083587f, -0.890448723f,
    0.460538711f, -0.887639620f,
    0.465976496f, -0.884797098f,
    0.471396737f, -0.881921264f,
    0.476799230f, -0.879012226f,
    0.482183772f, -0.876070094f,
    0.487550160f, -0.873094978f,
    0.492898192f, -0.870086991f,
    0.498227667f, -0.867046246f,
    0.503538384f, -0.863972856f,
    0.508830143f, -0.860866939f,
    0.514102744f, -0.857728610f,
    0.519355990f, -0.854557988f,
    0.524589683f, -0.851355193f,
    0.529803625f, -0.848120345f,
    0.534997620f, -0.844853565f,
    0.540171473f, -0.841554977f,
    0.545324988f, -0.838224706f,
    0.550457973f, -0.834862875f,
    0.555570233f, -0.831469612f,
    0.560661576f, -0.828045045f,
    0.565731811f, -0.824589303f,
    0.570780746f, -0.821102515f,
    0.575808191f, -0.817584813f,
    0.580813958f, -0.814036330f,
    0.585797857f, -0.810457198f,
    0.590759702f, -0.806847554f,
    0.595699304f, -0.803207531f,
    0.600616479f, -0.799537269f,
    0.605511041f, -0.795836905f,
    0.610382806f, -0.792106577f,
    0.615231591f, -0.788346428f,
    0.620057212f, -0.784556597f,
    0.624859488f, -0.780737229f,
    0.629638239f, -0.776888466f,
    0.634393284f, -0.773010453f,
    0.639124445f, -0.769103338f,
    0.643831543f, -0.765167266f,
    0.648514401f, -0.761202385f,
    0.653172843f, -0.757208847f,
    0.657806693f, -0.753186799f,
    0.662415778f, -0.749136395f,
    0.666999922f, -0.745057785f,
    0.671558955f, -0.740951125f,
    0.676092704f, -0.736816569f,
    0.680600998f, -0.732654272f,
    0.685083668f, -0.728464390f,
    0.689540545f, -0.724247083f,
    0.693971461f, -0.720002508f,
    0.698376249f, -0.715730825f,
    0.702754744f, -0.711432196f,
    0.707106781f, -0.707106781f,
    0.711432196f, -0.702754744f,
    0.715730825f, -0.698376249f,
    0.720002508f, -0.693971461f,
    0.724247083f, -0.689540545f,
    0.728464390f, -0.685083668f,
    0.732654272f, -0.680600998f,
    0.736816569f, -0.676092704f,
    0.740951125f, -0.671558955f,
    0.745057785f, -0.666999922f,
    0.749136395f, -0.662415778f,
    0.753186799f, -0.657806693f,
    0.757208847f, -0.653172843f,
    0.761202385f, -0.648514401f,
    0.765167266f, -0.643831543f,
    0.769103338f, -0.639124445f,
    0.773010453f, -0.634393284f,
    0.776888466f, -0.629638239f,
    0.780737229f, -0.624859488f,
    0.784556597f, -0.620057212f,
    0.788346428f, -0.615231591f,
    0.792106577f, -0.610382806f,
    0.795836905f, -0.605511041f,
    0.799537269f, -0.600616479f,
    0.803207531f, -0.595699304f,
    0.806847554f, -0.590759702f,
    0.810457198f, -0.585797857f,
    0.814036330f, -0.580813958f,
    0.817584813f, -0.575808191f,
    0.821102515f, -0.570780746f,
    0.824589303f, -0.565731811f,
    0.828045045f, -0.560661576f,
    0.831469612f, -0.555570233f,
    0.834862875f, -0.550457973f,
    0.838224706f, -0.545324988f,
    0.841554977f, -0.540171473f,
    0.844853565f, -0.534997620f,
    0.848120345f, -0.529803625f,
    0.851355193f, -0.524589683f,
    0.854557988f, -0.519355990f,
    0.857728610f, -0.514102744f,
    0.860866939f, -0.508830143f,
    0.863972856f, -0.503538384f,
    0.867046246f, -0.498227667f,
    0.870086991f, -0.492898192f,
    0.873094978f, -0.487550160f,
    0.876070094f, -0.482183772f,
    0.879012226f, -0.476799230f,
    0.881921264f, -0.471396737f,
    0.884797098f, -0.465976496f,
    0.887639620f, -0.460538711f,
    0.890448723f, -0.455083587f,
    0.893224301f, -0.449611330f,
    0.895966250f, -0.444122145f,
    0.898674466f, -0.438616239f,
    0.901348847f, -0.433093819f,
    0.903989293f, -0.427555093f,
    0.906595705f, -0.422000271f,
    0.909167983f, -0.416429560f,
    0.911706032f, -0.410843171f,
    0.914209756f, -0.405241314f,
    0.916679060f, -0.399624200f,
    0.919113852f, -0.393992040f,
    0.921514039f, -0.388345047f,
    0.923879533f, -0.382683432f,
    0.926210242f, -0.377007410f,
    0.928506080f, -0.371317194f,
    0.930766961f, -0.365612998f,
    0.932992799f, -0.359895037f,
    0.935183510f, -0.354163525f,
    0.937339012f, -0.348418680f,
    0.939459224f, -0.342660717f,
    0.941544065f, -0.336889853f,
    0.943593458f, -0.331106306f,
    0.945607325f, -0.325310292f,
    0.947585591f, -0.319502031f,
    0.949528181f, -0.313681740f,
    0.951435021f, -0.307849640f,
    0.953306040f, -0.302005949f,
    0.955141168f, -0.296150888f,
    0.956940336f, -0.290284677f,
    0.958703475f, -0.284407537f,
    0.960430519f, -0.278519689f,
    0.962121404f, -0.272621355f,
    0.963776066f, -0.266712757f,
    0.965394442f, -0.260794118f,
    0.966976471f, -0.254865660f,
    0.968522094f, -0.248927606f,
    0.970031253f, -0.242980180f,
    0.971503891f, -0.237023606f,
    0.972939952f, -0.231058108f,
    0.974339383f, -0.225083911f,
    0.975702130f, -0.219101240f,
    0.977028143f, -0.213110320f,
    0.978317371f, -0.207111376f,
    0.979569766f, -0.201104635f,
    0.980785280f, -0.195090322f,
    0.981963869f, -0.189068664f,
    0.983105487f, -0.183039888f,
    0.984210092f, -0.177004220f,
    0.985277642f, -0.170961889f,
    0.986308097f, -0.164913120f,
    0.987301418f, -0.158858143f,
    0.988257568f, -0.152797185f,
    0.989176510f, -0.146730474f,
    0.990058210f, -0.140658239f,
    0.990902635f, -0.134580709f,
    0.991709754f, -0.128498111f,
    0.992479535f, -0.122410675f,
    0.993211949f, -0.116318631f,
    0.993906970f, -0.110222207f,
    0.994564571f, -0.104121634f,
    0.995184727f, -0.098017140f,
    0.995767414f, -0.091908956f,
    0.996312612f, -0.085797312f,
    0.996820299f, -0.079682438f,
    0.997290457f, -0.073564564f,
    0.997723067f, -0.067443920f,
    0.998118113f, -0.061320736f,
    0.998475581f, -0.055195244f,
    0.998795456f, -0.049067674f,
    0.999077728f, -0.042938257f,
    0.999322385f, -0.036807223f,
    0.999529418f, -0.030674803f,
    0.999698819f, -0.024541229f,
    0.999830582f, -0.018406730f,
    0.999924702f, -0.012271538f,
    0.999981175f, -0.006135885f
};
#endif // INCLUDE_FFT_TABLES_1024


#if defined(INCLUDE_FFT_TABLES_2048)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 2048, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_2048[4096] = {
    1.000000000f,  0.000000000f,
    0.999995294f,  0.003067957f,
    0.999981175f,  0.006135885f,
    0.999957645f,  0.009203755f,
    0.999924702f,  0.012271538f,
    0.999882347f,  0.015339206f,
    0.999830582f,  0.018406730f,
    0.999769405f,  0.021474080f,
    0.999698819f,  0.024541229f,
    0.999618822f,  0.027608146f,
    0.999529418f,  0.030674803f,
    0.999430605f,  0.033741172f,
    0.999322385f,  0.036807223f,
    0.999204759f,  0.039872928f,
    0.999077728f,  0.042938257f,
    0.998941293f,  0.046003182f,
    0.998795456f,  0.049067674f,
    0.998640218f,  0.052131705f,
    0.998475581f,  0.055195244f,
    0.998301545f,  0.058258265f,
    0.998118113f,  0.061320736f,
    0.997925286f,  0.064382631f,
    0.997723067f,  0.067443920f,
    0.997511456f,  0.070504573f,
    0.997290457f,  0.073564564f,
    0.997060070f,  0.076623861f,
    0.996820299f,  0.079682438f,
    0.996571146f,  0.082740265f,
    0.996312612f,  0.085797312f,
    0.996044701f,  0.088853553f,
    0.995767414f,  0.091908956f,
    0.995480755f,  0.094963495f,
    0.995184727f,  0.098017140f,
    0.994879331f,  0.101069863f,
    0.994564571f,  0.104121634f,
    0.994240449f,  0.107172425f,
    0.993906970f,  0.110222207f,
    0.993564136f,  0.113270952f,
    0.993211949f,  0.116318631f,
    0.992850414f,  0.119365215f,
    0.992479535f,  0.122410675f,
    0.992099313f,  0.125454983f,
    0.991709754f,  0.128498111f,
    0.991310860f,  0.131540029f,
    0.990902635f,  0.134580709f,
    0.990485084f,  0.137620122f,
    0.990058210f,  0.140658239f,
    0.989622017f,  0.143695033f,
    0.989176510f,  0.146730474f,
    0.988721692f,  0.149764535f,
    0.988257568f,  0.152797185f,
    0.987784142f,  0.155828398f,
    0.987301418f,  0.158858143f,
    0.986809402f,  0.161886394f,
    0.986308097f,  0.164913120f,
    0.985797509f,  0.167938295f,
    0.985277642f,  0.170961889f,
    0.984748502f,  0.173983873f,
    0.984210092f,  0.177004220f,
    0.983662419f,  0.180022901f,
    0.983105487f,  0.183039888f,
    0.982539302f,  0.186055152f,
    0.981963869f,  0.189068664f,
    0.981379193f,  0.192080397f,
    0.980785280f,  0.195090322f,
    0.980182136f,  0.198098411f,
    0.979569766f,  0.201104635f,
    0.978948175f,  0.204108966f,
    0.978317371f,  0.207111376f,
    0.977677358f,  0.210111837f,
    0.977028143f,  0.213110320f,
    0.976369731f,  0.216106797f,
    0.975702130f,  0.219101240f,
    0.975025345f,  0.222093621f,
    0.974339383f,  0.225083911f,
    0.973644250f,  0.228072083f,
    0.972939952f,  0.231058108f,
    0.972226497f,  0.234041959f,
    0.971503891f,  0.237023606f,
    0.970772141f,  0.240003022f,
    0.970031253f,  0.242980180f,
    0.969281235f,  0.245955050f,
    0.968522094f,  0.248927606f,
    0.967753837f,  0.251897818f,
    0.966976471f,  0.254865660f,
    0.966190003f,  0.257831102f,
    0.965394442f,  0.260794118f,
    0.964589793f,  0.263754679f,
    0.963776066f,  0.266712757f,
    0.962953267f,  0.269668326f,
    0.962121404f,  0.272621355f,
    0.961280486f,  0.275571819f,
    0.960430519f,  0.278519689f,
    0.959571513f,  0.281464938f,
    0.958703475f,  0.284407537f,
    0.957826413f,  0.287347460f,
    0.956940336f,  0.290284677f,
    0.956045251f,  0.293219163f,
    0.955141168f,  0.296150888f,
    0.954228095f,  0.299079826f,
    0.953306040f,  0.302005949f,
    0.952375013f,  0.304929230f,
    0.951435021f,  0.307849640f,
    0.950486074f,  0.310767153f,
    0.949528181f,  0.313681740f,
    0.948561350f,  0.316593376f,
    0.947585591f,  0.319502031f,
    0.946600913f,  0.322407679f,
    0.945607325f,  0.325310292f,
    0.944604837f,  0.328209844f,
    0.943593458f,  0.331106306f,
    0.942573198f,  0.333999651f,
    0.941544065f,  0.336889853f,
    0.940506071f,  0.339776884f,
    0.939459224f,  0.342660717f,
    0.938403534f,  0.345541325f,
    0.937339012f,  0.348418680f,
    0.936265667f,  0.351292756f,
    0.935183510f,  0.354163525f,
    0.934092550f,  0.357030961f,
    0.932992799f,  0.359895037f,
    0.931884266f,  0.362755724f,
    0.930766961f,  0.365612998f,
    0.929640896f,  0.368466830f,
    0.928506080f,  0.371317194f,
    0.927362526f,  0.374164063f,
    0.926210242f,  0.377007410f,
    0.925049241f,  0.379847209f,
    0.923879533f,  0.382683432f,
    0.922701128f,  0.385516054f,
    0.921514039f,  0.388345047f,
    0.920318277f,  0.391170384f,
    0.919113852f,  0.393992040f,
    0.917900776f,  0.396809987f,
    0.916679060f,  0.399624200f,
    0.915448716f,  0.402434651f,
    0.914209756f,  0.405241314f,
    0.912962190f,  0.408044163f,
    0.911706032f,  0.410843171f,
    0.910441292f,  0.413638312f,
    0.909167983f,  0.416429560f,
    0.907886116f,  0.419216888f,
    0.906595705f,  0.422000271f,
    0.905296759f,  0.424779681f,
    0.903989293f,  0.427555093f,
    0.902673318f,  0.430326481f,
    0.901348847f,  0.433093819f,
    0.900015892f,  0.435857080f,
    0.898674466f,  0.438616239f,
    0.897324581f,  0.441371269f,
    0.895966250f,  0.444122145f,
    0.894599486f,  0.446868840f,
    0.893224301f,  0.449611330f,
    0.891840709f,  0.452349587f,
    0.890448723f,  0.455083587f,
    0.889048356f,  0.457813304f,
    0.887639620f,  0.460538711f,
    0.886222530f,  0.463259784f,
    0.884797098f,  0.465976496f,
    0.883363339f,  0.468688822f,
    0.881921264f,  0.471396737f,
    0.880470889f,  0.474100215f,
    0.879012226f,  0.476799230f,
    0.877545290f,  0.479493758f,
    0.876070094f,  0.482183772f,
    0.874586652f,  0.484869248f,
    0.873094978f,  0.487550160f,
    0.871595087f,  0.490226483f,
    0.870086991f,  0.492898192f,
    0.868570706f,  0.495565262f,
    0.867046246f,  0.498227667f,
    0.865513624f,  0.500885383f,
    0.863972856f,  0.503538384f,
    0.862423956f,  0.506186645f,
    0.860866939f,  0.508830143f,
    0.859301818f,  0.511468850f,
    0.857728610f,  0.514102744f,
    0.856147328f,  0.516731799f,
    0.854557988f,  0.519355990f,
    0.852960605f,  0.521975293f,
    0.851355193f,  0.524589683f,
    0.849741768f,  0.527199135f,
    0.848120345f,  0.529803625f,
    0.846490939f,  0.532403128f,
    0.844853565f,  0.534997620f,
    0.843208240f,  0.537587076f,
    0.841554977f,  0.540171473f,
    0.839893794f,  0.542750785f,
    0.838224706f,  0.545324988f,
    0.836547727f,  0.547894059f,
    0.834862875f,  0.550457973f,
    0.833170165f,  0.553016706f,
    0.831469612f,  0.555570233f,
    0.829761234f,  0.558118531f,
    0.828045045f,  0.560661576f,
    0.826321063f,  0.563199344f,
    0.824589303f,  0.565731811f,
    0.822849781f,  0.568258953f,
    0.821102515f,  0.570780746f,
    0.819347520f,  0.573297167f,
    0.817584813f,  0.575808191f,
    0.815814411f,  0.578313796f,
    0.814036330f,  0.580813958f,
    0.812250587f,  0.583308653f,
    0.810457198f,  0.585797857f,
    0.808656182f,  0.588281548f,
    0.806847554f,  0.590759702f,
    0.805031331f,  0.593232295f,
    0.803207531f,  0.595699304f,
    0.801376172f,  0.598160707f,
    0.799537269f,  0.600616479f,
    0.797690841f,  0.603066599f,
    0.795836905f,  0.605511041f,
    0.793975478f,  0.607949785f,
    0.792106577f,  0.610382806f,
    0.790230221f,  0.612810082f,
    0.788346428f,  0.615231591f,
    0.786455214f,  0.617647308f,
    0.784556597f,  0.620057212f,
    0.782650596f,  0.622461279f,
    0.780737229f,  0.624859488f,
    0.778816512f,  0.627251815f,
    0.776888466f,  0.629638239f,
    0.774953107f,  0.632018736f,
    0.773010453f,  0.634393284f,
    0.771060524f,  0.636761861f,
    0.769103338f,  0.639124445f,
    0.767138912f,  0.641481013f,
    0.765167266f,  0.643831543f,
    0.763188417f,  0.646176013f,
    0.761202385f,  0.648514401f,
    0.759209189f,  0.650846685f,
    0.757208847f,  0.653172843f,
    0.755201377f,  0.655492853f,
    0.753186799f,  0.657806693f,
    0.751165132f,  0.660114342f,
    0.749136395f,  0.662415778f,
    0.747100606f,  0.664710978f,
    0.745057785f,  0.666999922f,
    0.743007952f,  0.669282588f,
    0.740951125f,  0.671558955f,
    0.738887324f,  0.673829000f,
    0.736816569f,  0.676092704f,
    0.734738878f,  0.678350043f,
    0.732654272f,  0.680600998f,
    0.730562769f,  0.682845546f,
    0.728464390f,  0.685083668f,
    0.726359155f,  0.687315341f,
    0.724247083f,  0.689540545f,
    0.722128194f,  0.691759258f,
    0.720002508f,  0.693971461f,
    0.717870045f,  0.696177131f,
    0.715730825f,  0.698376249f,
    0.713584869f,  0.700568794f,
    0.711432196f,  0.702754744f,
    0.709272826f,  0.704934080f,
    0.707106781f,  0.707106781f,
    0.704934080f,  0.709272826f,
    0.702754744f,  0.711432196f,
    0.700568794f,  0.713584869f,
    0.698376249f,  0.715730825f,
    0.696177131f,  0.717870045f,
    0.693971461f,  0.720002508f,
    0.691759258f,  0.722128194f,
    0.689540545f,  0.724247083f,
    0.687315341f,  0.726359155f,
    0.685083668f,  0.728464390f,
    0.682845546f,  0.730562769f,
    0.680600998f,  0.732654272f,
    0.678350043f,  0.734738878f,
    0.676092704f,  0.736816569f,
    0.673829000f,  0.738887324f,
    0.671558955f,  0.740951125f,
    0.669282588f,  0.743007952f,
    0.666999922f,  0.745057785f,
    0.664710978f,  0.747100606f,
    0.662415778f,  0.749136395f,
    0.660114342f,  0.751165132f,
    0.657806693f,  0.753186799f,
    0.655492853f,  0.755201377f,
    0.653172843f,  0.757208847f,
    0.650846685f,  0.759209189f,
    0.648514401f,  0.761202385f,
    0.646176013f,  0.763188417f,
    0.643831543f,  0.765167266f,
    0.641481013f,  0.767138912f,
    0.639124445f,  0.769103338f,
    0.636761861f,  0.771060524f,
    0.634393284f,  0.773010453f,
    0.632018736f,  0.774953107f,
    0.629638239f,  0.776888466f,
    0.627251815f,  0.778816512f,
    0.624859488f,  0.780737229f,
    0.622461279f,  0.782650596f,
    0.620057212f,  0.784556597f,
    0.617647308f,  0.786455214f,
    0.615231591f,  0.788346428f,
    0.612810082f,  0.790230221f,
    0.610382806f,  0.792106577f,
    0.607949785f,  0.793975478f,
    0.605511041f,  0.795836905f,
    0.603066599f,  0.797690841f,
    0.600616479f,  0.799537269f,
    0.598160707f,  0.801376172f,
    0.595699304f,  0.803207531f,
    0.593232295f,  0.805031331f,
    0.590759702f,  0.806847554f,
    0.588281548f,  0.808656182f,
    0.585797857f,  0.810457198f,
    0.583308653f,  0.812250587f,
    0.580813958f,  0.814036330f,
    0.578313796f,  0.815814411f,
    0.575808191f,  0.817584813f,
    0.573297167f,  0.819347520f,
    0.570780746f,  0.821102515f,
    0.568258953f,  0.822849781f,
    0.565731811f,  0.824589303f,
    0.563199344f,  0.826321063f,
    0.560661576f,  0.828045045f,
    0.558118531f,  0.829761234f,
    0.555570233f,  0.831469612f,
    0.553016706f,  0.833170165f,
    0.550457973f,  0.834862875f,
    0.547894059f,  0.836547727f,
    0.545324988f,  0.838224706f,
    0.542750785f,  0.839893794f,
    0.540171473f,  0.841554977f,
    0.537587076f,  0.843208240f,
    0.534997620f,  0.844853565f,
    0.532403128f,  0.846490939f,
    0.529803625f,  0.848120345f,
    0.527199135f,  0.849741768f,
    0.524589683f,  0.851355193f,
    0.521975293f,  0.852960605f,
    0.519355990f,  0.854557988f,
    0.516731799f,  0.856147328f,
    0.514102744f,  0.857728610f,
    0.511468850f,  0.859301818f,
    0.508830143f,  0.860866939f,
    0.506186645f,  0.862423956f,
    0.503538384f,  0.863972856f,
    0.500885383f,  0.865513624f,
    0.498227667f,  0.867046246f,
    0.495565262f,  0.868570706f,
    0.492898192f,  0.870086991f,
    0.490226483f,  0.871595087f,
    0.487550160f,  0.873094978f,
    0.484869248f,  0.874586652f,
    0.482183772f,  0.876070094f,
    0.479493758f,  0.877545290f,
    0.476799230f,  0.879012226f,
    0.474100215f,  0.880470889f,
    0.471396737f,  0.881921264f,
    0.468688822f,  0.883363339f,
    0.465976496f,  0.884797098f,
    0.463259784f,  0.886222530f,
    0.460538711f,  0.887639620f,
    0.457813304f,  0.889048356f,
    0.455083587f,  0.890448723f,
    0.452349587f,  0.891840709f,
    0.449611330f,  0.893224301f,
    0.446868840f,  0.894599486f,
    0.444122145f,  0.895966250f,
    0.441371269f,  0.897324581f,
    0.438616239f,  0.898674466f,
    0.435857080f,  0.900015892f,
    0.433093819f,  0.901348847f,
    0.430326481f,  0.902673318f,
    0.427555093f,  0.903989293f,
    0.424779681f,  0.905296759f,
    0.422000271f,  0.906595705f,
    0.419216888f,  0.907886116f,
    0.416429560f,  0.909167983f,
    0.413638312f,  0.910441292f,
    0.410843171f,  0.911706032f,
    0.408044163f,  0.912962190f,
    0.405241314f,  0.914209756f,
    0.402434651f,  0.915448716f,
    0.399624200f,  0.916679060f,
    0.396809987f,  0.917900776f,
    0.393992040f,  0.919113852f,
    0.391170384f,  0.920318277f,
    0.388345047f,  0.921514039f,
    0.385516054f,  0.922701128f,
    0.382683432f,  0.923879533f,
    0.379847209f,  0.925049241f,
    0.377007410f,  0.926210242f,
    0.374164063f,  0.927362526f,
    0.371317194f,  0.928506080f,
    0.368466830f,  0.929640896f,
    0.365612998f,  0.930766961f,
    0.362755724f,  0.931884266f,
    0.359895037f,  0.932992799f,
    0.357030961f,  0.934092550f,
    0.354163525f,  0.935183510f,
    0.351292756f,  0.936265667f,
    0.348418680f,  0.937339012f,
    0.345541325f,  0.938403534f,
    0.342660717f,  0.939459224f,
    0.339776884f,  0.940506071f,
    0.336889853f,  0.941544065f,
    0.333999651f,  0.942573198f,
    0.331106306f,  0.943593458f,
    0.328209844f,  0.944604837f,
    0.325310292f,  0.945607325f,
    0.322407679f,  0.946600913f,
    0.319502031f,  0.947585591f,
    0.316593376f,  0.948561350f,
    0.313681740f,  0.949528181f,
    0.310767153f,  0.950486074f,
    0.307849640f,  0.951435021f,
    0.304929230f,  0.952375013f,
    0.302005949f,  0.953306040f,
    0.299079826f,  0.954228095f,
    0.296150888f,  0.955141168f,
    0.293219163f,  0.956045251f,
    0.290284677f,  0.956940336f,
    0.287347460f,  0.957826413f,
    0.284407537f,  0.958703475f,
    0.281464938f,  0.959571513f,
    0.278519689f,  0.960430519f,
    0.275571819f,  0.961280486f,
    0.272621355f,  0.962121404f,
    0.269668326f,  0.962953267f,
    0.266712757f,  0.963776066f,
    0.263754679f,  0.964589793f,
    0.260794118f,  0.965394442f,
    0.257831102f,  0.966190003f,
    0.254865660f,  0.966976471f,
    0.251897818f,  0.967753837f,
    0.248927606f,  0.968522094f,
    0.245955050f,  0.969281235f,
    0.242980180f,  0.970031253f,
    0.240003022f,  0.970772141f,
    0.237023606f,  0.971503891f,
    0.234041959f,  0.972226497f,
    0.231058108f,  0.972939952f,
    0.228072083f,  0.973644250f,
    0.225083911f,  0.974339383f,
    0.222093621f,  0.975025345f,
    0.219101240f,  0.975702130f,
    0.216106797f,  0.976369731f,
    0.213110320f,  0.977028143f,
    0.210111837f,  0.977677358f,
    0.207111376f,  0.978317371f,
    0.204108966f,  0.978948175f,
    0.201104635f,  0.979569766f,
    0.198098411f,  0.980182136f,
    0.195090322f,  0.980785280f,
    0.192080397f,  0.981379193f,
    0.189068664f,  0.981963869f,
    0.186055152f,  0.982539302f,
    0.183039888f,  0.983105487f,
    0.180022901f,  0.983662419f,
    0.177004220f,  0.984210092f,
    0.173983873f,  0.984748502f,
    0.170961889f,  0.985277642f,
    0.167938295f,  0.985797509f,
    0.164913120f,  0.986308097f,
    0.161886394f,  0.986809402f,
    0.158858143f,  0.987301418f,
    0.155828398f,  0.987784142f,
    0.152797185f,  0.988257568f,
    0.149764535f,  0.988721692f,
    0.146730474f,  0.989176510f,
    0.143695033f,  0.989622017f,
    0.140658239f,  0.990058210f,
    0.137620122f,  0.990485084f,
    0.134580709f,  0.990902635f,
    0.131540029f,  0.991310860f,
    0.128498111f,  0.991709754f,
    0.125454983f,  0.992099313f,
    0.122410675f,  0.992479535f,
    0.119365215f,  0.992850414f,
    0.116318631f,  0.993211949f,
    0.113270952f,  0.993564136f,
    0.110222207f,  0.993906970f,
    0.107172425f,  0.994240449f,
    0.104121634f,  0.994564571f,
    0.101069863f,  0.994879331f,
    0.098017140f,  0.995184727f,
    0.094963495f,  0.995480755f,
    0.091908956f,  0.995767414f,
    0.088853553f,  0.996044701f,
    0.085797312f,  0.996312612f,
    0.082740265f,  0.996571146f,
    0.079682438f,  0.996820299f,
    0.076623861f,  0.997060070f,
    0.073564564f,  0.997290457f,
    0.070504573f,  0.997511456f,
    0.067443920f,  0.997723067f,
    0.064382631f,  0.997925286f,
    0.061320736f,  0.998118113f,
    0.058258265f,  0.998301545f,
    0.055195244f,  0.998475581f,
    0.052131705f,  0.998640218f,
    0.049067674f,  0.998795456f,
    0.046003182f,  0.998941293f,
    0.042938257f,  0.999077728f,
    0.039872928f,  0.999204759f,
    0.036807223f,  0.999322385f,
    0.033741172f,  0.999430605f,
    0.030674803f,  0.999529418f,
    0.027608146f,  0.999618822f,
    0.024541229f,  0.999698819f,
    0.021474080f,  0.999769405f,
    0.018406730f,  0.999830582f,
    0.015339206f,  0.999882347f,
    0.012271538f,  0.999924702f,
    0.009203755f,  0.999957645f,
    0.006135885f,  0.999981175f,
    0.003067957f,  0.999995294f,
    0.000000000f,  1.000000000f,
   -0.003067957f,  0.999995294f,
   -0.006135885f,  0.999981175f,
   -0.009203755f,  0.999957645f,
   -0.012271538f,  0.999924702f,
   -0.015339206f,  0.999882347f,
   -0.018406730f,  0.999830582f,
   -0.021474080f,  0.999769405f,
   -0.024541229f,  0.999698819f,
   -0.027608146f,  0.999618822f,
   -0.030674803f,  0.999529418f,
   -0.033741172f,  0.999430605f,
   -0.036807223f,  0.999322385f,
   -0.039872928f,  0.999204759f,
   -0.042938257f,  0.999077728f,
   -0.046003182f,  0.998941293f,
   -0.049067674f,  0.998795456f,
   -0.052131705f,  0.998640218f,
   -0.055195244f,  0.998475581f,
   -0.058258265f,  0.998301545f,
   -0.061320736f,  0.998118113f,
   -0.064382631f,  0.997925286f,
   -0.067443920f,  0.997723067f,
   -0.070504573f,  0.997511456f,
   -0.073564564f,  0.997290457f,
   -0.076623861f,  0.997060070f,
   -0.079682438f,  0.996820299f,
   -0.082740265f,  0.996571146f,
   -0.085797312f,  0.996312612f,
   -0.088853553f,  0.996044701f,
   -0.091908956f,  0.995767414f,
   -0.094963495f,  0.995480755f,
   -0.098017140f,  0.995184727f,
   -0.101069863f,  0.994879331f,
   -0.104121634f,  0.994564571f,
   -0.107172425f,  0.994240449f,
   -0.110222207f,  0.993906970f,
   -0.113270952f,  0.993564136f,
   -0.116318631f,  0.993211949f,
   -0.119365215f,  0.992850414f,
   -0.122410675f,  0.992479535f,
   -0.125454983f,  0.992099313f,
   -0.128498111f,  0.991709754f,
   -0.131540029f,  0.991310860f,
   -0.134580709f,  0.990902635f,
   -0.137620122f,  0.990485084f,
   -0.140658239f,  0.990058210f,
   -0.143695033f,  0.989622017f,
   -0.146730474f,  0.989176510f,
   -0.149764535f,  0.988721692f,
   -0.152797185f,  0.988257568f,
   -0.155828398f,  0.987784142f,
   -0.158858143f,  0.987301418f,
   -0.161886394f,  0.986809402f,
   -0.164913120f,  0.986308097f,
   -0.167938295f,  0.985797509f,
   -0.170961889f,  0.985277642f,
   -0.173983873f,  0.984748502f,
   -0.177004220f,  0.984210092f,
   -0.180022901f,  0.983662419f,
   -0.183039888f,  0.983105487f,
   -0.186055152f,  0.982539302f,
   -0.189068664f,  0.981963869f,
   -0.192080397f,  0.981379193f,
   -0.195090322f,  0.980785280f,
   -0.198098411f,  0.980182136f,
   -0.201104635f,  0.979569766f,
   -0.204108966f,  0.978948175f,
   -0.207111376f,  0.978317371f,
   -0.210111837f,  0.977677358f,
   -0.213110320f,  0.977028143f,
   -0.216106797f,  0.976369731f,
   -0.219101240f,  0.975702130f,
   -0.222093621f,  0.975025345f,
   -0.225083911f,  0.974339383f,
   -0.228072083f,  0.973644250f,
   -0.231058108f,  0.972939952f,
   -0.234041959f,  0.972226497f,
   -0.237023606f,  0.971503891f,
   -0.240003022f,  0.970772141f,
   -0.242980180f,  0.970031253f,
   -0.245955050f,  0.969281235f,
   -0.248927606f,  0.968522094f,
   -0.251897818f,  0.967753837f,
   -0.254865660f,  0.966976471f,
   -0.257831102f,  0.966190003f,
   -0.260794118f,  0.965394442f,
   -0.263754679f,  0.964589793f,
   -0.266712757f,  0.963776066f,
   -0.269668326f,  0.962953267f,
   -0.272621355f,  0.962121404f,
   -0.275571819f,  0.961280486f,
   -0.278519689f,  0.960430519f,
   -0.281464938f,  0.959571513f,
   -0.284407537f,  0.958703475f,
   -0.287347460f,  0.957826413f,
   -0.290284677f,  0.956940336f,
   -0.293219163f,  0.956045251f,
   -0.296150888f,  0.955141168f,
   -0.299079826f,  0.954228095f,
   -0.302005949f,  0.953306040f,
   -0.304929230f,  0.952375013f,
   -0.307849640f,  0.951435021f,
   -0.310767153f,  0.950486074f,
   -0.313681740f,  0.949528181f,
   -0.316593376f,  0.948561350f,
   -0.319502031f,  0.947585591f,
   -0.322407679f,  0.946600913f,
   -0.325310292f,  0.945607325f,
   -0.328209844f,  0.944604837f,
   -0.331106306f,  0.943593458f,
   -0.333999651f,  0.942573198f,
   -0.336889853f,  0.941544065f,
   -0.339776884f,  0.940506071f,
   -0.342660717f,  0.939459224f,
   -0.345541325f,  0.938403534f,
   -0.348418680f,  0.937339012f,
   -0.351292756f,  0.936265667f,
   -0.354163525f,  0.935183510f,
   -0.357030961f,  0.934092550f,
   -0.359895037f,  0.932992799f,
   -0.362755724f,  0.931884266f,
   -0.365612998f,  0.930766961f,
   -0.368466830f,  0.929640896f,
   -0.371317194f,  0.928506080f,
   -0.374164063f,  0.927362526f,
   -0.377007410f,  0.926210242f,
   -0.379847209f,  0.925049241f,
   -0.382683432f,  0.923879533f,
   -0.385516054f,  0.922701128f,
   -0.388345047f,  0.921514039f,
   -0.391170384f,  0.920318277f,
   -0.393992040f,  0.919113852f,
   -0.396809987f,  0.917900776f,
   -0.399624200f,  0.916679060f,
   -0.402434651f,  0.915448716f,
   -0.405241314f,  0.914209756f,
   -0.408044163f,  0.912962190f,
   -0.410843171f,  0.911706032f,
   -0.413638312f,  0.910441292f,
   -0.416429560f,  0.909167983f,
   -0.419216888f,  0.907886116f,
   -0.422000271f,  0.906595705f,
   -0.424779681f,  0.905296759f,
   -0.427555093f,  0.903989293f,
   -0.430326481f,  0.902673318f,
   -0.433093819f,  0.901348847f,
   -0.435857080f,  0.900015892f,
   -0.438616239f,  0.898674466f,
   -0.441371269f,  0.897324581f,
   -0.444122145f,  0.895966250f,
   -0.446868840f,  0.894599486f,
   -0.449611330f,  0.893224301f,
   -0.452349587f,  0.891840709f,
   -0.455083587f,  0.890448723f,
   -0.457813304f,  0.889048356f,
   -0.460538711f,  0.887639620f,
   -0.463259784f,  0.886222530f,
   -0.465976496f,  0.884797098f,
   -0.468688822f,  0.883363339f,
   -0.471396737f,  0.881921264f,
   -0.474100215f,  0.880470889f,
   -0.476799230f,  0.879012226f,
   -0.479493758f,  0.877545290f,
   -0.482183772f,  0.876070094f,
   -0.484869248f,  0.874586652f,
   -0.487550160f,  0.873094978f,
   -0.490226483f,  0.871595087f,
   -0.492898192f,  0.870086991f,
   -0.495565262f,  0.868570706f,
   -0.498227667f,  0.867046246f,
   -0.500885383f,  0.865513624f,
   -0.503538384f,  0.863972856f,
   -0.506186645f,  0.862423956f,
   -0.508830143f,  0.860866939f,
   -0.511468850f,  0.859301818f,
   -0.514102744f,  0.857728610f,
   -0.516731799f,  0.856147328f,
   -0.519355990f,  0.854557988f,
   -0.521975293f,  0.852960605f,
   -0.524589683f,  0.851355193f,
   -0.527199135f,  0.849741768f,
   -0.529803625f,  0.848120345f,
   -0.532403128f,  0.846490939f,
   -0.534997620f,  0.844853565f,
   -0.537587076f,  0.843208240f,
   -0.540171473f,  0.841554977f,
   -0.542750785f,  0.839893794f,
   -0.545324988f,  0.838224706f,
   -0.547894059f,  0.836547727f,
   -0.550457973f,  0.834862875f,
   -0.553016706f,  0.833170165f,
   -0.555570233f,  0.831469612f,
   -0.558118531f,  0.829761234f,
   -0.560661576f,  0.828045045f,
   -0.563199344f,  0.826321063f,
   -0.565731811f,  0.824589303f,
   -0.568258953f,  0.822849781f,
   -0.570780746f,  0.821102515f,
   -0.573297167f,  0.819347520f,
   -0.575808191f,  0.817584813f,
   -0.578313796f,  0.815814411f,
   -0.580813958f,  0.814036330f,
   -0.583308653f,  0.812250587f,
   -0.585797857f,  0.810457198f,
   -0.588281548f,  0.808656182f,
   -0.590759702f,  0.806847554f,
   -0.593232295f,  0.805031331f,
   -0.595699304f,  0.803207531f,
   -0.598160707f,  0.801376172f,
   -0.600616479f,  0.799537269f,
   -0.603066599f,  0.797690841f,
   -0.605511041f,  0.795836905f,
   -0.607949785f,  0.793975478f,
   -0.610382806f,  0.792106577f,
   -0.612810082f,  0.790230221f,
   -0.615231591f,  0.788346428f,
   -0.617647308f,  0.786455214f,
   -0.620057212f,  0.784556597f,
   -0.622461279f,  0.782650596f,
   -0.624859488f,  0.780737229f,
   -0.627251815f,  0.778816512f,
   -0.629638239f,  0.776888466f,
   -0.632018736f,  0.774953107f,
   -0.634393284f,  0.773010453f,
   -0.636761861f,  0.771060524f,
   -0.639124445f,  0.769103338f,
   -0.641481013f,  0.767138912f,
   -0.643831543f,  0.765167266f,
   -0.646176013f,  0.763188417f,
   -0.648514401f,  0.761202385f,
   -0.650846685f,  0.759209189f,
   -0.653172843f,  0.757208847f,
   -0.655492853f,  0.755201377f,
   -0.657806693f,  0.753186799f,
   -0.660114342f,  0.751165132f,
   -0.662415778f,  0.749136395f,
   -0.664710978f,  0.747100606f,
   -0.666999922f,  0.745057785f,
   -0.669282588f,  0.743007952f,
   -0.671558955f,  0.740951125f,
   -0.673829000f,  0.738887324f,
   -0.676092704f,  0.736816569f,
   -0.678350043f,  0.734738878f,
   -0.680600998f,  0.732654272f,
   -0.682845546f,  0.730562769f,
   -0.685083668f,  0.728464390f,
   -0.687315341f,  0.726359155f,
   -0.689540545f,  0.724247083f,
   -0.691759258f,  0.722128194f,
   -0.693971461f,  0.720002508f,
   -0.696177131f,  0.717870045f,
   -0.698376249f,  0.715730825f,
   -0.700568794f,  0.713584869f,
   -0.702754744f,  0.711432196f,
   -0.704934080f,  0.709272826f,
   -0.707106781f,  0.707106781f,
   -0.709272826f,  0.704934080f,
   -0.711432196f,  0.702754744f,
   -0.713584869f,  0.700568794f,
   -0.715730825f,  0.698376249f,
   -0.717870045f,  0.696177131f,
   -0.720002508f,  0.693971461f,
   -0.722128194f,  0.691759258f,
   -0.724247083f,  0.689540545f,
   -0.726359155f,  0.687315341f,
   -0.728464390f,  0.685083668f,
   -0.730562769f,  0.682845546f,
   -0.732654272f,  0.680600998f,
   -0.734738878f,  0.678350043f,
   -0.736816569f,  0.676092704f,
   -0.738887324f,  0.673829000f,
   -0.740951125f,  0.671558955f,
   -0.743007952f,  0.669282588f,
   -0.745057785f,  0.666999922f,
   -0.747100606f,  0.664710978f,
   -0.749136395f,  0.662415778f,
   -0.751165132f,  0.660114342f,
   -0.753186799f,  0.657806693f,
   -0.755201377f,  0.655492853f,
   -0.757208847f,  0.653172843f,
   -0.759209189f,  0.650846685f,
   -0.761202385f,  0.648514401f,
   -0.763188417f,  0.646176013f,
   -0.765167266f,  0.643831543f,
   -0.767138912f,  0.641481013f,
   -0.769103338f,  0.639124445f,
   -0.771060524f,  0.636761861f,
   -0.773010453f,  0.634393284f,
   -0.774953107f,  0.632018736f,
   -0.776888466f,  0.629638239f,
   -0.778816512f,  0.627251815f,
   -0.780737229f,  0.624859488f,
   -0.782650596f,  0.622461279f,
   -0.784556597f,  0.620057212f,
   -0.786455214f,  0.617647308f,
   -0.788346428f,  0.615231591f,
   -0.790230221f,  0.612810082f,
   -0.792106577f,  0.610382806f,
   -0.793975478f,  0.607949785f,
   -0.795836905f,  0.605511041f,
   -0.797690841f,  0.603066599f,
   -0.799537269f,  0.600616479f,
   -0.801376172f,  0.598160707f,
   -0.803207531f,  0.595699304f,
   -0.805031331f,  0.593232295f,
   -0.806847554f,  0.590759702f,
   -0.808656182f,  0.588281548f,
   -0.810457198f,  0.585797857f,
   -0.812250587f,  0.583308653f,
   -0.814036330f,  0.580813958f,
   -0.815814411f,  0.578313796f,
   -0.817584813f,  0.575808191f,
   -0.819347520f,  0.573297167f,
   -0.821102515f,  0.570780746f,
   -0.822849781f,  0.568258953f,
   -0.824589303f,  0.565731811f,
   -0.826321063f,  0.563199344f,
   -0.828045045f,  0.560661576f,
   -0.829761234f,  0.558118531f,
   -0.831469612f,  0.555570233f,
   -0.833170165f,  0.553016706f,
   -0.834862875f,  0.550457973f,
   -0.836547727f,  0.547894059f,
   -0.838224706f,  0.545324988f,
   -0.839893794f,  0.542750785f,
   -0.841554977f,  0.540171473f,
   -0.843208240f,  0.537587076f,
   -0.844853565f,  0.534997620f,
   -0.846490939f,  0.532403128f,
   -0.848120345f,  0.529803625f,
   -0.849741768f,  0.527199135f,
   -0.851355193f,  0.524589683f,
   -0.852960605f,  0.521975293f,
   -0.854557988f,  0.519355990f,
   -0.856147328f,  0.516731799f,
   -0.857728610f,  0.514102744f,
   -0.859301818f,  0.511468850f,
   -0.860866939f,  0.508830143f,
   -0.862423956f,  0.506186645f,
   -0.863972856f,  0.503538384f,
   -0.865513624f,  0.500885383f,
   -0.867046246f,  0.498227667f,
   -0.868570706f,  0.495565262f,
   -0.870086991f,  0.492898192f,
   -0.871595087f,  0.490226483f,
   -0.873094978f,  0.487550160f,
   -0.874586652f,  0.484869248f,
   -0.876070094f,  0.482183772f,
   -0.877545290f,  0.479493758f,
   -0.879012226f,  0.476799230f,
   -0.880470889f,  0.474100215f,
   -0.881921264f,  0.471396737f,
   -0.883363339f,  0.468688822f,
   -0.884797098f,  0.465976496f,
   -0.886222530f,  0.463259784f,
   -0.887639620f,  0.460538711f,
   -0.889048356f,  0.457813304f,
   -0.890448723f,  0.455083587f,
   -0.891840709f,  0.452349587f,
   -0.893224301f,  0.449611330f,
   -0.894599486f,  0.446868840f,
   -0.895966250f,  0.444122145f,
   -0.897324581f,  0.441371269f,
   -0.898674466f,  0.438616239f,
   -0.900015892f,  0.435857080f,
   -0.901348847f,  0.433093819f,
   -0.902673318f,  0.430326481f,
   -0.903989293f,  0.427555093f,
   -0.905296759f,  0.424779681f,
   -0.906595705f,  0.422000271f,
   -0.907886116f,  0.419216888f,
   -0.909167983f,  0.416429560f,
   -0.910441292f,  0.413638312f,
   -0.911706032f,  0.410843171f,
   -0.912962190f,  0.408044163f,
   -0.914209756f,  0.405241314f,
   -0.915448716f,  0.402434651f,
   -0.916679060f,  0.399624200f,
   -0.917900776f,  0.396809987f,
   -0.919113852f,  0.393992040f,
   -0.920318277f,  0.391170384f,
   -0.921514039f,  0.388345047f,
   -0.922701128f,  0.385516054f,
   -0.923879533f,  0.382683432f,
   -0.925049241f,  0.379847209f,
   -0.926210242f,  0.377007410f,
   -0.927362526f,  0.374164063f,
   -0.928506080f,  0.371317194f,
   -0.929640896f,  0.368466830f,
   -0.930766961f,  0.365612998f,
   -0.931884266f,  0.362755724f,
   -0.932992799f,  0.359895037f,
   -0.934092550f,  0.357030961f,
   -0.935183510f,  0.354163525f,
   -0.936265667f,  0.351292756f,
   -0.937339012f,  0.348418680f,
   -0.938403534f,  0.345541325f,
   -0.939459224f,  0.342660717f,
   -0.940506071f,  0.339776884f,
   -0.941544065f,  0.336889853f,
   -0.942573198f,  0.333999651f,
   -0.943593458f,  0.331106306f,
   -0.944604837f,  0.328209844f,
   -0.945607325f,  0.325310292f,
   -0.946600913f,  0.322407679f,
   -0.947585591f,  0.319502031f,
   -0.948561350f,  0.316593376f,
   -0.949528181f,  0.313681740f,
   -0.950486074f,  0.310767153f,
   -0.951435021f,  0.307849640f,
   -0.952375013f,  0.304929230f,
   -0.953306040f,  0.302005949f,
   -0.954228095f,  0.299079826f,
   -0.955141168f,  0.296150888f,
   -0.956045251f,  0.293219163f,
   -0.956940336f,  0.290284677f,
   -0.957826413f,  0.287347460f,
   -0.958703475f,  0.284407537f,
   -0.959571513f,  0.281464938f,
   -0.960430519f,  0.278519689f,
   -0.961280486f,  0.275571819f,
   -0.962121404f,  0.272621355f,
   -0.962953267f,  0.269668326f,
   -0.963776066f,  0.266712757f,
   -0.964589793f,  0.263754679f,
   -0.965394442f,  0.260794118f,
   -0.966190003f,  0.257831102f,
   -0.966976471f,  0.254865660f,
   -0.967753837f,  0.251897818f,
   -0.968522094f,  0.248927606f,
   -0.969281235f,  0.245955050f,
   -0.970031253f,  0.242980180f,
   -0.970772141f,  0.240003022f,
   -0.971503891f,  0.237023606f,
   -0.972226497f,  0.234041959f,
   -0.972939952f,  0.231058108f,
   -0.973644250f,  0.228072083f,
   -0.974339383f,  0.225083911f,
   -0.975025345f,  0.222093621f,
   -0.975702130f,  0.219101240f,
   -0.976369731f,  0.216106797f,
   -0.977028143f,  0.213110320f,
   -0.977677358f,  0.210111837f,
   -0.978317371f,  0.207111376f,
   -0.978948175f,  0.204108966f,
   -0.979569766f,  0.201104635f,
   -0.980182136f,  0.198098411f,
   -0.980785280f,  0.195090322f,
   -0.981379193f,  0.192080397f,
   -0.981963869f,  0.189068664f,
   -0.982539302f,  0.186055152f,
   -0.983105487f,  0.183039888f,
   -0.983662419f,  0.180022901f,
   -0.984210092f,  0.177004220f,
   -0.984748502f,  0.173983873f,
   -0.985277642f,  0.170961889f,
   -0.985797509f,  0.167938295f,
   -0.986308097f,  0.164913120f,
   -0.986809402f,  0.161886394f,
   -0.987301418f,  0.158858143f,
   -0.987784142f,  0.155828398f,
   -0.988257568f,  0.152797185f,
   -0.988721692f,  0.149764535f,
   -0.989176510f,  0.146730474f,
   -0.989622017f,  0.143695033f,
   -0.990058210f,  0.140658239f,
   -0.990485084f,  0.137620122f,
   -0.990902635f,  0.134580709f,
   -0.991310860f,  0.131540029f,
   -0.991709754f,  0.128498111f,
   -0.992099313f,  0.125454983f,
   -0.992479535f,  0.122410675f,
   -0.992850414f,  0.119365215f,
   -0.993211949f,  0.116318631f,
   -0.993564136f,  0.113270952f,
   -0.993906970f,  0.110222207f,
   -0.994240449f,  0.107172425f,
   -0.994564571f,  0.104121634f,
   -0.994879331f,  0.101069863f,
   -0.995184727f,  0.098017140f,
   -0.995480755f,  0.094963495f,
   -0.995767414f,  0.091908956f,
   -0.996044701f,  0.088853553f,
   -0.996312612f,  0.085797312f,
   -0.996571146f,  0.082740265f,
   -0.996820299f,  0.079682438f,
   -0.997060070f,  0.076623861f,
   -0.997290457f,  0.073564564f,
   -0.997511456f,  0.070504573f,
   -0.997723067f,  0.067443920f,
   -0.997925286f,  0.064382631f,
   -0.998118113f,  0.061320736f,
   -0.998301545f,  0.058258265f,
   -0.998475581f,  0.055195244f,
   -0.998640218f,  0.052131705f,
   -0.998795456f,  0.049067674f,
   -0.998941293f,  0.046003182f,
   -0.999077728f,  0.042938257f,
   -0.999204759f,  0.039872928f,
   -0.999322385f,  0.036807223f,
   -0.999430605f,  0.033741172f,
   -0.999529418f,  0.030674803f,
   -0.999618822f,  0.027608146f,
   -0.999698819f,  0.024541229f,
   -0.999769405f,  0.021474080f,
   -0.999830582f,  0.018406730f,
   -0.999882347f,  0.015339206f,
   -0.999924702f,  0.012271538f,
   -0.999957645f,  0.009203755f,
   -0.999981175f,  0.006135885f,
   -0.999995294f,  0.003067957f,
   -1.000000000f,  0.000000000f,
   -0.999995294f, -0.003067957f,
   -0.999981175f, -0.006135885f,
   -0.999957645f, -0.009203755f,
   -0.999924702f, -0.012271538f,
   -0.999882347f, -0.015339206f,
   -0.999830582f, -0.018406730f,
   -0.999769405f, -0.021474080f,
   -0.999698819f, -0.024541229f,
   -0.999618822f, -0.027608146f,
   -0.999529418f, -0.030674803f,
   -0.999430605f, -0.033741172f,
   -0.999322385f, -0.036807223f,
   -0.999204759f, -0.039872928f,
   -0.999077728f, -0.042938257f,
   -0.998941293f, -0.046003182f,
   -0.998795456f, -0.049067674f,
   -0.998640218f, -0.052131705f,
   -0.998475581f, -0.055195244f,
   -0.998301545f, -0.058258265f,
   -0.998118113f, -0.061320736f,
   -0.997925286f, -0.064382631f,
   -0.997723067f, -0.067443920f,
   -0.997511456f, -0.070504573f,
   -0.997290457f, -0.073564564f,
   -0.997060070f, -0.076623861f,
   -0.996820299f, -0.079682438f,
   -0.996571146f, -0.082740265f,
   -0.996312612f, -0.085797312f,
   -0.996044701f, -0.088853553f,
   -0.995767414f, -0.091908956f,
   -0.995480755f, -0.094963495f,
   -0.995184727f, -0.098017140f,
   -0.994879331f, -0.101069863f,
   -0.994564571f, -0.104121634f,
   -0.994240449f, -0.107172425f,
   -0.993906970f, -0.110222207f,
   -0.993564136f, -0.113270952f,
   -0.993211949f, -0.116318631f,
   -0.992850414f, -0.119365215f,
   -0.992479535f, -0.122410675f,
   -0.992099313f, -0.125454983f,
   -0.991709754f, -0.128498111f,
   -0.991310860f, -0.131540029f,
   -0.990902635f, -0.134580709f,
   -0.990485084f, -0.137620122f,
   -0.990058210f, -0.140658239f,
   -0.989622017f, -0.143695033f,
   -0.989176510f, -0.146730474f,
   -0.988721692f, -0.149764535f,
   -0.988257568f, -0.152797185f,
   -0.987784142f, -0.155828398f,
   -0.987301418f, -0.158858143f,
   -0.986809402f, -0.161886394f,
   -0.986308097f, -0.164913120f,
   -0.985797509f, -0.167938295f,
   -0.985277642f, -0.170961889f,
   -0.984748502f, -0.173983873f,
   -0.984210092f, -0.177004220f,
   -0.983662419f, -0.180022901f,
   -0.983105487f, -0.183039888f,
   -0.982539302f, -0.186055152f,
   -0.981963869f, -0.189068664f,
   -0.981379193f, -0.192080397f,
   -0.980785280f, -0.195090322f,
   -0.980182136f, -0.198098411f,
   -0.979569766f, -0.201104635f,
   -0.978948175f, -0.204108966f,
   -0.978317371f, -0.207111376f,
   -0.977677358f, -0.210111837f,
   -0.977028143f, -0.213110320f,
   -0.976369731f, -0.216106797f,
   -0.975702130f, -0.219101240f,
   -0.975025345f, -0.222093621f,
   -0.974339383f, -0.225083911f,
   -0.973644250f, -0.228072083f,
   -0.972939952f, -0.231058108f,
   -0.972226497f, -0.234041959f,
   -0.971503891f, -0.237023606f,
   -0.970772141f, -0.240003022f,
   -0.970031253f, -0.242980180f,
   -0.969281235f, -0.245955050f,
   -0.968522094f, -0.248927606f,
   -0.967753837f, -0.251897818f,
   -0.966976471f, -0.254865660f,
   -0.966190003f, -0.257831102f,
   -0.965394442f, -0.260794118f,
   -0.964589793f, -0.263754679f,
   -0.963776066f, -0.266712757f,
   -0.962953267f, -0.269668326f,
   -0.962121404f, -0.272621355f,
   -0.961280486f, -0.275571819f,
   -0.960430519f, -0.278519689f,
   -0.959571513f, -0.281464938f,
   -0.958703475f, -0.284407537f,
   -0.957826413f, -0.287347460f,
   -0.956940336f, -0.290284677f,
   -0.956045251f, -0.293219163f,
   -0.955141168f, -0.296150888f,
   -0.954228095f, -0.299079826f,
   -0.953306040f, -0.302005949f,
   -0.952375013f, -0.304929230f,
   -0.951435021f, -0.307849640f,
   -0.950486074f, -0.310767153f,
   -0.949528181f, -0.313681740f,
   -0.948561350f, -0.316593376f,
   -0.947585591f, -0.319502031f,
   -0.946600913f, -0.322407679f,
   -0.945607325f, -0.325310292f,
   -0.944604837f, -0.328209844f,
   -0.943593458f, -0.331106306f,
   -0.942573198f, -0.333999651f,
   -0.941544065f, -0.336889853f,
   -0.940506071f, -0.339776884f,
   -0.939459224f, -0.342660717f,
   -0.938403534f, -0.345541325f,
   -0.937339012f, -0.348418680f,
   -0.936265667f, -0.351292756f,
   -0.935183510f, -0.354163525f,
   -0.934092550f, -0.357030961f,
   -0.932992799f, -0.359895037f,
   -0.931884266f, -0.362755724f,
   -0.930766961f, -0.365612998f,
   -0.929640896f, -0.368466830f,
   -0.928506080f, -0.371317194f,
   -0.927362526f, -0.374164063f,
   -0.926210242f, -0.377007410f,
   -0.925049241f, -0.379847209f,
   -0.923879533f, -0.382683432f,
   -0.922701128f, -0.385516054f,
   -0.921514039f, -0.388345047f,
   -0.920318277f, -0.391170384f,
   -0.919113852f, -0.393992040f,
   -0.917900776f, -0.396809987f,
   -0.916679060f, -0.399624200f,
   -0.915448716f, -0.402434651f,
   -0.914209756f, -0.405241314f,
   -0.912962190f, -0.408044163f,
   -0.911706032f, -0.410843171f,
   -0.910441292f, -0.413638312f,
   -0.909167983f, -0.416429560f,
   -0.907886116f, -0.419216888f,
   -0.906595705f, -0.422000271f,
   -0.905296759f, -0.424779681f,
   -0.903989293f, -0.427555093f,
   -0.902673318f, -0.430326481f,
   -0.901348847f, -0.433093819f,
   -0.900015892f, -0.435857080f,
   -0.898674466f, -0.438616239f,
   -0.897324581f, -0.441371269f,
   -0.895966250f, -0.444122145f,
   -0.894599486f, -0.446868840f,
   -0.893224301f, -0.449611330f,
   -0.891840709f, -0.452349587f,
   -0.890448723f, -0.455083587f,
   -0.889048356f, -0.457813304f,
   -0.887639620f, -0.460538711f,
   -0.886222530f, -0.463259784f,
   -0.884797098f, -0.465976496f,
   -0.883363339f, -0.468688822f,
   -0.881921264f, -0.471396737f,
   -0.880470889f, -0.474100215f,
   -0.879012226f, -0.476799230f,
   -0.877545290f, -0.479493758f,
   -0.876070094f, -0.482183772f,
   -0.874586652f, -0.484869248f,
   -0.873094978f, -0.487550160f,
   -0.871595087f, -0.490226483f,
   -0.870086991f, -0.492898192f,
   -0.868570706f, -0.495565262f,
   -0.867046246f, -0.498227667f,
   -0.865513624f, -0.500885383f,
   -0.863972856f, -0.503538384f,
   -0.862423956f, -0.506186645f,
   -0.860866939f, -0.508830143f,
   -0.859301818f, -0.511468850f,
   -0.857728610f, -0.514102744f,
   -0.856147328f, -0.516731799f,
   -0.854557988f, -0.519355990f,
   -0.852960605f, -0.521975293f,
   -0.851355193f, -0.524589683f,
   -0.849741768f, -0.527199135f,
   -0.848120345f, -0.529803625f,
   -0.846490939f, -0.532403128f,
   -0.844853565f, -0.534997620f,
   -0.843208240f, -0.537587076f,
   -0.841554977f, -0.540171473f,
   -0.839893794f, -0.542750785f,
   -0.838224706f, -0.545324988f,
   -0.836547727f, -0.547894059f,
   -0.834862875f, -0.550457973f,
   -0.833170165f, -0.553016706f,
   -0.831469612f, -0.555570233f,
   -0.829761234f, -0.558118531f,
   -0.828045045f, -0.560661576f,
   -0.826321063f, -0.563199344f,
   -0.824589303f, -0.565731811f,
   -0.822849781f, -0.568258953f,
   -0.821102515f, -0.570780746f,
   -0.819347520f, -0.573297167f,
   -0.817584813f, -0.575808191f,
   -0.815814411f, -0.578313796f,
   -0.814036330f, -0.580813958f,
   -0.812250587f, -0.583308653f,
   -0.810457198f, -0.585797857f,
   -0.808656182f, -0.588281548f,
   -0.806847554f, -0.590759702f,
   -0.805031331f, -0.593232295f,
   -0.803207531f, -0.595699304f,
   -0.801376172f, -0.598160707f,
   -0.799537269f, -0.600616479f,
   -0.797690841f, -0.603066599f,
   -0.795836905f, -0.605511041f,
   -0.793975478f, -0.607949785f,
   -0.792106577f, -0.610382806f,
   -0.790230221f, -0.612810082f,
   -0.788346428f, -0.615231591f,
   -0.786455214f, -0.617647308f,
   -0.784556597f, -0.620057212f,
   -0.782650596f, -0.622461279f,
   -0.780737229f, -0.624859488f,
   -0.778816512f, -0.627251815f,
   -0.776888466f, -0.629638239f,
   -0.774953107f, -0.632018736f,
   -0.773010453f, -0.634393284f,
   -0.771060524f, -0.636761861f,
   -0.769103338f, -0.639124445f,
   -0.767138912f, -0.641481013f,
   -0.765167266f, -0.643831543f,
   -0.763188417f, -0.646176013f,
   -0.761202385f, -0.648514401f,
   -0.759209189f, -0.650846685f,
   -0.757208847f, -0.653172843f,
   -0.755201377f, -0.655492853f,
   -0.753186799f, -0.657806693f,
   -0.751165132f, -0.660114342f,
   -0.749136395f, -0.662415778f,
   -0.747100606f, -0.664710978f,
   -0.745057785f, -0.666999922f,
   -0.743007952f, -0.669282588f,
   -0.740951125f, -0.671558955f,
   -0.738887324f, -0.673829000f,
   -0.736816569f, -0.676092704f,
   -0.734738878f, -0.678350043f,
   -0.732654272f, -0.680600998f,
   -0.730562769f, -0.682845546f,
   -0.728464390f, -0.685083668f,
   -0.726359155f, -0.687315341f,
   -0.724247083f, -0.689540545f,
   -0.722128194f, -0.691759258f,
   -0.720002508f, -0.693971461f,
   -0.717870045f, -0.696177131f,
   -0.715730825f, -0.698376249f,
   -0.713584869f, -0.700568794f,
   -0.711432196f, -0.702754744f,
   -0.709272826f, -0.704934080f,
   -0.707106781f, -0.707106781f,
   -0.704934080f, -0.709272826f,
   -0.702754744f, -0.711432196f,
   -0.700568794f, -0.713584869f,
   -0.698376249f, -0.715730825f,
   -0.696177131f, -0.717870045f,
   -0.693971461f, -0.720002508f,
   -0.691759258f, -0.722128194f,
   -0.689540545f, -0.724247083f,
   -0.687315341f, -0.726359155f,
   -0.685083668f, -0.728464390f,
   -0.682845546f, -0.730562769f,
   -0.680600998f, -0.732654272f,
   -0.678350043f, -0.734738878f,
   -0.676092704f, -0.736816569f,
   -0.673829000f, -0.738887324f,
   -0.671558955f, -0.740951125f,
   -0.669282588f, -0.743007952f,
   -0.666999922f, -0.745057785f,
   -0.664710978f, -0.747100606f,
   -0.662415778f, -0.749136395f,
   -0.660114342f, -0.751165132f,
   -0.657806693f, -0.753186799f,
   -0.655492853f, -0.755201377f,
   -0.653172843f, -0.757208847f,
   -0.650846685f, -0.759209189f,
   -0.648514401f, -0.761202385f,
   -0.646176013f, -0.763188417f,
   -0.643831543f, -0.765167266f,
   -0.641481013f, -0.767138912f,
   -0.639124445f, -0.769103338f,
   -0.636761861f, -0.771060524f,
   -0.634393284f, -0.773010453f,
   -0.632018736f, -0.774953107f,
   -0.629638239f, -0.776888466f,
   -0.627251815f, -0.778816512f,
   -0.624859488f, -0.780737229f,
   -0.622461279f, -0.782650596f,
   -0.620057212f, -0.784556597f,
   -0.617647308f, -0.786455214f,
   -0.615231591f, -0.788346428f,
   -0.612810082f, -0.790230221f,
   -0.610382806f, -0.792106577f,
   -0.607949785f, -0.793975478f,
   -0.605511041f, -0.795836905f,
   -0.603066599f, -0.797690841f,
   -0.600616479f, -0.799537269f,
   -0.598160707f, -0.801376172f,
   -0.595699304f, -0.803207531f,
   -0.593232295f, -0.805031331f,
   -0.590759702f, -0.806847554f,
   -0.588281548f, -0.808656182f,
   -0.585797857f, -0.810457198f,
   -0.583308653f, -0.812250587f,
   -0.580813958f, -0.814036330f,
   -0.578313796f, -0.815814411f,
   -0.575808191f, -0.817584813f,
   -0.573297167f, -0.819347520f,
   -0.570780746f, -0.821102515f,
   -0.568258953f, -0.822849781f,
   -0.565731811f, -0.824589303f,
   -0.563199344f, -0.826321063f,
   -0.560661576f, -0.828045045f,
   -0.558118531f, -0.829761234f,
   -0.555570233f, -0.831469612f,
   -0.553016706f, -0.833170165f,
   -0.550457973f, -0.834862875f,
   -0.547894059f, -0.836547727f,
   -0.545324988f, -0.838224706f,
   -0.542750785f, -0.839893794f,
   -0.540171473f, -0.841554977f,
   -0.537587076f, -0.843208240f,
   -0.534997620f, -0.844853565f,
   -0.532403128f, -0.846490939f,
   -0.529803625f, -0.848120345f,
   -0.527199135f, -0.849741768f,
   -0.524589683f, -0.851355193f,
   -0.521975293f, -0.852960605f,
   -0.519355990f, -0.854557988f,
   -0.516731799f, -0.856147328f,
   -0.514102744f, -0.857728610f,
   -0.511468850f, -0.859301818f,
   -0.508830143f, -0.860866939f,
   -0.506186645f, -0.862423956f,
   -0.503538384f, -0.863972856f,
   -0.500885383f, -0.865513624f,
   -0.498227667f, -0.867046246f,
   -0.495565262f, -0.868570706f,
   -0.492898192f, -0.870086991f,
   -0.490226483f, -0.871595087f,
   -0.487550160f, -0.873094978f,
   -0.484869248f, -0.874586652f,
   -0.482183772f, -0.876070094f,
   -0.479493758f, -0.877545290f,
   -0.476799230f, -0.879012226f,
   -0.474100215f, -0.880470889f,
   -0.471396737f, -0.881921264f,
   -0.468688822f, -0.883363339f,
   -0.465976496f, -0.884797098f,
   -0.463259784f, -0.886222530f,
   -0.460538711f, -0.887639620f,
   -0.457813304f, -0.889048356f,
   -0.455083587f, -0.890448723f,
   -0.452349587f, -0.891840709f,
   -0.449611330f, -0.893224301f,
   -0.446868840f, -0.894599486f,
   -0.444122145f, -0.895966250f,
   -0.441371269f, -0.897324581f,
   -0.438616239f, -0.898674466f,
   -0.435857080f, -0.900015892f,
   -0.433093819f, -0.901348847f,
   -0.430326481f, -0.902673318f,
   -0.427555093f, -0.903989293f,
   -0.424779681f, -0.905296759f,
   -0.422000271f, -0.906595705f,
   -0.419216888f, -0.907886116f,
   -0.416429560f, -0.909167983f,
   -0.413638312f, -0.910441292f,
   -0.410843171f, -0.911706032f,
   -0.408044163f, -0.912962190f,
   -0.405241314f, -0.914209756f,
   -0.402434651f, -0.915448716f,
   -0.399624200f, -0.916679060f,
   -0.396809987f, -0.917900776f,
   -0.393992040f, -0.919113852f,
   -0.391170384f, -0.920318277f,
   -0.388345047f, -0.921514039f,
   -0.385516054f, -0.922701128f,
   -0.382683432f, -0.923879533f,
   -0.379847209f, -0.925049241f,
   -0.377007410f, -0.926210242f,
   -0.374164063f, -0.927362526f,
   -0.371317194f, -0.928506080f,
   -0.368466830f, -0.929640896f,
   -0.365612998f, -0.930766961f,
   -0.362755724f, -0.931884266f,
   -0.359895037f, -0.932992799f,
   -0.357030961f, -0.934092550f,
   -0.354163525f, -0.935183510f,
   -0.351292756f, -0.936265667f,
   -0.348418680f, -0.937339012f,
   -0.345541325f, -0.938403534f,
   -0.342660717f, -0.939459224f,
   -0.339776884f, -0.940506071f,
   -0.336889853f, -0.941544065f,
   -0.333999651f, -0.942573198f,
   -0.331106306f, -0.943593458f,
   -0.328209844f, -0.944604837f,
   -0.325310292f, -0.945607325f,
   -0.322407679f, -0.946600913f,
   -0.319502031f, -0.947585591f,
   -0.316593376f, -0.948561350f,
   -0.313681740f, -0.949528181f,
   -0.310767153f, -0.950486074f,
   -0.307849640f, -0.951435021f,
   -0.304929230f, -0.952375013f,
   -0.302005949f, -0.953306040f,
   -0.299079826f, -0.954228095f,
   -0.296150888f, -0.955141168f,
   -0.293219163f, -0.956045251f,
   -0.290284677f, -0.956940336f,
   -0.287347460f, -0.957826413f,
   -0.284407537f, -0.958703475f,
   -0.281464938f, -0.959571513f,
   -0.278519689f, -0.960430519f,
   -0.275571819f, -0.961280486f,
   -0.272621355f, -0.962121404f,
   -0.269668326f, -0.962953267f,
   -0.266712757f, -0.963776066f,
   -0.263754679f, -0.964589793f,
   -0.260794118f, -0.965394442f,
   -0.257831102f, -0.966190003f,
   -0.254865660f, -0.966976471f,
   -0.251897818f, -0.967753837f,
   -0.248927606f, -0.968522094f,
   -0.245955050f, -0.969281235f,
   -0.242980180f, -0.970031253f,
   -0.240003022f, -0.970772141f,
   -0.237023606f, -0.971503891f,
   -0.234041959f, -0.972226497f,
   -0.231058108f, -0.972939952f,
   -0.228072083f, -0.973644250f,
   -0.225083911f, -0.974339383f,
   -0.222093621f, -0.975025345f,
   -0.219101240f, -0.975702130f,
   -0.216106797f, -0.976369731f,
   -0.213110320f, -0.977028143f,
   -0.210111837f, -0.977677358f,
   -0.207111376f, -0.978317371f,
   -0.204108966f, -0.978948175f,
   -0.201104635f, -0.979569766f,
   -0.198098411f, -0.980182136f,
   -0.195090322f, -0.980785280f,
   -0.192080397f, -0.981379193f,
   -0.189068664f, -0.981963869f,
   -0.186055152f, -0.982539302f,
   -0.183039888f, -0.983105487f,
   -0.180022901f, -0.983662419f,
   -0.177004220f, -0.984210092f,
   -0.173983873f, -0.984748502f,
   -0.170961889f, -0.985277642f,
   -0.167938295f, -0.985797509f,
   -0.164913120f, -0.986308097f,
   -0.161886394f, -0.986809402f,
   -0.158858143f, -0.987301418f,
   -0.155828398f, -0.987784142f,
   -0.152797185f, -0.988257568f,
   -0.149764535f, -0.988721692f,
   -0.146730474f, -0.989176510f,
   -0.143695033f, -0.989622017f,
   -0.140658239f, -0.990058210f,
   -0.137620122f, -0.990485084f,
   -0.134580709f, -0.990902635f,
   -0.131540029f, -0.991310860f,
   -0.128498111f, -0.991709754f,
   -0.125454983f, -0.992099313f,
   -0.122410675f, -0.992479535f,
   -0.119365215f, -0.992850414f,
   -0.116318631f, -0.993211949f,
   -0.113270952f, -0.993564136f,
   -0.110222207f, -0.993906970f,
   -0.107172425f, -0.994240449f,
   -0.104121634f, -0.994564571f,
   -0.101069863f, -0.994879331f,
   -0.098017140f, -0.995184727f,
   -0.094963495f, -0.995480755f,
   -0.091908956f, -0.995767414f,
   -0.088853553f, -0.996044701f,
   -0.085797312f, -0.996312612f,
   -0.082740265f, -0.996571146f,
   -0.079682438f, -0.996820299f,
   -0.076623861f, -0.997060070f,
   -0.073564564f, -0.997290457f,
   -0.070504573f, -0.997511456f,
   -0.067443920f, -0.997723067f,
   -0.064382631f, -0.997925286f,
   -0.061320736f, -0.998118113f,
   -0.058258265f, -0.998301545f,
   -0.055195244f, -0.998475581f,
   -0.052131705f, -0.998640218f,
   -0.049067674f, -0.998795456f,
   -0.046003182f, -0.998941293f,
   -0.042938257f, -0.999077728f,
   -0.039872928f, -0.999204759f,
   -0.036807223f, -0.999322385f,
   -0.033741172f, -0.999430605f,
   -0.030674803f, -0.999529418f,
   -0.027608146f, -0.999618822f,
   -0.024541229f, -0.999698819f,
   -0.021474080f, -0.999769405f,
   -0.018406730f, -0.999830582f,
   -0.015339206f, -0.999882347f,
   -0.012271538f, -0.999924702f,
   -0.009203755f, -0.999957645f,
   -0.006135885f, -0.999981175f,
   -0.003067957f, -0.999995294f,
   -0.000000000f, -1.000000000f,
    0.003067957f, -0.999995294f,
    0.006135885f, -0.999981175f,
    0.009203755f, -0.999957645f,
    0.012271538f, -0.999924702f,
    0.015339206f, -0.999882347f,
    0.018406730f, -0.999830582f,
    0.021474080f, -0.999769405f,
    0.024541229f, -0.999698819f,
    0.027608146f, -0.999618822f,
    0.030674803f, -0.999529418f,
    0.033741172f, -0.999430605f,
    0.036807223f, -0.999322385f,
    0.039872928f, -0.999204759f,
    0.042938257f, -0.999077728f,
    0.046003182f, -0.998941293f,
    0.049067674f, -0.998795456f,
    0.052131705f, -0.998640218f,
    0.055195244f, -0.998475581f,
    0.058258265f, -0.998301545f,
    0.061320736f, -0.998118113f,
    0.064382631f, -0.997925286f,
    0.067443920f, -0.997723067f,
    0.070504573f, -0.997511456f,
    0.073564564f, -0.997290457f,
    0.076623861f, -0.997060070f,
    0.079682438f, -0.996820299f,
    0.082740265f, -0.996571146f,
    0.085797312f, -0.996312612f,
    0.088853553f, -0.996044701f,
    0.091908956f, -0.995767414f,
    0.094963495f, -0.995480755f,
    0.098017140f, -0.995184727f,
    0.101069863f, -0.994879331f,
    0.104121634f, -0.994564571f,
    0.107172425f, -0.994240449f,
    0.110222207f, -0.993906970f,
    0.113270952f, -0.993564136f,
    0.116318631f, -0.993211949f,
    0.119365215f, -0.992850414f,
    0.122410675f, -0.992479535f,
    0.125454983f, -0.992099313f,
    0.128498111f, -0.991709754f,
    0.131540029f, -0.991310860f,
    0.134580709f, -0.990902635f,
    0.137620122f, -0.990485084f,
    0.140658239f, -0.990058210f,
    0.143695033f, -0.989622017f,
    0.146730474f, -0.989176510f,
    0.149764535f, -0.988721692f,
    0.152797185f, -0.988257568f,
    0.155828398f, -0.987784142f,
    0.158858143f, -0.987301418f,
    0.161886394f, -0.986809402f,
    0.164913120f, -0.986308097f,
    0.167938295f, -0.985797509f,
    0.170961889f, -0.985277642f,
    0.173983873f, -0.984748502f,
    0.177004220f, -0.984210092f,
    0.180022901f, -0.983662419f,
    0.183039888f, -0.983105487f,
    0.186055152f, -0.982539302f,
    0.189068664f, -0.981963869f,
    0.192080397f, -0.981379193f,
    0.195090322f, -0.980785280f,
    0.198098411f, -0.980182136f,
    0.201104635f, -0.979569766f,
    0.204108966f, -0.978948175f,
    0.207111376f, -0.978317371f,
    0.210111837f, -0.977677358f,
    0.213110320f, -0.977028143f,
    0.216106797f, -0.976369731f,
    0.219101240f, -0.975702130f,
    0.222093621f, -0.975025345f,
    0.225083911f, -0.974339383f,
    0.228072083f, -0.973644250f,
    0.231058108f, -0.972939952f,
    0.234041959f, -0.972226497f,
    0.237023606f, -0.971503891f,
    0.240003022f, -0.970772141f,
    0.242980180f, -0.970031253f,
    0.245955050f, -0.969281235f,
    0.248927606f, -0.968522094f,
    0.251897818f, -0.967753837f,
    0.254865660f, -0.966976471f,
    0.257831102f, -0.966190003f,
    0.260794118f, -0.965394442f,
    0.263754679f, -0.964589793f,
    0.266712757f, -0.963776066f,
    0.269668326f, -0.962953267f,
    0.272621355f, -0.962121404f,
    0.275571819f, -0.961280486f,
    0.278519689f, -0.960430519f,
    0.281464938f, -0.959571513f,
    0.284407537f, -0.958703475f,
    0.287347460f, -0.957826413f,
    0.290284677f, -0.956940336f,
    0.293219163f, -0.956045251f,
    0.296150888f, -0.955141168f,
    0.299079826f, -0.954228095f,
    0.302005949f, -0.953306040f,
    0.304929230f, -0.952375013f,
    0.307849640f, -0.951435021f,
    0.310767153f, -0.950486074f,
    0.313681740f, -0.949528181f,
    0.316593376f, -0.948561350f,
    0.319502031f, -0.947585591f,
    0.322407679f, -0.946600913f,
    0.325310292f, -0.945607325f,
    0.328209844f, -0.944604837f,
    0.331106306f, -0.943593458f,
    0.333999651f, -0.942573198f,
    0.336889853f, -0.941544065f,
    0.339776884f, -0.940506071f,
    0.342660717f, -0.939459224f,
    0.345541325f, -0.938403534f,
    0.348418680f, -0.937339012f,
    0.351292756f, -0.936265667f,
    0.354163525f, -0.935183510f,
    0.357030961f, -0.934092550f,
    0.359895037f, -0.932992799f,
    0.362755724f, -0.931884266f,
    0.365612998f, -0.930766961f,
    0.368466830f, -0.929640896f,
    0.371317194f, -0.928506080f,
    0.374164063f, -0.927362526f,
    0.377007410f, -0.926210242f,
    0.379847209f, -0.925049241f,
    0.382683432f, -0.923879533f,
    0.385516054f, -0.922701128f,
    0.388345047f, -0.921514039f,
    0.391170384f, -0.920318277f,
    0.393992040f, -0.919113852f,
    0.396809987f, -0.917900776f,
    0.399624200f, -0.916679060f,
    0.402434651f, -0.915448716f,
    0.405241314f, -0.914209756f,
    0.408044163f, -0.912962190f,
    0.410843171f, -0.911706032f,
    0.413638312f, -0.910441292f,
    0.416429560f, -0.909167983f,
    0.419216888f, -0.907886116f,
    0.422000271f, -0.906595705f,
    0.424779681f, -0.905296759f,
    0.427555093f, -0.903989293f,
    0.430326481f, -0.902673318f,
    0.433093819f, -0.901348847f,
    0.435857080f, -0.900015892f,
    0.438616239f, -0.898674466f,
    0.441371269f, -0.897324581f,
    0.444122145f, -0.895966250f,
    0.446868840f, -0.894599486f,
    0.449611330f, -0.893224301f,
    0.452349587f, -0.891840709f,
    0.455083587f, -0.890448723f,
    0.457813304f, -0.889048356f,
    0.460538711f, -0.887639620f,
    0.463259784f, -0.886222530f,
    0.465976496f, -0.884797098f,
    0.468688822f, -0.883363339f,
    0.471396737f, -0.881921264f,
    0.474100215f, -0.880470889f,
    0.476799230f, -0.879012226f,
    0.479493758f, -0.877545290f,
    0.482183772f, -0.876070094f,
    0.484869248f, -0.874586652f,
    0.487550160f, -0.873094978f,
    0.490226483f, -0.871595087f,
    0.492898192f, -0.870086991f,
    0.495565262f, -0.868570706f,
    0.498227667f, -0.867046246f,
    0.500885383f, -0.865513624f,
    0.503538384f, -0.863972856f,
    0.506186645f, -0.862423956f,
    0.508830143f, -0.860866939f,
    0.511468850f, -0.859301818f,
    0.514102744f, -0.857728610f,
    0.516731799f, -0.856147328f,
    0.519355990f, -0.854557988f,
    0.521975293f, -0.852960605f,
    0.524589683f, -0.851355193f,
    0.527199135f, -0.849741768f,
    0.529803625f, -0.848120345f,
    0.532403128f, -0.846490939f,
    0.534997620f, -0.844853565f,
    0.537587076f, -0.843208240f,
    0.540171473f, -0.841554977f,
    0.542750785f, -0.839893794f,
    0.545324988f, -0.838224706f,
    0.547894059f, -0.836547727f,
    0.550457973f, -0.834862875f,
    0.553016706f, -0.833170165f,
    0.555570233f, -0.831469612f,
    0.558118531f, -0.829761234f,
    0.560661576f, -0.828045045f,
    0.563199344f, -0.826321063f,
    0.565731811f, -0.824589303f,
    0.568258953f, -0.822849781f,
    0.570780746f, -0.821102515f,
    0.573297167f, -0.819347520f,
    0.575808191f, -0.817584813f,
    0.578313796f, -0.815814411f,
    0.580813958f, -0.814036330f,
    0.583308653f, -0.812250587f,
    0.585797857f, -0.810457198f,
    0.588281548f, -0.808656182f,
    0.590759702f, -0.806847554f,
    0.593232295f, -0.805031331f,
    0.595699304f, -0.803207531f,
    0.598160707f, -0.801376172f,
    0.600616479f, -0.799537269f,
    0.603066599f, -0.797690841f,
    0.605511041f, -0.795836905f,
    0.607949785f, -0.793975478f,
    0.610382806f, -0.792106577f,
    0.612810082f, -0.790230221f,
    0.615231591f, -0.788346428f,
    0.617647308f, -0.786455214f,
    0.620057212f, -0.784556597f,
    0.622461279f, -0.782650596f,
    0.624859488f, -0.780737229f,
    0.627251815f, -0.778816512f,
    0.629638239f, -0.776888466f,
    0.632018736f, -0.774953107f,
    0.634393284f, -0.773010453f,
    0.636761861f, -0.771060524f,
    0.639124445f, -0.769103338f,
    0.641481013f, -0.767138912f,
    0.643831543f, -0.765167266f,
    0.646176013f, -0.763188417f,
    0.648514401f, -0.761202385f,
    0.650846685f, -0.759209189f,
    0.653172843f, -0.757208847f,
    0.655492853f, -0.755201377f,
    0.657806693f, -0.753186799f,
    0.660114342f, -0.751165132f,
    0.662415778f, -0.749136395f,
    0.664710978f, -0.747100606f,
    0.666999922f, -0.745057785f,
    0.669282588f, -0.743007952f,
    0.671558955f, -0.740951125f,
    0.673829000f, -0.738887324f,
    0.676092704f, -0.736816569f,
    0.678350043f, -0.734738878f,
    0.680600998f, -0.732654272f,
    0.682845546f, -0.730562769f,
    0.685083668f, -0.728464390f,
    0.687315341f, -0.726359155f,
    0.689540545f, -0.724247083f,
    0.691759258f, -0.722128194f,
    0.693971461f, -0.720002508f,
    0.696177131f, -0.717870045f,
    0.698376249f, -0.715730825f,
    0.700568794f, -0.713584869f,
    0.702754744f, -0.711432196f,
    0.704934080f, -0.709272826f,
    0.707106781f, -0.707106781f,
    0.709272826f, -0.704934080f,
    0.711432196f, -0.702754744f,
    0.713584869f, -0.700568794f,
    0.715730825f, -0.698376249f,
    0.717870045f, -0.696177131f,
    0.720002508f, -0.693971461f,
    0.722128194f, -0.691759258f,
    0.724247083f, -0.689540545f,
    0.726359155f, -0.687315341f,
    0.728464390f, -0.685083668f,
    0.730562769f, -0.682845546f,
    0.732654272f, -0.680600998f,
    0.734738878f, -0.678350043f,
    0.736816569f, -0.676092704f,
    0.738887324f, -0.673829000f,
    0.740951125f, -0.671558955f,
    0.743007952f, -0.669282588f,
    0.745057785f, -0.666999922f,
    0.747100606f, -0.664710978f,
    0.749136395f, -0.662415778f,
    0.751165132f, -0.660114342f,
    0.753186799f, -0.657806693f,
    0.755201377f, -0.655492853f,
    0.757208847f, -0.653172843f,
    0.759209189f, -0.650846685f,
    0.761202385f, -0.648514401f,
    0.763188417f, -0.646176013f,
    0.765167266f, -0.643831543f,
    0.767138912f, -0.641481013f,
    0.769103338f, -0.639124445f,
    0.771060524f, -0.636761861f,
    0.773010453f, -0.634393284f,
    0.774953107f, -0.632018736f,
    0.776888466f, -0.629638239f,
    0.778816512f, -0.627251815f,
    0.780737229f, -0.624859488f,
    0.782650596f, -0.622461279f,
    0.784556597f, -0.620057212f,
    0.786455214f, -0.617647308f,
    0.788346428f, -0.615231591f,
    0.790230221f, -0.612810082f,
    0.792106577f, -0.610382806f,
    0.793975478f, -0.607949785f,
    0.795836905f, -0.605511041f,
    0.797690841f, -0.603066599f,
    0.799537269f, -0.600616479f,
    0.801376172f, -0.598160707f,
    0.803207531f, -0.595699304f,
    0.805031331f, -0.593232295f,
    0.806847554f, -0.590759702f,
    0.808656182f, -0.588281548f,
    0.810457198f, -0.585797857f,
    0.812250587f, -0.583308653f,
    0.814036330f, -0.580813958f,
    0.815814411f, -0.578313796f,
    0.817584813f, -0.575808191f,
    0.819347520f, -0.573297167f,
    0.821102515f, -0.570780746f,
    0.822849781f, -0.568258953f,
    0.824589303f, -0.565731811f,
    0.826321063f, -0.563199344f,
    0.828045045f, -0.560661576f,
    0.829761234f, -0.558118531f,
    0.831469612f, -0.555570233f,
    0.833170165f, -0.553016706f,
    0.834862875f, -0.550457973f,
    0.836547727f, -0.547894059f,
    0.838224706f, -0.545324988f,
    0.839893794f, -0.542750785f,
    0.841554977f, -0.540171473f,
    0.843208240f, -0.537587076f,
    0.844853565f, -0.534997620f,
    0.846490939f, -0.532403128f,
    0.848120345f, -0.529803625f,
    0.849741768f, -0.527199135f,
    0.851355193f, -0.524589683f,
    0.852960605f, -0.521975293f,
    0.854557988f, -0.519355990f,
    0.856147328f, -0.516731799f,
    0.857728610f, -0.514102744f,
    0.859301818f, -0.511468850f,
    0.860866939f, -0.508830143f,
    0.862423956f, -0.506186645f,
    0.863972856f, -0.503538384f,
    0.865513624f, -0.500885383f,
    0.867046246f, -0.498227667f,
    0.868570706f, -0.495565262f,
    0.870086991f, -0.492898192f,
    0.871595087f, -0.490226483f,
    0.873094978f, -0.487550160f,
    0.874586652f, -0.484869248f,
    0.876070094f, -0.482183772f,
    0.877545290f, -0.479493758f,
    0.879012226f, -0.476799230f,
    0.880470889f, -0.474100215f,
    0.881921264f, -0.471396737f,
    0.883363339f, -0.468688822f,
    0.884797098f, -0.465976496f,
    0.886222530f, -0.463259784f,
    0.887639620f, -0.460538711f,
    0.889048356f, -0.457813304f,
    0.890448723f, -0.455083587f,
    0.891840709f, -0.452349587f,
    0.893224301f, -0.449611330f,
    0.894599486f, -0.446868840f,
    0.895966250f, -0.444122145f,
    0.897324581f, -0.441371269f,
    0.898674466f, -0.438616239f,
    0.900015892f, -0.435857080f,
    0.901348847f, -0.433093819f,
    0.902673318f, -0.430326481f,
    0.903989293f, -0.427555093f,
    0.905296759f, -0.424779681f,
    0.906595705f, -0.422000271f,
    0.907886116f, -0.419216888f,
    0.909167983f, -0.416429560f,
    0.910441292f, -0.413638312f,
    0.911706032f, -0.410843171f,
    0.912962190f, -0.408044163f,
    0.914209756f, -0.405241314f,
    0.915448716f, -0.402434651f,
    0.916679060f, -0.399624200f,
    0.917900776f, -0.396809987f,
    0.919113852f, -0.393992040f,
    0.920318277f, -0.391170384f,
    0.921514039f, -0.388345047f,
    0.922701128f, -0.385516054f,
    0.923879533f, -0.382683432f,
    0.925049241f, -0.379847209f,
    0.926210242f, -0.377007410f,
    0.927362526f, -0.374164063f,
    0.928506080f, -0.371317194f,
    0.929640896f, -0.368466830f,
    0.930766961f, -0.365612998f,
    0.931884266f, -0.362755724f,
    0.932992799f, -0.359895037f,
    0.934092550f, -0.357030961f,
    0.935183510f, -0.354163525f,
    0.936265667f, -0.351292756f,
    0.937339012f, -0.348418680f,
    0.938403534f, -0.345541325f,
    0.939459224f, -0.342660717f,
    0.940506071f, -0.339776884f,
    0.941544065f, -0.336889853f,
    0.942573198f, -0.333999651f,
    0.943593458f, -0.331106306f,
    0.944604837f, -0.328209844f,
    0.945607325f, -0.325310292f,
    0.946600913f, -0.322407679f,
    0.947585591f, -0.319502031f,
    0.948561350f, -0.316593376f,
    0.949528181f, -0.313681740f,
    0.950486074f, -0.310767153f,
    0.951435021f, -0.307849640f,
    0.952375013f, -0.304929230f,
    0.953306040f, -0.302005949f,
    0.954228095f, -0.299079826f,
    0.955141168f, -0.296150888f,
    0.956045251f, -0.293219163f,
    0.956940336f, -0.290284677f,
    0.957826413f, -0.287347460f,
    0.958703475f, -0.284407537f,
    0.959571513f, -0.281464938f,
    0.960430519f, -0.278519689f,
    0.961280486f, -0.275571819f,
    0.962121404f, -0.272621355f,
    0.962953267f, -0.269668326f,
    0.963776066f, -0.266712757f,
    0.964589793f, -0.263754679f,
    0.965394442f, -0.260794118f,
    0.966190003f, -0.257831102f,
    0.966976471f, -0.254865660f,
    0.967753837f, -0.251897818f,
    0.968522094f, -0.248927606f,
    0.969281235f, -0.245955050f,
    0.970031253f, -0.242980180f,
    0.970772141f, -0.240003022f,
    0.971503891f, -0.237023606f,
    0.972226497f, -0.234041959f,
    0.972939952f, -0.231058108f,
    0.973644250f, -0.228072083f,
    0.974339383f, -0.225083911f,
    0.975025345f, -0.222093621f,
    0.975702130f, -0.219101240f,
    0.976369731f, -0.216106797f,
    0.977028143f, -0.213110320f,
    0.977677358f, -0.210111837f,
    0.978317371f, -0.207111376f,
    0.978948175f, -0.204108966f,
    0.979569766f, -0.201104635f,
    0.980182136f, -0.198098411f,
    0.980785280f, -0.195090322f,
    0.981379193f, -0.192080397f,
    0.981963869f, -0.189068664f,
    0.982539302f, -0.186055152f,
    0.983105487f, -0.183039888f,
    0.983662419f, -0.180022901f,
    0.984210092f, -0.177004220f,
    0.984748502f, -0.173983873f,
    0.985277642f, -0.170961889f,
    0.985797509f, -0.167938295f,
    0.986308097f, -0.164913120f,
    0.986809402f, -0.161886394f,
    0.987301418f, -0.158858143f,
    0.987784142f, -0.155828398f,
    0.988257568f, -0.152797185f,
    0.988721692f, -0.149764535f,
    0.989176510f, -0.146730474f,
    0.989622017f, -0.143695033f,
    0.990058210f, -0.140658239f,
    0.990485084f, -0.137620122f,
    0.990902635f, -0.134580709f,
    0.991310860f, -0.131540029f,
    0.991709754f, -0.128498111f,
    0.992099313f, -0.125454983f,
    0.992479535f, -0.122410675f,
    0.992850414f, -0.119365215f,
    0.993211949f, -0.116318631f,
    0.993564136f, -0.113270952f,
    0.993906970f, -0.110222207f,
    0.994240449f, -0.107172425f,
    0.994564571f, -0.104121634f,
    0.994879331f, -0.101069863f,
    0.995184727f, -0.098017140f,
    0.995480755f, -0.094963495f,
    0.995767414f, -0.091908956f,
    0.996044701f, -0.088853553f,
    0.996312612f, -0.085797312f,
    0.996571146f, -0.082740265f,
    0.996820299f, -0.079682438f,
    0.997060070f, -0.076623861f,
    0.997290457f, -0.073564564f,
    0.997511456f, -0.070504573f,
    0.997723067f, -0.067443920f,
    0.997925286f, -0.064382631f,
    0.998118113f, -0.061320736f,
    0.998301545f, -0.058258265f,
    0.998475581f, -0.055195244f,
    0.998640218f, -0.052131705f,
    0.998795456f, -0.049067674f,
    0.998941293f, -0.046003182f,
    0.999077728f, -0.042938257f,
    0.999204759f, -0.039872928f,
    0.999322385f, -0.036807223f,
    0.999430605f, -0.033741172f,
    0.999529418f, -0.030674803f,
    0.999618822f, -0.027608146f,
    0.999698819f, -0.024541229f,
    0.999769405f, -0.021474080f,
    0.999830582f, -0.018406730f,
    0.999882347f, -0.015339206f,
    0.999924702f, -0.012271538f,
    0.999957645f, -0.009203755f,
    0.999981175f, -0.006135885f,
    0.999995294f, -0.003067957f
};
#endif // INCLUDE_FFT_TABLES_2048


#if defined(INCLUDE_FFT_TABLES_4096)
/**
  @par
  Example code for Floating-point Twiddle factors Generation:
  @par
  <pre>for (i = 0; i< N/; i++)
  {
 	twiddleCoef[2*i]   = cos(i * 2*PI/(float)N);
 	twiddleCoef[2*i+1] = sin(i * 2*PI/(float)N);
  } </pre>
  @par
  where N = 4096, PI = 3.14159265358979
  @par
  Cos and Sin values are in interleaved fashion
*/
__attribute__ ((section(".rom_data")))
const float32_t twiddleCoef_4096[8192] = {
    1.000000000f,  0.000000000f,
    0.999998823f,  0.001533980f,
    0.999995294f,  0.003067957f,
    0.999989411f,  0.004601926f,
    0.999981175f,  0.006135885f,
    0.999970586f,  0.007669829f,
    0.999957645f,  0.009203755f,
    0.999942350f,  0.010737659f,
    0.999924702f,  0.012271538f,
    0.999904701f,  0.013805389f,
    0.999882347f,  0.015339206f,
    0.999857641f,  0.016872988f,
    0.999830582f,  0.018406730f,
    0.999801170f,  0.019940429f,
    0.999769405f,  0.021474080f,
    0.999735288f,  0.023007681f,
    0.999698819f,  0.024541229f,
    0.999659997f,  0.026074718f,
    0.999618822f,  0.027608146f,
    0.999575296f,  0.029141509f,
    0.999529418f,  0.030674803f,
    0.999481187f,  0.032208025f,
    0.999430605f,  0.033741172f,
    0.999377670f,  0.035274239f,
    0.999322385f,  0.036807223f,
    0.999264747f,  0.038340120f,
    0.999204759f,  0.039872928f,
    0.999142419f,  0.041405641f,
    0.999077728f,  0.042938257f,
    0.999010686f,  0.044470772f,
    0.998941293f,  0.046003182f,
    0.998869550f,  0.047535484f,
    0.998795456f,  0.049067674f,
    0.998719012f,  0.050599749f,
    0.998640218f,  0.052131705f,
    0.998559074f,  0.053663538f,
    0.998475581f,  0.055195244f,
    0.998389737f,  0.056726821f,
    0.998301545f,  0.058258265f,
    0.998211003f,  0.059789571f,
    0.998118113f,  0.061320736f,
    0.998022874f,  0.062851758f,
    0.997925286f,  0.064382631f,
    0.997825350f,  0.065913353f,
    0.997723067f,  0.067443920f,
    0.997618435f,  0.068974328f,
    0.997511456f,  0.070504573f,
    0.997402130f,  0.072034653f,
    0.997290457f,  0.073564564f,
    0.997176437f,  0.075094301f,
    0.997060070f,  0.076623861f,
    0.996941358f,  0.078153242f,
    0.996820299f,  0.079682438f,
    0.996696895f,  0.081211447f,
    0.996571146f,  0.082740265f,
    0.996443051f,  0.084268888f,
    0.996312612f,  0.085797312f,
    0.996179829f,  0.087325535f,
    0.996044701f,  0.088853553f,
    0.995907229f,  0.090381361f,
    0.995767414f,  0.091908956f,
    0.995625256f,  0.093436336f,
    0.995480755f,  0.094963495f,
    0.995333912f,  0.096490431f,
    0.995184727f,  0.098017140f,
    0.995033199f,  0.099543619f,
    0.994879331f,  0.101069863f,
    0.994723121f,  0.102595869f,
    0.994564571f,  0.104121634f,
    0.994403680f,  0.105647154f,
    0.994240449f,  0.107172425f,
    0.994074879f,  0.108697444f,
    0.993906970f,  0.110222207f,
    0.993736722f,  0.111746711f,
    0.993564136f,  0.113270952f,
    0.993389211f,  0.114794927f,
    0.993211949f,  0.116318631f,
    0.993032350f,  0.117842062f,
    0.992850414f,  0.119365215f,
    0.992666142f,  0.120888087f,
    0.992479535f,  0.122410675f,
    0.992290591f,  0.123932975f,
    0.992099313f,  0.125454983f,
    0.991905700f,  0.126976696f,
    0.991709754f,  0.128498111f,
    0.991511473f,  0.130019223f,
    0.991310860f,  0.131540029f,
    0.991107914f,  0.133060525f,
    0.990902635f,  0.134580709f,
    0.990695025f,  0.136100575f,
    0.990485084f,  0.137620122f,
    0.990272812f,  0.139139344f,
    0.990058210f,  0.140658239f,
    0.989841278f,  0.142176804f,
    0.989622017f,  0.143695033f,
    0.989400428f,  0.145212925f,
    0.989176510f,  0.146730474f,
    0.988950265f,  0.148247679f,
    0.988721692f,  0.149764535f,
    0.988490793f,  0.151281038f,
    0.988257568f,  0.152797185f,
    0.988022017f,  0.154312973f,
    0.987784142f,  0.155828398f,
    0.987543942f,  0.157343456f,
    0.987301418f,  0.158858143f,
    0.987056571f,  0.160372457f,
    0.986809402f,  0.161886394f,
    0.986559910f,  0.163399949f,
    0.986308097f,  0.164913120f,
    0.986053963f,  0.166425904f,
    0.985797509f,  0.167938295f,
    0.985538735f,  0.169450291f,
    0.985277642f,  0.170961889f,
    0.985014231f,  0.172473084f,
    0.984748502f,  0.173983873f,
    0.984480455f,  0.175494253f,
    0.984210092f,  0.177004220f,
    0.983937413f,  0.178513771f,
    0.983662419f,  0.180022901f,
    0.983385110f,  0.181531608f,
    0.983105487f,  0.183039888f,
    0.982823551f,  0.184547737f,
    0.982539302f,  0.186055152f,
    0.982252741f,  0.187562129f,
    0.981963869f,  0.189068664f,
    0.981672686f,  0.190574755f,
    0.981379193f,  0.192080397f,
    0.981083391f,  0.193585587f,
    0.980785280f,  0.195090322f,
    0.980484862f,  0.196594598f,
    0.980182136f,  0.198098411f,
    0.979877104f,  0.199601758f,
    0.979569766f,  0.201104635f,
    0.979260123f,  0.202607039f,
    0.978948175f,  0.204108966f,
    0.978633924f,  0.205610413f,
    0.978317371f,  0.207111376f,
    0.977998515f,  0.208611852f,
    0.977677358f,  0.210111837f,
    0.977353900f,  0.211611327f,
    0.977028143f,  0.213110320f,
    0.976700086f,  0.214608811f,
    0.976369731f,  0.216106797f,
    0.976037079f,  0.217604275f,
    0.975702130f,  0.219101240f,
    0.975364885f,  0.220597690f,
    0.975025345f,  0.222093621f,
    0.974683511f,  0.223589029f,
    0.974339383f,  0.225083911f,
    0.973992962f,  0.226578264f,
    0.973644250f,  0.228072083f,
    0.973293246f,  0.229565366f,
    0.972939952f,  0.231058108f,
    0.972584369f,  0.232550307f,
    0.972226497f,  0.234041959f,
    0.971866337f,  0.235533059f,
    0.971503891f,  0.237023606f,
    0.971139158f,  0.238513595f,
    0.970772141f,  0.240003022f,
    0.970402839f,  0.241491885f,
    0.970031253f,  0.242980180f,
    0.969657385f,  0.244467903f,
    0.969281235f,  0.245955050f,
    0.968902805f,  0.247441619f,
    0.968522094f,  0.248927606f,
    0.968139105f,  0.250413007f,
    0.967753837f,  0.251897818f,
    0.967366292f,  0.253382037f,
    0.966976471f,  0.254865660f,
    0.966584374f,  0.256348682f,
    0.966190003f,  0.257831102f,
    0.965793359f,  0.259312915f,
    0.965394442f,  0.260794118f,
    0.964993253f,  0.262274707f,
    0.964589793f,  0.263754679f,
    0.964184064f,  0.265234030f,
    0.963776066f,  0.266712757f,
    0.963365800f,  0.268190857f,
    0.962953267f,  0.269668326f,
    0.962538468f,  0.271145160f,
    0.962121404f,  0.272621355f,
    0.961702077f,  0.274096910f,
    0.961280486f,  0.275571819f,
    0.960856633f,  0.277046080f,
    0.960430519f,  0.278519689f,
    0.960002146f,  0.279992643f,
    0.959571513f,  0.281464938f,
    0.959138622f,  0.282936570f,
    0.958703475f,  0.284407537f,
    0.958266071f,  0.285877835f,
    0.957826413f,  0.287347460f,
    0.957384501f,  0.288816408f,
    0.956940336f,  0.290284677f,
    0.956493919f,  0.291752263f,
    0.956045251f,  0.293219163f,
    0.955594334f,  0.294685372f,
    0.955141168f,  0.296150888f,
    0.954685755f,  0.297615707f,
    0.954228095f,  0.299079826f,
    0.953768190f,  0.300543241f,
    0.953306040f,  0.302005949f,
    0.952841648f,  0.303467947f,
    0.952375013f,  0.304929230f,
    0.951906137f,  0.306389795f,
    0.951435021f,  0.307849640f,
    0.950961666f,  0.309308760f,
    0.950486074f,  0.310767153f,
    0.950008245f,  0.312224814f,
    0.949528181f,  0.313681740f,
    0.949045882f,  0.315137929f,
    0.948561350f,  0.316593376f,
    0.948074586f,  0.318048077f,
    0.947585591f,  0.319502031f,
    0.947094366f,  0.320955232f,
    0.946600913f,  0.322407679f,
    0.946105232f,  0.323859367f,
    0.945607325f,  0.325310292f,
    0.945107193f,  0.326760452f,
    0.944604837f,  0.328209844f,
    0.944100258f,  0.329658463f,
    0.943593458f,  0.331106306f,
    0.943084437f,  0.332553370f,
    0.942573198f,  0.333999651f,
    0.942059740f,  0.335445147f,
    0.941544065f,  0.336889853f,
    0.941026175f,  0.338333767f,
    0.940506071f,  0.339776884f,
    0.939983753f,  0.341219202f,
    0.939459224f,  0.342660717f,
    0.938932484f,  0.344101426f,
    0.938403534f,  0.345541325f,
    0.937872376f,  0.346980411f,
    0.937339012f,  0.348418680f,
    0.936803442f,  0.349856130f,
    0.936265667f,  0.351292756f,
    0.935725689f,  0.352728556f,
    0.935183510f,  0.354163525f,
    0.934639130f,  0.355597662f,
    0.934092550f,  0.357030961f,
    0.933543773f,  0.358463421f,
    0.932992799f,  0.359895037f,
    0.932439629f,  0.361325806f,
    0.931884266f,  0.362755724f,
    0.931326709f,  0.364184790f,
    0.930766961f,  0.365612998f,
    0.930205023f,  0.367040346f,
    0.929640896f,  0.368466830f,
    0.929074581f,  0.369892447f,
    0.928506080f,  0.371317194f,
    0.927935395f,  0.372741067f,
    0.927362526f,  0.374164063f,
    0.926787474f,  0.375586178f,
    0.926210242f,  0.377007410f,
    0.925630831f,  0.378427755f,
    0.925049241f,  0.379847209f,
    0.924465474f,  0.381265769f,
    0.923879533f,  0.382683432f,
    0.923291417f,  0.384100195f,
    0.922701128f,  0.385516054f,
    0.922108669f,  0.386931006f,
    0.921514039f,  0.388345047f,
    0.920917242f,  0.389758174f,
    0.920318277f,  0.391170384f,
    0.919717146f,  0.392581674f,
    0.919113852f,  0.393992040f,
    0.918508394f,  0.395401479f,
    0.917900776f,  0.396809987f,
    0.917290997f,  0.398217562f,
    0.916679060f,  0.399624200f,
    0.916064966f,  0.401029897f,
    0.915448716f,  0.402434651f,
    0.914830312f,  0.403838458f,
    0.914209756f,  0.405241314f,
    0.913587048f,  0.406643217f,
    0.912962190f,  0.408044163f,
    0.912335185f,  0.409444149f,
    0.911706032f,  0.410843171f,
    0.911074734f,  0.412241227f,
    0.910441292f,  0.413638312f,
    0.909805708f,  0.415034424f,
    0.909167983f,  0.416429560f,
    0.908528119f,  0.417823716f,
    0.907886116f,  0.419216888f,
    0.907241978f,  0.420609074f,
    0.906595705f,  0.422000271f,
    0.905947298f,  0.423390474f,
    0.905296759f,  0.424779681f,
    0.904644091f,  0.426167889f,
    0.903989293f,  0.427555093f,
    0.903332368f,  0.428941292f,
    0.902673318f,  0.430326481f,
    0.902012144f,  0.431710658f,
    0.901348847f,  0.433093819f,
    0.900683429f,  0.434475961f,
    0.900015892f,  0.435857080f,
    0.899346237f,  0.437237174f,
    0.898674466f,  0.438616239f,
    0.898000580f,  0.439994271f,
    0.897324581f,  0.441371269f,
    0.896646470f,  0.442747228f,
    0.895966250f,  0.444122145f,
    0.895283921f,  0.445496017f,
    0.894599486f,  0.446868840f,
    0.893912945f,  0.448240612f,
    0.893224301f,  0.449611330f,
    0.892533555f,  0.450980989f,
    0.891840709f,  0.452349587f,
    0.891145765f,  0.453717121f,
    0.890448723f,  0.455083587f,
    0.889749586f,  0.456448982f,
    0.889048356f,  0.457813304f,
    0.888345033f,  0.459176548f,
    0.887639620f,  0.460538711f,
    0.886932119f,  0.461899791f,
    0.886222530f,  0.463259784f,
    0.885510856f,  0.464618686f,
    0.884797098f,  0.465976496f,
    0.884081259f,  0.467333209f,
    0.883363339f,  0.468688822f,
    0.882643340f,  0.470043332f,
    0.881921264f,  0.471396737f,
    0.881197113f,  0.472749032f,
    0.880470889f,  0.474100215f,
    0.879742593f,  0.475450282f,
    0.879012226f,  0.476799230f,
    0.878279792f,  0.478147056f,
    0.877545290f,  0.479493758f,
    0.876808724f,  0.480839331f,
    0.876070094f,  0.482183772f,
    0.875329403f,  0.483527079f,
    0.874586652f,  0.484869248f,
    0.873841843f,  0.486210276f,
    0.873094978f,  0.487550160f,
    0.872346059f,  0.488888897f,
    0.871595087f,  0.490226483f,
    0.870842063f,  0.491562916f,
    0.870086991f,  0.492898192f,
    0.869329871f,  0.494232309f,
    0.868570706f,  0.495565262f,
    0.867809497f,  0.496897049f,
    0.867046246f,  0.498227667f,
    0.866280954f,  0.499557113f,
    0.865513624f,  0.500885383f,
    0.864744258f,  0.502212474f,
    0.863972856f,  0.503538384f,
    0.863199422f,  0.504863109f,
    0.862423956f,  0.506186645f,
    0.861646461f,  0.507508991f,
    0.860866939f,  0.508830143f,
    0.860085390f,  0.510150097f,
    0.859301818f,  0.511468850f,
    0.858516224f,  0.512786401f,
    0.857728610f,  0.514102744f,
    0.856938977f,  0.515417878f,
    0.856147328f,  0.516731799f,
    0.855353665f,  0.518044504f,
    0.854557988f,  0.519355990f,
    0.853760301f,  0.520666254f,
    0.852960605f,  0.521975293f,
    0.852158902f,  0.523283103f,
    0.851355193f,  0.524589683f,
    0.850549481f,  0.525895027f,
    0.849741768f,  0.527199135f,
    0.848932055f,  0.528502002f,
    0.848120345f,  0.529803625f,
    0.847306639f,  0.531104001f,
    0.846490939f,  0.532403128f,
    0.845673247f,  0.533701002f,
    0.844853565f,  0.534997620f,
    0.844031895f,  0.536292979f,
    0.843208240f,  0.537587076f,
    0.842382600f,  0.538879909f,
    0.841554977f,  0.540171473f,
    0.840725375f,  0.541461766f,
    0.839893794f,  0.542750785f,
    0.839060237f,  0.544038527f,
    0.838224706f,  0.545324988f,
    0.837387202f,  0.546610167f,
    0.836547727f,  0.547894059f,
    0.835706284f,  0.549176662f,
    0.834862875f,  0.550457973f,
    0.834017501f,  0.551737988f,
    0.833170165f,  0.553016706f,
    0.832320868f,  0.554294121f,
    0.831469612f,  0.555570233f,
    0.830616400f,  0.556845037f,
    0.829761234f,  0.558118531f,
    0.828904115f,  0.559390712f,
    0.828045045f,  0.560661576f,
    0.827184027f,  0.561931121f,
    0.826321063f,  0.563199344f,
    0.825456154f,  0.564466242f,
    0.824589303f,  0.565731811f,
    0.823720511f,  0.566996049f,
    0.822849781f,  0.568258953f,
    0.821977115f,  0.569520519f,
    0.821102515f,  0.570780746f,
    0.820225983f,  0.572039629f,
    0.819347520f,  0.573297167f,
    0.818467130f,  0.574553355f,
    0.817584813f,  0.575808191f,
    0.816700573f,  0.577061673f,
    0.815814411f,  0.578313796f,
    0.814926329f,  0.579564559f,
    0.814036330f,  0.580813958f,
    0.813144415f,  0.582061990f,
    0.812250587f,  0.583308653f,
    0.811354847f,  0.584553943f,
    0.810457198f,  0.585797857f,
    0.809557642f,  0.587040394f,
    0.808656182f,  0.588281548f,
    0.807752818f,  0.589521319f,
    0.806847554f,  0.590759702f,
    0.805940391f,  0.591996695f,
    0.805031331f,  0.593232295f,
    0.804120377f,  0.594466499f,
    0.803207531f,  0.595699304f,
    0.802292796f,  0.596930708f,
    0.801376172f,  0.598160707f,
    0.800457662f,  0.599389298f,
    0.799537269f,  0.600616479f,
    0.798614995f,  0.601842247f,
    0.797690841f,  0.603066599f,
    0.796764810f,  0.604289531f,
    0.795836905f,  0.605511041f,
    0.794907126f,  0.606731127f,
    0.793975478f,  0.607949785f,
    0.793041960f,  0.609167012f,
    0.792106577f,  0.610382806f,
    0.791169330f,  0.611597164f,
    0.790230221f,  0.612810082f,
    0.789289253f,  0.614021559f,
    0.788346428f,  0.615231591f,
    0.787401747f,  0.616440175f,
    0.786455214f,  0.617647308f,
    0.785506830f,  0.618852988f,
    0.784556597f,  0.620057212f,
    0.783604519f,  0.621259977f,
    0.782650596f,  0.622461279f,
    0.781694832f,  0.623661118f,
    0.780737229f,  0.624859488f,
    0.779777788f,  0.626056388f,
    0.778816512f,  0.627251815f,
    0.777853404f,  0.628445767f,
    0.776888466f,  0.629638239f,
    0.775921699f,  0.630829230f,
    0.774953107f,  0.632018736f,
    0.773982691f,  0.633206755f,
    0.773010453f,  0.634393284f,
    0.772036397f,  0.635578320f,
    0.771060524f,  0.636761861f,
    0.770082837f,  0.637943904f,
    0.769103338f,  0.639124445f,
    0.768122029f,  0.640303482f,
    0.767138912f,  0.641481013f,
    0.766153990f,  0.642657034f,
    0.765167266f,  0.643831543f,
    0.764178741f,  0.645004537f,
    0.763188417f,  0.646176013f,
    0.762196298f,  0.647345969f,
    0.761202385f,  0.648514401f,
    0.760206682f,  0.649681307f,
    0.759209189f,  0.650846685f,
    0.758209910f,  0.652010531f,
    0.757208847f,  0.653172843f,
    0.756206001f,  0.654333618f,
    0.755201377f,  0.655492853f,
    0.754194975f,  0.656650546f,
    0.753186799f,  0.657806693f,
    0.752176850f,  0.658961293f,
    0.751165132f,  0.660114342f,
    0.750151646f,  0.661265838f,
    0.749136395f,  0.662415778f,
    0.748119380f,  0.663564159f,
    0.747100606f,  0.664710978f,
    0.746080074f,  0.665856234f,
    0.745057785f,  0.666999922f,
    0.744033744f,  0.668142041f,
    0.743007952f,  0.669282588f,
    0.741980412f,  0.670421560f,
    0.740951125f,  0.671558955f,
    0.739920095f,  0.672694769f,
    0.738887324f,  0.673829000f,
    0.737852815f,  0.674961646f,
    0.736816569f,  0.676092704f,
    0.735778589f,  0.677222170f,
    0.734738878f,  0.678350043f,
    0.733697438f,  0.679476320f,
    0.732654272f,  0.680600998f,
    0.731609381f,  0.681724074f,
    0.730562769f,  0.682845546f,
    0.729514438f,  0.683965412f,
    0.728464390f,  0.685083668f,
    0.727412629f,  0.686200312f,
    0.726359155f,  0.687315341f,
    0.725303972f,  0.688428753f,
    0.724247083f,  0.689540545f,
    0.723188489f,  0.690650714f,
    0.722128194f,  0.691759258f,
    0.721066199f,  0.692866175f,
    0.720002508f,  0.693971461f,
    0.718937122f,  0.695075114f,
    0.717870045f,  0.696177131f,
    0.716801279f,  0.697277511f,
    0.715730825f,  0.698376249f,
    0.714658688f,  0.699473345f,
    0.713584869f,  0.700568794f,
    0.712509371f,  0.701662595f,
    0.711432196f,  0.702754744f,
    0.710353347f,  0.703845241f,
    0.709272826f,  0.704934080f,
    0.708190637f,  0.706021261f,
    0.707106781f,  0.707106781f,
    0.706021261f,  0.708190637f,
    0.704934080f,  0.709272826f,
    0.703845241f,  0.710353347f,
    0.702754744f,  0.711432196f,
    0.701662595f,  0.712509371f,
    0.700568794f,  0.713584869f,
    0.699473345f,  0.714658688f,
    0.698376249f,  0.715730825f,
    0.697277511f,  0.716801279f,
    0.696177131f,  0.717870045f,
    0.695075114f,  0.718937122f,
    0.693971461f,  0.720002508f,
    0.692866175f,  0.721066199f,
    0.691759258f,  0.722128194f,
    0.690650714f,  0.723188489f,
    0.689540545f,  0.724247083f,
    0.688428753f,  0.725303972f,
    0.687315341f,  0.726359155f,
    0.686200312f,  0.727412629f,
    0.685083668f,  0.728464390f,
    0.683965412f,  0.729514438f,
    0.682845546f,  0.730562769f,
    0.681724074f,  0.731609381f,
    0.680600998f,  0.732654272f,
    0.679476320f,  0.733697438f,
    0.678350043f,  0.734738878f,
    0.677222170f,  0.735778589f,
    0.676092704f,  0.736816569f,
    0.674961646f,  0.737852815f,
    0.673829000f,  0.738887324f,
    0.672694769f,  0.739920095f,
    0.671558955f,  0.740951125f,
    0.670421560f,  0.741980412f,
    0.669282588f,  0.743007952f,
    0.668142041f,  0.744033744f,
    0.666999922f,  0.745057785f,
    0.665856234f,  0.746080074f,
    0.664710978f,  0.747100606f,
    0.663564159f,  0.748119380f,
    0.662415778f,  0.749136395f,
    0.661265838f,  0.750151646f,
    0.660114342f,  0.751165132f,
    0.658961293f,  0.752176850f,
    0.657806693f,  0.753186799f,
    0.656650546f,  0.754194975f,
    0.655492853f,  0.755201377f,
    0.654333618f,  0.756206001f,
    0.653172843f,  0.757208847f,
    0.652010531f,  0.758209910f,
    0.650846685f,  0.759209189f,
    0.649681307f,  0.760206682f,
    0.648514401f,  0.761202385f,
    0.647345969f,  0.762196298f,
    0.646176013f,  0.763188417f,
    0.645004537f,  0.764178741f,
    0.643831543f,  0.765167266f,
    0.642657034f,  0.766153990f,
    0.641481013f,  0.767138912f,
    0.640303482f,  0.768122029f,
    0.639124445f,  0.769103338f,
    0.637943904f,  0.770082837f,
    0.636761861f,  0.771060524f,
    0.635578320f,  0.772036397f,
    0.634393284f,  0.773010453f,
    0.633206755f,  0.773982691f,
    0.632018736f,  0.774953107f,
    0.630829230f,  0.775921699f,
    0.629638239f,  0.776888466f,
    0.628445767f,  0.777853404f,
    0.627251815f,  0.778816512f,
    0.626056388f,  0.779777788f,
    0.624859488f,  0.780737229f,
    0.623661118f,  0.781694832f,
    0.622461279f,  0.782650596f,
    0.621259977f,  0.783604519f,
    0.620057212f,  0.784556597f,
    0.618852988f,  0.785506830f,
    0.617647308f,  0.786455214f,
    0.616440175f,  0.787401747f,
    0.615231591f,  0.788346428f,
    0.614021559f,  0.789289253f,
    0.612810082f,  0.790230221f,
    0.611597164f,  0.791169330f,
    0.610382806f,  0.792106577f,
    0.609167012f,  0.793041960f,
    0.607949785f,  0.793975478f,
    0.606731127f,  0.794907126f,
    0.605511041f,  0.795836905f,
    0.604289531f,  0.796764810f,
    0.603066599f,  0.797690841f,
    0.601842247f,  0.798614995f,
    0.600616479f,  0.799537269f,
    0.599389298f,  0.800457662f,
    0.598160707f,  0.801376172f,
    0.596930708f,  0.802292796f,
    0.595699304f,  0.803207531f,
    0.594466499f,  0.804120377f,
    0.593232295f,  0.805031331f,
    0.591996695f,  0.805940391f,
    0.590759702f,  0.806847554f,
    0.589521319f,  0.807752818f,
    0.588281548f,  0.808656182f,
    0.587040394f,  0.809557642f,
    0.585797857f,  0.810457198f,
    0.584553943f,  0.811354847f,
    0.583308653f,  0.812250587f,
    0.582061990f,  0.813144415f,
    0.580813958f,  0.814036330f,
    0.579564559f,  0.814926329f,
    0.578313796f,  0.815814411f,
    0.577061673f,  0.816700573f,
    0.575808191f,  0.817584813f,
    0.574553355f,  0.818467130f,
    0.573297167f,  0.819347520f,
    0.572039629f,  0.820225983f,
    0.570780746f,  0.821102515f,
    0.569520519f,  0.821977115f,
    0.568258953f,  0.822849781f,
    0.566996049f,  0.823720511f,
    0.565731811f,  0.824589303f,
    0.564466242f,  0.825456154f,
    0.563199344f,  0.826321063f,
    0.561931121f,  0.827184027f,
    0.560661576f,  0.828045045f,
    0.559390712f,  0.828904115f,
    0.558118531f,  0.829761234f,
    0.556845037f,  0.830616400f,
    0.555570233f,  0.831469612f,
    0.554294121f,  0.832320868f,
    0.553016706f,  0.833170165f,
    0.551737988f,  0.834017501f,
    0.550457973f,  0.834862875f,
    0.549176662f,  0.835706284f,
    0.547894059f,  0.836547727f,
    0.546610167f,  0.837387202f,
    0.545324988f,  0.838224706f,
    0.544038527f,  0.839060237f,
    0.542750785f,  0.839893794f,
    0.541461766f,  0.840725375f,
    0.540171473f,  0.841554977f,
    0.538879909f,  0.842382600f,
    0.537587076f,  0.843208240f,
    0.536292979f,  0.844031895f,
    0.534997620f,  0.844853565f,
    0.533701002f,  0.845673247f,
    0.532403128f,  0.846490939f,
    0.531104001f,  0.847306639f,
    0.529803625f,  0.848120345f,
    0.528502002f,  0.848932055f,
    0.527199135f,  0.849741768f,
    0.525895027f,  0.850549481f,
    0.524589683f,  0.851355193f,
    0.523283103f,  0.852158902f,
    0.521975293f,  0.852960605f,
    0.520666254f,  0.853760301f,
    0.519355990f,  0.854557988f,
    0.518044504f,  0.855353665f,
    0.516731799f,  0.856147328f,
    0.515417878f,  0.856938977f,
    0.514102744f,  0.857728610f,
    0.512786401f,  0.858516224f,
    0.511468850f,  0.859301818f,
    0.510150097f,  0.860085390f,
    0.508830143f,  0.860866939f,
    0.507508991f,  0.861646461f,
    0.506186645f,  0.862423956f,
    0.504863109f,  0.863199422f,
    0.503538384f,  0.863972856f,
    0.502212474f,  0.864744258f,
    0.500885383f,  0.865513624f,
    0.499557113f,  0.866280954f,
    0.498227667f,  0.867046246f,
    0.496897049f,  0.867809497f,
    0.495565262f,  0.868570706f,
    0.494232309f,  0.869329871f,
    0.492898192f,  0.870086991f,
    0.491562916f,  0.870842063f,
    0.490226483f,  0.871595087f,
    0.488888897f,  0.872346059f,
    0.487550160f,  0.873094978f,
    0.486210276f,  0.873841843f,
    0.484869248f,  0.874586652f,
    0.483527079f,  0.875329403f,
    0.482183772f,  0.876070094f,
    0.480839331f,  0.876808724f,
    0.479493758f,  0.877545290f,
    0.478147056f,  0.878279792f,
    0.476799230f,  0.879012226f,
    0.475450282f,  0.879742593f,
    0.474100215f,  0.880470889f,
    0.472749032f,  0.881197113f,
    0.471396737f,  0.881921264f,
    0.470043332f,  0.882643340f,
    0.468688822f,  0.883363339f,
    0.467333209f,  0.884081259f,
    0.465976496f,  0.884797098f,
    0.464618686f,  0.885510856f,
    0.463259784f,  0.886222530f,
    0.461899791f,  0.886932119f,
    0.460538711f,  0.887639620f,
    0.459176548f,  0.888345033f,
    0.457813304f,  0.889048356f,
    0.456448982f,  0.889749586f,
    0.455083587f,  0.890448723f,
    0.453717121f,  0.891145765f,
    0.452349587f,  0.891840709f,
    0.450980989f,  0.892533555f,
    0.449611330f,  0.893224301f,
    0.448240612f,  0.893912945f,
    0.446868840f,  0.894599486f,
    0.445496017f,  0.895283921f,
    0.444122145f,  0.895966250f,
    0.442747228f,  0.896646470f,
    0.441371269f,  0.897324581f,
    0.439994271f,  0.898000580f,
    0.438616239f,  0.898674466f,
    0.437237174f,  0.899346237f,
    0.435857080f,  0.900015892f,
    0.434475961f,  0.900683429f,
    0.433093819f,  0.901348847f,
    0.431710658f,  0.902012144f,
    0.430326481f,  0.902673318f,
    0.428941292f,  0.903332368f,
    0.427555093f,  0.903989293f,
    0.426167889f,  0.904644091f,
    0.424779681f,  0.905296759f,
    0.423390474f,  0.905947298f,
    0.422000271f,  0.906595705f,
    0.420609074f,  0.907241978f,
    0.419216888f,  0.907886116f,
    0.417823716f,  0.908528119f,
    0.416429560f,  0.909167983f,
    0.415034424f,  0.909805708f,
    0.413638312f,  0.910441292f,
    0.412241227f,  0.911074734f,
    0.410843171f,  0.911706032f,
    0.409444149f,  0.912335185f,
    0.408044163f,  0.912962190f,
    0.406643217f,  0.913587048f,
    0.405241314f,  0.914209756f,
    0.403838458f,  0.914830312f,
    0.402434651f,  0.915448716f,
    0.401029897f,  0.916064966f,
    0.399624200f,  0.916679060f,
    0.398217562f,  0.917290997f,
    0.396809987f,  0.917900776f,
    0.395401479f,  0.918508394f,
    0.393992040f,  0.919113852f,
    0.392581674f,  0.919717146f,
    0.391170384f,  0.920318277f,
    0.389758174f,  0.920917242f,
    0.388345047f,  0.921514039f,
    0.386931006f,  0.922108669f,
    0.385516054f,  0.922701128f,
    0.384100195f,  0.923291417f,
    0.382683432f,  0.923879533f,
    0.381265769f,  0.924465474f,
    0.379847209f,  0.925049241f,
    0.378427755f,  0.925630831f,
    0.377007410f,  0.926210242f,
    0.375586178f,  0.926787474f,
    0.374164063f,  0.927362526f,
    0.372741067f,  0.927935395f,
    0.371317194f,  0.928506080f,
    0.369892447f,  0.929074581f,
    0.368466830f,  0.929640896f,
    0.367040346f,  0.930205023f,
    0.365612998f,  0.930766961f,
    0.364184790f,  0.931326709f,
    0.362755724f,  0.931884266f,
    0.361325806f,  0.932439629f,
    0.359895037f,  0.932992799f,
    0.358463421f,  0.933543773f,
    0.357030961f,  0.934092550f,
    0.355597662f,  0.934639130f,
    0.354163525f,  0.935183510f,
    0.352728556f,  0.935725689f,
    0.351292756f,  0.936265667f,
    0.349856130f,  0.936803442f,
    0.348418680f,  0.937339012f,
    0.346980411f,  0.937872376f,
    0.345541325f,  0.938403534f,
    0.344101426f,  0.938932484f,
    0.342660717f,  0.939459224f,
    0.341219202f,  0.939983753f,
    0.339776884f,  0.940506071f,
    0.338333767f,  0.941026175f,
    0.336889853f,  0.941544065f,
    0.335445147f,  0.942059740f,
    0.333999651f,  0.942573198f,
    0.332553370f,  0.943084437f,
    0.331106306f,  0.943593458f,
    0.329658463f,  0.944100258f,
    0.328209844f,  0.944604837f,
    0.326760452f,  0.945107193f,
    0.325310292f,  0.945607325f,
    0.323859367f,  0.946105232f,
    0.322407679f,  0.946600913f,
    0.320955232f,  0.947094366f,
    0.319502031f,  0.947585591f,
    0.318048077f,  0.948074586f,
    0.316593376f,  0.948561350f,
    0.315137929f,  0.949045882f,
    0.313681740f,  0.949528181f,
    0.312224814f,  0.950008245f,
    0.310767153f,  0.950486074f,
    0.309308760f,  0.950961666f,
    0.307849640f,  0.951435021f,
    0.306389795f,  0.951906137f,
    0.304929230f,  0.952375013f,
    0.303467947f,  0.952841648f,
    0.302005949f,  0.953306040f,
    0.300543241f,  0.953768190f,
    0.299079826f,  0.954228095f,
    0.297615707f,  0.954685755f,
    0.296150888f,  0.955141168f,
    0.294685372f,  0.955594334f,
    0.293219163f,  0.956045251f,
    0.291752263f,  0.956493919f,
    0.290284677f,  0.956940336f,
    0.288816408f,  0.957384501f,
    0.287347460f,  0.957826413f,
    0.285877835f,  0.958266071f,
    0.284407537f,  0.958703475f,
    0.282936570f,  0.959138622f,
    0.281464938f,  0.959571513f,
    0.279992643f,  0.960002146f,
    0.278519689f,  0.960430519f,
    0.277046080f,  0.960856633f,
    0.275571819f,  0.961280486f,
    0.274096910f,  0.961702077f,
    0.272621355f,  0.962121404f,
    0.271145160f,  0.962538468f,
    0.269668326f,  0.962953267f,
    0.268190857f,  0.963365800f,
    0.266712757f,  0.963776066f,
    0.265234030f,  0.964184064f,
    0.263754679f,  0.964589793f,
    0.262274707f,  0.964993253f,
    0.260794118f,  0.965394442f,
    0.259312915f,  0.965793359f,
    0.257831102f,  0.966190003f,
    0.256348682f,  0.966584374f,
    0.254865660f,  0.966976471f,
    0.253382037f,  0.967366292f,
    0.251897818f,  0.967753837f,
    0.250413007f,  0.968139105f,
    0.248927606f,  0.968522094f,
    0.247441619f,  0.968902805f,
    0.245955050f,  0.969281235f,
    0.244467903f,  0.969657385f,
    0.242980180f,  0.970031253f,
    0.241491885f,  0.970402839f,
    0.240003022f,  0.970772141f,
    0.238513595f,  0.971139158f,
    0.237023606f,  0.971503891f,
    0.235533059f,  0.971866337f,
    0.234041959f,  0.972226497f,
    0.232550307f,  0.972584369f,
    0.231058108f,  0.972939952f,
    0.229565366f,  0.973293246f,
    0.228072083f,  0.973644250f,
    0.226578264f,  0.973992962f,
    0.225083911f,  0.974339383f,
    0.223589029f,  0.974683511f,
    0.222093621f,  0.975025345f,
    0.220597690f,  0.975364885f,
    0.219101240f,  0.975702130f,
    0.217604275f,  0.976037079f,
    0.216106797f,  0.976369731f,
    0.214608811f,  0.976700086f,
    0.213110320f,  0.977028143f,
    0.211611327f,  0.977353900f,
    0.210111837f,  0.977677358f,
    0.208611852f,  0.977998515f,
    0.207111376f,  0.978317371f,
    0.205610413f,  0.978633924f,
    0.204108966f,  0.978948175f,
    0.202607039f,  0.979260123f,
    0.201104635f,  0.979569766f,
    0.199601758f,  0.979877104f,
    0.198098411f,  0.980182136f,
    0.196594598f,  0.980484862f,
    0.195090322f,  0.980785280f,
    0.193585587f,  0.981083391f,
    0.192080397f,  0.981379193f,
    0.190574755f,  0.981672686f,
    0.189068664f,  0.981963869f,
    0.187562129f,  0.982252741f,
    0.186055152f,  0.982539302f,
    0.184547737f,  0.982823551f,
    0.183039888f,  0.983105487f,
    0.181531608f,  0.983385110f,
    0.180022901f,  0.983662419f,
    0.178513771f,  0.983937413f,
    0.177004220f,  0.984210092f,
    0.175494253f,  0.984480455f,
    0.173983873f,  0.984748502f,
    0.172473084f,  0.985014231f,
    0.170961889f,  0.985277642f,
    0.169450291f,  0.985538735f,
    0.167938295f,  0.985797509f,
    0.166425904f,  0.986053963f,
    0.164913120f,  0.986308097f,
    0.163399949f,  0.986559910f,
    0.161886394f,  0.986809402f,
    0.160372457f,  0.987056571f,
    0.158858143f,  0.987301418f,
    0.157343456f,  0.987543942f,
    0.155828398f,  0.987784142f,
    0.154312973f,  0.988022017f,
    0.152797185f,  0.988257568f,
    0.151281038f,  0.988490793f,
    0.149764535f,  0.988721692f,
    0.148247679f,  0.988950265f,
    0.146730474f,  0.989176510f,
    0.145212925f,  0.989400428f,
    0.143695033f,  0.989622017f,
    0.142176804f,  0.989841278f,
    0.140658239f,  0.990058210f,
    0.139139344f,  0.990272812f,
    0.137620122f,  0.990485084f,
    0.136100575f,  0.990695025f,
    0.134580709f,  0.990902635f,
    0.133060525f,  0.991107914f,
    0.131540029f,  0.991310860f,
    0.130019223f,  0.991511473f,
    0.128498111f,  0.991709754f,
    0.126976696f,  0.991905700f,
    0.125454983f,  0.992099313f,
    0.123932975f,  0.992290591f,
    0.122410675f,  0.992479535f,
    0.120888087f,  0.992666142f,
    0.119365215f,  0.992850414f,
    0.117842062f,  0.993032350f,
    0.116318631f,  0.993211949f,
    0.114794927f,  0.993389211f,
    0.113270952f,  0.993564136f,
    0.111746711f,  0.993736722f,
    0.110222207f,  0.993906970f,
    0.108697444f,  0.994074879f,
    0.107172425f,  0.994240449f,
    0.105647154f,  0.994403680f,
    0.104121634f,  0.994564571f,
    0.102595869f,  0.994723121f,
    0.101069863f,  0.994879331f,
    0.099543619f,  0.995033199f,
    0.098017140f,  0.995184727f,
    0.096490431f,  0.995333912f,
    0.094963495f,  0.995480755f,
    0.093436336f,  0.995625256f,
    0.091908956f,  0.995767414f,
    0.090381361f,  0.995907229f,
    0.088853553f,  0.996044701f,
    0.087325535f,  0.996179829f,
    0.085797312f,  0.996312612f,
    0.084268888f,  0.996443051f,
    0.082740265f,  0.996571146f,
    0.081211447f,  0.996696895f,
    0.079682438f,  0.996820299f,
    0.078153242f,  0.996941358f,
    0.076623861f,  0.997060070f,
    0.075094301f,  0.997176437f,
    0.073564564f,  0.997290457f,
    0.072034653f,  0.997402130f,
    0.070504573f,  0.997511456f,
    0.068974328f,  0.997618435f,
    0.067443920f,  0.997723067f,
    0.065913353f,  0.997825350f,
    0.064382631f,  0.997925286f,
    0.062851758f,  0.998022874f,
    0.061320736f,  0.998118113f,
    0.059789571f,  0.998211003f,
    0.058258265f,  0.998301545f,
    0.056726821f,  0.998389737f,
    0.055195244f,  0.998475581f,
    0.053663538f,  0.998559074f,
    0.052131705f,  0.998640218f,
    0.050599749f,  0.998719012f,
    0.049067674f,  0.998795456f,
    0.047535484f,  0.998869550f,
    0.046003182f,  0.998941293f,
    0.044470772f,  0.999010686f,
    0.042938257f,  0.999077728f,
    0.041405641f,  0.999142419f,
    0.039872928f,  0.999204759f,
    0.038340120f,  0.999264747f,
    0.036807223f,  0.999322385f,
    0.035274239f,  0.999377670f,
    0.033741172f,  0.999430605f,
    0.032208025f,  0.999481187f,
    0.030674803f,  0.999529418f,
    0.029141509f,  0.999575296f,
    0.027608146f,  0.999618822f,
    0.026074718f,  0.999659997f,
    0.024541229f,  0.999698819f,
    0.023007681f,  0.999735288f,
    0.021474080f,  0.999769405f,
    0.019940429f,  0.999801170f,
    0.018406730f,  0.999830582f,
    0.016872988f,  0.999857641f,
    0.015339206f,  0.999882347f,
    0.013805389f,  0.999904701f,
    0.012271538f,  0.999924702f,
    0.010737659f,  0.999942350f,
    0.009203755f,  0.999957645f,
    0.007669829f,  0.999970586f,
    0.006135885f,  0.999981175f,
    0.004601926f,  0.999989411f,
    0.003067957f,  0.999995294f,
    0.001533980f,  0.999998823f,
    0.000000000f,  1.000000000f,
   -0.001533980f,  0.999998823f,
   -0.003067957f,  0.999995294f,
   -0.004601926f,  0.999989411f,
   -0.006135885f,  0.999981175f,
   -0.007669829f,  0.999970586f,
   -0.009203755f,  0.999957645f,
   -0.010737659f,  0.999942350f,
   -0.012271538f,  0.999924702f,
   -0.013805389f,  0.999904701f,
   -0.015339206f,  0.999882347f,
   -0.016872988f,  0.999857641f,
   -0.018406730f,  0.999830582f,
   -0.019940429f,  0.999801170f,
   -0.021474080f,  0.999769405f,
   -0.023007681f,  0.999735288f,
   -0.024541229f,  0.999698819f,
   -0.026074718f,  0.999659997f,
   -0.027608146f,  0.999618822f,
   -0.029141509f,  0.999575296f,
   -0.030674803f,  0.999529418f,
   -0.032208025f,  0.999481187f,
   -0.033741172f,  0.999430605f,
   -0.035274239f,  0.999377670f,
   -0.036807223f,  0.999322385f,
   -0.038340120f,  0.999264747f,
   -0.039872928f,  0.999204759f,
   -0.041405641f,  0.999142419f,
   -0.042938257f,  0.999077728f,
   -0.044470772f,  0.999010686f,
   -0.046003182f,  0.998941293f,
   -0.047535484f,  0.998869550f,
   -0.049067674f,  0.998795456f,
   -0.050599749f,  0.998719012f,
   -0.052131705f,  0.998640218f,
   -0.053663538f,  0.998559074f,
   -0.055195244f,  0.998475581f,
   -0.056726821f,  0.998389737f,
   -0.058258265f,  0.998301545f,
   -0.059789571f,  0.998211003f,
   -0.061320736f,  0.998118113f,
   -0.062851758f,  0.998022874f,
   -0.064382631f,  0.997925286f,
   -0.065913353f,  0.997825350f,
   -0.067443920f,  0.997723067f,
   -0.068974328f,  0.997618435f,
   -0.070504573f,  0.997511456f,
   -0.072034653f,  0.997402130f,
   -0.073564564f,  0.997290457f,
   -0.075094301f,  0.997176437f,
   -0.076623861f,  0.997060070f,
   -0.078153242f,  0.996941358f,
   -0.079682438f,  0.996820299f,
   -0.081211447f,  0.996696895f,
   -0.082740265f,  0.996571146f,
   -0.084268888f,  0.996443051f,
   -0.085797312f,  0.996312612f,
   -0.087325535f,  0.996179829f,
   -0.088853553f,  0.996044701f,
   -0.090381361f,  0.995907229f,
   -0.091908956f,  0.995767414f,
   -0.093436336f,  0.995625256f,
   -0.094963495f,  0.995480755f,
   -0.096490431f,  0.995333912f,
   -0.098017140f,  0.995184727f,
   -0.099543619f,  0.995033199f,
   -0.101069863f,  0.994879331f,
   -0.102595869f,  0.994723121f,
   -0.104121634f,  0.994564571f,
   -0.105647154f,  0.994403680f,
   -0.107172425f,  0.994240449f,
   -0.108697444f,  0.994074879f,
   -0.110222207f,  0.993906970f,
   -0.111746711f,  0.993736722f,
   -0.113270952f,  0.993564136f,
   -0.114794927f,  0.993389211f,
   -0.116318631f,  0.993211949f,
   -0.117842062f,  0.993032350f,
   -0.119365215f,  0.992850414f,
   -0.120888087f,  0.992666142f,
   -0.122410675f,  0.992479535f,
   -0.123932975f,  0.992290591f,
   -0.125454983f,  0.992099313f,
   -0.126976696f,  0.991905700f,
   -0.128498111f,  0.991709754f,
   -0.130019223f,  0.991511473f,
   -0.131540029f,  0.991310860f,
   -0.133060525f,  0.991107914f,
   -0.134580709f,  0.990902635f,
   -0.136100575f,  0.990695025f,
   -0.137620122f,  0.990485084f,
   -0.139139344f,  0.990272812f,
   -0.140658239f,  0.990058210f,
   -0.142176804f,  0.989841278f,
   -0.143695033f,  0.989622017f,
   -0.145212925f,  0.989400428f,
   -0.146730474f,  0.989176510f,
   -0.148247679f,  0.988950265f,
   -0.149764535f,  0.988721692f,
   -0.151281038f,  0.988490793f,
   -0.152797185f,  0.988257568f,
   -0.154312973f,  0.988022017f,
   -0.155828398f,  0.987784142f,
   -0.157343456f,  0.987543942f,
   -0.158858143f,  0.987301418f,
   -0.160372457f,  0.987056571f,
   -0.161886394f,  0.986809402f,
   -0.163399949f,  0.986559910f,
   -0.164913120f,  0.986308097f,
   -0.166425904f,  0.986053963f,
   -0.167938295f,  0.985797509f,
   -0.169450291f,  0.985538735f,
   -0.170961889f,  0.985277642f,
   -0.172473084f,  0.985014231f,
   -0.173983873f,  0.984748502f,
   -0.175494253f,  0.984480455f,
   -0.177004220f,  0.984210092f,
   -0.178513771f,  0.983937413f,
   -0.180022901f,  0.983662419f,
   -0.181531608f,  0.983385110f,
   -0.183039888f,  0.983105487f,
   -0.184547737f,  0.982823551f,
   -0.186055152f,  0.982539302f,
   -0.187562129f,  0.982252741f,
   -0.189068664f,  0.981963869f,
   -0.190574755f,  0.981672686f,
   -0.192080397f,  0.981379193f,
   -0.193585587f,  0.981083391f,
   -0.195090322f,  0.980785280f,
   -0.196594598f,  0.980484862f,
   -0.198098411f,  0.980182136f,
   -0.199601758f,  0.979877104f,
   -0.201104635f,  0.979569766f,
   -0.202607039f,  0.979260123f,
   -0.204108966f,  0.978948175f,
   -0.205610413f,  0.978633924f,
   -0.207111376f,  0.978317371f,
   -0.208611852f,  0.977998515f,
   -0.210111837f,  0.977677358f,
   -0.211611327f,  0.977353900f,
   -0.213110320f,  0.977028143f,
   -0.214608811f,  0.976700086f,
   -0.216106797f,  0.976369731f,
   -0.217604275f,  0.976037079f,
   -0.219101240f,  0.975702130f,
   -0.220597690f,  0.975364885f,
   -0.222093621f,  0.975025345f,
   -0.223589029f,  0.974683511f,
   -0.225083911f,  0.974339383f,
   -0.226578264f,  0.973992962f,
   -0.228072083f,  0.973644250f,
   -0.229565366f,  0.973293246f,
   -0.231058108f,  0.972939952f,
   -0.232550307f,  0.972584369f,
   -0.234041959f,  0.972226497f,
   -0.235533059f,  0.971866337f,
   -0.237023606f,  0.971503891f,
   -0.238513595f,  0.971139158f,
   -0.240003022f,  0.970772141f,
   -0.241491885f,  0.970402839f,
   -0.242980180f,  0.970031253f,
   -0.244467903f,  0.969657385f,
   -0.245955050f,  0.969281235f,
   -0.247441619f,  0.968902805f,
   -0.248927606f,  0.968522094f,
   -0.250413007f,  0.968139105f,
   -0.251897818f,  0.967753837f,
   -0.253382037f,  0.967366292f,
   -0.254865660f,  0.966976471f,
   -0.256348682f,  0.966584374f,
   -0.257831102f,  0.966190003f,
   -0.259312915f,  0.965793359f,
   -0.260794118f,  0.965394442f,
   -0.262274707f,  0.964993253f,
   -0.263754679f,  0.964589793f,
   -0.265234030f,  0.964184064f,
   -0.266712757f,  0.963776066f,
   -0.268190857f,  0.963365800f,
   -0.269668326f,  0.962953267f,
   -0.271145160f,  0.962538468f,
   -0.272621355f,  0.962121404f,
   -0.274096910f,  0.961702077f,
   -0.275571819f,  0.961280486f,
   -0.277046080f,  0.960856633f,
   -0.278519689f,  0.960430519f,
   -0.279992643f,  0.960002146f,
   -0.281464938f,  0.959571513f,
   -0.282936570f,  0.959138622f,
   -0.284407537f,  0.958703475f,
   -0.285877835f,  0.958266071f,
   -0.287347460f,  0.957826413f,
   -0.288816408f,  0.957384501f,
   -0.290284677f,  0.956940336f,
   -0.291752263f,  0.956493919f,
   -0.293219163f,  0.956045251f,
   -0.294685372f,  0.955594334f,
   -0.296150888f,  0.955141168f,
   -0.297615707f,  0.954685755f,
   -0.299079826f,  0.954228095f,
   -0.300543241f,  0.953768190f,
   -0.302005949f,  0.953306040f,
   -0.303467947f,  0.952841648f,
   -0.304929230f,  0.952375013f,
   -0.306389795f,  0.951906137f,
   -0.307849640f,  0.951435021f,
   -0.309308760f,  0.950961666f,
   -0.310767153f,  0.950486074f,
   -0.312224814f,  0.950008245f,
   -0.313681740f,  0.949528181f,
   -0.315137929f,  0.949045882f,
   -0.316593376f,  0.948561350f,
   -0.318048077f,  0.948074586f,
   -0.319502031f,  0.947585591f,
   -0.320955232f,  0.947094366f,
   -0.322407679f,  0.946600913f,
   -0.323859367f,  0.946105232f,
   -0.325310292f,  0.945607325f,
   -0.326760452f,  0.945107193f,
   -0.328209844f,  0.944604837f,
   -0.329658463f,  0.944100258f,
   -0.331106306f,  0.943593458f,
   -0.332553370f,  0.943084437f,
   -0.333999651f,  0.942573198f,
   -0.335445147f,  0.942059740f,
   -0.336889853f,  0.941544065f,
   -0.338333767f,  0.941026175f,
   -0.339776884f,  0.940506071f,
   -0.341219202f,  0.939983753f,
   -0.342660717f,  0.939459224f,
   -0.344101426f,  0.938932484f,
   -0.345541325f,  0.938403534f,
   -0.346980411f,  0.937872376f,
   -0.348418680f,  0.937339012f,
   -0.349856130f,  0.936803442f,
   -0.351292756f,  0.936265667f,
   -0.352728556f,  0.935725689f,
   -0.354163525f,  0.935183510f,
   -0.355597662f,  0.934639130f,
   -0.357030961f,  0.934092550f,
   -0.358463421f,  0.933543773f,
   -0.359895037f,  0.932992799f,
   -0.361325806f,  0.932439629f,
   -0.362755724f,  0.931884266f,
   -0.364184790f,  0.931326709f,
   -0.365612998f,  0.930766961f,
   -0.367040346f,  0.930205023f,
   -0.368466830f,  0.929640896f,
   -0.369892447f,  0.929074581f,
   -0.371317194f,  0.928506080f,
   -0.372741067f,  0.927935395f,
   -0.374164063f,  0.927362526f,
   -0.375586178f,  0.926787474f,
   -0.377007410f,  0.926210242f,
   -0.378427755f,  0.925630831f,
   -0.379847209f,  0.925049241f,
   -0.381265769f,  0.924465474f,
   -0.382683432f,  0.923879533f,
   -0.384100195f,  0.923291417f,
   -0.385516054f,  0.922701128f,
   -0.386931006f,  0.922108669f,
   -0.388345047f,  0.921514039f,
   -0.389758174f,  0.920917242f,
   -0.391170384f,  0.920318277f,
   -0.392581674f,  0.919717146f,
   -0.393992040f,  0.919113852f,
   -0.395401479f,  0.918508394f,
   -0.396809987f,  0.917900776f,
   -0.398217562f,  0.917290997f,
   -0.399624200f,  0.916679060f,
   -0.401029897f,  0.916064966f,
   -0.402434651f,  0.915448716f,
   -0.403838458f,  0.914830312f,
   -0.405241314f,  0.914209756f,
   -0.406643217f,  0.913587048f,
   -0.408044163f,  0.912962190f,
   -0.409444149f,  0.912335185f,
   -0.410843171f,  0.911706032f,
   -0.412241227f,  0.911074734f,
   -0.413638312f,  0.910441292f,
   -0.415034424f,  0.909805708f,
   -0.416429560f,  0.909167983f,
   -0.417823716f,  0.908528119f,
   -0.419216888f,  0.907886116f,
   -0.420609074f,  0.907241978f,
   -0.422000271f,  0.906595705f,
   -0.423390474f,  0.905947298f,
   -0.424779681f,  0.905296759f,
   -0.426167889f,  0.904644091f,
   -0.427555093f,  0.903989293f,
   -0.428941292f,  0.903332368f,
   -0.430326481f,  0.902673318f,
   -0.431710658f,  0.902012144f,
   -0.433093819f,  0.901348847f,
   -0.434475961f,  0.900683429f,
   -0.435857080f,  0.900015892f,
   -0.437237174f,  0.899346237f,
   -0.438616239f,  0.898674466f,
   -0.439994271f,  0.898000580f,
   -0.441371269f,  0.897324581f,
   -0.442747228f,  0.896646470f,
   -0.444122145f,  0.895966250f,
   -0.445496017f,  0.895283921f,
   -0.446868840f,  0.894599486f,
   -0.448240612f,  0.893912945f,
   -0.449611330f,  0.893224301f,
   -0.450980989f,  0.892533555f,
   -0.452349587f,  0.891840709f,
   -0.453717121f,  0.891145765f,
   -0.455083587f,  0.890448723f,
   -0.456448982f,  0.889749586f,
   -0.457813304f,  0.889048356f,
   -0.459176548f,  0.888345033f,
   -0.460538711f,  0.887639620f,
   -0.461899791f,  0.886932119f,
   -0.463259784f,  0.886222530f,
   -0.464618686f,  0.885510856f,
   -0.465976496f,  0.884797098f,
   -0.467333209f,  0.884081259f,
   -0.468688822f,  0.883363339f,
   -0.470043332f,  0.882643340f,
   -0.471396737f,  0.881921264f,
   -0.472749032f,  0.881197113f,
   -0.474100215f,  0.880470889f,
   -0.475450282f,  0.879742593f,
   -0.476799230f,  0.879012226f,
   -0.478147056f,  0.878279792f,
   -0.479493758f,  0.877545290f,
   -0.480839331f,  0.876808724f,
   -0.482183772f,  0.876070094f,
   -0.483527079f,  0.875329403f,
   -0.484869248f,  0.874586652f,
   -0.486210276f,  0.873841843f,
   -0.487550160f,  0.873094978f,
   -0.488888897f,  0.872346059f,
   -0.490226483f,  0.871595087f,
   -0.491562916f,  0.870842063f,
   -0.492898192f,  0.870086991f,
   -0.494232309f,  0.869329871f,
   -0.495565262f,  0.868570706f,
   -0.496897049f,  0.867809497f,
   -0.498227667f,  0.867046246f,
   -0.499557113f,  0.866280954f,
   -0.500885383f,  0.865513624f,
   -0.502212474f,  0.864744258f,
   -0.503538384f,  0.863972856f,
   -0.504863109f,  0.863199422f,
   -0.506186645f,  0.862423956f,
   -0.507508991f,  0.861646461f,
   -0.508830143f,  0.860866939f,
   -0.510150097f,  0.860085390f,
   -0.511468850f,  0.859301818f,
   -0.512786401f,  0.858516224f,
   -0.514102744f,  0.857728610f,
   -0.515417878f,  0.856938977f,
   -0.516731799f,  0.856147328f,
   -0.518044504f,  0.855353665f,
   -0.519355990f,  0.854557988f,
   -0.520666254f,  0.853760301f,
   -0.521975293f,  0.852960605f,
   -0.523283103f,  0.852158902f,
   -0.524589683f,  0.851355193f,
   -0.525895027f,  0.850549481f,
   -0.527199135f,  0.849741768f,
   -0.528502002f,  0.848932055f,
   -0.529803625f,  0.848120345f,
   -0.531104001f,  0.847306639f,
   -0.532403128f,  0.846490939f,
   -0.533701002f,  0.845673247f,
   -0.534997620f,  0.844853565f,
   -0.536292979f,  0.844031895f,
   -0.537587076f,  0.843208240f,
   -0.538879909f,  0.842382600f,
   -0.540171473f,  0.841554977f,
   -0.541461766f,  0.840725375f,
   -0.542750785f,  0.839893794f,
   -0.544038527f,  0.839060237f,
   -0.545324988f,  0.838224706f,
   -0.546610167f,  0.837387202f,
   -0.547894059f,  0.836547727f,
   -0.549176662f,  0.835706284f,
   -0.550457973f,  0.834862875f,
   -0.551737988f,  0.834017501f,
   -0.553016706f,  0.833170165f,
   -0.554294121f,  0.832320868f,
   -0.555570233f,  0.831469612f,
   -0.556845037f,  0.830616400f,
   -0.558118531f,  0.829761234f,
   -0.559390712f,  0.828904115f,
   -0.560661576f,  0.828045045f,
   -0.561931121f,  0.827184027f,
   -0.563199344f,  0.826321063f,
   -0.564466242f,  0.825456154f,
   -0.565731811f,  0.824589303f,
   -0.566996049f,  0.823720511f,
   -0.568258953f,  0.822849781f,
   -0.569520519f,  0.821977115f,
   -0.570780746f,  0.821102515f,
   -0.572039629f,  0.820225983f,
   -0.573297167f,  0.819347520f,
   -0.574553355f,  0.818467130f,
   -0.575808191f,  0.817584813f,
   -0.577061673f,  0.816700573f,
   -0.578313796f,  0.815814411f,
   -0.579564559f,  0.814926329f,
   -0.580813958f,  0.814036330f,
   -0.582061990f,  0.813144415f,
   -0.583308653f,  0.812250587f,
   -0.584553943f,  0.811354847f,
   -0.585797857f,  0.810457198f,
   -0.587040394f,  0.809557642f,
   -0.588281548f,  0.808656182f,
   -0.589521319f,  0.807752818f,
   -0.590759702f,  0.806847554f,
   -0.591996695f,  0.805940391f,
   -0.593232295f,  0.805031331f,
   -0.594466499f,  0.804120377f,
   -0.595699304f,  0.803207531f,
   -0.596930708f,  0.802292796f,
   -0.598160707f,  0.801376172f,
   -0.599389298f,  0.800457662f,
   -0.600616479f,  0.799537269f,
   -0.601842247f,  0.798614995f,
   -0.603066599f,  0.797690841f,
   -0.604289531f,  0.796764810f,
   -0.605511041f,  0.795836905f,
   -0.606731127f,  0.794907126f,
   -0.607949785f,  0.793975478f,
   -0.609167012f,  0.793041960f,
   -0.610382806f,  0.792106577f,
   -0.611597164f,  0.791169330f,
   -0.612810082f,  0.790230221f,
   -0.614021559f,  0.789289253f,
   -0.615231591f,  0.788346428f,
   -0.616440175f,  0.787401747f,
   -0.617647308f,  0.786455214f,
   -0.618852988f,  0.785506830f,
   -0.620057212f,  0.784556597f,
   -0.621259977f,  0.783604519f,
   -0.622461279f,  0.782650596f,
   -0.623661118f,  0.781694832f,
   -0.624859488f,  0.780737229f,
   -0.626056388f,  0.779777788f,
   -0.627251815f,  0.778816512f,
   -0.628445767f,  0.777853404f,
   -0.629638239f,  0.776888466f,
   -0.630829230f,  0.775921699f,
   -0.632018736f,  0.774953107f,
   -0.633206755f,  0.773982691f,
   -0.634393284f,  0.773010453f,
   -0.635578320f,  0.772036397f,
   -0.636761861f,  0.771060524f,
   -0.637943904f,  0.770082837f,
   -0.639124445f,  0.769103338f,
   -0.640303482f,  0.768122029f,
   -0.641481013f,  0.767138912f,
   -0.642657034f,  0.766153990f,
   -0.643831543f,  0.765167266f,
   -0.645004537f,  0.764178741f,
   -0.646176013f,  0.763188417f,
   -0.647345969f,  0.762196298f,
   -0.648514401f,  0.761202385f,
   -0.649681307f,  0.760206682f,
   -0.650846685f,  0.759209189f,
   -0.652010531f,  0.758209910f,
   -0.653172843f,  0.757208847f,
   -0.654333618f,  0.756206001f,
   -0.655492853f,  0.755201377f,
   -0.656650546f,  0.754194975f,
   -0.657806693f,  0.753186799f,
   -0.658961293f,  0.752176850f,
   -0.660114342f,  0.751165132f,
   -0.661265838f,  0.750151646f,
   -0.662415778f,  0.749136395f,
   -0.663564159f,  0.748119380f,
   -0.664710978f,  0.747100606f,
   -0.665856234f,  0.746080074f,
   -0.666999922f,  0.745057785f,
   -0.668142041f,  0.744033744f,
   -0.669282588f,  0.743007952f,
   -0.670421560f,  0.741980412f,
   -0.671558955f,  0.740951125f,
   -0.672694769f,  0.739920095f,
   -0.673829000f,  0.738887324f,
   -0.674961646f,  0.737852815f,
   -0.676092704f,  0.736816569f,
   -0.677222170f,  0.735778589f,
   -0.678350043f,  0.734738878f,
   -0.679476320f,  0.733697438f,
   -0.680600998f,  0.732654272f,
   -0.681724074f,  0.731609381f,
   -0.682845546f,  0.730562769f,
   -0.683965412f,  0.729514438f,
   -0.685083668f,  0.728464390f,
   -0.686200312f,  0.727412629f,
   -0.687315341f,  0.726359155f,
   -0.688428753f,  0.725303972f,
   -0.689540545f,  0.724247083f,
   -0.690650714f,  0.723188489f,
   -0.691759258f,  0.722128194f,
   -0.692866175f,  0.721066199f,
   -0.693971461f,  0.720002508f,
   -0.695075114f,  0.718937122f,
   -0.696177131f,  0.717870045f,
   -0.697277511f,  0.716801279f,
   -0.698376249f,  0.715730825f,
   -0.699473345f,  0.714658688f,
   -0.700568794f,  0.713584869f,
   -0.701662595f,  0.712509371f,
   -0.702754744f,  0.711432196f,
   -0.703845241f,  0.710353347f,
   -0.704934080f,  0.709272826f,
   -0.706021261f,  0.708190637f,
   -0.707106781f,  0.707106781f,
   -0.708190637f,  0.706021261f,
   -0.709272826f,  0.704934080f,
   -0.710353347f,  0.703845241f,
   -0.711432196f,  0.702754744f,
   -0.712509371f,  0.701662595f,
   -0.713584869f,  0.700568794f,
   -0.714658688f,  0.699473345f,
   -0.715730825f,  0.698376249f,
   -0.716801279f,  0.697277511f,
   -0.717870045f,  0.696177131f,
   -0.718937122f,  0.695075114f,
   -0.720002508f,  0.693971461f,
   -0.721066199f,  0.692866175f,
   -0.722128194f,  0.691759258f,
   -0.723188489f,  0.690650714f,
   -0.724247083f,  0.689540545f,
   -0.725303972f,  0.688428753f,
   -0.726359155f,  0.687315341f,
   -0.727412629f,  0.686200312f,
   -0.728464390f,  0.685083668f,
   -0.729514438f,  0.683965412f,
   -0.730562769f,  0.682845546f,
   -0.731609381f,  0.681724074f,
   -0.732654272f,  0.680600998f,
   -0.733697438f,  0.679476320f,
   -0.734738878f,  0.678350043f,
   -0.735778589f,  0.677222170f,
   -0.736816569f,  0.676092704f,
   -0.737852815f,  0.674961646f,
   -0.738887324f,  0.673829000f,
   -0.739920095f,  0.672694769f,
   -0.740951125f,  0.671558955f,
   -0.741980412f,  0.670421560f,
   -0.743007952f,  0.669282588f,
   -0.744033744f,  0.668142041f,
   -0.745057785f,  0.666999922f,
   -0.746080074f,  0.665856234f,
   -0.747100606f,  0.664710978f,
   -0.748119380f,  0.663564159f,
   -0.749136395f,  0.662415778f,
   -0.750151646f,  0.661265838f,
   -0.751165132f,  0.660114342f,
   -0.752176850f,  0.658961293f,
   -0.753186799f,  0.657806693f,
   -0.754194975f,  0.656650546f,
   -0.755201377f,  0.655492853f,
   -0.756206001f,  0.654333618f,
   -0.757208847f,  0.653172843f,
   -0.758209910f,  0.652010531f,
   -0.759209189f,  0.650846685f,
   -0.760206682f,  0.649681307f,
   -0.761202385f,  0.648514401f,
   -0.762196298f,  0.647345969f,
   -0.763188417f,  0.646176013f,
   -0.764178741f,  0.645004537f,
   -0.765167266f,  0.643831543f,
   -0.766153990f,  0.642657034f,
   -0.767138912f,  0.641481013f,
   -0.768122029f,  0.640303482f,
   -0.769103338f,  0.639124445f,
   -0.770082837f,  0.637943904f,
   -0.771060524f,  0.636761861f,
   -0.772036397f,  0.635578320f,
   -0.773010453f,  0.634393284f,
   -0.773982691f,  0.633206755f,
   -0.774953107f,  0.632018736f,
   -0.775921699f,  0.630829230f,
   -0.776888466f,  0.629638239f,
   -0.777853404f,  0.628445767f,
   -0.778816512f,  0.627251815f,
   -0.779777788f,  0.626056388f,
   -0.780737229f,  0.624859488f,
   -0.781694832f,  0.623661118f,
   -0.782650596f,  0.622461279f,
   -0.783604519f,  0.621259977f,
   -0.784556597f,  0.620057212f,
   -0.785506830f,  0.618852988f,
   -0.786455214f,  0.617647308f,
   -0.787401747f,  0.616440175f,
   -0.788346428f,  0.615231591f,
   -0.789289253f,  0.614021559f,
   -0.790230221f,  0.612810082f,
   -0.791169330f,  0.611597164f,
   -0.792106577f,  0.610382806f,
   -0.793041960f,  0.609167012f,
   -0.793975478f,  0.607949785f,
   -0.794907126f,  0.606731127f,
   -0.795836905f,  0.605511041f,
   -0.796764810f,  0.604289531f,
   -0.797690841f,  0.603066599f,
   -0.798614995f,  0.601842247f,
   -0.799537269f,  0.600616479f,
   -0.800457662f,  0.599389298f,
   -0.801376172f,  0.598160707f,
   -0.802292796f,  0.596930708f,
   -0.803207531f,  0.595699304f,
   -0.804120377f,  0.594466499f,
   -0.805031331f,  0.593232295f,
   -0.805940391f,  0.591996695f,
   -0.806847554f,  0.590759702f,
   -0.807752818f,  0.589521319f,
   -0.808656182f,  0.588281548f,
   -0.809557642f,  0.587040394f,
   -0.810457198f,  0.585797857f,
   -0.811354847f,  0.584553943f,
   -0.812250587f,  0.583308653f,
   -0.813144415f,  0.582061990f,
   -0.814036330f,  0.580813958f,
   -0.814926329f,  0.579564559f,
   -0.815814411f,  0.578313796f,
   -0.816700573f,  0.577061673f,
   -0.817584813f,  0.575808191f,
   -0.818467130f,  0.574553355f,
   -0.819347520f,  0.573297167f,
   -0.820225983f,  0.572039629f,
   -0.821102515f,  0.570780746f,
   -0.821977115f,  0.569520519f,
   -0.822849781f,  0.568258953f,
   -0.823720511f,  0.566996049f,
   -0.824589303f,  0.565731811f,
   -0.825456154f,  0.564466242f,
   -0.826321063f,  0.563199344f,
   -0.827184027f,  0.561931121f,
   -0.828045045f,  0.560661576f,
   -0.828904115f,  0.559390712f,
   -0.829761234f,  0.558118531f,
   -0.830616400f,  0.556845037f,
   -0.831469612f,  0.555570233f,
   -0.832320868f,  0.554294121f,
   -0.833170165f,  0.553016706f,
   -0.834017501f,  0.551737988f,
   -0.834862875f,  0.550457973f,
   -0.835706284f,  0.549176662f,
   -0.836547727f,  0.547894059f,
   -0.837387202f,  0.546610167f,
   -0.838224706f,  0.545324988f,
   -0.839060237f,  0.544038527f,
   -0.839893794f,  0.542750785f,
   -0.840725375f,  0.541461766f,
   -0.841554977f,  0.540171473f,
   -0.842382600f,  0.538879909f,
   -0.843208240f,  0.537587076f,
   -0.844031895f,  0.536292979f,
   -0.844853565f,  0.534997620f,
   -0.845673247f,  0.533701002f,
   -0.846490939f,  0.532403128f,
   -0.847306639f,  0.531104001f,
   -0.848120345f,  0.529803625f,
   -0.848932055f,  0.528502002f,
   -0.849741768f,  0.527199135f,
   -0.850549481f,  0.525895027f,
   -0.851355193f,  0.524589683f,
   -0.852158902f,  0.523283103f,
   -0.852960605f,  0.521975293f,
   -0.853760301f,  0.520666254f,
   -0.854557988f,  0.519355990f,
   -0.855353665f,  0.518044504f,
   -0.856147328f,  0.516731799f,
   -0.856938977f,  0.515417878f,
   -0.857728610f,  0.514102744f,
   -0.858516224f,  0.512786401f,
   -0.859301818f,  0.511468850f,
   -0.860085390f,  0.510150097f,
   -0.860866939f,  0.508830143f,
   -0.861646461f,  0.507508991f,
   -0.862423956f,  0.506186645f,
   -0.863199422f,  0.504863109f,
   -0.863972856f,  0.503538384f,
   -0.864744258f,  0.502212474f,
   -0.865513624f,  0.500885383f,
   -0.866280954f,  0.499557113f,
   -0.867046246f,  0.498227667f,
   -0.867809497f,  0.496897049f,
   -0.868570706f,  0.495565262f,
   -0.869329871f,  0.494232309f,
   -0.870086991f,  0.492898192f,
   -0.870842063f,  0.491562916f,
   -0.871595087f,  0.490226483f,
   -0.872346059f,  0.488888897f,
   -0.873094978f,  0.487550160f,
   -0.873841843f,  0.486210276f,
   -0.874586652f,  0.484869248f,
   -0.875329403f,  0.483527079f,
   -0.876070094f,  0.482183772f,
   -0.876808724f,  0.480839331f,
   -0.877545290f,  0.479493758f,
   -0.878279792f,  0.478147056f,
   -0.879012226f,  0.476799230f,
   -0.879742593f,  0.475450282f,
   -0.880470889f,  0.474100215f,
   -0.881197113f,  0.472749032f,
   -0.881921264f,  0.471396737f,
   -0.882643340f,  0.470043332f,
   -0.883363339f,  0.468688822f,
   -0.884081259f,  0.467333209f,
   -0.884797098f,  0.465976496f,
   -0.885510856f,  0.464618686f,
   -0.886222530f,  0.463259784f,
   -0.886932119f,  0.461899791f,
   -0.887639620f,  0.460538711f,
   -0.888345033f,  0.459176548f,
   -0.889048356f,  0.457813304f,
   -0.889749586f,  0.456448982f,
   -0.890448723f,  0.455083587f,
   -0.891145765f,  0.453717121f,
   -0.891840709f,  0.452349587f,
   -0.892533555f,  0.450980989f,
   -0.893224301f,  0.449611330f,
   -0.893912945f,  0.448240612f,
   -0.894599486f,  0.446868840f,
   -0.895283921f,  0.445496017f,
   -0.895966250f,  0.444122145f,
   -0.896646470f,  0.442747228f,
   -0.897324581f,  0.441371269f,
   -0.898000580f,  0.439994271f,
   -0.898674466f,  0.438616239f,
   -0.899346237f,  0.437237174f,
   -0.900015892f,  0.435857080f,
   -0.900683429f,  0.434475961f,
   -0.901348847f,  0.433093819f,
   -0.902012144f,  0.431710658f,
   -0.902673318f,  0.430326481f,
   -0.903332368f,  0.428941292f,
   -0.903989293f,  0.427555093f,
   -0.904644091f,  0.426167889f,
   -0.905296759f,  0.424779681f,
   -0.905947298f,  0.423390474f,
   -0.906595705f,  0.422000271f,
   -0.907241978f,  0.420609074f,
   -0.907886116f,  0.419216888f,
   -0.908528119f,  0.417823716f,
   -0.909167983f,  0.416429560f,
   -0.909805708f,  0.415034424f,
   -0.910441292f,  0.413638312f,
   -0.911074734f,  0.412241227f,
   -0.911706032f,  0.410843171f,
   -0.912335185f,  0.409444149f,
   -0.912962190f,  0.408044163f,
   -0.913587048f,  0.406643217f,
   -0.914209756f,  0.405241314f,
   -0.914830312f,  0.403838458f,
   -0.915448716f,  0.402434651f,
   -0.916064966f,  0.401029897f,
   -0.916679060f,  0.399624200f,
   -0.917290997f,  0.398217562f,
   -0.917900776f,  0.396809987f,
   -0.918508394f,  0.395401479f,
   -0.919113852f,  0.393992040f,
   -0.919717146f,  0.392581674f,
   -0.920318277f,  0.391170384f,
   -0.920917242f,  0.389758174f,
   -0.921514039f,  0.388345047f,
   -0.922108669f,  0.386931006f,
   -0.922701128f,  0.385516054f,
   -0.923291417f,  0.384100195f,
   -0.923879533f,  0.382683432f,
   -0.924465474f,  0.381265769f,
   -0.925049241f,  0.379847209f,
   -0.925630831f,  0.378427755f,
   -0.926210242f,  0.377007410f,
   -0.926787474f,  0.375586178f,
   -0.927362526f,  0.374164063f,
   -0.927935395f,  0.372741067f,
   -0.928506080f,  0.371317194f,
   -0.929074581f,  0.369892447f,
   -0.929640896f,  0.368466830f,
   -0.930205023f,  0.367040346f,
   -0.930766961f,  0.365612998f,
   -0.931326709f,  0.364184790f,
   -0.931884266f,  0.362755724f,
   -0.932439629f,  0.361325806f,
   -0.932992799f,  0.359895037f,
   -0.933543773f,  0.358463421f,
   -0.934092550f,  0.357030961f,
   -0.934639130f,  0.355597662f,
   -0.935183510f,  0.354163525f,
   -0.935725689f,  0.352728556f,
   -0.936265667f,  0.351292756f,
   -0.936803442f,  0.349856130f,
   -0.937339012f,  0.348418680f,
   -0.937872376f,  0.346980411f,
   -0.938403534f,  0.345541325f,
   -0.938932484f,  0.344101426f,
   -0.939459224f,  0.342660717f,
   -0.939983753f,  0.341219202f,
   -0.940506071f,  0.339776884f,
   -0.941026175f,  0.338333767f,
   -0.941544065f,  0.336889853f,
   -0.942059740f,  0.335445147f,
   -0.942573198f,  0.333999651f,
   -0.943084437f,  0.332553370f,
   -0.943593458f,  0.331106306f,
   -0.944100258f,  0.329658463f,
   -0.944604837f,  0.328209844f,
   -0.945107193f,  0.326760452f,
   -0.945607325f,  0.325310292f,
   -0.946105232f,  0.323859367f,
   -0.946600913f,  0.322407679f,
   -0.947094366f,  0.320955232f,
   -0.947585591f,  0.319502031f,
   -0.948074586f,  0.318048077f,
   -0.948561350f,  0.316593376f,
   -0.949045882f,  0.315137929f,
   -0.949528181f,  0.313681740f,
   -0.950008245f,  0.312224814f,
   -0.950486074f,  0.310767153f,
   -0.950961666f,  0.309308760f,
   -0.951435021f,  0.307849640f,
   -0.951906137f,  0.306389795f,
   -0.952375013f,  0.304929230f,
   -0.952841648f,  0.303467947f,
   -0.953306040f,  0.302005949f,
   -0.953768190f,  0.300543241f,
   -0.954228095f,  0.299079826f,
   -0.954685755f,  0.297615707f,
   -0.955141168f,  0.296150888f,
   -0.955594334f,  0.294685372f,
   -0.956045251f,  0.293219163f,
   -0.956493919f,  0.291752263f,
   -0.956940336f,  0.290284677f,
   -0.957384501f,  0.288816408f,
   -0.957826413f,  0.287347460f,
   -0.958266071f,  0.285877835f,
   -0.958703475f,  0.284407537f,
   -0.959138622f,  0.282936570f,
   -0.959571513f,  0.281464938f,
   -0.960002146f,  0.279992643f,
   -0.960430519f,  0.278519689f,
   -0.960856633f,  0.277046080f,
   -0.961280486f,  0.275571819f,
   -0.961702077f,  0.274096910f,
   -0.962121404f,  0.272621355f,
   -0.962538468f,  0.271145160f,
   -0.962953267f,  0.269668326f,
   -0.963365800f,  0.268190857f,
   -0.963776066f,  0.266712757f,
   -0.964184064f,  0.265234030f,
   -0.964589793f,  0.263754679f,
   -0.964993253f,  0.262274707f,
   -0.965394442f,  0.260794118f,
   -0.965793359f,  0.259312915f,
   -0.966190003f,  0.257831102f,
   -0.966584374f,  0.256348682f,
   -0.966976471f,  0.254865660f,
   -0.967366292f,  0.253382037f,
   -0.967753837f,  0.251897818f,
   -0.968139105f,  0.250413007f,
   -0.968522094f,  0.248927606f,
   -0.968902805f,  0.247441619f,
   -0.969281235f,  0.245955050f,
   -0.969657385f,  0.244467903f,
   -0.970031253f,  0.242980180f,
   -0.970402839f,  0.241491885f,
   -0.970772141f,  0.240003022f,
   -0.971139158f,  0.238513595f,
   -0.971503891f,  0.237023606f,
   -0.971866337f,  0.235533059f,
   -0.972226497f,  0.234041959f,
   -0.972584369f,  0.232550307f,
   -0.972939952f,  0.231058108f,
   -0.973293246f,  0.229565366f,
   -0.973644250f,  0.228072083f,
   -0.973992962f,  0.226578264f,
   -0.974339383f,  0.225083911f,
   -0.974683511f,  0.223589029f,
   -0.975025345f,  0.222093621f,
   -0.975364885f,  0.220597690f,
   -0.975702130f,  0.219101240f,
   -0.976037079f,  0.217604275f,
   -0.976369731f,  0.216106797f,
   -0.976700086f,  0.214608811f,
   -0.977028143f,  0.213110320f,
   -0.977353900f,  0.211611327f,
   -0.977677358f,  0.210111837f,
   -0.977998515f,  0.208611852f,
   -0.978317371f,  0.207111376f,
   -0.978633924f,  0.205610413f,
   -0.978948175f,  0.204108966f,
   -0.979260123f,  0.202607039f,
   -0.979569766f,  0.201104635f,
   -0.979877104f,  0.199601758f,
   -0.980182136f,  0.198098411f,
   -0.980484862f,  0.196594598f,
   -0.980785280f,  0.195090322f,
   -0.981083391f,  0.193585587f,
   -0.981379193f,  0.192080397f,
   -0.981672686f,  0.190574755f,
   -0.981963869f,  0.189068664f,
   -0.982252741f,  0.187562129f,
   -0.982539302f,  0.186055152f,
   -0.982823551f,  0.184547737f,
   -0.983105487f,  0.183039888f,
   -0.983385110f,  0.181531608f,
   -0.983662419f,  0.180022901f,
   -0.983937413f,  0.178513771f,
   -0.984210092f,  0.177004220f,
   -0.984480455f,  0.175494253f,
   -0.984748502f,  0.173983873f,
   -0.985014231f,  0.172473084f,
   -0.985277642f,  0.170961889f,
   -0.985538735f,  0.169450291f,
   -0.985797509f,  0.167938295f,
   -0.986053963f,  0.166425904f,
   -0.986308097f,  0.164913120f,
   -0.986559910f,  0.163399949f,
   -0.986809402f,  0.161886394f,
   -0.987056571f,  0.160372457f,
   -0.987301418f,  0.158858143f,
   -0.987543942f,  0.157343456f,
   -0.987784142f,  0.155828398f,
   -0.988022017f,  0.154312973f,
   -0.988257568f,  0.152797185f,
   -0.988490793f,  0.151281038f,
   -0.988721692f,  0.149764535f,
   -0.988950265f,  0.148247679f,
   -0.989176510f,  0.146730474f,
   -0.989400428f,  0.145212925f,
   -0.989622017f,  0.143695033f,
   -0.989841278f,  0.142176804f,
   -0.990058210f,  0.140658239f,
   -0.990272812f,  0.139139344f,
   -0.990485084f,  0.137620122f,
   -0.990695025f,  0.136100575f,
   -0.990902635f,  0.134580709f,
   -0.991107914f,  0.133060525f,
   -0.991310860f,  0.131540029f,
   -0.991511473f,  0.130019223f,
   -0.991709754f,  0.128498111f,
   -0.991905700f,  0.126976696f,
   -0.992099313f,  0.125454983f,
   -0.992290591f,  0.123932975f,
   -0.992479535f,  0.122410675f,
   -0.992666142f,  0.120888087f,
   -0.992850414f,  0.119365215f,
   -0.993032350f,  0.117842062f,
   -0.993211949f,  0.116318631f,
   -0.993389211f,  0.114794927f,
   -0.993564136f,  0.113270952f,
   -0.993736722f,  0.111746711f,
   -0.993906970f,  0.110222207f,
   -0.994074879f,  0.108697444f,
   -0.994240449f,  0.107172425f,
   -0.994403680f,  0.105647154f,
   -0.994564571f,  0.104121634f,
   -0.994723121f,  0.102595869f,
   -0.994879331f,  0.101069863f,
   -0.995033199f,  0.099543619f,
   -0.995184727f,  0.098017140f,
   -0.995333912f,  0.096490431f,
   -0.995480755f,  0.094963495f,
   -0.995625256f,  0.093436336f,
   -0.995767414f,  0.091908956f,
   -0.995907229f,  0.090381361f,
   -0.996044701f,  0.088853553f,
   -0.996179829f,  0.087325535f,
   -0.996312612f,  0.085797312f,
   -0.996443051f,  0.084268888f,
   -0.996571146f,  0.082740265f,
   -0.996696895f,  0.081211447f,
   -0.996820299f,  0.079682438f,
   -0.996941358f,  0.078153242f,
   -0.997060070f,  0.076623861f,
   -0.997176437f,  0.075094301f,
   -0.997290457f,  0.073564564f,
   -0.997402130f,  0.072034653f,
   -0.997511456f,  0.070504573f,
   -0.997618435f,  0.068974328f,
   -0.997723067f,  0.067443920f,
   -0.997825350f,  0.065913353f,
   -0.997925286f,  0.064382631f,
   -0.998022874f,  0.062851758f,
   -0.998118113f,  0.061320736f,
   -0.998211003f,  0.059789571f,
   -0.998301545f,  0.058258265f,
   -0.998389737f,  0.056726821f,
   -0.998475581f,  0.055195244f,
   -0.998559074f,  0.053663538f,
   -0.998640218f,  0.052131705f,
   -0.998719012f,  0.050599749f,
   -0.998795456f,  0.049067674f,
   -0.998869550f,  0.047535484f,
   -0.998941293f,  0.046003182f,
   -0.999010686f,  0.044470772f,
   -0.999077728f,  0.042938257f,
   -0.999142419f,  0.041405641f,
   -0.999204759f,  0.039872928f,
   -0.999264747f,  0.038340120f,
   -0.999322385f,  0.036807223f,
   -0.999377670f,  0.035274239f,
   -0.999430605f,  0.033741172f,
   -0.999481187f,  0.032208025f,
   -0.999529418f,  0.030674803f,
   -0.999575296f,  0.029141509f,
   -0.999618822f,  0.027608146f,
   -0.999659997f,  0.026074718f,
   -0.999698819f,  0.024541229f,
   -0.999735288f,  0.023007681f,
   -0.999769405f,  0.021474080f,
   -0.999801170f,  0.019940429f,
   -0.999830582f,  0.018406730f,
   -0.999857641f,  0.016872988f,
   -0.999882347f,  0.015339206f,
   -0.999904701f,  0.013805389f,
   -0.999924702f,  0.012271538f,
   -0.999942350f,  0.010737659f,
   -0.999957645f,  0.009203755f,
   -0.999970586f,  0.007669829f,
   -0.999981175f,  0.006135885f,
   -0.999989411f,  0.004601926f,
   -0.999995294f,  0.003067957f,
   -0.999998823f,  0.001533980f,
   -1.000000000f,  0.000000000f,
   -0.999998823f, -0.001533980f,
   -0.999995294f, -0.003067957f,
   -0.999989411f, -0.004601926f,
   -0.999981175f, -0.006135885f,
   -0.999970586f, -0.007669829f,
   -0.999957645f, -0.009203755f,
   -0.999942350f, -0.010737659f,
   -0.999924702f, -0.012271538f,
   -0.999904701f, -0.013805389f,
   -0.999882347f, -0.015339206f,
   -0.999857641f, -0.016872988f,
   -0.999830582f, -0.018406730f,
   -0.999801170f, -0.019940429f,
   -0.999769405f, -0.021474080f,
   -0.999735288f, -0.023007681f,
   -0.999698819f, -0.024541229f,
   -0.999659997f, -0.026074718f,
   -0.999618822f, -0.027608146f,
   -0.999575296f, -0.029141509f,
   -0.999529418f, -0.030674803f,
   -0.999481187f, -0.032208025f,
   -0.999430605f, -0.033741172f,
   -0.999377670f, -0.035274239f,
   -0.999322385f, -0.036807223f,
   -0.999264747f, -0.038340120f,
   -0.999204759f, -0.039872928f,
   -0.999142419f, -0.041405641f,
   -0.999077728f, -0.042938257f,
   -0.999010686f, -0.044470772f,
   -0.998941293f, -0.046003182f,
   -0.998869550f, -0.047535484f,
   -0.998795456f, -0.049067674f,
   -0.998719012f, -0.050599749f,
   -0.998640218f, -0.052131705f,
   -0.998559074f, -0.053663538f,
   -0.998475581f, -0.055195244f,
   -0.998389737f, -0.056726821f,
   -0.998301545f, -0.058258265f,
   -0.998211003f, -0.059789571f,
   -0.998118113f, -0.061320736f,
   -0.998022874f, -0.062851758f,
   -0.997925286f, -0.064382631f,
   -0.997825350f, -0.065913353f,
   -0.997723067f, -0.067443920f,
   -0.997618435f, -0.068974328f,
   -0.997511456f, -0.070504573f,
   -0.997402130f, -0.072034653f,
   -0.997290457f, -0.073564564f,
   -0.997176437f, -0.075094301f,
   -0.997060070f, -0.076623861f,
   -0.996941358f, -0.078153242f,
   -0.996820299f, -0.079682438f,
   -0.996696895f, -0.081211447f,
   -0.996571146f, -0.082740265f,
   -0.996443051f, -0.084268888f,
   -0.996312612f, -0.085797312f,
   -0.996179829f, -0.087325535f,
   -0.996044701f, -0.088853553f,
   -0.995907229f, -0.090381361f,
   -0.995767414f, -0.091908956f,
   -0.995625256f, -0.093436336f,
   -0.995480755f, -0.094963495f,
   -0.995333912f, -0.096490431f,
   -0.995184727f, -0.098017140f,
   -0.995033199f, -0.099543619f,
   -0.994879331f, -0.101069863f,
   -0.994723121f, -0.102595869f,
   -0.994564571f, -0.104121634f,
   -0.994403680f, -0.105647154f,
   -0.994240449f, -0.107172425f,
   -0.994074879f, -0.108697444f,
   -0.993906970f, -0.110222207f,
   -0.993736722f, -0.111746711f,
   -0.993564136f, -0.113270952f,
   -0.993389211f, -0.114794927f,
   -0.993211949f, -0.116318631f,
   -0.993032350f, -0.117842062f,
   -0.992850414f, -0.119365215f,
   -0.992666142f, -0.120888087f,
   -0.992479535f, -0.122410675f,
   -0.992290591f, -0.123932975f,
   -0.992099313f, -0.125454983f,
   -0.991905700f, -0.126976696f,
   -0.991709754f, -0.128498111f,
   -0.991511473f, -0.130019223f,
   -0.991310860f, -0.131540029f,
   -0.991107914f, -0.133060525f,
   -0.990902635f, -0.134580709f,
   -0.990695025f, -0.136100575f,
   -0.990485084f, -0.137620122f,
   -0.990272812f, -0.139139344f,
   -0.990058210f, -0.140658239f,
   -0.989841278f, -0.142176804f,
   -0.989622017f, -0.143695033f,
   -0.989400428f, -0.145212925f,
   -0.989176510f, -0.146730474f,
   -0.988950265f, -0.148247679f,
   -0.988721692f, -0.149764535f,
   -0.988490793f, -0.151281038f,
   -0.988257568f, -0.152797185f,
   -0.988022017f, -0.154312973f,
   -0.987784142f, -0.155828398f,
   -0.987543942f, -0.157343456f,
   -0.987301418f, -0.158858143f,
   -0.987056571f, -0.160372457f,
   -0.986809402f, -0.161886394f,
   -0.986559910f, -0.163399949f,
   -0.986308097f, -0.164913120f,
   -0.986053963f, -0.166425904f,
   -0.985797509f, -0.167938295f,
   -0.985538735f, -0.169450291f,
   -0.985277642f, -0.170961889f,
   -0.985014231f, -0.172473084f,
   -0.984748502f, -0.173983873f,
   -0.984480455f, -0.175494253f,
   -0.984210092f, -0.177004220f,
   -0.983937413f, -0.178513771f,
   -0.983662419f, -0.180022901f,
   -0.983385110f, -0.181531608f,
   -0.983105487f, -0.183039888f,
   -0.982823551f, -0.184547737f,
   -0.982539302f, -0.186055152f,
   -0.982252741f, -0.187562129f,
   -0.981963869f, -0.189068664f,
   -0.981672686f, -0.190574755f,
   -0.981379193f, -0.192080397f,
   -0.981083391f, -0.193585587f,
   -0.980785280f, -0.195090322f,
   -0.980484862f, -0.196594598f,
   -0.980182136f, -0.198098411f,
   -0.979877104f, -0.199601758f,
   -0.979569766f, -0.201104635f,
   -0.979260123f, -0.202607039f,
   -0.978948175f, -0.204108966f,
   -0.978633924f, -0.205610413f,
   -0.978317371f, -0.207111376f,
   -0.977998515f, -0.208611852f,
   -0.977677358f, -0.210111837f,
   -0.977353900f, -0.211611327f,
   -0.977028143f, -0.213110320f,
   -0.976700086f, -0.214608811f,
   -0.976369731f, -0.216106797f,
   -0.976037079f, -0.217604275f,
   -0.975702130f, -0.219101240f,
   -0.975364885f, -0.220597690f,
   -0.975025345f, -0.222093621f,
   -0.974683511f, -0.223589029f,
   -0.974339383f, -0.225083911f,
   -0.973992962f, -0.226578264f,
   -0.973644250f, -0.228072083f,
   -0.973293246f, -0.229565366f,
   -0.972939952f, -0.231058108f,
   -0.972584369f, -0.232550307f,
   -0.972226497f, -0.234041959f,
   -0.971866337f, -0.235533059f,
   -0.971503891f, -0.237023606f,
   -0.971139158f, -0.238513595f,
   -0.970772141f, -0.240003022f,
   -0.970402839f, -0.241491885f,
   -0.970031253f, -0.242980180f,
   -0.969657385f, -0.244467903f,
   -0.969281235f, -0.245955050f,
   -0.968902805f, -0.247441619f,
   -0.968522094f, -0.248927606f,
   -0.968139105f, -0.250413007f,
   -0.967753837f, -0.251897818f,
   -0.967366292f, -0.253382037f,
   -0.966976471f, -0.254865660f,
   -0.966584374f, -0.256348682f,
   -0.966190003f, -0.257831102f,
   -0.965793359f, -0.259312915f,
   -0.965394442f, -0.260794118f,
   -0.964993253f, -0.262274707f,
   -0.964589793f, -0.263754679f,
   -0.964184064f, -0.265234030f,
   -0.963776066f, -0.266712757f,
   -0.963365800f, -0.268190857f,
   -0.962953267f, -0.269668326f,
   -0.962538468f, -0.271145160f,
   -0.962121404f, -0.272621355f,
   -0.961702077f, -0.274096910f,
   -0.961280486f, -0.275571819f,
   -0.960856633f, -0.277046080f,
   -0.960430519f, -0.278519689f,
   -0.960002146f, -0.279992643f,
   -0.959571513f, -0.281464938f,
   -0.959138622f, -0.282936570f,
   -0.958703475f, -0.284407537f,
   -0.958266071f, -0.285877835f,
   -0.957826413f, -0.287347460f,
   -0.957384501f, -0.288816408f,
   -0.956940336f, -0.290284677f,
   -0.956493919f, -0.291752263f,
   -0.956045251f, -0.293219163f,
   -0.955594334f, -0.294685372f,
   -0.955141168f, -0.296150888f,
   -0.954685755f, -0.297615707f,
   -0.954228095f, -0.299079826f,
   -0.953768190f, -0.300543241f,
   -0.953306040f, -0.302005949f,
   -0.952841648f, -0.303467947f,
   -0.952375013f, -0.304929230f,
   -0.951906137f, -0.306389795f,
   -0.951435021f, -0.307849640f,
   -0.950961666f, -0.309308760f,
   -0.950486074f, -0.310767153f,
   -0.950008245f, -0.312224814f,
   -0.949528181f, -0.313681740f,
   -0.949045882f, -0.315137929f,
   -0.948561350f, -0.316593376f,
   -0.948074586f, -0.318048077f,
   -0.947585591f, -0.319502031f,
   -0.947094366f, -0.320955232f,
   -0.946600913f, -0.322407679f,
   -0.946105232f, -0.323859367f,
   -0.945607325f, -0.325310292f,
   -0.945107193f, -0.326760452f,
   -0.944604837f, -0.328209844f,
   -0.944100258f, -0.329658463f,
   -0.943593458f, -0.331106306f,
   -0.943084437f, -0.332553370f,
   -0.942573198f, -0.333999651f,
   -0.942059740f, -0.335445147f,
   -0.941544065f, -0.336889853f,
   -0.941026175f, -0.338333767f,
   -0.940506071f, -0.339776884f,
   -0.939983753f, -0.341219202f,
   -0.939459224f, -0.342660717f,
   -0.938932484f, -0.344101426f,
   -0.938403534f, -0.345541325f,
   -0.937872376f, -0.346980411f,
   -0.937339012f, -0.348418680f,
   -0.936803442f, -0.349856130f,
   -0.936265667f, -0.351292756f,
   -0.935725689f, -0.352728556f,
   -0.935183510f, -0.354163525f,
   -0.934639130f, -0.355597662f,
   -0.934092550f, -0.357030961f,
   -0.933543773f, -0.358463421f,
   -0.932992799f, -0.359895037f,
   -0.932439629f, -0.361325806f,
   -0.931884266f, -0.362755724f,
   -0.931326709f, -0.364184790f,
   -0.930766961f, -0.365612998f,
   -0.930205023f, -0.367040346f,
   -0.929640896f, -0.368466830f,
   -0.929074581f, -0.369892447f,
   -0.928506080f, -0.371317194f,
   -0.927935395f, -0.372741067f,
   -0.927362526f, -0.374164063f,
   -0.926787474f, -0.375586178f,
   -0.926210242f, -0.377007410f,
   -0.925630831f, -0.378427755f,
   -0.925049241f, -0.379847209f,
   -0.924465474f, -0.381265769f,
   -0.923879533f, -0.382683432f,
   -0.923291417f, -0.384100195f,
   -0.922701128f, -0.385516054f,
   -0.922108669f, -0.386931006f,
   -0.921514039f, -0.388345047f,
   -0.920917242f, -0.389758174f,
   -0.920318277f, -0.391170384f,
   -0.919717146f, -0.392581674f,
   -0.919113852f, -0.393992040f,
   -0.918508394f, -0.395401479f,
   -0.917900776f, -0.396809987f,
   -0.917290997f, -0.398217562f,
   -0.916679060f, -0.399624200f,
   -0.916064966f, -0.401029897f,
   -0.915448716f, -0.402434651f,
   -0.914830312f, -0.403838458f,
   -0.914209756f, -0.405241314f,
   -0.913587048f, -0.406643217f,
   -0.912962190f, -0.408044163f,
   -0.912335185f, -0.409444149f,
   -0.911706032f, -0.410843171f,
   -0.911074734f, -0.412241227f,
   -0.910441292f, -0.413638312f,
   -0.909805708f, -0.415034424f,
   -0.909167983f, -0.416429560f,
   -0.908528119f, -0.417823716f,
   -0.907886116f, -0.419216888f,
   -0.907241978f, -0.420609074f,
   -0.906595705f, -0.422000271f,
   -0.905947298f, -0.423390474f,
   -0.905296759f, -0.424779681f,
   -0.904644091f, -0.426167889f,
   -0.903989293f, -0.427555093f,
   -0.903332368f, -0.428941292f,
   -0.902673318f, -0.430326481f,
   -0.902012144f, -0.431710658f,
   -0.901348847f, -0.433093819f,
   -0.900683429f, -0.434475961f,
   -0.900015892f, -0.435857080f,
   -0.899346237f, -0.437237174f,
   -0.898674466f, -0.438616239f,
   -0.898000580f, -0.439994271f,
   -0.897324581f, -0.441371269f,
   -0.896646470f, -0.442747228f,
   -0.895966250f, -0.444122145f,
   -0.895283921f, -0.445496017f,
   -0.894599486f, -0.446868840f,
   -0.893912945f, -0.448240612f,
   -0.893224301f, -0.449611330f,
   -0.892533555f, -0.450980989f,
   -0.891840709f, -0.452349587f,
   -0.891145765f, -0.453717121f,
   -0.890448723f, -0.455083587f,
   -0.889749586f, -0.456448982f,
   -0.889048356f, -0.457813304f,
   -0.888345033f, -0.459176548f,
   -0.887639620f, -0.460538711f,
   -0.886932119f, -0.461899791f,
   -0.886222530f, -0.463259784f,
   -0.885510856f, -0.464618686f,
   -0.884797098f, -0.465976496f,
   -0.884081259f, -0.467333209f,
   -0.883363339f, -0.468688822f,
   -0.882643340f, -0.470043332f,
   -0.881921264f, -0.471396737f,
   -0.881197113f, -0.472749032f,
   -0.880470889f, -0.474100215f,
   -0.879742593f, -0.475450282f,
   -0.879012226f, -0.476799230f,
   -0.878279792f, -0.478147056f,
   -0.877545290f, -0.479493758f,
   -0.876808724f, -0.480839331f,
   -0.876070094f, -0.482183772f,
   -0.875329403f, -0.483527079f,
   -0.874586652f, -0.484869248f,
   -0.873841843f, -0.486210276f,
   -0.873094978f, -0.487550160f,
   -0.872346059f, -0.488888897f,
   -0.871595087f, -0.490226483f,
   -0.870842063f, -0.491562916f,
   -0.870086991f, -0.492898192f,
   -0.869329871f, -0.494232309f,
   -0.868570706f, -0.495565262f,
   -0.867809497f, -0.496897049f,
   -0.867046246f, -0.498227667f,
   -0.866280954f, -0.499557113f,
   -0.865513624f, -0.500885383f,
   -0.864744258f, -0.502212474f,
   -0.863972856f, -0.503538384f,
   -0.863199422f, -0.504863109f,
   -0.862423956f, -0.506186645f,
   -0.861646461f, -0.507508991f,
   -0.860866939f, -0.508830143f,
   -0.860085390f, -0.510150097f,
   -0.859301818f, -0.511468850f,
   -0.858516224f, -0.512786401f,
   -0.857728610f, -0.514102744f,
   -0.856938977f, -0.515417878f,
   -0.856147328f, -0.516731799f,
   -0.855353665f, -0.518044504f,
   -0.854557988f, -0.519355990f,
   -0.853760301f, -0.520666254f,
   -0.852960605f, -0.521975293f,
   -0.852158902f, -0.523283103f,
   -0.851355193f, -0.524589683f,
   -0.850549481f, -0.525895027f,
   -0.849741768f, -0.527199135f,
   -0.848932055f, -0.528502002f,
   -0.848120345f, -0.529803625f,
   -0.847306639f, -0.531104001f,
   -0.846490939f, -0.532403128f,
   -0.845673247f, -0.533701002f,
   -0.844853565f, -0.534997620f,
   -0.844031895f, -0.536292979f,
   -0.843208240f, -0.537587076f,
   -0.842382600f, -0.538879909f,
   -0.841554977f, -0.540171473f,
   -0.840725375f, -0.541461766f,
   -0.839893794f, -0.542750785f,
   -0.839060237f, -0.544038527f,
   -0.838224706f, -0.545324988f,
   -0.837387202f, -0.546610167f,
   -0.836547727f, -0.547894059f,
   -0.835706284f, -0.549176662f,
   -0.834862875f, -0.550457973f,
   -0.834017501f, -0.551737988f,
   -0.833170165f, -0.553016706f,
   -0.832320868f, -0.554294121f,
   -0.831469612f, -0.555570233f,
   -0.830616400f, -0.556845037f,
   -0.829761234f, -0.558118531f,
   -0.828904115f, -0.559390712f,
   -0.828045045f, -0.560661576f,
   -0.827184027f, -0.561931121f,
   -0.826321063f, -0.563199344f,
   -0.825456154f, -0.564466242f,
   -0.824589303f, -0.565731811f,
   -0.823720511f, -0.566996049f,
   -0.822849781f, -0.568258953f,
   -0.821977115f, -0.569520519f,
   -0.821102515f, -0.570780746f,
   -0.820225983f, -0.572039629f,
   -0.819347520f, -0.573297167f,
   -0.818467130f, -0.574553355f,
   -0.817584813f, -0.575808191f,
   -0.816700573f, -0.577061673f,
   -0.815814411f, -0.578313796f,
   -0.814926329f, -0.579564559f,
   -0.814036330f, -0.580813958f,
   -0.813144415f, -0.582061990f,
   -0.812250587f, -0.583308653f,
   -0.811354847f, -0.584553943f,
   -0.810457198f, -0.585797857f,
   -0.809557642f, -0.587040394f,
   -0.808656182f, -0.588281548f,
   -0.807752818f, -0.589521319f,
   -0.806847554f, -0.590759702f,
   -0.805940391f, -0.591996695f,
   -0.805031331f, -0.593232295f,
   -0.804120377f, -0.594466499f,
   -0.803207531f, -0.595699304f,
   -0.802292796f, -0.596930708f,
   -0.801376172f, -0.598160707f,
   -0.800457662f, -0.599389298f,
   -0.799537269f, -0.600616479f,
   -0.798614995f, -0.601842247f,
   -0.797690841f, -0.603066599f,
   -0.796764810f, -0.604289531f,
   -0.795836905f, -0.605511041f,
   -0.794907126f, -0.606731127f,
   -0.793975478f, -0.607949785f,
   -0.793041960f, -0.609167012f,
   -0.792106577f, -0.610382806f,
   -0.791169330f, -0.611597164f,
   -0.790230221f, -0.612810082f,
   -0.789289253f, -0.614021559f,
   -0.788346428f, -0.615231591f,
   -0.787401747f, -0.616440175f,
   -0.786455214f, -0.617647308f,
   -0.785506830f, -0.618852988f,
   -0.784556597f, -0.620057212f,
   -0.783604519f, -0.621259977f,
   -0.782650596f, -0.622461279f,
   -0.781694832f, -0.623661118f,
   -0.780737229f, -0.624859488f,
   -0.779777788f, -0.626056388f,
   -0.778816512f, -0.627251815f,
   -0.777853404f, -0.628445767f,
   -0.776888466f, -0.629638239f,
   -0.775921699f, -0.630829230f,
   -0.774953107f, -0.632018736f,
   -0.773982691f, -0.633206755f,
   -0.773010453f, -0.634393284f,
   -0.772036397f, -0.635578320f,
   -0.771060524f, -0.636761861f,
   -0.770082837f, -0.637943904f,
   -0.769103338f, -0.639124445f,
   -0.768122029f, -0.640303482f,
   -0.767138912f, -0.641481013f,
   -0.766153990f, -0.642657034f,
   -0.765167266f, -0.643831543f,
   -0.764178741f, -0.645004537f,
   -0.763188417f, -0.646176013f,
   -0.762196298f, -0.647345969f,
   -0.761202385f, -0.648514401f,
   -0.760206682f, -0.649681307f,
   -0.759209189f, -0.650846685f,
   -0.758209910f, -0.652010531f,
   -0.757208847f, -0.653172843f,
   -0.756206001f, -0.654333618f,
   -0.755201377f, -0.655492853f,
   -0.754194975f, -0.656650546f,
   -0.753186799f, -0.657806693f,
   -0.752176850f, -0.658961293f,
   -0.751165132f, -0.660114342f,
   -0.750151646f, -0.661265838f,
   -0.749136395f, -0.662415778f,
   -0.748119380f, -0.663564159f,
   -0.747100606f, -0.664710978f,
   -0.746080074f, -0.665856234f,
   -0.745057785f, -0.666999922f,
   -0.744033744f, -0.668142041f,
   -0.743007952f, -0.669282588f,
   -0.741980412f, -0.670421560f,
   -0.740951125f, -0.671558955f,
   -0.739920095f, -0.672694769f,
   -0.738887324f, -0.673829000f,
   -0.737852815f, -0.674961646f,
   -0.736816569f, -0.676092704f,
   -0.735778589f, -0.677222170f,
   -0.734738878f, -0.678350043f,
   -0.733697438f, -0.679476320f,
   -0.732654272f, -0.680600998f,
   -0.731609381f, -0.681724074f,
   -0.730562769f, -0.682845546f,
   -0.729514438f, -0.683965412f,
   -0.728464390f, -0.685083668f,
   -0.727412629f, -0.686200312f,
   -0.726359155f, -0.687315341f,
   -0.725303972f, -0.688428753f,
   -0.724247083f, -0.689540545f,
   -0.723188489f, -0.690650714f,
   -0.722128194f, -0.691759258f,
   -0.721066199f, -0.692866175f,
   -0.720002508f, -0.693971461f,
   -0.718937122f, -0.695075114f,
   -0.717870045f, -0.696177131f,
   -0.716801279f, -0.697277511f,
   -0.715730825f, -0.698376249f,
   -0.714658688f, -0.699473345f,
   -0.713584869f, -0.700568794f,
   -0.712509371f, -0.701662595f,
   -0.711432196f, -0.702754744f,
   -0.710353347f, -0.703845241f,
   -0.709272826f, -0.704934080f,
   -0.708190637f, -0.706021261f,
   -0.707106781f, -0.707106781f,
   -0.706021261f, -0.708190637f,
   -0.704934080f, -0.709272826f,
   -0.703845241f, -0.710353347f,
   -0.702754744f, -0.711432196f,
   -0.701662595f, -0.712509371f,
   -0.700568794f, -0.713584869f,
   -0.699473345f, -0.714658688f,
   -0.698376249f, -0.715730825f,
   -0.697277511f, -0.716801279f,
   -0.696177131f, -0.717870045f,
   -0.695075114f, -0.718937122f,
   -0.693971461f, -0.720002508f,
   -0.692866175f, -0.721066199f,
   -0.691759258f, -0.722128194f,
   -0.690650714f, -0.723188489f,
   -0.689540545f, -0.724247083f,
   -0.688428753f, -0.725303972f,
   -0.687315341f, -0.726359155f,
   -0.686200312f, -0.727412629f,
   -0.685083668f, -0.728464390f,
   -0.683965412f, -0.729514438f,
   -0.682845546f, -0.730562769f,
   -0.681724074f, -0.731609381f,
   -0.680600998f, -0.732654272f,
   -0.679476320f, -0.733697438f,
   -0.678350043f, -0.734738878f,
   -0.677222170f, -0.735778589f,
   -0.676092704f, -0.736816569f,
   -0.674961646f, -0.737852815f,
   -0.673829000f, -0.738887324f,
   -0.672694769f, -0.739920095f,
   -0.671558955f, -0.740951125f,
   -0.670421560f, -0.741980412f,
   -0.669282588f, -0.743007952f,
   -0.668142041f, -0.744033744f,
   -0.666999922f, -0.745057785f,
   -0.665856234f, -0.746080074f,
   -0.664710978f, -0.747100606f,
   -0.663564159f, -0.748119380f,
   -0.662415778f, -0.749136395f,
   -0.661265838f, -0.750151646f,
   -0.660114342f, -0.751165132f,
   -0.658961293f, -0.752176850f,
   -0.657806693f, -0.753186799f,
   -0.656650546f, -0.754194975f,
   -0.655492853f, -0.755201377f,
   -0.654333618f, -0.756206001f,
   -0.653172843f, -0.757208847f,
   -0.652010531f, -0.758209910f,
   -0.650846685f, -0.759209189f,
   -0.649681307f, -0.760206682f,
   -0.648514401f, -0.761202385f,
   -0.647345969f, -0.762196298f,
   -0.646176013f, -0.763188417f,
   -0.645004537f, -0.764178741f,
   -0.643831543f, -0.765167266f,
   -0.642657034f, -0.766153990f,
   -0.641481013f, -0.767138912f,
   -0.640303482f, -0.768122029f,
   -0.639124445f, -0.769103338f,
   -0.637943904f, -0.770082837f,
   -0.636761861f, -0.771060524f,
   -0.635578320f, -0.772036397f,
   -0.634393284f, -0.773010453f,
   -0.633206755f, -0.773982691f,
   -0.632018736f, -0.774953107f,
   -0.630829230f, -0.775921699f,
   -0.629638239f, -0.776888466f,
   -0.628445767f, -0.777853404f,
   -0.627251815f, -0.778816512f,
   -0.626056388f, -0.779777788f,
   -0.624859488f, -0.780737229f,
   -0.623661118f, -0.781694832f,
   -0.622461279f, -0.782650596f,
   -0.621259977f, -0.783604519f,
   -0.620057212f, -0.784556597f,
   -0.618852988f, -0.785506830f,
   -0.617647308f, -0.786455214f,
   -0.616440175f, -0.787401747f,
   -0.615231591f, -0.788346428f,
   -0.614021559f, -0.789289253f,
   -0.612810082f, -0.790230221f,
   -0.611597164f, -0.791169330f,
   -0.610382806f, -0.792106577f,
   -0.609167012f, -0.793041960f,
   -0.607949785f, -0.793975478f,
   -0.606731127f, -0.794907126f,
   -0.605511041f, -0.795836905f,
   -0.604289531f, -0.796764810f,
   -0.603066599f, -0.797690841f,
   -0.601842247f, -0.798614995f,
   -0.600616479f, -0.799537269f,
   -0.599389298f, -0.800457662f,
   -0.598160707f, -0.801376172f,
   -0.596930708f, -0.802292796f,
   -0.595699304f, -0.803207531f,
   -0.594466499f, -0.804120377f,
   -0.593232295f, -0.805031331f,
   -0.591996695f, -0.805940391f,
   -0.590759702f, -0.806847554f,
   -0.589521319f, -0.807752818f,
   -0.588281548f, -0.808656182f,
   -0.587040394f, -0.809557642f,
   -0.585797857f, -0.810457198f,
   -0.584553943f, -0.811354847f,
   -0.583308653f, -0.812250587f,
   -0.582061990f, -0.813144415f,
   -0.580813958f, -0.814036330f,
   -0.579564559f, -0.814926329f,
   -0.578313796f, -0.815814411f,
   -0.577061673f, -0.816700573f,
   -0.575808191f, -0.817584813f,
   -0.574553355f, -0.818467130f,
   -0.573297167f, -0.819347520f,
   -0.572039629f, -0.820225983f,
   -0.570780746f, -0.821102515f,
   -0.569520519f, -0.821977115f,
   -0.568258953f, -0.822849781f,
   -0.566996049f, -0.823720511f,
   -0.565731811f, -0.824589303f,
   -0.564466242f, -0.825456154f,
   -0.563199344f, -0.826321063f,
   -0.561931121f, -0.827184027f,
   -0.560661576f, -0.828045045f,
   -0.559390712f, -0.828904115f,
   -0.558118531f, -0.829761234f,
   -0.556845037f, -0.830616400f,
   -0.555570233f, -0.831469612f,
   -0.554294121f, -0.832320868f,
   -0.553016706f, -0.833170165f,
   -0.551737988f, -0.834017501f,
   -0.550457973f, -0.834862875f,
   -0.549176662f, -0.835706284f,
   -0.547894059f, -0.836547727f,
   -0.546610167f, -0.837387202f,
   -0.545324988f, -0.838224706f,
   -0.544038527f, -0.839060237f,
   -0.542750785f, -0.839893794f,
   -0.541461766f, -0.840725375f,
   -0.540171473f, -0.841554977f,
   -0.538879909f, -0.842382600f,
   -0.537587076f, -0.843208240f,
   -0.536292979f, -0.844031895f,
   -0.534997620f, -0.844853565f,
   -0.533701002f, -0.845673247f,
   -0.532403128f, -0.846490939f,
   -0.531104001f, -0.847306639f,
   -0.529803625f, -0.848120345f,
   -0.528502002f, -0.848932055f,
   -0.527199135f, -0.849741768f,
   -0.525895027f, -0.850549481f,
   -0.524589683f, -0.851355193f,
   -0.523283103f, -0.852158902f,
   -0.521975293f, -0.852960605f,
   -0.520666254f, -0.853760301f,
   -0.519355990f, -0.854557988f,
   -0.518044504f, -0.855353665f,
   -0.516731799f, -0.856147328f,
   -0.515417878f, -0.856938977f,
   -0.514102744f, -0.857728610f,
   -0.512786401f, -0.858516224f,
   -0.511468850f, -0.859301818f,
   -0.510150097f, -0.860085390f,
   -0.508830143f, -0.860866939f,
   -0.507508991f, -0.861646461f,
   -0.506186645f, -0.862423956f,
   -0.504863109f, -0.863199422f,
   -0.503538384f, -0.863972856f,
   -0.502212474f, -0.864744258f,
   -0.500885383f, -0.865513624f,
   -0.499557113f, -0.866280954f,
   -0.498227667f, -0.867046246f,
   -0.496897049f, -0.867809497f,
   -0.495565262f, -0.868570706f,
   -0.494232309f, -0.869329871f,
   -0.492898192f, -0.870086991f,
   -0.491562916f, -0.870842063f,
   -0.490226483f, -0.871595087f,
   -0.488888897f, -0.872346059f,
   -0.487550160f, -0.873094978f,
   -0.486210276f, -0.873841843f,
   -0.484869248f, -0.874586652f,
   -0.483527079f, -0.875329403f,
   -0.482183772f, -0.876070094f,
   -0.480839331f, -0.876808724f,
   -0.479493758f, -0.877545290f,
   -0.478147056f, -0.878279792f,
   -0.476799230f, -0.879012226f,
   -0.475450282f, -0.879742593f,
   -0.474100215f, -0.880470889f,
   -0.472749032f, -0.881197113f,
   -0.471396737f, -0.881921264f,
   -0.470043332f, -0.882643340f,
   -0.468688822f, -0.883363339f,
   -0.467333209f, -0.884081259f,
   -0.465976496f, -0.884797098f,
   -0.464618686f, -0.885510856f,
   -0.463259784f, -0.886222530f,
   -0.461899791f, -0.886932119f,
   -0.460538711f, -0.887639620f,
   -0.459176548f, -0.888345033f,
   -0.457813304f, -0.889048356f,
   -0.456448982f, -0.889749586f,
   -0.455083587f, -0.890448723f,
   -0.453717121f, -0.891145765f,
   -0.452349587f, -0.891840709f,
   -0.450980989f, -0.892533555f,
   -0.449611330f, -0.893224301f,
   -0.448240612f, -0.893912945f,
   -0.446868840f, -0.894599486f,
   -0.445496017f, -0.895283921f,
   -0.444122145f, -0.895966250f,
   -0.442747228f, -0.896646470f,
   -0.441371269f, -0.897324581f,
   -0.439994271f, -0.898000580f,
   -0.438616239f, -0.898674466f,
   -0.437237174f, -0.899346237f,
   -0.435857080f, -0.900015892f,
   -0.434475961f, -0.900683429f,
   -0.433093819f, -0.901348847f,
   -0.431710658f, -0.902012144f,
   -0.430326481f, -0.902673318f,
   -0.428941292f, -0.903332368f,
   -0.427555093f, -0.903989293f,
   -0.426167889f, -0.904644091f,
   -0.424779681f, -0.905296759f,
   -0.423390474f, -0.905947298f,
   -0.422000271f, -0.906595705f,
   -0.420609074f, -0.907241978f,
   -0.419216888f, -0.907886116f,
   -0.417823716f, -0.908528119f,
   -0.416429560f, -0.909167983f,
   -0.415034424f, -0.909805708f,
   -0.413638312f, -0.910441292f,
   -0.412241227f, -0.911074734f,
   -0.410843171f, -0.911706032f,
   -0.409444149f, -0.912335185f,
   -0.408044163f, -0.912962190f,
   -0.406643217f, -0.913587048f,
   -0.405241314f, -0.914209756f,
   -0.403838458f, -0.914830312f,
   -0.402434651f, -0.915448716f,
   -0.401029897f, -0.916064966f,
   -0.399624200f, -0.916679060f,
   -0.398217562f, -0.917290997f,
   -0.396809987f, -0.917900776f,
   -0.395401479f, -0.918508394f,
   -0.393992040f, -0.919113852f,
   -0.392581674f, -0.919717146f,
   -0.391170384f, -0.920318277f,
   -0.389758174f, -0.920917242f,
   -0.388345047f, -0.921514039f,
   -0.386931006f, -0.922108669f,
   -0.385516054f, -0.922701128f,
   -0.384100195f, -0.923291417f,
   -0.382683432f, -0.923879533f,
   -0.381265769f, -0.924465474f,
   -0.379847209f, -0.925049241f,
   -0.378427755f, -0.925630831f,
   -0.377007410f, -0.926210242f,
   -0.375586178f, -0.926787474f,
   -0.374164063f, -0.927362526f,
   -0.372741067f, -0.927935395f,
   -0.371317194f, -0.928506080f,
   -0.369892447f, -0.929074581f,
   -0.368466830f, -0.929640896f,
   -0.367040346f, -0.930205023f,
   -0.365612998f, -0.930766961f,
   -0.364184790f, -0.931326709f,
   -0.362755724f, -0.931884266f,
   -0.361325806f, -0.932439629f,
   -0.359895037f, -0.932992799f,
   -0.358463421f, -0.933543773f,
   -0.357030961f, -0.934092550f,
   -0.355597662f, -0.934639130f,
   -0.354163525f, -0.935183510f,
   -0.352728556f, -0.935725689f,
   -0.351292756f, -0.936265667f,
   -0.349856130f, -0.936803442f,
   -0.348418680f, -0.937339012f,
   -0.346980411f, -0.937872376f,
   -0.345541325f, -0.938403534f,
   -0.344101426f, -0.938932484f,
   -0.342660717f, -0.939459224f,
   -0.341219202f, -0.939983753f,
   -0.339776884f, -0.940506071f,
   -0.338333767f, -0.941026175f,
   -0.336889853f, -0.941544065f,
   -0.335445147f, -0.942059740f,
   -0.333999651f, -0.942573198f,
   -0.332553370f, -0.943084437f,
   -0.331106306f, -0.943593458f,
   -0.329658463f, -0.944100258f,
   -0.328209844f, -0.944604837f,
   -0.326760452f, -0.945107193f,
   -0.325310292f, -0.945607325f,
   -0.323859367f, -0.946105232f,
   -0.322407679f, -0.946600913f,
   -0.320955232f, -0.947094366f,
   -0.319502031f, -0.947585591f,
   -0.318048077f, -0.948074586f,
   -0.316593376f, -0.948561350f,
   -0.315137929f, -0.949045882f,
   -0.313681740f, -0.949528181f,
   -0.312224814f, -0.950008245f,
   -0.310767153f, -0.950486074f,
   -0.309308760f, -0.950961666f,
   -0.307849640f, -0.951435021f,
   -0.306389795f, -0.951906137f,
   -0.304929230f, -0.952375013f,
   -0.303467947f, -0.952841648f,
   -0.302005949f, -0.953306040f,
   -0.300543241f, -0.953768190f,
   -0.299079826f, -0.954228095f,
   -0.297615707f, -0.954685755f,
   -0.296150888f, -0.955141168f,
   -0.294685372f, -0.955594334f,
   -0.293219163f, -0.956045251f,
   -0.291752263f, -0.956493919f,
   -0.290284677f, -0.956940336f,
   -0.288816408f, -0.957384501f,
   -0.287347460f, -0.957826413f,
   -0.285877835f, -0.958266071f,
   -0.284407537f, -0.958703475f,
   -0.282936570f, -0.959138622f,
   -0.281464938f, -0.959571513f,
   -0.279992643f, -0.960002146f,
   -0.278519689f, -0.960430519f,
   -0.277046080f, -0.960856633f,
   -0.275571819f, -0.961280486f,
   -0.274096910f, -0.961702077f,
   -0.272621355f, -0.962121404f,
   -0.271145160f, -0.962538468f,
   -0.269668326f, -0.962953267f,
   -0.268190857f, -0.963365800f,
   -0.266712757f, -0.963776066f,
   -0.265234030f, -0.964184064f,
   -0.263754679f, -0.964589793f,
   -0.262274707f, -0.964993253f,
   -0.260794118f, -0.965394442f,
   -0.259312915f, -0.965793359f,
   -0.257831102f, -0.966190003f,
   -0.256348682f, -0.966584374f,
   -0.254865660f, -0.966976471f,
   -0.253382037f, -0.967366292f,
   -0.251897818f, -0.967753837f,
   -0.250413007f, -0.968139105f,
   -0.248927606f, -0.968522094f,
   -0.247441619f, -0.968902805f,
   -0.245955050f, -0.969281235f,
   -0.244467903f, -0.969657385f,
   -0.242980180f, -0.970031253f,
   -0.241491885f, -0.970402839f,
   -0.240003022f, -0.970772141f,
   -0.238513595f, -0.971139158f,
   -0.237023606f, -0.971503891f,
   -0.235533059f, -0.971866337f,
   -0.234041959f, -0.972226497f,
   -0.232550307f, -0.972584369f,
   -0.231058108f, -0.972939952f,
   -0.229565366f, -0.973293246f,
   -0.228072083f, -0.973644250f,
   -0.226578264f, -0.973992962f,
   -0.225083911f, -0.974339383f,
   -0.223589029f, -0.974683511f,
   -0.222093621f, -0.975025345f,
   -0.220597690f, -0.975364885f,
   -0.219101240f, -0.975702130f,
   -0.217604275f, -0.976037079f,
   -0.216106797f, -0.976369731f,
   -0.214608811f, -0.976700086f,
   -0.213110320f, -0.977028143f,
   -0.211611327f, -0.977353900f,
   -0.210111837f, -0.977677358f,
   -0.208611852f, -0.977998515f,
   -0.207111376f, -0.978317371f,
   -0.205610413f, -0.978633924f,
   -0.204108966f, -0.978948175f,
   -0.202607039f, -0.979260123f,
   -0.201104635f, -0.979569766f,
   -0.199601758f, -0.979877104f,
   -0.198098411f, -0.980182136f,
   -0.196594598f, -0.980484862f,
   -0.195090322f, -0.980785280f,
   -0.193585587f, -0.981083391f,
   -0.192080397f, -0.981379193f,
   -0.190574755f, -0.981672686f,
   -0.189068664f, -0.981963869f,
   -0.187562129f, -0.982252741f,
   -0.186055152f, -0.982539302f,
   -0.184547737f, -0.982823551f,
   -0.183039888f, -0.983105487f,
   -0.181531608f, -0.983385110f,
   -0.180022901f, -0.983662419f,
   -0.178513771f, -0.983937413f,
   -0.177004220f, -0.984210092f,
   -0.175494253f, -0.984480455f,
   -0.173983873f, -0.984748502f,
   -0.172473084f, -0.985014231f,
   -0.170961889f, -0.985277642f,
   -0.169450291f, -0.985538735f,
   -0.167938295f, -0.985797509f,
   -0.166425904f, -0.986053963f,
   -0.164913120f, -0.986308097f,
   -0.163399949f, -0.986559910f,
   -0.161886394f, -0.986809402f,
   -0.160372457f, -0.987056571f,
   -0.158858143f, -0.987301418f,
   -0.157343456f, -0.987543942f,
   -0.155828398f, -0.987784142f,
   -0.154312973f, -0.988022017f,
   -0.152797185f, -0.988257568f,
   -0.151281038f, -0.988490793f,
   -0.149764535f, -0.988721692f,
   -0.148247679f, -0.988950265f,
   -0.146730474f, -0.989176510f,
   -0.145212925f, -0.989400428f,
   -0.143695033f, -0.989622017f,
   -0.142176804f, -0.989841278f,
   -0.140658239f, -0.990058210f,
   -0.139139344f, -0.990272812f,
   -0.137620122f, -0.990485084f,
   -0.136100575f, -0.990695025f,
   -0.134580709f, -0.990902635f,
   -0.133060525f, -0.991107914f,
   -0.131540029f, -0.991310860f,
   -0.130019223f, -0.991511473f,
   -0.128498111f, -0.991709754f,
   -0.126976696f, -0.991905700f,
   -0.125454983f, -0.992099313f,
   -0.123932975f, -0.992290591f,
   -0.122410675f, -0.992479535f,
   -0.120888087f, -0.992666142f,
   -0.119365215f, -0.992850414f,
   -0.117842062f, -0.993032350f,
   -0.116318631f, -0.993211949f,
   -0.114794927f, -0.993389211f,
   -0.113270952f, -0.993564136f,
   -0.111746711f, -0.993736722f,
   -0.110222207f, -0.993906970f,
   -0.108697444f, -0.994074879f,
   -0.107172425f, -0.994240449f,
   -0.105647154f, -0.994403680f,
   -0.104121634f, -0.994564571f,
   -0.102595869f, -0.994723121f,
   -0.101069863f, -0.994879331f,
   -0.099543619f, -0.995033199f,
   -0.098017140f, -0.995184727f,
   -0.096490431f, -0.995333912f,
   -0.094963495f, -0.995480755f,
   -0.093436336f, -0.995625256f,
   -0.091908956f, -0.995767414f,
   -0.090381361f, -0.995907229f,
   -0.088853553f, -0.996044701f,
   -0.087325535f, -0.996179829f,
   -0.085797312f, -0.996312612f,
   -0.084268888f, -0.996443051f,
   -0.082740265f, -0.996571146f,
   -0.081211447f, -0.996696895f,
   -0.079682438f, -0.996820299f,
   -0.078153242f, -0.996941358f,
   -0.076623861f, -0.997060070f,
   -0.075094301f, -0.997176437f,
   -0.073564564f, -0.997290457f,
   -0.072034653f, -0.997402130f,
   -0.070504573f, -0.997511456f,
   -0.068974328f, -0.997618435f,
   -0.067443920f, -0.997723067f,
   -0.065913353f, -0.997825350f,
   -0.064382631f, -0.997925286f,
   -0.062851758f, -0.998022874f,
   -0.061320736f, -0.998118113f,
   -0.059789571f, -0.998211003f,
   -0.058258265f, -0.998301545f,
   -0.056726821f, -0.998389737f,
   -0.055195244f, -0.998475581f,
   -0.053663538f, -0.998559074f,
   -0.052131705f, -0.998640218f,
   -0.050599749f, -0.998719012f,
   -0.049067674f, -0.998795456f,
   -0.047535484f, -0.998869550f,
   -0.046003182f, -0.998941293f,
   -0.044470772f, -0.999010686f,
   -0.042938257f, -0.999077728f,
   -0.041405641f, -0.999142419f,
   -0.039872928f, -0.999204759f,
   -0.038340120f, -0.999264747f,
   -0.036807223f, -0.999322385f,
   -0.035274239f, -0.999377670f,
   -0.033741172f, -0.999430605f,
   -0.032208025f, -0.999481187f,
   -0.030674803f, -0.999529418f,
   -0.029141509f, -0.999575296f,
   -0.027608146f, -0.999618822f,
   -0.026074718f, -0.999659997f,
   -0.024541229f, -0.999698819f,
   -0.023007681f, -0.999735288f,
   -0.021474080f, -0.999769405f,
   -0.019940429f, -0.999801170f,
   -0.018406730f, -0.999830582f,
   -0.016872988f, -0.999857641f,
   -0.015339206f, -0.999882347f,
   -0.013805389f, -0.999904701f,
   -0.012271538f, -0.999924702f,
   -0.010737659f, -0.999942350f,
   -0.009203755f, -0.999957645f,
   -0.007669829f, -0.999970586f,
   -0.006135885f, -0.999981175f,
   -0.004601926f, -0.999989411f,
   -0.003067957f, -0.999995294f,
   -0.001533980f, -0.999998823f,
   -0.000000000f, -1.000000000f,
    0.001533980f, -0.999998823f,
    0.003067957f, -0.999995294f,
    0.004601926f, -0.999989411f,
    0.006135885f, -0.999981175f,
    0.007669829f, -0.999970586f,
    0.009203755f, -0.999957645f,
    0.010737659f, -0.999942350f,
    0.012271538f, -0.999924702f,
    0.013805389f, -0.999904701f,
    0.015339206f, -0.999882347f,
    0.016872988f, -0.999857641f,
    0.018406730f, -0.999830582f,
    0.019940429f, -0.999801170f,
    0.021474080f, -0.999769405f,
    0.023007681f, -0.999735288f,
    0.024541229f, -0.999698819f,
    0.026074718f, -0.999659997f,
    0.027608146f, -0.999618822f,
    0.029141509f, -0.999575296f,
    0.030674803f, -0.999529418f,
    0.032208025f, -0.999481187f,
    0.033741172f, -0.999430605f,
    0.035274239f, -0.999377670f,
    0.036807223f, -0.999322385f,
    0.038340120f, -0.999264747f,
    0.039872928f, -0.999204759f,
    0.041405641f, -0.999142419f,
    0.042938257f, -0.999077728f,
    0.044470772f, -0.999010686f,
    0.046003182f, -0.998941293f,
    0.047535484f, -0.998869550f,
    0.049067674f, -0.998795456f,
    0.050599749f, -0.998719012f,
    0.052131705f, -0.998640218f,
    0.053663538f, -0.998559074f,
    0.055195244f, -0.998475581f,
    0.056726821f, -0.998389737f,
    0.058258265f, -0.998301545f,
    0.059789571f, -0.998211003f,
    0.061320736f, -0.998118113f,
    0.062851758f, -0.998022874f,
    0.064382631f, -0.997925286f,
    0.065913353f, -0.997825350f,
    0.067443920f, -0.997723067f,
    0.068974328f, -0.997618435f,
    0.070504573f, -0.997511456f,
    0.072034653f, -0.997402130f,
    0.073564564f, -0.997290457f,
    0.075094301f, -0.997176437f,
    0.076623861f, -0.997060070f,
    0.078153242f, -0.996941358f,
    0.079682438f, -0.996820299f,
    0.081211447f, -0.996696895f,
    0.082740265f, -0.996571146f,
    0.084268888f, -0.996443051f,
    0.085797312f, -0.996312612f,
    0.087325535f, -0.996179829f,
    0.088853553f, -0.996044701f,
    0.090381361f, -0.995907229f,
    0.091908956f, -0.995767414f,
    0.093436336f, -0.995625256f,
    0.094963495f, -0.995480755f,
    0.096490431f, -0.995333912f,
    0.098017140f, -0.995184727f,
    0.099543619f, -0.995033199f,
    0.101069863f, -0.994879331f,
    0.102595869f, -0.994723121f,
    0.104121634f, -0.994564571f,
    0.105647154f, -0.994403680f,
    0.107172425f, -0.994240449f,
    0.108697444f, -0.994074879f,
    0.110222207f, -0.993906970f,
    0.111746711f, -0.993736722f,
    0.113270952f, -0.993564136f,
    0.114794927f, -0.993389211f,
    0.116318631f, -0.993211949f,
    0.117842062f, -0.993032350f,
    0.119365215f, -0.992850414f,
    0.120888087f, -0.992666142f,
    0.122410675f, -0.992479535f,
    0.123932975f, -0.992290591f,
    0.125454983f, -0.992099313f,
    0.126976696f, -0.991905700f,
    0.128498111f, -0.991709754f,
    0.130019223f, -0.991511473f,
    0.131540029f, -0.991310860f,
    0.133060525f, -0.991107914f,
    0.134580709f, -0.990902635f,
    0.136100575f, -0.990695025f,
    0.137620122f, -0.990485084f,
    0.139139344f, -0.990272812f,
    0.140658239f, -0.990058210f,
    0.142176804f, -0.989841278f,
    0.143695033f, -0.989622017f,
    0.145212925f, -0.989400428f,
    0.146730474f, -0.989176510f,
    0.148247679f, -0.988950265f,
    0.149764535f, -0.988721692f,
    0.151281038f, -0.988490793f,
    0.152797185f, -0.988257568f,
    0.154312973f, -0.988022017f,
    0.155828398f, -0.987784142f,
    0.157343456f, -0.987543942f,
    0.158858143f, -0.987301418f,
    0.160372457f, -0.987056571f,
    0.161886394f, -0.986809402f,
    0.163399949f, -0.986559910f,
    0.164913120f, -0.986308097f,
    0.166425904f, -0.986053963f,
    0.167938295f, -0.985797509f,
    0.169450291f, -0.985538735f,
    0.170961889f, -0.985277642f,
    0.172473084f, -0.985014231f,
    0.173983873f, -0.984748502f,
    0.175494253f, -0.984480455f,
    0.177004220f, -0.984210092f,
    0.178513771f, -0.983937413f,
    0.180022901f, -0.983662419f,
    0.181531608f, -0.983385110f,
    0.183039888f, -0.983105487f,
    0.184547737f, -0.982823551f,
    0.186055152f, -0.982539302f,
    0.187562129f, -0.982252741f,
    0.189068664f, -0.981963869f,
    0.190574755f, -0.981672686f,
    0.192080397f, -0.981379193f,
    0.193585587f, -0.981083391f,
    0.195090322f, -0.980785280f,
    0.196594598f, -0.980484862f,
    0.198098411f, -0.980182136f,
    0.199601758f, -0.979877104f,
    0.201104635f, -0.979569766f,
    0.202607039f, -0.979260123f,
    0.204108966f, -0.978948175f,
    0.205610413f, -0.978633924f,
    0.207111376f, -0.978317371f,
    0.208611852f, -0.977998515f,
    0.210111837f, -0.977677358f,
    0.211611327f, -0.977353900f,
    0.213110320f, -0.977028143f,
    0.214608811f, -0.976700086f,
    0.216106797f, -0.976369731f,
    0.217604275f, -0.976037079f,
    0.219101240f, -0.975702130f,
    0.220597690f, -0.975364885f,
    0.222093621f, -0.975025345f,
    0.223589029f, -0.974683511f,
    0.225083911f, -0.974339383f,
    0.226578264f, -0.973992962f,
    0.228072083f, -0.973644250f,
    0.229565366f, -0.973293246f,
    0.231058108f, -0.972939952f,
    0.232550307f, -0.972584369f,
    0.234041959f, -0.972226497f,
    0.235533059f, -0.971866337f,
    0.237023606f, -0.971503891f,
    0.238513595f, -0.971139158f,
    0.240003022f, -0.970772141f,
    0.241491885f, -0.970402839f,
    0.242980180f, -0.970031253f,
    0.244467903f, -0.969657385f,
    0.245955050f, -0.969281235f,
    0.247441619f, -0.968902805f,
    0.248927606f, -0.968522094f,
    0.250413007f, -0.968139105f,
    0.251897818f, -0.967753837f,
    0.253382037f, -0.967366292f,
    0.254865660f, -0.966976471f,
    0.256348682f, -0.966584374f,
    0.257831102f, -0.966190003f,
    0.259312915f, -0.965793359f,
    0.260794118f, -0.965394442f,
    0.262274707f, -0.964993253f,
    0.263754679f, -0.964589793f,
    0.265234030f, -0.964184064f,
    0.266712757f, -0.963776066f,
    0.268190857f, -0.963365800f,
    0.269668326f, -0.962953267f,
    0.271145160f, -0.962538468f,
    0.272621355f, -0.962121404f,
    0.274096910f, -0.961702077f,
    0.275571819f, -0.961280486f,
    0.277046080f, -0.960856633f,
    0.278519689f, -0.960430519f,
    0.279992643f, -0.960002146f,
    0.281464938f, -0.959571513f,
    0.282936570f, -0.959138622f,
    0.284407537f, -0.958703475f,
    0.285877835f, -0.958266071f,
    0.287347460f, -0.957826413f,
    0.288816408f, -0.957384501f,
    0.290284677f, -0.956940336f,
    0.291752263f, -0.956493919f,
    0.293219163f, -0.956045251f,
    0.294685372f, -0.955594334f,
    0.296150888f, -0.955141168f,
    0.297615707f, -0.954685755f,
    0.299079826f, -0.954228095f,
    0.300543241f, -0.953768190f,
    0.302005949f, -0.953306040f,
    0.303467947f, -0.952841648f,
    0.304929230f, -0.952375013f,
    0.306389795f, -0.951906137f,
    0.307849640f, -0.951435021f,
    0.309308760f, -0.950961666f,
    0.310767153f, -0.950486074f,
    0.312224814f, -0.950008245f,
    0.313681740f, -0.949528181f,
    0.315137929f, -0.949045882f,
    0.316593376f, -0.948561350f,
    0.318048077f, -0.948074586f,
    0.319502031f, -0.947585591f,
    0.320955232f, -0.947094366f,
    0.322407679f, -0.946600913f,
    0.323859367f, -0.946105232f,
    0.325310292f, -0.945607325f,
    0.326760452f, -0.945107193f,
    0.328209844f, -0.944604837f,
    0.329658463f, -0.944100258f,
    0.331106306f, -0.943593458f,
    0.332553370f, -0.943084437f,
    0.333999651f, -0.942573198f,
    0.335445147f, -0.942059740f,
    0.336889853f, -0.941544065f,
    0.338333767f, -0.941026175f,
    0.339776884f, -0.940506071f,
    0.341219202f, -0.939983753f,
    0.342660717f, -0.939459224f,
    0.344101426f, -0.938932484f,
    0.345541325f, -0.938403534f,
    0.346980411f, -0.937872376f,
    0.348418680f, -0.937339012f,
    0.349856130f, -0.936803442f,
    0.351292756f, -0.936265667f,
    0.352728556f, -0.935725689f,
    0.354163525f, -0.935183510f,
    0.355597662f, -0.934639130f,
    0.357030961f, -0.934092550f,
    0.358463421f, -0.933543773f,
    0.359895037f, -0.932992799f,
    0.361325806f, -0.932439629f,
    0.362755724f, -0.931884266f,
    0.364184790f, -0.931326709f,
    0.365612998f, -0.930766961f,
    0.367040346f, -0.930205023f,
    0.368466830f, -0.929640896f,
    0.369892447f, -0.929074581f,
    0.371317194f, -0.928506080f,
    0.372741067f, -0.927935395f,
    0.374164063f, -0.927362526f,
    0.375586178f, -0.926787474f,
    0.377007410f, -0.926210242f,
    0.378427755f, -0.925630831f,
    0.379847209f, -0.925049241f,
    0.381265769f, -0.924465474f,
    0.382683432f, -0.923879533f,
    0.384100195f, -0.923291417f,
    0.385516054f, -0.922701128f,
    0.386931006f, -0.922108669f,
    0.388345047f, -0.921514039f,
    0.389758174f, -0.920917242f,
    0.391170384f, -0.920318277f,
    0.392581674f, -0.919717146f,
    0.393992040f, -0.919113852f,
    0.395401479f, -0.918508394f,
    0.396809987f, -0.917900776f,
    0.398217562f, -0.917290997f,
    0.399624200f, -0.916679060f,
    0.401029897f, -0.916064966f,
    0.402434651f, -0.915448716f,
    0.403838458f, -0.914830312f,
    0.405241314f, -0.914209756f,
    0.406643217f, -0.913587048f,
    0.408044163f, -0.912962190f,
    0.409444149f, -0.912335185f,
    0.410843171f, -0.911706032f,
    0.412241227f, -0.911074734f,
    0.413638312f, -0.910441292f,
    0.415034424f, -0.909805708f,
    0.416429560f, -0.909167983f,
    0.417823716f, -0.908528119f,
    0.419216888f, -0.907886116f,
    0.420609074f, -0.907241978f,
    0.422000271f, -0.906595705f,
    0.423390474f, -0.905947298f,
    0.424779681f, -0.905296759f,
    0.426167889f, -0.904644091f,
    0.427555093f, -0.903989293f,
    0.428941292f, -0.903332368f,
    0.430326481f, -0.902673318f,
    0.431710658f, -0.902012144f,
    0.433093819f, -0.901348847f,
    0.434475961f, -0.900683429f,
    0.435857080f, -0.900015892f,
    0.437237174f, -0.899346237f,
    0.438616239f, -0.898674466f,
    0.439994271f, -0.898000580f,
    0.441371269f, -0.897324581f,
    0.442747228f, -0.896646470f,
    0.444122145f, -0.895966250f,
    0.445496017f, -0.895283921f,
    0.446868840f, -0.894599486f,
    0.448240612f, -0.893912945f,
    0.449611330f, -0.893224301f,
    0.450980989f, -0.892533555f,
    0.452349587f, -0.891840709f,
    0.453717121f, -0.891145765f,
    0.455083587f, -0.890448723f,
    0.456448982f, -0.889749586f,
    0.457813304f, -0.889048356f,
    0.459176548f, -0.888345033f,
    0.460538711f, -0.887639620f,
    0.461899791f, -0.886932119f,
    0.463259784f, -0.886222530f,
    0.464618686f, -0.885510856f,
    0.465976496f, -0.884797098f,
    0.467333209f, -0.884081259f,
    0.468688822f, -0.883363339f,
    0.470043332f, -0.882643340f,
    0.471396737f, -0.881921264f,
    0.472749032f, -0.881197113f,
    0.474100215f, -0.880470889f,
    0.475450282f, -0.879742593f,
    0.476799230f, -0.879012226f,
    0.478147056f, -0.878279792f,
    0.479493758f, -0.877545290f,
    0.480839331f, -0.876808724f,
    0.482183772f, -0.876070094f,
    0.483527079f, -0.875329403f,
    0.484869248f, -0.874586652f,
    0.486210276f, -0.873841843f,
    0.487550160f, -0.873094978f,
    0.488888897f, -0.872346059f,
    0.490226483f, -0.871595087f,
    0.491562916f, -0.870842063f,
    0.492898192f, -0.870086991f,
    0.494232309f, -0.869329871f,
    0.495565262f, -0.868570706f,
    0.496897049f, -0.867809497f,
    0.498227667f, -0.867046246f,
    0.499557113f, -0.866280954f,
    0.500885383f, -0.865513624f,
    0.502212474f, -0.864744258f,
    0.503538384f, -0.863972856f,
    0.504863109f, -0.863199422f,
    0.506186645f, -0.862423956f,
    0.507508991f, -0.861646461f,
    0.508830143f, -0.860866939f,
    0.510150097f, -0.860085390f,
    0.511468850f, -0.859301818f,
    0.512786401f, -0.858516224f,
    0.514102744f, -0.857728610f,
    0.515417878f, -0.856938977f,
    0.516731799f, -0.856147328f,
    0.518044504f, -0.855353665f,
    0.519355990f, -0.854557988f,
    0.520666254f, -0.853760301f,
    0.521975293f, -0.852960605f,
    0.523283103f, -0.852158902f,
    0.524589683f, -0.851355193f,
    0.525895027f, -0.850549481f,
    0.527199135f, -0.849741768f,
    0.528502002f, -0.848932055f,
    0.529803625f, -0.848120345f,
    0.531104001f, -0.847306639f,
    0.532403128f, -0.846490939f,
    0.533701002f, -0.845673247f,
    0.534997620f, -0.844853565f,
    0.536292979f, -0.844031895f,
    0.537587076f, -0.843208240f,
    0.538879909f, -0.842382600f,
    0.540171473f, -0.841554977f,
    0.541461766f, -0.840725375f,
    0.542750785f, -0.839893794f,
    0.544038527f, -0.839060237f,
    0.545324988f, -0.838224706f,
    0.546610167f, -0.837387202f,
    0.547894059f, -0.836547727f,
    0.549176662f, -0.835706284f,
    0.550457973f, -0.834862875f,
    0.551737988f, -0.834017501f,
    0.553016706f, -0.833170165f,
    0.554294121f, -0.832320868f,
    0.555570233f, -0.831469612f,
    0.556845037f, -0.830616400f,
    0.558118531f, -0.829761234f,
    0.559390712f, -0.828904115f,
    0.560661576f, -0.828045045f,
    0.561931121f, -0.827184027f,
    0.563199344f, -0.826321063f,
    0.564466242f, -0.825456154f,
    0.565731811f, -0.824589303f,
    0.566996049f, -0.823720511f,
    0.568258953f, -0.822849781f,
    0.569520519f, -0.821977115f,
    0.570780746f, -0.821102515f,
    0.572039629f, -0.820225983f,
    0.573297167f, -0.819347520f,
    0.574553355f, -0.818467130f,
    0.575808191f, -0.817584813f,
    0.577061673f, -0.816700573f,
    0.578313796f, -0.815814411f,
    0.579564559f, -0.814926329f,
    0.580813958f, -0.814036330f,
    0.582061990f, -0.813144415f,
    0.583308653f, -0.812250587f,
    0.584553943f, -0.811354847f,
    0.585797857f, -0.810457198f,
    0.587040394f, -0.809557642f,
    0.588281548f, -0.808656182f,
    0.589521319f, -0.807752818f,
    0.590759702f, -0.806847554f,
    0.591996695f, -0.805940391f,
    0.593232295f, -0.805031331f,
    0.594466499f, -0.804120377f,
    0.595699304f, -0.803207531f,
    0.596930708f, -0.802292796f,
    0.598160707f, -0.801376172f,
    0.599389298f, -0.800457662f,
    0.600616479f, -0.799537269f,
    0.601842247f, -0.798614995f,
    0.603066599f, -0.797690841f,
    0.604289531f, -0.796764810f,
    0.605511041f, -0.795836905f,
    0.606731127f, -0.794907126f,
    0.607949785f, -0.793975478f,
    0.609167012f, -0.793041960f,
    0.610382806f, -0.792106577f,
    0.611597164f, -0.791169330f,
    0.612810082f, -0.790230221f,
    0.614021559f, -0.789289253f,
    0.615231591f, -0.788346428f,
    0.616440175f, -0.787401747f,
    0.617647308f, -0.786455214f,
    0.618852988f, -0.785506830f,
    0.620057212f, -0.784556597f,
    0.621259977f, -0.783604519f,
    0.622461279f, -0.782650596f,
    0.623661118f, -0.781694832f,
    0.624859488f, -0.780737229f,
    0.626056388f, -0.779777788f,
    0.627251815f, -0.778816512f,
    0.628445767f, -0.777853404f,
    0.629638239f, -0.776888466f,
    0.630829230f, -0.775921699f,
    0.632018736f, -0.774953107f,
    0.633206755f, -0.773982691f,
    0.634393284f, -0.773010453f,
    0.635578320f, -0.772036397f,
    0.636761861f, -0.771060524f,
    0.637943904f, -0.770082837f,
    0.639124445f, -0.769103338f,
    0.640303482f, -0.768122029f,
    0.641481013f, -0.767138912f,
    0.642657034f, -0.766153990f,
    0.643831543f, -0.765167266f,
    0.645004537f, -0.764178741f,
    0.646176013f, -0.763188417f,
    0.647345969f, -0.762196298f,
    0.648514401f, -0.761202385f,
    0.649681307f, -0.760206682f,
    0.650846685f, -0.759209189f,
    0.652010531f, -0.758209910f,
    0.653172843f, -0.757208847f,
    0.654333618f, -0.756206001f,
    0.655492853f, -0.755201377f,
    0.656650546f, -0.754194975f,
    0.657806693f, -0.753186799f,
    0.658961293f, -0.752176850f,
    0.660114342f, -0.751165132f,
    0.661265838f, -0.750151646f,
    0.662415778f, -0.749136395f,
    0.663564159f, -0.748119380f,
    0.664710978f, -0.747100606f,
    0.665856234f, -0.746080074f,
    0.666999922f, -0.745057785f,
    0.668142041f, -0.744033744f,
    0.669282588f, -0.743007952f,
    0.670421560f, -0.741980412f,
    0.671558955f, -0.740951125f,
    0.672694769f, -0.739920095f,
    0.673829000f, -0.738887324f,
    0.674961646f, -0.737852815f,
    0.676092704f, -0.736816569f,
    0.677222170f, -0.735778589f,
    0.678350043f, -0.734738878f,
    0.679476320f, -0.733697438f,
    0.680600998f, -0.732654272f,
    0.681724074f, -0.731609381f,
    0.682845546f, -0.730562769f,
    0.683965412f, -0.729514438f,
    0.685083668f, -0.728464390f,
    0.686200312f, -0.727412629f,
    0.687315341f, -0.726359155f,
    0.688428753f, -0.725303972f,
    0.689540545f, -0.724247083f,
    0.690650714f, -0.723188489f,
    0.691759258f, -0.722128194f,
    0.692866175f, -0.721066199f,
    0.693971461f, -0.720002508f,
    0.695075114f, -0.718937122f,
    0.696177131f, -0.717870045f,
    0.697277511f, -0.716801279f,
    0.698376249f, -0.715730825f,
    0.699473345f, -0.714658688f,
    0.700568794f, -0.713584869f,
    0.701662595f, -0.712509371f,
    0.702754744f, -0.711432196f,
    0.703845241f, -0.710353347f,
    0.704934080f, -0.709272826f,
    0.706021261f, -0.708190637f,
    0.707106781f, -0.707106781f,
    0.708190637f, -0.706021261f,
    0.709272826f, -0.704934080f,
    0.710353347f, -0.703845241f,
    0.711432196f, -0.702754744f,
    0.712509371f, -0.701662595f,
    0.713584869f, -0.700568794f,
    0.714658688f, -0.699473345f,
    0.715730825f, -0.698376249f,
    0.716801279f, -0.697277511f,
    0.717870045f, -0.696177131f,
    0.718937122f, -0.695075114f,
    0.720002508f, -0.693971461f,
    0.721066199f, -0.692866175f,
    0.722128194f, -0.691759258f,
    0.723188489f, -0.690650714f,
    0.724247083f, -0.689540545f,
    0.725303972f, -0.688428753f,
    0.726359155f, -0.687315341f,
    0.727412629f, -0.686200312f,
    0.728464390f, -0.685083668f,
    0.729514438f, -0.683965412f,
    0.730562769f, -0.682845546f,
    0.731609381f, -0.681724074f,
    0.732654272f, -0.680600998f,
    0.733697438f, -0.679476320f,
    0.734738878f, -0.678350043f,
    0.735778589f, -0.677222170f,
    0.736816569f, -0.676092704f,
    0.737852815f, -0.674961646f,
    0.738887324f, -0.673829000f,
    0.739920095f, -0.672694769f,
    0.740951125f, -0.671558955f,
    0.741980412f, -0.670421560f,
    0.743007952f, -0.669282588f,
    0.744033744f, -0.668142041f,
    0.745057785f, -0.666999922f,
    0.746080074f, -0.665856234f,
    0.747100606f, -0.664710978f,
    0.748119380f, -0.663564159f,
    0.749136395f, -0.662415778f,
    0.750151646f, -0.661265838f,
    0.751165132f, -0.660114342f,
    0.752176850f, -0.658961293f,
    0.753186799f, -0.657806693f,
    0.754194975f, -0.656650546f,
    0.755201377f, -0.655492853f,
    0.756206001f, -0.654333618f,
    0.757208847f, -0.653172843f,
    0.758209910f, -0.652010531f,
    0.759209189f, -0.650846685f,
    0.760206682f, -0.649681307f,
    0.761202385f, -0.648514401f,
    0.762196298f, -0.647345969f,
    0.763188417f, -0.646176013f,
    0.764178741f, -0.645004537f,
    0.765167266f, -0.643831543f,
    0.766153990f, -0.642657034f,
    0.767138912f, -0.641481013f,
    0.768122029f, -0.640303482f,
    0.769103338f, -0.639124445f,
    0.770082837f, -0.637943904f,
    0.771060524f, -0.636761861f,
    0.772036397f, -0.635578320f,
    0.773010453f, -0.634393284f,
    0.773982691f, -0.633206755f,
    0.774953107f, -0.632018736f,
    0.775921699f, -0.630829230f,
    0.776888466f, -0.629638239f,
    0.777853404f, -0.628445767f,
    0.778816512f, -0.627251815f,
    0.779777788f, -0.626056388f,
    0.780737229f, -0.624859488f,
    0.781694832f, -0.623661118f,
    0.782650596f, -0.622461279f,
    0.783604519f, -0.621259977f,
    0.784556597f, -0.620057212f,
    0.785506830f, -0.618852988f,
    0.786455214f, -0.617647308f,
    0.787401747f, -0.616440175f,
    0.788346428f, -0.615231591f,
    0.789289253f, -0.614021559f,
    0.790230221f, -0.612810082f,
    0.791169330f, -0.611597164f,
    0.792106577f, -0.610382806f,
    0.793041960f, -0.609167012f,
    0.793975478f, -0.607949785f,
    0.794907126f, -0.606731127f,
    0.795836905f, -0.605511041f,
    0.796764810f, -0.604289531f,
    0.797690841f, -0.603066599f,
    0.798614995f, -0.601842247f,
    0.799537269f, -0.600616479f,
    0.800457662f, -0.599389298f,
    0.801376172f, -0.598160707f,
    0.802292796f, -0.596930708f,
    0.803207531f, -0.595699304f,
    0.804120377f, -0.594466499f,
    0.805031331f, -0.593232295f,
    0.805940391f, -0.591996695f,
    0.806847554f, -0.590759702f,
    0.807752818f, -0.589521319f,
    0.808656182f, -0.588281548f,
    0.809557642f, -0.587040394f,
    0.810457198f, -0.585797857f,
    0.811354847f, -0.584553943f,
    0.812250587f, -0.583308653f,
    0.813144415f, -0.582061990f,
    0.814036330f, -0.580813958f,
    0.814926329f, -0.579564559f,
    0.815814411f, -0.578313796f,
    0.816700573f, -0.577061673f,
    0.817584813f, -0.575808191f,
    0.818467130f, -0.574553355f,
    0.819347520f, -0.573297167f,
    0.820225983f, -0.572039629f,
    0.821102515f, -0.570780746f,
    0.821977115f, -0.569520519f,
    0.822849781f, -0.568258953f,
    0.823720511f, -0.566996049f,
    0.824589303f, -0.565731811f,
    0.825456154f, -0.564466242f,
    0.826321063f, -0.563199344f,
    0.827184027f, -0.561931121f,
    0.828045045f, -0.560661576f,
    0.828904115f, -0.559390712f,
    0.829761234f, -0.558118531f,
    0.830616400f, -0.556845037f,
    0.831469612f, -0.555570233f,
    0.832320868f, -0.554294121f,
    0.833170165f, -0.553016706f,
    0.834017501f, -0.551737988f,
    0.834862875f, -0.550457973f,
    0.835706284f, -0.549176662f,
    0.836547727f, -0.547894059f,
    0.837387202f, -0.546610167f,
    0.838224706f, -0.545324988f,
    0.839060237f, -0.544038527f,
    0.839893794f, -0.542750785f,
    0.840725375f, -0.541461766f,
    0.841554977f, -0.540171473f,
    0.842382600f, -0.538879909f,
    0.843208240f, -0.537587076f,
    0.844031895f, -0.536292979f,
    0.844853565f, -0.534997620f,
    0.845673247f, -0.533701002f,
    0.846490939f, -0.532403128f,
    0.847306639f, -0.531104001f,
    0.848120345f, -0.529803625f,
    0.848932055f, -0.528502002f,
    0.849741768f, -0.527199135f,
    0.850549481f, -0.525895027f,
    0.851355193f, -0.524589683f,
    0.852158902f, -0.523283103f,
    0.852960605f, -0.521975293f,
    0.853760301f, -0.520666254f,
    0.854557988f, -0.519355990f,
    0.855353665f, -0.518044504f,
    0.856147328f, -0.516731799f,
    0.856938977f, -0.515417878f,
    0.857728610f, -0.514102744f,
    0.858516224f, -0.512786401f,
    0.859301818f, -0.511468850f,
    0.860085390f, -0.510150097f,
    0.860866939f, -0.508830143f,
    0.861646461f, -0.507508991f,
    0.862423956f, -0.506186645f,
    0.863199422f, -0.504863109f,
    0.863972856f, -0.503538384f,
    0.864744258f, -0.502212474f,
    0.865513624f, -0.500885383f,
    0.866280954f, -0.499557113f,
    0.867046246f, -0.498227667f,
    0.867809497f, -0.496897049f,
    0.868570706f, -0.495565262f,
    0.869329871f, -0.494232309f,
    0.870086991f, -0.492898192f,
    0.870842063f, -0.491562916f,
    0.871595087f, -0.490226483f,
    0.872346059f, -0.488888897f,
    0.873094978f, -0.487550160f,
    0.873841843f, -0.486210276f,
    0.874586652f, -0.484869248f,
    0.875329403f, -0.483527079f,
    0.876070094f, -0.482183772f,
    0.876808724f, -0.480839331f,
    0.877545290f, -0.479493758f,
    0.878279792f, -0.478147056f,
    0.879012226f, -0.476799230f,
    0.879742593f, -0.475450282f,
    0.880470889f, -0.474100215f,
    0.881197113f, -0.472749032f,
    0.881921264f, -0.471396737f,
    0.882643340f, -0.470043332f,
    0.883363339f, -0.468688822f,
    0.884081259f, -0.467333209f,
    0.884797098f, -0.465976496f,
    0.885510856f, -0.464618686f,
    0.886222530f, -0.463259784f,
    0.886932119f, -0.461899791f,
    0.887639620f, -0.460538711f,
    0.888345033f, -0.459176548f,
    0.889048356f, -0.457813304f,
    0.889749586f, -0.456448982f,
    0.890448723f, -0.455083587f,
    0.891145765f, -0.453717121f,
    0.891840709f, -0.452349587f,
    0.892533555f, -0.450980989f,
    0.893224301f, -0.449611330f,
    0.893912945f, -0.448240612f,
    0.894599486f, -0.446868840f,
    0.895283921f, -0.445496017f,
    0.895966250f, -0.444122145f,
    0.896646470f, -0.442747228f,
    0.897324581f, -0.441371269f,
    0.898000580f, -0.439994271f,
    0.898674466f, -0.438616239f,
    0.899346237f, -0.437237174f,
    0.900015892f, -0.435857080f,
    0.900683429f, -0.434475961f,
    0.901348847f, -0.433093819f,
    0.902012144f, -0.431710658f,
    0.902673318f, -0.430326481f,
    0.903332368f, -0.428941292f,
    0.903989293f, -0.427555093f,
    0.904644091f, -0.426167889f,
    0.905296759f, -0.424779681f,
    0.905947298f, -0.423390474f,
    0.906595705f, -0.422000271f,
    0.907241978f, -0.420609074f,
    0.907886116f, -0.419216888f,
    0.908528119f, -0.417823716f,
    0.909167983f, -0.416429560f,
    0.909805708f, -0.415034424f,
    0.910441292f, -0.413638312f,
    0.911074734f, -0.412241227f,
    0.911706032f, -0.410843171f,
    0.912335185f, -0.409444149f,
    0.912962190f, -0.408044163f,
    0.913587048f, -0.406643217f,
    0.914209756f, -0.405241314f,
    0.914830312f, -0.403838458f,
    0.915448716f, -0.402434651f,
    0.916064966f, -0.401029897f,
    0.916679060f, -0.399624200f,
    0.917290997f, -0.398217562f,
    0.917900776f, -0.396809987f,
    0.918508394f, -0.395401479f,
    0.919113852f, -0.393992040f,
    0.919717146f, -0.392581674f,
    0.920318277f, -0.391170384f,
    0.920917242f, -0.389758174f,
    0.921514039f, -0.388345047f,
    0.922108669f, -0.386931006f,
    0.922701128f, -0.385516054f,
    0.923291417f, -0.384100195f,
    0.923879533f, -0.382683432f,
    0.924465474f, -0.381265769f,
    0.925049241f, -0.379847209f,
    0.925630831f, -0.378427755f,
    0.926210242f, -0.377007410f,
    0.926787474f, -0.375586178f,
    0.927362526f, -0.374164063f,
    0.927935395f, -0.372741067f,
    0.928506080f, -0.371317194f,
    0.929074581f, -0.369892447f,
    0.929640896f, -0.368466830f,
    0.930205023f, -0.367040346f,
    0.930766961f, -0.365612998f,
    0.931326709f, -0.364184790f,
    0.931884266f, -0.362755724f,
    0.932439629f, -0.361325806f,
    0.932992799f, -0.359895037f,
    0.933543773f, -0.358463421f,
    0.934092550f, -0.357030961f,
    0.934639130f, -0.355597662f,
    0.935183510f, -0.354163525f,
    0.935725689f, -0.352728556f,
    0.936265667f, -0.351292756f,
    0.936803442f, -0.349856130f,
    0.937339012f, -0.348418680f,
    0.937872376f, -0.346980411f,
    0.938403534f, -0.345541325f,
    0.938932484f, -0.344101426f,
    0.939459224f, -0.342660717f,
    0.939983753f, -0.341219202f,
    0.940506071f, -0.339776884f,
    0.941026175f, -0.338333767f,
    0.941544065f, -0.336889853f,
    0.942059740f, -0.335445147f,
    0.942573198f, -0.333999651f,
    0.943084437f, -0.332553370f,
    0.943593458f, -0.331106306f,
    0.944100258f, -0.329658463f,
    0.944604837f, -0.328209844f,
    0.945107193f, -0.326760452f,
    0.945607325f, -0.325310292f,
    0.946105232f, -0.323859367f,
    0.946600913f, -0.322407679f,
    0.947094366f, -0.320955232f,
    0.947585591f, -0.319502031f,
    0.948074586f, -0.318048077f,
    0.948561350f, -0.316593376f,
    0.949045882f, -0.315137929f,
    0.949528181f, -0.313681740f,
    0.950008245f, -0.312224814f,
    0.950486074f, -0.310767153f,
    0.950961666f, -0.309308760f,
    0.951435021f, -0.307849640f,
    0.951906137f, -0.306389795f,
    0.952375013f, -0.304929230f,
    0.952841648f, -0.303467947f,
    0.953306040f, -0.302005949f,
    0.953768190f, -0.300543241f,
    0.954228095f, -0.299079826f,
    0.954685755f, -0.297615707f,
    0.955141168f, -0.296150888f,
    0.955594334f, -0.294685372f,
    0.956045251f, -0.293219163f,
    0.956493919f, -0.291752263f,
    0.956940336f, -0.290284677f,
    0.957384501f, -0.288816408f,
    0.957826413f, -0.287347460f,
    0.958266071f, -0.285877835f,
    0.958703475f, -0.284407537f,
    0.959138622f, -0.282936570f,
    0.959571513f, -0.281464938f,
    0.960002146f, -0.279992643f,
    0.960430519f, -0.278519689f,
    0.960856633f, -0.277046080f,
    0.961280486f, -0.275571819f,
    0.961702077f, -0.274096910f,
    0.962121404f, -0.272621355f,
    0.962538468f, -0.271145160f,
    0.962953267f, -0.269668326f,
    0.963365800f, -0.268190857f,
    0.963776066f, -0.266712757f,
    0.964184064f, -0.265234030f,
    0.964589793f, -0.263754679f,
    0.964993253f, -0.262274707f,
    0.965394442f, -0.260794118f,
    0.965793359f, -0.259312915f,
    0.966190003f, -0.257831102f,
    0.966584374f, -0.256348682f,
    0.966976471f, -0.254865660f,
    0.967366292f, -0.253382037f,
    0.967753837f, -0.251897818f,
    0.968139105f, -0.250413007f,
    0.968522094f, -0.248927606f,
    0.968902805f, -0.247441619f,
    0.969281235f, -0.245955050f,
    0.969657385f, -0.244467903f,
    0.970031253f, -0.242980180f,
    0.970402839f, -0.241491885f,
    0.970772141f, -0.240003022f,
    0.971139158f, -0.238513595f,
    0.971503891f, -0.237023606f,
    0.971866337f, -0.235533059f,
    0.972226497f, -0.234041959f,
    0.972584369f, -0.232550307f,
    0.972939952f, -0.231058108f,
    0.973293246f, -0.229565366f,
    0.973644250f, -0.228072083f,
    0.973992962f, -0.226578264f,
    0.974339383f, -0.225083911f,
    0.974683511f, -0.223589029f,
    0.975025345f, -0.222093621f,
    0.975364885f, -0.220597690f,
    0.975702130f, -0.219101240f,
    0.976037079f, -0.217604275f,
    0.976369731f, -0.216106797f,
    0.976700086f, -0.214608811f,
    0.977028143f, -0.213110320f,
    0.977353900f, -0.211611327f,
    0.977677358f, -0.210111837f,
    0.977998515f, -0.208611852f,
    0.978317371f, -0.207111376f,
    0.978633924f, -0.205610413f,
    0.978948175f, -0.204108966f,
    0.979260123f, -0.202607039f,
    0.979569766f, -0.201104635f,
    0.979877104f, -0.199601758f,
    0.980182136f, -0.198098411f,
    0.980484862f, -0.196594598f,
    0.980785280f, -0.195090322f,
    0.981083391f, -0.193585587f,
    0.981379193f, -0.192080397f,
    0.981672686f, -0.190574755f,
    0.981963869f, -0.189068664f,
    0.982252741f, -0.187562129f,
    0.982539302f, -0.186055152f,
    0.982823551f, -0.184547737f,
    0.983105487f, -0.183039888f,
    0.983385110f, -0.181531608f,
    0.983662419f, -0.180022901f,
    0.983937413f, -0.178513771f,
    0.984210092f, -0.177004220f,
    0.984480455f, -0.175494253f,
    0.984748502f, -0.173983873f,
    0.985014231f, -0.172473084f,
    0.985277642f, -0.170961889f,
    0.985538735f, -0.169450291f,
    0.985797509f, -0.167938295f,
    0.986053963f, -0.166425904f,
    0.986308097f, -0.164913120f,
    0.986559910f, -0.163399949f,
    0.986809402f, -0.161886394f,
    0.987056571f, -0.160372457f,
    0.987301418f, -0.158858143f,
    0.987543942f, -0.157343456f,
    0.987784142f, -0.155828398f,
    0.988022017f, -0.154312973f,
    0.988257568f, -0.152797185f,
    0.988490793f, -0.151281038f,
    0.988721692f, -0.149764535f,
    0.988950265f, -0.148247679f,
    0.989176510f, -0.146730474f,
    0.989400428f, -0.145212925f,
    0.989622017f, -0.143695033f,
    0.989841278f, -0.142176804f,
    0.990058210f, -0.140658239f,
    0.990272812f, -0.139139344f,
    0.990485084f, -0.137620122f,
    0.990695025f, -0.136100575f,
    0.990902635f, -0.134580709f,
    0.991107914f, -0.133060525f,
    0.991310860f, -0.131540029f,
    0.991511473f, -0.130019223f,
    0.991709754f, -0.128498111f,
    0.991905700f, -0.126976696f,
    0.992099313f, -0.125454983f,
    0.992290591f, -0.123932975f,
    0.992479535f, -0.122410675f,
    0.992666142f, -0.120888087f,
    0.992850414f, -0.119365215f,
    0.993032350f, -0.117842062f,
    0.993211949f, -0.116318631f,
    0.993389211f, -0.114794927f,
    0.993564136f, -0.113270952f,
    0.993736722f, -0.111746711f,
    0.993906970f, -0.110222207f,
    0.994074879f, -0.108697444f,
    0.994240449f, -0.107172425f,
    0.994403680f, -0.105647154f,
    0.994564571f, -0.104121634f,
    0.994723121f, -0.102595869f,
    0.994879331f, -0.101069863f,
    0.995033199f, -0.099543619f,
    0.995184727f, -0.098017140f,
    0.995333912f, -0.096490431f,
    0.995480755f, -0.094963495f,
    0.995625256f, -0.093436336f,
    0.995767414f, -0.091908956f,
    0.995907229f, -0.090381361f,
    0.996044701f, -0.088853553f,
    0.996179829f, -0.087325535f,
    0.996312612f, -0.085797312f,
    0.996443051f, -0.084268888f,
    0.996571146f, -0.082740265f,
    0.996696895f, -0.081211447f,
    0.996820299f, -0.079682438f,
    0.996941358f, -0.078153242f,
    0.997060070f, -0.076623861f,
    0.997176437f, -0.075094301f,
    0.997290457f, -0.073564564f,
    0.997402130f, -0.072034653f,
    0.997511456f, -0.070504573f,
    0.997618435f, -0.068974328f,
    0.997723067f, -0.067443920f,
    0.997825350f, -0.065913353f,
    0.997925286f, -0.064382631f,
    0.998022874f, -0.062851758f,
    0.998118113f, -0.061320736f,
    0.998211003f, -0.059789571f,
    0.998301545f, -0.058258265f,
    0.998389737f, -0.056726821f,
    0.998475581f, -0.055195244f,
    0.998559074f, -0.053663538f,
    0.998640218f, -0.052131705f,
    0.998719012f, -0.050599749f,
    0.998795456f, -0.049067674f,
    0.998869550f, -0.047535484f,
    0.998941293f, -0.046003182f,
    0.999010686f, -0.044470772f,
    0.999077728f, -0.042938257f,
    0.999142419f, -0.041405641f,
    0.999204759f, -0.039872928f,
    0.999264747f, -0.038340120f,
    0.999322385f, -0.036807223f,
    0.999377670f, -0.035274239f,
    0.999430605f, -0.033741172f,
    0.999481187f, -0.032208025f,
    0.999529418f, -0.030674803f,
    0.999575296f, -0.029141509f,
    0.999618822f, -0.027608146f,
    0.999659997f, -0.026074718f,
    0.999698819f, -0.024541229f,
    0.999735288f, -0.023007681f,
    0.999769405f, -0.021474080f,
    0.999801170f, -0.019940429f,
    0.999830582f, -0.018406730f,
    0.999857641f, -0.016872988f,
    0.999882347f, -0.015339206f,
    0.999904701f, -0.013805389f,
    0.999924702f, -0.012271538f,
    0.999942350f, -0.010737659f,
    0.999957645f, -0.009203755f,
    0.999970586f, -0.007669829f,
    0.999981175f, -0.006135885f,
    0.999989411f, -0.004601926f,
    0.999995294f, -0.003067957f,
    0.999998823f, -0.001533980f
};
#endif // INCLUDE_FFT_TABLES_4096


#if defined(INCLUDE_FFT_TABLES_16)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable16[RISCVBITREVINDEXTABLE_16_TABLE_LENGTH] =
{
   /* 8x2, size 20 */
   8,64, 24,72, 16,64, 40,80, 32,64, 56,88, 48,72, 88,104, 72,96, 104,112
};
#endif // INCLUDE_FFT_TABLES_16


#if defined(INCLUDE_FFT_TABLES_32)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable32[RISCVBITREVINDEXTABLE_32_TABLE_LENGTH] =
{
   /* 8x4, size 48 */
   8,64, 16,128, 24,192, 32,64, 40,72, 48,136, 56,200, 64,128, 72,80, 88,208,
   80,144, 96,192, 104,208, 112,152, 120,216, 136,192, 144,160, 168,208,
   152,224, 176,208, 184,232, 216,240, 200,224, 232,240
};
#endif // INCLUDE_FFT_TABLES_32


#if defined(INCLUDE_FFT_TABLES_64)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable64[RISCVBITREVINDEXTABLE_64_TABLE_LENGTH] =
{
   /* radix 8, size 56 */
   8,64, 16,128, 24,192, 32,256, 40,320, 48,384, 56,448, 80,136, 88,200,
   96,264, 104,328, 112,392, 120,456, 152,208, 160,272, 168,336, 176,400,
   184,464, 224,280, 232,344, 240,408, 248,472, 296,352, 304,416, 312,480,
   368,424, 376,488, 440,496
};
#endif // INCLUDE_FFT_TABLES_64


#if defined(INCLUDE_FFT_TABLES_128)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable128[RISCVBITREVINDEXTABLE_128_TABLE_LENGTH] =
{
   /* 8x2, size 208 */
   8,512, 16,64, 24,576, 32,128, 40,640, 48,192, 56,704, 64,256, 72,768,
   80,320, 88,832, 96,384, 104,896, 112,448, 120,960, 128,512, 136,520,
   144,768, 152,584, 160,520, 168,648, 176,200, 184,712, 192,264, 200,776,
   208,328, 216,840, 224,392, 232,904, 240,456, 248,968, 264,528, 272,320,
   280,592, 288,768, 296,656, 304,328, 312,720, 328,784, 344,848, 352,400,
   360,912, 368,464, 376,976, 384,576, 392,536, 400,832, 408,600, 416,584,
   424,664, 432,840, 440,728, 448,592, 456,792, 464,848, 472,856, 480,600,
   488,920, 496,856, 504,984, 520,544, 528,576, 536,608, 552,672, 560,608,
   568,736, 576,768, 584,800, 592,832, 600,864, 608,800, 616,928, 624,864,
   632,992, 648,672, 656,896, 664,928, 688,904, 696,744, 704,896, 712,808,
   720,912, 728,872, 736,928, 744,936, 752,920, 760,1000, 776,800, 784,832,
   792,864, 808,904, 816,864, 824,920, 840,864, 856,880, 872,944, 888,1008,
   904,928, 912,960, 920,992, 944,968, 952,1000, 968,992, 984,1008
};
#endif // INCLUDE_FFT_TABLES_128


#if defined(INCLUDE_FFT_TABLES_256)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable256[RISCVBITREVINDEXTABLE_256_TABLE_LENGTH] =
{
   /* 8x4, size 440 */
   8,512, 16,1024, 24,1536, 32,64, 40,576, 48,1088, 56,1600, 64,128, 72,640,
   80,1152, 88,1664, 96,192, 104,704, 112,1216, 120,1728, 128,256, 136,768,
   144,1280, 152,1792, 160,320, 168,832, 176,1344, 184,1856, 192,384,
   200,896, 208,1408, 216,1920, 224,448, 232,960, 240,1472, 248,1984,
   256,512, 264,520, 272,1032, 280,1544, 288,640, 296,584, 304,1096, 312,1608,
   320,768, 328,648, 336,1160, 344,1672, 352,896, 360,712, 368,1224, 376,1736,
   384,520, 392,776, 400,1288, 408,1800, 416,648, 424,840, 432,1352, 440,1864,
   448,776, 456,904, 464,1416, 472,1928, 480,904, 488,968, 496,1480, 504,1992,
   520,528, 512,1024, 528,1040, 536,1552, 544,1152, 552,592, 560,1104,
   568,1616, 576,1280, 584,656, 592,1168, 600,1680, 608,1408, 616,720,
   624,1232, 632,1744, 640,1032, 648,784, 656,1296, 664,1808, 672,1160,
   680,848, 688,1360, 696,1872, 704,1288, 712,912, 720,1424, 728,1936,
   736,1416, 744,976, 752,1488, 760,2000, 768,1536, 776,1552, 784,1048,
   792,1560, 800,1664, 808,1680, 816,1112, 824,1624, 832,1792, 840,1808,
   848,1176, 856,1688, 864,1920, 872,1936, 880,1240, 888,1752, 896,1544,
   904,1560, 912,1304, 920,1816, 928,1672, 936,1688, 944,1368, 952,1880,
   960,1800, 968,1816, 976,1432, 984,1944, 992,1928, 1000,1944, 1008,1496,
   1016,2008, 1032,1152, 1040,1056, 1048,1568, 1064,1408, 1072,1120,
   1080,1632, 1088,1536, 1096,1160, 1104,1184, 1112,1696, 1120,1552,
   1128,1416, 1136,1248, 1144,1760, 1160,1664, 1168,1312, 1176,1824,
   1184,1544, 1192,1920, 1200,1376, 1208,1888, 1216,1568, 1224,1672,
   1232,1440, 1240,1952, 1248,1560, 1256,1928, 1264,1504, 1272,2016,
   1288,1312, 1296,1408, 1304,1576, 1320,1424, 1328,1416, 1336,1640,
   1344,1792, 1352,1824, 1360,1920, 1368,1704, 1376,1800, 1384,1432,
   1392,1928, 1400,1768, 1416,1680, 1432,1832, 1440,1576, 1448,1936,
   1456,1832, 1464,1896, 1472,1808, 1480,1688, 1488,1936, 1496,1960,
   1504,1816, 1512,1944, 1520,1944, 1528,2024, 1560,1584, 1592,1648,
   1600,1792, 1608,1920, 1616,1800, 1624,1712, 1632,1808, 1640,1936,
   1648,1816, 1656,1776, 1672,1696, 1688,1840, 1704,1952, 1712,1928,
   1720,1904, 1728,1824, 1736,1952, 1744,1832, 1752,1968, 1760,1840,
   1768,1960, 1776,1944, 1784,2032, 1864,1872, 1848,1944, 1872,1888,
   1880,1904, 1888,1984, 1896,2000, 1912,2032, 1904,2016, 1976,2032,
   1960,1968, 2008,2032, 1992,2016, 2024,2032
};
#endif // INCLUDE_FFT_TABLES_256


#if defined(INCLUDE_FFT_TABLES_512)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable512[RISCVBITREVINDEXTABLE_512_TABLE_LENGTH] =
{
   /* radix 8, size 448 */
   8,512, 16,1024, 24,1536, 32,2048, 40,2560, 48,3072, 56,3584, 72,576,
   80,1088, 88,1600, 96,2112, 104,2624, 112,3136, 120,3648, 136,640, 144,1152,
   152,1664, 160,2176, 168,2688, 176,3200, 184,3712, 200,704, 208,1216,
   216,1728, 224,2240, 232,2752, 240,3264, 248,3776, 264,768, 272,1280,
   280,1792, 288,2304, 296,2816, 304,3328, 312,3840, 328,832, 336,1344,
   344,1856, 352,2368, 360,2880, 368,3392, 376,3904, 392,896, 400,1408,
   408,1920, 416,2432, 424,2944, 432,3456, 440,3968, 456,960, 464,1472,
   472,1984, 480,2496, 488,3008, 496,3520, 504,4032, 528,1032, 536,1544,
   544,2056, 552,2568, 560,3080, 568,3592, 592,1096, 600,1608, 608,2120,
   616,2632, 624,3144, 632,3656, 656,1160, 664,1672, 672,2184, 680,2696,
   688,3208, 696,3720, 720,1224, 728,1736, 736,2248, 744,2760, 752,3272,
   760,3784, 784,1288, 792,1800, 800,2312, 808,2824, 816,3336, 824,3848,
   848,1352, 856,1864, 864,2376, 872,2888, 880,3400, 888,3912, 912,1416,
   920,1928, 928,2440, 936,2952, 944,3464, 952,3976, 976,1480, 984,1992,
   992,2504, 1000,3016, 1008,3528, 1016,4040, 1048,1552, 1056,2064, 1064,2576,
   1072,3088, 1080,3600, 1112,1616, 1120,2128, 1128,2640, 1136,3152,
   1144,3664, 1176,1680, 1184,2192, 1192,2704, 1200,3216, 1208,3728,
   1240,1744, 1248,2256, 1256,2768, 1264,3280, 1272,3792, 1304,1808,
   1312,2320, 1320,2832, 1328,3344, 1336,3856, 1368,1872, 1376,2384,
   1384,2896, 1392,3408, 1400,3920, 1432,1936, 1440,2448, 1448,2960,
   1456,3472, 1464,3984, 1496,2000, 1504,2512, 1512,3024, 1520,3536,
   1528,4048, 1568,2072, 1576,2584, 1584,3096, 1592,3608, 1632,2136,
   1640,2648, 1648,3160, 1656,3672, 1696,2200, 1704,2712, 1712,3224,
   1720,3736, 1760,2264, 1768,2776, 1776,3288, 1784,3800, 1824,2328,
   1832,2840, 1840,3352, 1848,3864, 1888,2392, 1896,2904, 1904,3416,
   1912,3928, 1952,2456, 1960,2968, 1968,3480, 1976,3992, 2016,2520,
   2024,3032, 2032,3544, 2040,4056, 2088,2592, 2096,3104, 2104,3616,
   2152,2656, 2160,3168, 2168,3680, 2216,2720, 2224,3232, 2232,3744,
   2280,2784, 2288,3296, 2296,3808, 2344,2848, 2352,3360, 2360,3872,
   2408,2912, 2416,3424, 2424,3936, 2472,2976, 2480,3488, 2488,4000,
   2536,3040, 2544,3552, 2552,4064, 2608,3112, 2616,3624, 2672,3176,
   2680,3688, 2736,3240, 2744,3752, 2800,3304, 2808,3816, 2864,3368,
   2872,3880, 2928,3432, 2936,3944, 2992,3496, 3000,4008, 3056,3560,
   3064,4072, 3128,3632, 3192,3696, 3256,3760, 3320,3824, 3384,3888,
   3448,3952, 3512,4016, 3576,4080
};
#endif // INCLUDE_FFT_TABLES_512


#if defined(INCLUDE_FFT_TABLES_1024)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable1024[RISCVBITREVINDEXTABLE_1024_TABLE_LENGTH] =
{
   /* 8x2, size 1800 */
   8,4096, 16,512, 24,4608, 32,1024, 40,5120, 48,1536, 56,5632, 64,2048,
   72,6144, 80,2560, 88,6656, 96,3072, 104,7168, 112,3584, 120,7680, 128,2048,
   136,4160, 144,576, 152,4672, 160,1088, 168,5184, 176,1600, 184,5696,
   192,2112, 200,6208, 208,2624, 216,6720, 224,3136, 232,7232, 240,3648,
   248,7744, 256,2048, 264,4224, 272,640, 280,4736, 288,1152, 296,5248,
   304,1664, 312,5760, 320,2176, 328,6272, 336,2688, 344,6784, 352,3200,
   360,7296, 368,3712, 376,7808, 384,2112, 392,4288, 400,704, 408,4800,
   416,1216, 424,5312, 432,1728, 440,5824, 448,2240, 456,6336, 464,2752,
   472,6848, 480,3264, 488,7360, 496,3776, 504,7872, 512,2048, 520,4352,
   528,768, 536,4864, 544,1280, 552,5376, 560,1792, 568,5888, 576,2304,
   584,6400, 592,2816, 600,6912, 608,3328, 616,7424, 624,3840, 632,7936,
   640,2176, 648,4416, 656,832, 664,4928, 672,1344, 680,5440, 688,1856,
   696,5952, 704,2368, 712,6464, 720,2880, 728,6976, 736,3392, 744,7488,
   752,3904, 760,8000, 768,2112, 776,4480, 784,896, 792,4992, 800,1408,
   808,5504, 816,1920, 824,6016, 832,2432, 840,6528, 848,2944, 856,7040,
   864,3456, 872,7552, 880,3968, 888,8064, 896,2240, 904,4544, 912,960,
   920,5056, 928,1472, 936,5568, 944,1984, 952,6080, 960,2496, 968,6592,
   976,3008, 984,7104, 992,3520, 1000,7616, 1008,4032, 1016,8128, 1024,4096,
   1032,4104, 1040,4352, 1048,4616, 1056,4104, 1064,5128, 1072,1544,
   1080,5640, 1088,2056, 1096,6152, 1104,2568, 1112,6664, 1120,3080,
   1128,7176, 1136,3592, 1144,7688, 1152,6144, 1160,4168, 1168,6400,
   1176,4680, 1184,6152, 1192,5192, 1200,1608, 1208,5704, 1216,2120,
   1224,6216, 1232,2632, 1240,6728, 1248,3144, 1256,7240, 1264,3656,
   1272,7752, 1280,4160, 1288,4232, 1296,4416, 1304,4744, 1312,4168,
   1320,5256, 1328,1672, 1336,5768, 1344,2184, 1352,6280, 1360,2696,
   1368,6792, 1376,3208, 1384,7304, 1392,3720, 1400,7816, 1408,6208,
   1416,4296, 1424,6464, 1432,4808, 1440,6216, 1448,5320, 1456,1736,
   1464,5832, 1472,2248, 1480,6344, 1488,2760, 1496,6856, 1504,3272,
   1512,7368, 1520,3784, 1528,7880, 1536,4224, 1544,4360, 1552,4480,
   1560,4872, 1568,4232, 1576,5384, 1584,1800, 1592,5896, 1600,2312,
   1608,6408, 1616,2824, 1624,6920, 1632,3336, 1640,7432, 1648,3848,
   1656,7944, 1664,6272, 1672,4424, 1680,6528, 1688,4936, 1696,6280,
   1704,5448, 1712,1864, 1720,5960, 1728,2376, 1736,6472, 1744,2888,
   1752,6984, 1760,3400, 1768,7496, 1776,3912, 1784,8008, 1792,4288,
   1800,4488, 1808,4544, 1816,5000, 1824,4296, 1832,5512, 1840,1928,
   1848,6024, 1856,2440, 1864,6536, 1872,2952, 1880,7048, 1888,3464,
   1896,7560, 1904,3976, 1912,8072, 1920,6336, 1928,4552, 1936,6592,
   1944,5064, 1952,6344, 1960,5576, 1968,1992, 1976,6088, 1984,2504,
   1992,6600, 2000,3016, 2008,7112, 2016,3528, 2024,7624, 2032,4040,
   2040,8136, 2056,4112, 2064,2112, 2072,4624, 2080,4352, 2088,5136,
   2096,4480, 2104,5648, 2120,6160, 2128,2576, 2136,6672, 2144,3088,
   2152,7184, 2160,3600, 2168,7696, 2176,2560, 2184,4176, 2192,2816,
   2200,4688, 2208,2568, 2216,5200, 2224,2824, 2232,5712, 2240,2576,
   2248,6224, 2256,2640, 2264,6736, 2272,3152, 2280,7248, 2288,3664,
   2296,7760, 2312,4240, 2320,2432, 2328,4752, 2336,6400, 2344,5264,
   2352,6528, 2360,5776, 2368,2816, 2376,6288, 2384,2704, 2392,6800,
   2400,3216, 2408,7312, 2416,3728, 2424,7824, 2432,2624, 2440,4304,
   2448,2880, 2456,4816, 2464,2632, 2472,5328, 2480,2888, 2488,5840,
   2496,2640, 2504,6352, 2512,2768, 2520,6864, 2528,3280, 2536,7376,
   2544,3792, 2552,7888, 2568,4368, 2584,4880, 2592,4416, 2600,5392,
   2608,4544, 2616,5904, 2632,6416, 2640,2832, 2648,6928, 2656,3344,
   2664,7440, 2672,3856, 2680,7952, 2696,4432, 2704,2944, 2712,4944,
   2720,4432, 2728,5456, 2736,2952, 2744,5968, 2752,2944, 2760,6480,
   2768,2896, 2776,6992, 2784,3408, 2792,7504, 2800,3920, 2808,8016,
   2824,4496, 2840,5008, 2848,6464, 2856,5520, 2864,6592, 2872,6032,
   2888,6544, 2896,2960, 2904,7056, 2912,3472, 2920,7568, 2928,3984,
   2936,8080, 2952,4560, 2960,3008, 2968,5072, 2976,6480, 2984,5584,
   2992,3016, 3000,6096, 3016,6608, 3032,7120, 3040,3536, 3048,7632,
   3056,4048, 3064,8144, 3072,4608, 3080,4120, 3088,4864, 3096,4632,
   3104,4616, 3112,5144, 3120,4872, 3128,5656, 3136,4624, 3144,6168,
   3152,4880, 3160,6680, 3168,4632, 3176,7192, 3184,3608, 3192,7704,
   3200,6656, 3208,4184, 3216,6912, 3224,4696, 3232,6664, 3240,5208,
   3248,6920, 3256,5720, 3264,6672, 3272,6232, 3280,6928, 3288,6744,
   3296,6680, 3304,7256, 3312,3672, 3320,7768, 3328,4672, 3336,4248,
   3344,4928, 3352,4760, 3360,4680, 3368,5272, 3376,4936, 3384,5784,
   3392,4688, 3400,6296, 3408,4944, 3416,6808, 3424,4696, 3432,7320,
   3440,3736, 3448,7832, 3456,6720, 3464,4312, 3472,6976, 3480,4824,
   3488,6728, 3496,5336, 3504,6984, 3512,5848, 3520,6736, 3528,6360,
   3536,6992, 3544,6872, 3552,6744, 3560,7384, 3568,3800, 3576,7896,
   3584,4736, 3592,4376, 3600,4992, 3608,4888, 3616,4744, 3624,5400,
   3632,5000, 3640,5912, 3648,4752, 3656,6424, 3664,5008, 3672,6936,
   3680,4760, 3688,7448, 3696,3864, 3704,7960, 3712,6784, 3720,4440,
   3728,7040, 3736,4952, 3744,6792, 3752,5464, 3760,7048, 3768,5976,
   3776,6800, 3784,6488, 3792,7056, 3800,7000, 3808,6808, 3816,7512,
   3824,3928, 3832,8024, 3840,4800, 3848,4504, 3856,5056, 3864,5016,
   3872,4808, 3880,5528, 3888,5064, 3896,6040, 3904,4816, 3912,6552,
   3920,5072, 3928,7064, 3936,4824, 3944,7576, 3952,3992, 3960,8088,
   3968,6848, 3976,4568, 3984,7104, 3992,5080, 4000,6856, 4008,5592,
   4016,7112, 4024,6104, 4032,6864, 4040,6616, 4048,7120, 4056,7128,
   4064,6872, 4072,7640, 4080,7128, 4088,8152, 4104,4128, 4112,4160,
   4120,4640, 4136,5152, 4144,4232, 4152,5664, 4160,4352, 4168,6176,
   4176,4416, 4184,6688, 4192,4616, 4200,7200, 4208,4744, 4216,7712,
   4224,4608, 4232,4616, 4240,4672, 4248,4704, 4256,4640, 4264,5216,
   4272,4704, 4280,5728, 4288,4864, 4296,6240, 4304,4928, 4312,6752,
   4320,4632, 4328,7264, 4336,4760, 4344,7776, 4360,4640, 4368,4416,
   4376,4768, 4384,6152, 4392,5280, 4400,6280, 4408,5792, 4424,6304,
   4440,6816, 4448,6664, 4456,7328, 4464,6792, 4472,7840, 4480,4624,
   4488,4632, 4496,4688, 4504,4832, 4512,6168, 4520,5344, 4528,6296,
   4536,5856, 4544,4880, 4552,6368, 4560,4944, 4568,6880, 4576,6680,
   4584,7392, 4592,6808, 4600,7904, 4608,6144, 4616,6152, 4624,6208,
   4632,4896, 4640,6176, 4648,5408, 4656,6240, 4664,5920, 4672,6400,
   4680,6432, 4688,6464, 4696,6944, 4704,6432, 4712,7456, 4720,4808,
   4728,7968, 4736,6656, 4744,6664, 4752,6720, 4760,4960, 4768,6688,
   4776,5472, 4784,6752, 4792,5984, 4800,6912, 4808,6496, 4816,6976,
   4824,7008, 4832,6944, 4840,7520, 4848,7008, 4856,8032, 4864,6160,
   4872,6168, 4880,6224, 4888,5024, 4896,6216, 4904,5536, 4912,6344,
   4920,6048, 4928,6416, 4936,6560, 4944,6480, 4952,7072, 4960,6728,
   4968,7584, 4976,6856, 4984,8096, 4992,6672, 5000,6680, 5008,6736,
   5016,5088, 5024,6232, 5032,5600, 5040,6360, 5048,6112, 5056,6928,
   5064,6624, 5072,6992, 5080,7136, 5088,6744, 5096,7648, 5104,6872,
   5112,8160, 5128,5152, 5136,5376, 5144,5408, 5168,5384, 5176,5672,
   5184,5376, 5192,6184, 5200,5392, 5208,6696, 5216,5408, 5224,7208,
   5232,5400, 5240,7720, 5248,7168, 5256,7200, 5264,7424, 5272,7456,
   5280,7176, 5288,7208, 5296,7432, 5304,5736, 5312,7184, 5320,6248,
   5328,7440, 5336,6760, 5344,7192, 5352,7272, 5360,7448, 5368,7784,
   5384,5408, 5392,5440, 5400,5472, 5408,6184, 5416,7208, 5424,5448,
   5432,5800, 5448,6312, 5464,6824, 5472,6696, 5480,7336, 5488,6824,
   5496,7848, 5504,7232, 5512,7264, 5520,7488, 5528,7520, 5536,7240,
   5544,7272, 5552,7496, 5560,5864, 5568,7248, 5576,6376, 5584,7504,
   5592,6888, 5600,7256, 5608,7400, 5616,7512, 5624,7912, 5632,7168,
   5640,7176, 5648,7232, 5656,7240, 5664,7200, 5672,7208, 5680,7264,
   5688,5928, 5696,7424, 5704,6440, 5712,7488, 5720,6952, 5728,7456,
   5736,7464, 5744,7520, 5752,7976, 5760,7296, 5768,7328, 5776,7552,
   5784,7584, 5792,7304, 5800,7336, 5808,7560, 5816,5992, 5824,7312,
   5832,6504, 5840,7568, 5848,7016, 5856,7320, 5864,7528, 5872,7576,
   5880,8040, 5888,7184, 5896,7192, 5904,7248, 5912,7256, 5920,6248,
   5928,7272, 5936,6376, 5944,6056, 5952,7440, 5960,6568, 5968,7504,
   5976,7080, 5984,6760, 5992,7592, 6000,6888, 6008,8104, 6016,7360,
   6024,7392, 6032,7616, 6040,7648, 6048,7368, 6056,7400, 6064,7624,
   6072,6120, 6080,7376, 6088,6632, 6096,7632, 6104,7144, 6112,7384,
   6120,7656, 6128,7640, 6136,8168, 6168,6240, 6192,6216, 6200,7264,
   6232,6704, 6248,7216, 6256,6680, 6264,7728, 6272,6656, 6280,6664,
   6288,6912, 6296,6496, 6304,6688, 6312,6696, 6320,6944, 6328,7520,
   6336,6672, 6344,6680, 6352,6928, 6360,6768, 6368,6704, 6376,7280,
   6384,6744, 6392,7792, 6408,6432, 6424,6752, 6440,7432, 6448,6536,
   6456,7560, 6472,6944, 6488,6832, 6496,6920, 6504,7344, 6512,7048,
   6520,7856, 6528,6720, 6536,6728, 6544,6976, 6552,7008, 6560,6752,
   6568,7448, 6576,7008, 6584,7576, 6592,6736, 6600,6744, 6608,6992,
   6616,6896, 6624,6936, 6632,7408, 6640,7064, 6648,7920, 6712,7280,
   6744,6960, 6760,7472, 6768,6936, 6776,7984, 6800,6848, 6808,6856,
   6832,6880, 6840,6888, 6848,7040, 6856,7048, 6864,7104, 6872,7024,
   6880,7072, 6888,7536, 6896,7136, 6904,8048, 6952,7496, 6968,7624,
   6984,7008, 7000,7088, 7016,7600, 7024,7112, 7032,8112, 7056,7104,
   7064,7112, 7080,7512, 7088,7136, 7096,7640, 7128,7152, 7144,7664,
   7160,8176, 7176,7200, 7192,7216, 7224,7272, 7240,7264, 7256,7280,
   7288,7736, 7296,7680, 7304,7712, 7312,7936, 7320,7968, 7328,7688,
   7336,7720, 7344,7944, 7352,7976, 7360,7696, 7368,7728, 7376,7952,
   7384,7984, 7392,7704, 7400,7736, 7408,7960, 7416,7800, 7432,7456,
   7448,7472, 7480,7592, 7496,7520, 7512,7536, 7528,7976, 7544,7864,
   7552,7744, 7560,7776, 7568,8000, 7576,8032, 7584,7752, 7592,7784,
   7600,8008, 7608,8040, 7616,7760, 7624,7792, 7632,8016, 7640,8048,
   7648,7768, 7656,7800, 7664,8024, 7672,7928, 7688,7712, 7704,7728,
   7752,7776, 7768,7792, 7800,7992, 7816,7840, 7824,8064, 7832,8096,
   7856,8072, 7864,8104, 7872,8064, 7880,8072, 7888,8080, 7896,8112,
   7904,8096, 7912,8104, 7920,8088, 7928,8056, 7944,7968, 7960,7984,
   8008,8032, 8024,8048, 8056,8120, 8072,8096, 8080,8128, 8088,8160,
   8112,8136, 8120,8168, 8136,8160, 8152,8176
};
#endif // INCLUDE_FFT_TABLES_1024


#if defined(INCLUDE_FFT_TABLES_2048)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable2048[RISCVBITREVINDEXTABLE_2048_TABLE_LENGTH] =
{
   /* 8x2, size 3808 */
   8,4096, 16,8192, 24,12288, 32,512, 40,4608, 48,8704, 56,12800, 64,1024,
   72,5120, 80,9216, 88,13312, 96,1536, 104,5632, 112,9728, 120,13824,
   128,2048, 136,6144, 144,10240, 152,14336, 160,2560, 168,6656, 176,10752,
   184,14848, 192,3072, 200,7168, 208,11264, 216,15360, 224,3584, 232,7680,
   240,11776, 248,15872, 256,1024, 264,4160, 272,8256, 280,12352, 288,576,
   296,4672, 304,8768, 312,12864, 320,1088, 328,5184, 336,9280, 344,13376,
   352,1600, 360,5696, 368,9792, 376,13888, 384,2112, 392,6208, 400,10304,
   408,14400, 416,2624, 424,6720, 432,10816, 440,14912, 448,3136, 456,7232,
   464,11328, 472,15424, 480,3648, 488,7744, 496,11840, 504,15936, 512,2048,
   520,4224, 528,8320, 536,12416, 544,640, 552,4736, 560,8832, 568,12928,
   576,1152, 584,5248, 592,9344, 600,13440, 608,1664, 616,5760, 624,9856,
   632,13952, 640,2176, 648,6272, 656,10368, 664,14464, 672,2688, 680,6784,
   688,10880, 696,14976, 704,3200, 712,7296, 720,11392, 728,15488, 736,3712,
   744,7808, 752,11904, 760,16000, 768,3072, 776,4288, 784,8384, 792,12480,
   800,3200, 808,4800, 816,8896, 824,12992, 832,1216, 840,5312, 848,9408,
   856,13504, 864,1728, 872,5824, 880,9920, 888,14016, 896,2240, 904,6336,
   912,10432, 920,14528, 928,2752, 936,6848, 944,10944, 952,15040, 960,3264,
   968,7360, 976,11456, 984,15552, 992,3776, 1000,7872, 1008,11968, 1016,16064,
   1032,4352, 1040,8448, 1048,12544, 1056,3072, 1064,4864, 1072,8960,
   1080,13056, 1088,1280, 1096,5376, 1104,9472, 1112,13568, 1120,1792,
   1128,5888, 1136,9984, 1144,14080, 1152,2304, 1160,6400, 1168,10496,
   1176,14592, 1184,2816, 1192,6912, 1200,11008, 1208,15104, 1216,3328,
   1224,7424, 1232,11520, 1240,15616, 1248,3840, 1256,7936, 1264,12032,
   1272,16128, 1288,4416, 1296,8512, 1304,12608, 1312,3328, 1320,4928,
   1328,9024, 1336,13120, 1352,5440, 1360,9536, 1368,13632, 1376,1856,
   1384,5952, 1392,10048, 1400,14144, 1408,2368, 1416,6464, 1424,10560,
   1432,14656, 1440,2880, 1448,6976, 1456,11072, 1464,15168, 1472,3392,
   1480,7488, 1488,11584, 1496,15680, 1504,3904, 1512,8000, 1520,12096,
   1528,16192, 1536,2112, 1544,4480, 1552,8576, 1560,12672, 1568,2240,
   1576,4992, 1584,9088, 1592,13184, 1600,2368, 1608,5504, 1616,9600,
   1624,13696, 1632,1920, 1640,6016, 1648,10112, 1656,14208, 1664,2432,
   1672,6528, 1680,10624, 1688,14720, 1696,2944, 1704,7040, 1712,11136,
   1720,15232, 1728,3456, 1736,7552, 1744,11648, 1752,15744, 1760,3968,
   1768,8064, 1776,12160, 1784,16256, 1792,3136, 1800,4544, 1808,8640,
   1816,12736, 1824,3264, 1832,5056, 1840,9152, 1848,13248, 1856,3392,
   1864,5568, 1872,9664, 1880,13760, 1888,1984, 1896,6080, 1904,10176,
   1912,14272, 1920,2496, 1928,6592, 1936,10688, 1944,14784, 1952,3008,
   1960,7104, 1968,11200, 1976,15296, 1984,3520, 1992,7616, 2000,11712,
   2008,15808, 2016,4032, 2024,8128, 2032,12224, 2040,16320, 2048,4096,
   2056,4104, 2064,8200, 2072,12296, 2080,4224, 2088,4616, 2096,8712,
   2104,12808, 2112,4352, 2120,5128, 2128,9224, 2136,13320, 2144,4480,
   2152,5640, 2160,9736, 2168,13832, 2176,4104, 2184,6152, 2192,10248,
   2200,14344, 2208,2568, 2216,6664, 2224,10760, 2232,14856, 2240,3080,
   2248,7176, 2256,11272, 2264,15368, 2272,3592, 2280,7688, 2288,11784,
   2296,15880, 2304,5120, 2312,4168, 2320,8264, 2328,12360, 2336,5248,
   2344,4680, 2352,8776, 2360,12872, 2368,5376, 2376,5192, 2384,9288,
   2392,13384, 2400,5504, 2408,5704, 2416,9800, 2424,13896, 2432,5128,
   2440,6216, 2448,10312, 2456,14408, 2464,2632, 2472,6728, 2480,10824,
   2488,14920, 2496,3144, 2504,7240, 2512,11336, 2520,15432, 2528,3656,
   2536,7752, 2544,11848, 2552,15944, 2560,6144, 2568,4232, 2576,8328,
   2584,12424, 2592,6272, 2600,4744, 2608,8840, 2616,12936, 2624,6400,
   2632,5256, 2640,9352, 2648,13448, 2656,6528, 2664,5768, 2672,9864,
   2680,13960, 2688,6152, 2696,6280, 2704,10376, 2712,14472, 2720,6280,
   2728,6792, 2736,10888, 2744,14984, 2752,3208, 2760,7304, 2768,11400,
   2776,15496, 2784,3720, 2792,7816, 2800,11912, 2808,16008, 2816,7168,
   2824,4296, 2832,8392, 2840,12488, 2848,7296, 2856,4808, 2864,8904,
   2872,13000, 2880,7424, 2888,5320, 2896,9416, 2904,13512, 2912,7552,
   2920,5832, 2928,9928, 2936,14024, 2944,7176, 2952,6344, 2960,10440,
   2968,14536, 2976,7304, 2984,6856, 2992,10952, 3000,15048, 3008,3272,
   3016,7368, 3024,11464, 3032,15560, 3040,3784, 3048,7880, 3056,11976,
   3064,16072, 3072,4160, 3080,4360, 3088,8456, 3096,12552, 3104,4288,
   3112,4872, 3120,8968, 3128,13064, 3136,4416, 3144,5384, 3152,9480,
   3160,13576, 3168,4544, 3176,5896, 3184,9992, 3192,14088, 3200,4168,
   3208,6408, 3216,10504, 3224,14600, 3232,4296, 3240,6920, 3248,11016,
   3256,15112, 3264,3336, 3272,7432, 3280,11528, 3288,15624, 3296,3848,
   3304,7944, 3312,12040, 3320,16136, 3328,5184, 3336,4424, 3344,8520,
   3352,12616, 3360,5312, 3368,4936, 3376,9032, 3384,13128, 3392,5440,
   3400,5448, 3408,9544, 3416,13640, 3424,5568, 3432,5960, 3440,10056,
   3448,14152, 3456,5192, 3464,6472, 3472,10568, 3480,14664, 3488,5320,
   3496,6984, 3504,11080, 3512,15176, 3520,5448, 3528,7496, 3536,11592,
   3544,15688, 3552,3912, 3560,8008, 3568,12104, 3576,16200, 3584,6208,
   3592,4488, 3600,8584, 3608,12680, 3616,6336, 3624,5000, 3632,9096,
   3640,13192, 3648,6464, 3656,5512, 3664,9608, 3672,13704, 3680,6592,
   3688,6024, 3696,10120, 3704,14216, 3712,6216, 3720,6536, 3728,10632,
   3736,14728, 3744,6344, 3752,7048, 3760,11144, 3768,15240, 3776,6472,
   3784,7560, 3792,11656, 3800,15752, 3808,3976, 3816,8072, 3824,12168,
   3832,16264, 3840,7232, 3848,4552, 3856,8648, 3864,12744, 3872,7360,
   3880,5064, 3888,9160, 3896,13256, 3904,7488, 3912,5576, 3920,9672,
   3928,13768, 3936,7616, 3944,6088, 3952,10184, 3960,14280, 3968,7240,
   3976,6600, 3984,10696, 3992,14792, 4000,7368, 4008,7112, 4016,11208,
   4024,15304, 4032,7496, 4040,7624, 4048,11720, 4056,15816, 4064,7624,
   4072,8136, 4080,12232, 4088,16328, 4096,8192, 4104,4112, 4112,8208,
   4120,12304, 4128,8320, 4136,4624, 4144,8720, 4152,12816, 4160,8448,
   4168,5136, 4176,9232, 4184,13328, 4192,8576, 4200,5648, 4208,9744,
   4216,13840, 4224,8200, 4232,6160, 4240,10256, 4248,14352, 4256,8328,
   4264,6672, 4272,10768, 4280,14864, 4288,8456, 4296,7184, 4304,11280,
   4312,15376, 4320,8584, 4328,7696, 4336,11792, 4344,15888, 4352,9216,
   4360,9232, 4368,8272, 4376,12368, 4384,9344, 4392,4688, 4400,8784,
   4408,12880, 4416,9472, 4424,5200, 4432,9296, 4440,13392, 4448,9600,
   4456,5712, 4464,9808, 4472,13904, 4480,9224, 4488,6224, 4496,10320,
   4504,14416, 4512,9352, 4520,6736, 4528,10832, 4536,14928, 4544,9480,
   4552,7248, 4560,11344, 4568,15440, 4576,9608, 4584,7760, 4592,11856,
   4600,15952, 4608,10240, 4616,10256, 4624,8336, 4632,12432, 4640,10368,
   4648,4752, 4656,8848, 4664,12944, 4672,10496, 4680,5264, 4688,9360,
   4696,13456, 4704,10624, 4712,5776, 4720,9872, 4728,13968, 4736,10248,
   4744,6288, 4752,10384, 4760,14480, 4768,10376, 4776,6800, 4784,10896,
   4792,14992, 4800,10504, 4808,7312, 4816,11408, 4824,15504, 4832,10632,
   4840,7824, 4848,11920, 4856,16016, 4864,11264, 4872,11280, 4880,8400,
   4888,12496, 4896,11392, 4904,11408, 4912,8912, 4920,13008, 4928,11520,
   4936,5328, 4944,9424, 4952,13520, 4960,11648, 4968,5840, 4976,9936,
   4984,14032, 4992,11272, 5000,6352, 5008,10448, 5016,14544, 5024,11400,
   5032,6864, 5040,10960, 5048,15056, 5056,11528, 5064,7376, 5072,11472,
   5080,15568, 5088,11656, 5096,7888, 5104,11984, 5112,16080, 5120,8256,
   5128,8272, 5136,8464, 5144,12560, 5152,8384, 5160,8400, 5168,8976,
   5176,13072, 5184,8512, 5192,5392, 5200,9488, 5208,13584, 5216,8640,
   5224,5904, 5232,10000, 5240,14096, 5248,8264, 5256,6416, 5264,10512,
   5272,14608, 5280,8392, 5288,6928, 5296,11024, 5304,15120, 5312,8520,
   5320,7440, 5328,11536, 5336,15632, 5344,8648, 5352,7952, 5360,12048,
   5368,16144, 5376,9280, 5384,9296, 5392,8528, 5400,12624, 5408,9408,
   5416,9424, 5424,9040, 5432,13136, 5440,9536, 5448,5456, 5456,9552,
   5464,13648, 5472,9664, 5480,5968, 5488,10064, 5496,14160, 5504,9288,
   5512,6480, 5520,10576, 5528,14672, 5536,9416, 5544,6992, 5552,11088,
   5560,15184, 5568,9544, 5576,7504, 5584,11600, 5592,15696, 5600,9672,
   5608,8016, 5616,12112, 5624,16208, 5632,10304, 5640,10320, 5648,8592,
   5656,12688, 5664,10432, 5672,10448, 5680,9104, 5688,13200, 5696,10560,
   5704,10576, 5712,9616, 5720,13712, 5728,10688, 5736,6032, 5744,10128,
   5752,14224, 5760,10312, 5768,6544, 5776,10640, 5784,14736, 5792,10440,
   5800,7056, 5808,11152, 5816,15248, 5824,10568, 5832,7568, 5840,11664,
   5848,15760, 5856,10696, 5864,8080, 5872,12176, 5880,16272, 5888,11328,
   5896,11344, 5904,8656, 5912,12752, 5920,11456, 5928,11472, 5936,9168,
   5944,13264, 5952,11584, 5960,11600, 5968,9680, 5976,13776, 5984,11712,
   5992,6096, 6000,10192, 6008,14288, 6016,11336, 6024,6608, 6032,10704,
   6040,14800, 6048,11464, 6056,7120, 6064,11216, 6072,15312, 6080,11592,
   6088,7632, 6096,11728, 6104,15824, 6112,11720, 6120,8144, 6128,12240,
   6136,16336, 6144,12288, 6152,12304, 6160,8216, 6168,12312, 6176,12416,
   6184,12432, 6192,8728, 6200,12824, 6208,12544, 6216,12560, 6224,9240,
   6232,13336, 6240,12672, 6248,12688, 6256,9752, 6264,13848, 6272,12296,
   6280,12312, 6288,10264, 6296,14360, 6304,12424, 6312,6680, 6320,10776,
   6328,14872, 6336,12552, 6344,7192, 6352,11288, 6360,15384, 6368,12680,
   6376,7704, 6384,11800, 6392,15896, 6400,13312, 6408,13328, 6416,8280,
   6424,12376, 6432,13440, 6440,13456, 6448,8792, 6456,12888, 6464,13568,
   6472,13584, 6480,9304, 6488,13400, 6496,13696, 6504,13712, 6512,9816,
   6520,13912, 6528,13320, 6536,13336, 6544,10328, 6552,14424, 6560,13448,
   6568,6744, 6576,10840, 6584,14936, 6592,13576, 6600,7256, 6608,11352,
   6616,15448, 6624,13704, 6632,7768, 6640,11864, 6648,15960, 6656,14336,
   6664,14352, 6672,8344, 6680,12440, 6688,14464, 6696,14480, 6704,8856,
   6712,12952, 6720,14592, 6728,14608, 6736,9368, 6744,13464, 6752,14720,
   6760,14736, 6768,9880, 6776,13976, 6784,14344, 6792,14360, 6800,10392,
   6808,14488, 6816,14472, 6824,14488, 6832,10904, 6840,15000, 6848,14600,
   6856,7320, 6864,11416, 6872,15512, 6880,14728, 6888,7832, 6896,11928,
   6904,16024, 6912,15360, 6920,15376, 6928,8408, 6936,12504, 6944,15488,
   6952,15504, 6960,8920, 6968,13016, 6976,15616, 6984,15632, 6992,9432,
   7000,13528, 7008,15744, 7016,15760, 7024,9944, 7032,14040, 7040,15368,
   7048,15384, 7056,10456, 7064,14552, 7072,15496, 7080,15512, 7088,10968,
   7096,15064, 7104,15624, 7112,7384, 7120,11480, 7128,15576, 7136,15752,
   7144,7896, 7152,11992, 7160,16088, 7168,12352, 7176,12368, 7184,8472,
   7192,12568, 7200,12480, 7208,12496, 7216,8984, 7224,13080, 7232,12608,
   7240,12624, 7248,9496, 7256,13592, 7264,12736, 7272,12752, 7280,10008,
   7288,14104, 7296,12360, 7304,12376, 7312,10520, 7320,14616, 7328,12488,
   7336,12504, 7344,11032, 7352,15128, 7360,12616, 7368,7448, 7376,11544,
   7384,15640, 7392,12744, 7400,7960, 7408,12056, 7416,16152, 7424,13376,
   7432,13392, 7440,8536, 7448,12632, 7456,13504, 7464,13520, 7472,9048,
   7480,13144, 7488,13632, 7496,13648, 7504,9560, 7512,13656, 7520,13760,
   7528,13776, 7536,10072, 7544,14168, 7552,13384, 7560,13400, 7568,10584,
   7576,14680, 7584,13512, 7592,13528, 7600,11096, 7608,15192, 7616,13640,
   7624,13656, 7632,11608, 7640,15704, 7648,13768, 7656,8024, 7664,12120,
   7672,16216, 7680,14400, 7688,14416, 7696,8600, 7704,12696, 7712,14528,
   7720,14544, 7728,9112, 7736,13208, 7744,14656, 7752,14672, 7760,9624,
   7768,13720, 7776,14784, 7784,14800, 7792,10136, 7800,14232, 7808,14408,
   7816,14424, 7824,10648, 7832,14744, 7840,14536, 7848,14552, 7856,11160,
   7864,15256, 7872,14664, 7880,14680, 7888,11672, 7896,15768, 7904,14792,
   7912,8088, 7920,12184, 7928,16280, 7936,15424, 7944,15440, 7952,8664,
   7960,12760, 7968,15552, 7976,15568, 7984,9176, 7992,13272, 8000,15680,
   8008,15696, 8016,9688, 8024,13784, 8032,15808, 8040,15824, 8048,10200,
   8056,14296, 8064,15432, 8072,15448, 8080,10712, 8088,14808, 8096,15560,
   8104,15576, 8112,11224, 8120,15320, 8128,15688, 8136,15704, 8144,11736,
   8152,15832, 8160,15816, 8168,15832, 8176,12248, 8184,16344, 8200,8320,
   8208,8224, 8216,12320, 8232,10368, 8240,8736, 8248,12832, 8256,8448,
   8264,8384, 8272,9248, 8280,13344, 8288,9232, 8296,10432, 8304,9760,
   8312,13856, 8328,12416, 8336,10272, 8344,14368, 8352,12296, 8360,14464,
   8368,10784, 8376,14880, 8384,8456, 8392,12480, 8400,11296, 8408,15392,
   8416,12552, 8424,14528, 8432,11808, 8440,15904, 8448,9216, 8456,8576,
   8464,9232, 8472,12384, 8480,9248, 8488,10624, 8496,8800, 8504,12896,
   8512,9472, 8520,8640, 8528,9312, 8536,13408, 8544,9296, 8552,10688,
   8560,9824, 8568,13920, 8576,9224, 8584,12672, 8592,10336, 8600,14432,
   8608,13320, 8616,14720, 8624,10848, 8632,14944, 8640,9480, 8648,12736,
   8656,11360, 8664,15456, 8672,13576, 8680,14784, 8688,11872, 8696,15968,
   8704,12288, 8712,12416, 8720,12296, 8728,12448, 8736,12304, 8744,10376,
   8752,8864, 8760,12960, 8768,12352, 8776,12480, 8784,9376, 8792,13472,
   8800,12368, 8808,10440, 8816,9888, 8824,13984, 8832,12320, 8840,12424,
   8848,10400, 8856,14496, 8864,12312, 8872,14472, 8880,10912, 8888,15008,
   8896,12384, 8904,12488, 8912,11424, 8920,15520, 8928,12568, 8936,14536,
   8944,11936, 8952,16032, 8960,12544, 8968,12672, 8976,12552, 8984,12512,
   8992,12560, 9000,10632, 9008,12568, 9016,13024, 9024,12608, 9032,12736,
   9040,9440, 9048,13536, 9056,12624, 9064,10696, 9072,9952, 9080,14048,
   9088,9240, 9096,12680, 9104,10464, 9112,14560, 9120,13336, 9128,14728,
   9136,10976, 9144,15072, 9152,9496, 9160,12744, 9168,11488, 9176,15584,
   9184,13592, 9192,14792, 9200,12000, 9208,16096, 9224,9344, 9232,9248,
   9240,12576, 9256,11392, 9264,12560, 9272,13088, 9280,9472, 9288,9408,
   9296,9504, 9304,13600, 9312,9488, 9320,11456, 9328,10016, 9336,14112,
   9352,13440, 9360,10528, 9368,14624, 9376,12360, 9384,15488, 9392,11040,
   9400,15136, 9408,9480, 9416,13504, 9424,11552, 9432,15648, 9440,12616,
   9448,15552, 9456,12064, 9464,16160, 9480,9600, 9488,9504, 9496,12640,
   9512,11648, 9520,12624, 9528,13152, 9544,9664, 9552,9568, 9560,13664,
   9576,11712, 9584,10080, 9592,14176, 9608,13696, 9616,10592, 9624,14688,
   9632,13384, 9640,15744, 9648,11104, 9656,15200, 9672,13760, 9680,11616,
   9688,15712, 9696,13640, 9704,15808, 9712,12128, 9720,16224, 9728,13312,
   9736,13440, 9744,13320, 9752,12704, 9760,13328, 9768,11400, 9776,13336,
   9784,13216, 9792,13376, 9800,13504, 9808,13384, 9816,13728, 9824,13392,
   9832,11464, 9840,10144, 9848,14240, 9856,13344, 9864,13448, 9872,10656,
   9880,14752, 9888,12376, 9896,15496, 9904,11168, 9912,15264, 9920,13408,
   9928,13512, 9936,11680, 9944,15776, 9952,12632, 9960,15560, 9968,12192,
   9976,16288, 9984,13568, 9992,13696, 10000,13576, 10008,12768, 10016,13584,
   10024,11656, 10032,13592, 10040,13280, 10048,13632, 10056,13760,
   10064,13640, 10072,13792, 10080,13648, 10088,11720, 10096,10208,
   10104,14304, 10112,13600, 10120,13704, 10128,10720, 10136,14816,
   10144,13400, 10152,15752, 10160,11232, 10168,15328, 10176,13664,
   10184,13768, 10192,11744, 10200,15840, 10208,13656, 10216,15816,
   10224,12256, 10232,16352, 10248,10272, 10256,10368, 10264,12328,
   10280,10384, 10288,10376, 10296,12840, 10304,11264, 10312,11296,
   10320,11392, 10328,13352, 10336,11272, 10344,10448, 10352,11400,
   10360,13864, 10376,12432, 10392,14376, 10400,12328, 10408,14480,
   10416,10792, 10424,14888, 10432,11280, 10440,12496, 10448,11304,
   10456,15400, 10464,11288, 10472,14544, 10480,11816, 10488,15912,
   10496,11264, 10504,11272, 10512,11280, 10520,12392, 10528,11296,
   10536,10640, 10544,12496, 10552,12904, 10560,11328, 10568,11360,
   10576,11456, 10584,13416, 10592,11336, 10600,10704, 10608,11464,
   10616,13928, 10624,11392, 10632,12688, 10640,11304, 10648,14440,
   10656,13352, 10664,14736, 10672,10856, 10680,14952, 10688,11344,
   10696,12752, 10704,11368, 10712,15464, 10720,11352, 10728,14800,
   10736,11880, 10744,15976, 10752,14336, 10760,14368, 10768,14464,
   10776,12456, 10784,14344, 10792,14376, 10800,14472, 10808,12968,
   10816,15360, 10824,15392, 10832,15488, 10840,13480, 10848,15368,
   10856,15400, 10864,15496, 10872,13992, 10880,14352, 10888,12440,
   10896,14480, 10904,14504, 10912,14360, 10920,14488, 10928,14488,
   10936,15016, 10944,15376, 10952,12504, 10960,11432, 10968,15528,
   10976,15384, 10984,14552, 10992,11944, 11000,16040, 11008,14400,
   11016,14432, 11024,14528, 11032,12520, 11040,14408, 11048,14440,
   11056,14536, 11064,13032, 11072,15424, 11080,15456, 11088,15552,
   11096,13544, 11104,15432, 11112,15464, 11120,15560, 11128,14056,
   11136,14416, 11144,12696, 11152,14544, 11160,14568, 11168,14424,
   11176,14744, 11184,14552, 11192,15080, 11200,15440, 11208,12760,
   11216,11496, 11224,15592, 11232,15448, 11240,14808, 11248,12008,
   11256,16104, 11272,11296, 11280,11392, 11288,12584, 11304,11408,
   11312,12688, 11320,13096, 11328,11520, 11336,11552, 11344,11648,
   11352,13608, 11360,11528, 11368,11472, 11376,11656, 11384,14120,
   11400,13456, 11416,14632, 11424,12392, 11432,15504, 11440,14440,
   11448,15144, 11456,11536, 11464,13520, 11472,11560, 11480,15656,
   11488,11544, 11496,15568, 11504,12072, 11512,16168, 11528,11552,
   11536,11648, 11544,12648, 11560,11664, 11568,12752, 11576,13160,
   11592,11616, 11600,11712, 11608,13672, 11624,11728, 11632,11720,
   11640,14184, 11656,13712, 11672,14696, 11680,13416, 11688,15760,
   11696,15464, 11704,15208, 11720,13776, 11736,15720, 11744,13672,
   11752,15824, 11760,12136, 11768,16232, 11776,14592, 11784,14624,
   11792,14720, 11800,12712, 11808,14600, 11816,14632, 11824,14728,
   11832,13224, 11840,15616, 11848,15648, 11856,15744, 11864,13736,
   11872,15624, 11880,15656, 11888,15752, 11896,14248, 11904,14608,
   11912,13464, 11920,14736, 11928,14760, 11936,14616, 11944,15512,
   11952,14744, 11960,15272, 11968,15632, 11976,13528, 11984,15760,
   11992,15784, 12000,15640, 12008,15576, 12016,12200, 12024,16296,
   12032,14656, 12040,14688, 12048,14784, 12056,12776, 12064,14664,
   12072,14696, 12080,14792, 12088,13288, 12096,15680, 12104,15712,
   12112,15808, 12120,13800, 12128,15688, 12136,15720, 12144,15816,
   12152,14312, 12160,14672, 12168,13720, 12176,14800, 12184,14824,
   12192,14680, 12200,15768, 12208,14808, 12216,15336, 12224,15696,
   12232,13784, 12240,15824, 12248,15848, 12256,15704, 12264,15832,
   12272,15832, 12280,16360, 12312,12336, 12344,12848, 12352,12544,
   12360,12552, 12368,12560, 12376,13360, 12384,12576, 12392,12584,
   12400,13336, 12408,13872, 12424,12448, 12440,14384, 12456,14496,
   12464,14472, 12472,14896, 12480,12672, 12488,12512, 12496,12688,
   12504,15408, 12512,12680, 12520,14560, 12528,14728, 12536,15920,
   12544,13312, 12552,13320, 12560,13328, 12568,13336, 12576,13344,
   12584,13352, 12592,13360, 12600,12912, 12608,13568, 12616,13576,
   12624,13584, 12632,13424, 12640,13600, 12648,13608, 12656,13400,
   12664,13936, 12672,13440, 12680,12704, 12688,13456, 12696,14448,
   12704,13448, 12712,14752, 12720,15496, 12728,14960, 12736,13696,
   12744,12768, 12752,13712, 12760,15472, 12768,13704, 12776,14816,
   12784,15752, 12792,15984, 12800,14336, 12808,14464, 12816,14344,
   12824,14472, 12832,14352, 12840,14480, 12848,14360, 12856,12976,
   12864,14400, 12872,14528, 12880,14408, 12888,13488, 12896,14416,
   12904,14544, 12912,14424, 12920,14000, 12928,14368, 12936,14496,
   12944,14376, 12952,14512, 12960,14384, 12968,14504, 12976,14488,
   12984,15024, 12992,14432, 13000,14560, 13008,14440, 13016,15536,
   13024,14448, 13032,14568, 13040,14744, 13048,16048, 13056,14592,
   13064,14720, 13072,14600, 13080,14728, 13088,14608, 13096,14736,
   13104,14616, 13112,14744, 13120,14656, 13128,14784, 13136,14664,
   13144,13552, 13152,14672, 13160,14800, 13168,14680, 13176,14064,
   13184,14624, 13192,14752, 13200,14632, 13208,14576, 13216,13464,
   13224,14760, 13232,15512, 13240,15088, 13248,14688, 13256,14816,
   13264,14696, 13272,15600, 13280,13720, 13288,14824, 13296,15768,
   13304,16112, 13336,13360, 13368,14616, 13376,13568, 13384,13576,
   13392,13584, 13400,13616, 13408,13600, 13416,13608, 13424,13592,
   13432,14128, 13448,13472, 13464,14640, 13480,15520, 13488,14536,
   13496,15152, 13504,13696, 13512,13536, 13520,13712, 13528,15664,
   13536,13704, 13544,15584, 13552,14792, 13560,16176, 13592,13616,
   13624,14680, 13656,13680, 13688,14192, 13704,13728, 13720,14704,
   13736,15776, 13744,15560, 13752,15216, 13768,13792, 13784,15728,
   13800,15840, 13808,15816, 13816,16240, 13824,15360, 13832,15488,
   13840,15368, 13848,15496, 13856,15376, 13864,15504, 13872,15384,
   13880,15512, 13888,15424, 13896,15552, 13904,15432, 13912,15560,
   13920,15440, 13928,15568, 13936,15448, 13944,14256, 13952,15392,
   13960,15520, 13968,15400, 13976,14768, 13984,15408, 13992,15528,
   14000,14552, 14008,15280, 14016,15456, 14024,15584, 14032,15464,
   14040,15792, 14048,15472, 14056,15592, 14064,14808, 14072,16304,
   14080,15616, 14088,15744, 14096,15624, 14104,15752, 14112,15632,
   14120,15760, 14128,15640, 14136,15768, 14144,15680, 14152,15808,
   14160,15688, 14168,15816, 14176,15696, 14184,15824, 14192,15704,
   14200,14320, 14208,15648, 14216,15776, 14224,15656, 14232,14832,
   14240,15664, 14248,15784, 14256,15576, 14264,15344, 14272,15712,
   14280,15840, 14288,15720, 14296,15856, 14304,15728, 14312,15848,
   14320,15832, 14328,16368, 14392,14488, 14400,14592, 14408,14600,
   14416,14608, 14424,14616, 14432,14624, 14440,14632, 14448,14640,
   14456,15512, 14504,14512, 14520,14904, 14528,14720, 14536,14728,
   14544,14736, 14552,15416, 14560,14752, 14568,14576, 14584,15928,
   14576,14760, 14592,15360, 14600,15368, 14608,15376, 14616,15384,
   14624,15392, 14632,15400, 14640,15408, 14648,15416, 14656,15616,
   14664,15624, 14672,15632, 14680,15640, 14688,15648, 14696,15656,
   14704,15664, 14712,15576, 14720,15488, 14728,15496, 14736,15504,
   14744,15512, 14752,15520, 14760,14768, 14776,14968, 14768,15528,
   14784,15744, 14792,15752, 14800,15760, 14808,15480, 14816,15776,
   14824,14832, 14840,15992, 14832,15784, 14856,14864, 14864,14880,
   14872,14896, 14880,14976, 14888,14992, 14896,15008, 14904,15024,
   14912,15104, 14920,15120, 14928,15136, 14936,15152, 14944,15232,
   14952,15248, 14960,15264, 14968,15280, 14984,15008, 15000,15024,
   15016,15024, 15040,15112, 15048,15128, 15056,15144, 15064,15544,
   15072,15240, 15080,15256, 15088,15272, 15096,16056, 15104,15872,
   15112,15888, 15120,15904, 15128,15920, 15136,16000, 15144,16016,
   15152,16032, 15160,16048, 15168,16128, 15176,16144, 15184,16160,
   15192,16176, 15200,16256, 15208,16272, 15216,16288, 15224,16304,
   15232,15880, 15240,15896, 15248,15912, 15256,15928, 15264,16008,
   15272,16024, 15280,16040, 15288,16056, 15296,16136, 15304,16152,
   15312,16168, 15320,15608, 15328,16264, 15336,16280, 15344,16296,
   15352,16120, 15416,15512, 15424,15616, 15432,15624, 15440,15632,
   15448,15640, 15456,15648, 15464,15656, 15472,15664, 15480,15768,
   15528,15536, 15544,16048, 15552,15744, 15560,15752, 15568,15760,
   15576,15672, 15584,15776, 15592,15600, 15600,15784, 15608,16184,
   15672,15768, 15736,15832, 15784,15792, 15800,16304, 15848,15856,
   15880,16000, 15864,16248, 15888,16000, 15896,16008, 15904,16000,
   15912,16016, 15920,16008, 15928,16024, 15936,16128, 15944,16160,
   15952,16256, 15960,16288, 15968,16136, 15976,16168, 15984,16264,
   15992,16296, 16008,16032, 16024,16040, 16064,16144, 16040,16048,
   16072,16176, 16080,16272, 16088,16304, 16096,16152, 16104,16184,
   16112,16280, 16136,16256, 16120,16312, 16144,16256, 16152,16264,
   16160,16256, 16168,16272, 16176,16264, 16184,16280, 16200,16208,
   16208,16224, 16216,16240, 16224,16320, 16232,16336, 16240,16352,
   16248,16368, 16264,16288, 16280,16296, 16296,16304, 16344,16368,
   16328,16352, 16360,16368
};
#endif // INCLUDE_FFT_TABLES_2048


#if defined(INCLUDE_FFT_TABLES_4096)
__attribute__ ((section(".rom_data")))
const uint16_t riscvBitRevIndexTable4096[RISCVBITREVINDEXTABLE_4096_TABLE_LENGTH] =
{
   /* radix 8, size 4032 */
   8,4096, 16,8192, 24,12288, 32,16384, 40,20480, 48,24576, 56,28672, 64,512,
   72,4608, 80,8704, 88,12800, 96,16896, 104,20992, 112,25088, 120,29184,
   128,1024, 136,5120, 144,9216, 152,13312, 160,17408, 168,21504, 176,25600,
   184,29696, 192,1536, 200,5632, 208,9728, 216,13824, 224,17920, 232,22016,
   240,26112, 248,30208, 256,2048, 264,6144, 272,10240, 280,14336, 288,18432,
   296,22528, 304,26624, 312,30720, 320,2560, 328,6656, 336,10752, 344,14848,
   352,18944, 360,23040, 368,27136, 376,31232, 384,3072, 392,7168, 400,11264,
   408,15360, 416,19456, 424,23552, 432,27648, 440,31744, 448,3584, 456,7680,
   464,11776, 472,15872, 480,19968, 488,24064, 496,28160, 504,32256, 520,4160,
   528,8256, 536,12352, 544,16448, 552,20544, 560,24640, 568,28736, 584,4672,
   592,8768, 600,12864, 608,16960, 616,21056, 624,25152, 632,29248, 640,1088,
   648,5184, 656,9280, 664,13376, 672,17472, 680,21568, 688,25664, 696,29760,
   704,1600, 712,5696, 720,9792, 728,13888, 736,17984, 744,22080, 752,26176,
   760,30272, 768,2112, 776,6208, 784,10304, 792,14400, 800,18496, 808,22592,
   816,26688, 824,30784, 832,2624, 840,6720, 848,10816, 856,14912, 864,19008,
   872,23104, 880,27200, 888,31296, 896,3136, 904,7232, 912,11328, 920,15424,
   928,19520, 936,23616, 944,27712, 952,31808, 960,3648, 968,7744, 976,11840,
   984,15936, 992,20032, 1000,24128, 1008,28224, 1016,32320, 1032,4224,
   1040,8320, 1048,12416, 1056,16512, 1064,20608, 1072,24704, 1080,28800,
   1096,4736, 1104,8832, 1112,12928, 1120,17024, 1128,21120, 1136,25216,
   1144,29312, 1160,5248, 1168,9344, 1176,13440, 1184,17536, 1192,21632,
   1200,25728, 1208,29824, 1216,1664, 1224,5760, 1232,9856, 1240,13952,
   1248,18048, 1256,22144, 1264,26240, 1272,30336, 1280,2176, 1288,6272,
   1296,10368, 1304,14464, 1312,18560, 1320,22656, 1328,26752, 1336,30848,
   1344,2688, 1352,6784, 1360,10880, 1368,14976, 1376,19072, 1384,23168,
   1392,27264, 1400,31360, 1408,3200, 1416,7296, 1424,11392, 1432,15488,
   1440,19584, 1448,23680, 1456,27776, 1464,31872, 1472,3712, 1480,7808,
   1488,11904, 1496,16000, 1504,20096, 1512,24192, 1520,28288, 1528,32384,
   1544,4288, 1552,8384, 1560,12480, 1568,16576, 1576,20672, 1584,24768,
   1592,28864, 1608,4800, 1616,8896, 1624,12992, 1632,17088, 1640,21184,
   1648,25280, 1656,29376, 1672,5312, 1680,9408, 1688,13504, 1696,17600,
   1704,21696, 1712,25792, 1720,29888, 1736,5824, 1744,9920, 1752,14016,
   1760,18112, 1768,22208, 1776,26304, 1784,30400, 1792,2240, 1800,6336,
   1808,10432, 1816,14528, 1824,18624, 1832,22720, 1840,26816, 1848,30912,
   1856,2752, 1864,6848, 1872,10944, 1880,15040, 1888,19136, 1896,23232,
   1904,27328, 1912,31424, 1920,3264, 1928,7360, 1936,11456, 1944,15552,
   1952,19648, 1960,23744, 1968,27840, 1976,31936, 1984,3776, 1992,7872,
   2000,11968, 2008,16064, 2016,20160, 2024,24256, 2032,28352, 2040,32448,
   2056,4352, 2064,8448, 2072,12544, 2080,16640, 2088,20736, 2096,24832,
   2104,28928, 2120,4864, 2128,8960, 2136,13056, 2144,17152, 2152,21248,
   2160,25344, 2168,29440, 2184,5376, 2192,9472, 2200,13568, 2208,17664,
   2216,21760, 2224,25856, 2232,29952, 2248,5888, 2256,9984, 2264,14080,
   2272,18176, 2280,22272, 2288,26368, 2296,30464, 2312,6400, 2320,10496,
   2328,14592, 2336,18688, 2344,22784, 2352,26880, 2360,30976, 2368,2816,
   2376,6912, 2384,11008, 2392,15104, 2400,19200, 2408,23296, 2416,27392,
   2424,31488, 2432,3328, 2440,7424, 2448,11520, 2456,15616, 2464,19712,
   2472,23808, 2480,27904, 2488,32000, 2496,3840, 2504,7936, 2512,12032,
   2520,16128, 2528,20224, 2536,24320, 2544,28416, 2552,32512, 2568,4416,
   2576,8512, 2584,12608, 2592,16704, 2600,20800, 2608,24896, 2616,28992,
   2632,4928, 2640,9024, 2648,13120, 2656,17216, 2664,21312, 2672,25408,
   2680,29504, 2696,5440, 2704,9536, 2712,13632, 2720,17728, 2728,21824,
   2736,25920, 2744,30016, 2760,5952, 2768,10048, 2776,14144, 2784,18240,
   2792,22336, 2800,26432, 2808,30528, 2824,6464, 2832,10560, 2840,14656,
   2848,18752, 2856,22848, 2864,26944, 2872,31040, 2888,6976, 2896,11072,
   2904,15168, 2912,19264, 2920,23360, 2928,27456, 2936,31552, 2944,3392,
   2952,7488, 2960,11584, 2968,15680, 2976,19776, 2984,23872, 2992,27968,
   3000,32064, 3008,3904, 3016,8000, 3024,12096, 3032,16192, 3040,20288,
   3048,24384, 3056,28480, 3064,32576, 3080,4480, 3088,8576, 3096,12672,
   3104,16768, 3112,20864, 3120,24960, 3128,29056, 3144,4992, 3152,9088,
   3160,13184, 3168,17280, 3176,21376, 3184,25472, 3192,29568, 3208,5504,
   3216,9600, 3224,13696, 3232,17792, 3240,21888, 3248,25984, 3256,30080,
   3272,6016, 3280,10112, 3288,14208, 3296,18304, 3304,22400, 3312,26496,
   3320,30592, 3336,6528, 3344,10624, 3352,14720, 3360,18816, 3368,22912,
   3376,27008, 3384,31104, 3400,7040, 3408,11136, 3416,15232, 3424,19328,
   3432,23424, 3440,27520, 3448,31616, 3464,7552, 3472,11648, 3480,15744,
   3488,19840, 3496,23936, 3504,28032, 3512,32128, 3520,3968, 3528,8064,
   3536,12160, 3544,16256, 3552,20352, 3560,24448, 3568,28544, 3576,32640,
   3592,4544, 3600,8640, 3608,12736, 3616,16832, 3624,20928, 3632,25024,
   3640,29120, 3656,5056, 3664,9152, 3672,13248, 3680,17344, 3688,21440,
   3696,25536, 3704,29632, 3720,5568, 3728,9664, 3736,13760, 3744,17856,
   3752,21952, 3760,26048, 3768,30144, 3784,6080, 3792,10176, 3800,14272,
   3808,18368, 3816,22464, 3824,26560, 3832,30656, 3848,6592, 3856,10688,
   3864,14784, 3872,18880, 3880,22976, 3888,27072, 3896,31168, 3912,7104,
   3920,11200, 3928,15296, 3936,19392, 3944,23488, 3952,27584, 3960,31680,
   3976,7616, 3984,11712, 3992,15808, 4000,19904, 4008,24000, 4016,28096,
   4024,32192, 4040,8128, 4048,12224, 4056,16320, 4064,20416, 4072,24512,
   4080,28608, 4088,32704, 4112,8200, 4120,12296, 4128,16392, 4136,20488,
   4144,24584, 4152,28680, 4168,4616, 4176,8712, 4184,12808, 4192,16904,
   4200,21000, 4208,25096, 4216,29192, 4232,5128, 4240,9224, 4248,13320,
   4256,17416, 4264,21512, 4272,25608, 4280,29704, 4296,5640, 4304,9736,
   4312,13832, 4320,17928, 4328,22024, 4336,26120, 4344,30216, 4360,6152,
   4368,10248, 4376,14344, 4384,18440, 4392,22536, 4400,26632, 4408,30728,
   4424,6664, 4432,10760, 4440,14856, 4448,18952, 4456,23048, 4464,27144,
   4472,31240, 4488,7176, 4496,11272, 4504,15368, 4512,19464, 4520,23560,
   4528,27656, 4536,31752, 4552,7688, 4560,11784, 4568,15880, 4576,19976,
   4584,24072, 4592,28168, 4600,32264, 4624,8264, 4632,12360, 4640,16456,
   4648,20552, 4656,24648, 4664,28744, 4688,8776, 4696,12872, 4704,16968,
   4712,21064, 4720,25160, 4728,29256, 4744,5192, 4752,9288, 4760,13384,
   4768,17480, 4776,21576, 4784,25672, 4792,29768, 4808,5704, 4816,9800,
   4824,13896, 4832,17992, 4840,22088, 4848,26184, 4856,30280, 4872,6216,
   4880,10312, 4888,14408, 4896,18504, 4904,22600, 4912,26696, 4920,30792,
   4936,6728, 4944,10824, 4952,14920, 4960,19016, 4968,23112, 4976,27208,
   4984,31304, 5000,7240, 5008,11336, 5016,15432, 5024,19528, 5032,23624,
   5040,27720, 5048,31816, 5064,7752, 5072,11848, 5080,15944, 5088,20040,
   5096,24136, 5104,28232, 5112,32328, 5136,8328, 5144,12424, 5152,16520,
   5160,20616, 5168,24712, 5176,28808, 5200,8840, 5208,12936, 5216,17032,
   5224,21128, 5232,25224, 5240,29320, 5264,9352, 5272,13448, 5280,17544,
   5288,21640, 5296,25736, 5304,29832, 5320,5768, 5328,9864, 5336,13960,
   5344,18056, 5352,22152, 5360,26248, 5368,30344, 5384,6280, 5392,10376,
   5400,14472, 5408,18568, 5416,22664, 5424,26760, 5432,30856, 5448,6792,
   5456,10888, 5464,14984, 5472,19080, 5480,23176, 5488,27272, 5496,31368,
   5512,7304, 5520,11400, 5528,15496, 5536,19592, 5544,23688, 5552,27784,
   5560,31880, 5576,7816, 5584,11912, 5592,16008, 5600,20104, 5608,24200,
   5616,28296, 5624,32392, 5648,8392, 5656,12488, 5664,16584, 5672,20680,
   5680,24776, 5688,28872, 5712,8904, 5720,13000, 5728,17096, 5736,21192,
   5744,25288, 5752,29384, 5776,9416, 5784,13512, 5792,17608, 5800,21704,
   5808,25800, 5816,29896, 5840,9928, 5848,14024, 5856,18120, 5864,22216,
   5872,26312, 5880,30408, 5896,6344, 5904,10440, 5912,14536, 5920,18632,
   5928,22728, 5936,26824, 5944,30920, 5960,6856, 5968,10952, 5976,15048,
   5984,19144, 5992,23240, 6000,27336, 6008,31432, 6024,7368, 6032,11464,
   6040,15560, 6048,19656, 6056,23752, 6064,27848, 6072,31944, 6088,7880,
   6096,11976, 6104,16072, 6112,20168, 6120,24264, 6128,28360, 6136,32456,
   6160,8456, 6168,12552, 6176,16648, 6184,20744, 6192,24840, 6200,28936,
   6224,8968, 6232,13064, 6240,17160, 6248,21256, 6256,25352, 6264,29448,
   6288,9480, 6296,13576, 6304,17672, 6312,21768, 6320,25864, 6328,29960,
   6352,9992, 6360,14088, 6368,18184, 6376,22280, 6384,26376, 6392,30472,
   6416,10504, 6424,14600, 6432,18696, 6440,22792, 6448,26888, 6456,30984,
   6472,6920, 6480,11016, 6488,15112, 6496,19208, 6504,23304, 6512,27400,
   6520,31496, 6536,7432, 6544,11528, 6552,15624, 6560,19720, 6568,23816,
   6576,27912, 6584,32008, 6600,7944, 6608,12040, 6616,16136, 6624,20232,
   6632,24328, 6640,28424, 6648,32520, 6672,8520, 6680,12616, 6688,16712,
   6696,20808, 6704,24904, 6712,29000, 6736,9032, 6744,13128, 6752,17224,
   6760,21320, 6768,25416, 6776,29512, 6800,9544, 6808,13640, 6816,17736,
   6824,21832, 6832,25928, 6840,30024, 6864,10056, 6872,14152, 6880,18248,
   6888,22344, 6896,26440, 6904,30536, 6928,10568, 6936,14664, 6944,18760,
   6952,22856, 6960,26952, 6968,31048, 6992,11080, 7000,15176, 7008,19272,
   7016,23368, 7024,27464, 7032,31560, 7048,7496, 7056,11592, 7064,15688,
   7072,19784, 7080,23880, 7088,27976, 7096,32072, 7112,8008, 7120,12104,
   7128,16200, 7136,20296, 7144,24392, 7152,28488, 7160,32584, 7184,8584,
   7192,12680, 7200,16776, 7208,20872, 7216,24968, 7224,29064, 7248,9096,
   7256,13192, 7264,17288, 7272,21384, 7280,25480, 7288,29576, 7312,9608,
   7320,13704, 7328,17800, 7336,21896, 7344,25992, 7352,30088, 7376,10120,
   7384,14216, 7392,18312, 7400,22408, 7408,26504, 7416,30600, 7440,10632,
   7448,14728, 7456,18824, 7464,22920, 7472,27016, 7480,31112, 7504,11144,
   7512,15240, 7520,19336, 7528,23432, 7536,27528, 7544,31624, 7568,11656,
   7576,15752, 7584,19848, 7592,23944, 7600,28040, 7608,32136, 7624,8072,
   7632,12168, 7640,16264, 7648,20360, 7656,24456, 7664,28552, 7672,32648,
   7696,8648, 7704,12744, 7712,16840, 7720,20936, 7728,25032, 7736,29128,
   7760,9160, 7768,13256, 7776,17352, 7784,21448, 7792,25544, 7800,29640,
   7824,9672, 7832,13768, 7840,17864, 7848,21960, 7856,26056, 7864,30152,
   7888,10184, 7896,14280, 7904,18376, 7912,22472, 7920,26568, 7928,30664,
   7952,10696, 7960,14792, 7968,18888, 7976,22984, 7984,27080, 7992,31176,
   8016,11208, 8024,15304, 8032,19400, 8040,23496, 8048,27592, 8056,31688,
   8080,11720, 8088,15816, 8096,19912, 8104,24008, 8112,28104, 8120,32200,
   8144,12232, 8152,16328, 8160,20424, 8168,24520, 8176,28616, 8184,32712,
   8216,12304, 8224,16400, 8232,20496, 8240,24592, 8248,28688, 8272,8720,
   8280,12816, 8288,16912, 8296,21008, 8304,25104, 8312,29200, 8336,9232,
   8344,13328, 8352,17424, 8360,21520, 8368,25616, 8376,29712, 8400,9744,
   8408,13840, 8416,17936, 8424,22032, 8432,26128, 8440,30224, 8464,10256,
   8472,14352, 8480,18448, 8488,22544, 8496,26640, 8504,30736, 8528,10768,
   8536,14864, 8544,18960, 8552,23056, 8560,27152, 8568,31248, 8592,11280,
   8600,15376, 8608,19472, 8616,23568, 8624,27664, 8632,31760, 8656,11792,
   8664,15888, 8672,19984, 8680,24080, 8688,28176, 8696,32272, 8728,12368,
   8736,16464, 8744,20560, 8752,24656, 8760,28752, 8792,12880, 8800,16976,
   8808,21072, 8816,25168, 8824,29264, 8848,9296, 8856,13392, 8864,17488,
   8872,21584, 8880,25680, 8888,29776, 8912,9808, 8920,13904, 8928,18000,
   8936,22096, 8944,26192, 8952,30288, 8976,10320, 8984,14416, 8992,18512,
   9000,22608, 9008,26704, 9016,30800, 9040,10832, 9048,14928, 9056,19024,
   9064,23120, 9072,27216, 9080,31312, 9104,11344, 9112,15440, 9120,19536,
   9128,23632, 9136,27728, 9144,31824, 9168,11856, 9176,15952, 9184,20048,
   9192,24144, 9200,28240, 9208,32336, 9240,12432, 9248,16528, 9256,20624,
   9264,24720, 9272,28816, 9304,12944, 9312,17040, 9320,21136, 9328,25232,
   9336,29328, 9368,13456, 9376,17552, 9384,21648, 9392,25744, 9400,29840,
   9424,9872, 9432,13968, 9440,18064, 9448,22160, 9456,26256, 9464,30352,
   9488,10384, 9496,14480, 9504,18576, 9512,22672, 9520,26768, 9528,30864,
   9552,10896, 9560,14992, 9568,19088, 9576,23184, 9584,27280, 9592,31376,
   9616,11408, 9624,15504, 9632,19600, 9640,23696, 9648,27792, 9656,31888,
   9680,11920, 9688,16016, 9696,20112, 9704,24208, 9712,28304, 9720,32400,
   9752,12496, 9760,16592, 9768,20688, 9776,24784, 9784,28880, 9816,13008,
   9824,17104, 9832,21200, 9840,25296, 9848,29392, 9880,13520, 9888,17616,
   9896,21712, 9904,25808, 9912,29904, 9944,14032, 9952,18128, 9960,22224,
   9968,26320, 9976,30416, 10000,10448, 10008,14544, 10016,18640, 10024,22736,
   10032,26832, 10040,30928, 10064,10960, 10072,15056, 10080,19152,
   10088,23248, 10096,27344, 10104,31440, 10128,11472, 10136,15568,
   10144,19664, 10152,23760, 10160,27856, 10168,31952, 10192,11984,
   10200,16080, 10208,20176, 10216,24272, 10224,28368, 10232,32464,
   10264,12560, 10272,16656, 10280,20752, 10288,24848, 10296,28944,
   10328,13072, 10336,17168, 10344,21264, 10352,25360, 10360,29456,
   10392,13584, 10400,17680, 10408,21776, 10416,25872, 10424,29968,
   10456,14096, 10464,18192, 10472,22288, 10480,26384, 10488,30480,
   10520,14608, 10528,18704, 10536,22800, 10544,26896, 10552,30992,
   10576,11024, 10584,15120, 10592,19216, 10600,23312, 10608,27408,
   10616,31504, 10640,11536, 10648,15632, 10656,19728, 10664,23824,
   10672,27920, 10680,32016, 10704,12048, 10712,16144, 10720,20240,
   10728,24336, 10736,28432, 10744,32528, 10776,12624, 10784,16720,
   10792,20816, 10800,24912, 10808,29008, 10840,13136, 10848,17232,
   10856,21328, 10864,25424, 10872,29520, 10904,13648, 10912,17744,
   10920,21840, 10928,25936, 10936,30032, 10968,14160, 10976,18256,
   10984,22352, 10992,26448, 11000,30544, 11032,14672, 11040,18768,
   11048,22864, 11056,26960, 11064,31056, 11096,15184, 11104,19280,
   11112,23376, 11120,27472, 11128,31568, 11152,11600, 11160,15696,
   11168,19792, 11176,23888, 11184,27984, 11192,32080, 11216,12112,
   11224,16208, 11232,20304, 11240,24400, 11248,28496, 11256,32592,
   11288,12688, 11296,16784, 11304,20880, 11312,24976, 11320,29072,
   11352,13200, 11360,17296, 11368,21392, 11376,25488, 11384,29584,
   11416,13712, 11424,17808, 11432,21904, 11440,26000, 11448,30096,
   11480,14224, 11488,18320, 11496,22416, 11504,26512, 11512,30608,
   11544,14736, 11552,18832, 11560,22928, 11568,27024, 11576,31120,
   11608,15248, 11616,19344, 11624,23440, 11632,27536, 11640,31632,
   11672,15760, 11680,19856, 11688,23952, 11696,28048, 11704,32144,
   11728,12176, 11736,16272, 11744,20368, 11752,24464, 11760,28560,
   11768,32656, 11800,12752, 11808,16848, 11816,20944, 11824,25040,
   11832,29136, 11864,13264, 11872,17360, 11880,21456, 11888,25552,
   11896,29648, 11928,13776, 11936,17872, 11944,21968, 11952,26064,
   11960,30160, 11992,14288, 12000,18384, 12008,22480, 12016,26576,
   12024,30672, 12056,14800, 12064,18896, 12072,22992, 12080,27088,
   12088,31184, 12120,15312, 12128,19408, 12136,23504, 12144,27600,
   12152,31696, 12184,15824, 12192,19920, 12200,24016, 12208,28112,
   12216,32208, 12248,16336, 12256,20432, 12264,24528, 12272,28624,
   12280,32720, 12320,16408, 12328,20504, 12336,24600, 12344,28696,
   12376,12824, 12384,16920, 12392,21016, 12400,25112, 12408,29208,
   12440,13336, 12448,17432, 12456,21528, 12464,25624, 12472,29720,
   12504,13848, 12512,17944, 12520,22040, 12528,26136, 12536,30232,
   12568,14360, 12576,18456, 12584,22552, 12592,26648, 12600,30744,
   12632,14872, 12640,18968, 12648,23064, 12656,27160, 12664,31256,
   12696,15384, 12704,19480, 12712,23576, 12720,27672, 12728,31768,
   12760,15896, 12768,19992, 12776,24088, 12784,28184, 12792,32280,
   12832,16472, 12840,20568, 12848,24664, 12856,28760, 12896,16984,
   12904,21080, 12912,25176, 12920,29272, 12952,13400, 12960,17496,
   12968,21592, 12976,25688, 12984,29784, 13016,13912, 13024,18008,
   13032,22104, 13040,26200, 13048,30296, 13080,14424, 13088,18520,
   13096,22616, 13104,26712, 13112,30808, 13144,14936, 13152,19032,
   13160,23128, 13168,27224, 13176,31320, 13208,15448, 13216,19544,
   13224,23640, 13232,27736, 13240,31832, 13272,15960, 13280,20056,
   13288,24152, 13296,28248, 13304,32344, 13344,16536, 13352,20632,
   13360,24728, 13368,28824, 13408,17048, 13416,21144, 13424,25240,
   13432,29336, 13472,17560, 13480,21656, 13488,25752, 13496,29848,
   13528,13976, 13536,18072, 13544,22168, 13552,26264, 13560,30360,
   13592,14488, 13600,18584, 13608,22680, 13616,26776, 13624,30872,
   13656,15000, 13664,19096, 13672,23192, 13680,27288, 13688,31384,
   13720,15512, 13728,19608, 13736,23704, 13744,27800, 13752,31896,
   13784,16024, 13792,20120, 13800,24216, 13808,28312, 13816,32408,
   13856,16600, 13864,20696, 13872,24792, 13880,28888, 13920,17112,
   13928,21208, 13936,25304, 13944,29400, 13984,17624, 13992,21720,
   14000,25816, 14008,29912, 14048,18136, 14056,22232, 14064,26328,
   14072,30424, 14104,14552, 14112,18648, 14120,22744, 14128,26840,
   14136,30936, 14168,15064, 14176,19160, 14184,23256, 14192,27352,
   14200,31448, 14232,15576, 14240,19672, 14248,23768, 14256,27864,
   14264,31960, 14296,16088, 14304,20184, 14312,24280, 14320,28376,
   14328,32472, 14368,16664, 14376,20760, 14384,24856, 14392,28952,
   14432,17176, 14440,21272, 14448,25368, 14456,29464, 14496,17688,
   14504,21784, 14512,25880, 14520,29976, 14560,18200, 14568,22296,
   14576,26392, 14584,30488, 14624,18712, 14632,22808, 14640,26904,
   14648,31000, 14680,15128, 14688,19224, 14696,23320, 14704,27416,
   14712,31512, 14744,15640, 14752,19736, 14760,23832, 14768,27928,
   14776,32024, 14808,16152, 14816,20248, 14824,24344, 14832,28440,
   14840,32536, 14880,16728, 14888,20824, 14896,24920, 14904,29016,
   14944,17240, 14952,21336, 14960,25432, 14968,29528, 15008,17752,
   15016,21848, 15024,25944, 15032,30040, 15072,18264, 15080,22360,
   15088,26456, 15096,30552, 15136,18776, 15144,22872, 15152,26968,
   15160,31064, 15200,19288, 15208,23384, 15216,27480, 15224,31576,
   15256,15704, 15264,19800, 15272,23896, 15280,27992, 15288,32088,
   15320,16216, 15328,20312, 15336,24408, 15344,28504, 15352,32600,
   15392,16792, 15400,20888, 15408,24984, 15416,29080, 15456,17304,
   15464,21400, 15472,25496, 15480,29592, 15520,17816, 15528,21912,
   15536,26008, 15544,30104, 15584,18328, 15592,22424, 15600,26520,
   15608,30616, 15648,18840, 15656,22936, 15664,27032, 15672,31128,
   15712,19352, 15720,23448, 15728,27544, 15736,31640, 15776,19864,
   15784,23960, 15792,28056, 15800,32152, 15832,16280, 15840,20376,
   15848,24472, 15856,28568, 15864,32664, 15904,16856, 15912,20952,
   15920,25048, 15928,29144, 15968,17368, 15976,21464, 15984,25560,
   15992,29656, 16032,17880, 16040,21976, 16048,26072, 16056,30168,
   16096,18392, 16104,22488, 16112,26584, 16120,30680, 16160,18904,
   16168,23000, 16176,27096, 16184,31192, 16224,19416, 16232,23512,
   16240,27608, 16248,31704, 16288,19928, 16296,24024, 16304,28120,
   16312,32216, 16352,20440, 16360,24536, 16368,28632, 16376,32728,
   16424,20512, 16432,24608, 16440,28704, 16480,16928, 16488,21024,
   16496,25120, 16504,29216, 16544,17440, 16552,21536, 16560,25632,
   16568,29728, 16608,17952, 16616,22048, 16624,26144, 16632,30240,
   16672,18464, 16680,22560, 16688,26656, 16696,30752, 16736,18976,
   16744,23072, 16752,27168, 16760,31264, 16800,19488, 16808,23584,
   16816,27680, 16824,31776, 16864,20000, 16872,24096, 16880,28192,
   16888,32288, 16936,20576, 16944,24672, 16952,28768, 17000,21088,
   17008,25184, 17016,29280, 17056,17504, 17064,21600, 17072,25696,
   17080,29792, 17120,18016, 17128,22112, 17136,26208, 17144,30304,
   17184,18528, 17192,22624, 17200,26720, 17208,30816, 17248,19040,
   17256,23136, 17264,27232, 17272,31328, 17312,19552, 17320,23648,
   17328,27744, 17336,31840, 17376,20064, 17384,24160, 17392,28256,
   17400,32352, 17448,20640, 17456,24736, 17464,28832, 17512,21152,
   17520,25248, 17528,29344, 17576,21664, 17584,25760, 17592,29856,
   17632,18080, 17640,22176, 17648,26272, 17656,30368, 17696,18592,
   17704,22688, 17712,26784, 17720,30880, 17760,19104, 17768,23200,
   17776,27296, 17784,31392, 17824,19616, 17832,23712, 17840,27808,
   17848,31904, 17888,20128, 17896,24224, 17904,28320, 17912,32416,
   17960,20704, 17968,24800, 17976,28896, 18024,21216, 18032,25312,
   18040,29408, 18088,21728, 18096,25824, 18104,29920, 18152,22240,
   18160,26336, 18168,30432, 18208,18656, 18216,22752, 18224,26848,
   18232,30944, 18272,19168, 18280,23264, 18288,27360, 18296,31456,
   18336,19680, 18344,23776, 18352,27872, 18360,31968, 18400,20192,
   18408,24288, 18416,28384, 18424,32480, 18472,20768, 18480,24864,
   18488,28960, 18536,21280, 18544,25376, 18552,29472, 18600,21792,
   18608,25888, 18616,29984, 18664,22304, 18672,26400, 18680,30496,
   18728,22816, 18736,26912, 18744,31008, 18784,19232, 18792,23328,
   18800,27424, 18808,31520, 18848,19744, 18856,23840, 18864,27936,
   18872,32032, 18912,20256, 18920,24352, 18928,28448, 18936,32544,
   18984,20832, 18992,24928, 19000,29024, 19048,21344, 19056,25440,
   19064,29536, 19112,21856, 19120,25952, 19128,30048, 19176,22368,
   19184,26464, 19192,30560, 19240,22880, 19248,26976, 19256,31072,
   19304,23392, 19312,27488, 19320,31584, 19360,19808, 19368,23904,
   19376,28000, 19384,32096, 19424,20320, 19432,24416, 19440,28512,
   19448,32608, 19496,20896, 19504,24992, 19512,29088, 19560,21408,
   19568,25504, 19576,29600, 19624,21920, 19632,26016, 19640,30112,
   19688,22432, 19696,26528, 19704,30624, 19752,22944, 19760,27040,
   19768,31136, 19816,23456, 19824,27552, 19832,31648, 19880,23968,
   19888,28064, 19896,32160, 19936,20384, 19944,24480, 19952,28576,
   19960,32672, 20008,20960, 20016,25056, 20024,29152, 20072,21472,
   20080,25568, 20088,29664, 20136,21984, 20144,26080, 20152,30176,
   20200,22496, 20208,26592, 20216,30688, 20264,23008, 20272,27104,
   20280,31200, 20328,23520, 20336,27616, 20344,31712, 20392,24032,
   20400,28128, 20408,32224, 20456,24544, 20464,28640, 20472,32736,
   20528,24616, 20536,28712, 20584,21032, 20592,25128, 20600,29224,
   20648,21544, 20656,25640, 20664,29736, 20712,22056, 20720,26152,
   20728,30248, 20776,22568, 20784,26664, 20792,30760, 20840,23080,
   20848,27176, 20856,31272, 20904,23592, 20912,27688, 20920,31784,
   20968,24104, 20976,28200, 20984,32296, 21040,24680, 21048,28776,
   21104,25192, 21112,29288, 21160,21608, 21168,25704, 21176,29800,
   21224,22120, 21232,26216, 21240,30312, 21288,22632, 21296,26728,
   21304,30824, 21352,23144, 21360,27240, 21368,31336, 21416,23656,
   21424,27752, 21432,31848, 21480,24168, 21488,28264, 21496,32360,
   21552,24744, 21560,28840, 21616,25256, 21624,29352, 21680,25768,
   21688,29864, 21736,22184, 21744,26280, 21752,30376, 21800,22696,
   21808,26792, 21816,30888, 21864,23208, 21872,27304, 21880,31400,
   21928,23720, 21936,27816, 21944,31912, 21992,24232, 22000,28328,
   22008,32424, 22064,24808, 22072,28904, 22128,25320, 22136,29416,
   22192,25832, 22200,29928, 22256,26344, 22264,30440, 22312,22760,
   22320,26856, 22328,30952, 22376,23272, 22384,27368, 22392,31464,
   22440,23784, 22448,27880, 22456,31976, 22504,24296, 22512,28392,
   22520,32488, 22576,24872, 22584,28968, 22640,25384, 22648,29480,
   22704,25896, 22712,29992, 22768,26408, 22776,30504, 22832,26920,
   22840,31016, 22888,23336, 22896,27432, 22904,31528, 22952,23848,
   22960,27944, 22968,32040, 23016,24360, 23024,28456, 23032,32552,
   23088,24936, 23096,29032, 23152,25448, 23160,29544, 23216,25960,
   23224,30056, 23280,26472, 23288,30568, 23344,26984, 23352,31080,
   23408,27496, 23416,31592, 23464,23912, 23472,28008, 23480,32104,
   23528,24424, 23536,28520, 23544,32616, 23600,25000, 23608,29096,
   23664,25512, 23672,29608, 23728,26024, 23736,30120, 23792,26536,
   23800,30632, 23856,27048, 23864,31144, 23920,27560, 23928,31656,
   23984,28072, 23992,32168, 24040,24488, 24048,28584, 24056,32680,
   24112,25064, 24120,29160, 24176,25576, 24184,29672, 24240,26088,
   24248,30184, 24304,26600, 24312,30696, 24368,27112, 24376,31208,
   24432,27624, 24440,31720, 24496,28136, 24504,32232, 24560,28648,
   24568,32744, 24632,28720, 24688,25136, 24696,29232, 24752,25648,
   24760,29744, 24816,26160, 24824,30256, 24880,26672, 24888,30768,
   24944,27184, 24952,31280, 25008,27696, 25016,31792, 25072,28208,
   25080,32304, 25144,28784, 25208,29296, 25264,25712, 25272,29808,
   25328,26224, 25336,30320, 25392,26736, 25400,30832, 25456,27248,
   25464,31344, 25520,27760, 25528,31856, 25584,28272, 25592,32368,
   25656,28848, 25720,29360, 25784,29872, 25840,26288, 25848,30384,
   25904,26800, 25912,30896, 25968,27312, 25976,31408, 26032,27824,
   26040,31920, 26096,28336, 26104,32432, 26168,28912, 26232,29424,
   26296,29936, 26360,30448, 26416,26864, 26424,30960, 26480,27376,
   26488,31472, 26544,27888, 26552,31984, 26608,28400, 26616,32496,
   26680,28976, 26744,29488, 26808,30000, 26872,30512, 26936,31024,
   26992,27440, 27000,31536, 27056,27952, 27064,32048, 27120,28464,
   27128,32560, 27192,29040, 27256,29552, 27320,30064, 27384,30576,
   27448,31088, 27512,31600, 27568,28016, 27576,32112, 27632,28528,
   27640,32624, 27704,29104, 27768,29616, 27832,30128, 27896,30640,
   27960,31152, 28024,31664, 28088,32176, 28144,28592, 28152,32688,
   28216,29168, 28280,29680, 28344,30192, 28408,30704, 28472,31216,
   28536,31728, 28600,32240, 28664,32752, 28792,29240, 28856,29752,
   28920,30264, 28984,30776, 29048,31288, 29112,31800, 29176,32312,
   29368,29816, 29432,30328, 29496,30840, 29560,31352, 29624,31864,
   29688,32376, 29944,30392, 30008,30904, 30072,31416, 30136,31928,
   30200,32440, 30520,30968, 30584,31480, 30648,31992, 30712,32504,
   31096,31544, 31160,32056, 31224,32568, 31672,32120, 31736,32632,
   32248,32696
};
#endif // INCLUDE_FFT_TABLES_4096




