/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    sysctl_dig.c
*   @brief   digital system top level configuration
*   @details
*
*/


#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */


#include "encoder.h"
#include "spi.h"
/*
#include "PM_endat22_Include.h"
#include "PM_bissc_Include.h"
#include "PM_tformat_include.h"
*/
#if ((IS_GS32F00xx(0x30)))

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
#define ENC_BASE 0x400D4000
/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
//TFORMAT
PM_tformat_DataStruct tformatData;
uint32_t TFORMAT_SPI = SPIB_BASE;
volatile uint16_t tformatRxData[PM_TFORMAT_FIELDS_MAX];

//BISSC
BISSC_DATA_STRUCT bissc_data_struct;
uint32_t BISS_SPI = SPIB_BASE;

//ENDAT
ENDAT_DATA_STRUCT endat22Data;
uint32_t ENDAT_SPI = SPIB_BASE;
/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */
/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

//-------------------------     TA-Format START     -------------------------//
static inline void
tformat_configureSPILen(void)
{
    SPI_disableModule(PM_TFORMAT_SPI);
    SPI_enableModule(PM_TFORMAT_SPI);

    //
    // Set the SPI FIFO interrupt level for received data
    //
    HWREGH(PM_TFORMAT_SPI + SPI_O_FFRX) = (HWREGH(PM_TFORMAT_SPI + SPI_O_FFRX)
                                          & (~SPI_FFRX_RXFFIL_M))
                                          | (uint16_t)tformatData.fifoLevel;

    HWREGH(PM_TFORMAT_SPI + SPI_O_FFTX) = (HWREGH(PM_TFORMAT_SPI + SPI_O_FFTX)
                                          & (~SPI_FFTX_TXFFIL_M))
                                          | (uint16_t)tformatData.fifoLevel;
    SPI_resetTxFIFO(PM_TFORMAT_SPI);
    SPI_resetRxFIFO(PM_TFORMAT_SPI);
    SPI_disableModule(PM_TFORMAT_SPI);
    SPI_enableModule(PM_TFORMAT_SPI);
    return;
}

void PM_tformat_setupPeriph()
{
	//SPI init
	SPI_disableModule(PM_TFORMAT_SPI);
	SPI_setConfig(PM_TFORMAT_SPI, DEVICE_APBCLK_FREQ, SPI_PROT_POL1PHA0,
				  SPI_MODE_SLAVE, 500000, PM_TFORMAT_FIFO_WORD_BITS);

	SPI_enableFIFO(PM_TFORMAT_SPI);
	SPI_setFIFOInterruptLevel(PM_TFORMAT_SPI, SPI_FIFO_TX2, SPI_FIFO_RX2);
	SPI_clearInterruptStatus(PM_TFORMAT_SPI, SPI_INT_RXFF);
	SPI_enableInterrupt(PM_TFORMAT_SPI, SPI_INT_RXFF);
	SPI_disableLoopback(PM_TFORMAT_SPI);
	SPI_setEmulationMode(PM_TFORMAT_SPI, SPI_EMULATION_FREE_RUN);
	SPI_enableModule(PM_TFORMAT_SPI);

	//ENC init
	if(PM_TFORMAT_SPI == SPIA_BASE)
		ENC_TA_selRxdIn(ENC_BASE,SPIA);
	else if(PM_TFORMAT_SPI == SPIB_BASE)
		ENC_TA_selRxdIn(ENC_BASE,SPIB);
	ENC_TA_selRcvMode(ENC_BASE,HARDWARE);
	ENC_TA_selFormat(ENC_BASE,TFORMAT);
	ENC_TA_setBaudrate(ENC_BASE,DEVICE_APBCLK_FREQ/2500000);
	ENC_TA_setTxNum(ENC_BASE,1);
	ENC_TA_setRxNum(ENC_BASE,1);

	SPI_writeDataNonBlocking(PM_TFORMAT_SPI,0x3FF);
	ENC_TA_Enable(ENC_BASE);
	SPI_readDataBlockingFIFO(ENC_BASE);
	ENC_TA_Disable(ENC_BASE);
}

void PM_tformat_setupCommandReadEEPROM(uint16_t eepromAddr)
{
	//
	// IDD: Read from the encoder's EEPROM
	//
	// Create the txPacket to calculate the CRC
	// of the Control field (CF) + address field (ADF)
	// Address is sent significant bit first
	// txPacket = 0xCF:ADF (16-bits)
	//
	uint32_t txPacket;
	uint16_t crcSend;
	eepromAddr = __flip16(eepromAddr);
	txPacket = (eepromAddr >> PM_TFORMAT_ONE_DATA_FIELD);
	txPacket = txPacket | (PM_TFORMAT_CFIDD << PM_TFORMAT_ONE_DATA_FIELD);

	crcSend = tformat_getCRC(PM_TFORMAT_TX_CRC_BITS_IDD,
							(uint16_t *)&txPacket,
							tformatCRCtable,
							PM_TFORMAT_TX_CRC_BYTES_IDD);

	tformatData.dataReady = FALSE;
	tformatData.fifoLevel = PM_TFORMAT_FIFO_LEVEL_IDD;

	uint16_t controlField, addressField, crcField;
	controlField = PM_TFORMAT_IDD;

	//
	// addressField = delimiter:start:ADF:000000b
	// addressField = 1:0:ADF:000000b
	//                1 1  8    6   bits
	//
	addressField = PM_TFORMAT_ADD_DELIMITER_START(eepromAddr);

	//
	// Align left
	// Add delimiter (1) + start (0) before CRC
	//
	//                d:s:CRC
	// crcField =     1:0:CRC:000000b
	//                1 1  8    6   bits
	//
	crcField = crcSend << PM_TFORMAT_ONE_DATA_FIELD;
	crcField = PM_TFORMAT_ADD_DELIMITER_START(crcField);

	tformat_configureSPILen();
	ENC_TA_setTxNum(ENC_BASE,PM_TFORMAT_TX_FIELDS_IDD);
	ENC_TA_setRxNum(ENC_BASE,PM_TFORMAT_RX_FIELDS_IDD);

	//
	// Load the request into the SPI TX FIFO buffer.
	// Fill the remaining buffer slots with 0xFFFF
	// to keep SPI SOMI high while the response
	// is received.
	//
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = controlField;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = addressField;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = crcField;

	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
	HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;

	return;
}

void
PM_tformat_setupCommandWriteEEPROM(uint16_t eepromAddr,
                                   uint16_t eepromData)
{
    //
    // ID6: Write to the encoder's EEPROM
    //
    // Create the txPacket to calculate the CRC
    // Control Field (CF) + address field (ADF) + data field (EDF)
    // Address and data are sent significant bit first
    //
    uint32_t txPacket;
    uint32_t crcSend;

    //
    // eepromData = 0xEDF:00
    // txPacket   = 0x00:00:00:EDF
    //
    eepromData = (__flip16(eepromData));
    txPacket = eepromData >> PM_TFORMAT_ONE_DATA_FIELD;

    //
    // eepromAddr   = 0xADF:00
    // txPacket     = 0x00:00:ADF:EDF
    //
    eepromAddr = (__flip16(eepromAddr));
    txPacket = txPacket | eepromAddr;

    //
    //  txPacket = 0x00:CF:ADF:EDF
    //
    txPacket = txPacket
            | ((uint32_t)PM_TFORMAT_CFID6 << PM_TFORMAT_TWO_DATA_FIELDS);

	crcSend = tformat_getCRC(PM_TFORMAT_TX_CRC_BITS_ID6,
							(uint16_t *)&txPacket,
							tformatCRCtable,
							PM_TFORMAT_TX_CRC_BYTES_ID6);


    tformatData.dataReady = FALSE;
    tformatData.fifoLevel = PM_TFORMAT_FIFO_LEVEL_ID6;

    tformat_configureSPILen();
	ENC_TA_setTxNum(ENC_BASE,PM_TFORMAT_TX_FIELDS_ID6);
	ENC_TA_setRxNum(ENC_BASE,PM_TFORMAT_RX_FIELDS_ID6);

    uint16_t controlField, addressField, eepromDataField, crcField;
    controlField = PM_TFORMAT_ID6;

    //
    // Add delimiter bit (1) + start bit (0)
    //
    //                     d:s:
    // addressField     =  1:0:ADF:000000b
    // eepromDataField  =  1:0:EDF:000000b
    //                     1 1  8    6   bits
    //
    addressField = PM_TFORMAT_ADD_DELIMITER_START(eepromAddr);
    eepromDataField = PM_TFORMAT_ADD_DELIMITER_START(eepromData);

    //
    // Align left
    // Add delimiter bit (1) + start bit (0)
    //
    //                d:s:CRC
    // crcField =     1:0:CRC:000000b
    //                1 1  8    6   bits
    //
    crcField = crcSend << PM_TFORMAT_ONE_DATA_FIELD;
    crcField = PM_TFORMAT_ADD_DELIMITER_START(crcField);

    //
    // Load the request into the SPI TX FIFO buffer.
    // Fill the remaining buffer slots with 0xFFFF
    // to keep SPISOMI high while the response
    // is received.
    //
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = controlField;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = addressField;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = eepromDataField;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = crcField;

    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;

    return;
}

void
PM_tformat_setupCommandReadoutOrReset(uint16_t commandID0_1_2_3_7_8_C,
                              uint16_t tformatRXNum,
                              uint16_t tformatRXFields,
                              uint16_t tformatTXNum,
                              uint16_t tformatFIFOLevel)
{
    tformatData.dataReady = FALSE;
    tformatData.fifoLevel = tformatFIFOLevel;
    tformat_configureSPILen();
	ENC_TA_setTxNum(ENC_BASE,tformatTXNum);
	ENC_TA_setRxNum(ENC_BASE,tformatRXNum);

    //
    // Load the request into the TX FIFO buffer.
    // Fill unused FIFO slots with 0xFFFF to keep
    // the SPI SOMI line high while the response is
    // received.
    //
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = commandID0_1_2_3_7_8_C;

    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;

    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    HWREGH(PM_TFORMAT_SPI + SPI_O_TXBUF) = 0xFFFF;
    return;
}

void PM_tformat_receiveDataID0_1_7_8_C(void)
{
	//
	// The SPI RX FIFO clocks in data (all 0xFFFF) while the
	// command is transmitted.  Therefore, the response begins
	// at an offset corresponding to the number of TX FIELDS.
	//
	uint16_t rdataIndex;
	rdataIndex = PM_TFORMAT_TX_FIELDS_ID0;
	tformatData.controlField = tformatRxData[rdataIndex++];
	tformatData.statusField = tformatRxData[rdataIndex++];
	tformatData.dataField0 = tformatRxData[rdataIndex++];
	tformatData.dataField1 = tformatRxData[rdataIndex++];
	tformatData.dataField2 = tformatRxData[rdataIndex++];
	tformatData.crcField = tformatRxData[rdataIndex];

	uint32_t crcCheck;
	crcCheck = tformat_getCRCID0_1_7_8_C();

	tformatData.crcCheck = crcCheck;
	return;
}

uint16_t
tformat_getCRCID0_1_7_8_C()
{
    uint16_t crcCheck;
    //
    // This CRC is performed by the C28x CPU
    // Create a rxPacket to calculate the expected CRC of the
    // received data.
    //
    uint32_t rxPkts[2];
    rxPkts[0] =
        ((uint32_t) tformatData.controlField << PM_TFORMAT_THREE_DATA_FIELDS)
        | ((uint32_t) tformatData.statusField << PM_TFORMAT_TWO_DATA_FIELDS)
        | ((uint32_t) tformatData.dataField0 << PM_TFORMAT_ONE_DATA_FIELD)
        | ((uint32_t) tformatData.dataField1);

    rxPkts[1] = ((uint32_t) tformatData.dataField2) ;

    crcCheck = tformat_getCRC(PM_TFORMAT_RX_CRC_BITS_ID0,
                              (uint16_t *)&rxPkts,
                              tformatCRCtable,
                              PM_TFORMAT_RX_CRC_BYTES_ID0);
    return(crcCheck);
}

void PM_tformat_receiveDataID2(void)
{
    //
    // The SPI RX FIFO clocks in data (all 0xFFFF) while the
    // command is transmitted.  Therefore, the response begins
    // at an offset corresponding to the number of TX FIELDS.
    //
    uint16_t rdataIndex;
    rdataIndex = PM_TFORMAT_TX_FIELDS_ID2;
    tformatData.controlField = tformatRxData[rdataIndex++];
    tformatData.statusField = tformatRxData[rdataIndex++];
    tformatData.dataField0 = tformatRxData[rdataIndex++];
    tformatData.crcField = tformatRxData[rdataIndex];

    uint32_t crcCheck;

    crcCheck = tformat_getCRCID2();

    tformatData.crcCheck = crcCheck;
    return;
}

void
PM_tformat_receiveDataID3(void)
{
    //
    // The SPI RX FIFO clocks in data (all 0xFFFF) while the
    // command is transmitted.  Therefore, the response begins
    // at an offset corresponding to the number of TX FIELDS.
    //
    uint16_t rdataIndex;

    rdataIndex = PM_TFORMAT_TX_FIELDS_ID3;
    tformatData.controlField = tformatRxData[rdataIndex++];
    tformatData.statusField = tformatRxData[rdataIndex++];
    tformatData.dataField0 = tformatRxData[rdataIndex++];
    tformatData.dataField1 = tformatRxData[rdataIndex++];
    tformatData.dataField2 = tformatRxData[rdataIndex++];
    tformatData.dataField3 = tformatRxData[rdataIndex++];
    tformatData.dataField4 = tformatRxData[rdataIndex++];
    tformatData.dataField5 = tformatRxData[rdataIndex++];
    tformatData.dataField6 = tformatRxData[rdataIndex++];
    tformatData.dataField7 = tformatRxData[rdataIndex++];
    tformatData.crcField = tformatRxData[rdataIndex];

    uint32_t crcCheck;

    crcCheck = tformat_getCRCID3();

    tformatData.crcCheck = crcCheck;
    return;
}

uint16_t
PM_tformat_receiveDataIDD(void)
{
    //
    // Command was a read from EEPROM (IDD)
    //
    // The SPI RX FIFO clocks in data (all 0xFFFF) while the
    // command is transmitted.  Therefore, the response begins
    // at an offset corresponding to the number of TX FIELDS.
    //
    uint16_t rdataIndex;
    rdataIndex = PM_TFORMAT_TX_FIELDS_IDD;
    tformatData.controlField = tformatRxData[rdataIndex++];
    tformatData.eepromAddressField = tformatRxData[rdataIndex++];
    tformatData.eepromRdDataField = tformatRxData[rdataIndex++];
    tformatData.crcField = tformatRxData[rdataIndex];

    uint16_t crcCheck;
    crcCheck = tformat_getCRCIDD();

    uint16_t returnStatus;
    if( (tformatData.eepromAddressField & 0x1) != 0)
    {
        returnStatus = PM_TFORMAT_EEPROM_BUSY;
    }
    else
    {
        returnStatus = PM_TFORMAT_PASS;
    }

    tformatData.eepromAddressField = __flip16 (tformatData.eepromAddressField);
    tformatData.eepromAddressField =
                tformatData.eepromAddressField >> PM_TFORMAT_ONE_DATA_FIELD;

    tformatData.eepromRdDataField = __flip16 (tformatData.eepromRdDataField);
    tformatData.eepromRdDataField =
                tformatData.eepromRdDataField >> PM_TFORMAT_ONE_DATA_FIELD;

    tformatData.crcCheck = crcCheck;
    return(returnStatus);
}

uint16_t
PM_tformat_receiveDataID6(void)
{
    //
    // Command was a write to EEPROM (ID6)
    //
    // The SPI RX FIFO clocks in data (all 0xFFFF) while the
    // command is transmitted.  Therefore ,the response begins
    // at an offset corresponding to the number of TX FIELDS.
    //
    uint16_t rdataIndex;
    uint16_t returnStatus;
    rdataIndex = PM_TFORMAT_TX_FIELDS_ID6;
    tformatData.controlField = tformatRxData[rdataIndex++];
    tformatData.eepromAddressField = tformatRxData[rdataIndex++];
    tformatData.eepromWrDataField = tformatRxData[rdataIndex++];
    tformatData.crcField = tformatRxData[rdataIndex];

    uint16_t crcCheck;

    crcCheck = tformat_getCRCID6();


    if( (tformatData.eepromAddressField & 0x1) != 0)
    {
        returnStatus = PM_TFORMAT_EEPROM_BUSY;
    }
    else
    {
        returnStatus = PM_TFORMAT_PASS;
    }

    tformatData.eepromAddressField = __flip16 (tformatData.eepromAddressField);
    tformatData.eepromAddressField =
                tformatData.eepromAddressField >> PM_TFORMAT_ONE_DATA_FIELD;

    tformatData.eepromWrDataField = __flip16 (tformatData.eepromWrDataField);
    tformatData.eepromWrDataField =
                tformatData.eepromWrDataField >> PM_TFORMAT_ONE_DATA_FIELD;

    tformatData.crcCheck = crcCheck;
    return(returnStatus);
}

void
tformat_receiveDataError(uint16_t rdataIndex)
{
    tformatData.statusField = tformatRxData[rdataIndex++];
    tformatData.dataField0 = tformatRxData[rdataIndex++];
    tformatData.dataField1 = tformatRxData[rdataIndex++];
    tformatData.dataField2 = tformatRxData[rdataIndex++];
    tformatData.dataField3 = tformatRxData[rdataIndex++];
    tformatData.dataField4 = tformatRxData[rdataIndex++];
    tformatData.dataField5 = tformatRxData[rdataIndex++];
    tformatData.dataField6 = tformatRxData[rdataIndex++];
    tformatData.dataField7 = tformatRxData[rdataIndex++];
    tformatData.crcField = tformatRxData[rdataIndex];

    uint32_t crcCheck;

    crcCheck = tformat_getCRCID3();

    tformatData.crcCheck = crcCheck;
    return;
}

//Set baud rate
void PM_tformat_setFreq(uint32_t Freq_us)
{
	ENC_TA_setBaudrate(ENC_BASE,DEVICE_APBCLK_FREQ/Freq_us);
}


uint16_t
PM_tformat_updateEncoderID(uint16_t enidField)
{

    //
    // The mask to extract the 8-bits of data was already applied in the
    // appropriate receiveData function.
    //
    enidField = __flip16(enidField);
    return(enidField >> PM_TFORMAT_ONE_DATA_FIELD);
}

uint32_t
PM_tformat_updatePositionOrTurns(uint16_t field0,
                                 uint16_t field1,
                                 uint16_t field2)
{
    //
    // The mask to extract the 8-bits of data was already applied in the
    // SPI ISR routine.
    //
    field0 = __flip16(field0);
    field1 = __flip16(field1);
    field2 = __flip16(field2);

    uint32_t temp = (((uint32_t)field2) << PM_TFORMAT_ONE_DATA_FIELD) |
                    ((uint32_t)field1) |
                    (((uint32_t)field0) >> PM_TFORMAT_ONE_DATA_FIELD);

    return(temp);
}

//start
void PM_tformat_startOperation(void)
{
	ENC_TA_Enable(ENC_BASE);
}


//set SPI and ENC with CMD
uint16_t PM_tformat_setupCommand(uint16_t dataID, uint16_t eepromAddr, uint16_t eepromData, uint16_t crc)
{
	return 0;
}

//Unpack and fill the structure
uint16_t PM_tformat_receiveData(uint16_t dataID)
{
	return 0;
}

void PM_tformat_generateCRCTable(uint16_t nBits, uint16_t polynomial,          \
        uint16_t *pTable)
{
    uint16_t i, j;
    uint16_t accum;

    polynomial <<= (8 - nBits);
    for(i = 0; i < 256 ; i++)
    {
        accum  = i;
        for( j = 0; j < 8; j++)
        {
            if(accum & 0x80)
            {

                //
                // If leading bit is 1, shift accum to left, mask off unwanted
                // MSbs and xor the rest with poly
                //
                accum = ((accum << 1) & 0xFF) ^ polynomial;
            }
            else
            {

                //
                // If leading bit is 0, shift accum to left,
                // mask off unwanted most significant bits
                //
                accum = ((accum << 1) & 0xFF);
            }
        }
        pTable[i] = accum;
    }
    return;
}

uint16_t
tformat_getCRC(uint16_t nBitsData,
                        uint16_t *msg,
                        uint16_t *crcTable,
                        uint16_t rxLen)
{
    uint16_t i;
    uint16_t j;
    uint16_t index;
    uint16_t crcAccum;
    uint16_t crcValue;
    int *pdata;

    index = rxLen - 1;
    crcAccum = 0xFF;
    pdata = (int *)msg;

    //
    // Start from the end
    // Do the first two bytes, special case
    //
    i = crcAccum  ^ (__byte(pdata, index--));
    crcAccum = crcTable[i];
    i = crcAccum ^ (__byte(pdata, index--));
    crcAccum = crcTable[i];

    //
    // Do the CRC for the bytes
    //
    for (j = 0; j < rxLen - 2; j++, index--)
    {
        i = crcAccum  ^ (__byte(pdata, index));
        crcAccum = crcTable[i];
    }
    crcValue = crcAccum ^ (PM_TFORMAT_CRC_MASK);
    return(crcValue);
}

uint16_t
tformat_getCRCID2()
{
    uint32_t crcCheck;
    //
    // The CRC is performed by the C28x CPU
    // Create a rxPacket to calculate the expected CRC of the
    // received data.  The calculated CRC is compared with the
    // CRC received from the encoder
    //
    uint32_t rxPkts;
    rxPkts = ((uint32_t) tformatData.controlField << 16)
                            | ((uint32_t) tformatData.statusField << 8)
                            | ((uint32_t) tformatData.dataField0);

    crcCheck = tformat_getCRC(PM_TFORMAT_RX_CRC_BITS_ID2,
                              (uint16_t *)&rxPkts,
                              tformatCRCtable,
                              PM_TFORMAT_RX_CRC_BYTES_ID2);
    return(crcCheck);
}

uint16_t
tformat_getCRCID3()
{
    uint32_t crcCheck;
    //
    // The CRC is performed by the C28x CPU
    // Create a rxPacket to calculate the expected CRC of the
    // received data.  The calculated CRC is compared with the
    // CRC received from the encoder
    //
    uint32_t rxPkts[3];
    rxPkts[0] = ((uint32_t) tformatData.controlField << 24UL)
                          | ((uint32_t) tformatData.statusField << 16UL)
                          | ((uint32_t) tformatData.dataField0 << 8UL)
                          | (uint32_t)  tformatData.dataField1;
    rxPkts[1] = ((uint32_t) tformatData.dataField2 << 24UL) |
                            ((uint32_t) tformatData.dataField3 << 16UL) |
                            ((uint32_t) tformatData.dataField4 << 8UL) |
                            ((uint32_t) tformatData.dataField5);
    rxPkts[2] = ((uint32_t) tformatData.dataField6 << 8UL) |
                              ((uint32_t) tformatData.dataField7);


    crcCheck = tformat_getCRC(PM_TFORMAT_RX_CRC_BITS_ID3,
                              (uint16_t *)&rxPkts,
                              tformatCRCtable,
                              PM_TFORMAT_RX_CRC_BYTES_ID3);
    return(crcCheck);
}


uint16_t
tformat_getCRCIDD()
{
    uint16_t crcCheck;
    //
    // The CRC is performed by the C28x CPU
    // Create a rxPacket to calculate the expected CRC of the
    // received data.  The calculated CRC is compared with the
    // CRC received from the encoder
    //
    uint32_t rxPkts;
    rxPkts = ((uint32_t) tformatData.controlField << 16)
                          | ((uint32_t) tformatData.eepromAddressField << 8)
                          | ((uint32_t) tformatData.eepromRdDataField);

    crcCheck = tformat_getCRC(PM_TFORMAT_RX_CRC_BITS_IDD,
                                  (uint16_t *)&rxPkts,
                                  tformatCRCtable,
                                  PM_TFORMAT_RX_CRC_BYTES_IDD);
    return(crcCheck);
}
//-------------------------     TA-Format END     -------------------------//


//-------------------------     BISS-C START     -------------------------//

void PM_bissc_setupPeriph(void)
{
	BISS_SPI = (uint32_t)endat22Data.spi;
	//SPI init
    SPI_disableModule(BISS_SPI);
    SPI_setConfig(BISS_SPI, DEVICE_APBCLK_FREQ, SPI_PROT_POL1PHA1,
                  SPI_MODE_PERIPHERAL, 500000, 9);
    SPI_enableFIFO(BISS_SPI);
    SPI_setFIFOInterruptLevel(BISS_SPI, SPI_FIFO_TX2, SPI_FIFO_RX2);
    SPI_clearInterruptStatus(BISS_SPI, SPI_INT_RXFF);
    SPI_enableInterrupt(BISS_SPI, SPI_INT_RXFF);
    SPI_disableLoopback(BISS_SPI);
    SPI_enableModule(BISS_SPI);

    //ENC init
    ENC_BISS_selRcvMode(ENC_BASE,HARDWARE);
    ENC_BISS_SPI_MOD(ENC_BASE,BISS);
    if(BISS_SPI == SPIA_BASE)
    	ENC_BISS_selRxdIn(ENC_BASE,SPIA);
    else if(BISS_SPI == SPIB_BASE)
    	ENC_BISS_selRxdIn(ENC_BASE,SPIB);

    //bissc_data_struct init
	bissc_data_struct.dataReady = 0;
	bissc_data_struct.cd_status = 3;
	bissc_data_struct.cd_register_xfer_is_write = 0;
	bissc_data_struct.cd_register_xfer_rxdata = 0xFFFF;
	bissc_data_struct.cd_register_xfer_txdata = 0xFFFF;
	bissc_data_struct.crc_incorrect_count = 0;

}

void PM_bissc_setFreq(uint32_t Freq_us)
{
	ENC_BISS_setBaudrate(ENC_BASE,Freq_us);
}

void PM_bissc_setCDBit(uint32_t cdmBit)
{
	ENC_BISS_setCDM(ENC_BASE,cdmBit);
}

void PM_bissc_startOperation(void)
{
	ENC_BISS_Enable(ENC_BASE);
}

void PM_bissc_setupNewSCDTransfer(uint16_t nDataClks, uint16_t spi_fifo_width)
{
	uint16_t num;
	ENC_BISS_setRxNum(ENC_BASE,nDataClks+1);
	if((nDataClks%spi_fifo_width)>0)
		num = (nDataClks/spi_fifo_width) + 1;
	else
		num = nDataClks/spi_fifo_width;
	bissc_data_struct.fifo_level = num;
	ENC_BISS_setSpiBitNum(ENC_BASE,num*spi_fifo_width);
	//SPI_setFIFOInterruptLevel(bissc_data_struct.spi,num,num);
}

uint64_t PM_bissc_getBits (uint16_t len, uint16_t bitsParsed, uint16_t charBits)
{
    uint16_t index = 0;
    uint16_t len_surplus;
    uint16_t parsed;
    uint32_t *dPointer;
    uint64_t data;
    uint32_t i;
    uint32_t data_end;

    data = 0;
    dPointer = bissc_data_struct.rdata;
    index +=  bitsParsed / charBits;
    parsed = (charBits - bitsParsed % charBits);
    if(parsed>len)
    {
        data = (dPointer[index]>>(parsed-len)) & (((uint64_t)1<<len) -1);
        return data;
    }
    data = dPointer[index++] & (((uint64_t)1<<parsed)-1);
    len -= parsed;
    for(i = 0 ; i < (len / charBits) ; i++)
    {
        data = (data << charBits) | (dPointer[index++] & (((uint64_t)1<<charBits)-1));
    }
    len_surplus = len % charBits;
    data_end = charBits-len_surplus;
    if(len_surplus > 0)
    {
        data = (data << len_surplus) | ((dPointer[index++]>>data_end) & (((uint64_t)1<<len_surplus)-1));
    }
    return data;
}
void PM_bissc_generateCRCTable(uint16_t nBits, uint16_t polynomial, uint16_t *pTable)
{

}
uint16_t PM_bissc_getCRC(uint16_t input_crc_accum, uint16_t nBitsData,  uint16_t nBitsPoly, uint16_t * msg, uint16_t *crc_table, uint16_t rxLen)
{
	return 0;
}


//-------------------------     BISS-C END     -------------------------//

//-------------------------     ENDAT START     -------------------------//
void PM_endat22_setupPeriph(void)
{
	ENDAT_SPI = (uint32_t)endat22Data.spi;
	ENC_ENDAT_Disable(ENC_BASE);
	ENC_TA_selRcvMode(ENC_BASE,HARDWARE);
	if(ENDAT_SPI == SPIA_BASE)
	{
		ENC_ENDAT_selRxdIn(ENC_BASE,SPIA);
	}
	else if(ENDAT_SPI == SPIB_BASE)
	{
		ENC_ENDAT_selRxdIn(ENC_BASE,SPIB);
	}
	ENC_ENDAT_setTst(ENC_BASE,3);
    SPI_disableModule(ENDAT_SPI);
    SPI_setConfig(ENDAT_SPI, DEVICE_APBCLK_FREQ, SPI_PROT_POL1PHA1,
                  SPI_MODE_PERIPHERAL, 500000, 16);
    SPI_enableFIFO(ENDAT_SPI);
    SPI_setFIFOInterruptLevel(ENDAT_SPI, SPI_FIFO_TX3, SPI_FIFO_RX2);
    SPI_clearInterruptStatus(ENDAT_SPI, SPI_INT_RXFF);
    SPI_enableInterrupt(ENDAT_SPI, SPI_INT_RXFF);
    SPI_disableLoopback(ENDAT_SPI);
    SPI_enableModule(ENDAT_SPI);

}

void PM_endat22_Command12_23_E_1C_31_2A(uint16_t cmd, uint16_t data1, uint16_t data2, uint16_t nAddData)
{
	uint16_t i;
	endat22Data.dataReady = 0;
	endat22Data.fifo_level = 4;
	SPI_setFIFOInterruptLevel(ENDAT_SPI, SPI_FIFO_TX4, SPI_FIFO_RX4);
	endat22Data.sdata[0] = (cmd << 8) + (data1 & 0xFF);
	endat22Data.sdata[1] = data2;
	for(i = 0;i < endat22Data.fifo_level;i++)
	{
		SPI_writeDataNonBlocking(ENDAT_SPI,endat22Data.sdata[i]);
	}
//	ENC_ENDAT_setSpiBitNum(ENC_BASE,4*16);
//	ENC_ENDAT_setTxNum(ENC_BASE,2*16+1);
//	ENC_ENDAT_setRxNum(ENC_BASE,2*16-1);
}
void PM_endat22_Command9_1B_24_2D_36_38(uint16_t cmd, uint16_t data1, uint16_t data2, uint16_t nAddData)
{
	uint16_t i;
	uint16_t clknum;
	uint32_t exdata;
	exdata = 0x01000000 + (data1<16) + data2;
	endat22Data.dataReady = 0;
	clknum = 9+endat22Data.position_clocks+8+1;
	if(clknum&0xF)
		endat22Data.fifo_level = (clknum>>4) + 1 + 2;
	else
		endat22Data.fifo_level = clknum>>4 + 2;
	endat22Data.sdata[0] = cmd<<8;
	clknum -= 16;
	for(i = 1;i<((endat22Data.position_clocks+1)/16+1);i++)
	{
		endat22Data.sdata[i] = 0;
		clknum -= 16;
	}
	if(clknum)
	{
		endat22Data.sdata[i++] = exdata>>(clknum+16);
		endat22Data.sdata[i++] = (exdata>>clknum)&0xFFFF;
		endat22Data.sdata[i] = exdata&(0xFFFFFFFF>>(32-clknum));
	}
	else
	{
		endat22Data.sdata[i++] = exdata>>16;
		endat22Data.sdata[i] = exdata&0xFFFF;
	}
	for(i=0;i<endat22Data.fifo_level;i++)
	{
		SPI_writeDataNonBlocking(ENDAT_SPI,endat22Data.sdata[i]);
	}
/*	ENC_ENDAT_setSpiBitNum(ENC_BASE,endat22Data.fifo_level*16);
	ENC_ENDAT_setTxNum(ENC_BASE,9);
	ENC_ENDAT_setRxNum(ENC_BASE,endat22Data.fifo_level*16-9);*/

}
void PM_endat22_positionCommand(uint16_t cmd, uint16_t data1, uint16_t data2, uint16_t nAddData)
{
	uint16_t i;
	endat22Data.dataReady = 0;
	if(((endat22Data.position_clocks+7)&0xF))
		endat22Data.fifo_level = ((endat22Data.position_clocks+7)>>4) + 1;
	else
		endat22Data.fifo_level = (endat22Data.position_clocks+7)>>4;
	SPI_setFIFOInterruptLevel(ENDAT_SPI, endat22Data.fifo_level, endat22Data.fifo_level);
	endat22Data.sdata[0] = cmd << 8;
	for(i=0;i<endat22Data.fifo_level;i++)
	{
		SPI_writeDataNonBlocking(ENDAT_SPI,endat22Data.sdata[i]);
	}
/*	ENC_ENDAT_setSpiBitNum(ENC_BASE,endat22Data.fifo_level*16);
	ENC_ENDAT_setTxNum(ENC_BASE,9);
	ENC_ENDAT_setRxNum(ENC_BASE,endat22Data.fifo_level*16-9);*/
}
void PM_endat22_sendTestCommand(uint16_t cmd, uint16_t data1, uint16_t data2, uint16_t nAddData)
{
	uint16_t i;
	endat22Data.dataReady = 0;
	endat22Data.fifo_level = 4;
	SPI_setFIFOInterruptLevel(ENDAT_SPI, SPI_FIFO_TX4, SPI_FIFO_RX4);
	endat22Data.sdata[0] = cmd << 8;
	for(i = 0;i < endat22Data.fifo_level;i++)
	{
		SPI_writeDataNonBlocking(ENDAT_SPI,endat22Data.sdata[i]);
	}
/*	ENC_ENDAT_setSpiBitNum(ENC_BASE,4*16);
	ENC_ENDAT_setTxNum(ENC_BASE,9);
	ENC_ENDAT_setRxNum(ENC_BASE,4*16-9);*/
}


uint16_t PM_endat22_setupCommand(uint16_t cmd, uint16_t data1, uint16_t data2, uint16_t nAddData)
{
	uint16_t way = 0;
	if(cmd == 0x23)
	{

	}
	else if(cmd > 0x23)
	{
		if(cmd == 0x31)
		{

		}
		else if(cmd >0x31)
		{
			if(cmd == 0x36)
			{

			}
			else if(cmd == 0x38)
			{

			}
		}
		else
		{
			if(cmd == 0x24)
			{

			}
			else if(cmd == 0x2a)
			{

			}
			else if(cmd == 0x2d)
			{

			}
		}
	}
	else
	{
		if(cmd == 0x12)
		{

		}
		else if(cmd >0x12)
		{
			if(cmd == 0x15)
			{

			}
			else if(cmd == 0x16)
			{

			}
			else if(cmd == 0x1C)
			{

			}
		}
		else
		{
			if(cmd == 0x07)
			{

			}
			else if(cmd == 0x09)
			{

			}
			else if(cmd == 0x0E)
			{

			}
		}
	}
	return 0;
}

uint16_t PM_endat22_receiveData(uint16_t, uint16_t nAddData)
{
	return 0;
}

void PM_endat22_setFreq(uint32_t Freq_us)
{
	ENC_BISS_setBaudrate(ENC_BASE,Freq_us);
}

void PM_endat22_startOperation(void)
{
	ENC_ENDAT_Enable(ENC_BASE);
}

uint16_t PM_endat22_getDelayCompVal(void)
{
	return ENC_ENDAT_getTimeofWaitRpt(ENC_BASE);
}

//-------------------------     ENDAT END     -------------------------//


//-------------------------     abs2qep     -------------------------//

// Set the clock frequency of the cured abs2qep

void PM_abs2qep_configwidth (uint32_t base, uint32_t qclk_phase, uint32_t qclk_period)
{
	 HWREG(base + ABS2QEP_CNT0_REF) = qclk_phase<<0 | qclk_period<<16 ;
}

// How many cycles it takes to set the ABS2QEP
void PM_abs2qep_config_ptodown (uint32_t base, uint32_t pto_down)
{
	 HWREG( base+ ABS2QEP_CNT1_REF) = pto_down<<16 ;
}

//Set the pulse width of the qepi in the ABS2QEP
void PM_abs2qep_config_qepiwidth (uint32_t base, uint32_t qepi_low, uint32_t qepi_high)
{
	 HWREG( base+ ABS2QEP_CNT2_REF) = qepi_low<<0 | qepi_high<<16 ;
}

//Set the direction of the ABS2QEP
void PM_abs2qep_config_direction (uint32_t base, uint32_t direction)
{
	 HWREG( base+ ABS2QEP_CTRL)|= direction<< 1 ;
}

// enable abs2qep Start to wave
void PM_abs2qep_enable (uint32_t base)
{
	 HWREG( base+ ABS2QEP_CTRL) |= 0x1 ;
}

// disable abs2qep stop to wave
void PM_abs2qep_disable (uint32_t base)
{
	 HWREG( base+ ABS2QEP_CTRL) = HWREG( base+ ABS2QEP_CTRL) & 0x2;
}

//-------------------------     abs2qep     -------------------------//

//-------------------------     qepdiv     -------------------------//
// Select the input signal for the qepdiv
void PM_qepdiv_input (uint32_t base,QEPDIV_INPUT_SEL  qepa_input,QEPDIV_INPUT_SEL  qepb_input ,QEPDIV_INPUT_SEL qepi_input)
{
	 HWREG(base + QEPDIV_CTRL) = qepa_input<<4 | qepb_input<<8 | qepi_input<<12 ;
}

// Set the frequency separation coefficient of qepdiv
void PM_qepdiv_diviider (uint32_t base, uint32_t div)
{
	 HWREG( base+ QEPDIV_DIVIDER)= div ;
}

// Set the pulse width of the qepi
void PM_qepdiv_index_width (uint32_t base, uint32_t index_width)
{
	 HWREG( base+ QEPDIV_INDEX_WIDTH)= index_width ;
}

// enable qepdiv start to wave
void PM_qepdiv_enable (uint32_t base)
{
	 HWREG( base+ QEPDIV_CTRL) |= 0x1 ;
}

// disable qepdiv stop to wave
void PM_qepdiv_disable (uint32_t base)
{
	 HWREG( base+ QEPDIV_CTRL) = HWREG( base+ QEPDIV_CTRL) & 0xfffe;
}

//-------------------------     qepdiv     -------------------------//
#ifdef __cplusplus
}
#endif

#endif //0
