/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/******************************************************************************
 *
 * FILE:   pto_abs2qep.c
 *
 * Description: Absolute Position to QEP Pulse Train Out (PTO) Library
 *
 *****************************************************************************/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "stdint.h"
#include "stdlib.h"
#include "stdbool.h"
#include "pto_abs2qep.h"

#include "clb_config.h"
#include "clb.h"

//
// Globals
//
void pto_abs2qep_runPulseGen(uint16_t pto_direction);
//uint16_t pto_abs2qep_translatePosition(uint32_t positionNew);




//*****************************************************************************
//
//! Start the PTO Generation
//!
//! \param pto_direction is either forward (PTO_ABS2QEP_CLOCKWISE_PTO) 
//!        or reverse (PTO_ABS2QEP_COUNTERCLOCKWISE_PTO)
//!
//! This function confirms that the previous PTO has completed and then
//! triggers the load event to the HLC
//!
//! pto_abs2qep_translatePostion() must be called before this function
//! to load the HLC pullData FIFO
//!
//!
//! \return None
//
//*****************************************************************************

void pto_abs2qep_runPulseGen(uint16_t pto_direction)
{

    // Confirm that the previous PTO has completed
    // Start the load of a new configuration
    // Set the direction appropriately
    //
    // The GPIO is a visual marker for the start and
    // direction of the QEP-A/B signals and can be 
    // removed in an application
    //

	// CNT1.MATCH1=0              --- CLEAR LATCH,RUN,
	// CNT1.MATCH2=0xFFFFFFFF     --- PTO_DONE,
    CLB_configCounterLoadMatch(CLB1_BASE, CLB_CTR1, TILE1_COUNTER_1_LOAD_VAL, 0, 0xFFFFFFFF);

    while(CLB_getInterruptTag(CLB1_BASE) != PTO_ABS2QEP_PTO_DONE_TAG )
    {
    }
    CLB_setGPREG(CLB1_BASE, 0);
    CLB_clearInterruptTag(CLB1_BASE);
    if(pto_direction == PTO_ABS2QEP_CLOCKWISE_PTO)
    {
        CLB_setGPREG(CLB1_BASE, PTO_ABS2QEP_SET_CLOCKWISE |                    \
                     PTO_ABS2QEP_SET_LOAD);
        GPIO_writePin(PTO_ABSQEP_TEST_PIN_1, 1);

        CLB_setGPREG(CLB1_BASE,2);
    }
    else
    {
        CLB_setGPREG(CLB1_BASE, PTO_ABS2QEP_SET_COUNTERCLOCKWISE |             \
                     PTO_ABS2QEP_SET_LOAD);
        GPIO_writePin(PTO_ABSQEP_TEST_PIN_1, 0);
        CLB_setGPREG(CLB1_BASE,0);
    }

}


//*****************************************************************************
//
//! Translate a change in absolute position to QEP pulses
//!
//! \param  PositionNew is the new absolute position
//!
//! This function calculates the delta from the previous position and
//! translates it to an equivalent number of QCLKs needed to generate
//! QEP-A, QEP-B and QEP-I.
//!
//! The HLC FIFO is loaded with the configuration.
//!
//! \return None
//
//*****************************************************************************

uint16_t
pto_abs2qep_translatePosition(uint32_t positionNew)
{
    uint16_t numQclkSenddeduct,numQclkSendadd;
    uint16_t ptoDirection;
    int32_t  positionDelta;
    uint32_t indexHighEdge,clkadjust;
    uint16_t pulseWidth1,pulseWidth2;
    float32_t numQclkFloat;
    float32_t numQclkFrac;
    int16_t numQclkAdjust;
    static float32_t numQclkTotalFrac = 0.0f,numQclkpreFrac;
    static float32_t numQclkCarry = 0.0f,pulseclk=0.0;
    static uint32_t positionPrevious = 0ul;
    static uint32_t positionCurrent = 0ul;
    float32_t numQclkInt;
    Bool positionZeroCross;

    //
    // Uncomment the following to halt the CLB on a CPU halt.
    // EALLOW;
    // HWREG(CLB1_BASE + CLB_LOGICCTL + CLB_O_LOAD_EN) |= CLB_LOAD_EN_STOP;
    // EDIS;
    //
    //
    // Archive old absolute position
    // Find the delta between old and current position
    // Determine if change in position crosses the absolute zero position
    //
    positionPrevious = positionCurrent;
   positionCurrent = positionNew;
    positionDelta = positionCurrent - positionPrevious;

    //
    // Check if absolute zero is crossed over during the position delta
    // If so, adjust the calculations for the delta position value to
    // account for crossover.
    // Calculate the delta in two parts as described in the
    // user's guide.
    // Calculate the match 1 and match2 values that will
    // be used in counter 2 to generate the QEP index signal
    //
    positionZeroCross = labs(positionDelta) >
                   (PTO_ABS2QEP_ABS_MAX_DELTA_PER_SAMPLE) ? 1 : 0;

    if(positionZeroCross)
    {
        //
        // Crossed zero in the reverse direction
        // Change sign to indicate reverse
        //
        if(positionDelta > 0)
        {
            positionDelta = -1 *
                    (int32_t)((positionPrevious - 0) +
                         (PTO_ABS2QEP_ABS_MAX_POSITION - positionCurrent));
            indexHighEdge =
                    (int32_t)(positionPrevious * PTO_ABS2QEP_ABS_TO_INCR);
        }
        else
        {
            positionDelta =
                    (int32_t)((PTO_ABS2QEP_ABS_MAX_POSITION -
                                positionPrevious) +
                                positionCurrent);
            indexHighEdge =
                    (int32_t)((PTO_ABS2QEP_ABS_MAX_POSITION -
                                positionPrevious) *
                                PTO_ABS2QEP_ABS_TO_INCR);
        }
    }
    //
    // If zero was not crossed, use a large value so the match never occurs.
    //
    else
    {
        indexHighEdge = 0xFFFFFFFE;
    }
    //
    // Check to see if the QEP index signal occurs at 0
    // If it does, offset by 1 so it is not missed.
    //

    if(indexHighEdge == 0)
    {
        indexHighEdge += 1;
    }
    //
    // Calculate the number of QCLKs based on the delta
    // between the old and new absolute encoder positions.
    //
    // Determine the integer and fractional number of pulses.
    // If the accumulated fractional portion is > 1 or < -1,
    // increase the integer number of pulses by 1.
    //

    numQclkFloat = positionDelta * PTO_ABS2QEP_ABS_TO_INCR;


//    numQclkFrac = modff(numQclkFloat, &numQclkInt);
//
//
//    if(numQclkTotalFrac >= 1.0f)
//    {
//        numQclkAdjust = 1;
//    }
//    else if(numQclkTotalFrac <= -1.0f)
//    {
//        numQclkAdjust = -1;
//    }
//    else
//    {
//        numQclkAdjust = 0;
//    }
//    numQclkCarry = numQclkTotalFrac - numQclkAdjust;
//    numQclkSend = abs( (int16_t)numQclkInt + numQclkAdjust);





 #ifdef adiust_2.0
    if(numQclkSend != 0)
    {
        pulseWidth = (PTO_ABS2QEP_SAMPLING_RATIO / numQclkFloat);
    }
    else
    {
        pulseWidth = 0xFFFF;
    }

    if(numQclkTotalFrac < 0.1f)
    {
        numQclkTotalFrac + = numQclkFrac; //鏉烆剟锟界喕绻冪亸蹇ョ礉閻╁瓨甯寸槐顖氬閺冨娴嗘担宥囩枂

        // 瑜版挸澧犻崨銊︽埂鐏忓繑鏆熼柈銊ュ瀻娑撳秷顓哥粻妤佹闁界噦绱濆▽璺ㄦ暏娑撳﹣閲滈崨銊︽埂閻ㄥ嫭妞傞柦锟�
        pulseclk = (0+ pulseclk );
        // 瑜版挸澧犻崨銊︽埂鐠嬪啯鏆atch閸婏拷= 娑撳秷顓哥粻妤呯帛鐠併倕锟斤拷
        clkadjMatch1=0xFFFF ;


    }
    else if(numQclkTotalFrac > -0.1f)
    {
        numQclkTotalFrac - = numQclkFrac;
        
    }
    else
    {
        

        // 瑜版挸澧犻崨銊︽埂鐏忓繑鏆熼柈銊ュ瀻瀹歌尙绮＄拫鍐╂殻閻ㄥ嚋LB閺冨爼鎸撻弫锟�
        pulseclk = (pulseWidth* numQclkTotalFrac + pulseclk );
        // 鐠侊紕鐣诲鑼病鐠嬪啯鏆ｉ惃鍕闁界喐鏆熼柌蹇旀Ц閸氾箑銇囨禍搴ょ珶濞屽灝褰夐崠鏍劄闂�鍖＄礉婢堆傜艾閸掓瑥顤冮崝鐘辩娑擃亜鎳嗛張锟�

        numQclkTotalFrac = 0;  //鏉烆剟锟界喕绶濇径褝绱濋幋鏍拷鍛柈閸旂姴鍩屾潏鍐ㄣ亣娴ｅ秶鐤嗛敍宀冾吀缁犳妞傞柦鐔蜂焊缁夛拷

        if(pulseclk>pulseWidth)
            {
                pulseclk-=pulseWidth;
                numQclkInt+=1;
            }
        else
            {
            // 瑜版挸澧犻崨銊︽埂鐠嬪啯鏆atch閸婏拷= 鏉堣閮ㄩ崣妯哄濮濄儵鏆� - 娑撳﹣閲滈崨銊︽埂瀹歌尙绮＄拫鍐╂殻閻ㄥ嚋LB閺冨爼鎸撻弫浼村櫤
            clkadjMatch1=pulseWidth - pulseclk ;   
            }

    }

#endif


  	{
//  		pulseWidth1 = (PTO_ABS2QEP_SAMPLING_RATIO / numQclkFloat);
  	}
//  	else
//  	{
//  		pulseWidth = 0xFFFF;
//  	}

      numQclkFrac = modff(numQclkFloat, &numQclkInt);


    numQclkTotalFrac = numQclkpreFrac +numQclkFrac; //上周期剩下的小数+这个周期的小数


    if(numQclkTotalFrac >=1.0f)

    {
    	numQclkTotalFrac=numQclkTotalFrac-1;
        numQclkAdjust = 1;

    }
    else if(numQclkTotalFrac < -1.0f)
    {
        numQclkAdjust = -1;
    }
    else
    {

        numQclkAdjust = 0;
    }

 //   numQclkFloat = numQclkInt+numQclkAdjust+numQclkTotalFrac;  //当前周期边沿总数
    numQclkInt= numQclkInt+numQclkAdjust;

    pulseWidth1 = (PTO_ABS2QEP_SAMPLING_RATIO / numQclkFloat); //计算边沿之间的CLB时钟

    numQclkSenddeduct =(pulseWidth1+1)*numQclkFloat - PTO_ABS2QEP_SAMPLING_RATIO; //计算边沿之间低CLB时钟的数量

    if(numQclkInt > numQclkSenddeduct)   //防止只有一个边沿的情况
    {
    	numQclkSenddeduct+=1; 						//边沿之间低CLB时钟的数量向上取整
    }
    numQclkSendadd =numQclkFloat-numQclkSenddeduct; //边沿之间高CLB时钟的数量

    clkadjust=pulseWidth1 * numQclkpreFrac;  //时钟调整数量，上周期已经运行的时钟，直接加载到计数器
    if(clkadjust==pulseWidth1-1)

    {
    	clkadjust-=2;
    }

    numQclkpreFrac =numQclkTotalFrac;  //当前周期剩余小数



CLB_setHLCRegisters(CLB1_BASE, pulseWidth1,clkadjust,indexHighEdge,indexHighEdge + 1U);
CLB_writeDbgRx(CLB1_BASE ,4 ,pulseWidth1-1);
CLB_writeDbgRx(CLB1_BASE ,5 ,numQclkSenddeduct+1);
CLB_writeDbgRx(CLB1_BASE ,6 ,numQclkInt+1);
//#endif 

    //
    // Log the direction information for when the PTO is
    // started by the application
    //

    if(positionDelta > 0)
    {
        ptoDirection = PTO_ABS2QEP_CLOCKWISE_PTO;
    }
    else
    {
        ptoDirection = PTO_ABS2QEP_COUNTERCLOCKWISE_PTO;

    }
    return(ptoDirection);
}



//
// Simulate a new absolute position to test.
// Use a fraction of the max delta per sample
// Periodically switch directions and update
// the fraction.  The fractions were chosen to
// somewhat randomize the PTOs and where the
// index signal occurs.  Otherwise there is nothing
// special about the values selected.
//
//
uint32_t pto_generateTestAbsPosition(uint32_t currentPosition)
{
    uint32_t deltaPosition = 0ul;
    float32_t static fraction = 0.0f;
    uint16_t static setDirection = PTO_ABS2QEP_CLOCKWISE_PTO;
    uint16_t static sample = 0u;

    sample++;
    if(fraction > 1.0f)
    {
        fraction -= 1.0f;
        setDirection = setDirection == PTO_ABS2QEP_CLOCKWISE_PTO ?             \
                PTO_ABS2QEP_COUNTERCLOCKWISE_PTO : PTO_ABS2QEP_CLOCKWISE_PTO;
    }
    if(sample == 5u)
    {
        setDirection = setDirection == PTO_ABS2QEP_CLOCKWISE_PTO ?
                PTO_ABS2QEP_COUNTERCLOCKWISE_PTO : PTO_ABS2QEP_CLOCKWISE_PTO;  \
    }
    else if(sample == 10u)
    {
        fraction += .33f;
    }
    else if(sample == 15u)
    {
        fraction += .55f;
    }
    else if(sample == 20u)
    {
        sample = 0u;
    }
    //
    // test back-to-back zero
    //
    if(sample == 16u || sample == 17u || sample == 18u)
    {
        deltaPosition = 0ul;
    }
    else
    {
        deltaPosition = (uint32_t)                                             \
                ((float32_t)PTO_ABS2QEP_ABS_MAX_DELTA_PER_SAMPLE * fraction);
    }
    if(deltaPosition > (uint16_t)PTO_ABS2QEP_ABS_MAX_DELTA_PER_SAMPLE)
    {
        deltaPosition = (uint16_t)PTO_ABS2QEP_ABS_MAX_DELTA_PER_SAMPLE;
    }
    if(setDirection == PTO_ABS2QEP_CLOCKWISE_PTO)
    {
        //
        // Move absolute position clockwise
        //
        currentPosition = currentPosition + (uint32_t)deltaPosition;
        currentPosition = currentPosition > PTO_ABS2QEP_ABS_MAX_POSITION ?     \
                currentPosition - PTO_ABS2QEP_ABS_MAX_POSITION :               \
                currentPosition;
    }
    else
    {
        //
        // Move absolute position counter-clockwise
        //
        currentPosition = (uint32_t)deltaPosition > currentPosition ?          \
                PTO_ABS2QEP_ABS_MAX_POSITION + currentPosition -               \
                (uint32_t)deltaPosition :                                      \
                currentPosition - (uint32_t)deltaPosition;
    }
    return(currentPosition);
}

      






//
// End of File
//


//
// End of File
//

