/*
 * tformat.h
 *
 *  Created on: 2024 Oct 17
 *      Author: Gejian
 */

#ifndef _TFORMAT_SYSTEM_EXAMPLE_H
#define _TFORMAT_SYSTEM_EXAMPLE_H
/*
 * If building with a C++ compiler,
 * make all of the definitions in this header have a C binding.
 */
#ifdef __cplusplus
extern "C"
{
#endif

#include "PM_tformat_include.h"
#include "driverlib.h"
#include "device.h"
#include "board_cfg.h"

/*
 * Indicate which commands should be run.
 * 1 == run command
 * 0 == skip command
 */
#define TFORMAT_RUN_ID0     1
#define TFORMAT_RUN_ID1     1
#define TFORMAT_RUN_ID2     1
#define TFORMAT_RUN_ID3     1
#define TFORMAT_RUN_IDD_ID6 1  // EEPROM read and write
#define TFORMAT_RUN_ID7     1
#define TFORMAT_RUN_ID8     1
#define TFORMAT_RUN_IDC     1

/*
 * The duration of a single bit is 2.5 MHz (400ns).
 * Define the number of CLB clocks required to generate for the bus clock
 * Define the number of CLB clocks within a bit-width
 *
 * This macro of TFORMAT_FREQ_DIVIDER definition can
 * be used as an argument to the function of PM_tformat_setFreq
 *
 */
#define TFORMAT_CLB_CLOCK       250u		// MHz

#define TFORMAT_BUS_CLOCK       2.5f		// MHz
#define TFORMAT_FREQ_DIVIDER    (TFORMAT_CLB_CLOCK /                           \
                                    (uint16_t) (2.0f * TFORMAT_BUS_CLOCK) )
#define TFORMAT_BIT_WIDTH       (TFORMAT_CLB_CLOCK /                           \
                                    (uint16_t) (TFORMAT_BUS_CLOCK) )

/*
 * @brief For EEPROM read/write testing,specify the range of addresses to be used
 *
 * Note: Select addresses between 0 and 126 (0x7E).
 * 		 Address 127 (0x7F) is the EEPROM page control.
 */
#define TFORMAT_EEPROM_TEST_ADDR_START  0x73u
#define TFORMAT_EEPROM_TEST_ADDR_END    0x7Eu

/*
 * @brief These macro to delay
 *
 * Delay time, in micro-seconds, between each command sent
 * Delay time, in micro-seconds, between each command sent
 *
 * Time to wait after a write to the EEPROM before another write/read
 * Power-on to normal operation delay 1.5 seconds.
 *
 * Note:Refer to the data manual for a specific encoder.
 */
#define TFORMAT_IDLE_DELAY_US_START			10ul
#define TFORMAT_IDLE_DELAY_US_INCREMENT		15ul
#define TFORMAT_EEPROM_BUSY_US				18000ul
#define TFORMAT_POWER_ON_DELAY_US			1500000ul

/*
 * Global structures used to share information to the user through
 * the Code Composer Studio expressions window.
 */

/*
 * @brief TFORMAT_ENCODER_INFO_STRUCT_SNIPPET
 *
 * API function: PM_tformat_updatePositionOrTurns()
 * 				 PM_tformat_updateEncoderID()
 */
typedef struct PM_tformat_encoderStruct {
    uint32_t  position;
    uint32_t  turns;
    uint16_t  id;
} PM_tformat_encoderStruct;
extern PM_tformat_encoderStruct encoderData;

/*
 * @brief TFORMAT_ENCODER_ERROR_STRUCT_SNIPPET
 *
 * System example functions:tformat_exCommands()
 * 							tformat_checkError()
 */
typedef struct PM_tformat_errorStruct {
    uint16_t  idCheckError;
    uint16_t  crcCheckError;
    uint16_t  eepromAddressError;
    uint16_t  eepromDataError;
} PM_tformat_errorStruct;
extern PM_tformat_errorStruct errorData;

/*
 * @brief TFORMAT_COMMAND_STRUCT_SNIPPET
 *
 * System example functions:tformat_testEEPROMCommands()
 * 							tformat_exCommands()
 * 							tformat_checkError()
 */
typedef struct PM_tformat_commandStruct {
    uint32_t  controlField;
    uint32_t  eepromAddress;
    uint16_t  eepromData;
} PM_tformat_commandStruct;
extern PM_tformat_commandStruct commandData;

/*
 * @brief An array for storing data in a SPIRX
 */
extern volatile uint32_t tformatSpiRxIsrTicker;

/*
 * @brief Functions to demonstrate usage of the T-Format encode interface library
 */

/*
 * @brief Execute read-out and reset commands
 *
 * This function is the main test for the system example.
 * It tests the following command IDs: ID0, ID1, ID2, ID3, ID7, ID8 and IDC
 */
void tformat_exCommands(void) __attribute__ ((section (".RamFunc")))
                              __attribute__((noreturn));

/*
 * @brief Execute EEPROM read and write commands
 *
 * Test read (ID6) and write (IDD) EEPROM commands.
 */
void tformat_testEEPROMCommands(void) __attribute__ ((section (".RamFunc")));

/*
 * @brief Basic error check for received fields
 *
 */
uint16_t tformat_checkError(uint16_t controlFieldID) __attribute__ ((section (".RamFunc")));

/*
 * @brief Service the SPI FIFO receive interrupt
 */
void tformat_spiRxISR(void) __attribute__ ((section (".RamFunc")));

/*
 * @brief Initialize the T-Format subsystem
 */
void tformat_init(void);

/*
 * @brief Configure the SPI peripheral
 *
 * Configures the PM_TFORMAT_SPI peripheral
 * as required for T-Format communication
 */
void tformat_initSPIFIFO(uint32_t devLSPCLKFreq);


#ifdef __cplusplus
}
#endif

#endif /* PM_TFORMAT_SYSTEM_EXAMPLE_H */

/*
 * end of file
 */

