/*
 * tformat.c
 *
 *  Created on: 2024 Oct 17
 *      Author: Gejian
 */

#include "tformat.h"
#include "board.h"
#include "printf.h"

void tformat_init(void)
{
#if defined(PM_TFORMAT_CRC_TABLE)
	/*
	 * @brief If the CPU will be generating CRC,
	 * then initialize the lookup table.
	 */

    PM_tformat_generateCRCTable(PM_TFORMAT_NBITS_POLY,
                                PM_TFORMAT_POLY,
                                tformatCRCtable);
#endif

    /*
     * @brief Configure the CLB tile(s) for T-format
     */
    PM_tformat_setupPeriph();
	PM_tformat_setFreq(TFORMAT_FREQ_DIVIDER);

    /*
     * @brief Initializes the pin peripheral
     */
    GPIO_init();
    PinMux_init();
    CLB_OUTPUTXBAR_init();

    /*
     * @brief Initializes SPI
     */
    tformat_initSPIFIFO(DEVICE_APBCLK_FREQ);

    /*
     * @brief SPI interrupt
     */
    Interrupt_register(PM_TFORMAT_INT_SPI_RX, &tformat_spiRxISR);
    Interrupt_enable(PM_TFORMAT_INT_SPI_RX);
    SPI_disableAllInterrupt(PM_TFORMAT_SPI);
    SPI_clearAllInterruptStatus(PM_TFORMAT_SPI);
    SPI_enableInterrupt(PM_TFORMAT_SPI, SPI_INT_RXFF);
    SPI_enableModule(PM_TFORMAT_SPI);

    /*
     * @brief Power up the T-Format 5v supply
     */
    GPIO_writePin(TFORMAT_GPIO_PWRCTRL, 1);
    GPIO_writePin(TFORMAT_SPI_CS, 1);
    DEVICE_DELAY_US(TFORMAT_POWER_ON_DELAY_US);
}

void tformat_spiRxISR(void)
{
    uint16_t i;

	tformatSpiRxIsrTicker++;
	//#pragma MUST_ITERATE(5, 12, 1)
	for (i = 0; i < tformatData.fifoLevel; i++)
	{
		tformatRxData[i] = SPI_readDataNonBlocking(PM_TFORMAT_SPI) & 0xFF;
	}
	tformatData.dataReady = TRUE;

    //SPI_clearAllInterruptStatus(PM_TFORMAT_SPI);
}

void
tformat_initSPIFIFO(uint32_t devLSPCLKFreq)
{
    SPI_disableModule(PM_TFORMAT_SPI);
    SPI_disableLoopback(PM_TFORMAT_SPI);
    SPI_setConfig(PM_TFORMAT_SPI, devLSPCLKFreq, SPI_PROT_POL1PHA1,
                SPI_MODE_PERIPHERAL, 2500000, PM_TFORMAT_FIFO_WORD_BITS);

    SPI_clearAllInterruptStatus(PM_TFORMAT_SPI);
    SPI_setFIFOInterruptLevel(PM_TFORMAT_SPI, SPI_FIFO_TX0, SPI_FIFO_RX15);
    SPI_enableModule(PM_TFORMAT_SPI);
}

uint16_t
tformat_checkError(uint16_t controlFieldID)
{
    uint16_t errorFlag = PM_TFORMAT_PASS;

    //
    // Check sent/received controlField (data ID)
    // Check calculated CRC against received CRC
    //
    if(commandData.controlField != tformatData.controlField)
    {
        errorFlag = 1;
        errorData.idCheckError = 1;
    }
    if(tformatData.crcCheck != tformatData.crcField)
    {
        errorFlag = 1;
        errorData.crcCheckError = 1;
    }

    //
    // If this was an EEPROM write:
    // - Check sent Address against received Address (ADF)
    // - Check sent Data against received Data (EDF)
    //
    if(controlFieldID == PM_TFORMAT_CFID6)
    {
        if(commandData.eepromAddress != tformatData.eepromAddressField)
        {
            errorFlag = 1;
            errorData.eepromAddressError = 1;
        }

        if(commandData.eepromData != tformatData.eepromWrDataField)
        {
            errorFlag = 1;
            errorData.eepromDataError = 1;
        }
    }

    //
    // If this was an EEPROM read:
    // - Check sent Address against received Address (ADF)
    // - Check expected data against received data (EDF)
    //
    if(controlFieldID == PM_TFORMAT_CFIDD)
    {
        if(commandData.eepromAddress != tformatData.eepromAddressField)
        {
            errorFlag = 1;
            errorData.eepromAddressError = 1;
        }
        if(commandData.eepromData != tformatData.eepromRdDataField)
        {
            errorFlag = 1;
            errorData.eepromDataError = 1;
        }
    }

    return(errorFlag);
}

void
tformat_exCommands(void)
{
    commandData = (struct PM_tformat_commandStruct) {0};

    /*
     * @brief General communication process
     *
     * 1) Setup the command
     * 2) Start the operation (transmission)
     * 3) Wait for the response to be received by the SPI.
     * 	  Receive in SPI interrupt or other way,
     * 	  Receiving is to ensure that the SPI has received enough data
     * 4) Unpack, or receive, the data.
     * 5) System specific: check and respond to errors
     * 6) Update any encoder parameters (position, turns, ID)
     *
     */

    while(1)
    {

    #if TFORMAT_RUN_ID0

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFID0;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START);

        /*
         * @brief Setup the command ID0
         *
         *
         */
        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_ID0,
                                PM_TFORMAT_RX_CLOCKS_ID0,
                                PM_TFORMAT_RX_FIELDS_ID0,
                                PM_TFORMAT_TX_CLOCKS_ID0,
                                PM_TFORMAT_FIFO_LEVEL_ID0);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID0_1_7_8_C();
        if(tformat_checkError(PM_TFORMAT_CFID0) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

        /*
         * @brief Update position
         */
        encoderData.position =
                PM_tformat_updatePositionOrTurns(tformatData.dataField0,
                                                 tformatData.dataField1,
                                                 tformatData.dataField2);

    #endif

    #if TFORMAT_RUN_ID1

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFID1;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START
                        + TFORMAT_IDLE_DELAY_US_INCREMENT);

        /*
         * @brief Setup the command ID1
         */

        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_ID1,
                                PM_TFORMAT_RX_CLOCKS_ID1,
                                PM_TFORMAT_RX_FIELDS_ID1,
                                PM_TFORMAT_TX_CLOCKS_ID1,
                                PM_TFORMAT_FIFO_LEVEL_ID1);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID0_1_7_8_C();
        if(tformat_checkError(PM_TFORMAT_CFID1) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

        /*
         * @brief Update position
         */
        encoderData.turns =
            PM_tformat_updatePositionOrTurns(tformatData.dataField0,
                                            tformatData.dataField1,
                                            tformatData.dataField2);

    #endif
    #if TFORMAT_RUN_ID7

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFID7;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START
                        + 2ul * TFORMAT_IDLE_DELAY_US_INCREMENT);

        /*
         * @brief Setup the command ID7
         */
        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_ID7,
                                PM_TFORMAT_RX_CLOCKS_ID7,
                                PM_TFORMAT_RX_FIELDS_ID7,
                                PM_TFORMAT_TX_CLOCKS_ID7,
                                PM_TFORMAT_FIFO_LEVEL_ID7);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID0_1_7_8_C();
        if(tformat_checkError(PM_TFORMAT_CFID7) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

        /*
         * @brief Update position
         */
        encoderData.position =
            PM_tformat_updatePositionOrTurns(tformatData.dataField0,
                                            tformatData.dataField1,
                                            tformatData.dataField2);

    #endif

    #if TFORMAT_RUN_ID8

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFID8;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START
                        + 3ul * TFORMAT_IDLE_DELAY_US_INCREMENT);

        /*
         * @brief Setup the command ID8
         */
        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_ID8,
                                PM_TFORMAT_RX_CLOCKS_ID8,
                                PM_TFORMAT_RX_FIELDS_ID8,
                                PM_TFORMAT_TX_CLOCKS_ID8,
                                PM_TFORMAT_FIFO_LEVEL_ID8);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID0_1_7_8_C();
        if(tformat_checkError(PM_TFORMAT_CFID8) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

		/*
		 * @brief Update position
		 */
        encoderData.position =
            PM_tformat_updatePositionOrTurns(tformatData.dataField0,
                                            tformatData.dataField1,
                                            tformatData.dataField2);

    #endif

    #if TFORMAT_RUN_IDC

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFIDC;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START
                        + 5ul * TFORMAT_IDLE_DELAY_US_INCREMENT);

        /*
         * @brief Setup the command IDC
         */
        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_IDC,
                                PM_TFORMAT_RX_CLOCKS_IDC,
                                PM_TFORMAT_RX_FIELDS_IDC,
                                PM_TFORMAT_TX_CLOCKS_IDC,
                                PM_TFORMAT_FIFO_LEVEL_IDC);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID0_1_7_8_C();
        if(tformat_checkError(PM_TFORMAT_CFIDC) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

		/*
		 * @brief Update position
		 */
        encoderData.position =
        PM_tformat_updatePositionOrTurns(tformatData.dataField0,
                                    tformatData.dataField1,
                                    tformatData.dataField2);

    #endif

    #if TFORMAT_RUN_ID2

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFID2;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START
                        + 6ul * TFORMAT_IDLE_DELAY_US_INCREMENT);

        /*
         * @brief Setup the command ID2
         */
        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_ID2,
                                PM_TFORMAT_RX_CLOCKS_ID2,
                                PM_TFORMAT_RX_FIELDS_ID2,
                                PM_TFORMAT_TX_CLOCKS_ID2,
                                PM_TFORMAT_FIFO_LEVEL_ID2);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE);
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID2();
        if(tformat_checkError(PM_TFORMAT_CFID2) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

		/*
		 * @brief Update Encode ID
		 */
        encoderData.id = PM_tformat_updateEncoderID(tformatData.dataField0);

    #endif

    #if TFORMAT_RUN_ID3

        tformatData = (struct PM_tformat_DataStruct) {0};
        commandData.controlField = PM_TFORMAT_CFID3;
        DEVICE_DELAY_US(TFORMAT_IDLE_DELAY_US_START
                        + 7ul * TFORMAT_IDLE_DELAY_US_INCREMENT);

        /*
         * @brief Setup the command ID3
         */
        PM_tformat_setupCommandReadoutOrReset(PM_TFORMAT_ID3,
                                PM_TFORMAT_RX_CLOCKS_ID3,
                                PM_TFORMAT_RX_FIELDS_ID3,
                                PM_TFORMAT_TX_CLOCKS_ID3,
                                PM_TFORMAT_FIFO_LEVEL_ID3);

        /*
         * @brief Start transmission
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for the response to be received by the SPI.
         */
        while(tformatData.dataReady != TRUE);
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

        /*
         * @brief Unpack, or receive, the data.
         * 		  System specific: check and respond to errors
         */
        PM_tformat_receiveDataID3();
        if(tformat_checkError(PM_TFORMAT_CFID3) != PM_TFORMAT_PASS)
        {
            ESTOP0;
        }

		/*
		 * @brief Update position
		 */
        encoderData.position =
            PM_tformat_updatePositionOrTurns(tformatData.dataField0,
                                            tformatData.dataField1,
                                            tformatData.dataField2);
		/*
		 * @brief Update turns
		 */
        encoderData.turns =
            PM_tformat_updatePositionOrTurns(tformatData.dataField4,
                                            tformatData.dataField5,
                                            tformatData.dataField6);

		/*
		 * @brief Update Encode ID
		 */
        encoderData.id =
            PM_tformat_updateEncoderID(tformatData.dataField3);

    #endif

    }  // end while(1)
}

void
tformat_testEEPROMCommands(void)
{
    uint16_t eepromData;
    uint16_t eepromAddress;
    uint16_t returnStatus;
    uint16_t i;

    /*
     * @brief Write 0 to the EEPROM Addresses
     */
    eepromData = 0u;
    eepromAddress = TFORMAT_EEPROM_TEST_ADDR_START;

    while(eepromAddress <= TFORMAT_EEPROM_TEST_ADDR_END)
    {
    	/*
    	 * @brief initialize tformatData structure
    	 *
    	 */
        tformatData = (struct PM_tformat_DataStruct) {0};

        /*
         * @brief Next 3 lines are for debug and error checking
         *
         */
        commandData.eepromAddress = eepromAddress;
        commandData.eepromData = eepromData;
        commandData.controlField = PM_TFORMAT_CFID6;

        /*
         * @brief Pack for ID6
         *
         */
        PM_tformat_setupCommandWriteEEPROM(eepromAddress, eepromData);

        /*
         * @brief Initiate communication
         *
         */
    	GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for enough data to be received
         *
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

    	/*
    	 * @brief unpack for ID6
    	 *
    	 * Calculates the CRC verification code.
    	 * Fill RxDATAs and CRC value into tformatData structure
    	 *
    	 */

        returnStatus = PM_tformat_receiveDataID6();

        if(returnStatus != PM_TFORMAT_EEPROM_BUSY)
        {
            if(tformat_checkError(PM_TFORMAT_CFID6) != PM_TFORMAT_PASS)
            {


                ESTOP0;
            }

            eepromAddress++;
        }

        /*
         * @brief Wait some time and try the write again
         */
        else
        {
            DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
        }
    }

    /*
     * @brief Read from the EEPROM
     */
    DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
    eepromData = 0u;
    eepromAddress = TFORMAT_EEPROM_TEST_ADDR_START;

    while(eepromAddress <= TFORMAT_EEPROM_TEST_ADDR_END)
    {
    	/*
    	 * @brief initialize tformatData structure
    	 *
    	 */
        tformatData = (struct PM_tformat_DataStruct) {0};

        /*
         * @brief Next 2 lines are for debug and error checking
         *
         */
        commandData.eepromAddress = eepromAddress;
        commandData.eepromData = eepromData;
        commandData.controlField = PM_TFORMAT_CFIDD;

        /*
         * @brief Pack for IDD
         *
         */
        PM_tformat_setupCommandReadEEPROM(eepromAddress);


        /*
         * @brief Initiate communication
         *
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for enough data to be received
         *
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

    	/*
    	 * @brief unpack for IDD
    	 *
    	 * Calculates the CRC verification code.
    	 * Fill RxDATAs and CRC value into tformatData structure
    	 *
    	 */
        returnStatus = PM_tformat_receiveDataIDD();

        if(returnStatus != PM_TFORMAT_EEPROM_BUSY)
        {
            if(tformat_checkError(PM_TFORMAT_CFIDD) != PM_TFORMAT_PASS)
            {
                ESTOP0;
            }
            eepromAddress++;
        }

        /*
         * @brief Wait some time and try the write again
         */
        else
        {
            DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
        }
    }

    /*
     * @brief Write data == 0xFF, FE, FD .. pattern
     */
    DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
    eepromData = 0xFFu;
    eepromAddress = TFORMAT_EEPROM_TEST_ADDR_START;

    while(eepromAddress <= TFORMAT_EEPROM_TEST_ADDR_END)
    {
    	/*
    	 * @brief initialize tformatData structure
    	 *
    	 */
        tformatData = (struct PM_tformat_DataStruct) {0};

        /*
         * @brief Next 3 lines are for debug and error checking
         *
         */
        commandData.eepromAddress = eepromAddress;
        commandData.eepromData = eepromData;
        commandData.controlField = PM_TFORMAT_CFID6;

        /*
         * @brief Pack for ID6
         *
         */
        PM_tformat_setupCommandWriteEEPROM(eepromAddress, eepromData);

        /*
         * @brief Initiate communication
         *
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for enough data to be received
         *
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

    	/*
    	 * @brief unpack for ID6
    	 *
    	 * Calculates the CRC verification code.
    	 * Fill RxDATAs and CRC value into tformatData structure
    	 *
    	 */
        returnStatus = PM_tformat_receiveDataID6();

        if(returnStatus != PM_TFORMAT_EEPROM_BUSY)
        {
            if(tformat_checkError(PM_TFORMAT_CFID6) != PM_TFORMAT_PASS)
            {
                ESTOP0;
            }
            eepromData--;
            eepromAddress++;
        }

        /*
         * @brief Wait some time and try the write again
         */
        else
        {
            DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
        }
    }

    /*
     * @brief Read from the EEPROM
     */
    DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
    eepromData = 0xFFu;
    eepromAddress = TFORMAT_EEPROM_TEST_ADDR_START;

    while(eepromAddress <= TFORMAT_EEPROM_TEST_ADDR_END)
    {
    	/*
    	 * @brief initialize tformatData structure
    	 *
    	 */
        tformatData = (struct PM_tformat_DataStruct) {0};

        /*
         * @brief Next 2 lines are for debug and error checking
         *
         */
        commandData.eepromAddress = eepromAddress;
        commandData.eepromData = eepromData;
        commandData.controlField = PM_TFORMAT_CFIDD;

        /*
         * @brief Pack for IDD
         *
         */
        PM_tformat_setupCommandReadEEPROM(eepromAddress);

        /*
         * @brief Initiate communication
         *
         */
        GPIO_writePin(TFORMAT_SPI_CS, 0);
        PM_tformat_startOperation();

        /*
         * @brief Wait for enough data to be received
         *
         */
        while(tformatData.dataReady != TRUE) {}
    	GPIO_writePin(TFORMAT_SPI_CS, 1);

    	/*
    	 * @brief unpack for IDD
    	 *
    	 * Calculates the CRC verification code.
    	 * Fill RxDATAs and CRC value into tformatData structure
    	 *
    	 */
        returnStatus = PM_tformat_receiveDataIDD();

        if(returnStatus != PM_TFORMAT_EEPROM_BUSY)
        {
            if(tformat_checkError(PM_TFORMAT_CFIDD) != PM_TFORMAT_PASS)
            {
            	while(1);
                ESTOP0;
            }
            eepromData--;
            eepromAddress++;
        }

        /*
         * @brief Wait some time and try the write again
         */
        else
        {
            DEVICE_DELAY_US(TFORMAT_EEPROM_BUSY_US);
        }
    }
}

//
// End of file
//
