/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    gs32flash.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "flash_programming_gs32.h"
#include "gs32_flashAPI_lib.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/* MIN,MAX macros */
#ifndef MIN
#define MIN(a, b) (((a) < (b)) ? (a) : (b))
#endif
#ifndef MAX
#define MAX(a, b) (((a) > (b)) ? (a) : (b))
#endif


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/* None */

/**
 * \brief   "gs32 flash api init"
 *
 * \param  device system clock,Generally, it is DEVICE_SYSCLK_FREQ
 *
 *
 *
 * \retval  None
 */
RAMFUNC_T uint32 GS32Fapi_Init(uint32 inu32SysClk)
{
	volatile uint32 oReturnCheck = Fapi_Status_Success;
	volatile uint32 u32EfcCount = Fapi_GetEfcCount();
	volatile uint32 u32FlashBank = 0;

	Flash_initModule(0, 0, inu32SysClk);

    oReturnCheck = Fapi_initializeAPI( (Fapi_FmcRegistersType *)0,inu32SysClk/1000000U);
    if(oReturnCheck != Fapi_Status_Success)
    {
        return oReturnCheck;
    }

    for (u32FlashBank=0; u32FlashBank<u32EfcCount; u32FlashBank++)
    {
        oReturnCheck = Fapi_setActiveFlashBank(u32FlashBank);

        while (Fapi_checkFsmForReady(u32FlashBank) != Fapi_Status_FsmReady){}
        if(oReturnCheck != Fapi_Status_Success)
    	{

    		return oReturnCheck;
    	}
    }

	return oReturnCheck;
}

/**
 * \brief   "gs32 flash api erase one sector"
 *
 * \param   flash sector memory base addr,the example is
 *			Bzero_Sector0_start,include csxx_sector.h header file.
 *
 *
 * \retval  error code,For details, see Fapi_StatusType
 */
RAMFUNC_T uint32 GS32Fapi_EraseSector(uint32 inu32SectorAddr)
{
	volatile Fapi_StatusType  oReturnCheck = Fapi_Status_Success;
	volatile Fapi_FlashStatusWordType  oFlashStatusWord = {0};
	volatile Fapi_FlashBankType eBankId = 0;
	volatile Fapi_FlashStatusType oFlashStatus = 0;

#if IS_GS32F00xx(0x30)
	volatile uint32_t uPageCount = 2;
	volatile uint32_t uIndex = 0;

	if ( 0 != Fapi_isAddressMain(inu32SectorAddr))
	{
		uPageCount = 2;
	}
	else
	{
		uPageCount = 1;
	}

	if ( (inu32SectorAddr % (FlashMainPageSize*uPageCount)) != 0)
	{
		return Fapi_Error_InvalidAddress;
	}

	for ( uIndex=0; uIndex<uPageCount; uIndex++ )
	{
		eBankId = Fapi_GetIndexByEraseAddr(inu32SectorAddr+FlashMainPageSize*uIndex);
		if ( 0xdeadbeef == eBankId)
		{
			return Fapi_Error_InvalidAddress;
		}

		oReturnCheck = Fapi_issueAsyncCommand(eBankId, Fapi_ClearMore);
		if(oReturnCheck != Fapi_Status_Success)
		{
			return oReturnCheck;
		}


		oReturnCheck = Fapi_issueAsyncCommandWithAddress(eBankId, Fapi_EraseSector,  (0==Fapi_GetBankMode()) ? (uint32 *)inu32SectorAddr : (uint32 *)(inu32SectorAddr+FlashMainPageSize*uIndex));
		while (Fapi_checkFsmForReady(eBankId) != Fapi_Status_FsmReady){}
		if(oReturnCheck != Fapi_Status_Success)
		{
			return oReturnCheck;
		}

		oFlashStatus = Fapi_getFsmStatus(eBankId);
		if ( (FLASH_CONTROL_ERASE_OKAY != oFlashStatus) && (0x64 != oFlashStatus)  )
		{
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return oReturnCheck;
		}
	}

	MInvalDCacheLines(inu32SectorAddr, (FlashMainPageSize*uPageCount)/32);
	oReturnCheck = Fapi_doBlankCheck(eBankId, (uint32 *)inu32SectorAddr, (FlashMainPageSize*uPageCount)/sizeof(uint32), (Fapi_FlashStatusWordType *)&oFlashStatusWord);
	if ( oReturnCheck != Fapi_Status_Success )
	{
		return oReturnCheck;
	}

#else
	eBankId = Fapi_GetIndexByAddr(inu32SectorAddr);
	if ( 0xdeadbeef == eBankId)
	{
		return Fapi_Error_InvalidAddress;
	}

	oReturnCheck = Fapi_issueAsyncCommand(eBankId, Fapi_ClearMore);
	if(oReturnCheck != Fapi_Status_Success)
	{
		return oReturnCheck;
	}

	oReturnCheck = Fapi_issueAsyncCommandWithAddress(eBankId, Fapi_EraseSector, (uint32 *)inu32SectorAddr );
	while (Fapi_checkFsmForReady(eBankId) != Fapi_Status_FsmReady){}
	if(oReturnCheck != Fapi_Status_Success)
	{
		return oReturnCheck;
	}

	oFlashStatus = Fapi_getFsmStatus(eBankId);
	if ( FLASH_CONTROL_ERASE_OKAY != oFlashStatus  )
	{
		oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
		return oReturnCheck;
	}

#if IS_GS32F00xx(0x12)
		MInvalDCacheLines(inu32SectorAddr, FlashMainPageSize/32);
#elif IS_GS32F3xx(0x22)
		MInvalDCacheLines(inu32SectorAddr, FlashMainPageSize/64);
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif
	oReturnCheck = Fapi_doBlankCheck(eBankId, (uint32 *)inu32SectorAddr, FlashMainPageSize/sizeof(uint32), (Fapi_FlashStatusWordType *)&oFlashStatusWord);
	if ( oReturnCheck != Fapi_Status_Success )
	{
		return oReturnCheck;
	}
#endif
	return oReturnCheck;
}

/**
 * \brief   "gs32 flash api erase multiple sectors"
 *
 * \param   flash sector memory base addr
 * \param   Desired erase length,lenth of 32 bit words count.
 *          It is a multiple of FlashMainPageSize/4 frames
 *
 *
 * \retval  error code,For details, see Fapi_StatusType
 */
RAMFUNC_T uint32 GS32Fapi_EraseRegion(uint32 inu32Address, uint32 inu32LenthIn32BitsWords)
{
	volatile Fapi_StatusType  oReturnCheck = Fapi_Status_Success;
	volatile Fapi_FlashStatusType  oFlashStatus = Fapi_Status_Success;
	volatile Fapi_FlashStatusWordType  oFlashStatusWord = {0};

	volatile uint32 u32PageSize = FlashMainPageSize;
	volatile uint32 u32EraseAddr = inu32Address;
	volatile uint32 u32EraseLenth = inu32LenthIn32BitsWords*sizeof(uint32);

	volatile Fapi_FlashBankType eBankId = 0;

#if IS_GS32F00xx(0x30)
	volatile uint32_t uPageCount = 0;
	volatile uint32_t uIndex = 0;
	if( 0 == Fapi_GetBankMode() )
	{
		if ( 0 != Fapi_isAddressMain(inu32Address))
		{
			uPageCount = 2;
		}
		else
		{
			uPageCount = 1;
		}
	}
	else
	{
		uPageCount = 1;
	}

	if ( 0 != ( u32EraseLenth % (u32PageSize*uPageCount) ) )
	{
		return Fapi_Error_AsyncIncorrectDataBufferLength;
	}

	for ( ; ; )
	{
		eBankId = Fapi_GetIndexByEraseAddr(u32EraseAddr);
		if ( 0xdeadbeef == eBankId)
		{
			return Fapi_Error_InvalidAddress;
		}

		oReturnCheck = Fapi_issueAsyncCommand(eBankId, Fapi_ClearMore);
		if(oReturnCheck != Fapi_Status_Success)
		{
			return oReturnCheck+50;
		}

		oReturnCheck = Fapi_issueAsyncCommandWithAddress(eBankId, Fapi_EraseSector, (uint32 *)u32EraseAddr );
		while (Fapi_checkFsmForReady(eBankId) != Fapi_Status_FsmReady){}

		if(oReturnCheck != Fapi_Status_Success)
		{
			return oReturnCheck;
		}

		oFlashStatus = Fapi_getFsmStatus(eBankId);
		if ( (FLASH_CONTROL_ERASE_OKAY != oFlashStatus) && (0x64 != oFlashStatus)  )
		{
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return oReturnCheck;
		}

		u32EraseAddr += u32PageSize;
		u32EraseLenth -= u32PageSize;

		if ( 0 == u32EraseLenth )
		{
			break;
		}
	}

	MInvalDCacheLines(inu32Address, ((inu32LenthIn32BitsWords*sizeof(uint32_t))/32)+1);
	oReturnCheck = Fapi_doBlankCheck(Fapi_FlashBank0, (uint32 *)inu32Address, inu32LenthIn32BitsWords, (Fapi_FlashStatusWordType *)&oFlashStatusWord);
	if ( oReturnCheck != Fapi_Status_Success )
	{
		 return oReturnCheck;
	}

#else
	if ( 0 != ( u32EraseLenth % u32PageSize ) )
	{
		return Fapi_Error_AsyncIncorrectDataBufferLength;
	}

	for ( ; ; )
	{
		eBankId = Fapi_GetIndexByAddr(u32EraseAddr);
		if ( 0xdeadbeef == eBankId)
		{
			return Fapi_Error_InvalidAddress;
		}

		oReturnCheck = Fapi_issueAsyncCommand(eBankId, Fapi_ClearMore);
		if(oReturnCheck != Fapi_Status_Success)
		{
			return oReturnCheck;
		}

		oReturnCheck = Fapi_issueAsyncCommandWithAddress(eBankId, Fapi_EraseSector, (uint32 *)u32EraseAddr );
		while (Fapi_checkFsmForReady(eBankId) != Fapi_Status_FsmReady){}
		if(oReturnCheck != Fapi_Status_Success)
		{
			return oReturnCheck;
		}

		oFlashStatus = Fapi_getFsmStatus(eBankId);
		if ( FLASH_CONTROL_ERASE_OKAY != oFlashStatus  )
		{
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			return oReturnCheck;
		}

#if IS_GS32F00xx(0x12)
		MInvalDCacheLines(u32EraseAddr, (FlashMainPageSize/32)+1);
#elif IS_GS32F3xx(0x22)
		MInvalDCacheLines(u32EraseAddr, (FlashMainPageSize/64)+1);
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif
		oReturnCheck = Fapi_doBlankCheck(eBankId, (uint32 *)u32EraseAddr, FlashMainPageSize/sizeof(uint32), (Fapi_FlashStatusWordType *)&oFlashStatusWord);
		if ( oReturnCheck != Fapi_Status_Success )
		{
			 return oReturnCheck;
		}

		u32EraseAddr += u32PageSize;
		u32EraseLenth -= u32PageSize;

		if ( 0 == u32EraseLenth )
		{
			break;
		}
	}
#endif
	return oReturnCheck;
}

/**
 * \brief   "gs32 flash api program data to flash"
 *
 * \param   flash addr of memory
 * \param   data addr of memory
 * \param   lenth of 32-bit count,It must be a multiple of 4
 *
 * \retval  error code,For details, see Fapi_StatusType
 */

extern RAMFUNC_T uint32 Fapi_getWriteAddByAddr(Fapi_FlashBankType iFlashBank, uint32 Address);

RAMFUNC_T uint32 GS32Fapi_Program(uint32 inu32Address, uint32 inu32DataAddr,uint32 inu32LenthIn32BitsWords)
{
	volatile uint32 u32Index = 0;
	volatile uint32 i = 0;
	volatile Fapi_StatusType  oReturnCheck = Fapi_Status_Success;
	volatile Fapi_FlashStatusType  oFlashStatus;
	volatile Fapi_FlashStatusWordType  oFlashStatusWord = {0};

	volatile Fapi_FlashBankType eBankId = 0;

	volatile uint32 FlashStartAddr = inu32Address;
	volatile uint32 u32ProgramWords = FlashProgramBuffWordsOf16Bit;

	volatile uint32 *pDataBuff = (uint32 *)inu32DataAddr;

#if IS_GS32F00xx(0x30)
	volatile uint32_t uPageCount = 0;

	if( 0 == Fapi_GetBankMode() )
	{
		if ( 0 != Fapi_isAddressMain(inu32Address))
		{
			uPageCount = 2;
		}
		else
		{
			uPageCount = 1;
		}
	}
	else
	{
		uPageCount = 1;
	}

	if ( 0 != (inu32LenthIn32BitsWords%((u32ProgramWords*uPageCount)/sizeof(uint16))) )
	{
		return Fapi_Error_AsyncIncorrectDataBufferLength;
	}

	if ( 0 != (FlashStartAddr%(u32ProgramWords*uPageCount*sizeof(uint16))) )
	{
		return Fapi_Error_InvalidAddress;
	}
#else
	if ( 0 != (inu32LenthIn32BitsWords%(u32ProgramWords/sizeof(uint16))) )
	{
		return Fapi_Error_AsyncIncorrectDataBufferLength;
	}
#endif

	u32Index = FlashStartAddr;
	do
	{
		eBankId = Fapi_GetIndexByAddr(u32Index);
		if ( 0xdeadbeef == eBankId)
		{
			oReturnCheck = Fapi_Error_InvalidAddress;
			goto program_return;
		}

		oReturnCheck = Fapi_issueProgrammingCommand(eBankId, (uint32 *)u32Index, (uint16 *)(pDataBuff+i), u32ProgramWords, 0, 0, Fapi_AutoEccGeneration);
		while(Fapi_checkFsmForReady(eBankId) == Fapi_Status_FsmBusy);
		if(oReturnCheck != Fapi_Status_Success)
		{
			goto program_return;
		}

		oFlashStatus = Fapi_getFsmStatus(eBankId);
		if( (oFlashStatus != FLASH_CONTROL_ERASE_OKAY) && (oFlashStatus != 0x64) )
		{
			oReturnCheck = Fapi_Error_FlashControlExecuteCommandError;
			goto program_return;
		}

#if IS_GS32F00xx(0x12,0x30)
		MInvalDCacheLines(u32Index, ((u32ProgramWords*sizeof(uint16))/32) + 1 );
#elif IS_GS32F3xx(0x22)
		MInvalDCacheLines(u32Index, ((u32ProgramWords*sizeof(uint16))/64) + 1 );
#else
#error "Use the GS32F00xx or GS32F3xx macro to define the specific chip model."
#endif
		oReturnCheck = Fapi_doVerify((uint32 *)u32Index,
								u32ProgramWords/sizeof(uint16),
								(uint32 *)(pDataBuff+i),
								(Fapi_FlashStatusWordType *)&oFlashStatusWord);
		if(oReturnCheck != Fapi_Status_Success)
		{
			goto program_return;
		}

		u32Index += u32ProgramWords*sizeof(uint16);
		i += u32ProgramWords/sizeof(uint16);

	}while( u32Index < (FlashStartAddr + inu32LenthIn32BitsWords*sizeof(uint32)) );

program_return:
	return oReturnCheck;
}

#ifdef __cplusplus
}
#endif
