/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <time.h>
#include <stdlib.h>

#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"
#include "load_img.h"
#include "interrupt.h"
#include "stl_analog_io.h"
#include "stl_watchdog.h"
#include "stl_stack.h"
#include "stl_program_counter.h"
#include "stl_clock.h"
#include "stl_digital_io.h"
#include "stl_invariable_mem.h"
#include "stl_variable_mem.h"
#include "stl_cpu_register.h"
#include "stl_dma.h"
#include "stl_math_lib.h"

#define LED_GPIO_PIN	GPIO_PIN_LED1
#define LED_GPIO_CFG	GPIO_CFG_LED1

volatile uint32_t dspTicks = 0;

void TIMER1_IRQHandler(void)
{
	CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
	dspTicks += 1;
}

int32_t Timer1_init(uint32_t base, uint32_t period)
{
	/* Initialize CPUTimer with interrupt enable */
	CPUTimer_init(CPUTIMER1_BASE, period);

	Interrupt_register(INT_TIMER1, TIMER1_IRQHandler);
	Interrupt_enable(INT_TIMER1);

	return 0;
}

int main(void)
{
	uint32_t oldTicks;
	uint32_t tickCnt;
	uint8_t dma_src[4];
	uint8_t cache_mem[100];
	extern uint32_t _data;

	Device_init();

	GPIO_setPinConfig(LED_GPIO_CFG);
	GPIO_enableWritePin(LED_GPIO_PIN);

	UartPrint_init(LOG_SCI_BASE, 115200);

	log_set_level(LOG_DEBUG);

	log_info("Hello DSP template!\r\n");
	log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
	log_info("Code @ 0x%08X, Data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&oldTicks);

	/* 10Ms */
	Timer1_init(CPUTIMER1_BASE, DEVICE_APBCLK_FREQ/100);

	__enable_irq();

	oldTicks = dspTicks;

	__disable_irq();

	/* Stack test. */
	stl_stack_check_init();

	/* Watchdog reset cpu test. */
	if (stl_watchdog_reset_test(WD1_BASE, 1, 1))
		log_info("Watchdog test failed!\r\n");
	else
		log_info("Watchdog test successful!\r\n");

	/* Invariable memory test. */
	if (stl_invariable_mem_software_crc16modbus())
		log_info("Invariable memory test failed!\r\n");
	else
		log_info("Invariable memory test successful!\r\n");

	/* DMA test */
	if (stl_dma_test(DMA_BASE, DMA_CH1_BASE))
		log_info("dma test failed!\r\n");
	else
		log_info("dma test successful!\r\n");

	/* Math lib test. */
	for (uint32_t i = 0; i < MATH_TEST_COUNT; i++) {
		if (stl_mathlib_test(i))
			log_info("Mathlib id:%d test failed!\r\n", i);
		else
			log_info("Mathlib id:%d test successful!\r\n", i);
	}

	/* Variable memory test. */
	if (stl_mbist_march_x_algorithm((uintptr_t)&_data, 100))
		log_info("march x test failed!\r\n");
	else
		log_info("march x test successful!\r\n");

	if (stl_mbist_march_c_algorithm((uintptr_t)&_data, 100))
		log_info("march c test failed!\r\n");
	else
		log_info("march c test successful!\r\n");

	__enable_irq();

	/* RISCV register test. */
	for (uint32_t i = 0; i < 32; i++) {
		/* X2 register is stack register, we can not to access it. */
		if (i == 2 || i == 8)
			i++;

		if (stl_access_x_register(i))
			log_info("RISCV X%d register test failed!\r\n", i);
		else
			log_info("RISCV X%d register test successful!\r\n", i);
	}

	/* PC test. */
	uint32_t flag = stl_program_counter_test();
	if (flag != STL_SUCCESS)
		log_info("Program Counter test failed!\r\n");
	else
		log_info("Program Counter test successful!\r\n");

	/* Analog IO output/input test(DACA). */
	if (stl_analog_io_output(DACA_BASE, DAC_REF_INTERNAL,
		DAC_LOAD_SYSCLK, DAC_GAIN_TWO, 2.5f, 3.3f,
		ADC_SOC_NUMBER0, ADC_REFERENCE_INTERNAL, 3.3f))
		log_info("DACA Analog IO Output test failed!\r\n");
	else
		log_info("DACA Analog IO Output test successful!\r\n");

	/* Analog IO output test(DACB). */
	if (stl_analog_io_output(DACB_BASE, DAC_REF_INTERNAL,
		DAC_LOAD_SYSCLK, DAC_GAIN_TWO, 2.5f, 3.3f,
		ADC_SOC_NUMBER0, ADC_REFERENCE_INTERNAL, 3.3f))
		log_info("DACB Analog IO Output test failed!\r\n");
	else
		log_info("DACB Analog IO Output test successful!\r\n");

	if (stl_analog_io_input(ADCA_BASE, ADCARESULT_BASE, ADC_SOC_NUMBER0,
						ADC_CH_ADCIN11, ADC_CLK_DIV_4_0, ADC_REFERENCE_3_3V,
						ADC_REFERENCE_3_3V, 1.38, 3.3))
		log_info("ADC Analog IO Input test failed!\r\n");
	else
		log_info("ADC Analog IO Input test successful!\r\n");

	/* Digital IO input test. */
	GPIO_setPinConfig(GPIO_53_GPIO53);
	if (stl_digital_io_input(53, 0, 1))
		log_info("Digital IO Input test failed!\r\n");
	else
		log_info("Digital IO Input test successful!\r\n");

	/* Digital io output test. */
	GPIO_setPinConfig(GPIO_51_GPIO51);
	if (stl_digital_io_output(51))
		log_info("Digital IO Output test failed!\r\n");
	else
		log_info("Digital IO Output test successful!\r\n");

	/* Analog io digital function input test. */
	GPIO_setPinConfig(GPIO_13_GPIO13);
	if (stl_digital_aio_input(13, 0, 1))
		log_info("Digital AIO Input test failed!\r\n");
	else
		log_info("Digital AIO Input test successful!\r\n");

	/* Analog io digital function output test. */
	GPIO_setPinConfig(GPIO_20_GPIO20);
	if (stl_digital_aio_output(20))
		log_info("Digital AIO Output test failed!\r\n");
	else
		log_info("Digital AIO Output test successful!\r\n");

	/* Stack test. */
	if (stl_stack_overflow_check())
		log_info("Stack test failed!\r\n");
	else
		log_info("Stack test successful!\r\n");

	/* Clock test. */
	if (stl_clock_test(DEVICE_SYSCLK_FREQ))
		log_info("Clock test failed!\r\n");
	else
		log_info("Clock test successful!\r\n");

	while (1) {
		if (dspTicks != oldTicks) {
			oldTicks = dspTicks;
			tickCnt += 1;

			if ((tickCnt % 100) == 0) {
				GPIO_togglePin(LED_GPIO_PIN);
				log_info("ticks: %d\r\n", dspTicks);
			}
		}
	}

	return 0;
}
