/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <stdbool.h>

#include "stl_digital_io.h"
#include "stl_err.h"
#include "driverlib.h"

stl_err_t stl_digital_aio_input(uint16_t io_num,
							uint16_t expectation_value,
							uint32_t delay_cnt)
{
	uint32_t i;
	uint16_t val;

	GPIO_setAnalogMode(io_num, GPIO_ANALOG_DISABLED);
	GPIO_setInputEnableGPH(io_num, true);

	GPIO_setDirectionMode(io_num, GPIO_DIR_MODE_IN);

	/* Waiting for input capture to complete conversion only. */
	for (i = 0; i < delay_cnt; i++)
		asm volatile ("nop");

	/* Read the GPIO value. */
	val = GPIO_readPin(io_num);

	if (expectation_value != val)
		return STL_ERR_CHECK_FAILED;

	return STL_SUCCESS;
}

stl_err_t stl_digital_io_input(uint16_t io_num,
							uint16_t expectation_value,
							uint32_t delay_cnt)
{
	uint32_t i;
	uint16_t val;

	GPIO_setDirectionMode(io_num, GPIO_DIR_MODE_IN);

	/* Waiting for input capture to complete conversion only. */
	for (i = 0; i < delay_cnt; i++)
		asm volatile ("nop");

	/* Read the GPIO value. */
	val = GPIO_readPin(io_num);

	if (expectation_value != val)
		return STL_ERR_CHECK_FAILED;

	return STL_SUCCESS;
}

stl_err_t stl_digital_aio_output(uint16_t io_num)
{
	GPIO_setAnalogMode(io_num, GPIO_ANALOG_DISABLED);
	GPIO_setDirectionMode(io_num, GPIO_DIR_MODE_OUT);

	GPIO_writePin(io_num, 1);

	if (!GPIO_getPinDataOut(io_num))
		return STL_ERR_CHECK_FAILED;

	GPIO_writePin(io_num, 0);

	if (GPIO_getPinDataOut(io_num))
		return STL_ERR_CHECK_FAILED;

	return STL_SUCCESS;
}

stl_err_t stl_digital_io_output(uint16_t io_num)
{
	GPIO_setDirectionMode(io_num, GPIO_DIR_MODE_OUT);

	GPIO_writePin(io_num, 1);

	if (!GPIO_getPinDataOut(io_num))
		return STL_ERR_CHECK_FAILED;

	GPIO_writePin(io_num, 0);

	if (GPIO_getPinDataOut(io_num))
		return STL_ERR_CHECK_FAILED;

	return STL_SUCCESS;
}
