/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>
#include <math.h>
#include <stdlib.h>
#include "stl_math_lib.h"
#include "stl_err.h"

#define MATH_TEST_TOLERANCE 0.0001f
#define MATH_PI_DIV6   0.5235987756f    /* /6 */
#define MATH_PI_DIV3   1.0471975512f    /* /3 */
#define MATH_2PI_DIV3  2.0943951024f    /* 2/3 */
#define MATH_3PI_DIV4  2.3561944902f    /* 3/4 */
#define MATH_5PI_DIV6  2.6179938780f    /* 5/6 */
#define MATH_E         2.7182818284f    /* e */
#define MATH_1_DIV_E   0.3678794412f    /* 1/e */

static int float_compare(float a, float b)
{
    return fabsf(a - b) < MATH_TEST_TOLERANCE;
}

stl_err_t stl_mathlib_test(math_test_id_t test_id)
{
    switch(test_id) {
        /* Trigonometric function tests */
        case MATH_TEST_SIN:
            if(!float_compare(__sin(MATH_PI_DIV2), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_COS:
            if(!float_compare(__cos(MATH_PI_DIV3), 0.5f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_TAN:
            if(!float_compare(__tan(MATH_PI_DIV4), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
            
        /* Unit circle function tests */
        case MATH_TEST_SINPU:
            if(!float_compare(__sinpuf32(0.25f), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_COSPU:
            if(!float_compare(__cospuf32(0.16667f), 0.5f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_ATANPU:
            if(!float_compare(__atanpuf32(1.0f), 0.125f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_ATAN2PU:
            if(!float_compare(__atan2puf32(1.0f, 1.0f), 0.125f)) return STL_ERR_CHECK_FAILED;
            break;
            
        /* Inverse trigonometric function tests */
        case MATH_TEST_ASIN:
            if(!float_compare(__asin(0.5f), MATH_PI_DIV6)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_ACOS:
            if(!float_compare(__acos(0.5f), MATH_PI_DIV3)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_ATAN:
            if(!float_compare(__atan(1.0f), MATH_PI_DIV4)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_ATAN2:
            if(!float_compare(__atan2(1.0f, 1.0f), MATH_PI_DIV4)) return STL_ERR_CHECK_FAILED;
            break;
            
        /* Floating-point operation tests */
        case MATH_TEST_DIVF32:
            if(!float_compare(__divf32(5.0f, 2.5f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_SQSUMF:
            if(!float_compare(__sqsumf(5.0f, 2.5f), 31.25f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_SQRT:
            if(!float_compare(__sqrt(0.16f), 0.4f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FABS:
            if(!float_compare(__fabs(-5.0f), 5.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMIN:
            if(!float_compare(__fmin(-5.0f, 2.5f), -5.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMAX:
            if(!float_compare(__fmax(-5.0f, 2.5f), 2.5f)) return STL_ERR_CHECK_FAILED;
            break;
            
        /* Special operation tests */
        case MATH_TEST_FSAT:
            if(!float_compare(__fsat(5.0f, 1.0f, -1.0f), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FTRUNC:
            if(!float_compare(__ftrunc(-3.7f), -3.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FFRACT:
            if(!float_compare(__ffract(-3.7f), -0.7f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_MPY2PIF32:
            if(!float_compare(__mpy2pif32(1.0f), MATH_2PI)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_DIV2PIF32:
            if(!float_compare(__div2pif32(MATH_2PI), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FDIV3SQRT:
            if(!float_compare(__fdiv3sqrt(1.732f), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMUL2:
            if(!float_compare(__fmul2(2.0f), 4.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FDIV2:
            if(!float_compare(__fdiv2(4.0f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FDIV3:
            if(!float_compare(__fdiv3(6.0f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FDIV4:
            if(!float_compare(__fdiv4(8.0f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMUL3SQRT:
            if(!float_compare(__fmul3sqrt(1.0f), 1.732f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMUL2_DIV3:
            if(!float_compare(__fmul2_div3(3.0f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMUL3SQRT_DIV2:
            if(!float_compare(__fmul3sqrt_div2(2.0f), 1.73205f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMUL3SQRT_DIV6:
            if(!float_compare(__fmul3sqrt_div6(2.0f), 0.57735f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FDIV3SQRT_MUL2:
            if(!float_compare(__fdiv3sqrt_mul2(1.732f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FMUL2PI_NEG:
            if(!float_compare(__fmul2pi_neg(1.0f), -MATH_2PI)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FDIV2PI_NEG:
            if(!float_compare(__fdiv2pi_neg(MATH_2PI), -1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        
		/* Constant operation tests */
		case MATH_TEST_NEG1_FADD_CONST:
            if(!float_compare(__neg1_fadd_const(0.5f, 2), 0.5f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_NEG1_FSUB_CONST:
            if(!float_compare(__neg1_fsub_const(0.5f, 2), -1.5f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FADD_ONE:
            if(!float_compare(__fadd_one(1.0f), 2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FSUB_ONE:
            if(!float_compare(__fsub_one(2.0f), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FADD_3SQRT:
            if(!float_compare(__fadd_3sqrt(0.0f), 1.732f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FSUB_3SQRT:
            if(!float_compare(__fsub_3sqrt(1.732f), 0.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FADD_PI3RD:
            if(!float_compare(__fadd_pi3rd(0.0f), 1.0472f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_FSUB_PI3RD:
            if(!float_compare(__fsub_pi3rd(1.0472f), 0.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_NEG1_FADD_ONE:
            if(!float_compare(__neg1_fadd_one(1.0f), 0.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_NEG1_FSUB_ONE:
            if(!float_compare(__neg1_fsub_one(1.0f), -2.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_NEG1_FADD_3SQRT:
            if(!float_compare(__neg1_fadd_3sqrt(1.0f), 0.732f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_NEG1_FSUB_3SQRT:
            if(!float_compare(__neg1_fsub_3sqrt(-1.732f), 0.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_NEG1_FADD_PI3RD:
            if(!float_compare(__neg1_fadd_pi3rd(1.0f), 0.047198f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_EXPF32:
            if(!float_compare(__expf32(-1.0f), MATH_1_DIV_E)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_LOGF32:
            if(!float_compare(__logf32(MATH_1_DIV_E), -1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_POWF32:
            if(!float_compare(__powf32(MATH_1_DIV_E,-1.0f), MATH_E)) return STL_ERR_CHECK_FAILED;
            break;

		/* Fixed-point conversion tests */
		case MATH_TEST_IQ8TOF:
            if(!float_compare(_IQ8toF(0x100), 1.0f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ8FRAC:
            if(!float_compare(_IQ8frac(__float2iqn(0.1f, 8)), __float2iqn(0.1f, 8))) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ8SINPU:
            if(abs(_IQ8sinPU(__float2iqn(0.25f, 8)) - __float2iqn(1.0f, 8)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ8COSPU:
            if(abs(_IQ8cosPU(__float2iqn(0.25f, 8)) - __float2iqn(0.0f, 8)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16SIN:
            if(abs(_IQ16sin(__float2iqn(MATH_PI_DIV2, 16)) - __float2iqn(1.0f, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16COS:
            if(abs(_IQ16cos(__float2iqn(MATH_PI_DIV2, 16)) - __float2iqn(0.0f, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16ASIN:
            if(abs(_IQ16asin(__float2iqn(0.5f, 16)) - __float2iqn(MATH_PI_DIV6, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16ACOS:
            if(abs(_IQ16acos(__float2iqn(0.5f, 16)) - __float2iqn(MATH_PI_DIV3, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ24ATAN:
            if(abs(_IQ24atan(__float2iqn(1.0f, 24)) - __float2iqn(MATH_PI_DIV4, 24)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16ATAN2:
            if(abs(_IQ16atan2(__float2iqn(1.0f, 16), __float2iqn(1.0f, 16)) - __float2iqn(MATH_PI_DIV4, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQNSAT:
            if(__IQNsat(__float2iqn(2.0f, 16), __float2iqn(1.0f, 16), __float2iqn(-1.0f, 16)) != __float2iqn(1.0f, 16)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16MPY:
            if(abs(_IQ16mpy(__float2iqn(2.0f, 16), __float2iqn(3.0f, 16)) - __float2iqn(6.0f, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ16RMPY:
            if(abs(_IQ16rmpy(__float2iqn(2.0f, 16), __float2iqn(3.2f, 16)) - __float2iqn(6.4f, 16)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ24MPYIQX:
            if(abs(_IQ24mpyIQX(__float2iqn(2.0f, 10), 10, __float2iqn(3.0f, 16), 16) - __float2iqn(6.0f, 24)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ18DIV:
            if(abs(_IQ18div(__float2iqn(6.2f, 18), __float2iqn(3.1f, 18)) - __float2iqn(2.0f, 18)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_ATOIQ14:
            if(abs(_atoIQ14("2.1") - __float2iqn(2.1f, 14)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ24TOA:
            char str[20];
            _IQ24toa(str, "%3.6f", __float2iqn(2.1f, 24));
            if(!float_compare(atof(str), 2.1f)) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ15LOG:
            if(abs(_IQ15log(__float2iqn(MATH_E, 15)) - __float2iqn(1.0f, 15)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ8EXP:
            if(abs(_IQ8exp(__float2iqn(1.0f,8)) - __float2iqn(MATH_E, 8)) > 2) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_IQ10SQRT:
            if(abs(_IQ10sqrt(__float2iqn(4.0f, 10)) - __float2iqn(2.0f, 10)) > 2) return STL_ERR_CHECK_FAILED;
            break;

		/* Integer operation tests */
		case MATH_TEST_SMIN:
            if(__smin(10, 20) != 10) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_SMAX:
            if(__smax(10, 20) != 20) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_UMIN:
            if(__umin(10, 20) != 10) return STL_ERR_CHECK_FAILED;
            break;
        case MATH_TEST_UMAX:
            if(__umax(10, 20) != 20) return STL_ERR_CHECK_FAILED;
            break;

		/* Bit operation tests */
		 case MATH_TEST_RBIT:
			if(__rbit(0x0F0F0F0F) != 0xF0F0F0F0) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_FLIP32:
			if(__flip32(0x0F0F0F0F) != 0xF0F0F0F0) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_RBIT_MSBI:
			if(__rbit_msbi(0x0F0F0F0F, 16) != 0x1e1e1) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_ANDN:
			if(__andn(0x5555, 0x3333) != 0x2222) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_ORN:
			if(__orn(0x5555, 0x3333) != 0xFFFFBBBB) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_NXOR:
			if(__nxor(0x5555, 0x3333) != 0xFFFF9999) return STL_ERR_CHECK_FAILED;
			break;

		/* Saturation operation tests */
		case MATH_TEST_SAT16:
			if(__sat16(0x12345678) != 0x7FFF) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_SAT16U:
			if(__sat16u(0x12345678) != 0x7FFF) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_SAT32U:
			if(__sat32u(0x80000000) != 0x0) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_ABS_SAT:
			if(__abs_sat(-1) != 0x1) return STL_ERR_CHECK_FAILED;
			break;

		/* Bit counting tests */
		case MATH_TEST_CLO:
			if(__clo(0xffe00000) != 11) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_CLZ:
			if(__clz(0x00100000) != 11) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_CTZ:
			if(__ctz(0x00100000) != 20) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_CLRSB:
			if(__clrsb(0xFFFFFFF0) != 27) return STL_ERR_CHECK_FAILED;
			break;

		/* Endianness conversion tests */
		case MATH_TEST_BSWAP32:
			if(__bswap32(0x12345678) != 0x78563412) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_BSWAP16:
			if(__bswap16(0x1234) != 0x3412) return STL_ERR_CHECK_FAILED;
			break;

		/* Standard math functions tests*/
		case MATH_TEST_ROUNDF:
        if(roundf(3.5f) != 4.0f) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_FLOORF:
			if(floorf(3.9f) != 3.0f) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_CEILF:
			if(ceilf(2.1f) != 3.0f) return STL_ERR_CHECK_FAILED;
			break;
		case MATH_TEST_LABS:
			if(labs(-12345L) != 12345L) return STL_ERR_CHECK_FAILED;
			break;
            
        default:
            return STL_ERR_INVALID_PARAM;
    }
    
    return STL_SUCCESS;
}
