/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdint.h>

#include "stl_watchdog.h"
#include "stl_err.h"
#include "wd.h"

__attribute__((section(".unreset_ram_sec"))) uint32_t watch_dog_reset_flag;

/**
 * @brief Initialize watchdog.
 * 
 * @param base The base address of the watchdog module.
 * @param prescaler Watchdog clock prescaler coefficient.
 * @param timeout_cnt Watchdog timeout count value.
 */
static void stl_watchdog_init(uint32_t base, uint32_t prescaler, uint32_t timeout_cnt)
{
	/* Disable the watchdog. */
	SysCtl_disableWatchdog(base);

	/* configure the watchdog clock prescaler. */
	SysCtl_setWatchdogPrescaler(base, prescaler);

	/* configure the wd timer reload value. */
	SysCtl_initWatchdog(base, timeout_cnt);

	/* enable watchdog reset */
	SysCtl_setWatchdogMode(base, WD_MODE_RESET);

	/* turn on watchdog timer. */
	SysCtl_enableWatchdog(base);
}

stl_err_t stl_watchdog_reset_test(uint32_t base, uint32_t prescaler, uint32_t timeout_cnt)
{
	if (watch_dog_reset_flag != 0x12345678U) {
		watch_dog_reset_flag = 0x12345678U;
		stl_watchdog_init(base, prescaler, timeout_cnt);
		return STL_SUCCESS;
	} else if (watch_dog_reset_flag == 0x12345678U) {
		watch_dog_reset_flag = 0x56789;
		return STL_SUCCESS;
	}
	else
		return STL_ERR_CHECK_FAILED;
}
