/*
 *   Copyright (c) GeJian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    board.c
*   @brief   
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

//*****************************************************************************
//
// Board Configurations
//
//*****************************************************************************
//
// Initialize peripherals used by the abs2qep and the example
//
void
pto_initAbs2QEP(void)
{

    //
    // Setup the CLB
    //
    pto_abs2qep_setupPeriph();

    pto_configEPWM();
    pto_configEQEP();
    //
    // Setup the GPIO for this example
    //
    pto_setupGPIO();
}

//
// pto_initEQEP - Initialize EQEP module
//
void
pto_configEQEP(void)
{
	EQEP_SourceSelect source_myEQEP0 =
	{
		EQEP_SOURCE_PWMXBAR1, 		// eQEPA source
		EQEP_SOURCE_PWMXBAR2,		// eQEPB source
		EQEP_SOURCE_PWMXBAR3,  	// eQEP Index source
	};
	EQEP_selectSource(EQEP1_BASE, source_myEQEP0);

    EQEP_disableInterrupt(myEQEP0_BASE,
        (EQEP_INT_GLOBAL            |
        EQEP_INT_POS_CNT_ERROR      |
        EQEP_INT_PHASE_ERROR        |
        EQEP_INT_DIR_CHANGE         |
        EQEP_INT_WATCHDOG           |
        EQEP_INT_UNDERFLOW          |
        EQEP_INT_OVERFLOW           |
        EQEP_INT_POS_COMP_READY     |
        EQEP_INT_POS_COMP_MATCH     |
        EQEP_INT_STROBE_EVNT_LATCH  |
        EQEP_INT_INDEX_EVNT_LATCH   |
        EQEP_INT_UNIT_TIME_OUT      |
        EQEP_INT_QMA_ERROR));
    EQEP_clearInterruptStatus(myEQEP0_BASE,
        (EQEP_INT_GLOBAL            |
        EQEP_INT_POS_CNT_ERROR      |
        EQEP_INT_PHASE_ERROR        |
        EQEP_INT_DIR_CHANGE         |
        EQEP_INT_WATCHDOG           |
        EQEP_INT_UNDERFLOW          |
        EQEP_INT_OVERFLOW           |
        EQEP_INT_POS_COMP_READY     |
        EQEP_INT_POS_COMP_MATCH     |
        EQEP_INT_STROBE_EVNT_LATCH  |
        EQEP_INT_INDEX_EVNT_LATCH   |
        EQEP_INT_UNIT_TIME_OUT      |
        EQEP_INT_QMA_ERROR));

//
// Set the strobe input source of the eQEP module.
//
EQEP_setStrobeSource(myEQEP0_BASE,EQEP_STROBE_FROM_GPIO);
//
// Sets the polarity of the eQEP module's input signals.
//
EQEP_setInputPolarity(myEQEP0_BASE,false,false,false,false);
//
// Configures eQEP module's quadrature decoder unit.
//
EQEP_setDecoderConfig(myEQEP0_BASE, (EQEP_CONFIG_QUADRATURE | EQEP_CONFIG_2X_RESOLUTION | EQEP_CONFIG_NO_SWAP | EQEP_CONFIG_IGATE_DISABLE));
//
// Set the emulation mode of the eQEP module.
//
EQEP_setEmulationMode(myEQEP0_BASE,EQEP_EMULATIONMODE_RUNFREE);
//
// Configures eQEP module position counter unit.
//
EQEP_setPositionCounterConfig(myEQEP0_BASE,EQEP_POSITION_RESET_IDX,0xffffffffU);
//
// Sets the current encoder position.
//
EQEP_setPosition(myEQEP0_BASE,0U);
//
// Enables the eQEP module unit timer.
//
//EQEP_enableUnitTimer(myEQEP0_BASE,1200000U);

EQEP_disableUnitTimer(myEQEP0_BASE);
//
// Disables the eQEP module watchdog timer.
//
EQEP_disableWatchdog(myEQEP0_BASE);
//
// Configures the quadrature modes in which the position count can be latched.
//
EQEP_setLatchMode(myEQEP0_BASE,(EQEP_LATCH_SW_INDEX_MARKER));
//
// Set the quadrature mode adapter (QMA) module mode.
//
EQEP_setQMAModuleMode(myEQEP0_BASE,EQEP_QMA_MODE_BYPASS);
//
// Disable Direction Change During Index
//
EQEP_disableDirectionChangeDuringIndex(myEQEP0_BASE);
//
// Configures the mode in which the position counter is initialized.
//
EQEP_setPositionInitMode(myEQEP0_BASE,(EQEP_INIT_DO_NOTHING));
//
// Sets the software initialization of the encoder position counter.
//
EQEP_setSWPositionInit(myEQEP0_BASE,true);
// Sets the init value for the encoder position counter.
//
EQEP_setInitialPosition(myEQEP0_BASE,0U);
//
// Enables the eQEP module.
//
EQEP_enableModule(myEQEP0_BASE);
EQEP_setCaptureConfig(myEQEP0_BASE,EQEP_CAPTURE_CLK_DIV_128,EQEP_UNIT_POS_EVNT_DIV_256);
//
// Enables the eQEP module edge-capture unit.
//
EQEP_enableCapture(myEQEP0_BASE);
}

//*****************************************************************************
//
//! Setup CLB tiles
//!
//! \param None
//!
//! Setup the CLB and other interconnect XBARs used in this application.
//! This is called during the CLB initialization.
//!
//! \return None
//
//*****************************************************************************
void
pto_abs2qep_setupPeriph(void)
{

    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_CLB1);

    CLB_disableCLB(CLB1_BASE);
    //
    // The init of the tile must be done first. This
    // will configure the whole tile, including the
    // HLC program.
    //
    // When enabled, the HLC will receive a rising edge
    // on the e2 input. This corresponds to the end
    // of a PTO, and will set the expected INTERRUPT
    // TAG.
    //
    initTILE1(CLB1_BASE);


    //
    // Unused Inputs below
    //
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN0, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN1, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN2, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN3, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN4, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN5, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN6, CLB_GLOBAL_IN_MUX_EPWM1A);
    CLB_configGlobalInputMux(CLB1_BASE, CLB_IN7, CLB_GLOBAL_IN_MUX_EPWM1A);

    //
    // Inputs set to GP register
    //
    CLB_configGPInputMux(CLB1_BASE, CLB_IN0, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(CLB1_BASE, CLB_IN1, CLB_GP_IN_MUX_GP_REG);


    //
    // Unused inputs to GP register
    //
    CLB_configGPInputMux(CLB1_BASE, CLB_IN2, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(CLB1_BASE, CLB_IN3, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(CLB1_BASE, CLB_IN4, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(CLB1_BASE, CLB_IN5, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(CLB1_BASE, CLB_IN6, CLB_GP_IN_MUX_GP_REG);
    CLB_configGPInputMux(CLB1_BASE, CLB_IN7, CLB_GP_IN_MUX_GP_REG);

    CLB_selectInputFilter(CLB1_BASE, CLB_IN0, CLB_FILTER_RISING_EDGE);
    CLB_selectInputFilter(CLB1_BASE, CLB_IN1, CLB_FILTER_RISING_EDGE);
    //
    // Initialize the CLB XBAR
    //
    pto_abs2qep_initCLBXBAR();

    //
    // OUT2 (QEP-I) overrides PWM1B
    // OUT0 (RUN/HALT) overrides PWM1A
    //

    //
    // Sending a position 0 will setup the CLB
    // modules and set the interrupt tag to indicate
    // a PTO is not running.
    //
    pto_abs2qep_translatePosition(0);

	CLB_enableCLB(CLB1_BASE);

}

//*****************************************************************************
//
// PINMUX Configurations
//
//*****************************************************************************

void
	pto_setupGPIO(void)
	{
	GPIO_setPinConfig(myEPWM1_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM1_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM1_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(myEPWM1_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM1_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM1_EPWMB_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(GPIO_22_CLB_OUTPUTXBAR1);
	GPIO_setPadConfig(22, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(22, GPIO_QUAL_SYNC);


	GPIO_setPinConfig(GPIO_23_CLB_OUTPUTXBAR3);
	GPIO_setPadConfig(23, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(23, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(GPIO_2_OUTPUTXBAR1);
	GPIO_setPadConfig(24, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(24, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(GPIO_3_OUTPUTXBAR2);
	GPIO_setPadConfig(25, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(25, GPIO_QUAL_SYNC);

	//eqep1A
	GPIO_setPinConfig(GPIO_10_EQEP1_A);
	GPIO_setPadConfig(10, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(10, GPIO_QUAL_SYNC);
	//eqep1B
	GPIO_setPinConfig(GPIO_11_EQEP1_B);
	GPIO_setPadConfig(11, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(11, GPIO_QUAL_SYNC);

	//eqep1I
	GPIO_setPinConfig(GPIO_13_EQEP1_INDEX);
	GPIO_setPadConfig(13, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(13, GPIO_QUAL_SYNC);

	}

//*****************************************************************************
//
//! Initialize the CLB XBAR-connect
//!
//! \param None
//!
//! This function configures Output-XBars to route CLB outputs to GPIO pins
//!
//! \return None
//
//*****************************************************************************
void
pto_abs2qep_initCLBXBAR(void)
{
    //
    // PTO-QEP-A output
    //
    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT1, XBAR_OUT_MUX04_CLB1_OUT4);
    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT1, XBAR_MUX04);

    //
    // PTO-QEP-B output
    //
    XBAR_setOutputMuxConfig(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT3, XBAR_OUT_MUX05_CLB1_OUT5);
    XBAR_enableOutputMux(CLBOUTPUTXBAR_BASE,XBAR_OUTPUT3, XBAR_MUX05);
    //
    // PTO-QEP-I output
    //
    XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE,XBAR_OUTPUT1, XBAR_OUT_MUX32_CLB1_OUT0);
    XBAR_enableOutputMux(OUTPUTXBAR_BASE,XBAR_OUTPUT1, XBAR_MUX32);

    XBAR_setOutputMuxConfig(OUTPUTXBAR_BASE,XBAR_OUTPUT2, XBAR_OUT_MUX33_CLB1_OUT1);
    XBAR_enableOutputMux(OUTPUTXBAR_BASE,XBAR_OUTPUT2, XBAR_MUX33);
	//epeq1a
    XBAR_enableEPWMMux(XBAR_TRIP4, XBAR_MUX01);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT1, 6);
    XBAR_setEPWMMuxConfig(XBAR_TRIP4, XBAR_EPWM_MUX01_INPUTXBAR1);

	//epeq1b
    XBAR_enableEPWMMux(XBAR_TRIP5, XBAR_MUX03);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT2, 11);
    XBAR_setEPWMMuxConfig(XBAR_TRIP5, XBAR_EPWM_MUX03_INPUTXBAR2);
	//epeq1I
    XBAR_enableEPWMMux(XBAR_TRIP7, XBAR_MUX05);
    XBAR_setInputPin(INPUTXBAR_BASE, XBAR_INPUT3, 9);
    XBAR_setEPWMMuxConfig(XBAR_TRIP7, XBAR_EPWM_MUX05_INPUTXBAR3);
}

void pto_configEPWM(void)
{

	EPWM_setClockPrescaler(myEPWM1_BASE, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_1);
    EPWM_setTimeBasePeriod(myEPWM1_BASE, 250);
    EPWM_setTimeBaseCounter(myEPWM1_BASE, 0);
    EPWM_setTimeBaseCounterMode(myEPWM1_BASE, EPWM_COUNTER_MODE_UP);
    EPWM_disablePhaseShiftLoad(myEPWM1_BASE);
    EPWM_setPhaseShift(myEPWM1_BASE, 0);
    EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, 250);
    EPWM_setCounterCompareShadowLoadMode(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B, 367);
    EPWM_setCounterCompareShadowLoadMode(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B, EPWM_COMP_LOAD_ON_CNTR_ZERO);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_HIGH, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_LOW, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_A, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_TOGGLE, EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB);
    EPWM_setActionQualifierAction(myEPWM1_BASE, EPWM_AQ_OUTPUT_B, EPWM_AQ_OUTPUT_NO_CHANGE, EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
    EPWM_enableInterrupt(myEPWM1_BASE);
    EPWM_setInterruptSource(myEPWM1_BASE, EPWM_INT_TBCTR_ZERO);
    EPWM_setInterruptEventCount(myEPWM1_BASE, 2);
    HWREG(0x4003F000 + 0x120) = 0x10000;
    HWREG(0x4003F000 + 0x44) = 0x40000;
}
/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/* None */

#ifdef __cplusplus
}
#endif

