/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_dig.h
*   @brief   digital system top level configuration
*
*/


#ifndef PM_TFORMAT_INCLUDE_H_
#define PM_TFORMAT_INCLUDE_H_

#ifdef __cplusplus
extern "C"{
#endif
#include <stdint.h>
#include "inc/hw_types.h"


#define PM_TFORMAT_SPI              SPIB_BASE

#define PM_TFORMAT_INT_SPI_RX       INT_SPIB_RX
//*****************************************************************************
//
// Data ID Code
// Unique code for each command. Defined in the T-Format specification.
// IDx = code (4-bits) + parity (1-bit)
//
//*****************************************************************************
#define PM_TFORMAT_CODEID0  0x00u   // 0b0000:0 = code0:parity
#define PM_TFORMAT_CODEID1  0x11u   // 0b1000:1 = code1:parity
#define PM_TFORMAT_CODEID2  0x09u   // 0b0100:1 = code2:parity
#define PM_TFORMAT_CODEID3  0x18u   // 0b1100:0 = code3:parity
#define PM_TFORMAT_CODEID6  0x0Cu   // 0b0110:0 = code6:parity
#define PM_TFORMAT_CODEIDD  0x17u   // 0b1011:1 = codeD:parity
#define PM_TFORMAT_CODEID7  0x1Du   // 0b1110:1 = code7:parity
#define PM_TFORMAT_CODEID8  0x03u   // 0b0001:1 = code8:parity
#define PM_TFORMAT_CODEIDC  0x06u   // 0b0011:0 = codeC:parity

//*****************************************************************************
//
// Control Field (CF)
// First 10-bit field in any transmitted or received packet
//
//    idle   start  sink code   Data ID code   Delimiter
//  ...1,1     0      010          xxxx:x         1
//
// * 1-bit start:       fixed (0)
// * 3-bit sink code:   fixed (0,1,0)
// * 5-bit DataIDCode:  code (4-bits) + parity (1-bit)
// * 1-bit Delimiter:   fixed (1)
//
//  Note: The delimiter is added in the PM_TFORMAT_IDx enumerations
//
//*****************************************************************************
#define PM_TFORMAT_CFID0 (PM_TFORMAT_CODEID0 | 0x40u) //CF: 0b010:0000:0 = 0x40
#define PM_TFORMAT_CFID1 (PM_TFORMAT_CODEID1 | 0x40u) //CF: 0b010:1000:1 = 0x51
#define PM_TFORMAT_CFID2 (PM_TFORMAT_CODEID2 | 0x40u) //CF: 0b010:0100:1 = 0x49
#define PM_TFORMAT_CFID3 (PM_TFORMAT_CODEID3 | 0x40u) //CF: 0b010:1100:0 = 0x58
#define PM_TFORMAT_CFID6 (PM_TFORMAT_CODEID6 | 0x40u) //CF: 0b010:0110:0 = 0x4C
#define PM_TFORMAT_CFIDD (PM_TFORMAT_CODEIDD | 0x40u) //CF: 0b010:1011:1 = 0x57
#define PM_TFORMAT_CFID7 (PM_TFORMAT_CODEID7 | 0x40u) //CF: 0b010:1110:1 = 0x5D
#define PM_TFORMAT_CFID8 (PM_TFORMAT_CODEID8 | 0x40u) //CF: 0b010:0001:1 = 0x43
#define PM_TFORMAT_CFIDC (PM_TFORMAT_CODEIDC | 0x40u) //CF: 0b010:0011:0 = 0x46


//
// [doc-library-tformat-cf-end]
//
//*****************************************************************************
//
// This 16-bit value is the CF reformatted to be sent through the SPI:
// * The Control Field is left justified
//   (The SPI transfers bits out from the most significant bit)
// * An initial idle bit is added
// * The trailing delimiter is added
// * Trailing bits are filled with 1 (idle) This will keep SOMI
//   high, if needed, while waiting for the response.
//
//   bit15                                                 bit0
//   idle  start  sink code   Data ID code   Delimiter  Trailing bits
//    1      0      010          xxxx:x         1         11111
//
//*****************************************************************************

enum commandID6_D
    {
       PM_TFORMAT_ID6 = ((PM_TFORMAT_CFID6 << 7u ) | 0x40u),
       PM_TFORMAT_IDD = ((PM_TFORMAT_CFIDD << 7u ) | 0x40u)
    };


enum commandID0_1_2_3_7_8_C
    {
        PM_TFORMAT_ID0 = ((PM_TFORMAT_CFID0 << 7u ) | 0x40u),
        PM_TFORMAT_ID1 = ((PM_TFORMAT_CFID1 << 7u ) | 0x40u),
        PM_TFORMAT_ID2 = ((PM_TFORMAT_CFID2 << 7u ) | 0x40u),
        PM_TFORMAT_ID3 = ((PM_TFORMAT_CFID3 << 7u ) | 0x40u),
        PM_TFORMAT_ID7 = ((PM_TFORMAT_CFID7 << 7u ) | 0x40u),
        PM_TFORMAT_ID8 = ((PM_TFORMAT_CFID8 << 7u ) | 0x40u),
        PM_TFORMAT_IDC = ((PM_TFORMAT_CFIDC << 7u ) | 0x40u)
    };

//*****************************************************************************
//
// Command fields and expected response fields for each Data ID code.
// Each field is 10-bits including start (1-bit) and delimiter (1-bit)
// Abbreviations are from the T-Format specification:
//
//      CF:     Control Field
//      SF:     Status Field
//      DFx:    Data Field
//      CRC:    Cyclic-redundancy check Field
//      ABSx:   Absolute data in 1 revolution
//      ABMx:   Multi-turn data
//      ADF:    Address data field
//      EDF:    EEPROM data field
//      ENID:   Encoder ID
//      ALMC:   Encoder error
//
// [doc-library-tformat-command-format-start]
//*****************************************************************************
#define PM_TFORMAT_TX_FIELDS_ID0    1u	//ID0: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_ID1    1u	//ID1: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_ID2    1u	//ID2: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_ID3    1u	//ID3: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_ID7    1u	//ID7: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_ID8    1u	//ID8: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_IDC    1u	//IDC: TX 1 field    CF
#define PM_TFORMAT_TX_FIELDS_ID6    4u	//ID6: TX 4 fields   CF + ADF + EDF + CRC (EEPROM write)
#define PM_TFORMAT_TX_FIELDS_IDD    3u	//IDD: TX 3 fields   CF + ADF + CRC       (EEPROM read)


#define PM_TFORMAT_RX_FIELDS_ID0    6u	//ID0: RX 6 fields:  CF + SF + ABS0 + ABS1 + ABS2 + CRC
#define PM_TFORMAT_RX_FIELDS_ID1    6u	//ID1: RX 6 fields:  CF + SF + ABS0 + ABS1 + ABS2 + CRC
#define PM_TFORMAT_RX_FIELDS_ID2    4u	//ID2: RX 4 fields:  CF + SF + ENCID + CRC
//ID3: RX 11 fields: CF + SF + ABS0 + ABS1 + ABS2 + ENCID + ABM0 + ABM1 + ABM2 + ALMC + CRC
#define PM_TFORMAT_RX_FIELDS_ID3    11u
#define PM_TFORMAT_RX_FIELDS_ID7    6u	//ID7: RX 6 fields:  CF + SF + ABS0 + ABS1 + ABS2 + CRC
#define PM_TFORMAT_RX_FIELDS_ID8    6u	//ID8: RX 6 fields:  CF + SF + ABS0 + ABS1 + ABS2 + CRC
#define PM_TFORMAT_RX_FIELDS_IDC    6u	//IDC: RX 6 fields:  CF + SF + ABS0 + ABS1 + ABS2 + CRC
#define PM_TFORMAT_RX_FIELDS_ID6    4u	//ID6: RX 4 fields:  CF + ADF + EDF + CRC
#define PM_TFORMAT_RX_FIELDS_IDD    4u	//IDD: RX 4 fields:  CF + ADF + EDF + CRC

#define PM_TFORMAT_RX_FIELDS_MAX    11u
#define PM_TFORMAT_TX_FIELDS_MAX     4u
#define PM_TFORMAT_FIELDS_MAX                                                  \
           (PM_TFORMAT_TX_FIELDS_ID3 + PM_TFORMAT_RX_FIELDS_ID3)

//*****************************************************************************
//
//  - Number of bits in a field (start + data + delimiter)
//  - The number of bits of data, excludes the start and delimiter bits
//  - Transmit has an extra high bit at the start (to correspond to idle)
//  - Add the delimiter and start to the most significant bit.
//  - One field per SPI FIFO level.
//
//*****************************************************************************
#define PM_TFORMAT_FIELD_BITS               10u
#define PM_TFORMAT_FIELD_DATA_BITS          8u
#define PM_TFORMAT_ONE_DATA_FIELD           PM_TFORMAT_FIELD_DATA_BITS
#define PM_TFORMAT_TWO_DATA_FIELDS          (PM_TFORMAT_FIELD_DATA_BITS * 2ul)
#define PM_TFORMAT_THREE_DATA_FIELDS        (PM_TFORMAT_FIELD_DATA_BITS * 3ul)
#define PM_TFORMAT_IDLE_BIT                 1u
#define PM_TFORMAT_ALIGN_LEFT(a)            (a << 8)
#define PM_TFORMAT_ADD_DELIMITER_START(a) ( (a >> 1u) | (1u << 6u) )
#define PM_TFORMAT_FIFO_WORD_BITS           PM_TFORMAT_FIELD_BITS

//
// [doc-library-tformat-txrx-clocks-start]
//
//*****************************************************************************
//
// Number of SPI clocks required to transmit a request is:
// = 1 idle bit + (number of TX fields * number of field bits)
// = 1 idle bit + total number of TX bits
//
//*****************************************************************************
#define PM_TFORMAT_TX_CLOCKS_ID0 (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_ID0 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_ID1 (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_ID1 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_ID2 (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_ID2 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_ID3 (PM_TFORMAT_IDLE_BIT                          \
                +  (PM_TFORMAT_TX_FIELDS_ID3 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_ID6 (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_ID6 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_ID7 (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_ID7 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_ID8 (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_ID8 * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_IDC (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_IDC * PM_TFORMAT_FIELD_BITS))
#define PM_TFORMAT_TX_CLOCKS_IDD (PM_TFORMAT_IDLE_BIT                          \
                + (PM_TFORMAT_TX_FIELDS_IDD * PM_TFORMAT_FIELD_BITS))

//*****************************************************************************
//
// Number of SPI clocks required to receive a response is:
// = number of RX fields * number of field bits
// = total number of RX bits
//
// In the case of an error, the encoder will respond with ID3.  An error
// can be the wrong TX CRC or an invalid command.  Therefore the RX clocks
// will correspond to an ID3 response or PM_TFORMAT_TX_FIELDS_MAX
//
//*****************************************************************************

#define PM_TFORMAT_RX_CLOCKS_ID0                                               \
                    (PM_TFORMAT_RX_FIELDS_ID0 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_ID1                                               \
                    (PM_TFORMAT_RX_FIELDS_ID1 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_ID2                                               \
                    (PM_TFORMAT_RX_FIELDS_ID2 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_ID3                                               \
                    (PM_TFORMAT_RX_FIELDS_ID3 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_ID6                                               \
                    (PM_TFORMAT_RX_FIELDS_ID6 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_ID7                                               \
                    (PM_TFORMAT_RX_FIELDS_ID7 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_ID8                                               \
                    (PM_TFORMAT_RX_FIELDS_ID8 * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_IDC                                               \
                    (PM_TFORMAT_RX_FIELDS_IDC * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_IDD                                               \
                    (PM_TFORMAT_RX_FIELDS_IDD * PM_TFORMAT_FIELD_BITS)
#define PM_TFORMAT_RX_CLOCKS_MAX                                               \
                    (PM_TFORMAT_RX_FIELDS_MAX * PM_TFORMAT_FIELD_BITS)

#define PM_TFORMAT_RX_CLOCKS_IDX                                               \
                    (PM_TFORMAT_RX_FIELDS_ID3 * PM_TFORMAT_FIELD_BITS)

//
// [doc-library-tformat-txrx-clocks-end]
// [doc-library-tformat-fifo-start]
//
//*****************************************************************************
//
// Since SPISTEN is always active, the RX FIFO will see both the
// TX and RX data.  Thus the required FIFO level is the sum of the two.
// In the current design, this FIFO level is used for both TX and RX.
//
// On some devices the CLB can directly control the SPISTEN signal.  Changing
// the state of the signal is a potential design change that a developer may
// want to consider.
//
//*****************************************************************************
#define PM_TFORMAT_FIFO_LEVEL_ID0                                              \
                    (PM_TFORMAT_RX_FIELDS_ID0 + PM_TFORMAT_TX_FIELDS_ID0)
#define PM_TFORMAT_FIFO_LEVEL_ID1                                              \
                    (PM_TFORMAT_RX_FIELDS_ID1 + PM_TFORMAT_TX_FIELDS_ID1)
#define PM_TFORMAT_FIFO_LEVEL_ID2                                              \
                    (PM_TFORMAT_RX_FIELDS_ID2 + PM_TFORMAT_TX_FIELDS_ID2)
#define PM_TFORMAT_FIFO_LEVEL_ID3                                              \
                    (PM_TFORMAT_RX_FIELDS_ID3 + PM_TFORMAT_TX_FIELDS_ID3)
#define PM_TFORMAT_FIFO_LEVEL_ID6                                              \
                    (PM_TFORMAT_RX_FIELDS_ID6 + PM_TFORMAT_TX_FIELDS_ID6)
#define PM_TFORMAT_FIFO_LEVEL_ID7                                              \
                    (PM_TFORMAT_RX_FIELDS_ID7 + PM_TFORMAT_TX_FIELDS_ID7)
#define PM_TFORMAT_FIFO_LEVEL_ID8                                              \
                    (PM_TFORMAT_RX_FIELDS_ID8 + PM_TFORMAT_TX_FIELDS_ID8)
#define PM_TFORMAT_FIFO_LEVEL_IDC                                              \
                    (PM_TFORMAT_RX_FIELDS_IDC + PM_TFORMAT_TX_FIELDS_IDC)
#define PM_TFORMAT_FIFO_LEVEL_IDD                                              \
                    (PM_TFORMAT_RX_FIELDS_IDD + PM_TFORMAT_TX_FIELDS_IDD)
#define PM_TFORMAT_FIFO_LEVEL_MAX                                              \
                    (PM_TFORMAT_RX_FIELDS_MAX + PM_TFORMAT_TX_FIELDS_MAX)

//
// [doc-library-tformat-fifo-end]
// [doc-library-tformat-crc-len-start]
//
//*****************************************************************************
//
// CRC related declarations
// T-format polynomial: x^8 + 1
//
//*****************************************************************************
#define PM_TFORMAT_CRC_START        0
#define PM_TFORMAT_NBITS_POLY       8u
#define PM_TFORMAT_POLY             0x01u
#define PM_TFORMAT_CRCTABLE_SIZE    256u
#define PM_TFORMAT_CRC_MASK         0xFFu


//*****************************************************************************
//
// Number of bits to use in CRC calculations:
//
// Each field is a total of 10 bits, with 8 bits being actual data.
// This defines the number of bytes, or bits, of data used to calculate the CRC.
// The number of bytes used is one less than the total bytes in the packet.
// The extra byte received is the CRC sent from the encoder.
//
// For example:
//     ID7: RX 6 fields:  CF + SF + DF0 + DF1 + DF2 + CRC
//                        -------------------------
//                        5 fields used to calc CRC
//
//*****************************************************************************
#define PM_TFORMAT_RX_CRC_BYTES_ID0   (PM_TFORMAT_RX_FIELDS_ID0 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_ID1   (PM_TFORMAT_RX_FIELDS_ID1 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_ID2   (PM_TFORMAT_RX_FIELDS_ID2 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_ID3   (PM_TFORMAT_RX_FIELDS_ID3 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_ID6   (PM_TFORMAT_RX_FIELDS_ID6 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_IDD   (PM_TFORMAT_RX_FIELDS_IDD - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_ID7   (PM_TFORMAT_RX_FIELDS_ID7 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_ID8   (PM_TFORMAT_RX_FIELDS_ID8 - 1u)
#define PM_TFORMAT_RX_CRC_BYTES_IDC   (PM_TFORMAT_RX_FIELDS_IDC - 1u)

#define PM_TFORMAT_RX_CRC_BITS_ID0    (PM_TFORMAT_RX_CRC_BYTES_ID0 *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_ID1    (PM_TFORMAT_RX_CRC_BYTES_ID1 *           \
                                             PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_ID2    (PM_TFORMAT_RX_CRC_BYTES_ID2 *           \
                                             PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_ID3    (PM_TFORMAT_RX_CRC_BYTES_ID3 *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_ID6    (PM_TFORMAT_RX_CRC_BYTES_ID6 *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_IDD    (PM_TFORMAT_RX_CRC_BYTES_IDD *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_ID7    (PM_TFORMAT_RX_CRC_BYTES_ID7 *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_ID8    (PM_TFORMAT_RX_CRC_BYTES_ID8 *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_RX_CRC_BITS_IDC    (PM_TFORMAT_RX_CRC_BYTES_IDC *           \
                                            PM_TFORMAT_FIELD_DATA_BITS)

//*****************************************************************************
//
// The number of transmitted bytes, or bits, of data used to calculate the CRC
// for the EEPROM read/write commands.  Other commands do not have a transmit
// CRC.
//
//*****************************************************************************
#define PM_TFORMAT_TX_CRC_BYTES_ID6  (PM_TFORMAT_TX_FIELDS_ID6 - 1u)
#define PM_TFORMAT_TX_CRC_BYTES_IDD  (PM_TFORMAT_TX_FIELDS_IDD - 1u)

#define PM_TFORMAT_TX_CRC_BITS_ID6   (PM_TFORMAT_TX_CRC_BYTES_ID6 *            \
                                            PM_TFORMAT_FIELD_DATA_BITS)
#define PM_TFORMAT_TX_CRC_BITS_IDD   (PM_TFORMAT_TX_CRC_BYTES_IDD *            \
                                            PM_TFORMAT_FIELD_DATA_BITS)

//
// Error codes for certain T-format functions.
// Note: 16-bit error codes are used so they will be different from any valid
// 8-bit CRC value.
//
#define PM_TFORMAT_PASS               0
#define PM_TFORMAT_CRC_CLB_ERROR      0xFFFF
#define PM_TFORMAT_CRC_CHECK_ERROR    0xFFFEP
#define PM_TFORMAT_EEPROM_BUSY        0xFFFD
#define PM_TFORMAT_ID_CHECK_ERROR     0xFFFC

//
// [doc-library-tformat-crc-len-end]
//
//
//! [TFORMAT_API_DATA_STRUCT_SNIPPET]
//

typedef struct PM_tformat_DataStruct {
    uint16_t  controlField;
    uint16_t  statusField;
    uint16_t  dataField0;
    uint16_t  dataField1;
    uint16_t  dataField2;
    uint16_t  dataField3;
    uint16_t  dataField4;
    uint16_t  dataField5;
    uint16_t  dataField6;
    uint16_t  dataField7;
    uint16_t  crcField;
    uint16_t  crcCheck;
    uint16_t  eepromAddressField;
    uint16_t  eepromWrDataField;
    uint16_t  eepromRdDataField;
    volatile Bool  dataReady;
    uint16_t  fifoLevel;
} PM_tformat_DataStruct;

//
//! [TFORMAT_API_DATA_STRUCT_SNIPPET]
//

//*****************************************************************************
//! \addtogroup TFORMAT_API_DATA_STRUCT
//!
//! Information sharing structure
//! @{
//*****************************************************************************

//*****************************************************************************
//!
//! The received T-Format fields are written into this structure by the
//! PM_tformat_receiveDataIDx() functions.
//!
//! \snippet this TFORMAT_API_DATA_STRUCT_SNIPPET
//!
//! \note For debug, set this structure to a known value before calling one
//! of these functions.  This is not required, but can be helpful since these
//! functions do not clear out unused fields.
//!
//! \sa The functions which populate the tformatData fields are:
//!     * PM_tformat_receiveDataID0_1_7_8_C
//!     * PM_tformat_receiveDataID2
//!     * PM_tformat_receiveDataID3
//!     * PM_tformat_receiveDataIDD
//!     * PM_tformat_receiveDataID6
//
//*****************************************************************************
extern PM_tformat_DataStruct tformatData;


//*****************************************************************************
//!
//! The SPI interrupt service routine fills this array with the
//! received data from the encoder.
//! * The first PM_TFORMAT_TX_FIELDS_IDx RX FIFO outputs are skipped.
//!   These were filled during the request transmission and filled with
//!   0xFFFF.
//! * Each FIFO level is read, masked to include only the last 8 bits,
//!   and loaded into the corresponding tformatRxData location.
//!
//! The specific PM_tformat_receiveDataIDx function
//! then uses this array to populate the appropriate received fields in
//! the tformatData structure.
//!
//! \sa The functions which populate the tformatData fields are:
//!     * PM_tformat_receiveDataID0_1_7_8_C
//!     * PM_tformat_receiveDataID2
//!     * PM_tformat_receiveDataID3
//!     * PM_tformat_receiveDataIDD
//!     * PM_tformat_receiveDataID6
//
//*****************************************************************************
extern volatile uint16_t tformatRxData[PM_TFORMAT_FIELDS_MAX];

//
//! @} //end addtogroup TFORMAT_API_DATA_STRUCT
//
// globals
//
//*****************************************************************************

extern uint16_t tformatCRCtable[PM_TFORMAT_CRCTABLE_SIZE];

//*****************************************************************************
//! \defgroup TFORMAT_API_INIT Initialization Functions
//!
//! \brief These are the library functions used to initialize the
//! configurable logic block (CLB), SPI, and CRC lookup table.
//!
//! @{
//*****************************************************************************

//*****************************************************************************
//! \brief Initialize the CRC Lookup Table
//!
//! \note This function is used if the GS32 generates either the
//! transmit or receive CRC.  It is not used if the CLB generates a RX CRC or
//! if the VCRC module generates a TX CRC. This behavior is controlled by
//! the defines in the library header file.
//!
//! Generate a look-up table of 256 entries for a given CRC polynomial
//! with specified number of bits.  This table is only used if the GS32
//! generates a CRC.
//!
//! \param nBits        number of bits of the given polynomial
//! \param polynomial   polynomial used for the CRC calculations
//! \param pTable       pointer to the CRC table
//!
//! \return None.
//!
//*****************************************************************************
void PM_tformat_generateCRCTable(uint16_t nBits,
                                 uint16_t polynomial,
                                 uint16_t *pTable);

//*****************************************************************************
//! \brief Initialize the CLB tiles and SPI module
//!
//! The CLB input MUX was configured earlier in the system-level XBAR
//! configuration.
//!
//! \return none
//
//*****************************************************************************
void PM_tformat_setupPeriph();

//*****************************************************************************
//! \brief Scale to generate T-format clock
//!
//! This function configures the CLB counters in order to generate
//! a T-format clock at 400ns (2.5 MHz frequency).
//!
//! \param freqDiv Specified as TFORMAT_FREQ_DIVIDER in the tformat.h
//! \return none
//
//*****************************************************************************
void PM_tformat_setFreq(uint32_t freqDiv);

//
//! @} //end of addtogroup TFORMAT_API_INIT
//

//*****************************************************************************
//! \defgroup TFORMAT_API_RUN_START_COMMANDS Run-time Command Start Functions
//!
//! \brief Library functions used during run-time to start a command.
//!
//! @{
//*****************************************************************************
//*****************************************************************************
//!
//! \brief Setup a Data Readout or Reset Command
//!
//! This function:
//!     * Loads the PM_TFORMAT_SPI TX FIFO with the command field
//!     * Configures the SPI and CLB to send the command and receive the
//!       response.
//!     * Sets the global variable tformatData.dataReady to false.
//!
//! \param commandID0_1_2_3_7_8_C Command to be sent PM_TFORMAT_IDx
//! \param tformatRXClocks PM_TFORMAT_RX_CLOCKS_IDx SPI clocks required
//!  to send the request
//! \param tformatRXFields PM_TFORMAT_RX_FIELDS_IDx Number of fields in
//!  the response
//! \param tformatTXClocks PM_TFORMAT_TX_CLOCKS_IDx SPI clocks required
//!  to receive the response
//! \param tformatFIFOLevel PM_TFORMAT_FIFO_LEVEL_IDx FIFO level required
//!  to send the request and receive the response
//! \return None.
//!
//! \note
//!     * This function is valid for data readout commands (ID0, 1, 2) and
//!       for reset commands (ID7, 8 and C).
//!     * This function not valid for an EEPROM read (IDD) or
//!       EEPROM write (ID6).
//!
//!  \sa For read or write of EEPROM use:
//!     * PM_tformat_setupCommandReadEEPROM
//!     * PM_tformat_setupCommandWriteEEPROM
//*****************************************************************************
void PM_tformat_setupCommandReadoutOrReset(uint16_t commandID0_1_2_3_7_8_C,
                              uint16_t tformatRXNum,
                              uint16_t tformatRXFields,
                              uint16_t tformatTXNum,
                              uint16_t tformatFIFOLevel);

//*****************************************************************************
//!
//! \brief Setup the Command for an EEPROM Read
//!
//! This function:
//!     * Loads the PM_TFORMAT_SPI TX FIFO with the fields required to read
//!       the encoder's EEPROM.
//!     * Configures the SPI and CLB to send the command and receive the
//!       response
//!     * Sets the global variable tformatData.dataReady to false.
//!
//! \param eepromAddr Address to be read
//! \return None.
//!
//*****************************************************************************
void PM_tformat_setupCommandReadEEPROM(uint16_t eepromAddr);

//*****************************************************************************
//!
//! \brief Setup the Command for an EEPROM Write
//!
//! This function:
//!     * Loads the PM_TFORMAT_SPI TX FIFO with the fields required to write
//!       to the encoder's EEPROM.
//!     * Configures the SPI and CLB to send the command and receive the
//!       response.
//!     * Sets the global variable tformatData.dataReady to false.
//!
//! \param eepromAddr Address to be written to
//! \param eepromData Data to write
//! \return None.
//!
//*****************************************************************************
void PM_tformat_setupCommandWriteEEPROM(uint16_t eepromAddr,
                                        uint16_t eepromData);

//*****************************************************************************
//! \brief Start the T-Format Transaction
//!
//! This function starts the transmission of the request.
//!
//! \sa This function is called after the appropriate setupCommand function:
//!     * PM_tformat_setupCommandReadEEPROM
//!     * PM_tformat_setupCommandWriteEEPROM
//!     * PM_tformat_setupCommand
//!
//! \return none
//
//*****************************************************************************
void PM_tformat_startOperation(void);

//
//! @} //end of addtogroup TFORMAT_API_RUN_START_COMMANDS
//

//*****************************************************************************
//! \defgroup TFORMAT_API_RUN_RECEIVE_DATA Run-time Receive Data Functions
//!
//! \brief Library functions used during run-time to receive data from the
//! tformatRxData[] array.
//!
//! @{
//*****************************************************************************
//*****************************************************************************
//!
//! \brief Receive Data for ID0, ID1, ID7, ID8 or IDC
//!
//! Call this function when tformatData.dataReady is true indicating the data
//! is ready in the tformatRxData array.
//!
//! This function parses the tformatRxData array to populate the following
//! tformatData structure members with the received data:
//!     * controlField
//!     * statusField
//!     * dataField0, dataField1, and dataField2
//!     * crcField
//!
//! The tformatData.crcCheck variable is populated with the calculated CRC
//! of the received data.  This can be used for error checking against
//! the received crcField.
//!
//! \note
//!     * This function is valid for requests ID0, 1, 7, 8 and C only.
//!     * For convenience, the library header file includes aliases to allow
//!       this function to be called using the format:
//!       PM_tformat_receiveDataIDx where x is 0, 1, 7, 8 or C.
//!     * This function does not clear unused members of the tformatData
//!       structure.  This is an optional step the calling application can take
//!       if desired.
//!     * It is the responsibility of the calling application to perform
//!       error checking of the data.
//!
//! \sa Other recieveData functions are:
//!     * PM_tformat_receiveDataID2()
//!     * PM_tformat_receiveDataID3()
//!     * PM_tformat_receiveDataIDD()
//!     * PM_tformat_receiveDataID6()
//! \return none
//!
//*****************************************************************************
void PM_tformat_receiveDataID0_1_7_8_C(void);

//*****************************************************************************
//!
//! \brief Receive Data for ID2
//!
//! Call this function when tformatData.dataReady is true indicating the data
//! is ready in the tformatRxData array.
//!
//! This function parses the tformatRxData array to populate the following
//! tformatData structure members with the received data:
//!     * controlField
//!     * statusField
//!     * dataField0
//!     * crcField
//!
//! The tformatData.crcCheck variable is populated with the calculated CRC
//! of the received data.  This can be used for error checking against
//! the received crcField.
//!
//! \note
//!     * This function is valid for requests ID2 only
//!     * This function does not clear unused members of the tformatData
//!       structure.  This is an optional step the calling application can take
//!       if desired.
//!     * It is the responsibility of the calling application to perform
//!       error checking of the data.
//!
//! \sa Other recieveData functions are:
//!     * PM_tformat_receiveDataID0_1_7_8_C()
//!     * PM_tformat_receiveDataID3()
//!     * PM_tformat_receiveDataIDD()
//!     * PM_tformat_receiveDataID6()
//! \return none
//!
//*****************************************************************************
void PM_tformat_receiveDataID2(void);

//*****************************************************************************
//!
//! \brief Receive Data for ID3
//!
//! Call this function when tformatData.dataReady is true indicating the data
//! is ready in the tformatRxData array.
//!
//! This function parses the tformatRxData array to populate the following
//! tformatData structure members with the received data:
//!     * controlField (CF)
//!     * statusField (SF)
//!     * dataField0 - dataField7 (DF)
//!     * crcField
//!
//! The tformatData.crcCheck variable is populated with the calculated CRC
//! of the received data.  This can be used for error checking against
//! the received crcField.
//!
//! \note
//!     * This function is valid for requests ID3 only
//!     * This function does not clear unused members of the tformatData
//!       structure.  This is an optional step the calling application can take
//!       if desired.
//!     * It is the responsibility of the calling application to perform
//!       error checking of the data.
//!
//! \sa Other recieveData functions are:
//!     * PM_tformat_receiveDataID0_1_7_8_C()
//!     * PM_tformat_receiveDataID2()
//!     * PM_tformat_receiveDataIDD()
//!     * PM_tformat_receiveDataID6()
//! \return none
//!
//*****************************************************************************
void PM_tformat_receiveDataID3(void);

//*****************************************************************************
//!
//! \brief Receive data for IDD (read from EEPROM)
//!
//! Call this function when tformatData.dataReady is true indicating the data
//! is ready in the tformatRxData array.
//!
//! This function parses the tformatRxData array to populate the tformatData
//! structure members:
//!     * controlField (CF)
//!     * eepromAddressField (ADF)
//!     * eepromRdDataField (EDF)
//!     * crcField
//!
//! The tformatData.crcCheck variable is populated with the calculated CRC
//! of the received data.  This can be used for error checking against
//! the received crcField.
//!
//! \note
//!     * This function is valid for requests IDD only
//!     * This function does not clear unused members of the tformatData
//!       structure.  This is an optional step the calling application can take
//!       if desired.
//!     * It is the responsibility of the calling application to perform
//!       error checking of the data.
//!
//! \sa Other recieveData functions are:
//!    * PM_tformat_receiveDataID0_1_7_8_C()
//!    * PM_tformat_receiveDataID2()
//!    * PM_tformat_receiveDataID3()
//!    * PM_tformat_receiveDataID6()
//! \return indicates if the EEPROM was BUSY
//!    * PM_TFORMAT_EEPROM_BUSY  The busy bit was set in the response
//!    * PM_TFORMAT_PASS         The busy bit was not set
//!
//*****************************************************************************
uint16_t PM_tformat_receiveDataIDD(void);

//*****************************************************************************
//!
//! \brief Receive data for ID6 (write to EEPROM)
//!
//! Call this function when tformatData.dataReady is true indicating the data
//! is ready in the tformatRxData array.
//!
//! This function parses the tformatRxData array to populate the tformatData
//! structure members:
//!     * controlField (CF)
//!     * eepromAddressField (ADF)
//!     * eepromWrDataField (EDF)
//!     * crcField
//!
//! The tformatData.crcCheck variable is populated with the calculated CRC
//! of the received data.  This can be used for error checking against
//! the received crcField.
//!
//! \note
//!     * This function is valid for requests ID6 only
//!     * This function does not clear unused members of the tformatData
//!       structure.  This is an optional step the calling application can take
//!       if desired.
//!     * It is the responsibility of the calling application to perform
//!       error checking of the data.
//!
//! \sa Other recieveData functions are:
//!     * PM_tformat_receiveDataID0_1_7_8_C()
//!     * PM_tformat_receiveDataID2()
//!     * PM_tformat_receiveDataID3()
//!     * PM_tformat_receiveDataIDD()
//! \return
//!     * PM_TFORMAT_EEPROM_BUSY  The busy bit was set in the response
//!     * PM_TFORMAT_PASS         The busy bit was not set
//!
//*****************************************************************************
uint16_t PM_tformat_receiveDataID6(void);

//
//! @} //end of addtogroup TFORMAT_API_RUN_RECEIVE_DATA
//

//
// Alias of functions for readability and ease-of-use
//
#define PM_tformat_receiveDataID0 PM_tformat_receiveDataID0_1_7_8_C
#define PM_tformat_receiveDataID1 PM_tformat_receiveDataID0_1_7_8_C
#define PM_tformat_receiveDataID7 PM_tformat_receiveDataID0_1_7_8_C
#define PM_tformat_receiveDataID8 PM_tformat_receiveDataID0_1_7_8_C
#define PM_tformat_receiveDataIDC PM_tformat_receiveDataID0_1_7_8_C

//*****************************************************************************
//! \defgroup TFORMAT_API_RUN_UPDATE_PARAMETERS Run-time Encoder Functions
//!
//! \brief Library functions used during run-time to update the encoder
//! parameters such as turns, position and ID.
//!
//! @{
//*****************************************************************************
//*****************************************************************************
//!
//! \brief Update the position or turns
//!
//! Inverts the received bit sequence for the actual position and turns data
//! Refer to the T-format specification for which command fields contain
//! position or turn information.
//!
//! \param field0  first field received
//! \param field1  second field received
//! \param field2  third field received
//!
//! \return 32-bit position or turns
//!
//*****************************************************************************
uint32_t PM_tformat_updatePositionOrTurns(uint16_t field0,
                                          uint16_t field1,
                                          uint16_t field2 );

//*****************************************************************************
//!
//! \brief Update the encoder ID
//!
//! Inverts the received bit sequence of ENID to form the actual encoder ID
//! Refer to the T-format specification for which command fields contain
//! encoder ID (ENID) information.
//!
//! \param enidField ENID field received from the encoder
//!
//! \return 16-bit encoderID
//!
//*****************************************************************************
uint16_t PM_tformat_updateEncoderID(uint16_t enidField);

uint16_t
tformat_getCRC(uint16_t nBitsData,
                        uint16_t *msg,
                        uint16_t *crcTable,
                        uint16_t rxLen);

//*****************************************************************************
uint16_t tformat_getCRCID0_1_7_8_C(void);

//
//! @copybrief tformat_getCRCID0_1_7_8_C @sa tformat_getCRCID0_1_7_8_C
//
uint16_t tformat_getCRCID2(void);

//
//! @copybrief tformat_getCRCID0_1_7_8_C @sa tformat_getCRCID0_1_7_8_C
//
uint16_t tformat_getCRCID3(void);

//
//! @copybrief tformat_getCRCID0_1_7_8_C @sa tformat_getCRCID0_1_7_8_C
//
uint16_t tformat_getCRCIDD(void);

//
//! @copybrief tformat_getCRCID0_1_7_8_C @sa tformat_getCRCID0_1_7_8_C
//
uint16_t tformat_getCRCID6(void);

#ifdef __cplusplus
}
#endif

#endif /* PM_endat22_Include*/
