/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/**
*   @file    aes.h
*   @brief   Driver for the AES module.
*   @details
*	2024.7.12 Fanzhe add
*/

#ifndef DEVICE_DRIVERLIB_AES_H
#define DEVICE_DRIVERLIB_AES_H

//*****************************************************************************
//
// If building with a C++ compiler, make all of the definitions in this header
// have a C binding.
//
//*****************************************************************************
#ifdef __cplusplus
extern "C"
{
#endif



#include "gs32_version.h"

#if IS_GS32F00xx(0x12)
//*****************************************************************************

// Include files

//*****************************************************************************
#include "debug.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
//#include "device_gs32f00xx/hw_memmap_gs32f00xx.h"
#include "inc/hw_aes.h"


//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the \e configParams.direction
//! parameter.
//
//*****************************************************************************
typedef enum
{
	AES_DIRECTION_DECRYPT = 0x00000000U,	//!< Select Decryption Mode
	AES_DIRECTION_ENCRYPT = 0x00008000U		//!< Select Encryption Mode
} AES_Direction;

//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the \e configParams.keySize
//! parameter.
//
//*****************************************************************************
typedef enum
{
	AES_KEY_SIZE_NULL			= 0x00000000U,
    AES_HASH_KEY_SIZE_128BIT    = 0x00000010U, //!< Select 16-bytes key
    AES_HASH_KEY_SIZE_192BIT    = 0x00000018U, //!< Select 24-bytes key
    AES_HASH_KEY_SIZE_256BIT    = 0x00000020U, //!< Select 32-bytes key
    AES_CIPH_KEY_SIZE_128BIT    = 0x80000010U, //!< Select 16-bytes key
    AES_CIPH_KEY_SIZE_192BIT    = 0x80000018U, //!< Select 24-bytes key
    AES_CIPH_KEY_SIZE_256BIT    = 0x80000020U  //!< Select 32-bytes key
} AES_KeySize;

//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the \e configParams.cipherMode
//! parameter.
//
//*****************************************************************************
typedef enum
{
	AES_CIPH_MODE_NULL 			= 0x00000000U,	//!< Select NULL mode
	AES_CIPH_MODE_ECB 			= 0x00000000U,	//!< Select ECB mode
	AES_CIPH_MODE_CBC 			= 0x00000010U,	//!< Select CBC mode
	AES_CIPH_MODE_CTR_STREAM 	= 0x00000020U,	//!< Select CTR_STREAM mode
	AES_CIPH_MODE_CCM			= 0x00000030U,	//!< Select CCM mode
	AES_CIPH_MODE_GCM_AEAD		= 0x00000050U,	//!< Select GCM_AEAD mode
	AES_CIPH_MODE_OFB			= 0x00000070U,	//!< Select OFB mode
	AES_CIPH_MODE_CFB			= 0x00000080U,	//!< Select CFB mode
	AES_CIPH_MODE_F8			= 0x00000090U,	//!< Select F8 mode
	AES_CIPH_MODE_XTS			= 0x000000A0U	//!< Select XTS mode
} AES_CipherMode;

//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the \e configParams.cipherAlgorithm
//! parameter.
//
//*****************************************************************************
typedef enum
{
	AES_CIPH_ALG_NULL 			= 0x00000000U,
	AES_CIPH_ALG_DES			= 0x00000001U,
	AES_CIPH_ALG_AES			= 0x00000002U,
	AES_CIPH_ALG_RC4			= 0x00000003U,
	AES_CIPH_ALG_MULTI2			= 0x00000004U,
	AES_CIPH_ALG_KASUMI 		= 0x00000005U,
	AES_CIPH_ALG_SNOW_3G_UEA2	= 0x00000006U,
	AES_CIPH_ALG_ZUC_128_EE3	= 0x00000007U,
	AES_CIPH_ALG_CHACHA20		= 0x00000008U,
	AES_CIPH_ALG_SM4			= 0x00000009U
} AES_CipherAlgorithm;

//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the \e configParams.hashMode
//! parameter.
//
//*****************************************************************************
typedef enum
{
	AES_HASH_MODE_NULL			= 0x00000000U,
	AES_HASH_MODE_RAW			= 0x00000000U,
	AES_HASH_MODE_SHAKE			= 0x00000000U,
	AES_HASH_MODE_SSLMAC		= 0x00002000U,
	AES_HASH_MODE_cSHAKE		= 0x00002000U,
	AES_HASH_MODE_HMAC			= 0x00004000U,
	AES_HASH_MODE_KMAC			= 0x00004000U
} AES_HashMode;

//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the \e configParams.hashAlgorithm
//! parameter.
//
//*****************************************************************************
typedef enum
{
	AES_HASH_ALG_NULL				= 0x00000000U,
	AES_HASH_ALG_MD5				= 0X00000100U,
	AES_HASH_ALG_SHA_1				= 0X00000200U,
	AES_HASH_ALG_SHA_224			= 0x00000300U,
	AES_HASH_ALG_SHA_256			= 0x00000400U,
	AES_HASH_ALG_SHA_384			= 0x00000500U,
	AES_HASH_ALG_SHA_512			= 0x00000600U,
	AES_HASH_ALG_AES_XCBC_MAC		= 0x00000700U,
	AES_HASH_ALG_AES_CMAC			= 0x00000800U,
	AES_HASH_ALG_KASUMI_F9			= 0x00000900U,
	AES_HASH_ALG_SNOW_3G_UIA2		= 0x00000A00U,
	AES_HASH_ALG_CRC_32_IEEE_802_3	= 0x00000B00U,
	AES_HASH_ALG_ZUC_128_EIA3		= 0x00000C00U,
	AES_HASH_ALG_SHA_512_224		= 0x00000D00U,
	AES_HASH_ALG_SHA_512_256		= 0x00000E00U,
	AES_HASH_ALG_MICHAEL			= 0x00000F00U,
	AES_HASH_ALG_SHA3_224			= 0x00001000U,
	AES_HASH_ALG_SHA3_256			= 0x00001100U,
	AES_HASH_ALG_SHA3_384			= 0x00001200U,
	AES_HASH_ALG_SHA3_512			= 0x00001300U,
	AES_HASH_ALG_SHAKE128			= 0x00001400U,
	AES_HASH_ALG_SHAKE256			= 0x00001500U,
	AES_HASH_ALG_POLY1305			= 0x00001600U,
	AES_HASH_ALG_SM3				= 0x00001700U,
	AES_HASH_ALG_SM4_XCBC_MAC		= 0x00001800U,
	AES_HASH_ALG_SM4_CMAC			= 0x00001900U
} AES_HashAlgorithm;

//*****************************************************************************
//
//! Values that can be passed to AES_ConfigParams() as the configParams
//! parameter.
//
//*****************************************************************************
typedef struct
{
	AES_Direction			direction;				// Select encrypt or decrypt
	uint32_t*				srcAddr;				// The plaintext address to be encrypted (decrypted)
	uint32_t*				destAddr;				// Encrypted (decrypted) ciphertext address
	uint32_t				packetLength;			// Plaintext length, must be a multiple of 0x10 in cipher mode
	uint32_t				destLength;				// address size
	uint32_t*				keyAddr;				// Key address
	AES_KeySize				keySize;				// Key size
	AES_CipherMode			cipherMode;				// Select encryption (decryption) mode
	AES_CipherAlgorithm		cipherAlgorithm;		// Select the algorithm to use for encryption(decryption)
	AES_HashMode			hashMode;				// Select the Hash mode
	AES_HashAlgorithm		hashAlgorithm;			// Select the Hash algorithm
} AES_ConfigParams;


//*****************************************************************************
//
// Prototypes for the APIs.
//
//*****************************************************************************
//*****************************************************************************
//
//! \internal
//! Checks an AES base address.
//!
//! \param base specifies the AES module base address.
//!
//! This function determines if a AES module base address is valid.
//!
//! \return Returns \b true if the base address is valid and \b false
//! otherwise.
//
//*****************************************************************************
#ifdef DEBUG
__STATIC_FORCEINLINE bool
AES_isBaseVaild(uint32_t base)
{
	return base == AES_BASE;
}
#endif

//*****************************************************************************
//
//! Gets the status whether AES STAT FIFO is empty.
//!
//! \param base is the base address of the AES module.
//!
//! This function can get the status of whether the AES encryption is complete.
//!
//! \return \b false if AES encryption is complete
//! and \b true if it is not.
//
//*****************************************************************************
__STATIC_INLINE boolean
AES_IS_BUSY(uint32_t base)
{
	ASSERT(AES_isBaseVaild(base));

	return ((HWREG(base + AES_O_FIFO_STAT) & AES_FIFO_STAT_EMPTY) != 0U);
}

//*****************************************************************************
//
//! Configures the AES encryption mode.
//!
//! \param base is the base address of the AES module.
//! \param configParams is the desired mode configuration
//!
//! This function configures the mode selection of AES, and the AES module starts
//! encryption as soon as the modification function is executed.
//!
//! The \e configParams is the desired mode selection configuration which is
//! of type \e AES_ConfigParams.
//!
//! \return None.
//
//*****************************************************************************
extern void
AES_configMode(uint32_t base, const AES_ConfigParams *configParams);



#endif	/* #if IS_GS32F00xx(0x12) */

//*****************************************************************************
//
// Mark the end of the C bindings section for C++ compilers.
//
//*****************************************************************************
#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_AES_H */
