/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file     cmpss.h
*   @brief   
*
*   commit history
*   20240423, ZhaoLei, Compatible with 3.0
*   20241226, ZhengYuLong, Update comment
*/

#ifndef DEVICE_DRIVERLIB_CMPSS_H_
#define DEVICE_DRIVERLIB_CMPSS_H_

#ifdef __cplusplus
extern "C"{
#endif

/*
 * Include Files
 */
#include <stdbool.h>
#include <stdint.h>

#include "gs32_version.h"
#include "inc/hw_cmpss.h"
#include "inc/hw_memmap.h"
#include "cpu.h"
#include "debug.h"

#if IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22)||IS_GS32F00xx(0x30)
#define CMPSS_CMPxSETUPTIME_STEP	(CMPSS_O_CMPLSETUPTIME - CMPSS_O_CMPHSETUPTIME)

#define CMPSS_CALIBOFFSET_STEP		(CMPSS_O_CALIBOFFSETL - CMPSS_O_CALIBOFFSETH)

#define CMPSS_HICMP_CTL_M   (CMPSS_COMPCTL_COMPHSOURCE |                       \
                             CMPSS_COMPCTL_COMPHINV |                          \
                             CMPSS_COMPCTL_ASYNCHEN)

#define CMPSS_LOCMP_CTL_M   (CMPSS_COMPCTL_COMPLSOURCE |                       \
                             CMPSS_COMPCTL_COMPLINV |                          \
                             CMPSS_COMPCTL_ASYNCLEN)
#define CMPSS_MODULE_STEP	(CMPSS2_BASE - CMPSS1_BASE)
/*
 * @brief Comparator negative input source
 *
 * @details
 * Values that can be passed to CMPSS_configLowComparator() and
 * CMPSS_configHighComparator() as the config @parameter.
 *
 */
/*
 * @brief Input driven by internal DAC
 */
#define CMPSS_INSRC_DAC                 0x0000U
/*
 * @brief Input driven by external pin
 */
#define CMPSS_INSRC_PIN                 0x0001U

#if IS_GS32F3xx(0x22)
/*
 * @brief High Comparator positive input source
 *
 * @details
 * Values that can be passed to CMPSS_configLowComparator() and
 * CMPSS_configHighComparator() as the config @parameter.
 *
 */
/*
 * @brief Input driven by internal DAC
 */
#define CMPSS_INSRC_DAC_HIGH_POS                 0x0001U
/*
 * @brief Input driven by external pin
 */
#define CMPSS_INSRC_PIN_HIGH_POS                 0x0000U

/*
 * @brief Low Comparator positive input source
 *
 * @details
 * Values that can be passed to CMPSS_configLowComparator() and
 * CMPSS_configHighComparator() as the config @parameter.
 *
 */
/*
 * @brief Input driven by internal DAC.
 */
#define CMPSS_INSRC_DAC_LOW_NEG                 0x0000U
/*
 * @brief Input driven by external pin.
 */
#define CMPSS_INSRC_PIN_LOW_NEG	               	0x0001U
#endif

/*
 * @brief Comparator output is inverted
 */
#define CMPSS_INV_INVERTED              0x0002U
/*
 * @brief Asynch comparator output feeds into OR with latched digital filter output
 */
#define CMPSS_OR_ASYNC_OUT_W_FILT       0x0040U

/*
 * @brief Signal driving CTRIPOUT
 *
 * @details
 * Values that can be passed to CMPSS_configOutputsLow() and
 * CMPSS_configOutputsHigh() as the config @parameter.
 *
 */
/*
 * @brief Asynchronous comparator output drives CTRIPOUT
 */
#define CMPSS_TRIPOUT_ASYNC_COMP        0x0000U
/*
 * @brief Synchronous comparator output drives CTRIPOUT
 */
#define CMPSS_TRIPOUT_SYNC_COMP         0x0010U
/*
 * @brief Filter output drives CTRIPOUT
 */
#define CMPSS_TRIPOUT_FILTER            0x0020U
/*
 * @brief Latched filter output drives CTRIPOUT
 */
#define CMPSS_TRIPOUT_LATCH             0x0030U

/*
 * @brief Signal driving CTRIP
 *
 * @details
 * Values that can be passed to CMPSS_configOutputsLow() and
 * CMPSS_configOutputsHigh() as the config @parameter.
 *
 */
/*
 * @brief Asynchronous comparator output drives CTRIP
 */
#define CMPSS_TRIP_ASYNC_COMP           0x0000U
/*
 * @brief Synchronous comparator output drives CTRIP
 */
#define CMPSS_TRIP_SYNC_COMP            0x0004U
/*
 * @brief Filter output drives CTRIP
 */
#define CMPSS_TRIP_FILTER               0x0008U
/*
 * @brief Latched filter output drives CTRIP
 */
#define CMPSS_TRIP_LATCH                0x000CU

/*
 * @brief Values that can be returned by CMPSS_getStatus().
 */
/*
 * @brief High digital filter output
 */
#define CMPSS_STS_HI_FILTOUT            0x0001U
/*
 * @brief Latched value of high digital filter output
 */
#define CMPSS_STS_HI_LATCHFILTOUT       0x0002U
/*
 * @brief Low digital filter output
 */
#define CMPSS_STS_LO_FILTOUT            0x0100U
/*
 * @brief Latched value of low digital filter output
 */
#define CMPSS_STS_LO_LATCHFILTOUT       0x0200U

/*
 * @brief When is DAC value loaded from shadow register
 *
 * @details
 * Values that can be passed to CMPSS_configDAC() the config @parameter.
 */
/*
 * @brief	DAC value updated from SYSCLK
 */
#define CMPSS_DACVAL_SYSCLK             0x0000U
/*
 * @brief	DAC value updated from PWMSYNC
 */
#define CMPSS_DACVAL_PWMSYNC            0x0080U

/*
 * @brief DAC reference voltage
 */
/*
 * @brief VDDA is the voltage reference
 */
#define CMPSS_DACREF_VDDA               0x0000U
/*
 * @brief VDAC is the voltage reference
 */
#define CMPSS_DACREF_VDAC               0x0020U

/*
 * @brief DAC value source
 */
/*
 * @brief DAC value updated from shadow register
 */
#define CMPSS_DACSRC_SHDW               0x0000U
/*
 * @brief DAC value is updated from the ramp register
 */
#define CMPSS_DACSRC_RAMP               0x0001U

/*
 * @Values that can be passed to CMPSS_configRamp() as the pwmSyncSrc parameter.
 */
#define CMPSS_PWMSYNC1  1U /* PWMSYNC1 */
#define CMPSS_PWMSYNC2  2U /* PWMSYNC2 */
#define CMPSS_PWMSYNC3  3U /* PWMSYNC3 */
#define CMPSS_PWMSYNC4  4U /* PWMSYNC4 */
#define CMPSS_PWMSYNC5  5U /* PWMSYNC5 */
#define CMPSS_PWMSYNC6  6U /* PWMSYNC6 */
#define CMPSS_PWMSYNC7  7U /* PWMSYNC7 */
#define CMPSS_PWMSYNC8  8U /* PWMSYNC8 */
#endif
/*
 * @brief
 *
 * @param CMPSS_COMPARATORS_LOW  CMPSS module analog comparators low
 * @param CMPSS_COMPARATORS_HIGH CMPSS module analog comparators high
 */

#ifdef DEBUG
#if IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22)
__STATIC_INLINE bool
CMPSS_isBaseValid(uint32_t base)
{
    return(
           (base == CMPSS1_BASE) ||
           (base == CMPSS2_BASE) ||
           (base == CMPSS3_BASE) ||
           (base == CMPSS4_BASE)
#ifdef CMPSS5_BASE
        || (base == CMPSS5_BASE)
#endif
#ifdef CMPSS6_BASE
        || (base == CMPSS6_BASE)
#endif
#ifdef CMPSS7_BASE
        || (base == CMPSS7_BASE)
#endif
          );
}
#elif IS_GS32F00xx(0x30)
__STATIC_INLINE
bool CMPSS_isBaseValid(uint32_t base)
{
    return(
          (base == CMPSS1_BASE) ||
          (base == CMPSS2_BASE) ||
          (base == CMPSS3_BASE) ||
          (base == CMPSS4_BASE)
          );
}

//Lite
__STATIC_INLINE
bool CMPSS_isLiteBaseValid(uint32_t base)
{
    return(
               (base == CMPSS2_BASE) ||
               (base == CMPSS3_BASE) ||
               (base == CMPSS4_BASE)
              );
}
#endif
#endif
/*
 * @brief Enables the CMPSS module.
 *
 * @details
 * This function enables the CMPSS module passed into the base @parameter.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 */
#if IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22)||IS_GS32F00xx(0x30)
__STATIC_INLINE void
CMPSS_enableModule(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that enables the CMPSS module. */
   HWREGH(base + CMPSS_O_COMPCTL) |= CMPSS_COMPCTL_COMPDACE;

}

/*
 * @brief Disables the CMPSS module.
 *
 * @details
 * This function disables the CMPSS module passed into the base @parameter.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 *
 */
__STATIC_INLINE void
CMPSS_disableModule(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Clear the bit that enables the CMPSS module. */
    HWREGH(base + CMPSS_O_COMPCTL) &= ~CMPSS_COMPCTL_COMPDACE;

}

/*
 * @brief Sets the configuration for the high comparator.
 *
 * @details
 * This function configures a comparator. The config @parameter is the
 * result of a logical OR operation between a CMPSS_INSRC_xxx value and if
 * desired,CMPSS_INV_INVERTED and \b CMPSS_OR_ASYNC_OUT_W_FILT values.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in] 	config  Is the configuration of the high comparator.
 *
 * @remarks
 * - The CMPSS_INSRC_xxx term can take on the following values to specify
 *   the high comparator negative input source:
 *   - CMPSS_INSRC_DAC - The internal DAC.
 *   - CMPSS_INSRC_PIN - An external pin.
 * - CMPSS_INV_INVERTED may be ORed into config if the comparator output
 * 	 should be inverted.
 * - CMPSS_OR_ASYNC_OUT_W_FILT may be ORed into config if the asynchronous
 * 	 comparator output should be fed into an OR gate with the latched digital
 * 	 filter output before it is made available for CTRIPH or CTRIPOUTH.
 *
 */
__STATIC_INLINE void
CMPSS_configHighComparator(uint32_t base, uint16_t config)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the high comparator configuration to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPCTL) =
        (HWREGH(base + CMPSS_O_COMPCTL) & ~CMPSS_HICMP_CTL_M) | config;

    uint16_t shift = (base-CMPSS1_BASE) / CMPSS_MODULE_STEP;
#if IS_GS32F00xx(0x30)
    if (config & CMPSS_COMPCTL_COMPHSOURCE )
    {
    	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
    			(ANA_SYSCTRL_PARA_CMPX_HN_EN_M<<(shift * ANA_SYSCTRL_PARA_CMPMUX_STEP));
	}
    else {
    	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) &=
    			(~(ANA_SYSCTRL_PARA_CMPX_HN_EN_M<<(shift * ANA_SYSCTRL_PARA_CMPMUX_STEP)));
	}
#endif

}

/*
 * @brief Sets the configuration for the low comparator.
 *
 * @details
 * This function configures a comparator. The config @parameter is the
 * result of a logical OR operation between a CMPSS_INSRC_xxx value and if
 * desired,CMPSS_INV_INVERTED and \b CMPSS_OR_ASYNC_OUT_W_FILT values.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in] 	config  Is the configuration of the low comparator.
 *
 * @remarks
 * - The CMPSS_INSRC_xxx term can take on the following values to specify
 *   the low comparator negative input source:
 *   - CMPSS_INSRC_DAC - The internal DAC.
 *   - CMPSS_INSRC_PIN - An external pin.
 * - CMPSS_INV_INVERTED may be ORed into config if the comparator output
 * 	 should be inverted.
 * - CMPSS_OR_ASYNC_OUT_W_FILT may be ORed into config if the asynchronous
 * 	 comparator output should be fed into an OR gate with the latched digital
 * 	 filter output before it is made available for CTRIPH or CTRIPOUTH.
 *
 */
__STATIC_INLINE void
CMPSS_configLowComparator(uint32_t base, uint16_t config)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the low comparator configuration to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPCTL) =
        (HWREGH(base + CMPSS_O_COMPCTL) & ~CMPSS_LOCMP_CTL_M) | (config << 8U);
    uint16_t shift = (base-CMPSS1_BASE) / CMPSS_MODULE_STEP;
#if IS_GS32F00xx(0x30)
    if (config & CMPSS_COMPCTL_COMPHSOURCE )
    {
    	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) |=
    			(ANA_SYSCTRL_PARA_CMPX_LN_EN_M<<(shift * ANA_SYSCTRL_PARA_CMPMUX_STEP));
	}
    else {
    	HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_CMP_MX_EN) &=
    			(~(ANA_SYSCTRL_PARA_CMPX_LN_EN_M<<(shift * ANA_SYSCTRL_PARA_CMPMUX_STEP)));
	}
#endif
}

/*
 * @brief Sets the output signal configuration for the high comparator.
 *
 * @details
 * This function configures a comparator's output signals CTRIP and CTRIPOUT.
 * The config @parameter is the result of a logical OR operation between the
 * CMPSS_TRIPOUT_xxx and CMPSS_TRIP_xxx values.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in] 	config  Is the configuration of the high comparator output signals.
 *
 * @remarks
 * - The CMPSS_TRIPOUT_xxx term can take on the following values to specify
 *   which signal drives CTRIPOUTH:
 *   - CMPSS_TRIPOUT_ASYNC_COMP - The asynchronous comparator output.
 *   - CMPSS_TRIPOUT_SYNC_COMP - The synchronous comparator output.
 *   - CMPSS_TRIPOUT_FILTER - The output of the digital filter.
 *   - CMPSS_TRIPOUT_LATCH - The latched output of the digital filter.
 * - The CMPSS_TRIP_xxx term can take on the following values to specify
 * 	 which signal drives CTRIPH:
 * 	 - CMPSS_TRIP_ASYNC_COMP - The asynchronous comparator output.
 * 	 - CMPSS_TRIP_SYNC_COMP - The synchronous comparator output.
 * 	 - CMPSS_TRIP_FILTER - The output of the digital filter.
 * 	 - CMPSS_TRIP_LATCH - The latched output of the digital filter.
 *
 */
__STATIC_INLINE void
CMPSS_configOutputsHigh(uint32_t base, uint16_t config)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the high comparator output settings to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPCTL) = (HWREGH(base + CMPSS_O_COMPCTL) &
                                      ~(CMPSS_COMPCTL_CTRIPOUTHSEL_M |
                                        CMPSS_COMPCTL_CTRIPHSEL_M))  |
                                     config;

}

/*
 * @brief Sets the output signal configuration for the low comparator.
 *
 * @details
 * This function configures a comparator's output signals CTRIP and CTRIPOUT.
 * The config @parameter is the result of a logical OR operation between the
 * CMPSS_TRIPOUT_xxx and CMPSS_TRIP_xxx values.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in] 	config  Is the configuration of the low comparator output signals.
 *
 * @remarks
 * - The CMPSS_TRIPOUT_xxx term can take on the following values to specify
 *   which signal drives CTRIPOUTH:
 *   - CMPSS_TRIPOUT_ASYNC_COMP - The asynchronous comparator output.
 *   - CMPSS_TRIPOUT_SYNC_COMP - The synchronous comparator output.
 *   - CMPSS_TRIPOUT_FILTER - The output of the digital filter.
 *   - CMPSS_TRIPOUT_LATCH - The latched output of the digital filter.
 * - The CMPSS_TRIP_xxx term can take on the following values to specify
 * 	 which signal drives CTRIPH:
 * 	 - CMPSS_TRIP_ASYNC_COMP - The asynchronous comparator output.
 * 	 - CMPSS_TRIP_SYNC_COMP - The synchronous comparator output.
 * 	 - CMPSS_TRIP_FILTER - The output of the digital filter.
 * 	 - CMPSS_TRIP_LATCH - The latched output of the digital filter.
 *
 */
__STATIC_INLINE void
CMPSS_configOutputsLow(uint32_t base, uint16_t config)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the low comparator output settings to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPCTL) = (HWREGH(base + CMPSS_O_COMPCTL) &
                                      ~(CMPSS_COMPCTL_CTRIPOUTLSEL_M |
                                        CMPSS_COMPCTL_CTRIPLSEL_M)) |
                                     (config << 8U);

}

/*
 * @brief Gets the current comparator status.
 *
 * @details
 * This function returns the current status for the comparator, specifically
 * the digital filter output and latched digital filter output.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 *
 * @return	Returns the current interrupt status
 *
 * @remarks
 * - enumerated as a bit field of the following values:
 * 	 - CMPSS_STS_HI_FILTOUT - High digital filter output
 * 	 - CMPSS_STS_HI_LATCHFILTOUT - Latched value of high digital filter output
 * 	 - CMPSS_STS_LO_FILTOUT - Low digital filter output
 * 	 - CMPSS_STS_LO_LATCHFILTOUT - Latched value of low digital filter output
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getStatus(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Return contents of the status register. */
    return(HWREGH(base + CMPSS_O_COMPSTS));
}

/*
 * @brief Sets the configuration for the internal comparator DACs.
 *
 * @details
 * This function configures the comparator's internal DAC.  The config
 * @parameter is the result of a logical OR operation between the
 * CMPSS_DACVAL_xxx, CMPSS_DACREF_xxx, and CMPSS_DACSRC_xxx.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in]	config  Is the configuration of the internal DAC.
 *
 * @remarks
 * - The CMPSS_DACVAL_xxx term can take on the following values to specify
 * 	 when the DAC value is loaded from its shadow register:
 * 	 - CMPSS_DACVAL_SYSCLK - Value register updated on system clock.
 * 	 - CMPSS_DACVAL_PWMSYNC - Value register updated on PWM sync.
 * - The CMPSS_DACREF_xxx term can take on the following values to specify
 * 	 which voltage supply is used as reference for the DACs:
 * 	 - CMPSS_DACREF_VDDA - VDDA is the voltage reference for the DAC.
 * 	 - CMPSS_DACREF_VDAC - VDAC is the voltage reference for the DAC.
 * - The CMPSS_DACSRC_xxx term can take on the following values to specify
 * 	 the DAC value source for the high comparator's internal DAC:
 * 	 - CMPSS_DACSRC_SHDW - The user-programmed DACVALS register.
 * 	 - CMPSS_DACSRC_RAMP - The ramp generator RAMPSTS register.
 *
 * @note
 * - The CMPSS_DACVAL_xxx and CMPSS_DACREF_xxx terms apply to both the
 * 	 high and low comparators.CMPSS_DACSRC_xxx will only affect the high
 * 	 comparator's internal DAC.
 *
 */
#endif
#if IS_GS32F00xx(0x12)||IS_GS32F3xx(0x22)
__STATIC_INLINE void
CMPSS_configDAC(uint32_t base, uint16_t config)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the DAC configuration to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPDACCTL) =
                    (HWREGH(base + CMPSS_O_COMPDACCTL) &
                     ~(CMPSS_COMPDACCTL_SWLOADSEL | CMPSS_COMPDACCTL_SELREF |
                       CMPSS_COMPDACCTL_DACSOURCE)) | config;
}

/*
 * @brief Sets the value of the internal DAC of the high comparator.
 *
 * @details
 * This function sets the 12-bit value driven by the internal DAC of the high
 * comparator. This function will load the value into the shadow register from
 * which the actual DAC value register will be loaded. To configure which
 * event causes this shadow load to take place, use CMPSS_configDAC().
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in]	value 	Is the value actively driven by the DAC.
 *
 */
__STATIC_INLINE void
CMPSS_setDACValueHigh(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 4096U);

    /* Write the DAC value to the DAC value shadow register. */
    HWREGH(base + CMPSS_O_DACHVALS) = value;
}

/*
 * @brief Sets the value of the internal DAC of the low comparator.
 *
 * @details
 * This function sets the 12-bit value driven by the internal DAC of the low
 * comparator. This function will load the value into the shadow register from
 * which the actual DAC value register will be loaded. To configure which
 * event causes this shadow load to take place, use CMPSS_configDAC().
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value 	The value actively driven by the DAC.
 *
 */
__STATIC_INLINE void
CMPSS_setDACValueLow(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 4096U);

    /* Write the DAC value to the DAC value shadow register. */
    HWREGH(base + CMPSS_O_DACLVALS) = value;
}

/*
 * @brief Initializes the digital filter of the high comparator.
 *
 * @details
 * This function initializes all the samples in the high comparator digital
 * filter to the filter input value.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @remarks
 * See CMPSS_configFilterHigh() for the proper initialization sequence
 * to avoid glitches.
 *
 */
__STATIC_INLINE void
CMPSS_initFilterHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the high comparator filter initialization bit. */
    HWREGH(base + CMPSS_O_CTRIPHFILCTL) |= CMPSS_CTRIPHFILCTL_FILINIT;

}

/*
 * @brief Initializes the digital filter of the low comparator.
 *
 * @details
 * This function initializes all the samples in the low comparator digital
 * filter to the filter input value.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @remarks
 * See CMPSS_configFilterLow() for the proper initialization sequence
 * to avoid glitches.
 *
 */
__STATIC_INLINE void
CMPSS_initFilterLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the low comparator filter initialization bit. */
    HWREGH(base + CMPSS_O_CTRIPLFILCTL) |= CMPSS_CTRIPLFILCTL_FILINIT;

}

/*
 * @brief Gets the value of the internal DAC of the high comparator.
 *
 * @details
 * This function gets the value of the internal DAC of the high comparator.
 * The value is read from the active register--not the shadow register to
 * which CMPSS_setDACValueHigh() writes.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the value driven by the internal DAC of the high comparator.
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getDACValueHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Read the DAC value to the DAC value active register. */
    return(HWREGH(base + CMPSS_O_DACHVALA));
}

/*
 * @brief Gets the value of the internal DAC of the low comparator.
 *
 * @details
 * This function gets the value of the internal DAC of the low comparator.
 * The value is read from the active register--not the shadow register to
 * which CMPSS_setDACValueLow() writes.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the value driven by the internal DAC of the low comparator.
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getDACValueLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Read the DAC value to the DAC value active register. */
    return(HWREGH(base + CMPSS_O_DACLVALA));
}

/*
 * @brief
 * Causes a software reset of the high comparator digital filter output latch.
 *
 * @details
 * This function causes a software reset of the high comparator digital filter
 * output latch. It will generate a single pulse of the latch reset signal.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_clearFilterLatchHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that generates a reset pulse to the digital filter latch. */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_HLATCHCLR;

}

/*
 * @brief
 * Causes a software reset of the low comparator digital filter output latch.
 *
 * @details
 * This function causes a software reset of the low comparator digital filter
 * output latch. It will generate a single pulse of the latch reset signal.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_clearFilterLatchLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that generates a reset pulse to the digital filter latch. */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_LLATCHCLR;

}

/*
 * @brief Sets the ramp generator maximum reference value.
 *
 * @details
 * This function sets the ramp maximum reference value that will be loaded
 * into the ramp generator.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value   The ramp maximum reference value.
 *
 */
__STATIC_INLINE void
CMPSS_setMaxRampValue(uint32_t base,  uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the maximum ramp value to the shadow register. */
    HWREGH(base + CMPSS_O_RAMPMAXREFS) = value;
}

/*
 * @brief Gets the ramp generator maximum reference value.
 *
 * @details
 * This function sets the ramp maximum reference value that will be loaded
 * into the ramp generator.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the latched ramp maximum reference value that will be
 *		   loaded into the ramp generator.
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getMaxRampValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Read the maximum ramp value from the register. */
    return(HWREGH(base + CMPSS_O_RAMPMAXREFA));
}

/*
 * @brief Sets the ramp generator decrement value.
 *
 * @details
 * This function sets the value that is subtracted from the ramp value on
 * every system clock cycle.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in] 	value 	The ramp decrement value.
 *
 */
__STATIC_INLINE void
CMPSS_setRampDecValue(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the ramp decrement value to the shadow register. */
    HWREGH(base + CMPSS_O_RAMPDECVALS) = value;
}

/*
 * @brief Gets the ramp generator decrement value.
 *
 * @details
 * This function sets the value that is subtracted from the ramp value on
 * every system clock cycle.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the latched ramp decrement value that is subtracted from
 * 		   the ramp value on every system clock cycle.
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getRampDecValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Read the ramp decrement value from the register. */
    return(HWREGH(base + CMPSS_O_RAMPDECVALA));
}

/*
 * @brief Sets the ramp generator delay value.
 *
 * @details
 * This function sets the value that configures the number of system clock
 * cycles to delay the start of the ramp generator decrementer after a PWMSYNC
 * event is received. Delay value can be no greater than 8191.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value   The 13-bit ramp delay value.
 *
 */
__STATIC_INLINE void
CMPSS_setRampDelayValue(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
	ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 8192U);

    /* Write the ramp delay value to the shadow register. */
    HWREGH(base + CMPSS_O_RAMPDLYS) = value;
}

/*
 * @brief Gets the ramp generator delay value.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the latched ramp delay value that is subtracted from
 * 		   the ramp value on every system clock cycle.
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getRampDelayValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the ramp delay value to the shadow register. */
    return(HWREGH(base + CMPSS_O_RAMPDLYA));
}

/*
 * @brief Configures sync source for comparator
 *
 * @details
 * This function configures desired EPWMxSYNCPER source for comparator
 * blocks. Configured EPWMxSYNCPER signal can be used to synchronize loading
 * of DAC input value from shadow to active register. It can also be used to
 * synchronize Ramp generator, if applicable. Refer to device manual to check
 * if Ramp generator is available in the desired CMPSS instance.
 *
 * @param[in]	base		The base address of the comparator module.
 * @param[in]	syncSource	The desired EPWMxSYNCPER source.
 *
 * @remarks
 * - Valid values for syncSource parameter can be 1 to n, where n represents
 * 	 the maximum number of EPWMSYNCPER signals available on the device. For
 * 	 instance, passing 2 into syncSource will select EPWM2SYNCPER.
 *
 */
static inline void
CMPSS_configureSyncSource(uint32_t base, uint16_t syncSource)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the ramp delay value to the shadow register. */
    HWREGH(base + CMPSS_O_COMPDACCTL) = (HWREGH(base + CMPSS_O_COMPDACCTL) &
                                          ~CMPSS_COMPDACCTL_RAMPSOURCE_M)   |
                                         ((uint16_t)(syncSource - 1U)       <<
                                          CMPSS_COMPDACCTL_RAMPSOURCE_S);
}

/*
 * @brief Sets the comparator hysteresis settings.
 *
 * @details
 * This function sets the amount of hysteresis on the comparator inputs. The
 * value @parameter indicates the amount of hysteresis desired. Passing in 0
 * results in none, passing in 1 results in typical hysteresis, passing in 2
 * results in 2x of typical hysteresis, and so on where value x of typical
 * hysteresis is the amount configured.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value 	The amount of hysteresis on the comparator inputs.
 *
 */
__STATIC_INLINE void
CMPSS_setHysteresis(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value <= 4U);

    /* Read the ramp delay value from the register. */
    HWREGH(base + CMPSS_O_COMPHYSCTL) = value;

}

/*
 * @brief
 * Enables reset of HIGH comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function enables EPWMSYNCPER reset of High comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_enableLatchResetOnPWMSYNCHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Enable EPWMSYNCPER reset of high comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_HSYNCCLREN;

}

/*
 * @brief
 * Disables reset of HIGH comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function disables EPWMSYNCPER reset of High comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_disableLatchResetOnPWMSYNCHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Disable EPWMSYNCPER reset of high comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) &= ~CMPSS_COMPSTSCLR_HSYNCCLREN;

}

/*
 * @brief
 * Enables reset of LOW comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function enables EPWMSYNCPER reset of Low comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_enableLatchResetOnPWMSYNCLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Enable EPWMSYNCPER reset of low comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_LSYNCCLREN;

}

/*
 * @brief
 * Disables reset of LOW comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function disables EPWMSYNCPER reset of Low comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_disableLatchResetOnPWMSYNCLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Disable EPWMSYNCPER reset of low comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) &= ~CMPSS_COMPSTSCLR_LSYNCCLREN;

}

/*
 * @brief
 * Sets the ePWM module blanking signal that holds trip in reset.
 *
 * @details
 * This function configures which PWMBLANK signal from the ePWM module will
 * hold trip in reset when blanking is enabled.
 *
 * @param[in]	base		The base address of the comparator module.
 * @param[in]	pwmBlankSrc The number of the PWMBLANK source.
 *
 * @remarks
 * - The number of the PWMBLANK signal to be used to reset the ramp generator
 * 	 should be specified by passing it into the pwmBlankSrc @parameter.For
 * 	 instance, passing a 2 into pwmBlankSrc will select PWMBLANK2.
 *
 */
__STATIC_INLINE void
CMPSS_configBlanking(uint32_t base, uint16_t pwmBlankSrc)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((pwmBlankSrc >= 1U) && (pwmBlankSrc <= 16U));

    /* Write the blank source number to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPDACCTL) =
        (HWREGH(base + CMPSS_O_COMPDACCTL) & ~CMPSS_COMPDACCTL_BLANKSOURCE_M) |
        ((pwmBlankSrc - 1U) << CMPSS_COMPDACCTL_BLANKSOURCE_S);

}

/*
 * @brief
 * Enables an ePWM blanking signal to hold trip in reset.
 *
 * @details
 * This function enables a selected ePWM blanking signal to hold trip in
 * reset.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_enableBlanking(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that enables the PWMBLANK signal. */
    HWREGH(base + CMPSS_O_COMPDACCTL) |= CMPSS_COMPDACCTL_BLANKEN;

}

/*
 * @brief
 * Disables an ePWM blanking signal from holding trip in reset.
 *
 * @details
 * This function disables a selected ePWM blanking signal from holding trip in
 * reset.
 *
 * @param[in] base The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_disableBlanking(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Clear the bit that enables the PWMBLANK signal. */
    HWREGH(base + CMPSS_O_COMPDACCTL) &= ~CMPSS_COMPDACCTL_BLANKEN;
}

/*
 * @brief set LP source - internal DAC or external input
 *
 * @param[in] base 		The base address of the comparator module.
 * @param[in] lp_src	1-external input, 0-internal DAC(default)
 *
 */
__STATIC_INLINE uint16_t
CMPSS_setCmpLpSrc(uint32_t base, uint32_t lp_src)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* set HP source */
    HWREGH(base + CMPSS_O_CMPDACCTL)  = (HWREGH(base + CMPSS_O_CMPDACCTL) &
    										(~(CMPSS_CMPDACCTRL_LPSRC_M))) |
											(lp_src<<CMPSS_CMPDACCTRL_LPSRC_S);

    return 0;
}

/*
 * @brief set HP source - internal DAC or external input
 *
 * @param[in] base 		The base address of the comparator module.
 * @param[in] lp_src	1-external input, 0-internal DAC(default)
 *
 */
__STATIC_INLINE uint16_t
CMPSS_setCmpHpSrc(uint32_t base, uint32_t hp_src)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* set HP source */
    HWREGH(base + CMPSS_O_CMPDACCTL)  = (HWREGH(base + CMPSS_O_CMPDACCTL) & (~(CMPSS_CMPDACCTRL_HPSRC_M))) | (hp_src<<CMPSS_CMPDACCTRL_HPSRC_S);

    return 0;
}


/*
 * @brief Configures the digital filter of the high comparator.
 *
 * @details
 * This function configures the operation of the digital filter of the high
 * comparator.
 *
 * @param	base	The base address of the comparator module.
 * @param	samplePrescale	The number of system clock cycles between samples.
 * @param 	sampleWindow 	The number of FIFO samples to monitor.
 * @param 	threshold 		The majority threshold of samples to change state.
 *
 * @remarks
 * - The samplePrescale @parameter specifies the number of system clock cycles
 * 	 between samples. It is a 16-bit value so a number higher than 65535 should
 * 	 not be passed as this @parameter. The prescaler used by digital filter is 1
 * 	 more than samplePrescale value. So, the input provided should be 1 less than
 * 	 the expected prescaler.
 * - The sampleWindow @parameter configures the size of the window of FIFO
 * 	 samples taken from the input that will be monitored to determine when to
 * 	 change the filter output. This sample window may be no larger than 32
 * 	 samples.
 * - The threshold @parameter configures the threshold value to be used by
 * 	 the digital filter.
 * - The filter output resolves to the majority value of the sample window where
 * 	 majority is defined by the value passed into the threshold @parameter.
 * 	 For proper operation, the value of \e threshold must be greater than
 * 	 sampleWindow / 2.
 * - To ensure proper operation of the filter, the following is the recommended
 * 	 function call sequence for initialization:
 * 	 - Configure and enable the comparator using CMPSS_configHighComparator()
 * 	   and CMPSS_enableModule()
 * 	 - Configure the digital filter using CMPSS_configFilterHigh()
 * 	 - Initialize the sample values using CMPSS_initFilterHigh()
 * 	 - Configure the module output signals CTRIP and CTRIPOUT using
 * 	   CMPSS_configOutputsHigh()
 *
 */
extern void
CMPSS_configFilterHigh(uint32_t base, uint16_t samplePrescale,
                       uint16_t sampleWindow, uint16_t threshold);

/*
 * @brief Configures the digital filter of the low comparator.
 *
 * @details
 * This function configures the operation of the digital filter of the low
 * comparator.
 *
 * @param[in]	base			The base address of the comparator module.
 * @param[in]	samplePrescale	The number of system clock cycles between samples.
 * @param[in]	sampleWindow	The number of FIFO samples to monitor.
 * @param[in]	threshold 		The majority threshold of samples to change state.
 *
 * @remarks
 * - The samplePrescale @parameter specifies the number of system clock cycles
 * 	 between samples. It is a 16-bit value so a number higher than 65535 should
 * 	 not be passed as this @parameter. The prescaler used by digital filter is 1
 * 	 more than samplePrescale value. So, the input provided should be 1 less
 * 	 than the expected prescaler.
 * - The sampleWindow @parameter configures the size of the window of FIFO
 *   samples taken from the input that will be monitored to determine when to
 *   change the filter output. This sample window may be no larger than 32
 *   samples.
 * - The threshold @parameter configures the threshold value to be used by the
 *   digital filter.
 * - The filter output resolves to the majority value of the sample window where
 *   majority is defined by the value passed into the \e threshold @parameter.
 *   For proper operation, the value of \e threshold must be greater than
 *   sampleWindow / 2.
 * - To ensure proper operation of the filter, the following is the recommended
 *   function call sequence for initialization:
 *   - Configure and enable the comparator using CMPSS_configLowComparator()
 *     and CMPSS_enableModule()
 *   - Configure the digital filter using CMPSS_configFilterLow()
 *   - Initialize the sample values using CMPSS_initFilterLow()
 *   - Configure the module output signals CTRIP and CTRIPOUT using
 *     CMPSS_configOutputsLow()
 */
extern void
CMPSS_configFilterLow(uint32_t base, uint16_t samplePrescale,
                      uint16_t sampleWindow, uint16_t threshold);

/*****************************************************************************
 *
 * @brief Configures whether or not the digital filter latches are reset by PWMSYNC
 *
 * @details
 * This function configures whether or not the digital filter latches in both
 * the high and low comparators should be reset by PWMSYNC. If the highEnable
 * @parameter is true, the PWMSYNC will be allowed to reset the high comparator's
 * digital filter latch. If it is false, the ability of the PWMSYNC to reset the
 * latch will be disabled. The lowEnable @parameter has the same effect on the low
 * comparator's digital filter latch.
 *
 * @param[in]	base		The base address of the comparator module.
 * @param[in]	highEnable  Indicates filter latch settings in the high comparator.
 * @param[in]	lowEnable 	Indicates filter latch settings in the low comparator.
 *
 */
extern void
CMPSS_configLatchOnPWMSYNC(uint32_t base, bool highEnable, bool lowEnable);

/*
 * @brief Configures the comparator subsystem's ramp generator.
 *
 * @details
 * This function configures many of the main settings of the comparator
 * subsystem's ramp generator. The maxRampVal @parameter should be passed
 * the ramp maximum reference value that will be loaded into the ramp
 * generator. The decrementVal @parameter should be passed the decrement
 * value that will be subtracted from the ramp generator on each system clock
 * cycle. The delayVal @parameter should be passed the 13-bit number of
 * system clock cycles the ramp generator should delay before beginning to
 * decrement the ramp generator after a PWMSYNC signal is received.
 *
 * @param[in]	base 			The base address of the comparator module.
 * @param[in]	maxRampVal		The ramp maximum reference value.
 * @param[in]	decrementVal	Value is the ramp decrement value.
 * @param[in]	delayVal 		The ramp delay value.
 * @param[in]	pwmSyncSrc 		The number of the PWMSYNC source.
 * @param[in]	useRampValShdw 	Indicates if the max ramp shadow should be used.
 *
 *
 * These three values may be be set individually using the
 * CMPSS_setMaxRampValue(), CMPSS_setRampDecValue(), and
 * CMPSS_setRampDelayValue() APIs.
 *
 * The number of the PWMSYNC signal to be used to reset the ramp generator
 * should be specified by passing it into the pwmSyncSrc @parameter. For
 * instance, passing a 2 into pwmSyncSrc will select PWMSYNC2.
 *
 * To indicate whether the ramp generator should reset with the value from the
 * ramp max reference value shadow register or with the latched ramp max
 * reference value, use the useRampValShdw parameter. Passing it true
 * will result in the latched value being bypassed. The ramp generator will be
 * loaded right from the shadow register. A value of false will load the
 * ramp generator from the latched value.
 *
 */
extern void
CMPSS_configRamp(uint32_t base, uint16_t maxRampVal, uint16_t decrementVal,
                 uint16_t delayVal, uint16_t pwmSyncSrc, bool useRampValShdw);

#endif




#if IS_GS32F00xx(0x30)


typedef enum
{
    CMPSS_FILTIN_COMPOUT = 0x0000,  //!< Filter input driven by comp output
    CMPSS_FILTIN_EXT_H1  = 0x0001,  //!< Filter input driven by EXT_FILTIN_H[1]
    CMPSS_FILTIN_EXT_H2  = 0x0002,  //!< Filter input driven by EXT_FILTIN_H[2]
    CMPSS_FILTIN_EXT_H3  = 0x0003,  //!< Filter input driven by EXT_FILTIN_H[3]
    CMPSS_FILTIN_EXT_H4  = 0x0004,  //!< Filter input driven by EXT_FILTIN_H[4]
    CMPSS_FILTIN_EXT_H5  = 0x0005,  //!< Filter input driven by EXT_FILTIN_H[5]
    CMPSS_FILTIN_EXT_H6  = 0x0006,  //!< Filter input driven by EXT_FILTIN_H[6]
    CMPSS_FILTIN_EXT_H7  = 0x0007,  //!< Filter input driven by EXT_FILTIN_H[7]

} CMPSS_FilterInput;

//*****************************************************************************
//
// Values that can be passed to CMPSS_setRampDirectionHigh() as the dir
// parameter.
//
//*****************************************************************************
typedef enum
{
    CMPSS_RAMP_DIR_DOWN = 0x0000,                    //!< Decrementing Ramp
    CMPSS_RAMP_DIR_UP   = CMPSS_COMPDACHCTL_RAMPDIR, //!< Incrementing Ramp
} CMPSS_RampDirection;

//*****************************************************************************
//
//! Values that can be passed to CMPSS_configureRampXTriggerHigh() as the
//! \e trigger parameter.
//
//*****************************************************************************
typedef enum
{
    CMPSS_RAMP_XTRIGGER_DISABLE   = 0x0000,//!< RAMPH & RAMPL SOR trigger is
                                           //!< the independently configured
                                           //!< PWMSYNCx signal
    CMPSS_RAMP_XTRIGGER_RAMPH_EOR = 0x0001,//!< RAMPH SOR trigger is the
                                           //!< configured PWMSYNCx signal &
                                           //!< RAMPL SOR trigger is RAMPH EOR
    CMPSS_RAMP_XTRIGGER_RAMPL_EOR = 0x0002 //!< RAMPL SOR trigger is the
                                           //!< configured PWMSYNCx signal and
                                           //!< RAMPH SOR trigger is RAMPL EOR
} CMPSS_RampXTrigger;


//*****************************************************************************
//
//! Values that can be passed to CMPSS_selectDEACTIVESource() API
//! as the \e deactivesel parameter.
//
//*****************************************************************************
typedef enum
{
    CMPSS_DEACTIVE_EPWM1   = 0x00U,  //!< EPWM1 DEACTIVE
    CMPSS_DEACTIVE_EPWM2   = 0x01U,  //!< EPWM2 DEACTIVE
    CMPSS_DEACTIVE_EPWM3   = 0x02U,  //!< EPWM3 DEACTIVE
    CMPSS_DEACTIVE_EPWM4   = 0x03U,  //!< EPWM4 DEACTIVE
    CMPSS_DEACTIVE_EPWM5   = 0x04U,  //!< EPWM5 DEACTIVE
    CMPSS_DEACTIVE_EPWM6   = 0x05U,  //!< EPWM6 DEACTIVE
    CMPSS_DEACTIVE_EPWM7   = 0x06U,  //!< EPWM7 DEACTIVE
    CMPSS_DEACTIVE_EPWM8   = 0x07U,  //!< EPWM8 DEACTIVE
    CMPSS_DEACTIVE_EPWM9   = 0x08U,  //!< EPWM9 DEACTIVE
    CMPSS_DEACTIVE_EPWM10  = 0x09U,  //!< EPWM10 DEACTIVE
    CMPSS_DEACTIVE_EPWM11  = 0x0AU,  //!< EPWM11 DEACTIVE
    CMPSS_DEACTIVE_EPWM12  = 0x0BU,  //!< EPWM12 DEACTIVE
    CMPSS_DEACTIVE_EPWM13  = 0x0CU,  //!< EPWM13 DEACTIVE
    CMPSS_DEACTIVE_EPWM14  = 0x0DU,  //!< EPWM14 DEACTIVE
    CMPSS_DEACTIVE_EPWM15  = 0x0EU,  //!< EPWM15 DEACTIVE
    CMPSS_DEACTIVE_EPWM16  = 0x0FU,  //!< EPWM16 DEACTIVE
    CMPSS_DEACTIVE_EPWM17  = 0x10U,  //!< EPWM17 DEACTIVE
    CMPSS_DEACTIVE_EPWM18  = 0x11U,  //!< EPWM18 DEACTIVE
} CMPSS_DEActiveSelect;

//*****************************************************************************
//
// Values that can be passed to CMPSS_setRampClockDividerHigh() and
// CMPSS_setRampClockDividerLow() APIs as the divider parameter.
//
//*****************************************************************************
typedef enum
{
    CMPSS_RAMP_CLOCK_DIV1   = 0x0U,  //!< RAMPCLK = SYSCLK/1
    CMPSS_RAMP_CLOCK_DIV2   = 0x1U,  //!< RAMPCLK = SYSCLK/2
    CMPSS_RAMP_CLOCK_DIV3   = 0x2U,  //!< RAMPCLK = SYSCLK/3
    CMPSS_RAMP_CLOCK_DIV4   = 0x3U,  //!< RAMPCLK = SYSCLK/4
    CMPSS_RAMP_CLOCK_DIV5   = 0x4U,  //!< RAMPCLK = SYSCLK/5
    CMPSS_RAMP_CLOCK_DIV6   = 0x5U,  //!< RAMPCLK = SYSCLK/6
    CMPSS_RAMP_CLOCK_DIV7   = 0x6U,  //!< RAMPCLK = SYSCLK/7
    CMPSS_RAMP_CLOCK_DIV8   = 0x7U,  //!< RAMPCLK = SYSCLK/8
    CMPSS_RAMP_CLOCK_DIV9   = 0x8U,  //!< RAMPCLK = SYSCLK/9
    CMPSS_RAMP_CLOCK_DIV10  = 0x9U,  //!< RAMPCLK = SYSCLK/10
    CMPSS_RAMP_CLOCK_DIV11  = 0xAU,  //!< RAMPCLK = SYSCLK/11
    CMPSS_RAMP_CLOCK_DIV12  = 0xBU,  //!< RAMPCLK = SYSCLK/12
    CMPSS_RAMP_CLOCK_DIV13  = 0xCU,  //!< RAMPCLK = SYSCLK/13
    CMPSS_RAMP_CLOCK_DIV14  = 0xDU,  //!< RAMPCLK = SYSCLK/14
    CMPSS_RAMP_CLOCK_DIV15  = 0xEU,  //!< RAMPCLK = SYSCLK/15
    CMPSS_RAMP_CLOCK_DIV16  = 0xFU,  //!< RAMPCLK = SYSCLK/16
} CMPSS_RampClockDiv;

//*****************************************************************************
//
// Values that can be passed to CMPSS_selectBlankSourceGroupHigh(),
// CMPSS_selectBlankSourceGroupLow(), CMPSS_selectRampSourceGroupHigh() and
// CMPSS_selectRampSourceGroupLow() APIs as the group parameter.
//
//*****************************************************************************
typedef enum
{
    CMPSS_EPWM_GROUP_0TO15   = 0x0U,  //!< Select EPWM0-EPWM15 group
    CMPSS_EPWM_GROUP_16TO31  = 0x1U,  //!< Select EPWM16-EPWM31 group
} CMPSS_EPWMGroup;
/*
 * @brief
 *
 * @param CMPSS_COMPARATORS_LOW  CMPSS module analog comparators low
 * @param CMPSS_COMPARATORS_HIGH CMPSS module analog comparators high
 */
typedef enum
{
	CMPSS_COMPARATORS_LOW = 0U,
	CMPSS_COMPARATORS_HIGH = 1U
}CMPSS_CMPID;

/*
 * @brief Enables the CMPSS module.
 *
 * @details
 * This function enables the CMPSS module passed into the base @parameter.
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 */


__STATIC_INLINE void
CMPSS_configDAC(uint32_t base, uint16_t config)   
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the DAC configuration to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPDACHCTL) =
                    (HWREGH(base + CMPSS_O_COMPDACHCTL) &
                     ~(CMPSS_COMPDACHCTL_SWLOADSEL | CMPSS_COMPDACHCTL_SELREF |
                       CMPSS_COMPDACHCTL_DACSOURCE)) | config;

}
/*
 * @brief Sets the value of the internal DAC of the high comparator.
 *
 * @details
 * This function sets the 12-bit value driven by the internal DAC of the high
 * comparator. This function will load the value into the shadow register from
 * which the actual DAC value register will be loaded. To configure which
 * event causes this shadow load to take place, use CMPSS_configDAC().
 *
 * @param[in]	base	It is the base address of the CMPSS module.
 * @param[in]	value 	Is the value actively driven by the DAC.
 *
 */
__STATIC_INLINE void
CMPSS_configDACHigh(uint32_t base, uint16_t config)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the DAC configuration to the appropriate register.
    //
    EALLOW;

    HWREGH(base + CMPSS_O_COMPDACHCTL) = (HWREGH(base + CMPSS_O_COMPDACHCTL) &
                                          ~(CMPSS_COMPDACHCTL_SWLOADSEL      |
                                            CMPSS_COMPDACHCTL_SELREF         |
                                        CMPSS_COMPDACHCTL_DACSOURCE)) | config;
    EDIS;
}
__STATIC_INLINE void
CMPSS_configDACLow(uint32_t base, uint16_t config)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the DAC configuration to the appropriate register.
    //
    EALLOW;

    HWREGH(base + CMPSS_O_COMPDACLCTL) = (HWREGH(base + CMPSS_O_COMPDACLCTL) &
                                          ~(CMPSS_COMPDACLCTL_DACSOURCE))    |
                                          config;

    EDIS;
}
__STATIC_INLINE void
CMPSS_setDACValueHigh(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 4096U);

    /* Write the DAC value to the DAC value shadow register. */
    HWREGH(base + CMPSS_O_DACHVALS) = value;   
}

/*
 * @brief Sets the value of the internal DAC of the low comparator.
 *
 * @details
 * This function sets the 12-bit value driven by the internal DAC of the low
 * comparator. This function will load the value into the shadow register from
 * which the actual DAC value register will be loaded. To configure which
 * event causes this shadow load to take place, use CMPSS_configDAC().
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value 	The value actively driven by the DAC.
 *
 */
__STATIC_INLINE void
CMPSS_setDACValueLow(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 4096U);

    /* Write the DAC value to the DAC value shadow register. */
    HWREGH(base + CMPSS_O_DACLVALS) = value;   
}

/*
 * @brief Initializes the digital filter of the high comparator.
 *
 * @details
 * This function initializes all the samples in the high comparator digital
 * filter to the filter input value.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @remarks
 * See CMPSS_configFilterHigh() for the proper initialization sequence
 * to avoid glitches.
 *
 */
__STATIC_INLINE void
CMPSS_initFilterHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the high comparator filter initialization bit. */
    HWREGH(base + CMPSS_O_CTRIPHFILCTL) |= CMPSS_CTRIPHFILCTL_FILINIT;    

}

/*
 * @brief Initializes the digital filter of the low comparator.
 *
 * @details
 * This function initializes all the samples in the low comparator digital
 * filter to the filter input value.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @remarks
 * See CMPSS_configFilterLow() for the proper initialization sequence
 * to avoid glitches.
 *
 */
__STATIC_INLINE void
CMPSS_initFilterLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the low comparator filter initialization bit. */
    HWREGH(base + CMPSS_O_CTRIPLFILCTL) |= CMPSS_CTRIPLFILCTL_FILINIT;   

}

/*
 * @brief Gets the value of the internal DAC of the high comparator.
 *
 * @details
 * This function gets the value of the internal DAC of the high comparator.
 * The value is read from the active register--not the shadow register to
 * which CMPSS_setDACValueHigh() writes.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the value driven by the internal DAC of the high comparator.
 *
 */
__STATIC_INLINE void
CMPSS_configureFilterInputHigh(uint32_t base, CMPSS_FilterInput filtInput)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)filtInput <= 7U);

    //
    // Set the high comparator filter initialization bit.
    //
    EALLOW;

    HWREGH(base + CMPSS_O_CTRIPHFILCTL) = (HWREGH(base + CMPSS_O_CTRIPHFILCTL) &
                                           ~CMPSS_CTRIPHFILCTL_FILTINSEL_M)    |
                                          (uint16_t)filtInput ;

    EDIS;
}
__STATIC_INLINE void
CMPSS_configureFilterInputLow(uint32_t base, CMPSS_FilterInput filtInput)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)filtInput <= 7U);

    //
    // Set the low comparator filter initialization bit.
    //
    EALLOW;

    HWREGH(base + CMPSS_O_CTRIPLFILCTL) = (HWREGH(base + CMPSS_O_CTRIPLFILCTL) &
                                           ~CMPSS_CTRIPLFILCTL_FILTINSEL_M)    |
                                          (uint16_t)filtInput;

    EDIS;
}
__STATIC_INLINE uint16_t
CMPSS_getDACValueHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Read the DAC value to the DAC value active register. */
    return(HWREGH(base + CMPSS_O_DACHVALA));   
}

/*
 * @brief Gets the value of the internal DAC of the low comparator.
 *
 * @details
 * This function gets the value of the internal DAC of the low comparator.
 * The value is read from the active register--not the shadow register to
 * which CMPSS_setDACValueLow() writes.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the value driven by the internal DAC of the low comparator.
 *
 */
__STATIC_INLINE uint16_t
CMPSS_getDACValueLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Read the DAC value to the DAC value active register. */
    return(HWREGH(base + CMPSS_O_DACLVALA));   
}

/*
 * @brief
 * Causes a software reset of the high comparator digital filter output latch.
 *
 * @details
 * This function causes a software reset of the high comparator digital filter
 * output latch. It will generate a single pulse of the latch reset signal.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_clearFilterLatchHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that generates a reset pulse to the digital filter latch. */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_HLATCHCLR;

}

/*
 * @brief
 * Causes a software reset of the low comparator digital filter output latch.
 *
 * @details
 * This function causes a software reset of the low comparator digital filter
 * output latch. It will generate a single pulse of the latch reset signal.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_clearFilterLatchLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that generates a reset pulse to the digital filter latch. */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_LLATCHCLR;   

}

/*
 * @brief Sets the ramp generator maximum reference value.
 *
 * @details
 * This function sets the ramp maximum reference value that will be loaded
 * into the ramp generator.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value   The ramp maximum reference value.
 *
 */

__STATIC_INLINE void
CMPSS_setMaxRampValue(uint32_t base,  uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(CMPSS_isLiteBaseValid(base));
    /* Write the maximum ramp value to the shadow register. */
    HWREGH(base + CMPSS_O_RAMPHREFS) = value;   
}
/*
 * @brief Gets the ramp generator maximum reference value.
 *
 * @details
 * This function sets the ramp maximum reference value that will be loaded
 * into the ramp generator.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the latched ramp maximum reference value that will be
 *		   loaded into the ramp generator.
 *
 */

__STATIC_INLINE uint16_t
CMPSS_getMaxRampValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(CMPSS_isLiteBaseValid(base));
    /* Read the maximum ramp value from the register. */
    return(HWREGH(base + CMPSS_O_RAMPHREFS));   
}
/*
 * @brief Sets the ramp generator decrement value.
 *
 * @details
 * This function sets the value that is subtracted from the ramp value on
 * every system clock cycle.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in] 	value 	The ramp decrement value.
 *
 */

__STATIC_INLINE void
CMPSS_setRampDecValue(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(CMPSS_isLiteBaseValid(base));
    /* Write the ramp decrement value to the shadow register. */
    HWREGH(base + CMPSS_O_RAMPHSTEPVALS) = value;    
}

/*
 * @brief Gets the ramp generator decrement value.
 *
 * @details
 * This function sets the value that is subtracted from the ramp value on
 * every system clock cycle.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the latched ramp decrement value that is subtracted from
 * 		   the ramp value on every system clock cycle.
 *
 */

__STATIC_INLINE uint16_t
CMPSS_getRampDecValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(CMPSS_isLiteBaseValid(base));
    /* Read the ramp decrement value from the register. */
    return(HWREGH(base + CMPSS_O_RAMPHSTEPVALA));   
}
/*
 * @brief Sets the ramp generator delay value.
 *
 * @details
 * This function sets the value that configures the number of system clock
 * cycles to delay the start of the ramp generator decrementer after a PWMSYNC
 * event is received. Delay value can be no greater than 8191.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value   The 13-bit ramp delay value.
 *
 */

__STATIC_INLINE void
CMPSS_setRampDelayValue(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
	ASSERT(CMPSS_isBaseValid(base));
	ASSERT(CMPSS_isLiteBaseValid(base));
    ASSERT(value < 8192U);

    /* Write the ramp delay value to the shadow register. */
    HWREGH(base + CMPSS_O_RAMPHDLYS) = value;
}
/*
 * @brief Gets the ramp generator delay value.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 * @return Returns the latched ramp delay value that is subtracted from
 * 		   the ramp value on every system clock cycle.
 *
 */

__STATIC_INLINE uint16_t
CMPSS_getRampDelayValue(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(CMPSS_isLiteBaseValid(base));
    /* Write the ramp delay value to the shadow register. */
    return(HWREGH(base + CMPSS_O_RAMPHDLYA));  
}
__STATIC_INLINE void
CMPSS_setRampDirectionHigh(uint32_t base, CMPSS_RampDirection dir)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp direction value to the shadow register.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL) = (HWREGH(base + CMPSS_O_COMPDACHCTL) &
                                          ~(CMPSS_COMPDACHCTL_RAMPDIR)) |
                                          (uint16_t)dir;
    EDIS;
}
__STATIC_INLINE void
CMPSS_setRampReferenceHigh(uint32_t base, uint16_t value)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp reference value to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPHREFS) = value;
}
__STATIC_INLINE uint16_t
CMPSS_getRampReferenceHigh(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the maximum ramp value from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPHREFA));
}

__STATIC_INLINE void
CMPSS_setRampStepHigh(uint32_t base, uint16_t value)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp step value to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPHSTEPVALS) = value;
}
__STATIC_INLINE uint16_t
CMPSS_getRampStepHigh(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the high ramp generator step value from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPHSTEPVALA));
}
__STATIC_INLINE void
CMPSS_setRampDelayHigh(uint32_t base, uint16_t value)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 8192U);

    //
    // Write the ramp delay value to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPHDLYS) = value;
}
__STATIC_INLINE uint16_t
CMPSS_getRampDelayHigh(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the ramp delay value from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPHDLYA));
}
__STATIC_INLINE void
CMPSS_setRampDirectionLow(uint32_t base, CMPSS_RampDirection dir)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp direction value to the shadow register.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACLCTL) = (HWREGH(base + CMPSS_O_COMPDACLCTL) &
                                          ~(CMPSS_COMPDACLCTL_RAMPDIR)) |
                                          (uint16_t)dir;
    EDIS;
}
__STATIC_INLINE void
CMPSS_setRampReferenceLow(uint32_t base, uint16_t value)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp reference value to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPLREFS) = value;
}
__STATIC_INLINE uint16_t
CMPSS_getRampReferenceLow(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the ramp reference value from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPLREFA));
}
__STATIC_INLINE void
CMPSS_setRampStepLow(uint32_t base, uint16_t value)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp step value to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPLSTEPVALS) = value;
}
__STATIC_INLINE uint16_t
CMPSS_getRampStepLow(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the low ramp generator step value from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPLSTEPVALA));
}
__STATIC_INLINE void
CMPSS_setRampDelayLow(uint32_t base, uint16_t value)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value < 8192U);

    //
    // Write the ramp delay value to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPLDLYS) = value;
}
__STATIC_INLINE uint16_t
CMPSS_getRampDelayLow(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the ramp delay value from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPLDLYA));
}
__STATIC_INLINE void
CMPSS_configureRampXTriggerHigh(uint32_t base, CMPSS_RampXTrigger trigger)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp delay value to the shadow register.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL2) =
                                     (HWREGH(base + CMPSS_O_COMPDACHCTL2) &
                                      ~CMPSS_COMPDACHCTL2_XTRIGCFG_M)     |
                                     ((uint16_t)trigger <<
                                      CMPSS_COMPDACHCTL2_XTRIGCFG_S);
    EDIS;
}
__STATIC_INLINE void
CMPSS_configureSyncSourceHigh(uint32_t base, uint16_t syncSource)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp delay value to the shadow register.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL) = (HWREGH(base + CMPSS_O_COMPDACHCTL) &
                                          ~CMPSS_COMPDACHCTL_RAMPSOURCE_M)   |
                                         ((uint16_t)(syncSource - 1U)       <<
                                          CMPSS_COMPDACHCTL_RAMPSOURCE_S);
    EDIS;
}
__STATIC_INLINE void
CMPSS_configureSyncSourceLow(uint32_t base, uint16_t syncSource)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Write the ramp delay value to the shadow register.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACLCTL) = (HWREGH(base + CMPSS_O_COMPDACLCTL) &
                                          ~CMPSS_COMPDACLCTL_RAMPSOURCE_M)   |
                                         ((uint16_t)(syncSource - 1U)       <<
                                          CMPSS_COMPDACLCTL_RAMPSOURCE_S);
    EDIS;
}
__STATIC_INLINE void
CMPSS_setRampClockDividerHigh(uint32_t base, CMPSS_RampClockDiv divider)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    //ASSERT((uint16_t)divider <= CMPSS_RAMPHCTLS_RAMPCLKDIV_M);

    //
    // Write the high ramp generator clock divider to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPHCTLS) = (uint16_t)divider;
}
__STATIC_INLINE uint16_t
CMPSS_getRampClockDividerHigh(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the high ramp generator clock divider from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPHCTLA) & CMPSS_RAMPHCTLA_RAMPCLKDIV_M);
}
__STATIC_INLINE void
CMPSS_setRampClockDividerLow(uint32_t base, CMPSS_RampClockDiv divider)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)divider <= CMPSS_RAMPLCTLS_RAMPCLKDIV_M);

    //
    // Write the low ramp generator clock divider to the shadow register.
    //
    HWREGH(base + CMPSS_O_RAMPLCTLS) = (uint16_t)divider;
}
__STATIC_INLINE uint16_t
CMPSS_getRampClockDividerLow(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Read the low ramp generator clock divider from the register.
    //
    return(HWREGH(base + CMPSS_O_RAMPLCTLA) & CMPSS_RAMPLCTLA_RAMPCLKDIV_M);
}

__STATIC_INLINE void
CMPSS_configureSyncSource(uint32_t base, uint16_t syncSource)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Write the ramp delay value to the shadow register. */
    HWREGH(base + CMPSS_O_COMPDACHCTL) = (HWREGH(base + CMPSS_O_COMPDACHCTL) &
                                          ~CMPSS_COMPDACHCTL_RAMPSOURCE_M)   |
                                         ((uint16_t)(syncSource - 1U)       <<
                                          CMPSS_COMPDACHCTL_RAMPSOURCE_S);
}
/*
 * @brief Sets the comparator hysteresis settings.
 *
 * @details
 * This function sets the amount of hysteresis on the comparator inputs. The
 * value @parameter indicates the amount of hysteresis desired. Passing in 0
 * results in none, passing in 1 results in typical hysteresis, passing in 2
 * results in 2x of typical hysteresis, and so on where value x of typical
 * hysteresis is the amount configured.
 *
 * @param[in]	base	The base address of the comparator module.
 * @param[in]	value 	The amount of hysteresis on the comparator inputs.
 *
 */

__STATIC_INLINE void
CMPSS_setHysteresis(uint32_t base, uint16_t value)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(value <= 4U);

    /* Write the comparator hysteresis settings value to the register. */
    HWREGH(base + CMPSS_O_COMPHYSCTL) = value;     

}
/*
 * @brief
 * Enables reset of HIGH comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function enables EPWMSYNCPER reset of High comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */

__STATIC_INLINE void
CMPSS_enableLatchResetOnPWMSYNCHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Enable EPWMSYNCPER reset of high comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_HSYNCCLREN;    

}
/*
 * @brief
 * Disables reset of HIGH comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function disables EPWMSYNCPER reset of High comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_disableLatchResetOnPWMSYNCHigh(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Disable EPWMSYNCPER reset of high comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) &= ~CMPSS_COMPSTSCLR_HSYNCCLREN;   

}

/*
 * @brief
 * Enables reset of LOW comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function enables EPWMSYNCPER reset of Low comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_enableLatchResetOnPWMSYNCLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Enable EPWMSYNCPER reset of low comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) |= CMPSS_COMPSTSCLR_LSYNCCLREN;

}

/*
 * @brief
 * Disables reset of LOW comparator digital filter output latch on PWMSYNC
 *
 * @details
 * This function disables EPWMSYNCPER reset of Low comparator digital filter
 * output latch.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_disableLatchResetOnPWMSYNCLow(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Disable EPWMSYNCPER reset of low comparator digital filter output latch */
    HWREGH(base + CMPSS_O_COMPSTSCLR) &= ~CMPSS_COMPSTSCLR_LSYNCCLREN;   

}

/*
 * @brief
 * Sets the ePWM module blanking signal that holds trip in reset.
 *
 * @details
 * This function configures which PWMBLANK signal from the ePWM module will
 * hold trip in reset when blanking is enabled.
 *
 * @param[in]	base		The base address of the comparator module.
 * @param[in]	pwmBlankSrc The number of the PWMBLANK source.
 *
 * @remarks
 * - The number of the PWMBLANK signal to be used to reset the ramp generator
 * 	 should be specified by passing it into the pwmBlankSrc @parameter.For
 * 	 instance, passing a 2 into pwmBlankSrc will select PWMBLANK2.
 *
 */
__STATIC_INLINE void
CMPSS_configBlanking(uint32_t base, uint16_t pwmBlankSrc)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((pwmBlankSrc >= 1U) && (pwmBlankSrc <= 16U));

    /* Write the blank source number to the appropriate register. */
    HWREGH(base + CMPSS_O_COMPDACHCTL) =
        (HWREGH(base + CMPSS_O_COMPDACHCTL) & (~CMPSS_COMPDACHCTL_BLANKSOURCE_M)) |
        ((pwmBlankSrc - 1U) << CMPSS_COMPDACHCTL_BLANKSOURCE_S);

}

/*
 * @brief
 * Enables an ePWM blanking signal to hold trip in reset.
 *
 * @details
 * This function enables a selected ePWM blanking signal to hold trip in
 * reset.
 *
 * @param[in]	base	The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_enableBlanking(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Set the bit that enables the PWMBLANK signal. */
    HWREGH(base + CMPSS_O_COMPDACHCTL) |= CMPSS_COMPDACHCTL_BLANKEN;  

}

/*
 * @brief
 * Disables an ePWM blanking signal from holding trip in reset.
 *
 * @details
 * This function disables a selected ePWM blanking signal from holding trip in
 * reset.
 *
 * @param[in] base The base address of the comparator module.
 *
 */
__STATIC_INLINE void
CMPSS_disableBlanking(uint32_t base)
{
    /* Check the arguments. */
    ASSERT(CMPSS_isBaseValid(base));

    /* Clear the bit that enables the PWMBLANK signal. */
    HWREGH(base + CMPSS_O_COMPDACHCTL) &= ~CMPSS_COMPDACHCTL_BLANKEN; 
}

/*
 * @brief set LP source - internal DAC or external input
 *
 * @param[in] base 		The base address of the comparator module.
 * @param[in] lp_src	1-external input, 0-internal DAC(default)
 *
 */
__STATIC_INLINE void
CMPSS_configBlankingSourceHigh(uint32_t base, uint16_t pwmBlankSrc)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((pwmBlankSrc >= 1U) && (pwmBlankSrc <= 16U));

    //
    // Write the blank source number to the appropriate register.
    //
    EALLOW;

    HWREGH(base + CMPSS_O_COMPDACHCTL) =
       (HWREGH(base + CMPSS_O_COMPDACHCTL) & ~CMPSS_COMPDACHCTL_BLANKSOURCE_M) |
       ((pwmBlankSrc - 1U) << CMPSS_COMPDACHCTL_BLANKSOURCE_S);

    EDIS;
}
__STATIC_INLINE void
CMPSS_enableBlankingHigh(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Set the bit that enables the PWMBLANK signal.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL) |= CMPSS_COMPDACHCTL_BLANKEN;
    EDIS;
}
__STATIC_INLINE void
CMPSS_disableBlankingHigh(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Clear the bit that enables the PWMBLANK signal.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL) &= ~CMPSS_COMPDACHCTL_BLANKEN;
    EDIS;
}
__STATIC_INLINE void
CMPSS_configBlankingSourceLow(uint32_t base, uint16_t pwmBlankSrc)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((pwmBlankSrc >= 1U) && (pwmBlankSrc <= 16U));

    //
    // Write the blank source number to the appropriate register.
    //
    EALLOW;

    HWREGH(base + CMPSS_O_COMPDACLCTL) = (HWREGH(base + CMPSS_O_COMPDACLCTL) &
                        ~CMPSS_COMPDACLCTL_BLANKSOURCE_M) |
                        ((pwmBlankSrc - 1U) << CMPSS_COMPDACLCTL_BLANKSOURCE_S);

    EDIS;
}
__STATIC_INLINE void
CMPSS_enableBlankingLow(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Set the bit that enables the PWMBLANK signal.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACLCTL) |= CMPSS_COMPDACLCTL_BLANKEN;
    EDIS;
}
__STATIC_INLINE void
CMPSS_disableBlankingLow(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Clear the bit that enables the PWMBLANK signal.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACLCTL) &= ~CMPSS_COMPDACLCTL_BLANKEN;
    EDIS;
}
__STATIC_INLINE void
CMPSS_enableDEmode(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Enable diode emulation mode.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL2) |=  CMPSS_COMPDACHCTL2_DEENABLE;
    EDIS;
}
__STATIC_INLINE void
CMPSS_disableDEmode(uint32_t base)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));

    //
    // Disable diode emulation mode.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL2) &=  ~CMPSS_COMPDACHCTL2_DEENABLE;
    EDIS;
}
__STATIC_INLINE void
CMPSS_selectDEACTIVESource(uint32_t base, uint16_t deactivesel)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(deactivesel <= 31U);

    //
    // Select diode emulation active source.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL2) =
                             (HWREGH(base + CMPSS_O_COMPDACHCTL2)  &
                              ~(CMPSS_COMPDACHCTL2_DEACTIVESEL_M)) |
                             (deactivesel << CMPSS_COMPDACHCTL2_DEACTIVESEL_S);
    EDIS;
}
__STATIC_INLINE void
CMPSS_configHighDACShadowValueDE(uint32_t base, uint16_t dacval)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(dacval <= 4095U);

    //
    // Configure high comp DAC shadow value 2.
    //
    HWREGH(base + CMPSS_O_DACHVALS2) = (dacval & CMPSS_DACHVALS2_DACVAL_M);
}
__STATIC_INLINE void
CMPSS_configLowDACShadowValueDE(uint32_t base, uint16_t dacval)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT(dacval <= 4095U);

    //
    // Configure low comp DAC shadow value 2.
    //
    HWREGH(base + CMPSS_O_DACLVALS2) = (dacval & CMPSS_DACLVALS2_DACVAL_M);
}
__STATIC_INLINE void
CMPSS_selectBlankSourceGroupHigh(uint32_t base, CMPSS_EPWMGroup group)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)group <= 1U);

    //
    // Configure the Blanking source group.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL2) =
                                (HWREGH(base + CMPSS_O_COMPDACHCTL2) &
                                 ~CMPSS_COMPDACHCTL2_BLANKSOURCEUSEL_M) |
                                ((uint16_t)group << CMPSS_COMPDACHCTL2_BLANKSOURCEUSEL_S);
    EDIS;
}
__STATIC_INLINE void
CMPSS_selectBlankSourceGroupLow(uint32_t base, CMPSS_EPWMGroup group)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)group <= 1U);

    //
    // Configure the Blanking source group.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACLCTL2) =
                                (HWREGH(base + CMPSS_O_COMPDACLCTL2) &
                                 ~CMPSS_COMPDACLCTL2_BLANKSOURCEUSEL_M) |
                                ((uint16_t)group << CMPSS_COMPDACLCTL2_BLANKSOURCEUSEL_S);
    EDIS;
}
__STATIC_INLINE void
CMPSS_selectRampSourceGroupLow(uint32_t base, CMPSS_EPWMGroup group)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)group <= 1U);

    //
    // Configure the Blanking source group.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACLCTL2) =
                                    (HWREGH(base + CMPSS_O_COMPDACLCTL2) &
                                     ~CMPSS_COMPDACLCTL2_RAMPSOURCEUSEL_S) |
                                    ((uint16_t)group << CMPSS_COMPDACLCTL2_RAMPSOURCEUSEL_S);
    EDIS;
}




__STATIC_INLINE void
CMPSS_selectRampSourceGroupHigh(uint32_t base, CMPSS_EPWMGroup group)
{
    //
    // Check the arguments.
    //
    ASSERT(CMPSS_isBaseValid(base));
    ASSERT((uint16_t)group <= 1U);

    //
    // Configure the Blanking source group.
    //
    EALLOW;
    HWREGH(base + CMPSS_O_COMPDACHCTL2) =
                                    (HWREGH(base + CMPSS_O_COMPDACHCTL2) &
                                     ~CMPSS_COMPDACHCTL2_RAMPSOURCEUSEL_S) |
                                    ((uint16_t)group << CMPSS_COMPDACHCTL2_RAMPSOURCEUSEL_S);
    EDIS;
}




/*
 * @brief Configures the digital filter of the high comparator.
 *
 * @details
 * This function configures the operation of the digital filter of the high
 * comparator.
 *
 * @param	base	The base address of the comparator module.
 * @param	samplePrescale	The number of system clock cycles between samples.
 * @param 	sampleWindow 	The number of FIFO samples to monitor.
 * @param 	threshold 		The majority threshold of samples to change state.
 *
 * @remarks
 * - The samplePrescale @parameter specifies the number of system clock cycles
 * 	 between samples. It is a 16-bit value so a number higher than 65535 should
 * 	 not be passed as this @parameter. The prescaler used by digital filter is 1
 * 	 more than samplePrescale value. So, the input provided should be 1 less than
 * 	 the expected prescaler.
 * - The sampleWindow @parameter configures the size of the window of FIFO
 * 	 samples taken from the input that will be monitored to determine when to
 * 	 change the filter output. This sample window may be no larger than 32
 * 	 samples.
 * - The threshold @parameter configures the threshold value to be used by
 * 	 the digital filter.
 * - The filter output resolves to the majority value of the sample window where
 * 	 majority is defined by the value passed into the threshold @parameter.
 * 	 For proper operation, the value of \e threshold must be greater than
 * 	 sampleWindow / 2.
 * - To ensure proper operation of the filter, the following is the recommended
 * 	 function call sequence for initialization:
 * 	 - Configure and enable the comparator using CMPSS_configHighComparator()
 * 	   and CMPSS_enableModule()
 * 	 - Configure the digital filter using CMPSS_configFilterHigh()
 * 	 - Initialize the sample values using CMPSS_initFilterHigh()
 * 	 - Configure the module output signals CTRIP and CTRIPOUT using
 * 	   CMPSS_configOutputsHigh()
 *
 */
extern void
CMPSS_configFilterHigh(uint32_t base, uint32_t samplePrescale,
                       uint16_t sampleWindow, uint16_t threshold);

/*
 * @brief Configures the digital filter of the low comparator.
 *
 * @details
 * This function configures the operation of the digital filter of the low
 * comparator.
 *
 * @param[in]	base			The base address of the comparator module.
 * @param[in]	samplePrescale	The number of system clock cycles between samples.
 * @param[in]	sampleWindow	The number of FIFO samples to monitor.
 * @param[in]	threshold 		The majority threshold of samples to change state.
 *
 * @remarks
 * - The samplePrescale @parameter specifies the number of system clock cycles
 * 	 between samples. It is a 16-bit value so a number higher than 65535 should
 * 	 not be passed as this @parameter. The prescaler used by digital filter is 1
 * 	 more than samplePrescale value. So, the input provided should be 1 less
 * 	 than the expected prescaler.
 * - The sampleWindow @parameter configures the size of the window of FIFO
 *   samples taken from the input that will be monitored to determine when to
 *   change the filter output. This sample window may be no larger than 32
 *   samples.
 * - The threshold @parameter configures the threshold value to be used by the
 *   digital filter.
 * - The filter output resolves to the majority value of the sample window where
 *   majority is defined by the value passed into the \e threshold @parameter.
 *   For proper operation, the value of \e threshold must be greater than
 *   sampleWindow / 2.
 * - To ensure proper operation of the filter, the following is the recommended
 *   function call sequence for initialization:
 *   - Configure and enable the comparator using CMPSS_configLowComparator()
 *     and CMPSS_enableModule()
 *   - Configure the digital filter using CMPSS_configFilterLow()
 *   - Initialize the sample values using CMPSS_initFilterLow()
 *   - Configure the module output signals CTRIP and CTRIPOUT using
 *     CMPSS_configOutputsLow()
 */
extern void
CMPSS_configFilterLow(uint32_t base, uint32_t samplePrescale,
                      uint16_t sampleWindow, uint16_t threshold);

/*****************************************************************************
 *
 * @brief Configures whether or not the digital filter latches are reset by PWMSYNC
 *
 * @details
 * This function configures whether or not the digital filter latches in both
 * the high and low comparators should be reset by PWMSYNC. If the highEnable
 * @parameter is true, the PWMSYNC will be allowed to reset the high comparator's
 * digital filter latch. If it is false, the ability of the PWMSYNC to reset the
 * latch will be disabled. The lowEnable @parameter has the same effect on the low
 * comparator's digital filter latch.
 *
 * @param[in]	base		The base address of the comparator module.
 * @param[in]	highEnable  Indicates filter latch settings in the high comparator.
 * @param[in]	lowEnable 	Indicates filter latch settings in the low comparator.
 *
 */
extern void
CMPSS_configLatchOnPWMSYNC(uint32_t base, bool highEnable, bool lowEnable);

/*
 * @brief Configures the comparator subsystem's ramp generator.
 *
 * @details
 * This function configures many of the main settings of the comparator
 * subsystem's ramp generator. The maxRampVal @parameter should be passed
 * the ramp maximum reference value that will be loaded into the ramp
 * generator. The decrementVal @parameter should be passed the decrement
 * value that will be subtracted from the ramp generator on each system clock
 * cycle. The delayVal @parameter should be passed the 13-bit number of
 * system clock cycles the ramp generator should delay before beginning to
 * decrement the ramp generator after a PWMSYNC signal is received.
 *
 * @param[in]	base 			The base address of the comparator module.
 * @param[in]	maxRampVal		The ramp maximum reference value.
 * @param[in]	decrementVal	Value is the ramp decrement value.
 * @param[in]	delayVal 		The ramp delay value.
 * @param[in]	pwmSyncSrc 		The number of the PWMSYNC source.
 * @param[in]	useRampValShdw 	Indicates if the max ramp shadow should be used.
 *
 *
 * These three values may be be set individually using the
 * CMPSS_setMaxRampValue(), CMPSS_setRampDecValue(), and
 * CMPSS_setRampDelayValue() APIs.
 *
 * The number of the PWMSYNC signal to be used to reset the ramp generator
 * should be specified by passing it into the pwmSyncSrc @parameter. For
 * instance, passing a 2 into pwmSyncSrc will select PWMSYNC2.
 *
 * To indicate whether the ramp generator should reset with the value from the
 * ramp max reference value shadow register or with the latched ramp max
 * reference value, use the useRampValShdw parameter. Passing it true
 * will result in the latched value being bypassed. The ramp generator will be
 * loaded right from the shadow register. A value of false will load the
 * ramp generator from the latched value.
 *
 */
extern void
CMPSS_configRamp(uint32_t base, uint16_t maxRampVal, uint16_t decrementVal,
                 uint16_t delayVal, uint16_t pwmSyncSrc, bool useRampValShdw);
extern void
CMPSS_configRampHigh(uint32_t base, CMPSS_RampDirection rampDir,
                     uint16_t rampRef, uint16_t stepVal, uint16_t delayVal,
                     uint16_t pwmSyncSrc, bool useRampValShdw);
extern void
CMPSS_configRampLow(uint32_t base, CMPSS_RampDirection rampDir,
                    uint16_t rampRef, uint16_t stepVal, uint16_t delayVal,
                    uint16_t pwmSyncSrc, bool useRampValShdw);
/*
 * @brief Close the Doxygen group.
 */


#endif




/*
 * @brief Close the Doxygen group.
 */
#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_ */
