/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    crc.c
*   @brief
*   @details
*   commit history
*   20240306, Xu Haoze, add code annotation to function CRC_setTxRxMode.
*   20240325, Xu Haoze, add function crc_config() to simplify the configuration of CRC.
*   20241015, lihao, add crc8,crc16,crc32 calculation api
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "crc.h"
#include "nmsis_core.h"
#include "sysctl.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */


/**
 * @brief Configure CRC.
 * @details
 * This function is used to configure CRC.
 * @param CRC_ConfigParams specifies the CRC base address.
 * @param CRC crcConfigParams mode parameter.
 *
 */
GS32_DRIVER_CRC_FUNC_T void CRC_initConfig(uint32_t base, CRC_ConfigParams crcConfigParams)
{
	ASSERT(CRC_isBaseValid(base));
	HWREG(base + CRC_O_CRC_INIT) = crcConfigParams.initData;
 	/* RESET */
	HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_RESET;
 	/* Set CRC  Polynomial size */
 	CRC_setPolysize(base, crcConfigParams.polySize);
 	HWREG(base + CRC_O_CRC_POL) = crcConfigParams.polyValue;
 	HWREG(base + CRC_O_CRC_LEN) = crcConfigParams.dataSize - 1;
 	HWREG(base + CRC_O_CRC_GOLDEN) = crcConfigParams.goldenValue;
 	CRC_setInputReversed(base, crcConfigParams.revin);
	CRC_setOutputReversed(base, crcConfigParams.revout);
#if (CRC_VERSION == 110)
	CRC_setTxRxMode(base, crcConfigParams.mode); // Only in RX mode, CRC_FAIL flag will assert if there is a mismatch between the result and golden value.
#endif
	if (crcConfigParams.enableDoneInterrupt) {
		CRC_setDoneIntEnable(base, CRC_INTENDONE_ENABLE);
		CRC_clearIntRawFlagDone(base);
	}
	if (crcConfigParams.enableFailInterrupt){
		CRC_setFailIntEnable(base, CRC_INTENFALL_ENABLE);
		CRC_clearIntRawFlagFail(base);
	}
 }

/**
 * @brief Set CRC parameter configuration function
 * @details
 * This function is used to configure CRC parameter.
 * @param  the CRC base address.
 * @param CRC polysize.
 * @param CRC polyValue.
 * @param CRC revInput.
 * @param CRC revOutput.
 * @param CRC seadValue.
 *
 */
GS32_DRIVER_CRC_FUNC_T void CRC_setConfig(uint32_t base, CRC_POLYSIZE polySize, uint32_t polyValue, CRC_REVIN revInput, CRC_REVOUT revOutput, uint32_t seadValue)
{
	uint32_t cfgRegValue;
	ASSERT(CRC_isBaseValid(base));
	/* need configure INIT register before set RESET bit in CR */
	HWREG(base + CRC_O_CRC_INIT) = seadValue;
 	/* RESET CRC - initValue(seedValue) is fed into data register */
	HWREG(base + CRC_O_CRC_CR) = (CRC_CRC_CR_RESET) | (polySize<<CRC_CR_POLYSIZE_S) | (revInput<<CRC_CR_REV_IN_S) | (revOutput<<CRC_CR_REV_OUT_S);
 	/* Set CRC  Polynomial size, revInput, revOutput */
	//cfgRegValue = (polySize<<CRC_CR_POLYSIZE_S) | (revInput<<CRC_CR_REV_IN_S) | (revOutput<<CRC_CR_REV_OUT_S);
 	HWREG(base + CRC_O_CRC_POL) = polyValue;
}

/**
 * @brief CRC8 Calculation functions
 * @details
 * This function is used to set CRC8 Calculation functions .
 * @param  base the CRC base address.
 * @param  pSrcData Point to the data source where you want to calculate the CRC.
 * @param  nLength CRC The length of the CRC in bytes.
 *
 * @return CRC8 calculate result.
 */
GS32_DRIVER_CRC_FUNC_T uint8_t CRC_calcResult8(uint32_t base, uint8_t *pSrcData, uint32_t nLength)
{
	uint32_t i;

	/* set data length */
	if(!nLength)
		return 0;
	else
		HWREG(base + CRC_O_CRC_LEN) = (nLength-1);

	/* reset CRC, init value is placed into data register, it's not necessary */
	//HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_RESET;

	/* start CRC */
	HWREG(base + CRC_O_CRC_EN) = CRC_CRC_EN_START;
	/* wait till CRC HW is ready */
	while((HWREG(base + CRC_O_CRC_EN) & (CRC_CRC_EN_RUN_STS | CRC_CRC_EN_FMAT_READY))
			!= (CRC_CRC_EN_RUN_STS | CRC_CRC_EN_FMAT_READY))
	{

	}

	/* write data into DATA Register */
	for(i=0; i<nLength; i++)
	{
		HWREG(base + CRC_O_CRC_DR) = pSrcData[i];
	}
	__DSB();/* Ensure peripheral register update is complete */
	/* return the CRC result */
	return (uint32_t)HWREG(base + CRC_O_CRC_RESULT);
}

/**
 * @brief CRC16 Calculation functions
 * @details
 * This function is used to set CRC16 Calculation functions .
 * @param  base the CRC base address.
 * @param  pSrcData Point to the data source where you want to calculate the CRC.
 * @param  nLength CRC The length of the CRC in bytes.
 *
 * @return CRC16 calculate result.
 */
GS32_DRIVER_CRC_FUNC_T uint16_t CRC_calcResult16(uint32_t base, uint16_t *pSrcData, uint32_t nLength)
{
	uint32_t i;

	/* set data length */
	if(!nLength)
		return 0;
	else
		HWREG(base + CRC_O_CRC_LEN) = (nLength-1);

	/* reset CRC, init value is placed into data register, it's not necessary */
	//HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_RESET;

	/* start CRC */
	HWREG(base + CRC_O_CRC_EN) = CRC_CRC_EN_START;
	/* wait till CRC HW is ready */
	while((HWREG(base + CRC_O_CRC_EN) & (CRC_CRC_EN_RUN_STS | CRC_CRC_EN_FMAT_READY))!= (CRC_CRC_EN_RUN_STS | CRC_CRC_EN_FMAT_READY));

	/* write data into DATA Register */
	for(i=0; i<nLength; i++)
	{
		HWREG(base + CRC_O_CRC_DR) = pSrcData[i];
	}
	__DSB();/* Ensure peripheral register update is complete */
	/* return the CRC result */
	return (uint32_t)HWREG(base + CRC_O_CRC_RESULT);
}

/**
 * @brief CRC32 Calculation functions
 * @details
 * This function is used to set CRC32 Calculation functions .
 * @param  base the CRC base address.
 * @param  pSrcData Point to the data source where you want to calculate the CRC.
 * @param  nLength CRC The length of the CRC in bytes.
 *
 * @return CRC32 calculate result.
 */
GS32_DRIVER_CRC_FUNC_T uint32_t CRC_calcResult32(uint32_t base, uint32_t *pSrcData, uint32_t nLength)
{
	uint32_t i;

	/* set data length */
	if(!nLength)
		return 0;
	else
		HWREG(base + CRC_O_CRC_LEN) = (nLength-1);

	/* reset CRC, init value is placed into data register, it's not necessary */
	//HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_RESET;

	/* start CRC */
	HWREG(base + CRC_O_CRC_EN) = CRC_CRC_EN_START;
	/* wait till CRC HW is ready */
	while((HWREG(base + CRC_O_CRC_EN) & (CRC_CRC_EN_RUN_STS | CRC_CRC_EN_FMAT_READY))!= (CRC_CRC_EN_RUN_STS | CRC_CRC_EN_FMAT_READY));

	/* write data into DATA Register */
	for(i=0; i<nLength; i++)
	{
		HWREG(base + CRC_O_CRC_DR) = pSrcData[i];
	}
	__DSB();/* Ensure peripheral register update is complete */
	/* return the CRC result */
	return (uint32_t)HWREG(base + CRC_O_CRC_RESULT);
}

const uint16_t crc16_table[256] = {
	// Table entries as defined in the original code...
    0x0000, 0x1021, 0x2042, 0x3063, 0x4084, 0x50A5, 0x60C6, 0x70E7,
    0x8108, 0x9129, 0xA14A, 0xB16B, 0xC18C, 0xD1AD, 0xE1CE, 0xF1EF,
    0x1231, 0x0210, 0x3273, 0x2252, 0x52B5, 0x4294, 0x72F7, 0x62D6,
    0x9339, 0x8318, 0xB37B, 0xA35A, 0xD3BD, 0xC39C, 0xF3FF, 0xE3DE,
    0x2462, 0x3443, 0x0420, 0x1401, 0x64E6, 0x74C7, 0x44A4, 0x5485,
    0xA56A, 0xB54B, 0x8528, 0x9509, 0xE5EE, 0xF5CF, 0xC5AC, 0xD58D,
    0x3653, 0x2672, 0x1611, 0x0630, 0x76D7, 0x66F6, 0x5695, 0x46B4,
    0xB75B, 0xA77A, 0x9719, 0x8738, 0xF7DF, 0xE7FE, 0xD79D, 0xC7BC,
    0x48C4, 0x58E5, 0x6886, 0x78A7, 0x0840, 0x1861, 0x2802, 0x3823,
    0xC9CC, 0xD9ED, 0xE98E, 0xF9AF, 0x8948, 0x9969, 0xA90A, 0xB92B,
    0x5AF5, 0x4AD4, 0x7AB7, 0x6A96, 0x1A71, 0x0A50, 0x3A33, 0x2A12,
    0xDBFD, 0xCBDC, 0xFBBF, 0xEB9E, 0x9B79, 0x8B58, 0xBB3B, 0xAB1A,
    0x6CA6, 0x7C87, 0x4CE4, 0x5CC5, 0x2C22, 0x3C03, 0x0C60, 0x1C41,
    0xEDAE, 0xFD8F, 0xCDEC, 0xDDCD, 0xAD2A, 0xBD0B, 0x8D68, 0x9D49,
    0x7E97, 0x6EB6, 0x5ED5, 0x4EF4, 0x3E13, 0x2E32, 0x1E51, 0x0E70,
    0xFF9F, 0xEFBE, 0xDFDD, 0xCFFC, 0xBF1B, 0xAF3A, 0x9F59, 0x8F78,
    0x9188, 0x81A9, 0xB1CA, 0xA1EB, 0xD10C, 0xC12D, 0xF14E, 0xE16F,
    0x1080, 0x00A1, 0x30C2, 0x20E3, 0x5004, 0x4025, 0x7046, 0x6067,
    0x83B9, 0x9398, 0xA3FB, 0xB3DA, 0xC33D, 0xD31C, 0xE37F, 0xF35E,
    0x02B1, 0x1290, 0x22F3, 0x32D2, 0x4235, 0x5214, 0x6277, 0x7256,
    0xB5EA, 0xA5CB, 0x95A8, 0x8589, 0xF56E, 0xE54F, 0xD52C, 0xC50D,
    0x34E2, 0x24C3, 0x14A0, 0x0481, 0x7466, 0x6447, 0x5424, 0x4405,
    0xA7DB, 0xB7FA, 0x8799, 0x97B8, 0xE75F, 0xF77E, 0xC71D, 0xD73C,
    0x26D3, 0x36F2, 0x0691, 0x16B0, 0x6657, 0x7676, 0x4615, 0x5634,
    0xD94C, 0xC96D, 0xF90E, 0xE92F, 0x99C8, 0x89E9, 0xB98A, 0xA9AB,
    0x5844, 0x4865, 0x7806, 0x6827, 0x18C0, 0x08E1, 0x3882, 0x28A3,
    0xCB7D, 0xDB5C, 0xEB3F, 0xFB1E, 0x8BF9, 0x9BD8, 0xABBB, 0xBB9A,
    0x4A75, 0x5A54, 0x6A37, 0x7A16, 0x0AF1, 0x1AD0, 0x2AB3, 0x3A92,
    0xFD2E, 0xED0F, 0xDD6C, 0xCD4D, 0xBDAA, 0xAD8B, 0x9DE8, 0x8DC9,
    0x7C26, 0x6C07, 0x5C64, 0x4C45, 0x3CA2, 0x2C83, 0x1CE0, 0x0CC1,
    0xEF1F, 0xFF3E, 0xCF5D, 0xDF7C, 0xAF9B, 0xBFBA, 0x8FD9, 0x9FF8,
    0x6E17, 0x7E36, 0x4E55, 0x5E74, 0x2E93, 0x3EB2, 0x0ED1, 0x1EF0
};
/**
 * @brief CRC-16/CCITT-FALSE calculation using lookup table
 * @details
 * This software implementation uses a precomputed lookup table to calculate CRC-16.
 *
 * **Mathematical Expressions:**
 * **Polynomial**:
 *    CRC-16/CCITT-FALSE uses the polynomial:
 *    x^{16} + x^{12} + x^{5} + 1
 *    Hexadecimal representation: `0x1021` (polynomial).
 *
 * @param pSrcData Pointer to the data buffer
 * @param nLength Length of the data in bytes
 *
 * @return 16-bit CRC result
 */
GS32_DRIVER_CRC_FUNC_T uint16_t _CRC_run16BitPoly2(uint8_t *pSrcData, uint32_t nLength) {

    uint16_t crc = 0xFFFF;  // Initial value for CRC-16/CCITT-FALSE

    /* Process each byte in the data buffer */
    for (uint16_t i = 0; i < nLength; i++) {

    	/* Calculate table index: XOR of current CRC high byte and current data byte */
        uint8_t index = (crc >> 8) ^ pSrcData[i]; 

        /* Update CRC: shift left and XOR with table value */
        crc = (crc << 8) ^ crc16_table[index];   
    }
    return crc;
}

#ifdef __cplusplus
}
#endif

