/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    crc.h
*   @brief   
*   commit history
*   20240306, Xu Haoze, add code annotations to structures.
*   20240325, Xu Haoze, add struct CRC_ConfigParams to simplify the configuration of CRC
*   20240929, Jason, modify CRC APIs names and moved some APIs from .c to .h
*   20241015, lihao, add CRC_8,CRC_16,CRC_32 structions,add function declaration about crc.
*/

#ifndef DEVICE_DRIVERLIB_CRC_H_
#define DEVICE_DRIVERLIB_CRC_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "inc/hw_crc.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 * @brief Values that can be passed to CRC_setOutputReversed() as the revout parameter.
 *
 */
typedef enum
{
  //! Output data bit order unchanged
  CRC_REVOUT_OFF = 0,
  //! Output data bit order reversed
  CRC_REVOUT_ON = 1,
} CRC_REVOUT;

/**
 * @brief Values that can be passed to CRC_setInputReversed() as the revin parameter.
 *
 */
typedef enum
{
  //! input data bit order unchanged
  CRC_REVIN_NOTREV = 0,
  //! input data bit order is reversed within bytes
  CRC_REVIN_BYTEREV = 1,
  //! input data bit order is reversed within half-words
  CRC_REVIN_HALFWORDREV = 2,
  //! input data bit order is reversed within word
  CRC_REVIN_WORDREV = 3,
} CRC_REVIN;

/**
 * @brief Values that can be passed to CRC_setPolysize() as the bitsize parameter.
 *
 */
typedef enum
{
  //! Denotes that the CRC  Polynomial size will be 32BIT
  CRC_POLYSIZE_32BIT = 0,
  //! Denotes that the CRC  Polynomial size will be 24BIT
  CRC_POLYSIZE_24BIT = 1,
  //! Denotes that the CRC  Polynomial size will be 16BIT
  CRC_POLYSIZE_16BIT = 2,
  //! Denotes that the CRC  Polynomial size will be 8BIT
  CRC_POLYSIZE_8BIT = 3,
  //! Denotes that the CRC  Polynomial size will be 7BIT
  CRC_POLYSIZE_7BIT = 4,
} CRC_POLYSIZE;

/**
 * @brief Values that can be passed to CRC_setRUNSTSMODE() as the mode parameter.
 *
 */
typedef enum
{
  //! Denotes that the CRC  CRC module is IDLE
  CRC_RUNSTSMODE_IDLE = 0,
  //! Denotes that the CRC  CRC module is Active
  CRC_RUNSTSMODE_ACTIVE = 1,

} CRC_RUNSTSMODE;

/**
 * @brief  Values that can be passed to CRC_setINTENFALL() as the mode parameter.
 *
 */
typedef enum
{
  //! Denotes that the CRC  CRC module is IDLE
  CRC_INTENFALL_ENABLE = 1,
  //! Denotes that the CRC  CRC module is Active
  CRC_INTENFALL_DISABLE = 2,

} CRC_INTENFALL;

/**
 * @brief  Values that can be passed to CRC_setDoneIntEnable() as the mode parameter.
 *
 */
typedef enum
{
  //! Denotes that the CRC  CRC module is IDLE
  CRC_INTENDONE_ENABLE = 1,
  //! Denotes that the CRC  CRC module is Active
  CRC_INTENDONE_DISABLE = 2,

} CRC_INTENDONE;

/**
 * @brief  Values that can be passed to CRC_setTxRxMode() as the mode parameter.
 *
 */
typedef enum
{
  //! Denotes that the CRC  CRC module is IDLE
  CRC_TX_MODOE = 1,
  //! Denotes that the CRC  CRC module is Active
  CRC_RX_MODE = 2,

} CRC_TRANSFER_MODE;


/**
 * @brief  Values that can be passed to CRC_initConfig() as the mode parameter.
 *
 */
typedef struct{
	uint32_t 			initData;			/* Init Data */
	CRC_POLYSIZE 		polySize;			/* Polynomial Size */
	uint32_t 			polyValue;			/* Polynomial Value */
	uint32_t 			dataSize;			/* Size of Input Data */
	uint32_t 			goldenValue;		/* Golden Value */
	CRC_REVIN 			revin;				/* Type of Input Reverse */
	CRC_REVOUT 			revout;				/* Type of Output Reverse */
	CRC_TRANSFER_MODE 	mode;				/* Tx or Rx */
	boolean				enableDoneInterrupt; /* Enable/Disable Done interrupt */
	boolean				enableFailInterrupt; /* Enable/Disable Fail interrupt */
} CRC_ConfigParams;


/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */


/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */
extern void CRC_initConfig(uint32_t base, CRC_ConfigParams crcConfigParams);


/**
 * @brief Checks CRC base address.
 * @details
 * This function determines if a CRC base address is valid.
 * @parambase specifies the CRC base address.
 * @return true the base address is valid.
 * @return false otherwise.
 */
#ifdef DEBUG
static inline boolean CRC_isBaseValid(uint32_t base)
{
    return((base == CRC_BASE)); //need change
}
#endif

/**
 * @brief Set CRC data output reverse .
 * @details
 * This function controls the reversal of the bit order of the output data.
 * @param base is the base address of the timer module.
 * @param revout is ON or OFF.
 *
 */
static inline void CRC_setOutputReversed(uint32_t base, CRC_REVOUT revout)
{
    ASSERT(CRC_isBaseValid(base));

	if( revout == CRC_REVOUT_ON)
	{
		HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_REV_OUT;
	}
	else if ( revout == CRC_REVOUT_OFF)
	{
		HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_REV_OUT;
	}
	else
	{
		ASSERT(FALSE);
	}
}

/**
 * @brief Set CRC data input reverse.
 * @details
 * This function  control the reversal of the bit order of the input data
 * @param base is the base address of the timer module.
 * @param revin is NOTREV , BYTEREV , HALFWORDREV , WORDREV .
 *
 */
static inline void CRC_setInputReversed(uint32_t base, CRC_REVIN revin)
{
    ASSERT(CRC_isBaseValid(base));
    switch (revin)
    {
        case CRC_REVIN_NOTREV:
            HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_REV_INL;
			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_REV_INH;
            break;
        case CRC_REVIN_BYTEREV:
            HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_REV_INL;
			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_REV_INH;
            break;
		case CRC_REVIN_HALFWORDREV:
            HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_REV_INL;
			HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_REV_INH;
            break;
		case CRC_REVIN_WORDREV:
            HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_REV_INL;
			HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_REV_INH;
            break;

        default:
            ASSERT(FALSE);
            break;
    }
}

/**
 * @brief Set CRC  Polynomial size.
 * @details
 * This function control the size of the polynomial.
 * @param base is the base address of the timer module.
 * @param bitsize is 32BIT , 24BIT ,16BIT , 8BIT , 7BIT .
 *
 */
static inline void CRC_setPolysize(uint32_t base, CRC_POLYSIZE bitsize)
{
	uint32_t regval;
	regval = HWREG(base + CRC_O_CRC_CR);
	ASSERT(CRC_isBaseValid(base));

	regval &= ~CRC_CR_POLYSIZE_M;
	regval |= (bitsize<<CRC_CR_POLYSIZE_S);
	HWREG(base + CRC_O_CRC_CR) = regval;

//    ASSERT(CRC_isBaseValid(base));

//	uint32_t regval;
//	regval = HWREG(base + CRC_O_CRC_CR);
//	regval &= ~CRC_CR_POLYSIZE_M;
//	regval |= (bitsize<<CRC_CR_POLYSIZE_S);
//	HWREG(base + CRC_O_CRC_CR) = regval;
//
//    switch (bitsize)
//    {
//        case CRC_POLYSIZE_32BIT:
//            HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEH;
//			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEM;
//			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEL;
//            break;
//        case CRC_POLYSIZE_24BIT:
//            HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEH;
//			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEM;
//			HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_POLYSIZEL;
//            break;
//		case CRC_POLYSIZE_16BIT:
//            HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEH;
//			HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_POLYSIZEM;
//			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEL;
//            break;
//		case CRC_POLYSIZE_8BIT:
//            HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEH;
//			HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_POLYSIZEM;
//			HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_POLYSIZEL;
//            break;
//		case CRC_POLYSIZE_7BIT:
//            HWREG(base + CRC_O_CRC_CR) |=  CRC_CRC_CR_POLYSIZEH;
//			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEM;
//			HWREG(base + CRC_O_CRC_CR) &= ~CRC_CRC_CR_POLYSIZEL;
//            break;
//        default:
//            ASSERT(FALSE);
//            break;
//    }
}

/**
 * @brief  CRC RUN_STS SELECT.
 * @details
 * This function is set CRC runsys mode.
 * @param base specifies the CRC base address.
 * @param mode is IDLE or ACTIVE.
 * @return true the CRC RunStatus is ACTIVE.
 * @return false the CRC RunStatus is IDLE.
 *
 */
static inline uint32_t CRC_getRunStatus(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

	if ( HWREG(base + CRC_O_CRC_EN) & CRC_CRC_EN_RUN_STS)
		return CRC_RUNSTSMODE_ACTIVE;
	else
		return CRC_RUNSTSMODE_IDLE;
}

/**
 * @brief SET CRC_FAIL Interrupt.
 * @details
 * This function is set CRC_FAIL Interrupt mode.
 * @param base specifies the CRC base address.
 * @param  mode is CRC_INTENFALL_ENABLE or CRC_INTENFALL_DISABLE.
 *
 */
static inline void CRC_setFailIntEnable(uint32_t base,CRC_INTENFALL mode)
{
    ASSERT(CRC_isBaseValid(base));

#if (CRC_VERSION == 100)
	if ( mode == CRC_INTENFALL_ENABLE )
	{
		HWREG(base + CRC_O_CRC_INTEN) |=  CRC_CRC_INTEN_FAIL;
	}
	else if ( mode == CRC_INTENFALL_DISABLE )
	{
		HWREG(base + CRC_O_CRC_INTEN) &= ~CRC_CRC_INTEN_FAIL;
	}
	else
	{
		ASSERT(FALSE);
	}
#elif (CRC_VERSION == 110)
	if ( mode == CRC_INTENFALL_ENABLE )
	{
		HWREG(base + CRC_O_CRC_INTR_MASK) &=  ~CRC_CRC_INTR_MASK_FAIL;
	}
	else if ( mode == CRC_INTENFALL_DISABLE )
	{
		HWREG(base + CRC_O_CRC_INTR_MASK) |= CRC_CRC_INTR_MASK_FAIL;
	}
	else
	{
		ASSERT(FALSE);
	}
#endif


}

/**
 * @brief SET CRC_DONE Interrupt.
 * @details
 * This function is set CRC_DONE Interrupt mode.
 * @param base specifies the CRC base address.
 * @param mode is CRC_INTENDONE_ENABLE or CRC_INTENDONE_DISABLE.
 *
 */
static inline void CRC_setDoneIntEnable(uint32_t base,CRC_INTENDONE mode)
{
    ASSERT(CRC_isBaseValid(base));
#if (CRC_VERSION == 100)
	if ( mode == CRC_INTENDONE_ENABLE )
	{
		HWREG(base + CRC_O_CRC_INTEN) |=  CRC_CRC_INTEN_DONE;
	}
	else if (mode == CRC_INTENDONE_DISABLE )
	{
		HWREG(base + CRC_O_CRC_INTEN) &= ~CRC_CRC_INTEN_DONE;
	}
	else
	{
		ASSERT(FALSE);
	}
#elif (CRC_VERSION == 110)
	if ( mode == CRC_INTENDONE_ENABLE )
	{
		HWREG(base + CRC_O_CRC_INTR_MASK) &=  ~CRC_CRC_INTR_MASK_DONE;
	}
	else if (mode == CRC_INTENDONE_DISABLE )
	{
		HWREG(base + CRC_O_CRC_INTR_MASK) |= CRC_CRC_INTR_MASK_DONE;
	}
	else
	{
		ASSERT(FALSE);
	}
#endif
}


#if (CRC_VERSION == 110)

/**
 * @brief SET CRC_TRANSFER_MODE.
 * @details
 * This function is used to configure the transmission mode of the CRC.
 * @param base specifies the CRC base address.
 * @param mode is CRC_TX_MODOE or CRC_RX_MODOE.
 *
 */
static inline void CRC_setTxRxMode(uint32_t base, CRC_TRANSFER_MODE mode){
	ASSERT(CRC_isBaseValid(base));

	if (mode == CRC_TX_MODOE)
	{
		HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_TX ;
	}
	else if (mode == CRC_RX_MODE){
		HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_RX ;
	}
}

#endif

/**
 * @brief CRC_writeData.
 * @details
 * This function is that write CRC_DR data
 * @param base specifies the CRC base address.
 * @param data is write data.
 *
 */
static inline void CRC_writeData(uint32_t base,uint32_t data)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    HWREG(base + CRC_O_CRC_DR) = data;
}

/**
 * @brief CRC_DR_READ.
 * @details
 * This function is that read CRC_DR data
 * @param base specifies the CRC base address.
 * @return CRC DR.
 */
static inline boolean CRC_DR_READ(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    return HWREG(base + CRC_O_CRC_DR);
}

/**
 * @brief CRC_getResult.
 * @details
 * This function is that CRC_getResult data
 * @param base specifies the CRC base address.
 * @return CRC result.
 */
static inline uint32_t CRC_getResult(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    return HWREG(base + CRC_O_CRC_RESULT);
}

/**
 * @brief Reset the CRC calculation unit.
 * @details
 *  This function  is set by software to reset the CRC calculation unit and
 *  set the data register to the value stored in the CRC_INIT register
 * @param base specifies the CRC base address.
 *
 */
static inline void CRC_resetCalculation(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    HWREG(base + CRC_O_CRC_CR) |= CRC_CRC_CR_RESET;
}

/**
 * @brief CRC_setInitValue.
 * @details
 *  This function is that write CRC_INIT data
 * @param base specifies the CRC base address.
 * @param data is write data.
 */
static inline void CRC_setInitValue(uint32_t base,uint32_t data)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    HWREG(base + CRC_O_CRC_INIT) = data;
}

/**
 * @brief CRC_readInitValue.
 * @details
 * This function is that read CRC_INIT data
 * @param base specifies the CRC base address.
 * @return CRC init.
 */
static inline boolean CRC_readInitValue(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    return HWREG(base + CRC_O_CRC_INIT);
}

/**
 * @brief CRC_setPoly.
 * @details
 * This function is that write CRC_POL data
 * @param base specifies the CRC base address.
 * @param data is write data.
 */
static inline void CRC_setPoly(uint32_t base,uint32_t data)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    HWREG(base + CRC_O_CRC_POL) = data;
}

/**
 * @brief CRC_readPoly.
 * @details
 * This function is that write CRC_POL data
 * @param base specifies the CRC base address.
 * @return the CRC poly value.
 */
static inline boolean CRC_readPoly(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    return HWREG(base + CRC_O_CRC_POL);
}

/**
 * @brief Set CRC Length size.
 * @details
 * This function control the length of CRC calcuation.
 * @param base is the base address of the timer module.
 * @param datasize 0:1data 1:2data ...... 0xffffffff: 2^32 data .
 */
static inline void CRC_setDataLength(uint32_t base, uint32_t datasize)
{
  

    ASSERT(CRC_isBaseValid(base));
  

    HWREG(base + CRC_O_CRC_LEN) = datasize-1;
}

/**
 * @brief CRC_writeGoldenValue.
 * @details
 * This function is that write CRC_GOLDEN data
 * @param base specifies the CRC base address.
 * @param data is write data.
 */
static inline void CRC_writeGoldenValue(uint32_t base,uint32_t data)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    HWREG(base + CRC_O_CRC_GOLDEN) = data;
}

/**
 * @brief CRC_readGoldenValue.
 * @details
 * This function is that read CRC_GOLDEN data
 * @param base specifies the CRC base address.
 * @return CRC GoldenValue
 */
static inline boolean CRC_readGoldenValue(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    return HWREG(base + CRC_O_CRC_GOLDEN);
}

/**
 * @brief CRC calculation enable.
 * @details
 * This function is set this anytime during the CRC calculation
 *  will reset and restart the CRC calculation.
 * @param base specifies the CRC base address.
 *
 */
static inline void CRC_startCalculation(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    //
    // Reset the CRC calculation unit
    //
    HWREG(base + CRC_O_CRC_EN) |= CRC_CRC_EN_START;
}

/**
 * @brief CRC F matrix ready.
 * @details
 * This function is set CRC F matrix ready ENABLE.
 * @param base specifies the CRC base address.
 * @return CRC en value.
 */
static inline uint32_t CRC_getReady(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    
    return (HWREG(base + CRC_O_CRC_EN) & CRC_CRC_EN_FMAT_READY);
}

#if (CRC_VERSION == 100)
/**
 * @brief CRC Global Interrupt Status flag
 * @details
 * This function is that return CRC Global Interrupt Status flag
 * @param base specifies the CRC base address.
 * @return CRC Global Interrupt Status flag.
 */
static inline boolean CRC_INTFLG_INT(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    
    return HWREG(base + CRC_O_CRC_INTFLG) & CRC_CRC_INTFLG_INT;
}
#endif
/**
 * @brief CRC Done Interrupt Status flag
 * @details
 * This function is that return CRC Done Interrupt Status flag
 * @param base specifies the CRC base address.
 * @return CRC Done Interrupt Status flag.
 */
static inline boolean CRC_getIntFlagDone(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));
#if (CRC_VERSION == 100)
    return ((HWREG(base + CRC_O_CRC_INTFLG) & CRC_CRC_INTFLG_DONE)>>1);
#elif (CRC_VERSION == 110)
    return (HWREG(base + CRC_O_CRC_INTR) & CRC_CRC_INTR_DONE);
#endif
}

/**
 * @brief CRC Fail Interrupt Status flag
 * @details
 * This function is that return CRC Fail Interrupt Status flag
 * @param base specifies the CRC base address.
 * @return CRC Fail Interrupt Status flag.
 */
static inline boolean CRC_getIntFlagFail(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));
#if (CRC_VERSION == 100)
    return ((HWREG(base + CRC_O_CRC_INTFLG) & CRC_CRC_INTFLG_FAIL)>>2);
#elif (CRC_VERSION == 110)
    return ((HWREG(base + CRC_O_CRC_INTR) & CRC_CRC_INTR_FAIL)>>1);
#endif
}

#if (CRC_VERSION == 100)
/**
 * @brief CRC Global Interrupt Flag Clear.
 * @details
 * This function Clear CRC Global Interrupt Flag.
 * @param base specifies the CRC base address.
 */
static inline void CRC_INTCLR_INT(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    
    HWREG(base + CRC_O_CRC_INTCLR) |= CRC_CRC_INTCLR_INT ;
}
#endif

/**
 * @brief CRC Done Interrupt Flag Clear
 * @details
 * This function Clear CRC Done Interrupt Flag
 * @param base specifies the CRC base address.
 */
static inline void CRC_clearIntRawFlagDone(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));
#if (CRC_VERSION == 100)
    HWREG(base + CRC_O_CRC_INTCLR) |= CRC_CRC_INTCLR_DONE;
#elif (CRC_VERSION == 110)
    HWREG(base + CRC_O_CRC_INTR_RAW) = CRC_CRC_INTR_RAW_DONE;
#endif
}

/**
 * @brief CRC Fail Interrupt Flag Clear
 * @details
 * This function Clear CRC Fail Interrupt Flag
 * @param base specifies the CRC base address.
 */
static inline void CRC_clearIntRawFlagFail(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));
#if (CRC_VERSION == 100)
    HWREG(base + CRC_O_CRC_INTCLR) |= CRC_CRC_INTCLR_FAIL;
#elif (CRC_VERSION == 110)
    HWREG(base + CRC_O_CRC_INTR_RAW) = CRC_CRC_INTR_RAW_FAIL;
#endif
}

/**
 * @brief CRC interrupt Force Done flag
 * @details
 * This function CRC interrupt Force Done flag
 * @param base specifies the CRC base address.
 */
static inline void CRC_INTFRC_DONE(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    HWREG(base + CRC_O_CRC_INTFRC) |= CRC_CRC_INTFRC_DONE ;
}

/**
 * @brief CRC interrupt Force Fail flag
 * @details
 * This function CRC interrupt Force Fail flag
 * @param base specifies the CRC base address.
 */
static inline void CRC_INTFRC_FAIL(uint32_t base)
{
    ASSERT(CRC_isBaseValid(base));

    
    HWREG(base + CRC_O_CRC_INTFRC) |= CRC_CRC_INTFRC_FAIL ;
}
//*****************************************************************************



void CRC_initConfig(uint32_t base, CRC_ConfigParams crcConfigParams);

void CRC_setConfig(uint32_t base, CRC_POLYSIZE polySize, uint32_t polyValue, CRC_REVIN revInput, CRC_REVOUT revOutput, uint32_t seadValue);

uint8_t CRC_calcResult8(uint32_t base, uint8_t *pSrcData, uint32_t nLength);

uint16_t CRC_calcResult16(uint32_t base, uint16_t *pSrcData, uint32_t nLength);

uint32_t CRC_calcResult32(uint32_t base, uint32_t *pSrcData, uint32_t nLength);

uint16_t _CRC_run16BitPoly2(uint8_t *pSrcData, uint32_t nLength);

#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_CRC_H_ */
