/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_cc_aq.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_CC_AQ_H_
#define DEVICE_DRIVERLIB_EPWM_CC_AQ_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "inc/hw_types.h"
#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm_define.h"
#include "debug.h"
/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//*****************************************************************************
//
// Defines for the API.
//
//*****************************************************************************


/**
 * @brief  Sets up the Counter Compare shadow load mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @param [in]  loadMode is the shadow to active load mode.
 * @remarks
 * This function enables and sets up the counter compare shadow load mode.
 * Valid values for the variables are:
 * - compModule
 * - EPWM_COUNTER_COMPARE_A - counter compare A.
 * - EPWM_COUNTER_COMPARE_B - counter compare B.
 * - EPWM_COUNTER_COMPARE_C - counter compare C.
 * - EPWM_COUNTER_COMPARE_D - counter compare D.
 * - loadMode
 * - EPWM_COMP_LOAD_ON_CNTR_ZERO - load when counter equals zero
 * - EPWM_COMP_LOAD_ON_CNTR_PERIOD - load when counter equals period
 * - EPWM_COMP_LOAD_ON_CNTR_ZERO_PERIOD - load when counter equals
 * zero or period
 * - EPWM_COMP_LOAD_FREEZE - Freeze shadow to active load
 * - EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO - load when counter equals zero
 * - EPWM_COMP_LOAD_ON_SYNC_CNTR_PERIOD -load when counter equals period
 * - EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO_PERIOD - load when counter equals
 * zero or period
 * - EPWM_COMP_LOAD_ON_SYNC_ONLY - load on sync only
 */
__STATIC_INLINE void
EPWM_setCounterCompareShadowLoadMode(uint32_t base,
                                     EPWM_CounterCompareModule compModule,
                                     EPWM_CounterCompareLoadMode loadMode)
{
    uint16_t syncModeOffset;
    uint16_t loadModeOffset;
    uint16_t shadowModeOffset;
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if((compModule == EPWM_COUNTER_COMPARE_A) ||
       (compModule == EPWM_COUNTER_COMPARE_C))
    {
        syncModeOffset = 10U;
        loadModeOffset = 0U;
        shadowModeOffset = 4U;
    }
    else
    {
        syncModeOffset = 12U;
        loadModeOffset = 2U;
        shadowModeOffset = 6U;
    }

    //
    // Get the register offset.  EPWM_O_CMPCTL for A&B or
    // EPWM_O_CMPCTL2 for C&D
    //
    registerOffset = base + EPWM_O_CMPCTL + ((((uint32_t)compModule/2) & 0x1U)<<1);

    //
    // Set the appropriate sync and load mode bits and also enable shadow
    // load mode. Shadow to active load can also be frozen.
    //
    HWREGH(registerOffset) = ((HWREGH(registerOffset) &
                         ~((0x3U << syncModeOffset) | // Clear sync mode
                           (0x3U << loadModeOffset) | // Clear load mode
                           (0x1U << shadowModeOffset))) | // shadow mode
                         ((((uint16_t)loadMode >> 2U) << syncModeOffset) |
                         (((uint16_t)loadMode & 0x3U) << loadModeOffset)));
}

/**
 * @brief  Sets up the Counter Compare shadow load mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @remarks
 * This function disables counter compare shadow load mode.
 * Valid values for the variables are:
 * - compModule
 * - EPWM_COUNTER_COMPARE_A - counter compare A.
 * - EPWM_COUNTER_COMPARE_B - counter compare B.
 * - EPWM_COUNTER_COMPARE_C - counter compare C.
 * - EPWM_COUNTER_COMPARE_D - counter compare D.
 */
__STATIC_INLINE void
EPWM_disableCounterCompareShadowLoadMode(uint32_t base,
                                         EPWM_CounterCompareModule compModule)
{
    uint16_t shadowModeOffset;
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if((compModule == EPWM_COUNTER_COMPARE_A) ||
       (compModule == EPWM_COUNTER_COMPARE_C))
    {
        shadowModeOffset = 4U;
    }
    else
    {
        shadowModeOffset = 6U;
    }

    //
    // Get the register offset.  EPWM_O_CMPCTL for A&B or
    // EPWM_O_CMPCTL2 for C&D
    //
    registerOffset = base + EPWM_O_CMPCTL + ((((uint32_t)compModule/2) & 0x1U)<<1);

    //
    // Disable shadow load mode.
    //
    HWREGH(registerOffset) = (HWREGH(registerOffset) |
                             (0x1U << shadowModeOffset));
}

/**
 * @brief  Set counter compare values.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @param [in]  compCount is the counter compare count value.
 * @remarks
 * This function sets the counter compare value for counter compare registers.
 * The maximum value for compCount is 0xFFFF.
 * Valid values for compModule are:
 *   - EPWM_COUNTER_COMPARE_A - counter compare A.
 *   - EPWM_COUNTER_COMPARE_B - counter compare B.
 *   - EPWM_COUNTER_COMPARE_C - counter compare C.
 *   - EPWM_COUNTER_COMPARE_D - counter compare D.
 */

__STATIC_INLINE void
EPWM_setCounterCompareValue(uint32_t base, EPWM_CounterCompareModule compModule,
                            uint16_t compCount)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset for the Counter compare
    //
    registerOffset = EPWM_O_CMPA + (uint32_t)compModule;

    {
        //
        // Write to the counter compare registers.
        //
        if((compModule == EPWM_COUNTER_COMPARE_A) ||
            (compModule == EPWM_COUNTER_COMPARE_B))
        {

            //
            // Write to COMPA or COMPB bits
            //
            HWREGH(base + registerOffset + 0x2U) = compCount;
        }
        else
        {


            //
            // Write to COMPC or COMPD bits
            //
            HWREGH(base + registerOffset) = compCount;
        }
    }

}

/**
 * @brief  Get counter compare values.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @remarks
 * This function gets the counter compare value for counter compare registers.
 * Valid values for compModule are:
 *   - EPWM_COUNTER_COMPARE_A - counter compare A.
 *   - EPWM_COUNTER_COMPARE_B - counter compare B.
 *   - EPWM_COUNTER_COMPARE_C - counter compare C.
 *   - EPWM_COUNTER_COMPARE_D - counter compare D.
 * @return The counter compare count value.
 */
__STATIC_INLINE uint16_t
EPWM_getCounterCompareValue(uint32_t base, EPWM_CounterCompareModule compModule)
{
    uint32_t registerOffset;
    uint16_t compCount;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset for the Counter compare
    //
    registerOffset = EPWM_O_CMPA + (uint32_t)compModule;

    //
    // Read from the counter compare registers.
    //
    if((compModule == EPWM_COUNTER_COMPARE_A) ||
        (compModule == EPWM_COUNTER_COMPARE_B))
    {
        //
        // Read COMPA or COMPB bits
        //
        compCount = (uint16_t)((HWREG(base + registerOffset) &
                     0xFFFF0000UL) >> 16U);
    }
    else
    {
        //
        // Read COMPC or COMPD bits
        //
        compCount = HWREGH(base + registerOffset);
    }
    return(compCount);
}

/**
 * @brief  Return the counter compare shadow register full status.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @remarks
 * This function returns the counter Compare shadow register full status flag.
 * Valid values for compModule are:
 *   - EPWM_COUNTER_COMPARE_A - counter compare A.
 *   - EPWM_COUNTER_COMPARE_B - counter compare B.
 *   - EPWM_COUNTER_COMPARE_C - counter compare C.
 *   - EPWM_COUNTER_COMPARE_D - counter compare D.
 * @return Returns true if the shadow register is full.
 * 		   Returns false if the shadow register is not full.
 */
__STATIC_INLINE bool
EPWM_getCounterCompareShadowStatus(uint32_t base,
                                   EPWM_CounterCompareModule compModule)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Check the validity of input.
    // COMPA and COMPB are valid input arguments.
    //
    ASSERT((compModule == EPWM_COUNTER_COMPARE_A) ||
            (compModule == EPWM_COUNTER_COMPARE_B));

    //
    // Read the value of SHDWAFULL or SHDWBFULL bit
    //
    return((((HWREG(base + EPWM_O_CMPCTL) >>
              ((((uint16_t)compModule >> 1U) & 0x1U) + 8U)) &
              0x1U) == 0x1U) ? true:false);
}

/**
 * @brief Enable CMPAHR, CMPBHR register linking
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function enables CMPAHR and CMPBHR register linking. CMPBHR assumes
 * the same value as CMPAHR.
 */
static inline void
EPWM_enableLinkDutyHR(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set LINKDUTYHR bit in CMPCTL register
    //
    HWREGH(base + EPWM_O_CMPCTL) |= EPWM_CMPCTL_LINKDUTYHR;
}

/**
 * @brief disable CMPAHR, CMPBHR register linking
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * This function disables CMPAHR and CMPBHR register linking. CMPAHR and
 * CMPBHR operate independently.
 */
static inline void
EPWM_disableLinkDutyHR(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Clear LINKDUTYHR bit in CMPCTL register
    //
    HWREGH(base + EPWM_O_CMPCTL) &= ~EPWM_CMPCTL_LINKDUTYHR;
}

//
// Action Qualifier module related APIs
//

/**
 * @brief  Sets the Action Qualifier shadow load mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  aqModule is the Action Qualifier module value.
 * @param [in]  loadMode is the shadow to active load mode.
 * @remarks
 * This function enables and sets the Action Qualifier shadow load mode.
 * Valid values for the variables are:
 *  - aqModule
 *      - EPWM_ACTION_QUALIFIER_A - Action Qualifier A.
 *      - EPWM_ACTION_QUALIFIER_B - Action Qualifier B.
 *  - loadMode
 *      - EPWM_AQ_LOAD_ON_CNTR_ZERO - load when counter equals zero
 *      - EPWM_AQ_LOAD_ON_CNTR_PERIOD - load when counter equals period
 *      - EPWM_AQ_LOAD_ON_CNTR_ZERO_PERIOD - load when counter equals
 *                                               zero or period
 *      - EPWM_AQ_LOAD_FREEZE  - Freeze shadow to active load
 *      - EPWM_AQ_LOAD_ON_SYNC_CNTR_ZERO - load on sync or when counter
 *                                          equals zero
 *      - EPWM_AQ_LOAD_ON_SYNC_CNTR_PERIOD - load on sync or when counter
 *                                           equals period
 *      - EPWM_AQ_LOAD_ON_SYNC_CNTR_ZERO_PERIOD - load on sync or when
 *                                               counter equals zero or period
 *      - EPWM_AQ_LOAD_ON_SYNC_ONLY - load on sync only
 */
__STATIC_INLINE void
EPWM_setActionQualifierShadowLoadMode(uint32_t base,
                                      EPWM_ActionQualifierModule aqModule,
                                      EPWM_ActionQualifierLoadMode loadMode)
{
    uint16_t syncModeOffset;
    uint16_t shadowModeOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    syncModeOffset = 8U + (uint16_t)aqModule;
    shadowModeOffset = 4U + (uint16_t)aqModule;

    //
    // Set the appropriate sync and load mode bits and also enable shadow
    // load mode. Shadow to active load can also be frozen.
    //
    HWREGH(base + EPWM_O_AQCTL) = ( ((HWREGH(base + EPWM_O_AQCTL) & (~((0x3U << (uint16_t)aqModule) | (0x3U << (uint16_t)syncModeOffset)))) | (0x1U << shadowModeOffset)) |
                                   ((((uint16_t)loadMode >> 2U) << syncModeOffset) | (((uint16_t)loadMode & 0x3U) << (uint16_t)aqModule)) );
}

/**
 * @brief  Disable Action Qualifier shadow load mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  aqModule is the Action Qualifier module value.
 * @remarks
 * This function disables the Action Qualifier  shadow load mode.
 * Valid values for the variables are:
 *  - aqModule
 *      - EPWM_ACTION_QUALIFIER_A - Action Qualifier A.
 *      - EPWM_ACTION_QUALIFIER_B - Action Qualifier B.
 */
__STATIC_INLINE void
EPWM_disableActionQualifierShadowLoadMode(uint32_t base,
                                          EPWM_ActionQualifierModule aqModule)
{
    uint16_t shadowModeOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    shadowModeOffset = 4U + (uint16_t)aqModule;

    //
    // Disable shadow load mode. Action qualifier is loaded on
    // immediate mode only.
    //
    HWREGH(base + EPWM_O_AQCTL) &= ~(1U << shadowModeOffset);
}

/**
 * @brief  Set up Action qualifier trigger source for event T1
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  trigger sources for Action Qualifier triggers.
 * @remarks
 * This function sets up the sources for Action Qualifier event T1.
 * Valid values for trigger are:
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1       - Digital compare event A 1
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_2       - Digital compare event A 2
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCB_1       - Digital compare event B 1
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCB_2       - Digital compare event B 2
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_1        - Trip zone 1
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_2        - Trip zone 2
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_3        - Trip zone 3
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_EPWM_SYNCIN - ePWM sync
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DC_EVTFILT  - Digital compare filter event
 */
__STATIC_INLINE void
EPWM_setActionQualifierT1TriggerSource(uint32_t base,
                                     EPWM_ActionQualifierTriggerSource trigger)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set T1 trigger source
    //
    HWREGH(base + EPWM_O_AQTSRCSEL) =
         ((HWREGH(base + EPWM_O_AQTSRCSEL) & (~EPWM_AQTSRCSEL_T1SEL_M)) |
          ((uint16_t)trigger));
}

/**
 * @brief  Set up Action qualifier trigger source for event T2
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  trigger sources for Action Qualifier triggers.
 * @remarks
 * This function sets up the sources for Action Qualifier event T2.
 * Valid values for trigger are:
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1       - Digital compare event A 1
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_2       - Digital compare event A 2
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCB_1       - Digital compare event B 1
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DCB_2       - Digital compare event B 2
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_1        - Trip zone 1
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_2        - Trip zone 2
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_3        - Trip zone 3
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_EPWM_SYNCIN - ePWM sync
 *   - EPWM_AQ_TRIGGER_EVENT_TRIG_DC_EVTFILT  - Digital compare filter event
 */
__STATIC_INLINE void
EPWM_setActionQualifierT2TriggerSource(uint32_t base,
                                      EPWM_ActionQualifierTriggerSource trigger)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set T2 trigger source
    //
    HWREGH(base + EPWM_O_AQTSRCSEL) =
          ((HWREGH(base + EPWM_O_AQTSRCSEL) & (~EPWM_AQTSRCSEL_T2SEL_M)) |
           ((uint16_t)trigger << EPWM_AQTSRCSEL_T2SEL_S));
}

/**
 * @brief  Set up Action qualifier outputs
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  epwmOutput is the ePWM pin type.
 * @param [in]  output is the Action Qualifier output.
 * @param [in]  event is the event that causes a change in output.
 * @remarks
 * This function sets up the Action Qualifier output on ePWM A or ePWMB,
 * depending on the value of epwmOutput, to a value specified by outPut based
 * on the input events - specified by event.
 * The following are valid values for the parameters.
 *   - epwmOutput
 *       - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *       - EPWM_AQ_OUTPUT_B          - ePWMxB output
 *   - output
 *       - EPWM_AQ_OUTPUT_NO_CHANGE  - No change in the output pins
 *       - EPWM_AQ_OUTPUT_LOW        - Set output pins to low
 *       - EPWM_AQ_OUTPUT_HIGH       - Set output pins to High
 *       - EPWM_AQ_OUTPUT_TOGGLE     - Toggle the output pins
 *   - event
 *       - EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO       - Time base counter equals
 *                                                 zero
 *       - EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD     - Time base counter equals
 *                                                 period
 *       - EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA    - Time base counter up equals
 *                                                 COMPA
 *       - EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA  - Time base counter down
 *                                                 equals COMPA
 *       - EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB    - Time base counter up equals
 *                                                 COMPB
 *       - EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB  - Time base counter down
 *                                                 equals COMPB
 *       - EPWM_AQ_OUTPUT_ON_T1_COUNT_UP         - T1 event on count up
 *       - EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN       - T1 event on count down
 *       - EPWM_AQ_OUTPUT_ON_T2_COUNT_UP         - T2 event on count up
 *       - EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN       - T2 event on count down
 *
 *     - Extend compare
 *   	 - EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA1	 - Time base counter up equals COMPA
 *
 *   	 - EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA1 - Time base counter down equals COMPA
 *
 *   	 - EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB1	 - Time base counter up equals COMPB
 *
 *    	 - EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB1 - Time base counter down equals COMPB
 */
__STATIC_INLINE void
EPWM_setActionQualifierAction(uint32_t base,
                              EPWM_ActionQualifierOutputModule epwmOutput,
                              EPWM_ActionQualifierOutput output,
                              EPWM_ActionQualifierOutputEvent event)
{
    uint32_t registerOffset;
    uint32_t registerTOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset
    //
    registerOffset = EPWM_O_AQCTLA + (uint32_t)epwmOutput;
    registerTOffset = EPWM_O_AQCTLA2 + (uint32_t)epwmOutput;

    //
    // If the event occurs on T1 or T2 events
    //
    if(((uint16_t)event & 0x1U) == 1U)
    {
        //
        // Write to T1U,T1D,T2U or T2D of AQCTLA2 register
        //
        HWREGH(base + registerTOffset) =
         ((HWREGH(base + registerTOffset) & ~(3U << ((uint16_t)event - 1U))) |
          ((uint16_t)output << ((uint16_t)event - 1U)));
    }
    else
    {
        //
        // Write to ZRO,PRD,CAU,CAD,CBU or CBD bits of AQCTLA register
        //
        HWREGH(base + registerOffset) =
                ((HWREGH(base + registerOffset) & ~(3U << (uint16_t)event)) |
                 ((uint16_t)output << (uint16_t)event));
    }
}



/**
 * @brief  Set up Action qualifier event outputs
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  epwmOutput is the ePWM pin type.
 * @param [in]  output is the Action Qualifier output.
 * @param [in]  event is the event that causes a change in output.
 * @remarks
 *  This function sets up the Action Qualifier output on ePWMA or ePWMB,
 *  depending on the value of epwmOutput, to a value specified by action.
 *  Valid action param values from different time base counter scenarios
 *  should be OR'd together to configure complete action for a pwm output.
 *  The following are valid values for the parameters.
 *    - epwmOutput
 *        - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *        - EPWM_AQ_OUTPUT_B          - ePWMxB output
 *
 *    - action
 *        - When time base counter equals zero
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_ZERO   - Time base counter equals zero
 *                                              and no change in output pins
 *          - EPWM_AQ_OUTPUT_LOW_ZERO         - Time base counter equals zero
 *                                              and set output pins to low
 *          - EPWM_AQ_OUTPUT_HIGH_ZERO        - Time base counter equals zero
 *                                              and set output pins to high
 *          - EPWM_AQ_OUTPUT_TOGGLE_ZERO      - Time base counter equals zero
 *                                              and toggle the output pins
 *        - When time base counter equals period
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_PERIOD - Time base counter equals period
 *                                              and no change in output pins
 *          - EPWM_AQ_OUTPUT_LOW_PERIOD       - Time base counter equals period
 *                                              and set output pins to low
 *          - EPWM_AQ_OUTPUT_HIGH_PERIOD      - Time base counter equals period
 *                                              and set output pins to high
 *          - EPWM_AQ_OUTPUT_TOGGLE_PERIOD    - Time base counter equals period
 *                                              and toggle the output pins
 *        - When time base counter equals CMPA during up-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_UP_CMPA  - Time base counter up equals
 *                                                COMPA and no change in the
 *                                                output pins
 *          - EPWM_AQ_OUTPUT_LOW_UP_CMPA        - Time base counter up equals
 *                                                COMPA and set output pins low
 *          - EPWM_AQ_OUTPUT_HIGH_UP_CMPA       - Time base counter up equals
 *                                                COMPA and set output pins high
 *          - EPWM_AQ_OUTPUT_TOGGLE_UP_CMPA     - Time base counter up equals
 *                                                COMPA and toggle output pins
 *        - When time base counter equals CMPA during down-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_CMPA - Time base counter down equals
 *                                                 COMPA and no change in the
 *                                                 output pins
 *          - EPWM_AQ_OUTPUT_LOW_DOWN_CMPA      - Time base counter down equals
 *                                                COMPA and set output pins low
 *          - EPWM_AQ_OUTPUT_HIGH_DOWN_CMPA     - Time base counter down equals
 *                                                COMPA and set output pins high
 *          - EPWM_AQ_OUTPUT_TOGGLE_DOWN_CMPA   - Time base counter down equals
 *                                                COMPA and toggle output pins
 *        - When time base counter equals CMPB during up-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_UP_CMPB  - Time base counter up equals
 *                                                COMPB and no change in the
 *                                                output pins
 *          - EPWM_AQ_OUTPUT_LOW_UP_CMPB        - Time base counter up equals
 *                                                COMPB and set output pins low
 *          - EPWM_AQ_OUTPUT_HIGH_UP_CMPB       - Time base counter up equals
 *                                                COMPB and set output pins high
 *          - EPWM_AQ_OUTPUT_TOGGLE_UP_CMPB     - Time base counter up equals
 *                                                COMPB and toggle output pins
 *        - When time base counter equals CMPB during down-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_CMPB- Time base counter down equals
 *                                                COMPB and no change in the
 *                                                output pins
 *          - EPWM_AQ_OUTPUT_LOW_DOWN_CMPB      - Time base counter down equals
 *                                                COMPB and set output pins low
 *          - EPWM_AQ_OUTPUT_HIGH_DOWN_CMPB     - Time base counter down equals
 *                                                COMPB and set output pins high
 *          - EPWM_AQ_OUTPUT_TOGGLE_DOWN_CMPB   - Time base counter down equals
 *                                                COMPB and toggle output pins
 *
 *   note:  A logical OR of the valid values should be passed as the action
 *            parameter. Single action should be configured for each time base
 *            counter scenario.
 */
__STATIC_INLINE void
EPWM_setActionQualifierActionComplete(uint32_t base,
                                   EPWM_ActionQualifierOutputModule epwmOutput,
                                   uint16_t action)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset
    //
    registerOffset = EPWM_O_AQCTLA + (uint32_t)epwmOutput;

    //
    // Write to ZRO, PRD, CAU, CAD, CBU or CBD bits of AQCTLA register
    //
    HWREGH(base + registerOffset) = (uint16_t)action;
}

/**
 * @brief  Set up Additional action qualifier event outputs
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  epwmOutput is the ePWM pin type.
 * @param [in]  action is the desired action when the specified event occurs
 * @remarks
 *  This function sets up the Additional Action Qualifier output on ePWMA or
 *  ePWMB depending on the value of epwmOutput, to a value specified by action.
 *  Valid action param values from different event scenarios should be OR'd
 *  together to configure complete action for a pwm output.
 *  The following are valid values for the parameters.
 *    - epwmOutput
 *        - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *        - EPWM_AQ_OUTPUT_B          - ePWMxB output
 *    - action
 *        - When T1 event occurs during up-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_UP_T1  - T1 event on count up
 *                                              and no change in output pins
 *          - EPWM_AQ_OUTPUT_LOW_UP_T1        - T1 event on count up
 *                                            and set output pins to low
 *          - EPWM_AQ_OUTPUT_HIGH_UP_T1       - T1 event on count up
 *                                            and set output pins to high
 *          - EPWM_AQ_OUTPUT_TOGGLE_UP_T1     - T1 event on count up
 *                                            and toggle the output pins
 *        - When T1 event occurs during down-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_T1- T1 event on count down
 *                                            and no change in output pins
 *          - EPWM_AQ_OUTPUT_LOW_DOWN_T1      - T1 event on count down
 *                                            and set output pins to low
 *          - EPWM_AQ_OUTPUT_HIGH_DOWN_T1     - T1 event on count down
 *                                            and set output pins to high
 *          - EPWM_AQ_OUTPUT_TOGGLE_DOWN_T1   - T1 event on count down
 *                                            and toggle the output pins
 *        - When T2 event occurs during up-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_UP_T2  - T2 event on count up
 *                                              and no change in output pins
 *          - EPWM_AQ_OUTPUT_LOW_UP_T2        - T2 event on count up
 *                                              and set output pins to low
 *          - EPWM_AQ_OUTPUT_HIGH_UP_T2       - T2 event on count up
 *                                              and set output pins to high
 *          - EPWM_AQ_OUTPUT_TOGGLE_UP_T2     - T2 event on count up
 *                                              and toggle the output pins
 *        - When T2 event occurs during down-count
 *          - EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_T2 - T2 event on count down
 *                                               and no change in output pins
 *          - EPWM_AQ_OUTPUT_LOW_DOWN_T2       - T2 event on count down
 *                                               and set output pins to low
 *          - EPWM_AQ_OUTPUT_HIGH_DOWN_T2      - T2 event on count down
 *                                               and set output pins to high
 *          - EPWM_AQ_OUTPUT_TOGGLE_DOWN_T2    - T2 event on count down
 *                                               and toggle the output pins
 *
 *   note:  A logical OR of the valid values should be passed as the action
 *            parameter. Single action should be configured for each event
 *            scenario.
 */
__STATIC_INLINE void
EPWM_setAdditionalActionQualifierActionComplete(uint32_t base,
                               EPWM_ActionQualifierOutputModule epwmOutput,
                               uint16_t action)
{
    uint32_t registerTOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset
    //
    registerTOffset = EPWM_O_AQCTLA2 + (uint32_t)epwmOutput;

    //
    // Write to T1U, T1D, T2U or T2D of AQCTLA2 register
    //
    HWREGH(base + registerTOffset) = (uint16_t)action;
}

/**
 * @brief  Sets up Action qualifier continuous software load mode.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  mode is the mode for shadow to active load mode.
 * @remarks
 *  This function sets up the AQCFRSC register load mode for continuous
 *  software force reload mode. The software force actions are determined by
 *  the EPWM_setActionQualifierContSWForceAction() function.
 *  Valid values for mode are:
 *    - EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO        - shadow mode load when counter
 *                                                equals zero
 *    - EPWM_AQ_SW_SH_LOAD_ON_CNTR_PERIOD      - shadow mode load when counter
 *                                                equals period
 *    - EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO_PERIOD - shadow mode load when counter
 *                                                equals zero or period
 *    - EPWM_AQ_SW_IMMEDIATE_LOAD               - immediate mode load only
 */
__STATIC_INLINE void
EPWM_setActionQualifierContSWForceShadowMode(uint32_t base,
                                             EPWM_ActionQualifierContForce mode)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Action qualifier software action reload mode.
    // Write to RLDCSF bit
    //
    HWREGH(base + EPWM_O_AQSFRC) =
            ((HWREGH(base + EPWM_O_AQSFRC) & ~EPWM_AQSFRC_RLDCSF_M) |
             ((uint16_t)mode << EPWM_AQSFRC_RLDCSF_S));
}

/**
 * @brief  Triggers a continuous software forced event.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  epwmOutput is the ePWM pin type.
 * @param [in]  output is the Action Qualifier output.
 * @remarks
 *  This function triggers a continuous software forced Action Qualifier output
 *  on ePWM A or B based on the value of epwmOutput.
 *  Valid values for the parameters are:
 *    - epwmOutput
 *        - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *        - EPWM_AQ_OUTPUT_B          - ePWMxB output
 *    - output
 *        - EPWM_AQ_SW_DISABLED       - Software forcing disabled.
 *        - EPWM_AQ_SW_OUTPUT_LOW     - Set output pins to low
 *        - EPWM_AQ_SW_OUTPUT_HIGH    - Set output pins to High
 */
__STATIC_INLINE void
EPWM_setActionQualifierContSWForceAction(uint32_t base,
                                    EPWM_ActionQualifierOutputModule epwmOutput,
                                    EPWM_ActionQualifierSWOutput output)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Initiate a continuous software forced output
    //
    if(epwmOutput == EPWM_AQ_OUTPUT_A)
    {
        HWREGH(base + EPWM_O_AQCSFRC) =
                ((HWREGH(base + EPWM_O_AQCSFRC) & ~EPWM_AQCSFRC_CSFA_M) |
                 ((uint16_t)output));
    }
    else
    {
        HWREGH(base + EPWM_O_AQCSFRC) =
                ((HWREGH(base + EPWM_O_AQCSFRC) & ~EPWM_AQCSFRC_CSFB_M) |
                 ((uint16_t)output << EPWM_AQCSFRC_CSFB_S)) ;
    }
}

/**
 * @brief Set up one time software forced Action qualifier outputs
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  epwmOutput is the ePWM pin type.
 * @param [in]  output is the Action Qualifier output.
 * @remarks
 *  This function sets up the one time software forced Action Qualifier output
 *  on ePWM A or ePWMB, depending on the value of epwmOutput to a value
 *  specified by outPut.
 *  The following are valid values for the parameters.
 *    - epwmOutput
 *        - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *        - EPWM_AQ_OUTPUT_B          - ePWMxB output
 *    - output
 *        - EPWM_AQ_OUTPUT_NO_CHANGE  - No change in the output pins
 *        - EPWM_AQ_OUTPUT_LOW        - Set output pins to low
 *        - EPWM_AQ_OUTPUT_HIGH       - Set output pins to High
 *        - EPWM_AQ_OUTPUT_TOGGLE     - Toggle the output pins
 */
__STATIC_INLINE void
EPWM_setActionQualifierSWAction(uint32_t base,
                                EPWM_ActionQualifierOutputModule epwmOutput,
                                EPWM_ActionQualifierOutput output)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the one time software forced action
    //
    if(epwmOutput == EPWM_AQ_OUTPUT_A)
    {
        HWREGH(base + EPWM_O_AQSFRC) =
                    ((HWREGH(base + EPWM_O_AQSFRC) & ~EPWM_AQSFRC_ACTSFA_M) |
                     ((uint16_t)output));
    }
    else
    {
        HWREGH(base + EPWM_O_AQSFRC) =
                    ((HWREGH(base + EPWM_O_AQSFRC) & ~EPWM_AQSFRC_ACTSFB_M) |
                     ((uint16_t)output << EPWM_AQSFRC_ACTSFB_S));
    }
}

/**
 * @brief Triggers a one time software forced event on Action qualifier
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  epwmOutput is the ePWM pin type.
 * @remarks
 *  This function triggers a one time software forced Action Qualifier event
 *  on ePWM A or B based on the value of epwmOutput.
 *  Valid values for epwmOutput are:
 *    - EPWM_AQ_OUTPUT_A          - ePWMxA output
 *    - EPWM_AQ_OUTPUT_B          - ePWMxB output
 */
__STATIC_INLINE void
EPWM_forceActionQualifierSWAction(uint32_t base,
                                  EPWM_ActionQualifierOutputModule epwmOutput)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Initiate a software forced event
    //
    if(epwmOutput == EPWM_AQ_OUTPUT_A)
    {
        HWREGH(base + EPWM_O_AQSFRC) |= EPWM_AQSFRC_OTSFA;
    }
    else
    {
        HWREGH(base + EPWM_O_AQSFRC) |= EPWM_AQSFRC_OTSFB;
    }
}


/**
 * @brief enable CMPA1/B1
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * The EPWM B channel uses the CMPA1/B1 comparison event for AQ action,
 * in which case the CMPA/B comparison event is not valid for the EPWMB.
 */
static inline void
EPWM_enableCounterCompare_Extend(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

	HWREGH(base + EPWM_O_CMPSEL) |= 1;
}

/**
 * @brief disable CMPA1/B1
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 * The EPWM B channel uses the CMPA1/B1 comparison event for AQ action,
 * in which case the CMPA/B comparison event is not valid for the EPWMB.
 */
static inline void
EPWM_disableCounterCompare_Extend(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

	HWREGH(base + EPWM_O_CMPSEL) &= ~1;
}

/**
 * @brief  Set extend counter compare values.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @param [in]  compCount is the counter compare count value.
 * @remarks
 * This function sets the counter compare value for counter compare registers.
 * The maximum value for compCount is 0xFFFF.
 * Valid values for compModule are:
 *   - EPWM_COUNTER_COMPARE_A1 - counter compare A1.
 *   - EPWM_COUNTER_COMPARE_B1 - counter compare B1.
 */
static inline void
EPWM_setCounterCompareValue_Extend(uint32_t base, EPWM_CounterCompareModule_Extend compModule,
		uint16_t compCount)
{
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset for the Counter compare
    //
    registerOffset = EPWM_O_CMPA1 + (uint32_t)compModule;

#if(22 == EPWM_VERSION || 12 == EPWM_VERSION )
	//
	// Write to COMPA1 or COMPB1 bits
	//
	HWREGH(base + registerOffset) = compCount;

#elif(EPWM_VERSION == 30)
	//
	// Write to COMPA1 or COMPB1 bits
	//
	HWREGH(base + registerOffset + 0x2U) = compCount;

#endif
}

/**
 * @brief  Get extend counter compare values.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @remarks
 * This function gets the counter compare value for counter compare registers.
 * Valid values for compModule are:
 *   - EPWM_COUNTER_COMPARE_A1 - counter compare A.
 *   - EPWM_COUNTER_COMPARE_B1 - counter compare B.
 * @return The counter compare count value.
 */
__STATIC_INLINE uint16_t
EPWM_getCounterCompareValue_Extend(uint32_t base, EPWM_CounterCompareModule_Extend compModule)
{
    uint32_t registerOffset;
    uint16_t compCount;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Get the register offset for the Counter compare
    //
    registerOffset = EPWM_O_CMPA1 + (uint32_t)compModule;

    //
    // Read from the counter compare registers.
    //
	//
	// Read COMPA1 or COMPB1 bits
	//
#if(22 == EPWM_VERSION || 12 == EPWM_VERSION)

	compCount = (uint16_t)((HWREG(base + registerOffset) &
				 0xFFFF0000UL) >> 16U);

#elif(EPWM_VERSION == 30)

	compCount = (uint16_t)((HWREG(base + registerOffset + 0x2U) &
				 0xFFFF0000UL) >> 16U);

#endif

    return(compCount);
}


/**
 * @brief  Sets up the Counter Compare shadow load mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @param [in]  loadMode is the shadow to active load mode.
 * @remarks
 * This function enables and sets up the counter compare shadow load mode.
 * Valid values for the variables are:
 * - compModule
 * - EPWM_COUNTER_COMPARE_A1 - counter compare A1.
 * - EPWM_COUNTER_COMPARE_B1 - counter compare B1.
 * - loadMode
 * - EPWM_COMP_LOAD_ON_CNTR_ZERO - load when counter equals zero
 * - EPWM_COMP_LOAD_ON_CNTR_PERIOD - load when counter equals period
 * - EPWM_COMP_LOAD_ON_CNTR_ZERO_PERIOD - load when counter equals
 * zero or period
 * - EPWM_COMP_LOAD_FREEZE - Freeze shadow to active load
 * - EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO - load when counter equals zero
 * - EPWM_COMP_LOAD_ON_SYNC_CNTR_PERIOD -load when counter equals period
 * - EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO_PERIOD - load when counter equals
 * zero or period
 * - EPWM_COMP_LOAD_ON_SYNC_ONLY - load on sync only
 */
__STATIC_INLINE void
EPWM_setCounterCompareShadowLoadMode_Extend(uint32_t base,EPWM_CounterCompareModule_Extend compModule,
                                     EPWM_CounterCompareLoadMode loadMode)
{
    uint16_t syncModeOffset;
    uint16_t loadModeOffset;
    uint16_t shadowModeOffset;
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if((compModule == EPWM_COUNTER_COMPARE_A1))
    {
        syncModeOffset = 10U;
        loadModeOffset = 0U;
        shadowModeOffset = 4U;
    }
    else
    {
        syncModeOffset = 12U;
        loadModeOffset = 2U;
        shadowModeOffset = 6U;
    }

    //
    // Get the register offset.  EPWM_O_CMPCTL for A&B or
    // EPWM_O_CMPCTL2 for C&D
    //
    registerOffset = base + EPWM_O_CMPCTL3;

    //
    // Set the appropriate sync and load mode bits and also enable shadow
    // load mode. Shadow to active load can also be frozen.
    //
    HWREGH(registerOffset) = ((HWREGH(registerOffset) &
                         ~((0x3U << syncModeOffset) | // Clear sync mode
                           (0x3U << loadModeOffset) | // Clear load mode
                           (0x1U << shadowModeOffset))) | // shadow mode
                         ((((uint16_t)loadMode >> 2U) << syncModeOffset) |
                         (((uint16_t)loadMode & 0x3U) << loadModeOffset)));
}

/**
 * @brief  Sets up the Counter Compare shadow load mode
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @remarks
 * This function disables counter compare shadow load mode.
 * Valid values for the variables are:
 * - compModule
 * - EPWM_COUNTER_COMPARE_A1 - counter compare A1.
 * - EPWM_COUNTER_COMPARE_B1 - counter compare B1.
 */
__STATIC_INLINE void
EPWM_disableCounterCompareShadowLoadMode_Extend(uint32_t base,
		EPWM_CounterCompareModule_Extend compModule)
{
    uint16_t shadowModeOffset;
    uint32_t registerOffset;

    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    if(compModule == EPWM_COUNTER_COMPARE_A1)
    {
        shadowModeOffset = 4U;
    }
    else
    {
        shadowModeOffset = 6U;
    }

    //
    // Get the register offset.  EPWM_O_CMPCTL for A&B or
    // EPWM_O_CMPCTL2 for C&D
    //
    registerOffset = base + EPWM_O_CMPCTL3;

    //
    // Disable shadow load mode.
    //
    HWREGH(registerOffset) = (HWREGH(registerOffset) |
                             (0x1U << shadowModeOffset));
}

/**
 * @brief  Return the counter compare shadow register full status.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  compModule is the counter compare module.
 * @remarks
 * This function returns the counter Compare shadow register full status flag.
 * Valid values for compModule are:
 *   - EPWM_COUNTER_COMPARE_A1 - counter compare A1.
 *   - EPWM_COUNTER_COMPARE_B1 - counter compare B1.
 * @return Returns true if the shadow register is full.
 * 		   Returns false if the shadow register is not full.
 */
__STATIC_INLINE bool
EPWM_getCounterCompareShadowStatus_Extend(uint32_t base,
		EPWM_CounterCompareModule_Extend compModule)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Check the validity of input.
    // COMPA and COMPB are valid input arguments.
    //
    ASSERT((compModule == EPWM_COUNTER_COMPARE_A1) ||
            (compModule == EPWM_COUNTER_COMPARE_B1));

    //
    // Read the value of SHDWAFULL or SHDWBFULL bit
    //
    return((((HWREG(base + EPWM_O_CMPCTL3) >>
              ((((uint16_t)compModule >> 1U) & 0x1U) + 8U)) &
              0x1U) == 0x1U) ? true:false);
}




#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_CC_AQ_H_ */
