/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_define.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_DEFINE_H_
#define DEVICE_DRIVERLIB_EPWM_DEFINE_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "inc/hw_types.h"
#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "debug.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
//*****************************************************************************
//
// Defines for the API.
//
//*****************************************************************************

//*****************************************************************************
//
// Defines for the API.
//
//*****************************************************************************
//*****************************************************************************
//
// Define to specify mask for source parameter for
// EPWM_enableSyncOutPulseSource() & EPWM_disableSyncOutPulseSource()
//
//*****************************************************************************
#define EPWM_SYNC_OUT_SOURCE_M    ((uint16_t)EPWM_SYNCOUTEN_SWEN             |\
                                   (uint16_t)EPWM_SYNCOUTEN_ZEROEN           |\
                                   (uint16_t)EPWM_SYNCOUTEN_CMPBEN           |\
                                   (uint16_t)EPWM_SYNCOUTEN_CMPCEN           |\
                                   (uint16_t)EPWM_SYNCOUTEN_CMPDEN           |\
                                   (uint16_t)EPWM_SYNCOUTEN_DCAEVT1EN        |\
                                   (uint16_t)EPWM_SYNCOUTEN_DCBEVT1EN)


	//*****************************************************************************
//
// Values that can be passed to EPWM_enableSyncOutPulseSource() &
// EPWM_disableSyncOutPulseSource() as the \e mode parameter.
//
//*****************************************************************************
//! Software force generated EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_SOFTWARE         EPWM_SYNCOUTEN_SWEN
//! Counter zero event generates EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_CNTR_ZERO        EPWM_SYNCOUTEN_ZEROEN
//! Counter equal to CMPB event generates EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_B   EPWM_SYNCOUTEN_CMPBEN
//! Counter equal to CMPC event generates EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_C   EPWM_SYNCOUTEN_CMPCEN
//! Counter equal to CMPD event generates EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_CNTR_COMPARE_D   EPWM_SYNCOUTEN_CMPDEN
//! DCA Event 1 Sync signal generates EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_DCA_EVT1_SYNC    EPWM_SYNCOUTEN_DCAEVT1EN
//! DCB Event 1 Sync signal generates EPWM sync-out pulse
#define EPWM_SYNC_OUT_PULSE_ON_DCB_EVT1_SYNC    EPWM_SYNCOUTEN_DCBEVT1EN
//! Enable all the above sources
#define EPWM_SYNC_OUT_PULSE_ON_ALL              EPWM_SYNC_OUT_SOURCE_M


//
// Time Base Module
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_setEmulationMode() as the
//! \e emulationMode parameter.
//
//*****************************************************************************
typedef enum
{
   //! Stop after next Time Base counter increment or decrement
   EPWM_EMULATION_STOP_AFTER_NEXT_TB = 0,
   //! Stop when counter completes whole cycle
   EPWM_EMULATION_STOP_AFTER_FULL_CYCLE = 1,
   //! Free run
   EPWM_EMULATION_FREE_RUN = 2
} EPWM_EmulationMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setCountModeAfterSync() as the
//! \e mode parameter.
//
//*****************************************************************************
typedef enum
{
   EPWM_COUNT_MODE_DOWN_AFTER_SYNC = 0, //!< Count down after sync event
   EPWM_COUNT_MODE_UP_AFTER_SYNC = 1    //!< Count up after sync event
} EPWM_SyncCountMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setClockPrescaler() as the
//! \e prescaler parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_CLOCK_DIVIDER_1 = 0,     //!< Divide clock by 1
    EPWM_CLOCK_DIVIDER_2 = 1,     //!< Divide clock by 2
    EPWM_CLOCK_DIVIDER_4 = 2,     //!< Divide clock by 4
    EPWM_CLOCK_DIVIDER_8 = 3,     //!< Divide clock by 8
    EPWM_CLOCK_DIVIDER_16 = 4,    //!< Divide clock by 16
    EPWM_CLOCK_DIVIDER_32 = 5,    //!< Divide clock by 32
    EPWM_CLOCK_DIVIDER_64 = 6,    //!< Divide clock by 64
    EPWM_CLOCK_DIVIDER_128 = 7    //!< Divide clock by 128
} EPWM_ClockDivider;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setClockPrescaler() as the
//! \e highSpeedPrescaler parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_HSCLOCK_DIVIDER_1 = 0,     //!< Divide clock by 1
    EPWM_HSCLOCK_DIVIDER_2 = 1,     //!< Divide clock by 2
    EPWM_HSCLOCK_DIVIDER_4 = 2,     //!< Divide clock by 4
    EPWM_HSCLOCK_DIVIDER_6 = 3,     //!< Divide clock by 6
    EPWM_HSCLOCK_DIVIDER_8 = 4,     //!< Divide clock by 8
    EPWM_HSCLOCK_DIVIDER_10 = 5,    //!< Divide clock by 10
    EPWM_HSCLOCK_DIVIDER_12 = 6,    //!< Divide clock by 12
    EPWM_HSCLOCK_DIVIDER_14 = 7     //!< Divide clock by 14
} EPWM_HSClockDivider;


#if (EPWM_VERSION == 12)|| (EPWM_VERSION == 0) ||(EPWM_VERSION == 11) || (EPWM_VERSION == 21)

//*****************************************************************************
//
//! Values that can be passed to EPWM_setSyncInPulseSource() as the \e mode
//! parameter.
//
//*****************************************************************************
typedef enum
{
    //! Disable Sync-in
    EPWM_SYNC_IN_PULSE_SRC_DISABLE         = 0x0,
    //! Sync-in source is EPWM1 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM1   = 0x1,
    //! Sync-in source is EPWM2 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM2   = 0x2,
    //! Sync-in source is EPWM3 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM3   = 0x3,
    //! Sync-in source is EPWM4 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM4   = 0x4,
    //! Sync-in source is EPWM5 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM5   = 0x5,
    //! Sync-in source is EPWM6 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM6   = 0x6,
    //! Sync-in source is EPWM7 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM7   = 0x7,
    //! Sync-in source is EPWM8 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM8   = 0x8,
    //! Sync-in source is EPWM9 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM9   = 0x9,
    //! Sync-in source is EPWM10 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM10   = 0xA,
    //! Sync-in source is EPWM11 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM11   = 0xB,
    //! Sync-in source is EPWM12 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM12   = 0xC,
    //! Sync-in source is ECAP1 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP1   = 0x11,
    //! Sync-in source is ECAP2 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP2   = 0x12,
    //! Sync-in source is ECAP3 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP3   = 0x13,
    //! Sync-in source is ECAP4 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP4   = 0x14,
    //! Sync-in source is ECAP5 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP5   = 0x15,
    //! Sync-in source is ECAP6 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP6   = 0x16,
    //! Sync-in source is ECAP7 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP7   = 0x17,
    //! Sync-in source is Input XBAR out5 signal
    EPWM_SYNC_IN_PULSE_SRC_INPUTXBAR_OUT5 = 0x18,
    //! Sync-in source is Input XBAR out6 signal
    EPWM_SYNC_IN_PULSE_SRC_INPUTXBAR_OUT6 = 0x19,
} EPWM_SyncInPulseSource;
#elif (EPWM_VERSION == 22)||(EPWM_VERSION == 30)
typedef enum
{
    //! Disable Sync-in
    EPWM_SYNC_IN_PULSE_SRC_DISABLE         = 0x0,
    //! Sync-in source is EPWM1 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM1   = 0x1,
    //! Sync-in source is EPWM2 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM2   = 0x2,
    //! Sync-in source is EPWM3 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM3   = 0x3,
    //! Sync-in source is EPWM4 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM4   = 0x4,
    //! Sync-in source is EPWM5 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM5   = 0x5,
    //! Sync-in source is EPWM6 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM6   = 0x6,
    //! Sync-in source is EPWM7 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM7   = 0x7,
    //! Sync-in source is EPWM8 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM8   = 0x8,
    //! Sync-in source is EPWM9 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM9   = 0x9,
    //! Sync-in source is EPWM10 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM10   = 0xA,
    //! Sync-in source is EPWM11 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM11   = 0xB,
    //! Sync-in source is EPWM12 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM12   = 0xC,
    //! Sync-in source is EPWM13 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM13   = 0xD,
    //! Sync-in source is EPWM14 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM14   = 0xE,
    //! Sync-in source is EPWM15 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM15   = 0xF,
    //! Sync-in source is EPWM16 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM16   = 0x10,
    //! Sync-in source is ECAP1 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP1   = 0x11,
    //! Sync-in source is ECAP2 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP2   = 0x12,
    //! Sync-in source is ECAP3 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP3   = 0x13,
    //! Sync-in source is ECAP4 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP4   = 0x14,
    //! Sync-in source is ECAP5 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP5   = 0x15,
    //! Sync-in source is ECAP6 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP6   = 0x16,
    //! Sync-in source is ECAP7 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_ECAP7   = 0x17,
    //! Sync-in source is Input XBAR out5 signal
    EPWM_SYNC_IN_PULSE_SRC_INPUTXBAR_OUT5 = 0x18,
    //! Sync-in source is Input XBAR out6 signal
    EPWM_SYNC_IN_PULSE_SRC_INPUTXBAR_OUT6 = 0x19,
    //! Sync-in source is EPWM16 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM17 = 0x1C,
    //! Sync-in source is EPWM16 sync-out signal
    EPWM_SYNC_IN_PULSE_SRC_SYNCOUT_EPWM18 = 0x1D,
} EPWM_SyncInPulseSource;

#elif (EPWM_VERSION == 11) || (EPWM_VERSION == 21)
//*****************************************************************************
//
//! The following values define the \e syncInput parameter for
//! SysCtl_setSyncInputConfig().
//
//*****************************************************************************
typedef enum
{
    SYSCTL_SYNC_IN_EPWM4 = 0, //!< Sync input to EPWM4
    SYSCTL_SYNC_IN_EPWM7 = 3, //!< Sync input to EPWM7
	SYSCTL_SYNC_IN_EPWM10 = 6, //!< Sync input to EPWM7
    SYSCTL_SYNC_IN_ECAP1 = 9, //!< Sync input to ECAP1
    SYSCTL_SYNC_IN_ECAP4 = 12, //!< Sync input to ECAP4
    SYSCTL_SYNC_IN_ECAP6 = 15, //!< Sync input to ECAP6
	SYSCTL_SYNC_IN_EPWM1 = 29
} SysCtl_SyncInput;

//*****************************************************************************
//
//! The following values define the \e syncSrc parameter for
//! SysCtl_setSyncInputConfig(). Note that some of these are only valid for
//! certain values of \e syncInput. See device technical reference manual for
//! info on time-base counter synchronization for details.
//
//*****************************************************************************
typedef enum
{
    //! EPWM1SYNCOUT
    SYSCTL_SYNC_IN_SRC_EPWM1SYNCOUT     = 0,
    //! EPWM4SYNCOUT
    SYSCTL_SYNC_IN_SRC_EPWM4SYNCOUT     = 1,
    //! EPWM7SYNCOUT
    SYSCTL_SYNC_IN_SRC_EPWM7SYNCOUT     = 2,
    //! ECAP1SYNCOUT
    SYSCTL_SYNC_IN_SRC_ECAP1SYNCOUT     = 4,
    //! EXTSYNCIN1--Valid for all values of syncInput
    SYSCTL_SYNC_IN_SRC_EXTSYNCIN1       = 5,
    //! EXTSYNCIN2--Valid for all values of syncInput
    SYSCTL_SYNC_IN_SRC_EXTSYNCIN2       = 6,
} SysCtl_SyncInputSource;

//*****************************************************************************
//
//! The following values define the \e syncSrc parameter for
//! SysCtl_setSyncOutputConfig().
//
//*****************************************************************************
typedef enum
{
    SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT,   //!< EPWM1SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM4SYNCOUT,   //!< EPWM4SYNCOUT --> EXTSYNCOUT
    SYSCTL_SYNC_OUT_SRC_EPWM7SYNCOUT    //!< EPWM7SYNCOUT --> EXTSYNCOUT

} SysCtl_SyncOutputSource;

#endif

//*****************************************************************************
//
//! Values that can be passed to EPWM_setOneShotSyncOutTrigger() as the
//! \e trigger parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_OSHT_SYNC_OUT_TRIG_SYNC   = 0x0, //!< Trigger is OSHT sync
    EPWM_OSHT_SYNC_OUT_TRIG_RELOAD = 0x1  //!< Trigger is OSHT reload
} EPWM_OneShotSyncOutTrigger;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setPeriodLoadMode()  as the
//! \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! PWM Period register access is through shadow register
    EPWM_PERIOD_SHADOW_LOAD = 0,
    //! PWM Period register access is directly
    EPWM_PERIOD_DIRECT_LOAD = 1
} EPWM_PeriodLoadMode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTimeBaseCounterMode() as the
//! \e counterMode parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_COUNTER_MODE_UP = 0,         //!< Up - count mode
    EPWM_COUNTER_MODE_DOWN = 1,       //!< Down - count mode
    EPWM_COUNTER_MODE_UP_DOWN = 2,    //!< Up - down - count mode
    EPWM_COUNTER_MODE_STOP_FREEZE = 3 //!< Stop - Freeze counter
} EPWM_TimeBaseCountMode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectPeriodLoadEvent() as the
//! \e shadowLoadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Shadow to active load occurs when time base counter reaches 0
    EPWM_SHADOW_LOAD_MODE_COUNTER_ZERO = 0,
    //! Shadow to active load occurs when time base counter reaches 0 and a
    //! SYNC occurs
    EPWM_SHADOW_LOAD_MODE_COUNTER_SYNC = 1,
    //! Shadow to active load occurs only when a SYNC occurs
    EPWM_SHADOW_LOAD_MODE_SYNC         = 2
} EPWM_PeriodShadowLoadMode;

//*****************************************************************************
//
// Values that can be returned by the EPWM_getTimeBaseCounterDirection()
//
//*****************************************************************************
//! Time base counter is counting up
//!
#define EPWM_TIME_BASE_STATUS_COUNT_UP                1U
//! Time base counter is counting down
//!
#define EPWM_TIME_BASE_STATUS_COUNT_DOWN              0U

//*****************************************************************************
//
//! Values that can be passed to EPWM_setupEPWMLinks() as the \e epwmLink
//! parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_LINK_WITH_EPWM_1 = 0,   //!< link current ePWM with ePWM1
    EPWM_LINK_WITH_EPWM_2 = 1,   //!< link current ePWM with ePWM2
    EPWM_LINK_WITH_EPWM_3 = 2,   //!< link current ePWM with ePWM3
    EPWM_LINK_WITH_EPWM_4 = 3,   //!< link current ePWM with ePWM4
    EPWM_LINK_WITH_EPWM_5 = 4,   //!< link current ePWM with ePWM5
    EPWM_LINK_WITH_EPWM_6 = 5,   //!< link current ePWM with ePWM6
    EPWM_LINK_WITH_EPWM_7 = 6,   //!< link current ePWM with ePWM7
    EPWM_LINK_WITH_EPWM_8 = 7,   //!< link current ePWM with ePWM8
    EPWM_LINK_WITH_EPWM_9 = 8,   //!< link current ePWM with ePWM9
    EPWM_LINK_WITH_EPWM_10 = 9,   //!< link current ePWM with ePWM10
    EPWM_LINK_WITH_EPWM_11 = 10,   //!< link current ePWM with ePWM11
    EPWM_LINK_WITH_EPWM_12 = 11,   //!< link current ePWM with ePWM12
    EPWM_LINK_WITH_EPWM_13 = 12,   //!< link current ePWM with ePWM13
    EPWM_LINK_WITH_EPWM_14 = 13,   //!< link current ePWM with ePWM14
    EPWM_LINK_WITH_EPWM_15 = 14,   //!< link current ePWM with ePWM15
    EPWM_LINK_WITH_EPWM_16 = 15,    //!< link current ePWM with ePWM16
	EPWM_LINK_WITH_EPWM_17 = 16,   //!< link current ePWM with ePWM15
    EPWM_LINK_WITH_EPWM_18 = 17    //!< link current ePWM with ePWM16
} EPWM_CurrentLink;


typedef enum
{
    //link1
    EPWM_LINK_TBPRD = 0U,   //!< link TBPRD registers
	EPWM_LINK_COMP_A = 4U,   //!< link COMPA registers
	EPWM_LINK_COMP_B = 8U,   //!< link COMPB registers
	EPWM_LINK_COMP_C = 12U,   //!< link COMPC registers
	EPWM_LINK_COMP_D = 16U,   //!< link COMPD registers
#if (EPWM_VERSION == 30 || EPWM_VERSION == 12)
	EPWM_LINK_COMP_A1 = 20U,   //!< link COMPA1 registers
	EPWM_LINK_COMP_B1 = 24U,   //!< link COMPB1 registers
#endif
	EPWM_LINK_GLDCTL2 = 28U,    //!< link GLDCTL2 registers
    //link2
#if ( EPWM_VERSION == 30 || EPWM_VERSION == 23 )
	EPWM_LINK_DBFED = 4U+32U,    //!< link DBFED registers
	EPWM_LINK_DBRED = 0U+32U,    //!< link DBFED registers
#endif
#if (EPWM_VERSION == 22 || EPWM_VERSION == 23 )
    EPWM_LINK_COMP_A1 = 20U+32U,   //!< link COMPA1 registers
    EPWM_LINK_COMP_B1 = 24U+32U,   //!< link COMPB1 registers
#endif
    //link3
	EPWM_LINK_AQCSFRC  = 12U+64U,                   //!< link AQCSFRC registers
	EPWM_LINK_AQCTL  = 8U+64U,                      //!< link AQCTL registers
#if (EPWM_VERSION == 12 || EPWM_VERSION == 22)
    EPWM_LINK_DB = 4U+64U,    //!< link DB registers
#endif
	EPWM_LINK_TBPHS = 0U+64U,    //!< link TBPHS registers
    //XLOAD
#if (EPWM_VERSION == 30 || EPWM_VERSION == 23)
    //XCMP
	EPWM_LINK_XLOAD = 0U + 96U    //!< link XLOAD registers
#endif

} EPWM_LinkComponent;


//
// Counter Compare Module
//
//*****************************************************************************
//
//! Values that can be passed to the EPWM_getCounterCompareShadowStatus(),
//! EPWM_setCounterCompareValue(), EPWM_setCounterCompareShadowLoadMode(),
//! EPWM_disableCounterCompareShadowLoadMode()
//! as the \e compModule parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_COUNTER_COMPARE_A = 0, //!< Counter compare A
    EPWM_COUNTER_COMPARE_B = 4, //!< Counter compare B
    EPWM_COUNTER_COMPARE_C = 10, //!< Counter compare C
    EPWM_COUNTER_COMPARE_D = 14 //!< Counter compare D
} EPWM_CounterCompareModule;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setExtendCounterCompareValue() as the
//! \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_COUNTER_COMPARE_A1 = 0, //!< Counter compare A
    EPWM_COUNTER_COMPARE_B1 = 4, //!< Counter compare B
} EPWM_CounterCompareModule_Extend;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setCounterCompareShadowLoadMode() as the
//! \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load when counter equals zero
    EPWM_COMP_LOAD_ON_CNTR_ZERO = 0,
    //! Load when counter equals period
    EPWM_COMP_LOAD_ON_CNTR_PERIOD = 1,
    //! Load when counter equals zero or period
    EPWM_COMP_LOAD_ON_CNTR_ZERO_PERIOD = 2,
    //! Freeze shadow to active load
    EPWM_COMP_LOAD_FREEZE = 3,
    //! Load on sync or when counter equals zero
    EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO = 4,
    //! Load on sync or when counter equals period
    EPWM_COMP_LOAD_ON_SYNC_CNTR_PERIOD = 5,
    //! Load on sync or when counter equals zero or period
    EPWM_COMP_LOAD_ON_SYNC_CNTR_ZERO_PERIOD = 6,
    //! Load on sync only
    EPWM_COMP_LOAD_ON_SYNC_ONLY = 8
} EPWM_CounterCompareLoadMode;



//
// Action Qualifier Module
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierShadowLoadMode() and
//! EPWM_disableActionQualifierShadowLoadMode() as the \e aqModule parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_ACTION_QUALIFIER_A = 0, //!< Action Qualifier A
    EPWM_ACTION_QUALIFIER_B = 2  //!< Action Qualifier B
} EPWM_ActionQualifierModule;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierShadowLoadMode() as the
//! \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load when counter equals zero
    EPWM_AQ_LOAD_ON_CNTR_ZERO = 0,
    //! Load when counter equals period
    EPWM_AQ_LOAD_ON_CNTR_PERIOD = 1,
    //! Load when counter equals zero or period
    EPWM_AQ_LOAD_ON_CNTR_ZERO_PERIOD = 2,
    //! Freeze shadow to active load
    EPWM_AQ_LOAD_FREEZE = 3,
    //! Load on sync or when counter equals zero
    EPWM_AQ_LOAD_ON_SYNC_CNTR_ZERO = 4,
    //! Load on sync or when counter equals period
    EPWM_AQ_LOAD_ON_SYNC_CNTR_PERIOD = 5,
    //! Load on sync or when counter equals zero or period
    EPWM_AQ_LOAD_ON_SYNC_CNTR_ZERO_PERIOD = 6,
    //! Load on sync only
    EPWM_AQ_LOAD_ON_SYNC_ONLY = 8
} EPWM_ActionQualifierLoadMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierT1TriggerSource() and
//! EPWM_setActionQualifierT2TriggerSource() as the \e trigger parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_1 = 0,      //!< Digital compare event A 1
    EPWM_AQ_TRIGGER_EVENT_TRIG_DCA_2 = 1,      //!< Digital compare event A 2
    EPWM_AQ_TRIGGER_EVENT_TRIG_DCB_1 = 2,      //!< Digital compare event B 1
    EPWM_AQ_TRIGGER_EVENT_TRIG_DCB_2 = 3,      //!< Digital compare event B 2
    EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_1  = 4,      //!< Trip zone 1
    EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_2  = 5,      //!< Trip zone 2
    EPWM_AQ_TRIGGER_EVENT_TRIG_TZ_3  = 6,      //!< Trip zone 3
    EPWM_AQ_TRIGGER_EVENT_TRIG_EPWM_SYNCIN = 7,//!< ePWM sync
    EPWM_AQ_TRIGGER_EVENT_TRIG_DC_EVTFILT = 8  //!< Digital compare filter event
} EPWM_ActionQualifierTriggerSource;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierAction() as the \e
//! event parameter.
//
//*****************************************************************************
typedef enum
{
    //! Time base counter equals zero
    EPWM_AQ_OUTPUT_ON_TIMEBASE_ZERO       = 0,
    //! Time base counter equals period
    EPWM_AQ_OUTPUT_ON_TIMEBASE_PERIOD     = 2,
    //! Time base counter up equals COMPA
    EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA    = 4,
    //! Time base counter down equals COMPA
    EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA  = 6,
    //! Time base counter up equals COMPB
    EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB    = 8,
    //! Time base counter down equals COMPB
    EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB  = 10,
    //! T1 event on count up
    EPWM_AQ_OUTPUT_ON_T1_COUNT_UP         = 1,
    //! T1 event on count down
    EPWM_AQ_OUTPUT_ON_T1_COUNT_DOWN       = 3,
    //! T2 event on count up
    EPWM_AQ_OUTPUT_ON_T2_COUNT_UP         = 5,
    //! T2 event on count down
    EPWM_AQ_OUTPUT_ON_T2_COUNT_DOWN       = 7,

	/*Extend compare*/
    //! Time base counter up equals COMPA
    EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA1    = 4,
    //! Time base counter down equals COMPA
    EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPA1  = 6,
    //! Time base counter up equals COMPB
    EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPB1    = 8,
    //! Time base counter down equals COMPB
    EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB1  = 10
} EPWM_ActionQualifierOutputEvent;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierSWAction(),
//! EPWM_setActionQualifierAction() as the \e outPut parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_AQ_OUTPUT_NO_CHANGE = 0,  //!< No change in the output pins
    EPWM_AQ_OUTPUT_LOW       = 1,  //!< Set output pins to low
    EPWM_AQ_OUTPUT_HIGH      = 2,  //!< Set output pins to High
    EPWM_AQ_OUTPUT_TOGGLE    = 3   //!< Toggle the output pins
} EPWM_ActionQualifierOutput;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierContSWForceAction()
//! as the \e outPut parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_AQ_SW_DISABLED         = 0,  //!< Software forcing disabled
    EPWM_AQ_SW_OUTPUT_LOW       = 1,  //!< Set output pins to low
    EPWM_AQ_SW_OUTPUT_HIGH      = 2   //!< Set output pins to High
} EPWM_ActionQualifierSWOutput;



//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierActionComplete()
//! as the \e action parameter.
//
//*****************************************************************************
typedef enum
{
    //! Time base counter equals zero and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_ZERO = 0x0,
    //! Time base counter equals zero and set output pins to low
    EPWM_AQ_OUTPUT_LOW_ZERO = 0x1,
    //! Time base counter equals zero and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_ZERO = 0x2,
    //! Time base counter equals zero and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_ZERO = 0x3,
    //! Time base counter equals period and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_PERIOD = 0x0,
    //! Time base counter equals period and set output pins to low
    EPWM_AQ_OUTPUT_LOW_PERIOD = 0x4,
    //! Time base counter equals period and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_PERIOD = 0x8,
    //! Time base counter equals period and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_PERIOD = 0xC,
    //! Time base counter up equals COMPA and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_UP_CMPA = 0x00,
    //! Time base counter up equals COMPA and set output pins to low
    EPWM_AQ_OUTPUT_LOW_UP_CMPA = 0x10,
    //! Time base counter up equals COMPA and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_UP_CMPA = 0x20,
    //! Time base counter up equals COMPA and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_UP_CMPA = 0x30,
    //! Time base counter down equals COMPA and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_CMPA = 0x00,
    //! Time base counter down equals COMPA and set output pins to low
    EPWM_AQ_OUTPUT_LOW_DOWN_CMPA = 0x40,
    //! Time base counter down equals COMPA and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_DOWN_CMPA = 0x80,
    //! Time base counter down equals COMPA and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_DOWN_CMPA = 0xC0,
    //! Time base counter up equals COMPB and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_UP_CMPB = 0x000,
    //! Time base counter up equals COMPB and set output pins to low
    EPWM_AQ_OUTPUT_LOW_UP_CMPB = 0x100,
    //! Time base counter up equals COMPB and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_UP_CMPB = 0x200,
    //! Time base counter up equals COMPB and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_UP_CMPB = 0x300,
    //! Time base counter down equals COMPB and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_CMPB = 0x000,
    //! Time base counter down equals COMPB and set output pins to low
    EPWM_AQ_OUTPUT_LOW_DOWN_CMPB = 0x400,
    //! Time base counter down equals COMPB and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_DOWN_CMPB = 0x800,
    //! Time base counter down equals COMPB and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_DOWN_CMPB = 0xC00
} EPWM_ActionQualifierEventAction;



//*****************************************************************************
//
//! Values that can be passed to
//! EPWM_setAdditionalActionQualifierActionComplete()  as the \e action
//! parameter.
//
//*****************************************************************************
typedef enum
{
    //! T1 event on count up and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_UP_T1 = 0x0,
    //! T1 event on count up and set output pins to low
    EPWM_AQ_OUTPUT_LOW_UP_T1 = 0x1,
    //! T1 event on count up and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_UP_T1 = 0x2,
    //! T1 event on count up and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_UP_T1 = 0x3,
    //! T1 event on count down and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_T1 = 0x0,
    //! T1 event on count down and set output pins to low
    EPWM_AQ_OUTPUT_LOW_DOWN_T1 = 0x4,
    //! T1 event on count down and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_DOWN_T1 = 0x8,
    //! T1 event on count down and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_DOWN_T1 = 0xC,
    //! T2 event on count up and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_UP_T2 = 0x00,
    //! T2 event on count up and set output pins to low
    EPWM_AQ_OUTPUT_LOW_UP_T2 = 0x10,
    //! T2 event on count up and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_UP_T2 = 0x20,
    //! T2 event on count up and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_UP_T2 = 0x30,
    //! T2 event on count down and no change in the output pins
    EPWM_AQ_OUTPUT_NO_CHANGE_DOWN_T2 = 0x00,
    //! T2 event on count down and set output pins to low
    EPWM_AQ_OUTPUT_LOW_DOWN_T2 = 0x40,
    //! T2 event on count down and set output pins to high
    EPWM_AQ_OUTPUT_HIGH_DOWN_T2 = 0x80,
    //! T2 event on count down and toggle the output pins
    EPWM_AQ_OUTPUT_TOGGLE_DOWN_T2 = 0xC0
} EPWM_AdditionalActionQualifierEventAction;



//*****************************************************************************
//
//! Values that can be passed to EPWM_forceActionQualifierSWAction(),
//! EPWM_setActionQualifierSWAction(), EPWM_setActionQualifierAction()
//! EPWM_setActionQualifierContSWForceAction() as the \e epwmOutput parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_AQ_OUTPUT_A = 0, //!< ePWMxA output
    EPWM_AQ_OUTPUT_B = 4  //!< ePWMxB output  //huwei 20230526
} EPWM_ActionQualifierOutputModule;



//*****************************************************************************
//
//! Values that can be passed to EPWM_setActionQualifierContSWForceShadowMode()
//! as the \e mode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Shadow mode load when counter equals zero
    EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO        = 0,
    //! Shadow mode load when counter equals period
    EPWM_AQ_SW_SH_LOAD_ON_CNTR_PERIOD      = 1,
    //! Shadow mode load when counter equals zero or period
    EPWM_AQ_SW_SH_LOAD_ON_CNTR_ZERO_PERIOD = 2,
    //! No shadow load mode. Immediate mode only.
    EPWM_AQ_SW_IMMEDIATE_LOAD   = 3
} EPWM_ActionQualifierContForce;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setDeadBandOutputSwapMode()
//! as the \e output parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DB_OUTPUT_A = 1, //!< DB output is ePWMA
    EPWM_DB_OUTPUT_B = 0  //!< DB output is ePWMB
} EPWM_DeadBandOutput;



//*****************************************************************************
//
//! Values that can be passed to EPWM_setDeadBandDelayPolarity(),
//! EPWM_setDeadBandDelayMode() as the \e delayMode parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DB_RED = 1, //!< DB RED (Rising Edge Delay) mode
    EPWM_DB_FED = 0  //!< DB FED (Falling Edge Delay) mode
} EPWM_DeadBandDelayMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setDeadBandDelayPolarity as the
//! \e polarity parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DB_POLARITY_ACTIVE_HIGH = 0, //!< DB polarity is not inverted
    EPWM_DB_POLARITY_ACTIVE_LOW  = 1  //!< DB polarity is inverted
} EPWM_DeadBandPolarity;

//*****************************************************************************
//
// Values that can be passed to EPWM_setRisingEdgeDeadBandDelayInput(),
// EPWM_setFallingEdgeDeadBandDelayInput() as the input parameter.
//
//*****************************************************************************
//! Input signal is ePWMA
//!
#define EPWM_DB_INPUT_EPWMA       0U
//! Input signal is ePWMB
//!
#define EPWM_DB_INPUT_EPWMB       1U
//! Input signal is the output of Rising Edge delay
//!
#define EPWM_DB_INPUT_DB_RED      2U

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDeadBandControlShadowLoadMode() as
//! the \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load when counter equals zero
    EPWM_DB_LOAD_ON_CNTR_ZERO        = 0,
    //! Load when counter equals period
    EPWM_DB_LOAD_ON_CNTR_PERIOD      = 1,
    //! Load when counter equals zero or period
    EPWM_DB_LOAD_ON_CNTR_ZERO_PERIOD = 2,
    //! Freeze shadow to active load
    EPWM_DB_LOAD_FREEZE = 3
} EPWM_DeadBandControlLoadMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setRisingEdgeDelayCountShadowLoadMode()
//! as the \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load when counter equals zero
    EPWM_RED_LOAD_ON_CNTR_ZERO        = 0,
    //! Load when counter equals period
    EPWM_RED_LOAD_ON_CNTR_PERIOD      = 1,
    //! Load when counter equals zero or period
    EPWM_RED_LOAD_ON_CNTR_ZERO_PERIOD = 2,
    //! Freeze shadow to active load
    EPWM_RED_LOAD_FREEZE = 3
} EPWM_RisingEdgeDelayLoadMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setFallingEdgeDelayCountShadowLoadMode()
//! as the \e loadMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load when counter equals zero
    EPWM_FED_LOAD_ON_CNTR_ZERO        = 0,
    //! Load when counter equals period
    EPWM_FED_LOAD_ON_CNTR_PERIOD      = 1,
    //! Load when counter equals zero or period
    EPWM_FED_LOAD_ON_CNTR_ZERO_PERIOD = 2,
    //! Freeze shadow to active load
    EPWM_FED_LOAD_FREEZE = 3
} EPWM_FallingEdgeDelayLoadMode;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setDeadBandCounterClock() as the
//! \e clockMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Dead band counter runs at TBCLK rate
    EPWM_DB_COUNTER_CLOCK_FULL_CYCLE = 0,
    //! Dead band counter runs at 2*TBCLK rate
    EPWM_DB_COUNTER_CLOCK_HALF_CYCLE = 1
} EPWM_DeadBandClockMode;

//
// Trip Zone
//
//*****************************************************************************
//
// Values that can be passed to EPWM_enableTripZoneSignals() and
// EPWM_disableTripZoneSignals() as the tzSignal parameter.
//
//*****************************************************************************
//! TZ1 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_CBC1          0x1U
//! TZ2 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_CBC2          0x2U
//! TZ3 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_CBC3          0x4U
//! TZ4 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_CBC4          0x8U
//! TZ5 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_CBC5          0x10U
//! TZ6 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_CBC6          0x20U
//! DCAEVT2 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_DCAEVT2       0x40U
//! DCBEVT2 Cycle By Cycle
//!
#define EPWM_TZ_SIGNAL_DCBEVT2       0x80U
//! One-shot TZ1
//!
#define EPWM_TZ_SIGNAL_OSHT1         0x100U
//! One-shot TZ2
//!
#define EPWM_TZ_SIGNAL_OSHT2         0x200U
//! One-shot TZ3
//!
#define EPWM_TZ_SIGNAL_OSHT3         0x400U
//! One-shot TZ4
//!
#define EPWM_TZ_SIGNAL_OSHT4         0x800U
//! One-shot TZ5
//!
#define EPWM_TZ_SIGNAL_OSHT5         0x1000U
//! One-shot TZ6
//!
#define EPWM_TZ_SIGNAL_OSHT6         0x2000U
//! One-shot DCAEVT1
//!
#define EPWM_TZ_SIGNAL_DCAEVT1       0x4000U
//! One-shot DCBEVT1
//!
#define EPWM_TZ_SIGNAL_DCBEVT1        0x8000U


//*****************************************************************************
//
// Values that can be passed to EPWM_enableTripZone2Signals() and
// EPWM_disableTripZone2Signals() as the tzSignal parameter.
//
//*****************************************************************************
//! Cycle by cycle capture event
#define EPWM_TZ_SIGNAL_CAPEVT_CBC    (0x1U)
//! One-shot Capture event
#define EPWM_TZ_SIGNAL_CAPEVT_OST    (0x100U)

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneDigitalCompareEventCondition()
//! as the \e dcType parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_TZ_DC_OUTPUT_A1 = 0, //!< Digital Compare output 1 A
    EPWM_TZ_DC_OUTPUT_A2 = 3, //!< Digital Compare output 2 A
    EPWM_TZ_DC_OUTPUT_B1 = 6, //!< Digital Compare output 1 B
    EPWM_TZ_DC_OUTPUT_B2 = 9  //!< Digital Compare output 2 B
} EPWM_TripZoneDigitalCompareOutput;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneDigitalCompareEventCondition()
//! as the \e dcEvent parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_TZ_EVENT_DC_DISABLED = 0,       //!< Event is disabled
    EPWM_TZ_EVENT_DCXH_LOW    = 1,       //!< Event when DCxH low
    EPWM_TZ_EVENT_DCXH_HIGH   = 2,       //!< Event when DCxH high
    EPWM_TZ_EVENT_DCXL_LOW    = 3,       //!< Event when DCxL low
    EPWM_TZ_EVENT_DCXL_HIGH   = 4,       //!< Event when DCxL high
    EPWM_TZ_EVENT_DCXL_HIGH_DCXH_LOW = 5 //!< Event when DCxL high DCxH low
} EPWM_TripZoneDigitalCompareOutputEvent;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneAction() as the \e tzEvent
//! parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_TZ_ACTION_EVENT_TZA = 0,     //!< TZ1 - TZ6, DCAEVT2, DCAEVT1
    EPWM_TZ_ACTION_EVENT_TZB = 2,     //!< TZ1 - TZ6, DCBEVT2, DCBEVT1
    EPWM_TZ_ACTION_EVENT_DCAEVT1 = 4, //!< DCAEVT1 (Digital Compare A event 1)
    EPWM_TZ_ACTION_EVENT_DCAEVT2 = 6, //!< DCAEVT2 (Digital Compare A event 2)
    EPWM_TZ_ACTION_EVENT_DCBEVT1 = 8, //!< DCBEVT1 (Digital Compare B event 1)
    EPWM_TZ_ACTION_EVENT_DCBEVT2 = 10 //!< DCBEVT2 (Digital Compare B event 2)
} EPWM_TripZoneEvent;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneAction() as the
//! \e tzAction parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_TZ_ACTION_HIGH_Z  = 0, //!< High impedance output
    EPWM_TZ_ACTION_HIGH    = 1, //!< High voltage state
    EPWM_TZ_ACTION_LOW     = 2, //!< Low voltage state
    EPWM_TZ_ACTION_DISABLE = 3  //!< Disable action
} EPWM_TripZoneAction;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneAdvAction() as the
//! \e tzAdvEvent parameter.
//
//*****************************************************************************
typedef enum
{
    //! TZ1 - TZ6, DCBEVT2, DCBEVT1 while counting down
    EPWM_TZ_ADV_ACTION_EVENT_TZB_D = 9,
    //! TZ1 - TZ6, DCBEVT2, DCBEVT1 while counting up
    EPWM_TZ_ADV_ACTION_EVENT_TZB_U = 6,
    //! TZ1 - TZ6, DCAEVT2, DCAEVT1 while counting down
    EPWM_TZ_ADV_ACTION_EVENT_TZA_D = 3,
    //! TZ1 - TZ6, DCAEVT2, DCAEVT1 while counting up
    EPWM_TZ_ADV_ACTION_EVENT_TZA_U = 0
} EPWM_TripZoneAdvancedEvent;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneAdvDigitalCompareActionA(),
//! EPWM_setTripZoneAdvDigitalCompareActionB(),EPWM_setTripZoneAdvAction()
//! as the \e tzAdvDCAction parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_TZ_ADV_ACTION_HIGH_Z  = 0, //!< High impedance output
    EPWM_TZ_ADV_ACTION_HIGH    = 1, //!< High voltage state
    EPWM_TZ_ADV_ACTION_LOW     = 2, //!< Low voltage state
    EPWM_TZ_ADV_ACTION_TOGGLE  = 3, //!< Toggle the output
    EPWM_TZ_ADV_ACTION_DISABLE = 7  //!< Disable action
} EPWM_TripZoneAdvancedAction;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setTripZoneAdvDigitalCompareActionA() and
//! EPWM_setTripZoneAdvDigitalCompareActionB() as the \e tzAdvDCEvent
//! parameter.
//
//*****************************************************************************
typedef enum
{
    //! Digital Compare event A/B 1 while counting up
    EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_U = 0,
    //! Digital Compare event A/B 1 while counting down
    EPWM_TZ_ADV_ACTION_EVENT_DCxEVT1_D = 3,
    //! Digital Compare event A/B 2 while counting up
    EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_U = 6,
    //! Digital Compare event A/B 2 while counting down
    EPWM_TZ_ADV_ACTION_EVENT_DCxEVT2_D =  9
} EPWM_TripZoneAdvDigitalCompareEvent;


//*****************************************************************************
//
// Values that can be passed to EPWM_enableTripZoneInterrupt()and
// EPWM_disableTripZoneInterrupt() as the tzInterrupt parameter .
//
//*****************************************************************************
//! Trip Zones Cycle By Cycle interrupt
//!
#define EPWM_TZ_INTERRUPT_CBC      0x2U
//! Trip Zones One Shot interrupt
//!
#define EPWM_TZ_INTERRUPT_OST      0x4U
//! Digital Compare A Event 1 interrupt
//!
#define EPWM_TZ_INTERRUPT_DCAEVT1  0x8U
//! Digital Compare A Event 2 interrupt
//!
#define EPWM_TZ_INTERRUPT_DCAEVT2  0x10U
//! Digital Compare B Event 1 interrupt
//!
#define EPWM_TZ_INTERRUPT_DCBEVT1  0x20U
//! Digital Compare B Event 2 interrupt
//!
#define EPWM_TZ_INTERRUPT_DCBEVT2  0x40U
//! Trip Zones Capture Event interrupt
//!
#define EPWM_TZ_INTERRUPT_CAPEVT   0x80U
//*****************************************************************************
//
// Values that can be returned by EPWM_getTripZoneFlagStatus() .
//
//*****************************************************************************
//! Trip Zones Cycle By Cycle flag
//!
#define EPWM_TZ_FLAG_CBC      0x2U
//! Trip Zones One Shot flag
//!
#define EPWM_TZ_FLAG_OST      0x4U
//! Digital Compare A Event 1 flag
//!
#define EPWM_TZ_FLAG_DCAEVT1  0x8U
//! Digital Compare A Event 2 flag
//!
#define EPWM_TZ_FLAG_DCAEVT2  0x10U
//! Digital Compare B Event 1 flag
//!
#define EPWM_TZ_FLAG_DCBEVT1  0x20U
//! Digital Compare B Event 2 flag
//!
#define EPWM_TZ_FLAG_DCBEVT2  0x40U
//! Trip Zones Capture Event flag
//!
#define EPWM_TZ_FLAG_CAPEVT   0x80U
//*****************************************************************************
//
// Value can be passed to EPWM_clearTripZoneFlag() as the
// tzInterrupt parameter and returned by EPWM_getTripZoneFlagStatus().
//
//*****************************************************************************
//! Trip Zone interrupt
//!
#define EPWM_TZ_INTERRUPT     0x1U

//*****************************************************************************
//
// Values that can be passed to EPWM_clearCycleByCycleTripZoneFlag()
// as the tzCbcFlag parameter and returned by
// EPWM_getCycleByCycleTripZoneFlagStatus().
//
//*****************************************************************************
//! CBC flag 1
//!
#define EPWM_TZ_CBC_FLAG_1  0x1U
//! CBC flag 2
//!
#define EPWM_TZ_CBC_FLAG_2  0x2U
//! CBC flag 3
//!
#define EPWM_TZ_CBC_FLAG_3  0x4U
//! CBC flag 4
//!
#define EPWM_TZ_CBC_FLAG_4  0x8U
//! CBC flag 5
//!
#define EPWM_TZ_CBC_FLAG_5  0x10U
//! CBC flag 6
//!
#define EPWM_TZ_CBC_FLAG_6   0x20U
//! CBC flag Digital compare event A2
//!
#define EPWM_TZ_CBC_FLAG_DCAEVT2  0x40U
//! CBC flag Digital compare event B2
//!
#define EPWM_TZ_CBC_FLAG_DCBEVT2  0x80U
//! CBC flag capture event
//!
#define EPWM_TZ_CBC_FLAG_CAPEVT  0x100U


//*****************************************************************************
//
// Values that can be passed to EPWM_clearOneShotTripZoneFlag() as
// the tzCbcFlag parameter and returned by the
// EPWM_getOneShotTripZoneFlagStatus() .
//
//*****************************************************************************
//! OST flag OST1
//!
#define EPWM_TZ_OST_FLAG_OST1  0x1U
//! OST flag OST2
//!
#define EPWM_TZ_OST_FLAG_OST2  0x2U
//! OST flag OST3
//!
#define EPWM_TZ_OST_FLAG_OST3  0x4U
//! OST flag OST4
//!
#define EPWM_TZ_OST_FLAG_OST4  0x8U
//! OST flag OST5
//!
#define EPWM_TZ_OST_FLAG_OST5  0x10U
//! OST flag OST6
//!
#define EPWM_TZ_OST_FLAG_OST6  0x20U
//! OST flag Digital compare event A1
//!
#define EPWM_TZ_OST_FLAG_DCAEVT1   0x40U
//! OST flag Digital compare event B1
//!
#define EPWM_TZ_OST_FLAG_DCBEVT1   0x80U
//! OST flag capture event
//!
#define EPWM_TZ_OST_FLAG_CAPEVT		0x100U

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectCycleByCycleTripZoneClearEvent() as
//! the \e clearMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Clear CBC pulse when counter equals zero
    EPWM_TZ_CBC_PULSE_CLR_CNTR_ZERO = 0,
    //! Clear CBC pulse when counter equals period
    EPWM_TZ_CBC_PULSE_CLR_CNTR_PERIOD = 1,
    //! Clear CBC pulse when counter equals zero or period
    EPWM_TZ_CBC_PULSE_CLR_CNTR_ZERO_PERIOD = 2
} EPWM_CycleByCycleTripZoneClearMode;

//*****************************************************************************
//
// Values that can be passed to EPWM_forceTripZoneEvent() as the
// tzForceEvent parameter.
//
//*****************************************************************************
//! Force Cycle By Cycle trip event
//!
#define EPWM_TZ_FORCE_EVENT_CBC  0x2U
//! Force a One-Shot Trip Event
//!
#define EPWM_TZ_FORCE_EVENT_OST  0x4U
//! Force Digital Compare Output A Event 1
//!
#define EPWM_TZ_FORCE_EVENT_DCAEVT1  0x8U
//! Force Digital Compare Output A Event 2
//!
#define EPWM_TZ_FORCE_EVENT_DCAEVT2  0x10U
//! Force Digital Compare Output B Event 1
//!
#define EPWM_TZ_FORCE_EVENT_DCBEVT1  0x20U
//! Force Digital Compare Output B Event 2
//!
#define EPWM_TZ_FORCE_EVENT_DCBEVT2  0x40U

//*****************************************************************************
//
// Values that can be passed to EPWM_setInterruptSource() as the
// interruptSource parameter.
//
//*****************************************************************************
//! Time-base counter is disabled
//!
#define EPWM_INT_TBCTR_DISABLED 0U
//! Time-base counter equal to zero
//!
#define EPWM_INT_TBCTR_ZERO  1U
//! Time-base counter equal to period
//!
#define EPWM_INT_TBCTR_PERIOD  2U
//! Time-base counter equal to zero or period
//!
#define EPWM_INT_TBCTR_ZERO_OR_PERIOD  3U
//! Time-base counter based on mixed events (ETINTMIX)
//!
#define EPWM_INT_TBCTR_ETINTMIX  3U
//! time-base counter equal to CMPA when the timer is incrementing
//!
#define EPWM_INT_TBCTR_U_CMPA  4U
//! time-base counter equal to CMPC when the timer is incrementing
//!
#define EPWM_INT_TBCTR_U_CMPC  8U
//! time-base counter equal to CMPA when the timer is decrementing
//!
#define EPWM_INT_TBCTR_D_CMPA  5U
//! time-base counter equal to CMPC when the timer is decrementing
//!
#define EPWM_INT_TBCTR_D_CMPC  10U
//! time-base counter equal to CMPB when the timer is incrementing
//!
#define EPWM_INT_TBCTR_U_CMPB  6U
//! time-base counter equal to CMPD when the timer is incrementing
//!
#define EPWM_INT_TBCTR_U_CMPD  12U
//! time-base counter equal to CMPB when the timer is decrementing
//!
#define EPWM_INT_TBCTR_D_CMPB  7U
//! time-base counter equal to CMPD when the timer is decrementing
//!
#define EPWM_INT_TBCTR_D_CMPD  14U
//! Force a Capture Event
//!
#define EPWM_TZ_FORCE_EVENT_CAPEVT  0x80U

//*****************************************************************************
//
//! Values that can be passed to EPWM_enableADCTrigger(),
//! EPWM_disableADCTrigger(),EPWM_setADCTriggerSource(),
//! EPWM_setADCTriggerEventPrescale(),EPWM_getADCTriggerFlagStatus(),
//! EPWM_clearADCTriggerFlag(),EPWM_enableADCTriggerEventCountInit(),
//! EPWM_disableADCTriggerEventCountInit(),EPWM_forceADCTriggerEventCountInit(),
//! EPWM_setADCTriggerEventCountInitValue(),EPWM_getADCTriggerEventCount(),
//! EPWM_forceADCTrigger() as the \e adcSOCType parameter
//
//*****************************************************************************
typedef enum
{
    EPWM_SOC_A = 0,  //!< SOC A
    EPWM_SOC_B = 1   //!< SOC B
} EPWM_ADCStartOfConversionType;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setADCTriggerSource() as the
//! \e socSource parameter.
//
//*****************************************************************************
typedef enum
{
    //! Event is based on DCxEVT1
    EPWM_SOC_DCxEVT1 = 0,
    //! Time-base counter equal to zero
    EPWM_SOC_TBCTR_ZERO = 1,
    //! Time-base counter equal to period
    EPWM_SOC_TBCTR_PERIOD = 2,
    //! Time-base counter equal to zero or period
    EPWM_SOC_TBCTR_ZERO_OR_PERIOD = 3,
    //! Time-base counter based on mixed events (ETSOCAMIX)
    EPWM_SOC_TBCTR_ETSOCAMIX = 3,
    //! Time-base counter equal to CMPA when the timer is incrementing
    EPWM_SOC_TBCTR_U_CMPA = 4,
    //! Time-base counter equal to CMPC when the timer is incrementing
    EPWM_SOC_TBCTR_U_CMPC = 8,
    //! Time-base counter equal to CMPA when the timer is decrementing
    EPWM_SOC_TBCTR_D_CMPA = 5,
    //! Time-base counter equal to CMPC when the timer is decrementing
    EPWM_SOC_TBCTR_D_CMPC = 10,
    //! Time-base counter equal to CMPB when the timer is incrementing
    EPWM_SOC_TBCTR_U_CMPB = 6,
    //! Time-base counter equal to CMPD when the timer is incrementing
    EPWM_SOC_TBCTR_U_CMPD = 12,
    //! Time-base counter equal to CMPB when the timer is decrementing
    EPWM_SOC_TBCTR_D_CMPB = 7,
    //! Time-base counter equal to CMPD when the timer is decrementing
    EPWM_SOC_TBCTR_D_CMPD = 14
} EPWM_ADCStartOfConversionSource;

//
// Digital Compare Module
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_selectDigitalCompareTripInput(),
//! EPWM_enableDigitalCompareTripCombinationInput(),
//! EPWM_disableDigitalCompareTripCombinationInput() as the \e dcType
//! parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_TYPE_DCAH = 0,  //!< Digital Compare A High
    EPWM_DC_TYPE_DCAL = 2,  //!< Digital Compare A Low
    EPWM_DC_TYPE_DCBH = 4,  //!< Digital Compare B High
    EPWM_DC_TYPE_DCBL = 6   //!< Digital Compare B Low
} EPWM_DigitalCompareType;

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectDigitalCompareTripInput()
//! as the \e tripSource parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_TRIP_TRIPIN1 = 0,   //!< Trip 1
    EPWM_DC_TRIP_TRIPIN2 = 1,   //!< Trip 2
    EPWM_DC_TRIP_TRIPIN3 = 2,   //!< Trip 3
    EPWM_DC_TRIP_TRIPIN4 = 3,   //!< Trip 4
    EPWM_DC_TRIP_TRIPIN5 = 4,   //!< Trip 5
    EPWM_DC_TRIP_TRIPIN6 = 5,   //!< Trip 6
    EPWM_DC_TRIP_TRIPIN7 = 6,   //!< Trip 7
    EPWM_DC_TRIP_TRIPIN8 = 7,   //!< Trip 8
    EPWM_DC_TRIP_TRIPIN9 = 8,   //!< Trip 9
    EPWM_DC_TRIP_TRIPIN10 = 9,  //!< Trip 10
    EPWM_DC_TRIP_TRIPIN11 = 10, //!< Trip 11
    EPWM_DC_TRIP_TRIPIN12 = 11, //!< Trip 12
    EPWM_DC_TRIP_TRIPIN14 = 13, //!< Trip 14
    EPWM_DC_TRIP_TRIPIN15 = 14, //!< Trip 15
    EPWM_DC_TRIP_COMBINATION = 15, //!< All Trips (Trip1 - Trip 15) are selected

	//extened trip signals
    EPWM_DC_TRIP_TZ7 = 0,   //!< TZ7
    EPWM_DC_TRIP_TZ8 = 1,   //!< TZ8
    EPWM_DC_TRIP_TZ9 = 2,   //!< TZ9
    EPWM_DC_TRIP_TRIPIN16 = 3,   //!< Trip 16
    EPWM_DC_TRIP_TRIPIN17 = 4,   //!< Trip 17
    EPWM_DC_TRIP_TRIPIN18 = 5,   //!< Trip 18
    EPWM_DC_TRIP_TRIPIN19 = 6,   //!< Trip 19
    EPWM_DC_TRIP_TRIPIN20 = 7,   //!< Trip 20
    EPWM_DC_TRIP_TRIPIN21 = 8,   //!< Trip 21
    EPWM_DC_TRIP_TRIPIN22 = 9,  //!< Trip 22
    EPWM_DC_TRIP_TRIPIN23 = 10, //!< Trip 23
    EPWM_DC_TRIP_TRIPIN24 = 11, //!< Trip 24
    EPWM_DC_TRIP_TRIPIN25 = 12, //!< Trip 25
    EPWM_DC_TRIP_TRIPIN26 = 13, //!< Trip 26
    EPWM_DC_TRIP_TRIPIN27 = 14, //!< Trip 27
    EPWM_DC_TRIP_TRIPIN28 = 15, //!< Trip 28

	//extened trip signals
    EPWM_DC_TRIP_TRIPIN29 = 0,   //!< Trip 29
    EPWM_DC_TRIP_TRIPIN30 = 1,   //!< Trip 30
    EPWM_DC_TRIP_TRIPIN31 = 2,   //!< Trip 31
	EPWM_DC_TRIP_COMBINATION_EX = 3,   //!< All Trips (Trip1 - Trip 15) are selected
	EPWM_DC_TRIP_COMBINATION_NEW1 = 4,   //!< All Trips (Trip1 - Trip 15) are selected
	EPWM_DC_TRIP_COMBINATION_NEW2 = 5,   //!< All Trips (Trip1 - Trip 15) are selected
	EPWM_DC_TRIP_COMBINATION_NEW3 = 6,   //!< All Trips (Trip1 - Trip 15) are selected
} EPWM_DigitalCompareTripInput;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareBlankingEvent() as the
//! the \e blankingPulse parameter.
//
//*****************************************************************************
typedef enum
{
    //! Time base counter equals period
    EPWM_DC_WINDOW_START_TBCTR_PERIOD = 0,
    //! Time base counter equals zero
    EPWM_DC_WINDOW_START_TBCTR_ZERO   = 1,
    //! Time base counter equals zero or period
    EPWM_DC_WINDOW_START_TBCTR_ZERO_PERIOD  = 2
} EPWM_DigitalCompareBlankingPulse;

////*****************************************************************************
////
////! Values that can be passed to EPWM_setDigitalCompareBlankingEvent() as the
////! the \e blankingPulse parameter.
////
////*****************************************************************************
//typedef enum
//{
//    //! Time base counter equals period
//    EPWM_DC_WINDOW_START_TBCTR_PERIOD = 0,
//    //! Time base counter equals zero
//    EPWM_DC_WINDOW_START_TBCTR_ZERO   = 1,
//    //! Time base counter equals zero or period
//    EPWM_DC_WINDOW_START_TBCTR_ZERO_PERIOD  = 2,
//    //! Blank pulse mix
//    EPWM_DC_WINDOW_START_BLANK_PULSE_MIX  = 3
//} EPWM_DigitalCompareBlankingPulse;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareFilterInput()
//! as the \e filterInput parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_WINDOW_SOURCE_DCAEVT1 = 0, //!< DC filter signal source is DCAEVT1
    EPWM_DC_WINDOW_SOURCE_DCAEVT2 = 1, //!< DC filter signal source is DCAEVT2
    EPWM_DC_WINDOW_SOURCE_DCBEVT1 = 2, //!< DC filter signal source is DCBEVT1
    EPWM_DC_WINDOW_SOURCE_DCBEVT2 = 3  //!< DC filter signal source is DCBEVT2
} EPWM_DigitalCompareFilterInput;


//*****************************************************************************
//
//! Values that can be assigned to EPWM_setDigitalCompareEventSource(),
//! EPWM_setDigitalCompareEventSyncMode(),EPWM_enableDigitalCompareSyncEvent()
//! EPWM_enableDigitalCompareADCTrigger(),EPWM_disableDigitalCompareSyncEvent()
//! EPWM_disableDigitalCompareADCTrigger() as the \e dcModule parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_MODULE_A = 0, //!< Digital Compare Module A
    EPWM_DC_MODULE_B = 2  //!< Digital Compare Module B
} EPWM_DigitalCompareModule;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareEventSource(),
//! EPWM_setDigitalCompareEventSyncMode as the \e dcEvent parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_EVENT_1 = 0, //!< Digital Compare Event number 1
    EPWM_DC_EVENT_2 = 1  //!< Digital Compare Event number 2
} EPWM_DigitalCompareEvent;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareEventSource() as the
//! \e dcEventSource parameter.
//
//*****************************************************************************
typedef enum
{
    //! Signal source is unfiltered (DCAEVT1/2)
    EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL = 0,
    //! Signal source is filtered (DCEVTFILT)
    EPWM_DC_EVENT_SOURCE_FILT_SIGNAL = 1
} EPWM_DigitalCompareEventSource;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareEventSyncMode() as the
//! \e syncMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! DC input signal is synced with TBCLK
    EPWM_DC_EVENT_INPUT_SYNCED = 0,
    //! DC input signal is not synced with TBCLK
    EPWM_DC_EVENT_INPUT_NOT_SYNCED = 1
} EPWM_DigitalCompareSyncMode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareCBCLatchMode() as the
//! \e latchMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! DC cycle-by-cycle(CBC) latch is disabled
    EPWM_DC_CBC_LATCH_DISABLED = 0x0,
    //! DC cycle-by-cycle(CBC) latch is enabled
    EPWM_DC_CBC_LATCH_ENABLED  = 0x1
} EPWM_DigitalCompareCBCLatchMode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectDigitalCompareCBCLatchClearEvent()
//! as the \e latchMode parameter.
//
//*****************************************************************************
typedef enum
{
    //! Clear CBC latch when counter equals zero
    EPWM_DC_CBC_LATCH_CLR_CNTR_ZERO           = 0x0,
    //! Clear CBC latch when counter equals period
    EPWM_DC_CBC_LATCH_CLR_ON_CNTR_PERIOD      = 0x1,
    //! Clear CBC latch when counter equals zero or period
    EPWM_DC_CBC_LATCH_CLR_ON_CNTR_ZERO_PERIOD = 0x2
} EPWM_DigitalCompareCBCLatchClearEvent;

//*****************************************************************************
//
//! Values that can be passed to EPWM_configCaptureGateInputPolarity()
//! as the \e polSel parameter.
//
//*****************************************************************************
typedef enum
{
    //! Capture gate is always on
    EPWM_CAPGATE_INPUT_ALWAYS_ON            = 0U,
    //! Capture gate is always off
    EPWM_CAPGATE_INPUT_ALWAYS_OFF           = 1U,
    //! Capture gate input is CAPGATE.sync
    EPWM_CAPGATE_INPUT_SYNC                 = 2U,
    //! Capture gate input is CAPGATE.sync inverted
    EPWM_CAPGATE_INPUT_SYNC_INVERT          = 3U
} EPWM_selectCaptureGateInputPolarity;

//*****************************************************************************
//
//! Values that can be passed to EPWM_invertCaptureInputPolarity()
//! as the \e polSel parameter.
//
//*****************************************************************************
typedef enum
{
    //! Capture input is not inverted
    EPWM_CAPTURE_INPUT_CAPIN_SYNC            = 0U,
    //! Capture input is inverted
    EPWM_CAPTURE_INPUT_CAPIN_SYNC_INVERT     = 1U
} EPWM_selectCaptureInputPolarity;

//*****************************************************************************
//
//! Values that can be passed to EPWM_selectCaptureTripInput(),
//! EPWM_enableCaptureTripCombinationInput(),
//! EPWM_disableCaptureTripCombinationInput()
//! as the \e dcType parameter.
//
//*****************************************************************************
typedef enum
{
    //! Capture Gate
    EPWM_CAPTURE_GATE             = 1U,
    //! Capture Input
    EPWM_CAPTURE_INPUT            = 0U
} EPWM_CaptureInputType;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setGlobalLoadTrigger() as the
//! \e loadTrigger parameter.
//
//*****************************************************************************
typedef enum
{
    //! Load when counter is equal to zero
    EPWM_GL_LOAD_PULSE_CNTR_ZERO = 0x0,
    //! Load when counter is equal to period
    EPWM_GL_LOAD_PULSE_CNTR_PERIOD = 0x1,
    //! Load when counter is equal to zero or period
    EPWM_GL_LOAD_PULSE_CNTR_ZERO_PERIOD = 0x2,
    //! Load on sync event
    EPWM_GL_LOAD_PULSE_SYNC = 0x3,
    //! Load on sync event or when counter  is equal to zero
    EPWM_GL_LOAD_PULSE_SYNC_OR_CNTR_ZERO = 0x4,
    //! Load on sync event or when counter  is equal to period
    EPWM_GL_LOAD_PULSE_SYNC_OR_CNTR_PERIOD = 0x5,
    //! Load on sync event or when counter is equal to period or zero
    EPWM_GL_LOAD_PULSE_SYNC_CNTR_ZERO_PERIOD = 0x6,
    //! Load when counter is equal to CMPC while incrementing
    EPWM_GL_LOAD_PULSE_CNTR_U_CMPC = 0x8,
    //! Load when counter is equal to CMPC while decrementing
    EPWM_GL_LOAD_PULSE_CNTR_D_CMPC = 0x9,
    //! Load when counter is equal to CMPD while incrementing
    EPWM_GL_LOAD_PULSE_CNTR_U_CMPD = 0xA,
    //! Load when counter is equal to CMPD while decrementing
    EPWM_GL_LOAD_PULSE_CNTR_D_CMPD = 0xB,
    //! Load on global force
    EPWM_GL_LOAD_PULSE_GLOBAL_FORCE = 0xF
} EPWM_GlobalLoadTrigger;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setValleyTriggerSource() as the \e
//! trigger parameter.
//
//*****************************************************************************
typedef enum
{
    //! Valley capture trigged by software
    EPWM_VALLEY_TRIGGER_EVENT_SOFTWARE = 0U,
    //! Valley capture trigged by when counter is equal to zero
    EPWM_VALLEY_TRIGGER_EVENT_CNTR_ZERO = 1U,
    //! Valley capture trigged by when counter is equal period
    EPWM_VALLEY_TRIGGER_EVENT_CNTR_PERIOD = 2U,
    //! Valley capture trigged when counter is equal to zero or period
    EPWM_VALLEY_TRIGGER_EVENT_CNTR_ZERO_PERIOD = 3U,
    //! Valley capture trigged by DCAEVT1 (Digital Compare A event 1)
    EPWM_VALLEY_TRIGGER_EVENT_DCAEVT1 = 4U,
    //! Valley capture trigged by DCAEVT2 (Digital Compare A event 2)
    EPWM_VALLEY_TRIGGER_EVENT_DCAEVT2 = 5U,
    //! Valley capture trigged by DCBEVT1 (Digital Compare B event 1)
    EPWM_VALLEY_TRIGGER_EVENT_DCBEVT1 = 6U,
    //! Valley capture trigged by DCBEVT2 (Digital Compare B event 2)
    EPWM_VALLEY_TRIGGER_EVENT_DCBEVT2 = 7U
} EPWM_ValleyTriggerSource;

//*****************************************************************************
//
//! Values that can be passed to EPWM_getValleyCountEdgeStatus() as the \e edge
//! parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_VALLEY_COUNT_START_EDGE = 0, //!< Valley count start edge
    EPWM_VALLEY_COUNT_STOP_EDGE  = 1  //!< Valley count stop edge
} EPWM_ValleyCounterEdge;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setValleyDelayValue() as the \e delayMode
//! parameter.
//
//*****************************************************************************
typedef enum
{
    //! Delay value equals the offset value defines by software
    EPWM_VALLEY_DELAY_MODE_SW_DELAY   = 0U,
    //! Delay value equals the sum of the Hardware counter value and the offset
    //! value defines by software
    EPWM_VALLEY_DELAY_MODE_VCNT_DELAY_SW_DELAY = 1U,
    //! Delay value equals the the Hardware counter shifted by
    //! (1 + the offset value defines by software)
    EPWM_VALLEY_DELAY_MODE_VCNT_DELAY_SHIFT_1_SW_DELAY = 2U,
    //! Delay value equals the the Hardware counter shifted by
    //! (2 + the offset value defines by software)
    EPWM_VALLEY_DELAY_MODE_VCNT_DELAY_SHIFT_2_SW_DELAY = 3U,
    //! Delay value equals the the Hardware counter shifted by
    //! (4 + the offset value defines by software)
    EPWM_VALLEY_DELAY_MODE_VCNT_DELAY_SHIFT_4_SW_DELAY = 4U
} EPWM_ValleyDelayMode;

//
// DC Edge Filter
//
//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareEdgeFilterMode()
//! as the \e edgeMode parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_EDGEFILT_MODE_RISING  = 0, //!< Digital Compare Edge filter low
                                       //!< to high edge mode
    EPWM_DC_EDGEFILT_MODE_FALLING = 1, //!< Digital Compare Edge filter high
                                       //!< to low edge mode
    EPWM_DC_EDGEFILT_MODE_BOTH    = 2  //!< Digital Compare Edge filter both
                                       //!< edges mode
} EPWM_DigitalCompareEdgeFilterMode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setDigitalCompareEdgeFilterEdgeCount()
//! as the \e edgeCount parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_DC_EDGEFILT_EDGECNT_0  = 0, //!< Digital Compare Edge filter edge
                                     //!< count = 0
    EPWM_DC_EDGEFILT_EDGECNT_1  = 1, //!< Digital Compare Edge filter edge
                                     //!< count = 1
    EPWM_DC_EDGEFILT_EDGECNT_2  = 2, //!< Digital Compare Edge filter edge
                                     //!< count = 2
    EPWM_DC_EDGEFILT_EDGECNT_3  = 3, //!< Digital Compare Edge filter edge
                                     //!< count = 3
    EPWM_DC_EDGEFILT_EDGECNT_4  = 4, //!< Digital Compare Edge filter edge
                                     //!< count = 4
    EPWM_DC_EDGEFILT_EDGECNT_5  = 5, //!< Digital Compare Edge filter edge
                                     //!< count = 5
    EPWM_DC_EDGEFILT_EDGECNT_6  = 6, //!< Digital Compare Edge filter edge
                                     //!< count = 6
    EPWM_DC_EDGEFILT_EDGECNT_7  = 7  //!< Digital Compare Edge filter edge
                                     //!< count = 7
} EPWM_DigitalCompareEdgeFilterEdgeCount;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setMixEvtTriggerSource() as the
//! \e interruptSource parameter.
//
//*****************************************************************************
typedef enum
{
    //! Time-base counter equal to zero
    EPWM_ETMIX_TBCTR_ZERO = 0x1,
    //! Time-base counter equal to period
    EPWM_ETMIX_TBCTR_PERIOD = 0x2,
    //! Time-base counter equal to CMPA when the timer is incrementing
    EPWM_ETMIX_TBCTR_U_CMPA = 0x4,
    //! Time-base counter equal to CMPA when the timer is decrementing
    EPWM_ETMIX_TBCTR_D_CMPA = 0x8,
    //! Time-base counter equal to CMPB when the timer is incrementing
    EPWM_ETMIX_TBCTR_U_CMPB = 0x10,
    //! Time-base counter equal to CMPB when the timer is decrementing
    EPWM_ETMIX_TBCTR_D_CMPB = 0x20,
    //! Time-base counter equal to CMPC when the timer is incrementing
    EPWM_ETMIX_TBCTR_U_CMPC = 0x40,
    //! Time-base counter equal to CMPC when the timer is decrementing
    EPWM_ETMIX_TBCTR_D_CMPC = 0x80,
    //! Time-base counter equal to CMPD when the timer is incrementing
    EPWM_ETMIX_TBCTR_U_CMPD = 0x100,
    //! Time-base counter equal to CMPD when the timer is decrementing
    EPWM_ETMIX_TBCTR_D_CMPD = 0x200,
    //! Event is based on DCAEVT1
    EPWM_ETMIX_DCAEVT1 = 0x400,
} EPWM_ETMixTriggerSource;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setMixEvtTriggerSource()
//! and EPWM_clearMixEvtTriggerSource() as the
//! \e etmixSignal parameter.
//
//*****************************************************************************
typedef enum
{
    //! Mixed ET Interrupt signal
    EPWM_ETMIX_INTERRUPT = 0x0U,
    //! Mixed ET SOCA signal
    EPWM_ETMIX_SOCA = 0x4U,
    //! Mixed ET SOCB signal
    EPWM_ETMIX_SOCB = 0x8U
} EPWM_ETMixSignalSelect;

//*****************************************************************************
//
//! Values that can be passed to EPWM_lockRegisters() as the \e registerGroup
//! parameter.
//
//*****************************************************************************
typedef enum
{
    EPWM_REGISTER_GROUP_GLOBAL_LOAD = 0x2,     //!< Global load register group
    EPWM_REGISTER_GROUP_TRIP_ZONE = 0x4,       //!< Trip zone register group
    EPWM_REGISTER_GROUP_TRIP_ZONE_CLEAR = 0x8, //!< Trip zone clear group
    EPWM_REGISTER_GROUP_DIGITAL_COMPARE = 0x10 //!< Digital compare group
} EPWM_LockRegisterGroup;

//*****************************************************************************
//
//! Values that can be passed to EPWM_configureSignal() as the
//! \e signalParams parameter.
//
//*****************************************************************************
typedef struct
{
    float32_t              freqInHz;    //!< Desired Signal Frequency(in Hz)
    float32_t              dutyValA;    //!< Desired ePWMxA Signal Duty
    float32_t              dutyValB;    //!< Desired ePWMxB Signal Duty
    bool                   invertSignalB; //!< Invert ePWMxB Signal if true
    float32_t              sysClkInHz;  //!< SYSCLK Frequency(in Hz)
    EPWM_TimeBaseCountMode tbCtrMode;   //!< Time Base Counter Mode
    EPWM_ClockDivider      tbClkDiv;    //!< Time Base Counter Clock Divider
    EPWM_HSClockDivider    tbHSClkDiv;  //!< Time Base Counter HS Clock Divider

} EPWM_SignalParams;


//*****************************************************************************
//
//! Values that can be passed to EPWM_enablePwmHalt()/EPWM_disablePwmHalt() as the
//! \e haltenslect parameter.
//
//*****************************************************************************
typedef enum
{
    PREEPWM_PWM_LOCK_EN   = 0x0, //!< Used to control the source of TZ5
    PREEPWM_PWM_HALT_EN  = 0x1  //!< Used to control the source of TZ6
} PREPWM_Haltenslect;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setEpwmCfgUpdate() as the
//! \e   parameter.
//
//*****************************************************************************
typedef enum
{
	PREEPWM_CFG_UPDATE   = 0x1, //!< Used to enhance preprocessing shadow loading triggers
    
} PREPWM_Cfgupdate;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setEpwmCfgUpdate() as the
//! \e  parameter.
//
//*****************************************************************************
typedef enum
{
	PREEPWM_CFG_UPDATE_MODE_SHAW   = 0x0,  //!< Used to enhance preprocessing shadow mode selection
    PREEPWM_CFG_UPDATE_MODE_ACTIVE  = 0x1  //!< Used to enhance preprocessing activity pattern selection
} PREPWM_Cfgupdatemode;

//*****************************************************************************
//
//! Values that can be passed to EPWM_enableTripOutSource()and
//! EPWM_disableTripOutSource() as the \e tzTripOut parameter .
//
//*****************************************************************************
typedef enum
{
    EPWM_TZ_TRIPOUT_OST = 0x1, //!< OST as TRIPOUT source
    EPWM_TZ_TRIPOUT_CBC = 0x2, //!< CBC as TRIPOUT source
    EPWM_TZ_TRIPOUT_TZ1 = 0x4, //!< TZ1 as TRIPOUT source
    EPWM_TZ_TRIPOUT_TZ2 = 0x8, //!< TZ2 as TRIPOUT source
    EPWM_TZ_TRIPOUT_TZ3 = 0x10, //!< TZ3 as TRIPOUT source
    EPWM_TZ_TRIPOUT_TZ4 = 0x20, //!< TZ4 as TRIPOUT source
    EPWM_TZ_TRIPOUT_TZ5 = 0x40, //!< TZ5 as TRIPOUT source
    EPWM_TZ_TRIPOUT_TZ6 = 0x80, //!< TZ6 as TRIPOUT source
    EPWM_TZ_TRIPOUT_DCAEVT1 = 0x100, //!< DCAEVT1 as TRIPOUT source
    EPWM_TZ_TRIPOUT_DCAEVT2 = 0x200, //!< DCAEVT2 as TRIPOUT source
    EPWM_TZ_TRIPOUT_DCBEVT1 = 0x400, //!< DCBEVT1 as TRIPOUT source
    EPWM_TZ_TRIPOUT_DCBEVT2 = 0x800, //!< DCBEVT2 as TRIPOUT source
    EPWM_TZ_TRIPOUT_CAPEVT = 0x1000, //!< Capture Event as TRIPOUT source

} EPWM_selectTripOutSource;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setExtendControlSyncEventOutputMode()and
//! EPWM_setExtendControlSyncEventOutputPolarity() as the \e EPWM_ExtendSyncEventOutputChannel parameter .
//
//*****************************************************************************
typedef enum
{
    EPWM_Event_OutputChannel_A = 1, //!< //!<ePWM Channel_A
    EPWM_Event_OutputChannel_B = 5, //!< //!<ePWM Channel_B

} EPWM_SyncEventOutputChannel_Extend;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setExtendControlSyncEventOutputMode()
//! as the \e EPWM_ExtendSyncEventOutputMode parameter .
//
//*****************************************************************************
typedef enum
{
    EPWM_Event_OutputMode_Hardware  = 0, //!<  Hardware Control
    EPWM_Event_OutputMode_Software = 1,  //!<  Software Control

} EPWM_SyncEventOutputMode_Extend;

//*****************************************************************************
//
//! Values that can be passed to EPWM_setExtendControlSyncEventOutputPolarity()
//! as the \e EPWM_ExtendSyncEventOutputMode parameter .
//
//*****************************************************************************
typedef enum
{
    EPWM_Event_OutputPolarity_LOW  = 0, //!< Output LOW
    EPWM_Event_OutputPolarity_HIGH = 1, //!< Output HIGH

} EPWM_SyncEventOutputPolarity_Extend;


//*****************************************************************************
//
//! Values that can be passed to EPWM_setExtendControlSyncEventOutputPolarity()
//! as the \e EPWM_ExtendSyncEventOutputMode parameter .
//
//*****************************************************************************
typedef enum
{
    EPWM_Event_LoadMode_ZERO        = 0, //!< ZERO Load
    EPWM_Event_LoadMode_ZERO_PERIOD = 1, //!< ZERO_PERIOD Load(Up-Down Mode valid)

} EPWM_SyncEventLoadMode_Extend;

//*****************************************************************************
//
//! Values that can be passed to EPWM_enable_ExtendedDigitalCompareTripInput()
//! as the \e EPWM_enable_ExtendedDigitalCompareTripInput parameter.
//
//*****************************************************************************
typedef enum
{
	DCInput_Extend1        = 1, //!< TZ7~9,Trip16~28
	DCInput_Extend2        = 2, //!< Trip29~31,Combination_input, Combination_input_new1~3

} DCInput_Extend;

//*****************************************************************************
//
// Values that can be passed to EPWM_enableDigitalCompareTripCombinationInput(),
// EPWM_disableDigitalCompareTripCombinationInput() as the tripInput
// parameter.
//
//*****************************************************************************
//! Combinational Trip 1 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN1   0x1U
//! Combinational Trip 2 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN2   0x2U
//! Combinational Trip 3 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN3   0x4U
//! Combinational Trip 4 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN4   0x8U
//! Combinational Trip 5 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN5   0x10U
//! Combinational Trip 6 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN6   0x20U
//! Combinational Trip 7 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN7   0x40U
//! Combinational Trip 8 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN8   0x80U
//! Combinational Trip 9 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN9   0x100U
//! Combinational Trip 10 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN10  0x200U
//! Combinational Trip 11 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN11  0x400U
//! Combinational Trip 12 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN12  0x800U
//! Combinational Trip 14 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN14  0x2000U
//! Combinational Trip 15 input
//!
#define EPWM_DC_COMBINATIONAL_TRIPIN15  0x4000U



//*****************************************************************************
//
// Values that can be passed to EPWM_enableGlobalLoadRegisters(),
// EPWM_disableGlobalLoadRegisters() as theloadRegister parameter.
//
//*****************************************************************************
//! Global load TBPRD:TBPRDHR
//!
#define EPWM_GL_REGISTER_TBPRD_TBPRDHR   0x1U
//! Global load CMPA:CMPAHR
//!
#define EPWM_GL_REGISTER_CMPA_CMPAHR     0x2U
//! Global load CMPB:CMPBHR
//!
#define EPWM_GL_REGISTER_CMPB_CMPBHR     0x4U
//! Global load CMPC
//!
#define EPWM_GL_REGISTER_CMPC            0x8U
//! Global load CMPD
//!
#define EPWM_GL_REGISTER_CMPD            0x10U
//! Global load DBRED:DBREDHR
//!
#define EPWM_GL_REGISTER_DBRED_DBREDHR   0x20U
//! Global load DBFED:DBFEDHR
//!
#define EPWM_GL_REGISTER_DBFED_DBFEDHR   0x40U
//! Global load DBCTL
//!
#define EPWM_GL_REGISTER_DBCTL           0x80U
//! Global load AQCTLA/A2
//!
#define EPWM_GL_REGISTER_AQCTLA_AQCTLA2  0x100U
//! Global load AQCTLB/B2
//!
#define EPWM_GL_REGISTER_AQCTLB_AQCTLB2  0x200U
//! Global load AQCSFRC
//!
#define EPWM_GL_REGISTER_AQCSFRC         0x400U

//*****************************************************************************
//
// Functions APIs shared with HRPWM module
//
//*****************************************************************************

//
// Period Control related API
//
//#define EPWM_setSyncPulseSource                 HRPWM_setSyncPulseSource


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

//*****************************************************************************
//
//! Enhanced pattern selection for preprocessing
//!
//
//*****************************************************************************
typedef enum
{
    EPWM_MODE_SPWM_0 ,     //!< Three-phase two-level SPWM
    EPWM_MODE_SPWM_1 ,     //!< Three-phase three-level SPWM
    EPWM_MODE_SVPWM_2 ,     //!< Three-phase two-level SVPWM
    EPWM_MODE_SVPWM_3 ,     //!< Three-phase three-level SVPWM

} EPWM_EnhancedMode;

//*****************************************************************************
//
// Prototypes for the API.
//
//*****************************************************************************

/**
 *  @brief Checks ePWM base address.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function determines if an ePWM module base address is valid.
 *
 *  @return Returns \b true if the base address is valid and \b false
 *  otherwise.
*/
#ifdef DEBUG
__STATIC_INLINE bool EPWM_isBaseValid(uint32_t base)
{
    return(
           (base == EPWM1_BASE)
           || (base == EPWM2_BASE)
           || (base == EPWM3_BASE)
		   || (base == EPWM4_BASE)
		   || (base == EPWM5_BASE)
		   || (base == EPWM6_BASE)
		   || (base == EPWM7_BASE)
		   || (base == EPWM8_BASE)
#ifdef EPWM9_BASE
		   || (base == EPWM9_BASE)
		   || (base == EPWM10_BASE)
		   || (base == EPWM11_BASE)
		   || (base == EPWM12_BASE)
#endif

#ifdef EPWM13_BASE
		   || (base == EPWM13_BASE)
		   || (base == EPWM14_BASE)
		   || (base == EPWM15_BASE)
		   || (base == EPWM16_BASE)
#endif

#ifdef EPWM17_BASE
		   || (base == EPWM17_BASE)
		   || (base == EPWM18_BASE)
#endif
          );
}
#endif


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_DEFINE_H_ */
