/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_link_gld.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_LINK_GLD_H_
#define DEVICE_DRIVERLIB_EPWM_LINK_GLD_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include <stdbool.h>
#include <stdint.h>
#include <string.h>
#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm_define.h"
#include "debug.h"

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

//*****************************************************************************
//
// Functions APIs shared with HRPWM module
//
//*****************************************************************************

//
// Period Control related API
//
#define EPWM_setSyncPulseSource                 HRPWM_setSyncPulseSource

//*****************************************************************************
//
//! Sets the EPWM links.
//!
//! \param base is the base address of the EPWM module.
//! \param epwmLink is the ePWM instance to link with.
//! \param linkComp is the ePWM component to link.
//!
//! This function links the component defined in linkComp in the current ePWM
//! instance with the linkComp component of the ePWM instance defined by
//! epwmLink. A change (a write) in the value of linkComp component of epwmLink
//! instance, causes a change in the current ePWM linkComp component.
//! For example if the current ePWM is ePWM3 and the values of epwmLink and
//! linkComp are EPWM_LINK_WITH_EPWM_1 and EPWM_LINK_COMP_C respectively,
//! then a write to COMPC register in ePWM1, will result in a simultaneous
//! write to COMPC register in ePWM3.
//! Valid values for epwmLink are:
//!   - EPWM_LINK_WITH_EPWM_17  - link current ePWM with ePWM17
//!   - EPWM_LINK_WITH_EPWM_18  - link current ePWM with ePWM18
//!   - EPWM_LINK_WITH_EPWM_1  - link current ePWM with ePWM1
//!   - EPWM_LINK_WITH_EPWM_2  - link current ePWM with ePWM2
//!   - EPWM_LINK_WITH_EPWM_3  - link current ePWM with ePWM3
//!   - EPWM_LINK_WITH_EPWM_4  - link current ePWM with ePWM4
//!   - EPWM_LINK_WITH_EPWM_5  - link current ePWM with ePWM5
//!   - EPWM_LINK_WITH_EPWM_6  - link current ePWM with ePWM6
//!   - EPWM_LINK_WITH_EPWM_7  - link current ePWM with ePWM7
//!   - EPWM_LINK_WITH_EPWM_8  - link current ePWM with ePWM8
//!   - EPWM_LINK_WITH_EPWM_9  - link current ePWM with ePWM9
//!   - EPWM_LINK_WITH_EPWM_10  - link current ePWM with ePWM10
//!   - EPWM_LINK_WITH_EPWM_11  - link current ePWM with ePWM11
//!   - EPWM_LINK_WITH_EPWM_12  - link current ePWM with ePWM12
//!   - EPWM_LINK_WITH_EPWM_13  - link current ePWM with ePWM13
//!   - EPWM_LINK_WITH_EPWM_14  - link current ePWM with ePWM14
//!   - EPWM_LINK_WITH_EPWM_15  - link current ePWM with ePWM15
//!   - EPWM_LINK_WITH_EPWM_16  - link current ePWM with ePWM16
//!
//! Valid values for linkComp are:
//!   - EPWM_LINK_TBPRD   - link TBPRD:TBPRDHR registers
//!   - EPWM_LINK_COMP_A   - link COMPA registers
//!   - EPWM_LINK_COMP_B   - link COMPB registers
//!   - EPWM_LINK_COMP_C   - link COMPC registers
//!   - EPWM_LINK_COMP_D   - link COMPD registers
//!   - EPWM_LINK_GLDCTL2  - link GLDCTL2 registers
//!   - EPWM_LINK_DBRED    - link DBRED registers
//!   - EPWM_LINK_DBFED    - link DBFED registers
//!   - EPWM_LINK_XLOAD    - link XLOAD registers
//!
//! \return None.
//
//*****************************************************************************
static inline void
EPWM_setupEPWMLinks(uint32_t base, EPWM_CurrentLink epwmLink,
					EPWM_LinkComponent linkComp)
{
   //
   // Check the arguments
   //
   ASSERT(EPWM_isBaseValid(base));

   uint32_t registerOffset;
   uint32_t link;
   int8_t Link1Offset[] = {0, 1, 2, 4, 5, 0, 0, -2};
   int8_t Link2Offset[] = {0, 1, 0, 0, 0, 2, 3, 0};
#if (EPWM_VERSION == 23)
   int8_t Link3Offset[] = {0, 1, 0, 4, 0, 0, 0, 0};
#else
   int8_t Link3Offset[] = {0, 1, 2, 3, 0, 0, 0, 0};
#endif
#if (EPWM_VERSION == 30 || EPWM_VERSION == 12)
    memset(Link1Offset,0x00,sizeof(Link1Offset));
    memset(Link2Offset,0x00,sizeof(Link2Offset));
    memset(Link3Offset,0x00,sizeof(Link3Offset));
#endif

   if ( (uint16_t)linkComp > 95U )
   {
       registerOffset = base + EPWM_O_XCMP + EPWM_O_XLINKXLOAD;
       link = (uint32_t)linkComp - 96U;
   }
   else if( (uint16_t)linkComp > 63U )   //link3
   {
       registerOffset = base + EPWM_O_XLINK3;
       link = (uint32_t)linkComp - 64U;
       link += (uint32_t)Link3Offset[link/4];
   }
   else if((uint16_t)linkComp > 31U)
   {
       registerOffset = base + EPWM_O_XLINK2;  //link2
       link = (uint16_t)linkComp - 32U;
       link += (uint32_t)Link2Offset[link/4];
   }else
   {
       registerOffset = base + EPWM_O_XLINK;   //link1
       link = (uint16_t)linkComp;
       link += (uint32_t)Link1Offset[link/4];

   }
   //
   // Configure EPWM links
   //
   HWREG(registerOffset) =
       ((HWREG(registerOffset) & ~((uint32_t)EPWM_XLINK_TBPRDLINK_M << link)) |
       ((uint32_t)epwmLink << link));

}

/**
 * @brief Enable Global shadow load mode.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables Global shadow to active load mode of registers.
 *  The trigger source for loading shadow to active is determined by
 *  EPWM_setGlobalLoadTrigger() function.
 */
__STATIC_INLINE void
EPWM_enableGlobalLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Shadow to active load is controlled globally
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL) |= EPWM_GLDCTL_GLD;
    EDIS;
}

/**
 * @brief Disable Global shadow load mode.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function disables Global shadow to active load mode of registers.
 *  Loading shadow to active is determined individually.
 */
__STATIC_INLINE void
EPWM_disableGlobalLoad(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Shadow to active load is controlled individually
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL) &= ~EPWM_GLDCTL_GLD;
    EDIS;
}

/**
 * @brief Set the Global shadow load pulse.
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  loadTrigger is the pulse that causes global shadow load.
 * @remarks
 *  This function sets the pulse that causes Global shadow to active load.
 *  Valid values for the loadTrigger parameter are:
 *
 *    - EPWM_GL_LOAD_PULSE_CNTR_ZERO              - load when counter is equal
 *                                                  to zero
 *    - EPWM_GL_LOAD_PULSE_CNTR_PERIOD            - load when counter is equal
 *                                                  to period
 *    - EPWM_GL_LOAD_PULSE_CNTR_ZERO_PERIOD       - load when counter is equal
 *                                                  to zero or period
 *    - EPWM_GL_LOAD_PULSE_SYNC                    - load on sync event
 *    - EPWM_GL_LOAD_PULSE_SYNC_OR_CNTR_ZERO      - load on sync event or when
 *                                                  counter is equal to zero
 *    - EPWM_GL_LOAD_PULSE_SYNC_OR_CNTR_PERIOD    - load on sync event or when
 *                                                  counter is equal to period
 *    - EPWM_GL_LOAD_PULSE_SYNC_CNTR_ZERO_PERIOD  - load on sync event or when
 *                                                  counter is equal to period
 *                                                  or zero
 *    - EPWM_GL_LOAD_PULSE_GLOBAL_FORCE            - load on global force
 */
__STATIC_INLINE void
EPWM_setGlobalLoadTrigger(uint32_t base, EPWM_GlobalLoadTrigger loadTrigger)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set the Global shadow to active load pulse
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL) =
               ((HWREGH(base + EPWM_O_GLDCTL) & ~EPWM_GLDCTL_GLDMODE_M) |
                ((uint16_t)loadTrigger << EPWM_GLDCTL_GLDMODE_S));
    EDIS;
}

/**
 * @brief Set the number of Global load pulse event counts
 * @param [in]  base is the base address of the EPWM module.
 * @param [in]  prescalePulseCount is the pulse event counts.
 * @remarks
 *  This function sets the number of Global Load pulse events that have to
 *  occurred before a global load pulse is issued. Valid values for
 *  prescaleCount range from 0 to 7. 0 being no event (disables counter), and 7
 *  representing 7 events.
 */
__STATIC_INLINE void
EPWM_setGlobalLoadEventPrescale(uint32_t base, uint16_t prescalePulseCount)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT(prescalePulseCount < 8U);

    //
    // Set the number of counts that have to occur before
    // a load strobe is issued
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL) =
                ((HWREGH(base + EPWM_O_GLDCTL) & ~EPWM_GLDCTL_GLDPRD_M) |
                 (prescalePulseCount << EPWM_GLDCTL_GLDPRD_S));
    EDIS;
}

/**
 * @brief Return the number of Global load pulse event counts
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function returns the number of Global Load pulse events that have
 *  occurred. These pulse events are set by the EPWM_setGlobalLoadTrigger()
 *  function.
 *  @return the number of Global Load pulse events that have occurred
 */
__STATIC_INLINE uint16_t
EPWM_getGlobalLoadEventCount(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Return the number of events that have occurred
    //
    return((HWREGH(base + EPWM_O_GLDCTL) >> EPWM_GLDCTL_GLDCNT_S) & 0x7U);
}

/**
 * @brief Enable continuous global shadow to active load.
 * @param [in]  base is the base address of the EPWM module.
 * @remarks
 *  This function enables global continuous shadow to active load. Register
 *  load happens every time the event set by the
 *  EPWM_setGlobalLoadTrigger() occurs.
 */
__STATIC_INLINE void
EPWM_disableGlobalLoadOneShotMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable global continuous shadow to active load
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL) &= ~EPWM_GLDCTL_OSHTMODE;
    EDIS;
}

/**
 *  @brief Enable One shot global shadow to active load.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function enables a one time global shadow to active load. Register
 *  load happens every time the event set by the
 *  EPWM_setGlobalLoadTrigger() occurs.
*/
__STATIC_INLINE void
EPWM_enableGlobalLoadOneShotMode(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Enable global continuous shadow to active load
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL) |= EPWM_GLDCTL_OSHTMODE;
    EDIS;
}

/**
 *  @brief Set One shot global shadow to active load pulse.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function sets a one time global shadow to active load pulse. The pulse
 *  propagates to generate a load signal if any of the events set by
 *  EPWM_setGlobalLoadTrigger() occur.
*/
__STATIC_INLINE void
EPWM_setGlobalLoadOneShotLatch(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Set a one shot Global shadow load pulse.
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL2) |= EPWM_GLDCTL2_OSHTLD;
    EDIS;
}

/**
 *  @brief Force a software One shot global shadow to active load pulse.
 *  @param [in] base is the base address of the EPWM module.
 *  @remarks
 *  This function forces a software a one time global shadow to active load
 *  pulse.
*/
__STATIC_INLINE void
EPWM_forceGlobalLoadOneShotEvent(uint32_t base)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));

    //
    // Force a Software Global shadow load pulse
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCTL2) |= EPWM_GLDCTL2_GFRCLD;
    EDIS;
}

/**
 *  @brief Enable a register to be loaded Globally.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] oadRegister is the register.
 *  @remarks
 *  This function enables the register specified by loadRegister to be globally
 *  loaded.
 *  Valid values for loadRegister are:
 *   - EPWM_GL_REGISTER_TBPRD_TBPRDHR  - Register TBPRD:TBPRDHR
 *   - EPWM_GL_REGISTER_CMPA_CMPAHR    - Register CMPA:CMPAHR
 *   - EPWM_GL_REGISTER_CMPB_CMPBHR    - Register CMPB:CMPBHR
 *   - EPWM_GL_REGISTER_CMPC           - Register CMPC
 *   - EPWM_GL_REGISTER_CMPD           - Register CMPD
 *   - EPWM_GL_REGISTER_DBRED_DBREDHR  - Register DBRED:DBREDHR
 *   - EPWM_GL_REGISTER_DBFED_DBFEDHR  - Register DBFED:DBFEDHR
 *   - EPWM_GL_REGISTER_DBCTL          - Register DBCTL
 *   - EPWM_GL_REGISTER_AQCTLA_AQCTLA2 - Register AQCTLA/A2
 *   - EPWM_GL_REGISTER_AQCTLB_AQCTLB2 - Register AQCTLB/B2
 *   - EPWM_GL_REGISTER_AQCSFRC        - Register AQCSFRC
*/
__STATIC_INLINE void
EPWM_enableGlobalLoadRegisters(uint32_t base, uint16_t loadRegister)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((loadRegister > 0x0000U) && (loadRegister < 0x0800U));

    //
    // The register specified by loadRegister is loaded globally
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCFG) |= loadRegister;
    EDIS;
}

/**
 *  @brief Disable a register to be loaded Globally.
 *  @param [in] base is the base address of the EPWM module.
 *  @param [in] oadRegister is the register.
 *  @remarks
 *  This function disables the register specified by loadRegister from being
 *  loaded globally. The shadow to active load happens as specified by the
 *  register control
 *  Valid values for loadRegister are:
 *   - EPWM_GL_REGISTER_TBPRD_TBPRDHR  - Register TBPRD:TBPRDHR
 *   - EPWM_GL_REGISTER_CMPA_CMPAHR    - Register CMPA:CMPAHR
 *   - EPWM_GL_REGISTER_CMPB_CMPBHR    - Register CMPB:CMPBHR
 *   - EPWM_GL_REGISTER_CMPC           - Register CMPC
 *   - EPWM_GL_REGISTER_CMPD           - Register CMPD
 *   - EPWM_GL_REGISTER_DBRED_DBREDHR  - Register DBRED:DBREDHR
 *   - EPWM_GL_REGISTER_DBFED_DBFEDHR  - Register DBFED:DBFEDHR
 *   - EPWM_GL_REGISTER_DBCTL          - Register DBCTL
 *   - EPWM_GL_REGISTER_AQCTLA_AQCTLA2 - Register AQCTLA/A2
 *   - EPWM_GL_REGISTER_AQCTLB_AQCTLB2 - Register AQCTLB/B2
 *   - EPWM_GL_REGISTER_AQCSFRC        - Register AQCSFRC
*/
__STATIC_INLINE void
EPWM_disableGlobalLoadRegisters(uint32_t base, uint16_t loadRegister)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(base));
    ASSERT((loadRegister > 0x0000U) && (loadRegister < 0x0800U));

    //
    // The register specified by loadRegister is loaded by individual
    // register configuration setting
    //
    EALLOW;
    HWREGH(base + EPWM_O_GLDCFG) &= ~loadRegister;
    EDIS;
}


#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_LINK_GLD_H_ */
