/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    epwm_top.h
*   @brief   
*
*   commit history
*   20240804, Jason, initial creation
*/

#ifndef DEVICE_DRIVERLIB_EPWM_TOP_H_
#define DEVICE_DRIVERLIB_EPWM_TOP_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_version.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "inc/hw_epwm.h"
#include "inc/hw_sysctl_ahb.h"
#include "epwm_define.h"

#include "debug.h"



#if (EPWM_VERSION == 0)
/**
 *  @brief Three-phase two-level/three-phase three-level SPWM enhanced duty cycle configuration
 *  @param [in] prebase Base address and Duty value
 *  @remarks
 *  Set the value of duty cycle ratio DutyA, DutyB, and DutyC for three-phase two-level
*/
__STATIC_INLINE void
ESPWM_setSpwmUpdateDuty(uint32_t prebase,float32_t dutyA,float32_t dutyB,float32_t dutyC)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(prebase));

    //
    // Force a Software Global shadow load pulse
    //
    //EALLOW;
    //HWREGH(base + EPWM_O_GLDCTL2) |= EPWM_GLDCTL2_GFRCLD;
    //EDIS;
	  HWREG(prebase + EPWMTOP_O_EPWMA_DUTY) = dutyA;
	  HWREG(prebase + EPWMTOP_O_EPWMB_DUTY) = dutyB;	  
	  HWREG(prebase + EPWMTOP_O_EPWMC_DUTY) = dutyC;	
}

/**
 *  @brief Three-phase two-level/three-phase three-level SPWM enhancement cycle configuration
 *  @param [in] prebase Base address and Duty value
 *  @remarks
 *  Set the values of the duty cycle ratio EPWMA_PERIOD, EPWMB_PERIOD
 *  and EPWMC_PERIOD for three-phase two-level
*/
__STATIC_INLINE void
ESPWM_setSpwmUpdatePeriod(uint32_t prebase,float32_t period0,float32_t period1,float32_t period2)
{
	  ASSERT(EPWM_isBaseValid(prebase));
	  HWREG(prebase + EPWMTOP_O_EPWMA_PERIOD) = period0;
	  HWREG(prebase + EPWMTOP_O_EPWMB_PERIOD) = period1;	  
	  HWREG(prebase + EPWMTOP_O_EPWMC_PERIOD) = period2;	


}

/**
 *  @brief Three-phase two-level/three-phase three-level SVPWM enhancement period configuration
 *  @param [in] prebase Base address and time value
 *  @remarks
 *  Set the value time0, time1, and time2 of three-phase two-level SVPWM
*/
__STATIC_INLINE void
ESPWM_setSvpwmUpdateTimer(uint32_t prebase,float32_t epwmt0,float32_t epwmt1,float32_t epwmt2)
{
    //
    // Check the arguments
    //
    ASSERT(EPWM_isBaseValid(prebase));

    //
    // Force a Software Global shadow load pulse
    //
    //EALLOW;
    //HWREGH(base + EPWM_O_GLDCTL2) |= EPWM_GLDCTL2_GFRCLD;
    //EDIS;
	  HWREG(prebase + EPWMTOP_O_EPWM_0T) = epwmt0;
	  HWREG(prebase + EPWMTOP_O_EPWM_1T) = epwmt1;	  
	  HWREG(prebase + EPWMTOP_O_EPWM_2T) = epwmt2;	
}

/**
 *  @brief Three-phase two-level/three-phase three-level SVPWM enhanced slope configuration
 *  @param [in] prebase Base address and time value
 *  @remarks
 *  Set the slope values of EPWM_0K, EPWM_1K, and EPWM_2K for the three-phase two-level
*/
__STATIC_INLINE void
ESPWM_setSvpwmUpdateSlope(uint32_t prebase,uint32_t epwmk0,uint32_t epwmk1,uint32_t epwmk2)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_0K) = epwmk0;
	  HWREG(prebase + EPWMTOP_O_EPWM_1K) = epwmk1;	  
	  HWREG(prebase + EPWMTOP_O_EPWM_2K) = epwmk2;	


}

/**
 *  @brief SPWM enhanced and SVPWM enhanced static mapping 0 Configuration
 *  @param [in] prebaseBase address and EPWM_CFG_0SEL value
 *  @remarks
 *  Set the static mapping value: EPWM1A_CFG_SEL, EPWM1B_CFG_SEL, EPWM2A_CFG_SEL, EPWM2B_CFG_SEL
 *  EPWM3B_CFG_SEL, EPWM4A_CFG_SEL, EPWM4B_CFG_SEL, EPWM5A_CFG_SEL, EPWM5B_CFG_SEL
 *  EPWM6A_CFG_SEL, EPWM6B_CFG_SEL
*/
__STATIC_INLINE void
EPWM_set3Phase2LevelStaicMapping0(uint32_t prebase,uint32_t epwmCfgSel)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_CFG_0SEL) |= epwmCfgSel;
	
}

/**
 *  @brief SPWM enhanced and SVPWM enhanced static mapping 1 Configure static mapping
 *  @param [in] prebase Base address and EPWM_CFG_0SEL value
 *  @remarks
 *  Set the static mapping value: EPWM7A_CFG_SEL, EPWM7B_CFG_SEL, EPWM8A_CFG_SEL, EPWM8B_CFG_SEL, EPWM9A_CFG_SEL
 *  EPWM9B_CFG_SEL, EPWM10A_CFG_SEL, EPWM10B_CFG_SEL, EPWM11A_CFG_SEL, EPWM11B_CFG_SEL
 *  EPWM12A_CFG_SEL, EPWM12B_CFG_SEL
*/
__STATIC_INLINE void
EPWM_set3Phase2LevelStaicMapping1(uint32_t prebase,uint32_t epwmCfgSel)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_CFG_1SEL) |= epwmCfgSel;
	
}

/**
 *  @brief Set the enhanced preprocessing mode
 *  @param [in] prebase Base address and epwmmode value
 *  @remarks
 *  Setting mode selection: 0: three-phase two-level SPWM
 *  1: three-phase three-level SPWM
 *  2: three-phase two-level SVPWM
 *  3: three-phase three-level SVPWM
*/
__STATIC_INLINE void
EPWM_setPreEpwmMode(uint32_t prebase,uint32_t epwmmode)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_MODE) = epwmmode;
	
}

/**
 *  @brief Set up the enhanced preprocessing MUX configuration
 *  @param [in] prebase Base address and epwmmux value
 *  @remarks
 *  Set the MUX configuration:  0x1<<0: epwm1_mux
 *                0x1<<1: epwm2_mux
 *                0x1<<2: epwm3_mux
 *                0x1<<3: epwm4_mux
 *                0x1<<4: epwm5_mux
 *                0x1<<5: epwm6_mux
 *                0x1<<6: epwm7_mux
 *                0x1<<7: epwm8_mux
 *                0x1<<8: epwm9_mux
 *                0x1<<9: epwm10_mux
 *                0x1<<10: epwm11_mux
 *                0x1<<11: epwm12_mux
*/
__STATIC_INLINE void
EPWM_setPreEpwmMux(uint32_t prebase,uint32_t epwmmux)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_MUX_SEL) |= epwmmux;
	
}

/**
 *  @brief Set the three-phase three-level SPWM static mapping
 *  @param [in] Base address and phase static mapping values
 *  @remarks
 *  epwm_cfg0_sel :Configure ABC three-phase for each pwm channel
 *  Set the MUX configuration:  0x1: A
 *                              0x2: B
 *                              0x3: C
*/
__STATIC_INLINE void
EPWM_setEpwm_cfg0_sel(uint32_t base,uint32_t epwmcfgsel)
{
	
	  HWREG(base + EPWMTOP_O_EPWM_CFG_0SEL) |= epwmcfgsel;
	
}

/**
 *  @brief Set the three-phase three-level SPWM static mapping
 *  @param [in] Base address and phase static mapping values
 *  @remarks
 *  epwm_cfg1_sel :Configure ABC three-phase for each pwm channel
 *  Set the MUX configuration:  0x1: A
 *                              0x2: B
 *                              0x3: C
*/
__STATIC_INLINE void
EPWM_setEpwm_cfg1_sel(uint32_t base,uint32_t epwmcfgsel)
{
	
	  HWREG(base + EPWMTOP_O_EPWM_CFG_1SEL) |= epwmcfgsel;
	
}

/**
 *  @brief Set whether the output channel of EPWM is enabled
 *  @param [in] Base address and phase static mapping value
 *  @remarks
 *  EPWM_DISABLE_00CFG : epwm1a :  (0x1: A, 0x2: B, 0x3: C)
 *                       epwm1b :  (0x1: A, 0x2: B, 0x3: C)
 *  Set the MUX configuration:  0x1: A
 *                              0x2: B
 *                              0x3: C
*/
__STATIC_INLINE void
EPWM_setEpwm_disable_00cfg(uint32_t base,uint32_t epwmcfg)
{
	
	  HWREG(base + EPWMTOP_O_EPWM_DISABLE_00CFG) |= epwmcfg;
	
}

/**
 *  @brief Set whether the output channel of EPWM is enabled
 *  @param [in] Base address and phase static mapping value
 *  @remarks
 *  EPWM_DISABLE_01CFG : epwm1a : (0x1: A, 0x2: B, 0x3: C)
 *                       epwm1b : (0x1: A, 0x2: B, 0x3: C)
 *  Set the MUX configuration:  0x1: A
 *                              0x2: B
 *                              0x3: C
*/
__STATIC_INLINE void
EPWM_setEpwm_disable_01cfg(uint32_t base,uint32_t epwmcfg)
{
	
	  HWREG(base + EPWMTOP_O_EPWM_DISABLE_01CFG) |= epwmcfg;
	
}

/**
 *  @brief Set whether the output channel of EPWM is enabled
 *  @param [in] Base address and phase static mapping value
 *  @remarks
 *  EPWM_DISABLE_10CFG : epwm1a :  (0x1: A, 0x2: B, 0x3: C)
 *                       epwm1b :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration:  0x1: A
 *                           0x2: B
 *                           0x3: C
*/
__STATIC_INLINE void
EPWM_setEpwm_disable_10cfg(uint32_t base,uint32_t epwmcfg)
{
	
	  HWREG(base + EPWMTOP_O_EPWM_DISABLE_10CFG) |= epwmcfg;
	
}

/**
 *  @brief Set whether the output channel of EPWM is enabled
 *  @param [in] Base address and phase static mapping value
 *  @remarks
 *  EPWM_DISABLE_11CFG : epwmxa :  (0x1: A, 0x2: B, 0x3: C)
 *                       epwmxb :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setEpwm_disable_11cfg(uint32_t base,uint32_t epwmcfg)
{
	
	  HWREG(base + EPWMTOP_O_EPWM_DISABLE_11CFG) |= epwmcfg;
	
}

/**
 *  @brief Set a three-phase two-level SVPWM seven-segment A0 group channel
 *  @param [in] base address and SVPWM seven-stage phase setting
 *  @remarks
 *  SVPWM_CMPA_0SEL : svpwm_n0_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n1_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setTwoLevelSvpwm_cmpa0_sel(uint32_t base,uint32_t svpwmcmpasel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPA_0SEL) |= svpwmcmpasel;
	
}

/**
 *  @brief Set a three-phase two-level SVPWM seven-segment Group B0 channel
 *  @param [in] Base address and SVPWM seven-stage phase setup
 *  @remarks
 *  SVPWM_CMPB_0SEL : svpwm_n0_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n1_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setTwoLevelSvpwm_cmpb0_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPB_0SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set up three-phase two-level SVPWM seven-segment C0 group channel
 *  @param [in] Base address and SVPWM seven-stage phase setup
 *  @remarks
 *  SVPWM_CMPC_0SEL : svpwm_n0_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n1_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration:  0x1: A
 *                           0x2: B
 *                           0x3: C
*/
__STATIC_INLINE void
EPWM_setTwoLevelSvpwm_cmpc0_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPC_0SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set a three-phase three-level SVPWM seven-segment A0 group channel
 *  @param [in] Base  address and SVPWM seven-stage phase setting
 *  @remarks
 *  SVPWM_CMPA_0SEL : svpwm_n1n0_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n1n1_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                0x2: B
 *                0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpa0_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPA_0SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set up three-phase three-level SVPWM seven-segment A1 group channel
 *  @param [in] Base address and SVPWM seven-stage phase setup
 *  @remarks
 *  SVPWM_CMPA_1SEL : svpwm_n2n0_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n2n1_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpa1_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPA_1SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set a three-phase three-level SVPWM seven-segment A2 group channel
 *  @param [in] Base address and SVPWM seven-stage phase setting
 *  @remarks
 *  SVPWM_CMPA_2SEL : svpwm_n4n0_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n4n1_cmpa_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpa2_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPA_2SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set a three-phase three-level SVPWM seven-segment Group B0 channel
 *  @param [in] Base address and SVPWM seven-stage phase setup
 *  @remarks
 *  SVPWM_CMPB_0SEL : svpwm_n1n0_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n1n1_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpb0_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPB_0SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set a three-phase three-level SVPWM seven-segment Group B0 channel
 *  @param [in] Base address and SVPWM seven-stage phase setup
 *  @remarks
 *  SVPWM_CMPB_1SEL : svpwm_n2n0_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n2n1_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpb1_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPB_1SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set the three-phase three-level SVPWM seven-segment B2 group channel
 *  @param [in] Base address and SVPWM seven-stage phase setup
 *  @remarks
 *  SVPWM_CMPB_2SEL : svpwm_n4n0_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n4n1_cmpb_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpb2_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPB_2SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set up three-phase three-level SVPWM seven-segment C0 group channel
 *  @param [in] Base address and SVPWM seven-stage phase setting
 *  @remarks
 *  SVPWM_CMPC_0SEL : svpwm_n1n0_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n1n1_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpc0_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPC_0SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set a three-phase three-level SVPWM seven-segment C1 channel
 *  @param [in] Base address and SVPWM seven-stage phase setting
 *  @remarks
 *  SVPWM_CMPC_1SEL : svpwm_n2n0_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n2n1_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpc1_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPC_1SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set the three-phase three-level SVPWM seven-segment C2 channel
 *  @param [in] Base address and SVPWM seven-stage phase setting
 *  @remarks
 *  SVPWM_CMPC_1SEL : svpwm_n4n0_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *                    svpwm_n4n1_cmpc_sel :  (0x1: A, 0x2: B, 0x3: C)
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_cmpc2_sel(uint32_t base,uint32_t svpwmcmpbsel)
{
	
	  HWREG(base + EPWMTOP_O_SVPWM_CMPC_2SEL) |= svpwmcmpbsel;
	
}

/**
 *  @brief Set the three-phase 3-level SVPWM PO/ON mapping table
 *  @param [in] Base address and three-phase three-level SVPWM PO/ON mapping configuration
 *  @remarks
 *  SVPWM_DISABLEA_0CFG : 0x00 ON
 *                        0x01 OFF
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_disablea_0cfg(uint32_t base,uint32_t svpwmdisablecfg)
{
 	
	  HWREG(base + EPWMTOP_O_SVPWM_DISABLEA_0CFG) |= svpwmdisablecfg;
	
}

/**
 *  @brief Set the three-phase 3-level SVPWM PO/ON mapping table
 *  @param [in] Base address and three-phase three-level SVPWM PO/ON mapping configuration
 *  @remarks
 *  SVPWM_DISABLEA_1CFG : 0x00 ON
 *                        0x01 OFF
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_disablea_1cfg(uint32_t base,uint32_t svpwmdisablecfg)
{
 	
	  HWREG(base + EPWMTOP_O_SVPWM_DISABLEA_1CFG) |= svpwmdisablecfg;
	
}

/**
 *  @brief Set the three-phase 3-level SVPWM PO/ON mapping table
 *  @param [in] Base address and three-phase three-level SVPWM PO/ON mapping configuration
 *  @remarks
 *  SVPWM_DISABLEB_0CFG : 0x00 ON
 *                        0x01 OFF
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_disableb_0cfg(uint32_t base,uint32_t svpwmdisablecfg)
{
 	
	  HWREG(base + EPWMTOP_O_SVPWM_DISABLEB_0CFG) |= svpwmdisablecfg;
	
}

/**
 *  @brief Set the three-phase 3-level SVPWM PO/ON mapping table
 *  @param [in] Base address and three-phase three-level SVPWM PO/ON mapping configuration
 *  @remarks
 *  SVPWM_DISABLEB_1CFG : 0x00 ON
 *                        0x01 OFF
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_disableb_1cfg(uint32_t base,uint32_t svpwmdisablecfg)
{
 	
	  HWREG(base + EPWMTOP_O_SVPWM_DISABLEB_1CFG) |= svpwmdisablecfg;
 	
}

/**
 *  @brief Set the three-phase 3-level SVPWM PO/ON mapping table
 *  @param [in] Base address and three-phase three-level SVPWM PO/ON mapping configuration
 *  @remarks
 *  SVPWM_DISABLEB_0CFG : 0x00 ON
 *                        0x01 OFF
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_disablec_0cfg(uint32_t base,uint32_t svpwmdisablecfg)
{
 	
	  HWREG(base + EPWMTOP_O_SVPWM_DISABLEC_0CFG) |= svpwmdisablecfg;
	
}

/**
 *  @brief Set the three-phase 3-level SVPWM PO/ON mapping table
 *  @param [in] Base address and three-phase three-level SVPWM PO/ON mapping configuration
 *  @remarks
 *  SVPWM_DISABLEB_1CFG : 0x00 ON
 *                        0x01 OFF
 *  Set each configuration: 0x1: A
 *                          0x2: B
 *                          0x3: C
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwm_disablec_1cfg(uint32_t base,uint32_t svpwmdisablecfg)
{
 	
	  HWREG(base + EPWMTOP_O_SVPWM_DISABLEC_1CFG) |= svpwmdisablecfg;
 	
}

/**
 *  @brief Set the three-phase two-level SVPWM sector
 *  @param [in] prebase Base address and EPWM_CTL value
 *  @remarks
 *  Set mode selection: 0:
 *                      1:
 *                      2:
 *                      3:
*/
__STATIC_INLINE void
EPWM_setTwoLevelSvpwmSectors(uint32_t prebase,uint32_t epwmsector)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_CTL) = epwmsector;
	
}

/**
 *  @brief Set the three-phase three-level SVPWM sector
 *  @param [in] Base address and EPWM_CTL value
 *  @remarks
 *  Set mode selection: 0:
 *  1:
 *  2: Set mode selection
 *  3:
*/
__STATIC_INLINE void
EPWM_setThreeLevelSvpwmSectors(uint32_t prebase,uint32_t epwmsector)
{
	
	  HWREG(prebase + EPWMTOP_O_EPWM_CTL) = epwmsector;
	
}

/**
 *  @brief The PWM TZ5/TZ6 trigger source was enabled
 *  @param [in] prebase Base address and PWM_HALT_EN value
 *  @remarks
 *  Set mode selection:
 *  bit[5] : 1:Indicates TZ5 source enabled      0: Indicates TZ5 source disabled
 *  bit[4] : 1:dll lockflag setting 0:Indicates dlThe PWM TZ5/TZ6 trigger source was enabledl_lockflag unset
 *  bit[1] : 1 Indicates TZ6 source enabled          0: Indicates TZ6 source disabled
 *  bit[0] : 1 Indicates debug stop time setting 0 Indicates debug stop time unset
*/
__STATIC_INLINE void
EPWM_enablePwmHalt(uint32_t prebase,PREPWM_Haltenslect pwmhalten)
{
	
	if(pwmhalten==PREEPWM_PWM_LOCK_EN)
	{
		//Set TZ6 as the trigger enable
		HWREG(prebase + EPWMTOP_O_PWM_HALT_EN) |= PWM_HALT_EN_M<<PWM_HALT_EN_S;
	}
	else
	{
		//Set TZ5 as the trigger enable
		HWREG(prebase + EPWMTOP_O_PWM_HALT_EN) |= PWM_LOCK_EN_M<<PWM_LOCK_EN_S;
	}
	  
	
}

/**
 *  @brief Disable the PWM TZ5/TZ6 trigger source
 *  @param [in] prebase Base address and PWM_HALT_EN value
 *  @remarks
 *  Set mode selection:
 *  bit[5] : 1:Indicates TZ5 source enabled      0: Indicates TZ5 source disabled
 *  bit[4] : 1:Indicates dll lockflag setting 0:Indicates dll_lockflag unset
 *  bit[1] : 1 Indicates TZ6 source enabled          0: Indicates TZ6 source disabled
 *  bit[0] : 1 Indicates debug stop time Indicates 0:Indicates debug stop time unset
*/
__STATIC_INLINE void
EPWM_disablePwmHalt(uint32_t prebase,PREPWM_Haltenslect pwmhalten)
{
	
	if(pwmhalten==PREEPWM_PWM_LOCK_EN)
	{
		//Set TZ6 as triggered disable
		HWREG(prebase + EPWMTOP_O_PWM_HALT_EN) &= ~(PWM_HALT_EN_M<<PWM_HALT_EN_S);
	}
	else
	{
		//Set TZ5 as triggered disable
		HWREG(prebase + EPWMTOP_O_PWM_HALT_EN) &= ~(PWM_LOCK_EN_M<<PWM_LOCK_EN_S);
	}
	
}

/**
 *  @brief Set enhanced and optimized EPWM update loading mode
 *  @param [in] prebase Base address and EPWM CFG UPDATE value
 *  @remarks
 *  Set mode selection:
 *  bit[0] : 0: Indicates that register write is disabled
 * 			 1: Indicates that when bit[4] is 0, it is written to 1 and loaded immediately
 *  bit[4] : 0: Indicates that bits are allowed to load immediately
 * 			 1: Indicates that the register value takes effect immediately and is equivalent
 * 			 to the active register function
*/
__STATIC_INLINE void
EPWM_setEpwmCfgUpdate(uint32_t prebase,PREPWM_Cfgupdatemode cfgupdatemode,PREPWM_Cfgupdate cfgupdate)
{
	  //Set enhanced and optimized EPWM update loading mode
	  HWREG(prebase + EPWMTOP_O_EPWM_CFG_UPDATE) |= cfgupdate|(cfgupdatemode<<4);
	
}

/**
 *  @brief Set high precision PWM DLL configuration 0
 *  @param [in] prebaseBase address and EPWMTOP_O_HRPWM_0CFG value
 *  @remarks
 *  Set mode selection:
 *
*/
__STATIC_INLINE void
EPWM_setHrpwmDllCfg0(uint32_t prebase,uint32_t value)
{
	//Set high precision PWM DLL configuration 0
	HWREG(prebase + EPWMTOP_O_HRPWM_0CFG) = value;

}

/**
 *  @brief Set high precision PWM DLL configuration 1
 *  @param [in] prebaseBase address and EPWMTOP_O_HRPWM_1CFG value
 *  @remarks
 *  Set mode selection:
 *
*/
__STATIC_INLINE void
EPWM_setHrpwmDllCfg1(uint32_t prebase,uint32_t value)
{
	//Set high precision PWM DLL configuration 1
	HWREG(prebase + EPWMTOP_O_HRPWM_1CFG) = value;

}

/**
 *  @brief Set EPWMTOP_O_HRPWM_BYPASS configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_HRPWM_BYPASS value
 *  @remarks
 *  Set mode selection:
 *
 *
 * EPWMTOP_O_HRPWM_BYPASS
*/
__STATIC_INLINE void
EPWM_setEpwmtopHrpwmBypass(uint32_t prebase,uint32_t value)
{
	//Set EPWMTOP_O_HRPWM_BYPASS configuration
	HWREG(prebase + EPWMTOP_O_HRPWM_BYPASS) = value;

}
#elif (EPWM_VERSION == 11)

/**
 *  @brief Configures the sync input source for the ePWM and eCAP signals.
 *  @param [in] syncInput is the sync input being configured
 *  @param [in] syncSrc is sync input source selection.
 *  @remarks
 *  This function configures the sync input source for the ePWM and eCAP
 *  modules. The \e syncInput parameter is the sync input being configured. It
 *  should be passed a value of \b SYSCTL_SYNC_IN_XXXX, where XXXX is the ePWM
 *  or eCAP instance the sync signal is entering.
 *
 *  The \e syncSrc parameter is the sync signal selected as the source of the
 *  sync input. It should be passed a value of \b SYSCTL_SYNC_IN_SRC_XXXX,
 *  XXXX is a sync signal coming from an ePWM, eCAP or external sync output.
 *  where For example, a \e syncInput value of \b SYSCTL_SYNC_IN_ECAP1 and a
 *  \e syncSrc value of \b SYSCTL_SYNC_IN_SRC_EPWM1SYNCOUT will make the
 *  EPWM1SYNCOUT signal drive eCAP1's SYNCIN signal.
 *
 *  Note that some \e syncSrc values are only valid for certain values of
 *  \e syncInput. See device technical reference manual for details on
 *  time-base counter synchronization.
*/
static inline void
SysCtl_setSyncInputConfig(SysCtl_SyncInput syncInput,
                          SysCtl_SyncInputSource syncSrc)
{
    uint32_t clearMask;

    //
    // Write the input sync source selection to the appropriate register.
    //
    EALLOW;
    if(syncInput == SYSCTL_SYNC_IN_EPWM1)
    {
        //
        // Note that EXTSYNCIN1 is the only valid option for ePWM1
        //
        HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) &=
            ~SYSCTL_SYNCSELECT_EPWM1SYNCIN_M;
    }
    else
    {
        clearMask = (uint32_t)SYSCTL_SYNCSELECT_SYNCIN_M <<
                    (uint32_t)syncInput;

        HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) =
            (HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) & ~clearMask) |
            ((uint32_t)syncSrc << (uint32_t)syncInput);
    }

    EDIS;
}

/**
 *  @brief Configures the sync output source.
 *  @param [in] syncSrc is sync output source selection.
 *  @remarks
 *  This function configures the sync output source from the ePWM modules. The
 *  \e syncSrc parameter is a value \b SYSCTL_SYNC_OUT_SRC_XXXX, where XXXX is
 *  a sync signal coming from an ePWM such as SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT
*/
static inline void
SysCtl_setSyncOutputConfig(SysCtl_SyncOutputSource syncSrc)
{
    //
    // Write the sync output source selection to the appropriate register.
    //
    EALLOW;
    HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) =
        (HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) &
         ~((uint32_t)SYSCTL_SYNCSELECT_SYNCOUT_M)) |
        ((uint32_t)syncSrc << SYSCTL_SYNCSELECT_SYNCOUT_S);
    EDIS;

}

/**
 *  @brief Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 *  @remarks
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical OR
 *  of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
*/
static inline void
SysCtl_enableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT) |= adcsocSrc;
    EDIS;
}

/**
 *  @brief Disables ePWM SOC signals from driving an external ADCSOC signal.
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 *  @remarks
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical
 *  OR of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
*/
static inline void
SysCtl_disableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT) &= ~adcsocSrc;
    EDIS;
}

#elif (EPWM_VERSION==12)

#elif (EPWM_VERSION==21)

#elif (EPWM_VERSION==22)

#endif



/**
 *  @brief Set high precision TRANS_MODE configuration 1
 *  @param [in] prebaseBase address and EPWMTOP_O_TRANS_MODE value
 *  @remarks
 *  Set mode selection:
*/
__STATIC_INLINE void
EPWM_setTRANSMODE(uint32_t prebase,uint32_t value)
{
	//Set high precision PWM DLL configuration 1
	HWREG(prebase + EPWMTOP_O_TRANS_MODE) |= value;

}

/**
 *  @brief Set high precision PERIOD configuration 1
 *  @param [in] prebaseBase address and EPWMTOP_O_PERIOD value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopPeriod(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_PERIOD configuration 1
	HWREG(prebase + EPWMTOP_O_PERIOD) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_DUTYCYCLE1 configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_DUTYCYCLE1 value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopDutycycle1(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_DUTYCYCLE1 configuration
	HWREG(prebase + EPWMTOP_O_DUTYCYCLE1) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_DUTYCYCLE2 configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_DUTYCYCLE2 value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopDutycycle2(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_DUTYCYCLE2 configuration
	HWREG(prebase + EPWMTOP_O_DUTYCYCLE2) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_DEADZONE_A1 configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_DEADZONE_A1 value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopDeadzeroA1(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_DUTYCYCLE2 configuration
	HWREG(prebase + EPWMTOP_O_DEADZONE_A1) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_DEADZONE_B1 configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_DEADZONE_B1 value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopDeadzeroB1(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_DEADZONE_B1 configuration
	HWREG(prebase + EPWMTOP_O_DEADZONE_B1) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_DEADZONE_A2 configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_DEADZONE_A2 value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopDeadzeroA2(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_DEADZONE_A2 configuration
	HWREG(prebase + EPWMTOP_O_DEADZONE_A2) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_DEADZONE_B2 configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_DEADZONE_B2 value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopDeadzeroB2(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_DEADZONE_B2 configuration
	HWREG(prebase + EPWMTOP_O_DEADZONE_B2) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_PHASESHIFT configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_PHASESHIFT value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopPhaseShift(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_PHASESHIFT configuration
	HWREG(prebase + EPWMTOP_O_PHASESHIFT) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_PHASESHIFT_INI configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_PHASESHIFT_INI value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopPhaseShiftIni(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_PHASESHIFT_INI configuration
	HWREG(prebase + EPWMTOP_O_PHASESHIFT_INI) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_AQCSFRC configuration
 *  @param [in]  prebaseBase address and EPWMTOP_O_AQCSFRC value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopAqcsfrc(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_AQCSFRC configuration
	HWREG(prebase + EPWMTOP_O_AQCSFRC) |= value;

}

//
//end of transmode related APIs
//

/**
 *  @brief Set  EPWMTOP_O_EPWM_MUX_SEL configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_EPWM_MUX_SEL value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopEpwmMuxSel(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_EPWM_MUX_SEL configuration
	HWREG(prebase + EPWMTOP_O_EPWM_MUX_SEL) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_TZ_INV_CFG configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_TZ_INV_CFG value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopEpwmTzInvCfg(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_TZ_INV_CFG configuration
	HWREG(prebase + EPWMTOP_O_TZ_INV_CFG) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_TRIPIN_INV_CFG configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_TRIPIN_INV_CFG value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopTrioinInvCfg(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_TRIPIN_INV_CFG configuration
	HWREG(prebase + EPWMTOP_O_TRIPIN_INV_CFG) |= value;

}

/**
 *  @brief Set  EPWMTOP_O_PWM_HALT_EN configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_PWM_HALT_EN value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setEpwmTopPwmHaltEn(uint32_t prebase,uint32_t value)
{
	//Set high precision EPWMTOP_O_PWM_HALT_EN configuration
	HWREG(prebase + EPWMTOP_O_PWM_HALT_EN) |= value;

}

/**
 *  @brief Set high precision PWM DLL configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_HRPWM_0CFG value
 *  @remarks
 *  Set mode selection:
 *
 *
 *
*/
__STATIC_INLINE void
EPWM_setHrpwmDllCfg0(uint32_t prebase,uint32_t value)
{
	//Set high precision PWM DLL configuration 0
	HWREG(prebase + EPWMTOP_O_HRPWM_0CFG) = value;

}

#if (EPWM_VERSION==11) || (EPWM_VERSION==12)
/**
 *  @brief Set high precision PWM DLL configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_HRPWM_1CFG value
 *  @remarks
 *  Set mode selection:
 *
 *
 * EPWMTOP_O_HRPWM_1CFG
*/
__STATIC_INLINE void
EPWM_setHrpwmDllCfg1(uint32_t prebase,uint32_t value)
{
	//Set high precision PWM DLL configuration 1
	HWREG(prebase + EPWMTOP_O_HRPWM_1CFG) = value;

}

/**
 *  @brief Set EPWMTOP_O_DLL_LOCKFLAG configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_DLL_LOCKFLAG value
 *  @remarks
 *  Set mode selection:
 *
 *
 * EPWMTOP_O_DLL_LOCKFLAG
*/
__STATIC_INLINE void
EPWM_setEpwmtopDllLockflag(uint32_t prebase,uint32_t value)
{
	//Set high precision PWM DLL configuration 1
	HWREG(prebase + EPWMTOP_O_DLL_LOCKFLAG) |= value;

}

/**
 *  @brief Set EPWMTOP_O_HRPWM_BYPASS configuration
 *  @param [in] prebaseBase address and EPWMTOP_O_HRPWM_BYPASS value
 *  @remarks
 *  Set mode selection:
 *
 *
 * EPWMTOP_O_HRPWM_BYPASS
*/
__STATIC_INLINE void
EPWM_setEpwmtopHrpwmBypass(uint32_t prebase,uint32_t value)
{
	//Set EPWMTOP_O_HRPWM_BYPASS configuration
	HWREG(prebase + EPWMTOP_O_HRPWM_BYPASS) = value;

}

#endif


#if (EPWM_VERSION == 21)
/**
 *  @brief Configures the sync input source for the ePWM and eCAP signals.
 *  @param [in] syncInput is the sync input being configured
 *  @param [in] syncSrc is sync input source selection.
 *  @remarks
 *  This function configures the sync input source for the ePWM and eCAP
 *  modules. The \e syncInput parameter is the sync input being configured. It
 *  should be passed a value of \b SYSCTL_SYNC_IN_XXXX, where XXXX is the ePWM
 *  or eCAP instance the sync signal is entering.
 *
 *  The \e syncSrc parameter is the sync signal selected as the source of the
 *  sync input. It should be passed a value of \b SYSCTL_SYNC_IN_SRC_XXXX,
 *  XXXX is a sync signal coming from an ePWM, eCAP or external sync output.
 *  where For example, a \e syncInput value of \b SYSCTL_SYNC_IN_ECAP1 and a
 *  \e syncSrc value of \b SYSCTL_SYNC_IN_SRC_EPWM1SYNCOUT will make the
 *  EPWM1SYNCOUT signal drive eCAP1's SYNCIN signal.
 *
 *  Note that some \e syncSrc values are only valid for certain values of
 *  \e syncInput. See device technical reference manual for details on
 *  time-base counter synchronization.
*/
static inline void
SysCtl_setSyncInputConfig(SysCtl_SyncInput syncInput,
                          SysCtl_SyncInputSource syncSrc)
{
    uint32_t clearMask;

    //
    // Write the input sync source selection to the appropriate register.
    //
    EALLOW;
    clearMask = (uint32_t)SYSCTL_SYNCSELECT_SYNCIN_M << (uint32_t)syncInput;

    HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) =
        (HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) & ~clearMask) |
        ((uint32_t)syncSrc << (uint32_t)syncInput);

    EDIS;
}

/**
 *  @brief Configures the sync output source.
 *  @param [in] syncSrc is sync input source selection.
 *  @remarks
 *  This function configures the sync output source from the ePWM modules. The
 *  \e syncSrc parameter is a value \b SYSCTL_SYNC_OUT_SRC_XXXX, where XXXX is
 *  a sync signal coming from an ePWM such as SYSCTL_SYNC_OUT_SRC_EPWM1SYNCOUT
*/
static inline void
SysCtl_setSyncOutputConfig(SysCtl_SyncOutputSource syncSrc)
{
    //
    // Write the sync output source selection to the appropriate register.
    //
    EALLOW;
    HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) =
        (HWREG(PREEPWM_BASE + EPWMTOP_O_SYNCSELECT) &
         ~((uint32_t)SYSCTL_SYNCSELECT_SYNCOUT_M)) |
        ((uint32_t)syncSrc << SYSCTL_SYNCSELECT_SYNCOUT_S);
    EDIS;

}

/**
 *  @brief Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 *  @remarks
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical OR
 *  of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
*/
static inline void
SysCtl_enableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT) |= adcsocSrc;
    EDIS;
}

/**
 *  @brief Disables ePWM SOC signals from driving an external ADCSOC signal.
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 *  @remarks
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical
 *  OR of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
*/
static inline void
SysCtl_disableExtADCSOCSource(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT) &= ~adcsocSrc;
    EDIS;
}

/**
 *  @brief Enables ePWM SOC signals to drive an external (off-chip) ADCSOC signal.
 *  @param [in] adcsocSrc is a bit field of the selected signals to be enabled
 *  @remarks
 *  This function configures which ePWM SOC signals are enabled as a source for
 *  either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical OR
 *  of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
*/
static inline void
SysCtl_enableExtADCSOCSource1(uint32_t adcsocSrc)
{
    //
    // Set the bits that correspond to signal to be enabled.
    //
    EALLOW;
    //HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT1) |= adcsocSrc;
    HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT) |= adcsocSrc;
    EDIS;
}

/**
 *  @brief Disables ePWM SOC signals from driving an external ADCSOC signal.
 *  @param [in] adcsocSrc is a bit field of the selected signals to be disabled
 *  @remarks
 *  This function configures which ePWM SOC signals are disabled as a source
 *  for either ADCSOCAO or ADCSOCBO. The \e adcsocSrc parameter takes a logical
 *  OR of \b SYSCTL_ADCSOC_SRC_PWMxSOCA/B values that correspond to different
 *  signals.
*/
static inline void
SysCtl_disableExtADCSOCSource1(uint32_t adcsocSrc)
{
    //
    // Clear the bits that correspond to signal to be disabled.
    //
    EALLOW;
    //HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT1) &= ~adcsocSrc;
    HWREG(PREEPWM_BASE + EPWMTOP_O_ADCSOCOUTSELECT) &= ~adcsocSrc;
    EDIS;
}

#endif



#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_EPWM_TOP_H_ */
