/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    hw_esc_ss.h
*   @brief   Definitions for the EtherCAT slave controller registers.
*
*/

#ifndef DEVICE_DRIVERLIB_HW_ESC_SS_H_
#define DEVICE_DRIVERLIB_HW_ESC_SS_H_

#ifdef __cplusplus
extern "C"{
#endif

/**
 * @brief The following are defines for the ESCSS register offsets
 * 
 */
#define ESCSS_O_IPREVNUM            0x0U    /* IP Revision Number */
#define ESCSS_O_INTR_RIS            0x2U    /* EtherCATSS Interrupt Raw Status */
#define ESCSS_O_INTR_MASK           0x4U    /* EtherCATSS Interrupt Mask */
#define ESCSS_O_INTR_MIS            0x6U    /* EtherCATSS Masked Interrupt Status */
#define ESCSS_O_INTR_CLR            0x8U    /* EtherCATSS Interrupt Clear */
#define ESCSS_O_INTR_SET            0xAU    /* EtherCATSS Interrupt Set to emulate */
#define ESCSS_O_LATCH_SEL           0xCU    /* Select for Latch0/1 inputs and LATCHIN input */
#define ESCSS_O_ACCESS_CTRL         0xEU    /* PDI interface access control config. */
#define ESCSS_O_GPIN_DAT            0x10U   /* GPIN data capture for debug & override */
#define ESCSS_O_GPIN_PIPE           0x12U   /* GPIN pipeline select */
#define ESCSS_O_GPIN_GRP_CAP_SEL    0x14U   /* GPIN pipe group capture trigger */
#define ESCSS_O_GPOUT_DAT           0x16U   /* GPOUT data capture for debug & override */
#define ESCSS_O_GPOUT_PIPE          0x18U   /* GPOUT pipeline select */
#define ESCSS_O_GPOUT_GRP_CAP_SEL   0x1AU   /* GPOUT pipe group capture trigger */
#define ESCSS_O_MEM_TEST            0x1CU   /* Memory Test Control */
#define ESCSS_O_RESET_DEST_CONFIG   0x1EU   /* ResetOut impact or destination config */
#define ESCSS_O_SYNC0_CONFIG        0x20U   /* SYNC0 Configuration for various triggers */
#define ESCSS_O_SYNC1_CONFIG        0x22U   /* SYNC1 Configuration for various triggers */

/**
 * @brief The following are defines for the ESCSS CONFIG register offsets
 * 
 */
#define ESCSS_O_CONFIG_LOCK         0x0U    /* EtherCATSS Configuration Lock */
#define ESCSS_O_MISC_IO_CONFIG      0x2U    /* RESET_IN, EEPROM IO connections select */
#define ESCSS_O_PHY_IO_CONFIG       0x4U    /* Control Register of ESCSS */
#define ESCSS_O_SYNC_IO_CONFIG      0x6U    /* SYNC Signals IO configurations */
#define ESCSS_O_LATCH_IO_CONFIG     0x8U    /* LATCH inputs IO pad select */
#define ESCSS_O_GPIN_SEL            0xAU    /* GPIN Select between IO PAD & tieoff */
#define ESCSS_O_GPIN_IOPAD_SEL      0xCU    /* GPIN IO pad Select */
#define ESCSS_O_GPOUT_SEL           0xEU    /* GPOUT IO pad connect select */
#define ESCSS_O_GPOUT_IOPAD_SEL     0x10U   /* GPOUT IO pad select */
#define ESCSS_O_LED_CONFIG          0x12U   /* Selection of LED o/p connect to IO pad */
#define ESCSS_O_MISC_CONFIG         0x14U   /* Miscelleneous Configuration */

/**
 * @brief The following are defines for the bit fields in the ESCSS_IPREVNUM register
 * 
 */
#define ESCSS_IPREVNUM_IP_REV_MINOR_S       0U
#define ESCSS_IPREVNUM_IP_REV_MINOR_M       0xFU    /* Minor IP Revision Number */
#define ESCSS_IPREVNUM_IP_REV_MAJOR_S       4U
#define ESCSS_IPREVNUM_IP_REV_MAJOR_M       0xF0U   /* Major IP Revision Number */

/**
 * @brief The following are defines for the bit fields in the ESCSS_INTR_RIS register
 * 
 */
#define ESCSS_INTR_RIS_SYNC0_RIS_S          0x1U    /* SYNC0 feature RIS */
#define ESCSS_INTR_RIS_SYNC1_RIS_S          0x2U    /* SYNC1 feature RIS */
#define ESCSS_INTR_RIS_IRQ_RIS_S            0x4U    /* EtherCATSS IRQ RIS */
#define ESCSS_INTR_RIS_DMA_DONE_RIS_S       0x8U    /* DMA Done RIS */
#define ESCSS_INTR_RIS_TIMEOUT_ERR_RIS_S    0x10U   /* PDI bus Timeout Error RIS */
#define ESCSS_INTR_RIS_MDEVICE_RESET_RIS_S  0x20U   /* ECAT RESET RIS */
#define ESCSS_INTR_RIS_M                    0x3FU   /* ESCSS_INTR_RIS register bit mask */

/**
 * @brief The following are defines for the bit fields in the ESCSS_INTR_MASK register
 * 
 */
#define ESCSS_INTR_MASK_SYNC0_MASK          0x1U    /* SYNC0 feature Mask */
#define ESCSS_INTR_MASK_SYNC1_MASK          0x2U    /* SYNC1 feature Mask */
#define ESCSS_INTR_MASK_IRQ_MASK            0x4U    /* EtherCATSS IRQ Mask */
#define ESCSS_INTR_MASK_DMA_DONE_MASK       0x8U    /* DMA Done Mask */
#define ESCSS_INTR_MASK_TIMEOUT_ERR_MASK    0x10U   /* PDI Access Timeout Error Mask */
#define ESCSS_INTR_MASK_MDEVICE_RESET_MASK  0x20U   /* EtherCAT MainDevice Reset Mask */
#define ESCSS_INTR_MASK_M                   0x3FU   /* ESCSS_INTR_MASK register bit mask */

/**
 * @brief The following are defines for the bit fields in the ESCSS_INTR_MIS register
 * 
 */
#define ESCSS_INTR_MIS_SYNC0_MIS            0x1U    /* SYNC0 feature MIS */
#define ESCSS_INTR_MIS_SYNC1_MIS            0x2U    /* SYNC1 feature MIS */
#define ESCSS_INTR_MIS_IRQ_MIS              0x4U    /* EtherCATSS IRQ MIS */
#define ESCSS_INTR_MIS_DMA_DONE_MIS         0x8U    /* DMA Done MIS */
#define ESCSS_INTR_MIS_TIMEOUT_ERR_MIS      0x10U   /* PDI bus Timeout Error MIS */
#define ESCSS_INTR_MIS_MDEVICE_RESET_MIS    0x20U   /* EtherCAT MainDevice Reset MIS */
#define ESCSS_INTR_MIS_M                    0x3FU   /* ESCSS_INTR_MIS register bit mask */

/**
 * @brief The following are defines for the bit fields in the ESCSS_INTR_CLR register
 * 
 */
#define ESCSS_INTR_CLR_SYNC0_CLR            0x1U    /* SYNC0 feature Clear */
#define ESCSS_INTR_CLR_SYNC1_CLR            0x2U    /* SYNC1 feature Clear */
#define ESCSS_INTR_CLR_IRQ_CLR              0x4U    /* EtherCATSS IRQ Clear */
#define ESCSS_INTR_CLR_DMA_DONE_CLR         0x8U    /* DMA Done Clear */
#define ESCSS_INTR_CLR_TIMEOUT_ERR_CLR      0x10U   /* PDI Access Timeout Error Clear */
#define ESCSS_INTR_CLR_MDEVICE_RESET_CLR    0x20U   /* EtherCAT MainDevice Reset Clear */
#define ESCSS_INTR_CLR_M                    0x3FU   /* ESCSS_INTR_CLR register bit mask */

/**
 * @brief The following are defines for the bit fields in the ESCSS_INTR_SET register
 * 
 */
#define ESCSS_INTR_SET_SYNC0_SET            0x1U    /* SYNC0 Set Emulate */
#define ESCSS_INTR_SET_SYNC1_SET            0x2U    /* SYNC1 Set Emulate */
#define ESCSS_INTR_SET_IRQ_SET              0x4U    /* EtherCATSS IRQ Set Emulate */
#define ESCSS_INTR_SET_DMA_DONE_SET         0x8U    /* DMA Done Set Emulate */
#define ESCSS_INTR_SET_TIMEOUT_ERR_SET      0x10U   /* PDI Access Timeout Error Set Emulate */
#define ESCSS_INTR_SET_MDEVICE_RESET_SET    0x20U   /* EtherCAT MainDevice Reset Emulate */
#define ESCSS_INTR_SET_INT_CAUSE_M          0x3FU   /* ESCSS_INTR */
#define ESCSS_INTR_SET_WRITE_KEY_S          8U
#define ESCSS_INTR_SET_WRITE_KEY_M          0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_LATCH_SEL register
 * 
 */
#define ESCSS_LATCH_SEL_LATCH0_SELECT_S     0U
#define ESCSS_LATCH_SEL_LATCH0_SELECT_M     0x1FU   /* LATCH0 Inputs mux select */
#define ESCSS_LATCH_SEL_LATCH1_SELECT_S     8U
#define ESCSS_LATCH_SEL_LATCH1_SELECT_M     0x1F00U /* LATCH1 Inputs mux select */

/**
 * @brief The following are defines for the bit fields in the ESCSS_ACCESS_CTRL register
 * 
 */
#define ESCSS_ACCESS_CTRL_WAIT_STATES_S                 0U
#define ESCSS_ACCESS_CTRL_WAIT_STATES_M                 0x7FU        /* Minimum Wait States
                                                                        for VBUS Bridge */
#define ESCSS_ACCESS_CTRL_EN_TIMEOUT                    0x80U        /* PDI Timeout enable */
#define ESCSS_ACCESS_CTRL_ENABLE_DEBUG_ACCESS           0x200U       /* Debug access enable */
#define ESCSS_ACCESS_CTRL_ENABLE_PARALLEL_PORT_ACCESS   0x400U       /* Parallel port access
                                                                        enable */
#define ESCSS_ACCESS_CTRL_TIMEOUT_COUNT_S               16U
#define ESCSS_ACCESS_CTRL_TIMEOUT_COUNT_M               0xFFF0000U   /* Max timecount programmed
                                                                        and count while
                                                                        enabled. */
#define ESCSS_ACCESS_CTRL_TIMEOUT_COUNT                 (ESCSS_ACCESS_CTRL_TIMEOUT_COUNT_M >> \
                                                        ESCSS_ACCESS_CTRL_TIMEOUT_COUNT_S)

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPIN_DAT register
 * 
 */
#define ESCSS_GPIN_DAT_M    0xFFFFFFFFU     /* GPIN data capture for debug & override */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPIN_PIPE register
 * 
 */
#define ESCSS_GPIN_PIPE_M   0xFFFFFFFFU     /* GPIN pipeline select */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPIN_GRP_CAP_SEL register
 * 
 */
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL0_S   0U
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL0_M   0x7U    /* GPI7-0 capture trigger select */

#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL1_S   4U
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL1_M   0x70U   /* GPI15-8 capture trigger select */
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL1     (ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL1_M >> \
                                                    ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL1_S)
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL2_S   8U
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL2_M   0x700U  /* GPI23-16 capture trigger select */
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL2     (ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL2_M >> \
                                                    ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL2_S)
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL3_S   12U
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL3_M   0x7000U /* GPI31-24 capture trigger select */
#define ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL3     (ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL3_M >> \
                                                    ESCSS_GPIN_GRP_CAP_SEL_GPI_GRP_CAP_SEL3_S)

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPOUT_DAT register
 * 
 */
#define ESCSS_GPOUT_DAT_M           0xFFFFFFFFU     /* GPOUT data capture for debug & override */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPOUT_PIPE register
 * 
 */
#define ESCSS_GPOUT_PIPE_M          0xFFFFFFFFU    /* GPOUT pipeline select */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPOUT_GRP_CAP_SEL register
 * 
 */
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL0_S  0U
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL0_M  0x3U    /* GPO7-0 capture trigger select */

#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL1_S  4U
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL1_M  0x30U   /* GPO15-8 capture trigger select */
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL1    (ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL1_M >> \
                                                    ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL1_S)

#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL2_S  8U
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL2_M  0x300U  /* GPO23-16 capture trigger select */
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL2    (ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL2_M >> \
                                                    ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL2_S)

#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL3_S  12U
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL3_M  0x3000U /* GPO31-24 capture trigger select */
#define ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL3    (ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL3_M >> \
                                                    ESCSS_GPOUT_GRP_CAP_SEL_GPO_GRP_CAP_SEL3_S)

/**
 * @brief The following are defines for the bit fields in the ESCSS_MEM_TEST register
 * 
 */
#define ESCSS_MEM_TEST_INITIATE_MEM_INIT            0x1U    /* Initialize memory init */
#define ESCSS_MEM_TEST_MEM_INIT_DONE                0x2U    /* Memory Init done status */

/**
 * @brief The following are defines for the bit fields in the ESCSS_RESET_DEST_CONFIG register
 * 
 */
#define ESCSS_RESET_DEST_CONFIG_CPU_RESET_EN        0x1U    /* CPU reset enable for ResetOut */
#define ESCSS_RESET_DEST_CONFIG_CPU_NMI_EN          0x2U    /* CPU NMI enable for ResetOut */
#define ESCSS_RESET_DEST_CONFIG_CPU_INT_EN          0x4U    /* CPU Interrupt enable for ResetOut */
#define ESCSS_RESET_DEST_CONFIG_DEVICE_RESET_EN     0x80U   /* Enables RESET_OUT to impact the
                                                               device reset */
#define ESCSS_RESET_DEST_CONFIG_WRITE_KEY_S         8U
#define ESCSS_RESET_DEST_CONFIG_WRITE_KEY_M         0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_SYNC0_CONFIG register
 * 
 */
#define ESCSS_SYNC0_CONFIG_PIE_EN           0x1U    /* Connects the SYNC0 to PIE Interrupt */
#define ESCSS_SYNC0_CONFIG_CLA_INT_EN       0x2U    /* Connects the SYNC0 to CLA Interrupt */
#define ESCSS_SYNC0_CONFIG_DMA_EN           0x4U    /* Connects the SYNC0 to DMA Trigger */
#define ESCSS_SYNC0_CONFIG_WRITE_KEY_S      8U
#define ESCSS_SYNC0_CONFIG_WRITE_KEY_M      0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_SYNC1_CONFIG register
 * 
 */
#define ESCSS_SYNC1_CONFIG_PIE_EN           0x1U    /* Connects the SYNC1 to PIE Interrupt */
#define ESCSS_SYNC1_CONFIG_CLA_INT_EN       0x2U    /* Connects the SYNC1 to CLA Interrupt */
#define ESCSS_SYNC1_CONFIG_DMA_EN           0x4U    /* Connects the SYNC1 to DMA Trigger */
#define ESCSS_SYNC1_CONFIG_WRITE_KEY_S      8U
#define ESCSS_SYNC1_CONFIG_WRITE_KEY_M      0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_CONFIG_LOCK register
 * 
 */
#define ESCSS_CONFIG_LOCK_LOCK_ENABLE               0x1U    /* Locking writes to ECATSS */
#define ESCSS_CONFIG_LOCK_IO_CONFIG_ENABLE          0x10U   /* Locking the IO Configuration */
#define ESCSS_CONFIG_LOCK_WRITE_KEY_S               8U
#define ESCSS_CONFIG_LOCK_WRITE_KEY_M               0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_MISC_IO_CONFIG register
 * 
 */
#define ESCSS_MISC_IO_CONFIG_RESETIN_GPIO_EN        0x1U    /* Enabled ResetIN from GPIO */
#define ESCSS_MISC_IO_CONFIG_EEPROM_I2C_IO_EN       0x2U    /* Enables the EEPROM I2C IOPAD
                                                            connection */
#define ESCSS_MISC_IO_CONFIG_WRITE_KEY_S            8U
#define ESCSS_MISC_IO_CONFIG_WRITE_KEY_M            0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_PHY_IO_CONFIG register
 * 
 */
#define ESCSS_PHY_IO_CONFIG_PHY_PORT_CNT_S          2U
#define ESCSS_PHY_IO_CONFIG_PHY_PORT_CNT_M          0xCU    /* Number of PHY port counts */
#define ESCSS_PHY_IO_CONFIG_PHY_PORT_CNT            (ESCSS_PHY_IO_CONFIG_PHY_PORT_CNT_M >> \
                                                    ESCSS_PHY_IO_CONFIG_PHY_PORT_CNT_S)
#define ESCSS_PHY_IO_CONFIG_PHY_INTF_IOPAD_SEL_S    4U
#define ESCSS_PHY_IO_CONFIG_PHY_INTF_IOPAD_SEL_M    0x30U   /* IO Combination select for PHY
                                                               Interface */
#define ESCSS_PHY_IO_CONFIG_TX_CLK_AUTO_COMP        0x40U   /* Selects TX_CLK IO to do Auto
                                                               compensation */
#define ESCSS_PHY_IO_CONFIG_WRITE_KEY_S             8U
#define ESCSS_PHY_IO_CONFIG_WRITE_KEY_M             0xFF00U /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_SYNC_IO_CONFIG register
 * 
 */
#define ESCSS_SYNC_IO_CONFIG_SYNC0_IOPAD_SEL_S      0U
#define ESCSS_SYNC_IO_CONFIG_SYNC0_IOPAD_SEL_M      0x3U       /* SYNC0 IO PAD select option */
#define ESCSS_SYNC_IO_CONFIG_SYNC0_GPIO_EN          0x8U       /* SYNC0 connection to OUT pad
                                                                enabled */
#define ESCSS_SYNC_IO_CONFIG_SYNC1_IOPAD_SEL_S      4U
#define ESCSS_SYNC_IO_CONFIG_SYNC1_IOPAD_SEL_M      0x30U      /* SYNC1 IO PAD select option */
#define ESCSS_SYNC_IO_CONFIG_SYNC1_IOPAD_SEL        (ESCSS_SYNC_IO_CONFIG_SYNC1_IOPAD_SEL_M >> \
                                                    ESCSS_SYNC_IO_CONFIG_SYNC1_IOPAD_SEL_S)
#define ESCSS_SYNC_IO_CONFIG_SYNC1_GPIO_EN          0x80U      /* SYNC1 connection to OUT pad
                                                                enabled */
#define ESCSS_SYNC_IO_CONFIG_WRITE_KEY_S            8U
#define ESCSS_SYNC_IO_CONFIG_WRITE_KEY_M            0xFF00U    /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_LATCH_IO_CONFIG register
 * 
 */
#define ESCSS_LATCH_IO_CONFIG_LATCH0_IOPAD_SEL_S    0U
#define ESCSS_LATCH_IO_CONFIG_LATCH0_IOPAD_SEL_M    0x3U     /* LATCH0 IO PAD select option */
#define ESCSS_LATCH_IO_CONFIG_LATCH0_GPIO_EN        0x8U     /* LATCH0 connection to IN pad
                                                                enabled */
#define ESCSS_LATCH_IO_CONFIG_LATCH1_IOPAD_SEL_S    4U
#define ESCSS_LATCH_IO_CONFIG_LATCH1_IOPAD_SEL_M    0x30U    /* LATCH1 IO PAD select option */
#define ESCSS_LATCH_IO_CONFIG_LATCH1_IOPAD_SEL      (ESCSS_LATCH_IO_CONFIG_LATCH1_IOPAD_SEL_M >> \
                                                    ESCSS_LATCH_IO_CONFIG_LATCH1_IOPAD_SEL_S)
#define ESCSS_LATCH_IO_CONFIG_LATCH1_GPIO_EN        0x80U    /* LATCH1 connection to IN pad
                                                                enabled */
#define ESCSS_LATCH_IO_CONFIG_WRITE_KEY_S           8U
#define ESCSS_LATCH_IO_CONFIG_WRITE_KEY_M           0xFF00U  /* Key to enable writing lock */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPIN_SEL register
 * 
 */
#define ESCSS_GPIN_SEL_M        0xFFFFFFFFU     /* GPIN Select between IO PAD & tieoff */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPIN_IOPAD_SEL register
 * 
 */
#define ESCSS_GPIN_IOPAD_SEL_M  0xFFFFFFFFU     /* GPIN IO pad Select */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPOUT_SEL register
 * 
 */
#define ESCSS_GPOUT_SEL_M       0xFFFFFFFFU     /* GPOUT IO pad connect select */

/**
 * @brief The following are defines for the bit fields in the ESCSS_GPOUT_IOPAD_SEL register
 * 
 */
#define ESCSS_GPOUT_IOPAD_SEL_M 0xFFFFFFFFU     /* GPOUT IO pad select */

/**
 * @brief The following are defines for the bit fields in the ESCSS_LED_CONFIG register
 * 
 */
#define ESCSS_LED_CONFIG_LINKACT0               0x1U    /* GPIO enable for LINKACT0 LED */
#define ESCSS_LED_CONFIG_LINKACT1               0x2U    /* GPIO enable for LINKACT1 LED */
#define ESCSS_LED_CONFIG_STATE                  0x4U    /* GPIO enable for STATE LED */
#define ESCSS_LED_CONFIG_ERR                    0x8U    /* GPIO enable for ERR LED */
#define ESCSS_LED_CONFIG_RUN                    0x10U   /* GPIO enable for RUN LED */
#define ESCSS_LED_CONFIG_LINKACT0_IOPAD_SEL_S   6U
#define ESCSS_LED_CONFIG_LINKACT0_IOPAD_SEL_M   0xC0U   /* LINKACT0 LED IO PAD select */
#define ESCSS_LED_CONFIG_LINKACT0_IOPAD_SEL     (ESCSS_LED_CONFIG_LINKACT0_IOPAD_SEL_M >> \
                                                ESCSS_LED_CONFIG_LINKACT0_IOPAD_SEL_S)
#define ESCSS_LED_CONFIG_LINKACT1_IOPAD_SEL_S   8U
#define ESCSS_LED_CONFIG_LINKACT1_IOPAD_SEL_M   0x300U  /* LINKACT1 LED IO PAD select */
#define ESCSS_LED_CONFIG_LINKACT1_IOPAD_SEL     (ESCSS_LED_CONFIG_LINKACT1_IOPAD_SEL_M >> \
                                                ESCSS_LED_CONFIG_LINKACT1_IOPAD_SEL_S)
#define ESCSS_LED_CONFIG_STATE_IOPAD_SEL_S      10U
#define ESCSS_LED_CONFIG_STATE_IOPAD_SEL_M      0xC00U  /* STATE LED IO PAD select */
#define ESCSS_LED_CONFIG_STATE_IOPAD_SEL        (ESCSS_LED_CONFIG_STATE_IOPAD_SEL_M >> \
                                                ESCSS_LED_CONFIG_STATE_IOPAD_SEL_S)
#define ESCSS_LED_CONFIG_ERR_IOPAD_SEL_S        12U
#define ESCSS_LED_CONFIG_ERR_IOPAD_SEL_M        0x3000U /* ERROR LED IO PAD select */
#define ESCSS_LED_CONFIG_ERR_IOPAD_SEL          (ESCSS_LED_CONFIG_ERR_IOPAD_SEL_M >> \
                                                ESCSS_LED_CONFIG_ERR_IOPAD_SEL_S)
#define ESCSS_LED_CONFIG_RUN_IOPAD_SEL_S        14U
#define ESCSS_LED_CONFIG_RUN_IOPAD_SEL_M        0xC000U /* RUN LED IO PAD select */
#define ESCSS_LED_CONFIG_RUN_IOPAD_SEL          (ESCSS_LED_CONFIG_RUN_IOPAD_SEL_M >> \
                                                ESCSS_LED_CONFIG_RUN_IOPAD_SEL_S)

/**
 * @brief The following are defines for the bit fields in the ESCSS_MISC_CONFIG register
 * 
 */
#define ESCSS_MISC_CONFIG_TX0_SHIFT_CONFIG_S    0U
#define ESCSS_MISC_CONFIG_TX0_SHIFT_CONFIG_M    0x3U    /* TX Shift configuration for Port0 */
#define ESCSS_MISC_CONFIG_TX1_SHIFT_CONFIG_S    2U
#define ESCSS_MISC_CONFIG_TX1_SHIFT_CONFIG_M    0xCU    /* TX Shift configuration for Port1 */
#define ESCSS_MISC_CONFIG_TX1_SHIFT_CONFIG      (ESCSS_MISC_CONFIG_TX1_SHIFT_CONFIG_M >> \
                                                ESCSS_MISC_CONFIG_TX1_SHIFT_CONFIG_S)
#define ESCSS_MISC_CONFIG_EEPROM_SIZE           0x10U   /* EEPROM Size bound select */
#define ESCSS_MISC_CONFIG_PDI_EMULATION         0x20U   /* PDI Emulation enable */
#define ESCSS_MISC_CONFIG_PHY_ADDR_S            6U
#define ESCSS_MISC_CONFIG_PHY_ADDR_M            0x7C0U  /* PHY Address Offset */
#define ESCSS_MISC_CONFIG_PHY_ADDR              (ESCSS_MISC_CONFIG_PHY_ADDR_M >> \
                                                ESCSS_MISC_CONFIG_PHY_ADDR_S)

/**
 * @brief The write enable key of EtherCAT slave controller register.
 * 
 */
#define ESCSS_REGISTER_WRITE_EN_KEY             0xa5U

/**
 * @brief The Maximum number of EtherCAT GPI/GPO pins.
 * 
 */
#define ESCSS_MAX_NUM_OF_GPI_PINS               31U
#define ESCSS_MAX_NUM_OF_GPO_PINS               31U

#ifdef __cplusplus
}
#endif

#endif
