/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifndef __LIN_H__
#define __LIN_H__

#include <stdbool.h>
#include <stdint.h>

#include "inc/hw_lin.h"
#include "inc/hw_memmap.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22) || IS_GS32F00xx(0x30)

/**
 * @brief Check if the base address is valid or not.
 *
 */
#define LIN_CHECK_BASE_VALID		1U

#define LIN_NUMBER_OF_INSTANCES		2U

#define LIN_BASE_ADDR_ARRAY		{	\
		LINA_BASE,					\
		LINB_BASE,					\
}

/**
 * @brief Maximum length of LIN frame.
 *
 */
#define LIN_FRAME_MAX_LENGTH		256U

typedef enum {
	LIN_SUCCESS				= 0U,
	LIN_ERROR				= 1U,
	LIN_BUSY				= 2U,
	LIN_TIMEOUT				= 3U,
	LIN_INVALID_PARAMTER	= 5U,
	LIN_RXEMPTY				= 6U,
	LIN_TXFULL				= 7U,
	LIN_RXERR				= 8U,
	LIN_TXERR				= 9U,
} LIN_errNumber_t;

/**
 * @brief LIN operation mode.
 *
 * @param LIN_UART_MODE UART mode.
 * @param LIN_LIN_MASTER_MODE LIN master mode.
 * @param LIN_LIN_SLAVE_MODE LIN slave mode.
 */
typedef enum {
	LIN_UART_MODE			= 0U,
	LIN_LIN_MASTER_MODE		= 0xAU,
	LIN_LIN_SLAVE_MODE		= 0xBU,
} LIN_opMode_t;

/**
 * @brief LIN character length.
 *
 * @param LIN_CHAR_LENGTH_5BITS length of 5 bits.
 * @param LIN_CHAR_LENGTH_6BITS length of 6 bits.
 * @param LIN_CHAR_LENGTH_7BITS length of 7 bits.
 * @param LIN_CHAR_LENGTH_8BITS length of 8 bits.
 * @param LIN_CHAR_LENGTH_9BITS length of 9 bits.
 */
typedef enum {
	LIN_CHAR_LENGTH_5BITS	= 0U,
	LIN_CHAR_LENGTH_6BITS	= 1U,
	LIN_CHAR_LENGTH_7BITS	= 2U,
	LIN_CHAR_LENGTH_8BITS	= 3U,
	LIN_CHAR_LENGTH_9BITS	= 4U,
} LIN_charLength_t;

/**
 * @brief LIN module clock source select.
 *
 * @param LIN_CLK_SRC_MCK MCK as LIN clock source.
 * @param LIN_CLK_SRC_MCK_DIV MCK/DIV as LIN clock source.
 * @param LIN_CLK_SRC_LIN_SCK Serial Clock SLK as LIN clock source.
 */
typedef enum {
	LIN_CLK_SRC_MCK			= 0U,
	LIN_CLK_SRC_MCK_DIV		= 1U,
	LIN_CLK_SRC_LIN_SCK		= 3U,
} LIN_clkSource_t;

/**
 * @brief LIN synchronous mode select.
 *
 * @param LIN_OP_ASYNC_MODE USART operates in Asynchronous Mode.
 * @param LIN_OP_SYNC_MODE USART operates in Synchronous Mode.
 */
typedef enum {
	LIN_OP_ASYNC_MODE		= 0U,
	LIN_OP_SYNC_MODE		= 1U,
} LIN_syncMode_t;

/**
 * @brief LIN parity type.
 *
 * @param LIN_PARITY_EVEN Even parity.
 * @param LIN_PARITY_ODD Odd parity.
 * @param LIN_PARITY_SPACE Parity forced to 0 (Space).
 * @param LIN_PARITY_MARK Parity forced to 1 (Mark).
 * @param LIN_PARITY_NONE No parity.
 * @param LIN_PARITY_MULTIDROP Multidrop mode.
 */
typedef enum {
	LIN_PARITY_EVEN			= 0U,
	LIN_PARITY_ODD			= 1U,
	LIN_PARITY_SPACE		= 2U,
	LIN_PARITY_MARK			= 3U,
	LIN_PARITY_NONE			= 4U,
	LIN_PARITY_MULTIDROP	= 6U,
} LIN_parityType_t;

/**
 * @brief Number of LIN module stop bits.
 *
 * @param LIN_STOP_BIT_1 1 stop bit.
 * @param LIN_STOP_BIT_1_5 1.5 stop bits.
 * @param LIN_STOP_BIT_2 2 stop bits.
 */
typedef enum {
	LIN_STOP_BIT_1			= 0U,
	LIN_STOP_BIT_1_5		= 1U,
	LIN_STOP_BIT_2			= 2U,
} LIN_stopBitMode_t;

/**
 * @brief LIN Channel Mode.
 *
 * @param LIN_CHAL_NORMAL_MODE Normal mode.
 * @param LIN_CHAL_AUTO_ECHO_MODE Automatic Echo. Receiver input is
 * connected to the TXD pin.
 * @param LIN_CHAL_LOCAL_LOOPBACK_MODE Local loopback mode. Transmitter output is
 * connected to the Receiver Input.
 * @param LIN_CHAL_REMOTE_LOOPBACK_MODE Remote loopback mode. RXD pin is internally
 * connected to the TXD pin.
 */
typedef enum {
	LIN_CHAL_NORMAL_MODE			= 0U,
	LIN_CHAL_AUTO_ECHO_MODE			= 1U,
	LIN_CHAL_LOCAL_LOOPBACK_MODE	= 2U,
	LIN_CHAL_REMOTE_LOOPBACK_MODE	= 3U,
} LIN_channelMode_t;

/**
 * @brief LIN transfer Bit Order mode.
 *
 * @param LIN_TRANSFER_LSB LSB first.
 * @param LIN_TRANSFER_MSB MSB first.
 */
typedef enum {
	LIN_TRANSFER_LSB		= 0U,
	LIN_TRANSFER_MSB		= 1U,
} LIN_bitOrderMode_t;

/**
 * @brief LIN module Oversampling Mode.
 *
 * @param LIN_OVERSAMP_16 16x oversampling.
 * @param LIN_OVERSAMP_8 8x oversampling.
 */
typedef enum {
	LIN_OVERSAMP_16			= 0U,
	LIN_OVERSAMP_8			= 1U,
} LIN_overSampMode_t;

/**
 * @brief Start Frame Delimiter Selector.
 *
 * @param LIN_ONEBIT_CMD_OR_DATA_SYNC Start Frame delimiter is COMMAND or DATA SYNC.
 * @param LIN_ONEBIT_IS_ONE_BIT Start Frame delimiter is One Bit.
 */
typedef enum {
	LIN_ONEBIT_CMD_OR_DATA_SYNC = 0U,
	LIN_ONEBIT_IS_ONE_BIT = 1U,
} LIN_startFrameBitType_t;

/**
 * @brief LIN module initialization parameters.
 *
 * @param opmode LIN operation mode.
 * @param char_length character length.
 * @param sync_mode LIN synchronous mode select.
 * @param parity LIN parity type.
 * @param stopbit Number of LIN module stop bits.
 * @param chl_mode LIN Channel Mode.
 * @param bit_order LIN transfer Bit Order mode.
 * @param onebit Start Frame Delimiter Selector.
 * @param sck_clk_out LIN module SCK clock output enable.
 * @param inverted_data_en LIN module inverted data enable.
 * @param rx_filter_en LIN module RX filter enable.
 */
typedef struct LIN_initParam {
	LIN_opMode_t opmode;
	LIN_charLength_t char_length;
	LIN_syncMode_t sync_mode;
	LIN_parityType_t parity;
	LIN_stopBitMode_t stopbit;
	LIN_channelMode_t chl_mode;
	LIN_bitOrderMode_t bit_order;
	LIN_startFrameBitType_t onebit;
	bool sck_clk_out;
	bool inverted_data_en;
	bool rx_filter_en;
} LIN_initParam_t;

/**
 * @brief LIN module LIN mode data length mode.
 *
 * @param LIN_LIN_DLM_BY_DLC The response data length is defined by the
 * field DLC of this register.
 * @param LIN_LIN_DLM_BY_IDCHR The response data length is defined by the
 * bits 5 and 6 of the Identifier (IDCHR in US_LINIR).
 */
typedef enum {
	LIN_LIN_DLM_BY_DLC		= 0U,
	LIN_LIN_DLM_BY_IDCHR	= 1U,
} LIN_LinDataLengthMode_t;

typedef enum {
	LIN_LIN_ISO_MODE_1_3			= 0U,
	LIN_LIN_ISO_MODE_ENHANCED_2_0	= 1U,
} LIN_IsoMode_t;

/**
 * @brief LIN module LIN mode initialization parameters.
 *
 * @param parity_en Enable/Disable the Identifier Parity is computed and
 * sent automatically.
 * @param checksum_en Enable/Disable the checksum is computed and sent automatically.
 * @param frame_slot_en LIN frame slot enable.
 * @param dma_en LIN DMA mode enable.
 * @param slave_sync_en Enable/Disable the Synchronization procedure is performed
 * in LIN Slave node configuration.
 * @param iso_mode The LIN standard communication protocol mode.
 * @param data_length_mode LIN data length mode.
 */
typedef struct LIN_LinModeInitParam {
	bool parity_en;
	bool checksum_en;
	bool frame_slot_en;
	bool dma_en;
	bool slave_sync_en;
	LIN_IsoMode_t iso_mode;
	LIN_LinDataLengthMode_t data_length_mode;
} LIN_LinModeInitParam_t;

/**
 * @brief LIN module LIN mode node action.
 *
 * @param LIN_LIN_OP_MODE_PUBLISH The LIN module transmits the response.
 * @param LIN_LIN_OP_MODE_SUBSCRIBE The LIN module receives the response.
 * @param LIN_LIN_OP_MODE_IGNORE The LIN module does not transmit and
 * does not receive the response.
 */
typedef enum {
	LIN_LIN_OP_MODE_PUBLISH			= 0U,
	LIN_LIN_OP_MODE_SUBSCRIBE		= 1U,
	LIN_LIN_OP_MODE_IGNORE			= 2U,
} LIN_LinModeNodeAction_t;

/**
 * @brief The interrupt type of LIN module.
 *
 * @param LIN_LIN_MODULE_INTR_RXRDY RXRDY interrupt.
 * @param LIN_LIN_MODULE_INTR_TXRDY TXRDY interrupt.
 * @param LIN_LIN_MODULE_INTR_ENDRX End of RX interrupt.
 * @param LIN_LIN_MODULE_INTR_ENDTX End of TX interrupt.
 * @param LIN_LIN_MODULE_INTR_OVRE Overrun Error interrupt.
 * @param LIN_LIN_MODULE_INTR_FRAME_ERR Frame Error interrupt.
 * @param LIN_LIN_MODULE_INTR_PARE Parity Error interrupt.
 * @param LIN_LIN_MODULE_INTR_TIMEOUT Timeout Error interrupt.
 * @param LIN_LIN_MODULE_INTR_TXEMPTY Tx EMPTY interrupt.
 * @param LIN_LIN_MODULE_INTR_TXBUFE Tx Buffer Empty Interrupt.
 * @param LIN_LIN_MODULE_INTR_RXBUFF Rx Buffer Full Interrupt.
 * @param LIN_LIN_MODULE_INTR_LINBK LIN Break Detection interrupt.
 * @param LIN_LIN_MODULE_INTR_LINID LIN Identifier Received interrupt.
 * @param LIN_LIN_MODULE_INTR_LINTC LIN Transmission Completed interrupt.
 * @param LIN_LIN_MODULE_INTR_LINBE LIN Break Error interrupt.
 * @param LIN_LIN_MODULE_INTR_LINISFE LIN Inconsistent Synch Field Error interrupt.
 * @param LIN_LIN_MODULE_INTR_LINIPE LIN Inconsistent Parity Error interrupt.
 * @param LIN_LIN_MODULE_INTR_LINCE LIN Checksum Error interrupt.
 * @param LIN_LIN_MODULE_INTR_LINSNRE LIN Slave Not Responding Error Interrupt.
 * @param LIN_LIN_MODULE_INTR_LINSTE LIN Synch Tolerance Error Interrupt.
 * @param LIN_LIN_MODULE_INTR_HTE LIN Header Timeout Error Interrupt Enable.
 */
typedef enum {
	LIN_LIN_MODULE_INTR_RXRDY				= LIN_US_LIN_IER_RXRDY_M,
	LIN_LIN_MODULE_INTR_TXRDY				= LIN_US_LIN_IER_TXRDY_M,
	LIN_LIN_MODULE_INTR_ENDRX				= LIN_US_LIN_IER_ENDRX_M,
	LIN_LIN_MODULE_INTR_ENDTX				= LIN_US_LIN_IER_ENDTX_M,
	LIN_LIN_MODULE_INTR_OVRE				= LIN_US_LIN_IER_OVER_M,
	LIN_LIN_MODULE_INTR_FRAME_ERR			= LIN_US_LIN_IER_FRAME_M,
	LIN_LIN_MODULE_INTR_PARE				= LIN_US_LIN_IER_PARE_M,
	LIN_LIN_MODULE_INTR_TIMEOUT				= LIN_US_LIN_IER_TIMEOUT_M,
	LIN_LIN_MODULE_INTR_TXEMPTY				= LIN_US_LIN_IER_TXEMPTY_M,
	LIN_LIN_MODULE_INTR_TXBUFE				= LIN_US_LIN_IER_TXBUFE_M,
	LIN_LIN_MODULE_INTR_RXBUFF				= LIN_US_LIN_IER_RXBUFF_M,
	LIN_LIN_MODULE_INTR_LINBK				= LIN_US_LIN_IER_LINBK_M,
	LIN_LIN_MODULE_INTR_LINID				= LIN_US_LIN_IER_LINID_M,
	LIN_LIN_MODULE_INTR_LINTC				= LIN_US_LIN_IER_LINTC_M,
	LIN_LIN_MODULE_INTR_LINBE				= LIN_US_LIN_IER_LINBE_M,
	LIN_LIN_MODULE_INTR_LINISFE				= LIN_US_LIN_IER_LINISFE_M,
	LIN_LIN_MODULE_INTR_LINIPE				= LIN_US_LIN_IER_LINIPE_M,
	LIN_LIN_MODULE_INTR_LINCE				= LIN_US_LIN_IER_LINCE_M,
	LIN_LIN_MODULE_INTR_LINSNRE				= LIN_US_LIN_IER_LINSNRE_M,
	LIN_LIN_MODULE_INTR_LINSTE				= LIN_US_LIN_IER_LINSTE_M,
	LIN_LIN_MODULE_INTR_HTE					= LIN_US_LIN_IER_HTE_M,
} LIN_LinModeInterrupt_t;

/**
 * @brief The interrupt type of USART module.
 *
 * @param LIN_USART_MODULE_INTR_RXRDY Rx ready interrupt.
 * @param LIN_USART_MODULE_INTR_TXRDY TX ready interrupt.
 * @param LIN_USART_MODULE_INTR_RXBRK Receiver Break Interrupt.
 * @param LIN_USART_MODULE_INTR_ENDRX End of RX interrupt.
 * @param LIN_USART_MODULE_INTR_ENDTX End of TX interrupt.
 * @param LIN_USART_MODULE_INTR_OVRE Overrun Error interrupt.
 * @param LIN_USART_MODULE_INTR_FRAME_ERR Frame Error interrupt.
 * @param LIN_USART_MODULE_INTR_PARE Parity Error interrupt.
 * @param LIN_USART_MODULE_INTR_TIMEOUT Timeout Error interrupt.
 * @param LIN_USART_MODULE_INTR_TXEMPTY TXEMPTY interrupt.
 * @param LIN_USART_MODULE_INTR_TXBUFE Tx Buffer Empty Interrupt.
 * @param LIN_USART_MODULE_INTR_RXBUFF Rx Buffer Full Interrupt.
 * @param LIN_USART_MODULE_INTR_NACK Non Acknowledge Interrupt.
 * @param LIN_USART_MODULE_INTR_RIIC Ring Indicator Input Change.
 */
typedef enum {
	LIN_USART_MODULE_INTR_RXRDY				= LIN_US_USART_IER_RXRDY_M,
	LIN_USART_MODULE_INTR_TXRDY				= LIN_US_USART_IER_TXRDY_M,
	LIN_USART_MODULE_INTR_RXBRK				= LIN_US_USART_IER_RXBRK_M,
	LIN_USART_MODULE_INTR_ENDRX				= LIN_US_USART_IER_ENDRX_M,
	LIN_USART_MODULE_INTR_ENDTX				= LIN_US_USART_IER_ENDTX_M,
	LIN_USART_MODULE_INTR_OVRE				= LIN_US_USART_IER_OVRE_M,
	LIN_USART_MODULE_INTR_FRAME_ERR			= LIN_US_USART_IER_FRAME_M,
	LIN_USART_MODULE_INTR_PARE				= LIN_US_USART_IER_PARE_M,
	LIN_USART_MODULE_INTR_TIMEOUT			= LIN_US_USART_IER_TIMEOUT_M,
	LIN_USART_MODULE_INTR_TXEMPTY			= LIN_US_USART_IER_TXEMPTY_M,
	LIN_USART_MODULE_INTR_ITER				= LIN_US_USART_IER_ITER_M,
	LIN_USART_MODULE_INTR_TXBUFE			= LIN_US_USART_IER_TXBUFE_M,
	LIN_USART_MODULE_INTR_RXBUFF			= LIN_US_USART_IER_RXBUFF_M,
	LIN_USART_MODULE_INTR_NACK				= LIN_US_USART_IER_NACK_M,
	LIN_USART_MODULE_INTR_RIIC				= LIN_US_USART_IER_RIIC_M,
	LIN_USART_MODULE_INTR_DSRIC				= LIN_US_USART_IER_DSRIC_M,
	LIN_USART_MODULE_INTR_DCDIC				= LIN_US_USART_IER_DCDIC_M,
	LIN_USART_MODULE_INTR_CTSIC				= LIN_US_USART_IER_CTSIC_M,
	LIN_USART_MODULE_INTR_MANE				= LIN_US_USART_IER_MANE_M,
} LIN_UsartModeInterrupt_t;

/**
 * @brief The channel status of LIN module.
 *
 * @param LIN_LIN_CHANNEL_RXRDY RX Ready channel status.
 * @param LIN_LIN_CHANNEL_TXRDY TX Ready channel status.
 * @param LIN_LIN_CHANNEL_ENDRX End of RX channel status.
 * @param LIN_LIN_CHANNEL_ENDTX End of TX channel status.
 * @param LIN_LIN_CHANNEL_OVRE Overrun Error channel status.
 * @param LIN_LIN_CHANNEL_FRAME_ERR Frame Error channel status.
 * @param LIN_LIN_CHANNEL_PARE Parity Error channel status.
 * @param LIN_LIN_CHANNEL_TIMEOUT Timeout Error channel status.
 * @param LIN_LIN_CHANNEL_TXEMPTY TXEMPTY channel status.
 * @param LIN_LIN_CHANNEL_TXBUFE Tx Buffer Empty channel status.
 * @param LIN_LIN_CHANNEL_RXBUFF Rx Buffer Full channel status.
 * @param LIN_LIN_CHANNEL_LINBK LIN Break Detection channel status.
 * @param LIN_LIN_CHANNEL_LINID LIN Identifier Received channel status.
 * @param LIN_LIN_CHANNEL_LINTC LIN Transmission Completed channel status.
 * @param LIN_LIN_CHANNEL_LINBLS LIN Bus Line Status.
 * @param LIN_LIN_CHANNEL_LINBE LIN Break Error channel status.
 * @param LIN_LIN_CHANNEL_LINISFE LIN Inconsistent Synch Field Error channel status.
 * @param LIN_LIN_CHANNEL_LINIPE LIN Inconsistent Parity Error channel status.
 * @param LIN_LIN_CHANNEL_LINCE LIN Checksum Error channel status.
 * @param LIN_LIN_CHANNEL_LINSNRE LIN Slave Not Responding Error channel status.
 * @param LIN_LIN_CHANNEL_LINSTE LIN Synch Tolerance Error channel status.
 * @param LIN_LIN_CHANNEL_LINHTE LIN Header Timeout Error channel status.
 */
typedef enum {
	LIN_LIN_CHANNEL_RXRDY					= (1U << 0U),
	LIN_LIN_CHANNEL_TXRDY					= (1U << 1U),
	LIN_LIN_CHANNEL_ENDRX					= (1U << 2U),
	LIN_LIN_CHANNEL_ENDTX					= (1U << 3U),
	LIN_LIN_CHANNEL_OVRE					= (1U << 5U),
	LIN_LIN_CHANNEL_FRAME_ERR				= (1U << 6U),
	LIN_LIN_CHANNEL_PARE					= (1U << 7U),
	LIN_LIN_CHANNEL_TIMEOUT					= (1U << 8U),
	LIN_LIN_CHANNEL_TXEMPTY					= (1U << 9U),
	LIN_LIN_CHANNEL_TXBUFE					= (1U << 11U),
	LIN_LIN_CHANNEL_RXBUFF					= (1U << 12U),
	LIN_LIN_CHANNEL_LINBK					= (1U << 13U),
	LIN_LIN_CHANNEL_LINID					= (1U << 14U),
	LIN_LIN_CHANNEL_LINTC					= (1U << 15U),
	LIN_LIN_CHANNEL_LINBLS					= (1U << 23U),
	LIN_LIN_CHANNEL_LINBE					= (1U << 25U),
	LIN_LIN_CHANNEL_LINISFE					= (1U << 26U),
	LIN_LIN_CHANNEL_LINIPE					= (1U << 27U),
	LIN_LIN_CHANNEL_LINCE					= (1U << 28U),
	LIN_LIN_CHANNEL_LINSNRE					= (1U << 29U),
	LIN_LIN_CHANNEL_LINSTE					= (1U << 30U),
	LIN_LIN_CHANNEL_LINHTE					= (1U << 31U),
} LIN_LinChannelStatus_t;

/**
 * @brief The channel status of USART module.
 *
 * @param LIN_USART_CHANNEL_RXRDY RX Ready channel status.
 * @param LIN_USART_CHANNEL_TXRDY TX Ready channel status.
 * @param LIN_USART_CHANNEL_RXBRK Break Received or End of Break detected.
 * @param LIN_USART_CHANNEL_ENDRX End of RX channel status.
 * @param LIN_USART_CHANNEL_ENDTX End of TX channel status.
 * @param LIN_USART_CHANNEL_OVRE Overrun Error channel status.
 * @param LIN_USART_CHANNEL_FRAME_ERR Frame Error channel status.
 * @param LIN_USART_CHANNEL_PARE Parity Error channel status.
 * @param LIN_USART_CHANNEL_TIMEOUT Timeout Error channel status.
 * @param LIN_USART_CHANNEL_TXEMPTY TXEMPTY channel status.
 * @param LIN_USART_CHANNEL_TXBUFE Tx Buffer Empty channel status.
 * @param LIN_USART_CHANNEL_RXBUFF Rx Buffer Full channel status.
 * @param LIN_USART_CHANNEL_NACK Non Acknowledge Interrupt.
 * @param LIN_USART_CHANNEL_RIIC Ring Indicator Input Change.
 * @param LIN_USART_CHANNEL_DSRIC Data Set Ready Input Change Flag
 * @param LIN_USART_CHANNEL_DCDIC Data Carrier Detect Input Change Flag
 * @param LIN_USART_CHANNEL_CTSIC Clear to Send Input Change Flag.
 * @param LIN_USART_CHANNEL_RI Image of RI Input.
 * @param LIN_USART_CHANNEL_DSR Image of DSR Input.
 * @param LIN_USART_CHANNEL_DCD Image of DCD Input.
 * @param LIN_USART_CHANNEL_CTS Clear to Send Input Change Flag.
 * @param LIN_USART_CHANNEL_MANE Manchester Error.
 */
typedef enum {
	LIN_USART_CHANNEL_RXRDY					= (1U << 0U),
	LIN_USART_CHANNEL_TXRDY					= (1U << 1U),
	LIN_USART_CHANNEL_RXBRK					= (1U << 2U),
	LIN_USART_CHANNEL_ENDRX					= (1U << 3U),
	LIN_USART_CHANNEL_ENDTX					= (1U << 4U),
	LIN_USART_CHANNEL_OVRE					= (1U << 5U),
	LIN_USART_CHANNEL_FRAME_ERR				= (1U << 6U),
	LIN_USART_CHANNEL_PARE					= (1U << 7U),
	LIN_USART_CHANNEL_TIMEOUT				= (1U << 8U),
	LIN_USART_CHANNEL_TXEMPTY				= (1U << 9U),
	LIN_USART_CHANNEL_ITER					= (1U << 10U),
	LIN_USART_CHANNEL_TXBUFE				= (1U << 11U),
	LIN_USART_CHANNEL_RXBUFF				= (1U << 12U),
	LIN_USART_CHANNEL_NACK					= (1U << 13U),
	LIN_USART_CHANNEL_RIIC					= (1U << 16U),
	LIN_USART_CHANNEL_DSRIC					= (1U << 17U),
	LIN_USART_CHANNEL_DCDIC					= (1U << 18U),
	LIN_USART_CHANNEL_CTSIC					= (1U << 19U),
	LIN_USART_CHANNEL_RI					= (1U << 20U),
	LIN_USART_CHANNEL_DSR					= (1U << 21U),
	LIN_USART_CHANNEL_DCD					= (1U << 22U),
	LIN_USART_CHANNEL_CTS					= (1U << 23U),
	LIN_USART_CHANNEL_MANE					= (1U << 24U),
} LIN_UsartChannelStatus_t;

/**
 * @brief The LIN message structure.
 *
 * @param id LIN message ID.
 * @param data LIN message data.
 */
typedef struct LIN_LinMsg {
	uint8_t id;
	uint8_t *data;
} LIN_LinMsg_t;

static const uint32_t LIN_BaseAddressArray[] = LIN_BASE_ADDR_ARRAY;


/**
 * @brief Checks if the base address is valid.
 *
 * @param base LIN module base address.
 * @return true if the base address is valid; otherwise, return false.
 * @return false if the base address is invalid.
 */
static inline bool LIN_isBaseValid(uint32_t base)
{
#if LIN_CHECK_BASE_VALID

	for (uint8_t i = 0; i < LIN_NUMBER_OF_INSTANCES; i++) {
		if (LIN_BaseAddressArray[i] == base)
			return true;
	}

	return false;

#else

	return true;

#endif
}

static inline void LIN_writeRegister(uintptr_t base, uint32_t value)
{
	*(volatile uint32_t *)(base) = value;
}

static inline uint32_t LIN_readRegister(uintptr_t base)
{
	return *(volatile uint32_t *)(base);
}

/**
 * @brief Resets the module, include LIN and USART mode.
 *
 * @param base LIN module base address.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_resetModule(uintptr_t base)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	/*
	 * reset module. After executing the reset tx/rx operation,
	 * the tx/rx enable will be turned off.
	 */
	LIN_writeRegister(base + LIN_US_CR, LIN_US_CR_RSTRX_M |
										LIN_US_CR_RSTTX_M |
										LIN_US_CR_RSTSTA_M |
										LIN_US_CR_RSTNACK_M);

	return LIN_SUCCESS;
}

/**
 * @brief Gets the current operating mode.
 *
 * @param base LIN module base address.
 * @return LIN_opMode_t The current operating mode.
 */
static inline LIN_opMode_t LIN_getCurrentMode(uintptr_t base)
{
	return (LIN_opMode_t)(LIN_readRegister(base + LIN_US_MR) & LIN_US_MR_USART_MODE_M);
}

/**
 * @brief Sets the synchronization mode.
 *
 * @param base LIN module base address.
 * @param sync_mode The synchronous/asynchronous mode.
 */
static inline void LIN_setSyncMode(uintptr_t base, LIN_syncMode_t sync_mode)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (sync_mode == LIN_OP_ASYNC_MODE)
		reg_val &= ~(LIN_US_MR_SYNC_M);
	else
		reg_val |= LIN_US_MR_SYNC_M;

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Module global initialization.
 *
 * @param base LIN module base address.
 * @param initParam LIN module initialization parameters.
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_initModule(uintptr_t base, LIN_initParam_t *initParam);

/**
 * @brief Configuration baudrate.
 *
 * @param base LIN module base address.
 * @param clock_frq LIN module main clock frequency.
 * @param baudrate Bus baud rate.
 * @param sync_mode Select the synchronous/asynchronous
 * communication mode(@LIN_syncMode_t).
 * @param over_sampling Select the oversampling mode(@LIN_overSampMode_t).
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_setBaudRate(uintptr_t base, uint32_t clock_frq,
					uint32_t baudrate, LIN_syncMode_t sync_mode,
					LIN_overSampMode_t over_sampling);

/**
 * @brief Enable/Disable receiver.
 *
 * @param base LIN module base address.
 * @param enable Enable/Disable receiver.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_enableReceiver(uintptr_t base, bool enable)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	if (enable)
		LIN_writeRegister(base + LIN_US_CR, LIN_US_CR_RXEN_M);
	else
		LIN_writeRegister(base + LIN_US_CR, LIN_US_CR_RXDIS_M);

	return LIN_SUCCESS;
}

/**
 * @brief Enable/Disable transmitter.
 *
 * @param base LIN module base address.
 * @param enable Enable/Disable transmitter.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_enableTransmitter(uintptr_t base, bool enable)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	if (enable)
		LIN_writeRegister(base + LIN_US_CR, LIN_US_CR_TXEN_M);
	else
		LIN_writeRegister(base + LIN_US_CR, LIN_US_CR_TXDIS_M);

	return LIN_SUCCESS;
}

/**
 * @brief Get data form Rx buffer in UART mode.
 * Bit 15 is 0 for data and 1 for command, Bit 0-8 is data.
 *
 * @param base LIN module base address.
 * @param data Pointer to data.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_UartGetCharData(uintptr_t base, uint16_t *data)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*data = (uint16_t)LIN_readRegister(base + LIN_US_RHR);

	return LIN_SUCCESS;
}

/**
 * @brief Put data to Tx buffer in UART mode.
 *
 * @param base LIN module base address.
 * @param data Data to be sent.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_UartPutCharData(uintptr_t base, uint16_t data)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	LIN_writeRegister(base + LIN_US_THR, data);

	return LIN_SUCCESS;
}

/**
 * @brief Get module status.
 *
 * @param base LIN module base address.
 * @param status LIN status(For LIN mode, @LIN_LinChannelStatus_t.
 * For USART mode, @LIN_UsartChannelStatus_t).
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_getStatus(uintptr_t base, uint32_t *status)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*status = LIN_readRegister(base + LIN_US_CSR);

	return LIN_SUCCESS;
}

/**
 * @brief Reset transfer status.
 * Resets the status bits PARE, FRAME, OVRE, MANERR, LINBE, LINISFE,
 * LINIPE, LINCE, LINSNRE, LINSTE, LINHTE, LINID, LINTC, LINBK and RXBRK in US_CSR.
 *
 * @param base LIN module base address.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_resetTransferStatus(uintptr_t base)
{
	uint32_t reg_val;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	reg_val = LIN_readRegister(base + LIN_US_CR);

	reg_val |= LIN_US_CR_RSTSTA_M;

	LIN_writeRegister(base + LIN_US_CR, reg_val);

	return LIN_SUCCESS;
}

/**
 * @brief Module Interrupt Enable.
 * For LIN mode, Please use the @LIN_LinModeInterrupt_t enum.
 * For USART mode, Please use the @LIN_UsartModeInterrupt_t enum.
 *
 * @param base LIN module base address.
 * @param intr interrupt type.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_InterruptEnable(uintptr_t base, uint32_t intr)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	LIN_writeRegister(base + LIN_US_IER, intr);

	return LIN_SUCCESS;
}

/**
 * @brief Module Interrupt Disable.
 * For LIN mode, Please use the @LIN_LinModeInterrupt_t enum.
 * For USART mode, Please use the @LIN_UsartModeInterrupt_t enum.
 *
 * @param base LIN module base address.
 * @param intr interrupt type.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_InterruptDisable(uintptr_t base, uint32_t intr)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	LIN_writeRegister(base + LIN_US_IDR, intr);

	return LIN_SUCCESS;
}

/**
 * @brief Module Interrupt enable Status.
 * For LIN mode, Please use the @LIN_LinModeInterrupt_t enum.
 * For USART mode, Please use the @LIN_UsartModeInterrupt_t enum.
 *
 * @param base LIN module base address.
 * @param status interrupt status(@LIN_LinModeInterrupt_t or @LIN_UsartModeInterrupt_t).
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_getInterrupEnablet(uintptr_t base, uint32_t *status)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*status = LIN_readRegister(base + LIN_US_IMR);

	return LIN_SUCCESS;
}

/**
 * @brief LIN mode initialization.
 *
 * @param base LIN module base address.
 * @param initParam LIN mode initialization parameters.
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_LinModeInit(uintptr_t base, LIN_LinModeInitParam_t *initParam);

/**
 * @brief Sets LIN Node Action.
 *
 * @param base LIN module base address.
 * @param mode node action mode(@LIN_LinModeNodeAction_t).
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_LinSetNodeAction(uintptr_t base, LIN_LinModeNodeAction_t mode)
{
	uint32_t reg_val;

	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	reg_val = LIN_readRegister(base + LIN_US_LINMR);

	reg_val &= ~(LIN_US_LINMR_NACT_M);

	reg_val |= (mode & LIN_US_LINMR_NACT_M);

	LIN_writeRegister(base + LIN_US_LINMR, reg_val);

	return LIN_SUCCESS;
}

/**
 * @brief Sets the frame bit order.
 *
 * @param base LIN module base address.
 * @param bit_order The frame bit order.
 */
static inline void LIN_setBitOrder(uintptr_t base, LIN_bitOrderMode_t bit_order)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (bit_order == LIN_TRANSFER_MSB)
		reg_val |= LIN_US_MR_MSBF_M;
	else
		reg_val &= ~(LIN_US_MR_MSBF_M);

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Sets the SCK output to pin.
 *
 * @param base LIN module base address.
 * @param enable Enable or disable the SCK output to pin.
 */
static inline void LIN_setSCKOutput(uintptr_t base, bool enable)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (enable)
		reg_val |= LIN_US_MR_CLKO_M;
	else
		reg_val &= ~(LIN_US_MR_CLKO_M);

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Get LIN Node Action.
 *
 * @param base LIN module base address.
 * @param mode node action mode(@LIN_LinModeNodeAction_t).
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_LinGetNodeAction(uintptr_t base, LIN_LinModeNodeAction_t *mode)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*mode = (LIN_LinModeNodeAction_t)((LIN_readRegister(base + LIN_US_LINMR) &
										LIN_US_LINMR_NACT_M) >> LIN_US_LINMR_NACT_S);

	return LIN_SUCCESS;
}

/**
 * @brief Sets the over sampling mode.
 *
 * @param base LIN module base address.
 * @param over_sampling The over sampling mode(@LIN_overSampMode_t).
 */
static inline void LIN_setOverSampling(uintptr_t base, LIN_overSampMode_t over_sampling)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (over_sampling == LIN_OVERSAMP_8)
		reg_val |= LIN_US_MR_OVER_M;
	else
		reg_val &= ~(LIN_US_MR_OVER_M);

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Sets the inverted data mode.
 *
 * @param base LIN module base address.
 * @param enable Enable or disable the inverted data mode.
 */
static inline void LIN_setInvertedData(uintptr_t base, bool enable)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (enable)
		reg_val |= LIN_US_MR_INVDATA_M;
	else
		reg_val &= ~(LIN_US_MR_INVDATA_M);

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Receive Line Filter Enable/Disable.
 *
 * @param base LIN module base address.
 * @param enable Enable or disable the receive line filter.
 */
static inline void LIN_ReceiveLineFilterEnable(uintptr_t base, bool enable)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (enable)
		reg_val |= LIN_US_MR_FILTER_M;
	else
		reg_val &= ~(LIN_US_MR_FILTER_M);

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Start Frame Delimiter Type.
 *
 * @param base LIN module base address.
 * @param type The start frame delimiter type.
 */
static inline void LIN_startFrameDeliniter(uintptr_t base, LIN_startFrameBitType_t type)
{
	uint32_t reg_val;

	reg_val = LIN_readRegister(base + LIN_US_MR);

	if (type == LIN_ONEBIT_IS_ONE_BIT)
		reg_val |= LIN_US_MR_ONEBIT_M;
	else
		reg_val &= ~(LIN_US_MR_ONEBIT_M);

	LIN_writeRegister(base + LIN_US_MR, reg_val);
}

/**
 * @brief Get the Identifier character to be transmitted in LIN mode.
 *
 * @param base LIN module base address.
 * @param identifier Points to the LIN ID memory space.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_LinGetIdentifier(uintptr_t base, uint8_t *identifier)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*identifier = (uint8_t)LIN_readRegister(base + LIN_US_LINIR);

	return LIN_SUCCESS;
}

/**
 * @brief Defines the response data length if DLM = 0,
 * in that case the response data length is equal to DLC+1 bytes.
 *
 * @param base LIN module base address.
 * @param length data length(valid value : 1-256).
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_LinSetDataLength(uintptr_t base, uint16_t length);



/**
 * @brief Set the Identifier character to be transmitted in LIN master mode.
 *
 * @param base LIN module base address.
 * @param id Identifier character to be transmitted.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_LinSetID(uintptr_t base, uint8_t id)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	LIN_writeRegister(base + LIN_US_LINIR, id);

	return LIN_SUCCESS;
}

/**
 * @brief Get the Identifier character to be transmitted in LIN mode.
 *
 * @param base LIN module base address.
 * @param id Identifier character to be transmitted.
 */
static inline uint8_t LIN_LinGetID(uintptr_t base)
{
	return (uint8_t)LIN_readRegister(base + LIN_US_LINIR);
}

/**
 * @brief Get the frame length in LIN mode.
 *
 * @param base LIN module base address.
 * @param length frame length.
 * @return int If successful, return 0; otherwise, return an error.
 */
static inline int LIN_LinGetFrameLength(uintptr_t base, uint8_t *length)
{
	if (!LIN_isBaseValid(base))
		return -LIN_INVALID_PARAMTER;

	*length = ((LIN_readRegister(base + LIN_US_LINMR) & LIN_US_LINMR_DLC_M) >>
														LIN_US_LINMR_DLC_S) + 1;

	return LIN_SUCCESS;
}

/**
 * @brief Block received frame in LIN slave mode.
 *
 * @param base LIN module base address.
 * @param msg LIN message.
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_LinSlaveReceiveFrameBlock(uintptr_t base, LIN_LinMsg_t *msg);

/**
 * @brief Block send frame in LIN slave mode.
 *
 * @param base LIN module base address.
 * @param msg LIN message.
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_LinSlaveSendFrameBlock(uintptr_t base, LIN_LinMsg_t *msg);

/**
 * @brief Block send frame in LIN master mode.
 *
 * @param base LIN module base address.
 * @param msg LIN message.
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_LinMasterSendFrameBlock(uintptr_t base, LIN_LinMsg_t *msg);

/**
 * @brief Block receive frame in LIN master mode.
 *
 * @param base LIN module base address.
 * @param msg LIN message.
 * @return int If successful, return 0; otherwise, return an error.
 */
int LIN_LinMasterReceiveFrameBlock(uintptr_t base, LIN_LinMsg_t *msg);

#endif

#endif
