/*
 * pmbus_v30.h
 *
 *  Created on: 2025 Aug 1
 *      Author: Windows11
 */

#ifndef DEVICE_DRIVERLIB_PMBUS_H_
#define DEVICE_DRIVERLIB_PMBUS_H_

#ifdef __cplusplus
extern "C"{
#endif

#include "gs32_version.h"

#include "inc/hw_i2c.h"
#include "inc/hw_pmbus.h"
#include "inc/hw_memmap.h"
#include "inc/hw_types.h"
#include "debug.h"
#include "device.h"

//*****************************************************************************
//
//! Operation Mode Descriptor
//!
//! Used in PMBUS_setOptMode() to set up the module operation. There are two
//! possible modes of operation:
//! -# Master Mode
//! -# SLAVE_Mode
//
//*****************************************************************************
typedef enum{
	PMBUS_MASTER_MODE 		= 0x0U,
	PMBUS_SLAVE_MODE  		= 0x1U,
}PMBUS_optMode_t;

//*****************************************************************************
//
//! Speed Mode Descriptor
//!
//! Used in PMBUS_setSpeedMode() to set up the module speed. There are two
//! possible modes of operation:
//! -# Standard Mode
//! -# Fast Mode
//
//*****************************************************************************
typedef enum{
	PMBUS_STANDARD_MODE 	= 0x1U,
	PMBUS_FAST_MODE  		= 0x2U,
}PMBUS_speedMode_t;

//*****************************************************************************
//
//! Addr Mode Descriptor
//!
//! Used in PMBUS_setAddrMode() to set up the addressing mode. There are four
//! possible modes of operation:
//! -# Slave 7bit Mode
//! -# Slave 10bit Mode
//! -# Master 7bit Mode
//! -# Master 10bit Mode
//
//*****************************************************************************
typedef enum{
	PMBUS_SLAVE_7BIT		= 0x1U,
	PMBUS_SLAVE_10BIT		= 0x2U,
	PMBUS_MASTER_7BIT		= 0x4U,
	PMBUS_MASTER_10BIT		= 0x8U,
}PMBUS_addrMode_t;

//*****************************************************************************
//
//! TX EMPTY Triggering Mode Descriptor
//!
//! Used in PMBUS_setTxEmptyMode() to set up the TX EMPTY triggering mode. There
//! are two possible modes of operation:
//! -# Base Mode
//! -# Enhanced Mode
//
//*****************************************************************************
typedef enum{
	PMBUS_TX_EMPTY_BASIC_TRIGGER		= 0x0U,
	PMBUS_TX_EMPTY_ENHANCED_TRIGGER		= 0x1U,
}PMBUS_txEmptyCtrlMode_t;

//*****************************************************************************
//
//! Device ID Mode Descriptor
//!
//! Used in PMBUS_setDeviceIDMode() to set up the send device id mode. There
//! are four possible modes of operation:
//! -# Send Device ID
//! -# Send General Call
//! -# Send Start Byte
//! -# Send Quick Cmd
//
//*****************************************************************************
typedef enum{
	PMBUS_SAR 				= 0x0U,
	PMBUS_GENERAIL_CALL	= 0x1U,
	PMBUS_START_BYTE		= 0x2U,
	PMBUS_QUICK_CMD		= 0x3U,
}PMBUS_deviceIDMode_t;

//*****************************************************************************
//
//! Duty Cycle Descriptor
//!
//! Used in PMBUS_setBaudRate() to set up the scl duty cycle. There are two
//! possible modes of operation:
//! -# Duty Cycle 33%
//! -# Duty Cycle 50%
//
//*****************************************************************************
typedef enum{
    PMBUS_DUTYCYCLE_33			= 0x0U,
    PMBUS_DUTYCYCLE_50			= 0x1U,
} PMBUS_DutyCycle_t;

//*****************************************************************************
//
//! Slave Addr Descriptor
//!
//! Used in PMBUS_setSMBUSARP() to set up the ARP.Used in PMBUS_setSMBUSPSA()
//! to set up the PSA.Used in PMBUS_setOwnerAddress() to set the slave addr.
//! Used in PMBUS_enableSAR() to enable the slave addr
//! There are four possible modes of operation:
//! -# PMBUS_SLAVER_ADDR1
//! -# PMBUS_SLAVER_ADDR2
//! -# PMBUS_SLAVER_ADDR3
//! -# PMBUS_SLAVER_ADDR4
//
//*****************************************************************************
typedef enum{
	PMBUS_SLAVER_ADDR1			= 0x1U,
	PMBUS_SLAVER_ADDR2			= 0x2U,
	PMBUS_SLAVER_ADDR3			= 0x3U,
	PMBUS_SLAVER_ADDR4			= 0x4U,
}PMBUS_SlaveAddrID_t;

//*****************************************************************************
//
//! Quick Command Descriptor
//!
//! Used in PMBUS_sendQuickCommand() to send a quick command.
//
//*****************************************************************************
typedef enum{
	PMBUS_QUICKCOM_WRITE		= 0x0U,
	PMBUS_QUICKCOM_READ			= 0x1U,
}PMBUS_quickComType_t;

//*****************************************************************************
//
//! Values that can be passed to PMBUS_enableInterrupt() as the \e intFlags
//! parameter,values that can be passed to PMBUS_disableInterrupt() as the
//! \e intFlags,values that can be passed to PMBUS_clearInterruptStatus() as the
//! \e intFlags,returned by PMBUS_getInterruptRawStatus() in the \e intFlags
//! and returned by PMBUS_getInterruptStatus().
//
//*****************************************************************************
typedef enum{
	PMBUS_RX_UNDER				= PMBUS_INTR_STAT_R_RX_UNDER,
	PMBUS_RX_OVER				= PMBUS_INTR_STAT_R_RX_OVER,
	PMBUS_RX_FULL				= PMBUS_INTR_STAT_R_RX_FULL,
	PMBUS_TX_OVER				= PMBUS_INTR_STAT_R_TX_OVER,
	PMBUS_TX_EMPTY				= PMBUS_INTR_STAT_R_TX_EMPTY,
	PMBUS_RD_REQ				= PMBUS_INTR_STAT_R_RD_REQ,
	PMBUS_TX_ABRT				= PMBUS_INTR_STAT_R_TX_ABRT,
	PMBUS_RX_DONE				= PMBUS_INTR_STAT_R_RX_DONE,
	PMBUS_INTR_ACTIVITY			= PMBUS_INTR_STAT_R_ACTIVITY,
	PMBUS_STOP_DET				= PMBUS_INTR_STAT_R_STOP_DET,
	PMBUS_START_DET				= PMBUS_INTR_STAT_R_START_DET,
	PMBUS_GEN_CALL				= PMBUS_INTR_STAT_R_GEN_CALL,
	PMBUS_RESTART_DET			= PMBUS_INTR_STAT_R_RESTART_DET,
	PMBUS_MASTER_ON_HOLD		= PMBUS_INTR_STAT_R_MASTER_ON_HOLD,
	PMBUS_SCL_STUCK_AT_LOW		= PMBUS_INTR_STAT_R_SCL_STUCK_AT_LOW,
	PMBUS_WR_REQ				= PMBUS_INTR_STAT_R_WR_REQ,
	PMBUS_SLV_ADDR1_TAG			= PMBUS_INTR_STAT_R_SLV_ADDR1_TAG,
	PMBUS_SLV_ADDR2_TAG			= PMBUS_INTR_STAT_R_SLV_ADDR2_TAG,
	PMBUS_SLV_ADDR3_TAG			= PMBUS_INTR_STAT_R_SLV_ADDR3_TAG,
	PMBUS_SLV_ADDR4_TAG			= PMBUS_INTR_STAT_R_SLV_ADDR4_TAG,
	PMBUS_INT_ALL				= PMBUS_INTR_STAT_R_ALL,
}PMBUS_InterruptRawStatus_t;

//*****************************************************************************
//
//! Values that can be passed to PMBUS_setFIFOInterruptLevel() as the \e txLevel
//! parameter,and vaules that can be passed to PMBUS_getFIFOInterruptLevel() as the
//! \e txLevel parameter,
//
//*****************************************************************************
typedef enum{
    PMBUS_FIFO_LEVEL_TX2        = 0x0001U,
    PMBUS_FIFO_LEVEL_TX1        = 0x0000U,
    PMBUS_FIFO_LEVEL_TX3        = 0x0002U,
    PMBUS_FIFO_LEVEL_TX4        = 0x0003U,
    PMBUS_FIFO_LEVEL_TX5        = 0x0004U,
    PMBUS_FIFO_LEVEL_TX6        = 0x0005U,
    PMBUS_FIFO_LEVEL_TX7        = 0x0006U,
    PMBUS_FIFO_LEVEL_TX8        = 0x0007U,
    PMBUS_FIFO_LEVEL_TX9        = 0x0008U,
    PMBUS_FIFO_LEVEL_TX10       = 0x0009U,
    PMBUS_FIFO_LEVEL_TX11       = 0x000AU,
    PMBUS_FIFO_LEVEL_TX12       = 0x000BU,
    PMBUS_FIFO_LEVEL_TX13       = 0x000CU,
    PMBUS_FIFO_LEVEL_TX14       = 0x000DU,
    PMBUS_FIFO_LEVEL_TX15       = 0x000EU,
    PMBUS_FIFO_LEVEL_TX16       = 0x000FU,
} PMBUS_TxFIFOLevel_t;

//*****************************************************************************
//
//! Values that can be passed to PMBUS_setFIFOInterruptLevel() as the \e rxLevel
//! parameter,and vaules that can be passed to PMBUS_getFIFOInterruptLevel() as the
//! \e rxLevel parameter,
//
//*****************************************************************************
typedef enum{
    PMBUS_FIFO_LEVEL_RX1        = 0x0000U,
    PMBUS_FIFO_LEVEL_RX2        = 0x0001U,
    PMBUS_FIFO_LEVEL_RX3        = 0x0002U,
    PMBUS_FIFO_LEVEL_RX4        = 0x0003U,
    PMBUS_FIFO_LEVEL_RX5        = 0x0004U,
    PMBUS_FIFO_LEVEL_RX6        = 0x0005U,
    PMBUS_FIFO_LEVEL_RX7        = 0x0006U,
    PMBUS_FIFO_LEVEL_RX8        = 0x0007U,
    PMBUS_FIFO_LEVEL_RX9        = 0x0008U,
    PMBUS_FIFO_LEVEL_RX10       = 0x0009U,
    PMBUS_FIFO_LEVEL_RX11       = 0x000AU,
    PMBUS_FIFO_LEVEL_RX12       = 0x000BU,
    PMBUS_FIFO_LEVEL_RX13       = 0x000CU,
    PMBUS_FIFO_LEVEL_RX14       = 0x000DU,
    PMBUS_FIFO_LEVEL_RX15       = 0x000EU,
    PMBUS_FIFO_LEVEL_RX16       = 0x000FU,
} PMBUS_RxFIFOLevel_t;

//*****************************************************************************
//
//! Returned by PMBUS_getStatus()
//
//*****************************************************************************
typedef enum{
	PMBUS_ACTIVITY							= PMBUS_STATUS_ACTIVITY,
	PMBUS_TFNF								= PMBUS_STATUS_TFNF,
	PMBUS_TFE								= PMBUS_STATUS_TFE,
	PMBUS_RFBE								= PMBUS_STATUS_RFNE,
	PMBUS_RFF								= PMBUS_STATUS_RFF,
	PMBUS_MST_ACTIVITY						= PMBUS_STATUS_MST_ACTIVITY,
	PMBUS_SLV_ACTIVITY						= PMBUS_STATUS_SLV_ACTIVITY,
	PMBUS_MST_HOLD_TX_FIFO_EMPTY			= PMBUS_STATUS_MST_HOLD_TX_FIFO_EMPTY,
	PMBUS_MST_HOLD_RX_FIFO_FULL				= PMBUS_STATUS_MST_HOLD_RX_FIFO_FULL,
	PMBUS_SLV_HOLD_TX_FIFO_EMPTY			= PMBUS_STATUS_SLV_HOLD_TX_FIFO_EMPTY,
	PMBUS_SLV_HOLD_RX_FIFO_FULL				= PMBUS_STATUS_SLV_HOLD_RX_FIFO_FULL,
	PMBUS_SDA_STUCK_NOT_RECOVERED			= PMBUS_STATUS_SDA_STUCK_NOT_RECOVERED,
	PMBUS_SLV_ISO_SAR_DATA_CLK_STRETCH		= PMBUS_STATUS_SLV_ISO_SAR_DATA_CLK_STRETCH,
	PMBUS_SMBUS_QUICK_CMD_BIT				= PMBUS_STATUS_SMBUS_QUICK_CMD_BIT,
	PMBUS_SMBUS_SLAVE_ADDR_VALID			= PMBUS_STATUS_SMBUS_SLAVE_ADDR_VALID,
	PMBUS_SMBUS_SLAVE_ADDR_RESOLVED			= PMBUS_STATUS_SMBUS_SLAVE_ADDR_RESOLVED,
	PMBUS_SMBUS_SUSPEND_STATUS				= PMBUS_STATUS_SMBUS_SUSPEND_STATUS,
	PMBUS_SMBUS_ALERT_STATUS				= PMBUS_STATUS_SMBUS_ALERT_STATUS,
	PMBUS_SMBUS_SLAVE_ADDR2_VALID			= PMBUS_STATUS_SMBUS_SLAVE_ADDR2_VALID,
	PMBUS_SMBUS_SLAVE_ADDR2_RESOLVED		= PMBUS_STATUS_SMBUS_SLAVE_ADDR2_RESOLVED,
	PMBUS_SMBUS_SLAVE_ADDR3_VALID			= PMBUS_STATUS_SMBUS_SLAVE_ADDR3_VALID,
	PMBUS_SMBUS_SLAVE_ADDR3_RESOLVED		= PMBUS_STATUS_SMBUS_SLAVE_ADDR3_RESOLVED,
	PMBUS_SMBUS_SLAVE_ADDR4_VALID			= PMBUS_STATUS_SMBUS_SLAVE_ADDR4_VALID,
	PMBUS_SMBUS_SLAVE_ADDR4_RESOLVED		= PMBUS_STATUS_SMBUS_SLAVE_ADDR4_RESOLVED,
}PMBUS_Status_t;

//*****************************************************************************
//
//! Returned by PMBUS_getTxFIFOStatus()
//
//*****************************************************************************
typedef enum{
    PMBUS_FIFO_TXEMPTY    = 0x0000U,
    PMBUS_FIFO_TX0        = 0x0000U,
    PMBUS_FIFO_TX1        = 0x0001U,
    PMBUS_FIFO_TX2        = 0x0002U,
    PMBUS_FIFO_TX3        = 0x0003U,
    PMBUS_FIFO_TX4        = 0x0004U,
    PMBUS_FIFO_TX5        = 0x0005U,
    PMBUS_FIFO_TX6        = 0x0006U,
    PMBUS_FIFO_TX7        = 0x0007U,
    PMBUS_FIFO_TX8        = 0x0008U,
    PMBUS_FIFO_TX9        = 0x0009U,
    PMBUS_FIFO_TX10       = 0x000AU,
    PMBUS_FIFO_TX11       = 0x000BU,
    PMBUS_FIFO_TX12       = 0x000CU,
    PMBUS_FIFO_TX13       = 0x000DU,
    PMBUS_FIFO_TX14       = 0x000EU,
    PMBUS_FIFO_TX15       = 0x000FU,
    PMBUS_FIFO_TX16       = 0x0010U,
    PMBUS_FIFO_TXFULL     = 0x0010U
}PMBUS_TxFIFOStatus_t;

//*****************************************************************************
//
//! Returned by PMBUS_getRxFIFOStatus()
//
//*****************************************************************************
typedef enum{
    PMBUS_FIFO_RXEMPTY    = 0x0000U,
    PMBUS_FIFO_RX0        = 0x0000U,
    PMBUS_FIFO_RX1        = 0x0001U,
    PMBUS_FIFO_RX2        = 0x0002U,
    PMBUS_FIFO_RX3        = 0x0003U,
    PMBUS_FIFO_RX4        = 0x0004U,
    PMBUS_FIFO_RX5        = 0x0005U,
    PMBUS_FIFO_RX6        = 0x0006U,
    PMBUS_FIFO_RX7        = 0x0007U,
    PMBUS_FIFO_RX8        = 0x0008U,
    PMBUS_FIFO_RX9        = 0x0009U,
    PMBUS_FIFO_RX10       = 0x000AU,
    PMBUS_FIFO_RX11       = 0x000BU,
    PMBUS_FIFO_RX12       = 0x000CU,
    PMBUS_FIFO_RX13       = 0x000DU,
    PMBUS_FIFO_RX14       = 0x000EU,
    PMBUS_FIFO_RX15       = 0x000FU,
    PMBUS_FIFO_RX16       = 0x0010U,
    PMBUS_FIFO_RXFULL     = 0x0010U
}PMBUS_RxFIFOStatus_t;

//*****************************************************************************
//
//! Returned by PMBUS_getTxAbrtSource()
//
//*****************************************************************************
typedef enum{
	PMBUS_ABRT_7B_ADDR_NOACK			= PMBUS_TX_ABRT_SOURCE_ABRT_7B_ADDR_NOACK,
	PMBUS_ABRT_10ADDR1_NOCAK			= PMBUS_TX_ABRT_SOURCE_ABRT_10ADDR1_NOACK,
	PMBUS_ABRT_10ADDR2_NOACK			= PMBUS_TX_ABRT_SOURCE_ABRT_10ADDR2_NOACK,
	PMBUS_ABRT_TXDATA_NOACK				= PMBUS_TX_ABRT_SOURCE_ABRT_TXDATA_NOACK,
	PMBUS_ABRT_GCALL_NOACK				= PMBUS_TX_ABRT_SOURCE_ABRT_GCALL_NOACK,
	PMBUS_ABRT_GCALL_READ				= PMBUS_TX_ABRT_SOURCE_ABRT_GCALL_READ,
	PMBUS_ABRT_HS_ACKDET				= PMBUS_TX_ABRT_SOURCE_ABRT_HS_ACKDET,
	PMBUS_ABRT_SBYTE_ACKDET				= PMBUS_TX_ABRT_SOURCE_ABRT_SBYTE_ACKDET,
	PMBUS_ABRT_HS_NORSTRT				= PMBUS_TX_ABRT_SOURCE_ABRT_HS_NORSTRT,
	PMBUS_ABRT_SBYTE_NORSTRT			= PMBUS_TX_ABRT_SOURCE_ABRT_SBYTE_NORSTRT,
	PMBUS_ABRT_10B_RD_NORSTRT			= PMBUS_TX_ABRT_SOURCE_ABRT_10B_RD_NORSTRT,
	PMBUS_ABRT_MASTER_DIS				= PMBUS_TX_ABRT_SOURCE_ABRT_MASTER_DIS,
	PMBUS_ABRT_LOST						= PMBUS_TX_ABRT_SOURCE_ARB_LOST,
	PMBUS_ABRT_SLVFLUSH_TXFIFO			= PMBUS_TX_ABRT_SOURCE_ABRT_SLVFLUSH_TXFIFO,
	PMBUS_ABRT_SLV_ARBLOST				= PMBUS_TX_ABRT_SOURCE_ABRT_SLV_ARBLOST,
	PMBUS_ABRT_SLVRD_INTX				= PMBUS_TX_ABRT_SOURCE_ABRT_SLVRD_INTX,
	PMBUS_ABRT_USER_ABRT				= PMBUS_TX_ABRT_SOURCE_ABRT_USER_ABRT,
	PMBUS_ABRT_SDA_STUCK_AT_LOW			= PMBUS_TX_ABRT_SOURCE_ABRT_SDA_STUCK_AT_LOW,
	PMBUS_ABRT_DEVICE_NOACK				= PMBUS_TX_ABRT_SOURCE_ABRT_DEVICE_NOACK,
	PMBUS_ABRT_DEVICE_SLVADDR_NOACK		= PMBUS_TX_ABRT_SOURCE_ABRT_DEVICE_SLVADDR_NOACK,
	PMBUS_ABRT_DEVICE_WRITE				= PMBUS_TX_ABRT_SOURCE_ABRT_DEVICE_WRITE,
}PMBUS_TxAbrtSource_t;

//*****************************************************************************
//
//! Returned by PMBUS_setTxDMALevel()
//
//*****************************************************************************
typedef enum{
	PMBUS_TXDMA_LEVEL0		= 0U,
	PMBUS_TXDMA_LEVEL1		= 1U,
	PMBUS_TXDMA_LEVEL2		= 2U,
	PMBUS_TXDMA_LEVEL3		= 3U,
	PMBUS_TXDMA_LEVEL4		= 4U,
	PMBUS_TXDMA_LEVEL5		= 5U,
	PMBUS_TXDMA_LEVEL6		= 6U,
	PMBUS_TXDMA_LEVEL7		= 7U,
	PMBUS_TXDMA_LEVEL8		= 8U,
	PMBUS_TXDMA_LEVEL9		= 9U,
	PMBUS_TXDMA_LEVEL10		= 10U,
	PMBUS_TXDMA_LEVEL11		= 11U,
	PMBUS_TXDMA_LEVEL12		= 12U,
	PMBUS_TXDMA_LEVEL13		= 13U,
	PMBUS_TXDMA_LEVEL14		= 14U,
	PMBUS_TXDMA_LEVEL15		= 15U,
	PMBUS_TXDMA_LEVEL16		= 16U,
}PMBUS_TxDMALevel_t;

//*****************************************************************************
//
//! Returned by PMBUS_setRxDMALevel()
//
//*****************************************************************************
typedef enum{
	PMBUS_RXDMA_LEVEL1		= 0U,
	PMBUS_RXDMA_LEVEL2		= 1U,
	PMBUS_RXDMA_LEVEL3		= 2U,
	PMBUS_RXDMA_LEVEL4		= 3U,
	PMBUS_RXDMA_LEVEL5		= 4U,
	PMBUS_RXDMA_LEVEL6		= 5U,
	PMBUS_RXDMA_LEVEL7		= 6U,
	PMBUS_RXDMA_LEVEL8		= 7U,
	PMBUS_RXDMA_LEVEL9		= 8U,
	PMBUS_RXDMA_LEVEL10		= 9U,
	PMBUS_RXDMA_LEVEL11		= 10U,
	PMBUS_RXDMA_LEVEL12		= 11U,
	PMBUS_RXDMA_LEVEL13		= 12U,
	PMBUS_XDMA_LEVEL14		= 13U,
	PMBUS_RXDMA_LEVEL15		= 14U,
	PMBUS_RXDMA_LEVEL16		= 15U,
}PMBUS_RxDMALevel_t;

//*****************************************************************************
//
//! Values that can be passed to PMBUS_enableSMBUSInterrupt() as the \e intFlags
//! parameter,values that can be passed to PMBUS_disableSMBUSInterrupt() as the
//! \e intFlags,values that can be passed to PMBUS_clearSMBUSInterrupt() as the
//! \e intFlags,returned by PMBUS_getSMBUSInterruptRawStatus() in the \e intFlags
//! and returned by PMBUS_getSMBUSInterruptStatus().
//
//*****************************************************************************
typedef enum{
	PMBUS_SMBUS_SLV_CLOCK_EXTND_TIMEOUT		= PMBUS_SMBUS_RAW_INTR_STAT_SLV_CLOCK_EXTND_TIMEOUT,
	PMBUS_SMBUS_MST_CLOCK_EXTND_TIMEOUT		= PMBUS_SMBUS_RAW_INTR_STAT_MST_CLOCK_EXTND_TIMEOUT,
	PMBUS_SMBUS_QUICK_CMD_DET				= PMBUS_SMBUS_RAW_INTR_STAT_QUICK_CMD_DET,
	PMBUS_SMBUS_HOST_NTFY_MST_DET			= PMBUS_SMBUS_RAW_INTR_STAT_HOST_NTFY_MST_DET,
	PMBUS_SMBUS_ARP_PREPARE_CMD_DET			= PMBUS_SMBUS_RAW_INTR_STAT_ARP_PREPARE_CMD_DET,
	PMBUS_SMBUS_ARP_RST_CMD_DET				= PMBUS_SMBUS_RAW_INTR_STAT_ARP_RST_CMD_DET,
	PMBUS_SMBUS_ARP_GET_UDID_CMD_DET		= PMBUS_SMBUS_RAW_INTR_STAT_ARP_GET_UDID_CMD_DET,
	PMBUS_SMBUS_ASSGN_ADDR_CMD_DET			= PMBUS_SMBUS_RAW_INTR_STAT_ARP_ASSGN_ADDR_CMD_DET,
	PMBUS_SMBUS_RX_PEC_NACK					= PMBUS_SMBUS_RAW_INTR_STAT_SLV_RX_PEC_NACK,
	PMBUS_SMBUS_SUSPEND_DET					= PMBUS_SMBUS_RAW_INTR_STAT_SMBUS_SUSPEND_DET,
	PMBUS_SMBUS_ALERT_DET					= PMBUS_SMBUS_RAW_INTR_STAT_SMBUS_ALERT_DET,
	PMBUS_SMBUS_ALL_INT						= PMBUS_SMBUS_RAW_INTR_STAT_ALL,
}PMBUS_SMBUSInterruptRawStatus_t;

__STATIC_INLINE bool PMBUS_isBaseValid(uint32_t base)
{
    return (base == PMBUSA_BASE);
}

static inline void PMBUS_writeRegister(uintptr_t base, uint32_t value)
{
	*(volatile uint32_t *)(base) = value;
}

static inline uint32_t PMBUS_readRegister(uintptr_t base)
{
	return *(volatile uint32_t *)(base);
}

//*****************************************************************************
//
//! The operation mode of the setting module.
//!
//! \param base is the base address of the PMBus instance used.
//! \param mode is is used to configure the operation mode of the module.

//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setOptMode(uint32_t base,PMBUS_optMode_t mode)
{

	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~(PMBUS_CON_MASTER_MODE|PMBUS_CON_IC_SLAVE_DISABLE);

	if(mode == PMBUS_MASTER_MODE)
		reg_val |= (PMBUS_CON_MASTER_MODE|PMBUS_CON_IC_SLAVE_DISABLE);

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);

}

//*****************************************************************************
//
//! The speed of setting up the module.
//!
//! \param base is the base address of the PMBus instance used.
//! \param mode is set the module speed.
//!
//! \return None.
//
//*****************************************************************************
void PMBUS_setSpeedMode(uint32_t base,PMBUS_speedMode_t mode);

//*****************************************************************************
//
//! Get the speed of the module.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The speed mode of this module.
//
//*****************************************************************************
static inline PMBUS_speedMode_t PMBUS_getSpeedMode(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val = ((reg_val&PMBUS_CON_SPEED_M)>>PMBUS_CON_SPEED_S);

	return (PMBUS_speedMode_t)reg_val;
}

//*****************************************************************************
//
//! Set the address mode of this module.
//!
//! \param base is the base address of the PMBus instance used.
//! \param mode is the address mode of this module.
//!
//! \return None.
//
//*****************************************************************************
void PMBUS_setAddrMode(uint32_t base,PMBUS_addrMode_t mode);

//*****************************************************************************
//
//! Set the restart mode of this module.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the restart function
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setRestart(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_IC_RESTART_EN;

	if(enable)
		reg_val |= PMBUS_CON_IC_RESTART_EN;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);

}

//*****************************************************************************
//
//! Set the restart mode of this module.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the stop filter function
//!
//!	In slave mode, when the stop filter is enabled, the STOP_DET interrupt
//! signal will only be issued upon receiving the corresponding slave address.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setStopFilter(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_STOP_DET_IFADDRESSED;

	if(enable)
		reg_val |= PMBUS_CON_STOP_DET_IFADDRESSED;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Set the trigger mode for transmit empty.
//!
//! \param base is the base address of the PMBus instance used.
//! \param mode is set the trigger mode for transmit empty.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setTxEmptyMode(uint32_t base,PMBUS_txEmptyCtrlMode_t mode)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_TX_EMPTY_CTRL;

	if(mode == PMBUS_TX_EMPTY_ENHANCED_TRIGGER)
		reg_val |= PMBUS_CON_TX_EMPTY_CTRL;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Set the restart mode of this module.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the hold bus on receive fifo is full.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setRxFIFOFullHoldBus(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_RX_FIFO_FULL_HLD_CTRL;

	if(enable)
		reg_val |= PMBUS_CON_RX_FIFO_FULL_HLD_CTRL;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! The mode that generates STOP_DET is set in the master mode.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE in master mode, the STOP_DET interrupt
//! 			  will only be generated when in active mode.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setStopDetIfMasterActive(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_STOP_DET_IF_MASTER_ACTIVE;

	if(enable)
		reg_val |= PMBUS_CON_STOP_DET_IF_MASTER_ACTIVE;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Configure the bus clearing function.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the bus clearing function.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setBusClear(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_BUS_CLEAR_FEATURE_CTRL;

	if(enable)
		reg_val |= PMBUS_CON_BUS_CLEAR_FEATURE_CTRL;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Enables the usage of IC_OPTIONAL_SAR register.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the usage of IC_OPTIONAL_SAR register.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setExtraAddr(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_OPTIONAL_SAR_CTRL;

	if(enable)
		reg_val |= PMBUS_CON_OPTIONAL_SAR_CTRL;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Set to receive Quick commands signals in slave mode
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE receive Quick commands signals in slave mode.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSSlaveQuick(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~PMBUS_CON_SMBUS_SLAVE_QUICK_EN;

	if(enable)
		reg_val |= PMBUS_CON_SMBUS_SLAVE_QUICK_EN;

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Enable the ARP function by setting the corresponding SAR.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the ARP function by setting the corresponding
//!   			  SAR ID.
//! \param sar_num Corresponding SAR ID
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSARP(uint32_t base,bool enable,PMBUS_SlaveAddrID_t sar_num)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	if(sar_num == 1){
		reg_val &= ~PMBUS_CON_SMBUS_ARP_EN;
		if(enable)
			reg_val |= PMBUS_CON_SMBUS_ARP_EN;
	}else {
		reg_val &= ~(PMBUS_CON_IC_SAR2_SMBUS_ARP_EN << (sar_num-2));
		if(enable)
			reg_val |= (PMBUS_CON_IC_SAR2_SMBUS_ARP_EN << (sar_num-2));
	}

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Enable the PSA function by setting the corresponding SAR.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the PSA function by setting the corresponding
//!   			  SAR ID.
//! \param sar_num Corresponding SAR ID
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSPSA(uint32_t base,bool enable,PMBUS_SlaveAddrID_t sar_num)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_CON);

	reg_val &= ~(PMBUS_CON_SMBUS_PERSISTENT_SLV_ADDR_EN << (sar_num - 1));

	if(enable)
		reg_val |= (PMBUS_CON_SMBUS_PERSISTENT_SLV_ADDR_EN << (sar_num - 1));

	PMBUS_writeRegister(base + PMBUS_O_IC_CON, reg_val);
}

//*****************************************************************************
//
//! Set the target address
//!
//! \param base is the base address of the PMBus instance used.
//! \param address : Target Address to be configured.
//!
//! This function configures the current device address, this
//! will be the target address.
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setTargetAddress(uint32_t base, uint16_t ownerAddr)
{

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_TAR);

	reg_val &= ~PMBUS_TAR_IC_TAR_M;

	reg_val |= ((ownerAddr<<PMBUS_TAR_IC_TAR_S)&PMBUS_TAR_IC_TAR_M);

	PMBUS_writeRegister(base + PMBUS_O_IC_TAR, reg_val);
}

//*****************************************************************************
//
//! Set the mode for sending the device ID.
//!
//! \param base is the base address of the PMBus instance used.
//! \param mode is set the mode for sending the device ID.
//!
//! \return None.
//
//*****************************************************************************
void PMBUS_setDeviceIDMode(uint32_t base,PMBUS_deviceIDMode_t mode);

//*****************************************************************************
//
//! Set the 10bit address transmission in the master mode.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the 10bit address transmission in the master mode.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setMaster10BitAddrTransmit(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_TAR);

	reg_val &= ~PMBUS_TAR_IC_10BITADDR_MASTER;

	if(enable)
		reg_val |= PMBUS_TAR_IC_10BITADDR_MASTER;

	PMBUS_writeRegister(base + PMBUS_O_IC_TAR, reg_val);
}

//*****************************************************************************
//
//! Set the current device address
//!
//! \param base is the base address of the PMBus instance used.
//! \param address : Address to be configured.
//!
//! This function configures the current device address, this
//! will be the own address of the module.
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setOwnerAddress(uint32_t base, uint16_t targetAddr,PMBUS_SlaveAddrID_t sar_id)
{
	ASSERT(PMBUS_isBaseValid(base));

	if(sar_id == PMBUS_SLAVER_ADDR1){
		PMBUS_writeRegister(base + PMBUS_O_IC_SAR, targetAddr);
	}else if(sar_id == PMBUS_SLAVER_ADDR2){
		PMBUS_writeRegister(base + PMBUS_O_IC_SAR2, targetAddr);
	}else if(sar_id == PMBUS_SLAVER_ADDR3){
		PMBUS_writeRegister(base + PMBUS_O_IC_SAR3, targetAddr);
	}else if(sar_id == PMBUS_SLAVER_ADDR4){
		PMBUS_writeRegister(base + PMBUS_O_IC_SAR4, targetAddr);
	}
}

//*****************************************************************************
//
//! Set the master ID in the high-speed mode
//!
//! \param base is the base address of the PMBus instance used.
//! \param id : the master ID in the high-speed mode
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setHSMasterID(uint32_t base,uint8_t id)
{
	ASSERT(PMBUS_isBaseValid(base));

	if(id > 7)
		id = 7;

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_HS_MADDR);

	reg_val = (id<<PMBUS_HS_MADDR_S);

	PMBUS_writeRegister(base + PMBUS_O_IC_HS_MADDR, reg_val);
}

//*****************************************************************************
//
//! Get the received first data status
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return  Received first data status.Return true when the first data is received.
//
//*****************************************************************************
static inline int PMBUS_getFirstDataState(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_DATA_CMD);

	return (reg_val & PMBUS_DATA_CMD_FIRST_DATA_BYTE);
}

//*****************************************************************************
//
//! Send the data to be written.
//!
//! \param base is the base address of the PMBus instance used.
//! \param data is the information to be written.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdWrite(uint32_t base,uint8_t data)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= ((data<<PMBUS_DATA_CMD_DAT_S)&PMBUS_DATA_CMD_DAT_M);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the data to be written and the STOP signal.
//!
//! \param base is the base address of the PMBus instance used.
//! \param data is the information to be written.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdWriteAndStop(uint32_t base,uint8_t data)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= (PMBUS_DATA_CMD_STOP|((data<<PMBUS_DATA_CMD_DAT_S)&PMBUS_DATA_CMD_DAT_M));

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the data to be written and the STOP signal.
//!
//! \param base is the base address of the PMBus instance used.
//! \param data is the information to be written.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdRestartAndWrite(uint32_t base,uint8_t data)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= (PMBUS_DATA_CMD_RESTART|((data<<PMBUS_DATA_CMD_DAT_S)&PMBUS_DATA_CMD_DAT_M));

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the data to be written, the STOP signal and the START signal.
//!
//! \param base is the base address of the PMBus instance used.
//! \param data is the information to be written.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdRestartAndWriteAndStop(uint32_t base,uint8_t data)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= (PMBUS_DATA_CMD_RESTART|((data<<PMBUS_DATA_CMD_DAT_S)&PMBUS_DATA_CMD_DAT_M)\
				|PMBUS_DATA_CMD_STOP);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the read command and the START signal.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdRestartAndRead(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= (PMBUS_DATA_CMD_RESTART|PMBUS_DATA_CMD_CMD);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the read command and the STOP signal.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdReadAndStop(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= (PMBUS_DATA_CMD_STOP|PMBUS_DATA_CMD_CMD);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the read command, the START signal and the STOP signal.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdRestartAndReadAndStop(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= (PMBUS_DATA_CMD_STOP|PMBUS_DATA_CMD_CMD|PMBUS_DATA_CMD_RESTART);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send the read command.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendCmdRead(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= PMBUS_DATA_CMD_CMD;

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Get the received data.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return data refers to the received data.
//
//*****************************************************************************
static inline uint8_t PMBUS_getData(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base + PMBUS_O_IC_DATA_CMD);

	return ((reg_val>>PMBUS_DATA_CMD_DAT_S)&PMBUS_DATA_CMD_DAT_M);

}

//*****************************************************************************
//
//! Send the data to be written.
//!
//! \param base is the base address of the PMBus instance used.
//! \param data is the information to be written.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_putData(uint32_t base, uint32_t data)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	reg_val |= ((data<<PMBUS_DATA_CMD_DAT_S)&PMBUS_DATA_CMD_DAT_M);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Send a Quick Command.
//!
//! \param base is the base address of the PMBus instance used.
//! \param command is quick command type.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendQuickCommand(uint32_t base, PMBUS_quickComType_t command)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = 0;

	if(command == PMBUS_QUICKCOM_WRITE)
		reg_val |= (PMBUS_DATA_CMD_RESTART|PMBUS_DATA_CMD_STOP);
	else if(command == PMBUS_QUICKCOM_READ)
		reg_val |= (PMBUS_DATA_CMD_RESTART|PMBUS_DATA_CMD_STOP|PMBUS_DATA_CMD_CMD);

	PMBUS_writeRegister(base + PMBUS_O_IC_DATA_CMD, reg_val);
}

//*****************************************************************************
//
//! Set length of the longest spike that is filtered out by the spike
//! suppression logic when the component is operating in SS, FS or FM+ modes.
//!
//! \param base is the base address of the PMBus instance used.
//! \param spklen of the longest spike that is filtered out by the spike
//!  suppression logic when the component is operating in SS, FS or FM+ modes.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setFSSpkLen(uint32_t base, uint8_t spklen)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base + PMBUS_O_IC_FS_SPKLEN, spklen);
}

//*****************************************************************************
//
//! Get length of the longest spike that is filtered out by the spike
//! suppression logic when the component is operating in SS, FS or FM+ modes.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return number of the longest spike that is filtered out by the spike
//! suppression logic when the component is operating in SS, FS or FM+ modes.
//
//*****************************************************************************
static inline int PMBUS_getFSSpkLen(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_FS_SPKLEN);

	return ((reg_val & PMBUS_FS_SPKLEN_IC_FS_SPKLEN_M) >> PMBUS_FS_SPKLEN_IC_FS_SPKLEN_S);
}


//*****************************************************************************
//
//! Set the baud rate of the PMBUS bus.
//!
//! \param base is the base address of the PMBus instance used.
//! \param sysclkHz is the rate of the clock supplied to the PMBUS module (SYSCLK) in Hz.
//! \param bitRate is the rate of the controller clock signal, SCL.
//! \param dutyCycle is duty cycle of the SCL signal.
//!
//! This function initializes operation of the PMBUS Controller by configuring the
//! bus speed for the controller. Note that the PMBUS module \b must be put into
//! reset before calling this function. You can do this with the function
//! PMBUS_disableModule().
//!
//!	That clock is then divided down further to configure the SCL signal to run at the
//! rate specified by \e bitRate. The \e dutyCycle parameter determines the percentage
//!  of time high and time low on the clock signal. The valid values are
//!  \b I2C_DUTYCYCLE_33 for 33% and \b I2C_DUTYCYCLE_50 for 50%.
//!
//! \return None.
//
//*****************************************************************************

void PMBUS_setBaudRate(uint32_t base, uint32_t sysclkHz, uint32_t bitRate,
						PMBUS_DutyCycle_t dutyCycle);


static inline PMBUS_InterruptRawStatus_t PMBUS_getInterruptRawStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_RAW_INTR_STAT);

	return (PMBUS_InterruptRawStatus_t)reg_val;
}

static inline PMBUS_InterruptRawStatus_t PMBUS_getInterruptStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_INTR_STAT);

	return (PMBUS_InterruptRawStatus_t)reg_val;
}

//*****************************************************************************
//
//! Enables PMBus interrupt sources.
//!
//! \param base is the base address of the PMBus instance used.
//! \param intFlags is the bit mask of the interrupt sources to be enabled.
//!
//! This function enables the indicated PMBus interrupt sources.  Only the
//! sources that are enabled can be reflected to the processor interrupt.
//! Disabled sources have no effect on the processor.
//!
//! The \e intFlags parameter is the logical OR of any of the following:
//!
//! - \b PMBUS_RX_UNDER
//! - \b PMBUS_RX_OVER
//! - \b PMBUS_RX_FULL
//! - \b PMBUS_TX_OVER
//! - \b PMBUS_TX_EMPTY
//! - \b PMBUS_RD_REQ
//! - \b PMBUS_TX_ABRT
//! - \b PMBUS_RX_DONE
//! - \b PMBUS_INTR_ACTIVITY
//! - \b PMBUS_STOP_DET
//! - \b PMBUS_START_DET
//! - \b PMBUS_GEN_CALL
//! - \b PMBUS_RESTART_DET
//! - \b PMBUS_MASTER_ON_HOLD
//! - \b PMBUS_SCL_STUCK_AT_LOW
//! - \b PMBUS_WR_REQ
//! - \b PMBUS_SLV_ADDR1_TAG
//! - \b PMBUS_SLV_ADDR2_TAG
//! - \b PMBUS_SLV_ADDR3_TAG
//! - \b PMBUS_SLV_ADDR4_TAG
//! - \b PMBUS_INT_ALL
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableInterrupt(uint32_t base,PMBUS_InterruptRawStatus_t intFlags)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_INTR_MASK);

	reg_val |= intFlags;

	PMBUS_writeRegister(base+PMBUS_O_IC_INTR_MASK,reg_val);
}

//*****************************************************************************
//
//! Disables PMBus interrupt sources.
//!
//! \param base is the base address of the PMBus instance used.
//! \param intFlags is the bit mask of the interrupt sources to be disabled.
//!
//! This function disables the indicated PMBus interrupt sources.  Only
//! the sources that are enabled can be reflected to the processor interrupt.
//! Disabled sources have no effect on the processor.
//!
//! The \e intFlags parameter has the same definition as the \e intFlags
//! parameter to PMBUS_enableInterrupt().
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_disableInterrupt(uint32_t base,PMBUS_InterruptRawStatus_t intFlags)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_INTR_MASK);

	reg_val &= (~intFlags);

	PMBUS_writeRegister(base+PMBUS_O_IC_INTR_MASK,reg_val);
}

//*****************************************************************************
//
//! Clear PMBus interrupt status.
//!
//! \param base is the base address of the PMBus instance used.
//! \param intFlags is the bit mask of the interrupt status to be clear.
//!
//! This function clear the indicated PMBus interrupt status.  Only
//! the sources that are enabled can be reflected to the processor interrupt.
//! Disabled sources have no effect on the processor.
//!
//! The \e intFlags parameter has the same definition as the \e intFlags
//! parameter to PMBUS_enableInterrupt().
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_clearInterruptStatus(uint32_t base, PMBUS_InterruptRawStatus_t intFlags);

//*****************************************************************************
//
//! Set the PMBUS FIFO Threshold.
//!
//! \param base is the base address of the PMBus instance used.
//! \param txLevel is set the tx fifo threshold.
//! \param rxLevel is set the rx fifo threshold.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setFIFOInterruptLevel(uint32_t base, PMBUS_TxFIFOLevel_t txLevel,
		PMBUS_RxFIFOLevel_t rxLevel)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_TX_TL,txLevel);

	PMBUS_writeRegister(base+PMBUS_O_IC_RX_TL,rxLevel);

}

//*****************************************************************************
//
//! Get the PMBUS FIFO Threshold.
//!
//! \param base is the base address of the PMBus instance used.
//! \param *txLevel pointer is used to receive tx threshold.
//! \param *rxLevel pointer is used to receive rx threshold.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_getFIFOInterruptLevel(uint32_t base, PMBUS_TxFIFOLevel_t *txLevel,
				PMBUS_RxFIFOLevel_t *rxLevel)
{
	ASSERT(PMBUS_isBaseValid(base));

	*txLevel = (PMBUS_TxFIFOLevel_t)PMBUS_readRegister(base+PMBUS_O_IC_TX_TL);

	*rxLevel = (PMBUS_RxFIFOLevel_t)PMBUS_readRegister(base+PMBUS_O_IC_RX_TL);
}

//*****************************************************************************
//
//! Enables the PMBus module.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! This function enables operation of the PMBus module by removing it from the
//! reset state
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableModule(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val |= PMBUS_ENABLE_ENABLE;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Disables the PMBus module.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! This function resets the internal state machine of the PMBus module and
//! holds it in that state
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_disableModule(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val &= ~PMBUS_ENABLE_ENABLE;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! When set, the controller initiates the transfer abort.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_transmitAbort(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val |= PMBUS_ENABLE_ABORT;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Enbale blocks the transmission of data on PMBUS bus even if
//! Tx FIFO has data to transmit.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableBlockTx(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val |= PMBUS_ENABLE_TX_CMD_BLOCK;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Disable blocks the transmission of data on PMBUS bus even if
//! Tx FIFO has data to transmit.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_disableBlockTx(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val &= ~PMBUS_ENABLE_TX_CMD_BLOCK;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Master initiates the SDA stuck at low recovery mechanism.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableSDARecovery(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val |= PMBUS_ENABLE_SDA_STUCK_RECOVERY_ENABLE;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Set slave initates the Alert signal to indicate SMBus Host.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable Enable/Disable slave initates the Alert signal
//!					to indicate SMBus Host.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSAlert(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val &= ~PMBUS_ENABLE_SMBUS_ALERT_EN;

	if(enable)
		reg_val |= ~PMBUS_ENABLE_SMBUS_ALERT_EN;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Set master initates the SMBUS system to enter Suspend Mode.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable Enable master initates the SMBUS system to enter Suspend Mode.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableSMBUSSuspend(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val |= PMBUS_ENABLE_SMBUS_SUSPEND_EN;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Set Master initates the SMBUS Clock Reset Mechanism.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable Enable/Disable master initates the SMBUS Clock Reset Mechanism.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setCLKReset(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val &= ~PMBUS_ENABLE_SMBUS_CLK_RESET;

	if(enable)
		reg_val |= PMBUS_ENABLE_SMBUS_CLK_RESET;

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Set the address validity in the corresponding SAR register.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable ENABLE/DISABLE the address validity in the corresponding SAR register.
//! \param sar_num Corresponding SAR ID
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableSAR(uint32_t base,bool enable,PMBUS_SlaveAddrID_t sar_num)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_ENABLE);

	reg_val |= (PMBUS_ENABLE_SAR_EN << (sar_num-1));

	PMBUS_writeRegister(base+PMBUS_O_IC_ENABLE,reg_val);
}

//*****************************************************************************
//
//! Get the status of the PMBUS module.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return Return the current status of PMBUS.
//
//*****************************************************************************
static inline PMBUS_Status_t PMBUS_getStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_STATUS);

	return (PMBUS_Status_t)reg_val;
}

//*****************************************************************************
//
//! Get the TXFIFO status of the PMBUS module.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return Return the number of data currently in the TXFIFO.
//
//*****************************************************************************
static inline PMBUS_TxFIFOStatus_t PMBUS_getTxFIFOStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_TXFLR);

	return (PMBUS_TxFIFOStatus_t)reg_val;
}

//*****************************************************************************
//
//! Get the RXFIFO status of the PMBUS module.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return Return the number of data currently in the RXFIFO.
//
//*****************************************************************************
static inline PMBUS_RxFIFOStatus_t PMBUS_getRxFIFOStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_RXFLR);

	return (PMBUS_RxFIFOStatus_t)reg_val;
}

//*****************************************************************************
//
//! Set the holding time of SDA when sending
//!
//! \param base is the base address of the PMBus instance used.
//! \param hold_time is holding time of SDA when sending.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSDATxHold(uint32_t base,uint32_t hold_time)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SDA_HOLD);

	reg_val |= ((hold_time<<PMBUS_SDA_HOLD_SDA_TX_HOLD_S)&PMBUS_SDA_HOLD_SDA_TX_HOLD_M);

	PMBUS_writeRegister(base+PMBUS_O_IC_SDA_HOLD,reg_val);
}

//*****************************************************************************
//
//! Set the holding time of SDA during receiving
//!
//! \param base is the base address of the PMBus instance used.
//! \param hold_time is holding time of SDA when receiving.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSDARxHold(uint32_t base,uint32_t hold_time)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SDA_HOLD);

	reg_val |= ((hold_time<<PMBUS_SDA_HOLD_SDA_RX_HOLD_S)&PMBUS_SDA_HOLD_SDA_RX_HOLD_M);

	PMBUS_writeRegister(base+PMBUS_O_IC_SDA_HOLD,reg_val);
}

//*****************************************************************************
//
//! Set the holding time of SDA when sending
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return number of holding time of SDA when sending.
//
//*****************************************************************************
static inline int PMBUS_getSDATxHold(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SDA_HOLD);

	reg_val = ((reg_val&PMBUS_SDA_HOLD_SDA_TX_HOLD_M)>>PMBUS_SDA_HOLD_SDA_TX_HOLD_S);

	return reg_val;
}

//*****************************************************************************
//
//! Set the holding time of SDA when sending
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return number of holding time of SDA when receiving.
//
//*****************************************************************************
static inline int PMBUS_getSDARxHold(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SDA_HOLD);

	reg_val = ((reg_val&PMBUS_SDA_HOLD_SDA_RX_HOLD_M)>>PMBUS_SDA_HOLD_SDA_RX_HOLD_S);

	return reg_val;
}

//*****************************************************************************
//
//! Get the status of the TX arbitration source.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return number of holding time of SDA when receiving.
//
//*****************************************************************************
static inline PMBUS_TxAbrtSource_t PMBUS_getTxAbrtSource(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_TX_ABRT_SOURCE);

	return (PMBUS_TxAbrtSource_t)reg_val;
}

//*****************************************************************************
//
//! Get the number of Tx FIFO Data Commands which are flushed due to
//!  TX_ABRT interrupt.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of Tx FIFO Data Commands which are flushed due to
//!  TX_ABRT interrupt.
//
//*****************************************************************************
static inline int PMBUS_getTxFlushCount(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_TX_ABRT_SOURCE);

	reg_val = ((reg_val&PMBUS_TX_ABRT_SOURCE_TX_FLUSH_CNT_M)>>PMBUS_TX_ABRT_SOURCE_TX_FLUSH_CNT_S);

	return reg_val;
}

//*****************************************************************************
//
//! Send the NACK signal.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_sendNack(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SLV_DATA_NACK_ONLY);

	reg_val |= PMBUS_SLV_DATA_NACK_ONLY_NACK;

	PMBUS_writeRegister(base+PMBUS_O_IC_SLV_DATA_NACK_ONLY,reg_val);
}

//*****************************************************************************
//
//! Set tx dma.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable Enable/Disable tx dma.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setTxDMA(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_DMA_CR);

	reg_val &= ~PMBUS_DMA_CR_TDMAE;

	if(enable)
		reg_val |= PMBUS_DMA_CR_TDMAE;

	PMBUS_writeRegister(base+PMBUS_O_IC_DMA_CR,reg_val);
}

//*****************************************************************************
//
//! Set rx dma.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable Enable/Disable rx dma.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setRxDMA(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_DMA_CR);

	reg_val &= ~PMBUS_DMA_CR_RDMAE;

	if(enable)
		reg_val |= PMBUS_DMA_CR_RDMAE;

	PMBUS_writeRegister(base+PMBUS_O_IC_DMA_CR,reg_val);
}

//*****************************************************************************
//
//! Set the trigger level for the TX DMA request.
//!
//! \param base is the base address of the PMBus instance used.
//! \param dma_level The level at which a TX DMA request is generated.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setTxDMALevel(uint32_t base,PMBUS_TxDMALevel_t dma_level)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_DMA_TDLR,dma_level);
}

//*****************************************************************************
//
//! Set the trigger level for the RX DMA request.
//!
//! \param base is the base address of the PMBus instance used.
//! \param dma_level The level at which a RX DMA request is generated
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setRxDMALevel(uint32_t base,PMBUS_RxDMALevel_t dma_level)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_DMA_RDLR,dma_level);
}

//*****************************************************************************
//
//! Set the delay time for the rising edge of SDA.
//!
//! \param base is the base address of the PMBus instance used.
//! \param dma_level delay time for the rising edge of SDA.
//! \param sysclk is the rate of the clock supplied to the PMBUS module
//!					(SYSCLK) in Hz.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSDASetup(uint32_t base,uint8_t setuptime_ns,uint32_t sysclk)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = (1000000000/(sysclk*setuptime_ns)+1);

	reg_val = ((reg_val<<PMBUS_SDA_SETUP_SDA_SETUP_S)&PMBUS_SDA_SETUP_SDA_SETUP_M);

	PMBUS_writeRegister(base+PMBUS_O_IC_SDA_SETUP,reg_val);
}

//*****************************************************************************
//
//! Get the delay time for the rising edge of SDA.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of delay time for the rising edge of SDA.
//
//*****************************************************************************
static inline int PMBUS_getSDASetup(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_DMA_CR);
}

//*****************************************************************************
//
//! Set the corresponding General Call in Slave Mode.
//!
//! \param base is the base address of the PMBus instance used.
//! \param enable Enable/Disable corresponding General Call in Slave Mode.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setAckGenCall(uint32_t base,bool enable)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SDA_SETUP,enable);
}

//*****************************************************************************
//
//! Get the status of enabling the module
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The enabling status of the module
//
//*****************************************************************************
static inline int PMBUS_checkEnable(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return (PMBUS_readRegister(base+PMBUS_O_IC_ENABLE_STATUS)&PMBUS_ENABLE_STATUS_IC_EN);
}

//*****************************************************************************
//
//! Check the status when the slave is disable.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The status when the slave is disable.
//
//*****************************************************************************
static inline int PMBUS_checkSlaveDisableStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return (PMBUS_readRegister(base+PMBUS_O_IC_ENABLE_STATUS)&PMBUS_ENABLE_STATUS_SLV_DISABLED_WHILE_BUSY);
}

//*****************************************************************************
//
//! Check the status when slave RX data is lost
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The status when the slave RX data is lost.
//
//*****************************************************************************
static inline int PMBUS_checkSlaveRxDataLost(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return (PMBUS_readRegister(base+PMBUS_O_IC_ENABLE_STATUS)&PMBUS_ENABLE_STATUS_SLV_RX_DATA_LOST);
}

//*****************************************************************************
//
//! Set the low-level timeout value for SCL.
//!
//! \param base is the base address of the PMBus instance used.
//! \param timeout_count is the timeout value.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSCLAtLowTimeout(uint32_t base,uint32_t timeout_count)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SCL_STUCK_AT_LOW_TIMEOUT,timeout_count);
}

//*****************************************************************************
//
//! Set the low-level timeout value for SDA.
//!
//! \param base is the base address of the PMBus instance used.
//! \param timeout_count is the timeout value.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSDAAtLowTimeout(uint32_t base,uint32_t timeout_count)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SDA_STUCK_AT_LOW_TIMEOUT,timeout_count);
}

//*****************************************************************************
//
//! Get the low-level timeout value for SCL.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of timeout when SCL is low-level.
//
//*****************************************************************************
static inline uint32_t PMBUS_getSCLAtLowTimeout(uint32_t base,uint32_t timeout_count)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_SCL_STUCK_AT_LOW_TIMEOUT);
}

//*****************************************************************************
//
//! Get the low-level timeout value for SDA.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of timeout when SDA is low-level.
//
//*****************************************************************************
static inline uint32_t PMBUS_getSDAAtLowTimeout(uint32_t base,uint32_t timeout_count)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_SDA_STUCK_AT_LOW_TIMEOUT);
}

//*****************************************************************************
//
//! Set the DEVICE ID.
//!
//! \param base is the base address of the PMBus instance used.
//! \param device_id is DEVICE ID.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setDeviceID(uint32_t base,uint32_t device_id)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_DEVICE_ID,device_id);
}

//*****************************************************************************
//
//! Get the DEVICE ID.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of DEVICE ID.
//
//*****************************************************************************
static inline uint32_t PMBUS_getDeviceID(uint32_t base,uint32_t device_id)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_DEVICE_ID);
}

//*****************************************************************************
//
//! Set the slave clock extend timeout value.
//!
//! \param base is the base address of the PMBus instance used.
//! \param timeout_count is the slave clock extend timeout value.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSSlaveClkExtendTimeout(uint32_t base,uint32_t timeout_count)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SMBUS_CLK_LOW_SEXT,timeout_count);
}

//*****************************************************************************
//
//! Get the slave clock extend timeout value.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of the slave clock extend timeout value.
//
//*****************************************************************************
static inline uint32_t PMBUS_getSMBUSSlaveClkExtendTimeout(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_CLK_LOW_SEXT);
}

//*****************************************************************************
//
//! Set the master clock extend timeout value.
//!
//! \param base is the base address of the PMBus instance used.
//! \param timeout_count is the master clock extend timeout value.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSMasterClkExtendTimeout(uint32_t base,uint32_t timeout_count)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SMBUS_CLK_LOW_MEXT,timeout_count);
}

//*****************************************************************************
//
//! Get the master clock extend timeout value.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of the master clock extend timeout value.
//
//*****************************************************************************
static inline uint32_t PMBUS_getSMBUSMasterClkExtendTimeout(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_CLK_LOW_MEXT);
}

//*****************************************************************************
//
//! Set the master THigh MAX Bus-idle count.
//!
//! \param base is the base address of the PMBus instance used.
//! \param count is the the master THigh MAX Bus-idle count.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUSMasterMaxBusidleConut(uint32_t base,uint32_t count)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SMBUS_THIGH_MAX_IDLE_COUNT,count);
}

//*****************************************************************************
//
//! Getthe master THigh MAX Bus-idle count.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of the master THigh MAX Bus-idle count.
//
//*****************************************************************************
static inline int32_t PMBUS_getSMBUSMasterMaxBusidleConut(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_THIGH_MAX_IDLE_COUNT);
}

static inline PMBUS_SMBUSInterruptRawStatus_t PMBUS_getSMBUSInterruptRawStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_RAW_INTR_STAT);

	return (PMBUS_SMBUSInterruptRawStatus_t)reg_val;
}

static inline PMBUS_SMBUSInterruptRawStatus_t PMBUS_getSMBUSInterruptStatus(uint32_t base)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_INTR_STAT);

	return (PMBUS_SMBUSInterruptRawStatus_t)reg_val;
}

//*****************************************************************************
//
//! Enables SMBUS mode interrupt sources.
//!
//! \param base is the base address of the PMBus instance used.
//! \param intFlags is the bit mask of the interrupt sources to be enabled.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_enableSMBUSInterrupt(uint32_t base,PMBUS_SMBUSInterruptRawStatus_t intFlags)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_INTR_MASK);

	reg_val |= intFlags;

	PMBUS_writeRegister(base+PMBUS_O_IC_SMBUS_INTR_MASK,reg_val);
}

//*****************************************************************************
//
//! Disables SMBUS mode interrupt sources.
//!
//! \param base is the base address of the PMBus instance used.
//! \param intFlags is the bit mask of the interrupt sources to be disabled.
//!
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_disableSMBUSInterrupt(uint32_t base,PMBUS_SMBUSInterruptRawStatus_t intFlags)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_SMBUS_INTR_MASK);

	reg_val &= (~intFlags);

	PMBUS_writeRegister(base+PMBUS_O_IC_SMBUS_INTR_MASK,reg_val);
}

//*****************************************************************************
//
//! Clear SMBUS mode interrupt status.
//!
//! \param base is the base address of the PMBus instance used.
//! \param intFlags is the bit mask of the interrupt status to be clear.
//!
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_clearSMBUSInterrupt(uint32_t base,PMBUS_SMBUSInterruptRawStatus_t intFlags)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_CLR_SMBUS_INTR,intFlags);
}

//*****************************************************************************
//
//! Set the optional slave address.
//!
//! \param base is the base address of the PMBus instance used.
//! \param opt_sar is optional slave address.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setOptSAR(uint32_t base,uint8_t opt_sar)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_OPTIONAL_SAR,opt_sar);
}

//*****************************************************************************
//
//! Set the register timeout counter reset value
//!
//! \param base is the base address of the PMBus instance used.
//! \param count is register timeout counter reset value.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setRegisterTimeoutResetValue(uint32_t base,uint8_t count)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_REG_TIMEOUT_RST,count);
}

//*****************************************************************************
//
//! Get the register timeout counter reset value.
//!
//! \param base is the base address of the PMBus instance used.
//!
//! \return The number of register timeout counter reset value.
//
//*****************************************************************************
static inline int PMBUS_getRegisterTimeoutResetValue(uint32_t base,uint8_t count)
{
	ASSERT(PMBUS_isBaseValid(base));

	return PMBUS_readRegister(base+PMBUS_O_REG_TIMEOUT_RST);
}

//*****************************************************************************
//
//! Get the PMBUS FIFO depth.
//!
//! \param base is the base address of the PMBus instance used.
//! \param txLevel is the tx fifo depth.
//! \param rxLevel is the rx fifo depth.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_getFIFODepth(uint32_t base,uint8_t* txfifo_depth,uint8_t* rxfifo_depth)
{
	ASSERT(PMBUS_isBaseValid(base));

	uint32_t reg_val = PMBUS_readRegister(base+PMBUS_O_IC_COMP_PARAM_1);

	*txfifo_depth = ((reg_val&PMBUS_COMP_PARAM_1_RX_BUFFER_DEPTH_M)>>PMBUS_COMP_PARAM_1_RX_BUFFER_DEPTH_S);

	*rxfifo_depth = ((reg_val&PMBUS_COMP_PARAM_1_TX_BUFFER_DEPTH_M)>>PMBUS_COMP_PARAM_1_TX_BUFFER_DEPTH_S);
}

//*****************************************************************************
//
//! Set the LSB 32 bit value of slave unique device identifier used in
//! Address Resolution Protocol.
//!
//! \param base is the base address of the PMBus instance used.
//! \param udid is the LSB 32 bit value of slave unique device identifier used in
//!				 Address Resolution Protocol.
//!
//! \return None.
//
//*****************************************************************************
static inline void PMBUS_setSMBUS_UDID_LSB(uint32_t base,uint32_t udid)
{
	ASSERT(PMBUS_isBaseValid(base));

	PMBUS_writeRegister(base+PMBUS_O_IC_SMBUS_UDID_LSB,udid);
}


#ifdef __cplusplus
}
#endif

#endif /* DRIVERLIB_PMBUS_H_ */
