/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_apb.h
*   @brief   do NOT include this file directly, include "sysctl.h" instead
*
*   commit history:
*   2024/05/15 ZhaoLei, disable 3.0 spi mode config
*
*/

#ifndef DEVICE_DRIVERLIB_SYSCTL_APB_PARA_H_
#define DEVICE_DRIVERLIB_SYSCTL_APB_PARA_H_

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "gs32_version.h"

#include "inc/hw_sysctl_apb.h"
#include "inc/hw_memmap.h"
#include "spi.h"


typedef enum
{
    CAN0_PORT = 0,
    CAN1_PORT = 1,
#ifdef CANC_BASE
    CAN2_PORT = 2,
#endif
#ifdef CAND_BASE
    CAN3_PORT = 3,
#endif
    CANX_PORT_MAX,
} E_CANX_PORT;

typedef enum
{
	E_SPI_MODE_NONE = 0,
	E_SPI_MODE_SEL = 1
} E_SPI_PORT;


typedef enum
{
    APB_OWNER_CPU1 = 0x0,
	APB_OWNER_CPU2 = 0x1
}APB_SemOwner;
/* ========================================================================== */

/*
 *   Module Name                    Base_Addr
 *
 *   APB_COMM_PARA                  0x4038_1200 (HPEC1X)
 *   APB_COMM_PARA                  0x40223000U (HPEC2X)
*/

/* ========================================================================== */
#if  IS_GS32F00xx(0x12,0x30) || IS_GS32F3xx(0x22)
/*
 * SPI_MASTER_MODE   SPI master mode configuration   offset 0x00
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 5
 *  spi1_master_mode      4
 *  reserved              3  : 1
 *  spi0_master_mode      0
 *  
 *
 */

/*
 * Set the operating mode of SPIx as master/slave
*/

__STATIC_INLINE void SysCtl_Apb_setSpiMasterMode(uint32_t spi_base, E_SPI_PORT mode)
{
	uint32_t Index = 0;
    uint32_t value = HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_SPI_MASTER_MODE);

    switch (spi_base)
    {
#ifdef SPIA_BASE
        case SPIA_BASE: Index = 0;  break;
#endif
#ifdef SPIB_BASE
        case SPIB_BASE: Index = 4;  break;
#endif
#ifdef SPIC_BASE
        case SPIC_BASE: Index = 8;  break;
#endif
#ifdef SPID_BASE
        case SPID_BASE: Index = 12; break;
#endif
#ifdef SPIE_BASE
        case SPIE_BASE: Index = 16; break;
#endif
    }

	value &= (~(0x01 << Index));
	value |= ((mode == E_SPI_MODE_SEL) ? (0x01 << Index) : (0x00 << Index));

    HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_SPI_MASTER_MODE) = value;
}


/*
 * Retrieve the operating mode of SPIx as master/slave
*/
__STATIC_INLINE uint32_t SysCtl_Apb_getSpiMasterMode(uint32_t spi_base)
{
	uint32_t Index = 0;
    uint32_t value = HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_SPI_MASTER_MODE);

    switch (spi_base)
    {
#ifdef SPIA_BASE
        case SPIA_BASE: Index = 0;  break;
#endif
#ifdef SPIB_BASE
        case SPIB_BASE: Index = 4;  break;
#endif
#ifdef SPIC_BASE
        case SPIC_BASE: Index = 8;  break;
#endif
#ifdef SPID_BASE
        case SPID_BASE: Index = 12; break;
#endif
#ifdef SPIE_BASE
        case SPIE_BASE: Index = 16; break;
#endif
    }

    return ((value >> Index) & 0x01);
}

/*
 * CAN_FD_ENABLE   CAN FD enable configuration   offset 0x20
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 5
 *  can1_fd_enable        4
 *  reserved              3  : 1
 *  can0_fd_enable        0
 *  
 *
 */

/*
 * Configure CANx FD enable
 * enable = 1, enable CANx FD
 * enable = 0, disable CANx FD
*/
__STATIC_INLINE void SysCtl_setCanFdEnable(E_CANX_PORT port, uint32_t enable)
{
    HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN_FD_ENABLE) = \
            (HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN_FD_ENABLE) & ~(1<<(port*4))) | ((enable & 0x01) << (port*4));
}

/*
 * Retrieve the enable status of CANx FD
*/
__STATIC_INLINE uint32_t SysCtl_getCanFdEnable(E_CANX_PORT port)
{
    return (HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN_FD_ENABLE) >> (port * 4)) & 0x01;
}


#ifdef CANA_BASE
/*
 * CAN0_HOST_IF   CAN0 HOST IF reporting   offset 0x24
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 19
 *  can0_host_if          18 : 0
 *  
 *
 */

/*
 * Retrieve the value of can0_host_if
*/
__STATIC_INLINE uint32_t SysCtl_getCAN0_HOST_IF(void)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN0_HOST_IF) & 0x7FFFF;
}

#endif

#ifdef CANB_BASE
/*
 * CAN1_HOST_IF   CAN1 HOST IF reporting   offset 0x28
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 19
 *  can1_host_if          18 : 0
 *  
 *
 */

/*
 * Retrieve the value of can1_host_if
*/
__STATIC_INLINE uint32_t SysCtl_getCAN1_HOST_IF(void)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN1_HOST_IF) & 0x7FFFF;
}

#endif

#ifdef CANC_BASE
/*
 * CAN2_HOST_IF   CAN2 HOST IF reporting   offset 0x28
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 19
 *  can2_host_if          18 : 0
 *
 *
 */

/*
 * Retrieve the value of can2_host_if
*/
__STATIC_INLINE uint32_t SysCtl_getCAN2_HOST_IF(void)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN2_HOST_IF) & 0x7FFFF;
}
#endif

#ifdef CANC_BASE
/*
 * CAN3_HOST_IF   CAN3 HOST IF reporting   offset 0x28
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 19
 *  can3_host_if          18 : 0
 *
 *
 */

/*
 * Retrieve the value of can3_host_if
*/
__STATIC_INLINE uint32_t SysCtl_getCAN3_HOST_IF(void)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_CAN3_HOST_IF) & 0x7FFFF;
}
#endif

#endif


/*
 * APB_MEM_CTRL   APB domain memory parameter configuration   offset 0x30
 *
 *  Register Field Name   Field Range
 *  can_ram_mem_ctrl      31 : 0
 *
 */

/*
 * Set the value of can_ram_mem_ctrl
*/
__STATIC_INLINE void SysCtl_Apb_setApbMemCtrl(uint32_t value)
{
    HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_APB_MEM_CTRL) = value;
}

/*
 * Retrieve the value of can_ram_mem_ctrl
*/
__STATIC_INLINE uint32_t SysCtl_Apb_getApbMemCtrl(void)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_APB_MEM_CTRL) & 0xFFFFFFFF;
}


/*
 * APB_MEM_PWR_CTRL   APB domain memory low power configuration   offset 0x34
 *
 *  Register Field Name   Field Range
 *  can_ram_mem_pwr_ctrl  31 : 0
 *
 */

/*
 * Set the value of can_ram_mem_pwr_ctrl
*/
__STATIC_INLINE void SysCtl_Apb_setApbMemPwrCtrl(uint32_t value)
{
    HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_APB_MEM_PWR_CTRL) = value;
}

/*
 * Retrieve the value of can_ram_mem_pwr_ctrl
*/
__STATIC_INLINE uint32_t SysCtl_Apb_getApbMemPwrCtrl(void)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_APB_MEM_PWR_CTRL) & 0xFFFFFFFF;
}


/*
 *  TIMER_DEBUG_STOP_MASK   Debug Pause Timer Source Selection   offset 0x040+0x4*t   t = 1~4 Number of Timers
 *
 *  Register Field Name   Field Range
 *  reserved              31 : 2
 *  timer_debug_stop_mask 1  : 0
 *
 *  Bit 1: Corresponding to DSP debugging, set to 1 to enable, set to 0 to disable
 *
 */
/*
 * Set the value of timer_debug_stop_mask
*/
__STATIC_INLINE void Sysctl_ApbPara_setTimerDebugStopMask(uint32_t nTimer, uint32_t value)
{
    HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_TIMER_DEBUG_STOP_MASK_1 + 4 * nTimer) = value;
}

/*
 * Get the value of timer_debug_stop_mask
*/
__STATIC_INLINE uint32_t Sysctl_ApbPara_getTimerDebugStopMask(uint32_t nTimer)
{
    return HWREG(APB_COMM_PARA_BASE + APB1_COMM_PARA_TIMER_DEBUG_STOP_MASK_1 + 4 * nTimer);
}

#ifdef __cplusplus
}
#endif


#endif /* DEVICE_DRIVERLIB_SYSCTL_APB_PARA_H_ */
