/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
 *   @file    gpio.c
 *   @brief   GS32 GPIO driver.
 *   @details
 *
 */

#ifdef __cplusplus
extern "C" {
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "xbar.h"
#include "gs32_version.h"
#include "sysctl_io_v30.h"
#include "inc/hw_asysctl_v30.h"
#include "inc/hw_sysctl_dig_v30.h"


#if IS_GS32F00xx(0x30)

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#define GPIO_MAP_SIZE (sizeof(gpio_map) / sizeof(gpio_map[0]))

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

// GPIO mapping table
const gpio_config_t gpio_map[] = {

#if !(GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027)
    {0, 0, GPIO_TYPE_PI2C},   // gpio0
    {1, 3, GPIO_TYPE_PI2C},   // gpio1
    {2, 5, GPIO_TYPE_PI2C},   // gpio2
    {3, 6, GPIO_TYPE_PI2C},   // gpio3
    {4, 8, GPIO_TYPE_PI2C},   // gpio4
    {5, 10, GPIO_TYPE_PI2C},  // gpio5
    {6, 12, GPIO_TYPE_GPIO},  // gpio6
    {7, 14, GPIO_TYPE_GPIO},  // gpio7
    {8, 16, GPIO_TYPE_PI2C},  // gpio8
    {9, 18, GPIO_TYPE_PI2C},  // gpio9
    {10, 21, GPIO_TYPE_PI2C}, // gpio10
    {11, 22, GPIO_TYPE_GPIO}, // gpio11
    {12, 25, GPIO_TYPE_PI2C}, // gpio12
    {13, 26, GPIO_TYPE_PI2C}, // gpio13
    {14, 28, GPIO_TYPE_PI2C}, // gpio14
    {15, 30, GPIO_TYPE_PI2C}, // gpio15

    {16, 0, GPIO_TYPE_PI2C},  // gpio16
    {17, 2, GPIO_TYPE_PI2C},  // gpio17
    {18, 4, GPIO_TYPE_PI2C},  // gpio18
    {19, 6, GPIO_TYPE_PI2C},  // gpio19
    {20, 8, GPIO_TYPE_PI2C},  // gpio20
    {21, 10, GPIO_TYPE_PI2C}, // gpio21
    {22, 12, GPIO_TYPE_GPIO}, // gpio22
    {23, 14, GPIO_TYPE_GPIO}, // gpio23
    {24, 17, GPIO_TYPE_PI2C}, // gpio24
    {25, 18, GPIO_TYPE_PI2C}, // gpio25
    {26, 21, GPIO_TYPE_PI2C}, // gpio26
    {27, 22, GPIO_TYPE_PI2C}, // gpio27
    {28, 24, GPIO_TYPE_PI2C}, // gpio28
    {29, 26, GPIO_TYPE_PI2C}, // gpio29
    {30, 28, GPIO_TYPE_GPIO}, // gpio30
    {31, 30, GPIO_TYPE_GPIO}, // gpio31

    {32, 1, GPIO_TYPE_PI2C},  // gpio32
    {33, 2, GPIO_TYPE_PI2C},  // gpio33
    {34, 4, GPIO_TYPE_PI2C},  // gpio34
    {35, 6, GPIO_TYPE_PI2C},  // gpio35
    {36, 8, GPIO_TYPE_PI2C},  // gpio36
    {37, 10, GPIO_TYPE_PI2C}, // gpio37
    {38, 12, GPIO_TYPE_PI2C}, // gpio38
    {39, 14, GPIO_TYPE_GPIO}, // gpio39
    {40, 16, GPIO_TYPE_PI2C}, // gpio40
    {41, 18, GPIO_TYPE_PI2C}, // gpio41
    {42, 21, GPIO_TYPE_PI2C}, // gpio42
    {43, 22, GPIO_TYPE_PI2C}, // gpio43
    {44, 25, GPIO_TYPE_PI2C}, // gpio44
    {45, 26, GPIO_TYPE_PI2C}, // gpio45
    {46, 28, GPIO_TYPE_PI2C}, // gpio46

    {48, 0, GPIO_TYPE_PI2C}, // gpio48
    {49, 2, GPIO_TYPE_GPIO}, // gpio49

    {53, 10, GPIO_TYPE_GPIO}, // gpio53
    {54, 12, GPIO_TYPE_GPIO}, // gpio54
    {55, 14, GPIO_TYPE_GPIO}, // gpio55
    {56, 16, GPIO_TYPE_PI2C}, // gpio56
    {57, 18, GPIO_TYPE_PI2C}, // gpio57
    {58, 20, GPIO_TYPE_GPIO}, // gpio58
    {59, 22, GPIO_TYPE_GPIO}, // gpio59
    {60, 24, GPIO_TYPE_GPIO}, // gpio60
    {61, 26, GPIO_TYPE_GPIO}, // gpio61
    {62, 28, GPIO_TYPE_PI2C}, // gpio62
    {63, 30, GPIO_TYPE_GPIO}, // gpio63

    {200, 0, GPIO_TYPE_GPIO}, // gpio200

    // Adding gpio224 to gpio255
    {224, 0, GPIO_TYPE_GPIO},  // gpio224
    {225, 2, GPIO_TYPE_GPIO},  // gpio225
    {226, 4, GPIO_TYPE_GPIO},  // gpio226
    {227, 6, GPIO_TYPE_PI2C},  // gpio227
    {228, 8, GPIO_TYPE_GPIO},  // gpio228
    {230, 12, GPIO_TYPE_PI2C}, // gpio230
    {233, 18, GPIO_TYPE_GPIO}, // gpio233
    {236, 24, GPIO_TYPE_PI2C}, // gpio236
    {237, 26, GPIO_TYPE_GPIO}, // gpio237

    {242, 4, GPIO_TYPE_GPIO}, // gpio242

    {245, 10, GPIO_TYPE_GPIO}, // gpio245
#else
    {0, 0, GPIO_TYPE_PI2C},   // gpio0
    {1, 2, GPIO_TYPE_GPIO},   // gpio1
    {2, 4, GPIO_TYPE_GPIO},   // gpio2
    {3, 6, GPIO_TYPE_PI2C},   // gpio3
    {4, 8, GPIO_TYPE_PI2C},   // gpio4
    {5, 10, GPIO_TYPE_PI2C},  // gpio5
    {6, 13, GPIO_TYPE_PI2C},  // gpio6
    {7, 15, GPIO_TYPE_PI2C},  // gpio7
    {8, 16, GPIO_TYPE_PI2C},  // gpio8
    {9, 18, GPIO_TYPE_PI2C},  // gpio9
    {10, 20, GPIO_TYPE_GPIO}, // gpio10
    {11, 23, GPIO_TYPE_PI2C}, // gpio11
    {12, 24, GPIO_TYPE_GPIO}, // gpio12
    {13, 26, GPIO_TYPE_PI2C}, // gpio13
    {14, 28, GPIO_TYPE_PI2C}, // gpio14
    {15, 30, GPIO_TYPE_PI2C}, // gpio15

    {16, 0, GPIO_TYPE_PI2C},  // gpio16
    {17, 2, GPIO_TYPE_PI2C},  // gpio17
    {18, 4, GPIO_TYPE_PI2C},  // gpio18
    {19, 6, GPIO_TYPE_PI2C},  // gpio19
    {20, 8, GPIO_TYPE_PI2C},  // gpio20
    {21, 10, GPIO_TYPE_PI2C}, // gpio21
    {22, 12, GPIO_TYPE_GPIO}, // gpio22
    {23, 15, GPIO_TYPE_PI2C}, // gpio23
    {24, 16, GPIO_TYPE_GPIO}, // gpio24
    {25, 18, GPIO_TYPE_PI2C}, // gpio25
    {26, 20, GPIO_TYPE_GPIO}, // gpio26
    {27, 22, GPIO_TYPE_PI2C}, // gpio27
    {28, 24, GPIO_TYPE_PI2C}, // gpio28
    {29, 26, GPIO_TYPE_PI2C}, // gpio29
    {30, 29, GPIO_TYPE_PI2C}, // gpio30
    {31, 30, GPIO_TYPE_GPIO}, // gpio31

    {32, 0, GPIO_TYPE_GPIO},  // gpio32
    {33, 2, GPIO_TYPE_PI2C},  // gpio33
    {34, 4, GPIO_TYPE_PI2C},  // gpio34
    {35, 6, GPIO_TYPE_PI2C},  // gpio35
    {36, 8, GPIO_TYPE_PI2C},  // gpio36
    {37, 10, GPIO_TYPE_PI2C}, // gpio37
    {38, 12, GPIO_TYPE_PI2C}, // gpio38
    {39, 14, GPIO_TYPE_GPIO}, // gpio39
    {40, 16, GPIO_TYPE_PI2C}, // gpio40
    {41, 18, GPIO_TYPE_PI2C}, // gpio41
    {42, 20, GPIO_TYPE_GPIO}, // gpio42
    {43, 22, GPIO_TYPE_PI2C}, // gpio43
    {44, 24, GPIO_TYPE_GPIO}, // gpio44
    {45, 26, GPIO_TYPE_PI2C}, // gpio45
    {46, 28, GPIO_TYPE_PI2C}, // gpio46

    {48, 0, GPIO_TYPE_PI2C}, // gpio48
    {49, 2, GPIO_TYPE_GPIO}, // gpio49

    {53, 10, GPIO_TYPE_GPIO}, // gpio53
    {54, 12, GPIO_TYPE_GPIO}, // gpio54
    {55, 14, GPIO_TYPE_GPIO}, // gpio55
    {56, 16, GPIO_TYPE_PI2C}, // gpio56
    {57, 18, GPIO_TYPE_PI2C}, // gpio57
    {58, 20, GPIO_TYPE_GPIO}, // gpio58
    {59, 22, GPIO_TYPE_GPIO}, // gpio59
    {60, 24, GPIO_TYPE_GPIO}, // gpio60
    {61, 26, GPIO_TYPE_GPIO}, // gpio61
    {62, 28, GPIO_TYPE_PI2C}, // gpio62
    {63, 30, GPIO_TYPE_GPIO}, // gpio63

    {200, 0, GPIO_TYPE_GPIO}, // gpio200

    // Adding gpio224 to gpio255
    {224, 0, GPIO_TYPE_GPIO},  // gpio224
    {225, 2, GPIO_TYPE_GPIO},  // gpio225
    {226, 4, GPIO_TYPE_GPIO},  // gpio226
    {227, 6, GPIO_TYPE_PI2C},  // gpio227
    {228, 8, GPIO_TYPE_GPIO},  // gpio228
    {230, 12, GPIO_TYPE_PI2C}, // gpio230
    {233, 18, GPIO_TYPE_GPIO}, // gpio233
    {236, 24, GPIO_TYPE_PI2C}, // gpio236
    {237, 26, GPIO_TYPE_GPIO}, // gpio237

    {242, 4, GPIO_TYPE_GPIO}, // gpio242

    {245, 10, GPIO_TYPE_GPIO}, // gpio245
#endif

};

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

// Find the gpio_comfig_t structure corresponding to the given pin value
GS32_DRIVER_GPIO_FUNC_T const gpio_config_t *GPIO_getConfigByPin(uint32_t pin)
{
    for (uint32_t i = 0; i < GPIO_MAP_SIZE; ++i) {
        if (gpio_map[i].pin_number == pin) {
            return &gpio_map[i];
        }
    }
    return NULL; // If not found, return NULL
}
#if !(GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027)
/**
 * @brief   Determine whether the pin type is AIO or AGPIO
 * @param pin_num Pin number
 * @return  HW_PIN_TYPE_AIO: AIO pin
 *          HW_PIN_TYPE_AGPIO: AGPIO pin
 *          HW_PIN_TYPE_GPIO: GPIO pin
 *          HW_PIN_TYPE_UNKNOWN: Invalid pin
 */
GS32_DRIVER_GPIO_FUNC_T HwPinType
check_pin_type(int pin_num)
{
    // AIO pin judgment
    switch (pin_num) {
    case 231:
    case 232:
    case 234:
    case 235:
    case 238:
    case 239:
    case 241:
    case 244:
    case 248:
        return HW_PIN_TYPE_AIO;
    }

    // AGPIO pin determination
    switch (pin_num) {
    case 11:
    case 12:
    case 13:
    case 16:
    case 17:
    case 20:
    case 21:
    case 24:
    case 28:
    case 33:
    case 200:
    case 224:
    case 225:
    case 226:
    case 227:
    case 228:
    case 230:
    case 233:
    case 236:
    case 237:
    case 242:
    case 245:
        return HW_PIN_TYPE_AGPIO;
    }

    // If it is not AIO or AGPIO, it is considered a regular GPIO
    if (pin_num >= 0) {
        return HW_PIN_TYPE_GPIO;
    }

    // Negative or other invalid valuess
    return HW_PIN_TYPE_UNKNOWN;
}

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/*******************************************************************************
 *
 *  GPIO_setDirectionMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setDirectionMode(uint32_t pin, GPIO_Direction pinIO)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           Index;
    uint32_t           pinMask;
    uint32_t           pinType;
    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)
    switch (pin) {
    case 56:
        if (pinIO == GPIO_DIR_MODE_OUT) {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE2) |= 0x1;
        } else if (pinIO == GPIO_DIR_MODE_IN) {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE2) &= 0xFFFFFFFE;
        }
        return;
    case 57:
        if (pinIO == GPIO_DIR_MODE_OUT) {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE2) |= 0x4;
        } else if (pinIO == GPIO_DIR_MODE_IN) {
            HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE2) &= 0xFFFFFFFB;
        }
        return;
    default:
        break;
    }
#endif

    pinType = check_pin_type(pin);
    if (pinType == HW_PIN_TYPE_GPIO || pinType == HW_PIN_TYPE_AGPIO) {
        if (pin != 200) // Non GPOG configuration
        {
            gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + ((pin / 32U) * GPIO_CTRL_REGS_STEP));
            Index        = GPIO_GPxDIR_INDEX;
            pinMask      = (uint32_t)1U << (pin % 32U);
        } else {
            gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
            Index        = (GPIO_O_GPGDIR - GPIO_O_GPGCTRL) / 4;
            pinMask      = (uint32_t)1U;
        }

        //
        // Set the data direction
        //
        if (pinIO == GPIO_DIR_MODE_OUT) {
            //
            // Output
            //
            gpioBaseAddr[Index] |= pinMask;
        } else if (pinIO == GPIO_DIR_MODE_IN) {
            //
            // Input
            //
            gpioBaseAddr[Index] &= ~pinMask;
        }
    } else if (pinType == HW_PIN_TYPE_AIO) {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPHPDD);
        Index        = ((GPIO_O_GPHIE - GPIO_O_GPHPDD) / 4);
        pinMask      = (uint32_t)1U << (pin % 32U);
        //
        // Set the data direction
        //
        if (pinIO == AIO_IE_MODE_NO_IN) {
            //
            // Input disable
            //
            gpioBaseAddr[Index] &= ~pinMask;
        } else if (pinIO == AIO_IE_MODE_IN) {
            //
            // Input enable
            //
            gpioBaseAddr[Index] |= pinMask;
        } else {
            ASSERT(0);
        }
    }
}

/*******************************************************************************
 *
 *  GPIO_getDirectionMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T GPIO_Direction
GPIO_getDirectionMode(uint32_t pin)
{

#if (GS32_PART_NUM==0x5000 || GS32_PART_NUM==0x4000)

    switch (pin) {
    case 56:
        return (HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE2) & 0x1) ? 1U : 0U;
    case 57:
        return (HWREG(DSP_APB_CFG_BASE + DIG_SYS_SPARE2) & 0x4) ? 1U : 0U;
    default:
        break;
    }

#endif

    volatile uint32_t *gpioBaseAddr;
    uint32_t           Index;
    uint32_t           RegValue;
    uint32_t           pinType;
    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
    pinType = check_pin_type(pin);
    if (pinType == HW_PIN_TYPE_GPIO || pinType == HW_PIN_TYPE_AGPIO) {

        if (pin != 200) // Non GPOG configuration
        {
            gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + ((pin / 32U) * GPIO_CTRL_REGS_STEP));
            Index        = GPIO_GPxDIR_INDEX;
            RegValue     = (uint32_t)((gpioBaseAddr[Index] >> (pin % 32U)) & 1U);
            return ((GPIO_Direction)(RegValue));
        } else {
            gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
            Index        = (GPIO_O_GPGDIR - GPIO_O_GPGCTRL) / 4;
            RegValue     = (uint32_t)((gpioBaseAddr[Index]) & 1U);
            return ((GPIO_Direction)(RegValue));
        }
    } else if (pinType == HW_PIN_TYPE_AIO) {
        gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);
        Index        = (GPIO_O_GPHIE - GPIO_O_GPHPDD) / 4;
        RegValue     = (uint32_t)((gpioBaseAddr[Index] >> (pin % 32U)) & 1U);
        return ((GPIO_Direction)(3 - RegValue));
    }else { /* HW_PIN_TYPE_UNKNOWN */
        return GPIO_DIR_MODE_IN;
    }
}

/*******************************************************************************
 *
 *  GPIO_setInterruptPin
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setInterruptPin(uint32_t pin, GPIO_ExternalIntNum extIntNum)
{
    XBAR_InputNum input;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    //
    // Pick the X-BAR input that corresponds to the requested XINT.
    //
    switch (extIntNum) {
    case GPIO_INT_XINT1:
        input = XBAR_INPUT1;
        break;

    case GPIO_INT_XINT2:
        input = XBAR_INPUT2;
        break;

    case GPIO_INT_XINT3:
        input = XBAR_INPUT3;
        break;

    case GPIO_INT_XINT4:
        input = XBAR_INPUT4;
        break;

    case GPIO_INT_XINT5:
        input = XBAR_INPUT5;
        break;

    case GPIO_INT_XINT6:
        input = XBAR_INPUT6;
        break;

    case GPIO_INT_XINT7:
        input = XBAR_INPUT7;
        break;

    case GPIO_INT_XINT8:
        input = XBAR_INPUT8;
        break;

    case GPIO_INT_XINT9:
        input = XBAR_INPUT9;
        break;

    case GPIO_INT_XINT10:
        input = XBAR_INPUT10;
        break;

    case GPIO_INT_XINT11:
        input = XBAR_INPUT11;
        break;

    case GPIO_INT_XINT12:
        input = XBAR_INPUT12;
        break;

    case GPIO_INT_XINT13:
        input = XBAR_INPUT13;
        break;

    case GPIO_INT_XINT14:
        input = XBAR_INPUT14;
        break;

    case GPIO_INT_XINT15:
        input = XBAR_INPUT15;
        break;

    case GPIO_INT_XINT16:
        input = XBAR_INPUT16;
        break;
    }

    XBAR_setInputPin(INPUTXBAR_BASE, input, (uint16_t)pin);
}

/*******************************************************************************
 *
 *  GPIO_setPadConfig
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setPadConfig(uint32_t pin, uint32_t pinType)
{
    volatile uint32_t *gpioBaseAddr1;
    volatile uint32_t *gpioBaseAddr2;
    uint32_t           PudIndex;
    uint32_t           InvIndex;
    uint32_t           PddIndex;
    uint32_t           OutInvIndex;
    uint32_t           pinMask;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    if (pin != 200) // Non GPOG configuration
    {
        gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        PudIndex      = GPIO_GPxPUD_INDEX;
        InvIndex      = GPIO_GPxINV_INDEX;
        PddIndex      = GPIO_GPxPDD_INDEX;
        OutInvIndex   = GPIO_GPxOUTINV_INDEX;
        pinMask       = (uint32_t)1U << (pin % 32U);
    } else {
        gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
        pinMask       = (uint32_t)1U << (0U);

        PudIndex    = (GPIO_O_GPGPUD - GPIO_O_GPGCTRL) / 4;
        InvIndex    = (GPIO_O_GPGINV - GPIO_O_GPGCTRL) / 4;
        PddIndex    = (GPIO_O_GPGPDD - GPIO_O_GPGPDD) / 4;
        OutInvIndex = (GPIO_O_GPGOUTINV - GPIO_O_GPGPDD) / 4;
    }
    gpioBaseAddr2 = (uint32_t *)(GPIOEXT_BASE +
                                (pin / 32U) * GPIO_EXT_REGS_STEP);

    //
    // Enable pull-up if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_PULLUP) != 0U) {
        gpioBaseAddr1[PudIndex] &= ~pinMask;
    } else {
        gpioBaseAddr1[PudIndex] |= pinMask;
    }

    //
    // Invert polarity if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_INVERT) != 0U) {
        gpioBaseAddr1[InvIndex] |= pinMask;
    } else {
        gpioBaseAddr1[InvIndex] &= ~pinMask;
    }

    //
    // Pull-Down if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_PULLDOWN) != 0U) {
        gpioBaseAddr2[PddIndex] &= ~pinMask;
    } else {
        gpioBaseAddr2[PddIndex] |= pinMask;
    }

    //
    // Inverted Push-Pull if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_OUTPUT_INVERT) != 0U) {
        gpioBaseAddr2[OutInvIndex] |= pinMask;
    } else {
        gpioBaseAddr2[OutInvIndex] &= ~pinMask;
    }
}

/*******************************************************************************
 *
 *  GPIO_getPadConfig
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_getPadConfig(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr1;
    volatile uint32_t *gpioBaseAddr2;

    uint32_t PudIndex;
    uint32_t InvIndex;
    uint32_t PddIndex;
    uint32_t OutInvIndex;

    uint32_t pinMask;
    uint32_t pinTypeRes;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    if (pin != 200) // Non GPOG configuration
    {
        gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);

        PudIndex    = GPIO_GPxPUD_INDEX;
        InvIndex    = GPIO_GPxINV_INDEX;
        PddIndex    = GPIO_GPxPDD_INDEX;
        OutInvIndex = GPIO_GPxOUTINV_INDEX;

        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);

        PudIndex    = (GPIO_O_GPGPUD - GPIO_O_GPGCTRL) / 4;
        InvIndex    = (GPIO_O_GPGINV - GPIO_O_GPGCTRL) / 4;
        PddIndex    = (GPIO_O_GPGPDD - GPIO_O_GPGPDD) / 4;
        OutInvIndex = (GPIO_O_GPGOUTINV - GPIO_O_GPGPDD) / 4;

        pinMask = (uint32_t)1U << (0U);
    }
    gpioBaseAddr2 = (uint32_t *)(GPIOEXT_BASE +
                                (pin / 32U) * GPIO_EXT_REGS_STEP);
    pinTypeRes    = GPIO_PIN_TYPE_STD;

    //
    // Get pull-up value
    //
    if ((gpioBaseAddr1[PudIndex] & pinMask) == 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_PULLUP;
    }

    //
    // Get polarity value
    //
    if ((gpioBaseAddr1[InvIndex] & pinMask) != 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_INVERT;
    }

    //
    // Get Pull-Down value
    //
    if ((gpioBaseAddr2[PddIndex] & pinMask) != 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_PULLDOWN;
    }

    //
    // Get Inverted Push-Pull value
    //
    if ((gpioBaseAddr2[OutInvIndex] & pinMask) != 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_OUTPUT_INVERT;
    }

    return (pinTypeRes);
}

/*******************************************************************************
 *
 *  GPIO_setQualificationMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setQualificationMode(uint32_t pin, GPIO_QualificationMode qualification)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           qSelIndex;
    uint32_t           shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    if (pin != 200) // Non GPOG configuration
    {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        shiftAmt     = (uint32_t)GPIO_GPAQSEL1_GPIO1_S * (pin % 16U);
        qSelIndex    = GPIO_GPxQSEL_INDEX + ((pin % 32U) / 16U);
    } else {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
        shiftAmt     = 0U;
        qSelIndex    = (GPIO_O_GPGQSEL1 - GPIO_O_GPGCTRL) / 4;
    }

    //
    // Write the input qualification mode to the register.
    //

    gpioBaseAddr[qSelIndex] &= ~((uint32_t)GPIO_GPAQSEL1_GPIO0_M << shiftAmt);
    gpioBaseAddr[qSelIndex] |= (uint32_t)qualification << shiftAmt;
}

/*******************************************************************************
 *
 *  GPIO_getQualificationMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T GPIO_QualificationMode
GPIO_getQualificationMode(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           qSelIndex;
    uint32_t           qualRes;
    uint32_t           shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    if (pin != 200) // Non GPOG configuration
    {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        qSelIndex    = GPIO_GPxQSEL_INDEX + ((pin % 32U) / 16U);
        shiftAmt     = (uint32_t)GPIO_GPAQSEL1_GPIO1_S * (pin % 16U);
    } else {
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
        qSelIndex    = (GPIO_O_GPGQSEL1 - GPIO_O_GPGCTRL) / 4;
        shiftAmt     = 0U;
    }

    //
    // Read the qualification mode register and shift and mask to get the
    // value for the specified pin.
    //
    qualRes = (gpioBaseAddr[qSelIndex] >> shiftAmt) &
            (uint32_t)GPIO_GPAQSEL1_GPIO0_M;
    return ((GPIO_QualificationMode)qualRes);
}

/*******************************************************************************
 *
 *  GPIO_setQualificationPeriod
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setQualificationPeriod(uint32_t pin, uint32_t divider)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           CtrlIndex;
    uint32_t           pinMask, regVal, shiftAmt;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));
    ASSERT((divider >= 1U) && (divider <= 510U));

    if (pin != 200) // Non GPOG configuration
    {
        shiftAmt = (pin % 32U) & ~((uint32_t)0x7U);
        pinMask  = (uint32_t)0xFFU << shiftAmt;

        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        CtrlIndex    = GPIO_GPxCTRL_INDEX;
    } else {
        shiftAmt = 0U;
        pinMask  = (uint32_t)0xFFU << shiftAmt;

        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE + GPIO_O_GPGCTRL);
        CtrlIndex    = (GPIO_O_GPGCTRL - GPIO_O_GPGCTRL) / 4;
    }

    //
    // Divide divider by two to get the value that needs to go into the field.
    // Then shift it into the right place.
    //
    regVal = (divider / 2U) << shiftAmt;

    gpioBaseAddr[CtrlIndex] &= ~pinMask;
    gpioBaseAddr[CtrlIndex] |= regVal;
}

/*******************************************************************************
 *
 *  GPIO_setAnalogMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setAnalogMode(uint32_t pin, GPIO_AnalogMode mode)
{
    uint32_t pinMask;
    HwPinType pinType;

    pinType = check_pin_type(pin);

    //
    // Check the arguments.
    //
    ASSERT((pinType == HW_PIN_TYPE_AIO) ||
        (pinType == HW_PIN_TYPE_AGPIO));

    if (pin != 200) // Non GPOG configuration
    {
        volatile uint32_t *gpioBaseAddr;
        gpioBaseAddr = (uint32_t *)(GPIOCTRL_BASE +
                                    (pin / 32U) * GPIO_CTRL_REGS_STEP);
        pinMask      = (uint32_t)1U << (pin % 32U);

        //
        // Set the analog mode selection.
        //
        if (mode == GPIO_ANALOG_ENABLED)
        {
            //
            // Enable analog mode
            //
            gpioBaseAddr[GPIO_GPxAMSEL_INDEX] |= pinMask;
            if(pinType == HW_PIN_TYPE_AGPIO)
            {
                //
                // Set AGPIOCTL
                //
                HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLA +
                       ((pin / 32U) * 0x4U)) |= (pinMask);
            }
        }
        else
        {
            //
            // Disable analog mode
            //
            gpioBaseAddr[GPIO_GPxAMSEL_INDEX] &= ~pinMask;
            if(pinType == HW_PIN_TYPE_AGPIO)
            {
                //
                // Clear AGPIOCTL
                //
                HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_AGPIOCTRLA +
                       ((pin / 32U) * 0x4U)) &= ~(pinMask);
            }
        }
    }
    else
    {
        pinMask = (uint32_t)1U;
        //
        // Set the analog mode selection.
        //
        if (mode == GPIO_ANALOG_ENABLED)
        {
            //
            // Enable analog mode
            //
            HWREG(GPIOCTRL_BASE + GPIO_O_GPGAMSEL) |= pinMask;
        }
        else
        {
            //
            // Disable analog mode
            //
            HWREG(GPIOCTRL_BASE + GPIO_O_GPGAMSEL) &= ~pinMask;
        }
    }
}

/*******************************************************************************
 *
 *  GPIO_setPinConfig
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setPinConfig(uint32_t pinConfig)
{
    uint32_t muxRegAddr;
    uint32_t pinMask, shiftAmt;

    muxRegAddr = (uint32_t)GPIOCTRL_BASE + (pinConfig >> 16);
    shiftAmt   = ((pinConfig >> 8) & (uint32_t)0xFFU);
    pinMask    = (uint32_t)0x3U << shiftAmt;

    //
    // Clear fields in MUX register first to avoid glitches
    //
    HWREG(muxRegAddr) &= ~pinMask;

    if (muxRegAddr != (GPIOCTRL_BASE + GPIO_O_GPGMUX1)) // Non GPOG configuration
    {
        //
        // Write value into GMUX register
        //
        HWREG(muxRegAddr + GPIO_MUX_TO_GMUX) =
            (HWREG(muxRegAddr + GPIO_MUX_TO_GMUX) & ~pinMask) |
            (((pinConfig >> 2) & (uint32_t)0x3U) << shiftAmt);

        //
        // Write value into MUX register
        //
        HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U) << shiftAmt);
    } else // GPOG configuration
    {
        //
        // Write value into GMUX register
        //
        HWREG(GPIOCTRL_BASE + GPIO_O_GPGGMUX1) =
            (HWREG(GPIOCTRL_BASE + GPIO_O_GPGGMUX1) & ~pinMask) |
            (((pinConfig >> 2) & (uint32_t)0x3U));

        //
        // Write value into MUX register
        //
        HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U));
    }
}

/*******************************************************************************
 *
 * GPIO_setStrength : I2C GPIO : 4MA/20MA   Standard GPIO : 2MA/4MA/8MA/12MA
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setStrength(uint32_t pin, GPIO_DriveStrength StrengthLevel)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           DsIndex;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    // Find the gpio_comfig_t structure corresponding to the pin
    const gpio_config_t *config = GPIO_getConfigByPin(pin);
    ASSERT(config);

    if (config->type == GPIO_TYPE_PI2C) {
        pinMask = (uint32_t)StrengthLevel >> 2;
    } else if (config->type == GPIO_TYPE_GPIO) {
        pinMask = (uint32_t)StrengthLevel & 0x3;
    }

    if (pin != 200) // Non GPOG configuration
    {
        DsIndex = GPIO_GPxDS_INDEX + ((pin % 32U) / 16U);
    } else {
        DsIndex = (GPIO_O_GPGDS1 - GPIO_O_GPGPDD) / 4;
    }

    //
    // Enable driver strength
    //
    gpioBaseAddr[DsIndex] &= ~(0x3 << (2 * (pin % 16U)));
    gpioBaseAddr[DsIndex] |= (pinMask << (config->pin_S));
}

/*******************************************************************************
 *
 *  GPIO_setSchmittTrigger
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setSchmittTrigger(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           IsIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
        IsIndex = GPIO_GPxIS_INDEX;
    } else {
        pinMask = (uint32_t)1U;
        IsIndex = (GPIO_O_GPGIS - GPIO_O_GPGPDD) / 4;
    }

    if (pinType) {
        gpioBaseAddr[IsIndex] |= pinMask;
    } else {
        gpioBaseAddr[IsIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_setOutPolarityInv
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setOutPolarityInv(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           OutInvIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if (pin != 200) // Non GPOG configuration
    {
        pinMask     = (uint32_t)1U << (pin % 32U);
        OutInvIndex = GPIO_GPxOUTINV_INDEX;
    } else {
        pinMask     = (uint32_t)1U;
        OutInvIndex = (GPIO_O_GPGOUTINV - GPIO_O_GPGPDD) / 4;
    }

    if (pinType) {
        gpioBaseAddr[OutInvIndex] |= pinMask;
    } else {
        gpioBaseAddr[OutInvIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_setLoopBack
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setLoopBack(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           LoopEnIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if (pin != 200) // Non GPOG configuration
    {
        pinMask     = (uint32_t)1U << (pin % 32U);
        LoopEnIndex = GPIO_GPxLOOPEN_INDEX;
    } else {
        pinMask     = (uint32_t)1U;
        LoopEnIndex = (GPIO_O_GPGLOOPEN - GPIO_O_GPGPDD) / 4;
    }

    if (pinType) {
        gpioBaseAddr[LoopEnIndex] |= pinMask;
    } else {
        gpioBaseAddr[LoopEnIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_setI2CMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setI2CMode(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           I2cModeIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if ((pin != 227) || (pin != 230)) // Non GPOH configuration
    {
        pinMask      = (uint32_t)1U << (pin % 32U);
        I2cModeIndex = GPIO_GPxPI2C_MODE_INDEX;
    } else {
        pinMask      = (uint32_t)1U << (pin % 32U);
        I2cModeIndex = (GPIO_O_GPHPI2C_MODE - GPIO_O_GPHPDD) / 4;
    }

    if (pinType) {
        gpioBaseAddr[I2cModeIndex] |= pinMask;
    } else {
        gpioBaseAddr[I2cModeIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_readPinOutReport
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_readPinOutReport(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           RegValue;
    uint32_t           PoutRptIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if (pin == 200) // GPIOG configuration
    {
        PoutRptIndex = (GPIO_O_GPG_POUT_RPT - GPIO_O_GPGPDD) / 4;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((RegValue >> (0U)) & 0x1);
    } else if (pin == 224) // GPIOH configuration
    {
        PoutRptIndex = (GPIO_O_GPH_POUT_RPT - GPIO_O_GPHPDD) / 4;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((RegValue >> (0U)) & 0x1);
    } else // Non GPIOG/GPIOH configuration
    {
        PoutRptIndex = GPIO_GPxPOUT_RPT_INDEX;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((RegValue >> (pin % 32U)) & 0x1);
    }
}

/*******************************************************************************
 *
 *  GPIO_readPortOutReport
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_readPortOutReport(GPIO_Port_32BIT port)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           RegValue;
    uint32_t           PoutRptIndex;

    gpioBaseAddr = (uint32_t *)(IO_CFG_BASE + (port * GPIO_EXT_REGS_STEP));
    if (port == GPIO_PORTG) {
        PoutRptIndex = (GPIO_O_GPG_POUT_RPT - GPIO_O_GPGPDD) / 4;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((uint32_t)(RegValue & 0x01));
    } else if (port == GPIO_PORTH) {
        PoutRptIndex = (GPIO_O_GPH_POUT_RPT - GPIO_O_GPHPDD) / 4;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((uint32_t)(RegValue & 0x01));
    } else {
        PoutRptIndex = GPIO_GPxPOUT_RPT_INDEX;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((uint32_t)(RegValue & 0xFFFFFFFF));
    }
}

/*******************************************************************************
 *
 * GPIO_setInterruptType_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setInterruptType_NO_XInt(uint32_t pin, GPIO_InterruptType_NO_XInt intTypePolarity)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntTypeIndex;
    uint32_t           IntPolIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntTypeIndex = GPIO_INTTYPE_INDEX;
    IntPolIndex  = GPIO_INTPOL_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    if (intTypePolarity == GPIO_INTERRUPT_TYPE_LEVEL_LOW) {
        gpioBaseAddr[IntTypeIndex] &= (~pinMask);
        gpioBaseAddr[IntPolIndex] &= (~pinMask);
    } else if (intTypePolarity == GPIO_INTERRUPT_TYPE_LEVEL_HIGH) {
        gpioBaseAddr[IntTypeIndex] &= (~pinMask);
        gpioBaseAddr[IntPolIndex] |= pinMask;
    } else if (intTypePolarity == GPIO_INTERRUPT_TYPE_EDGE_RISE) {
        gpioBaseAddr[IntTypeIndex] |= pinMask;
        gpioBaseAddr[IntPolIndex] |= pinMask;
    } else if (intTypePolarity == GPIO_INTERRUPT_TYPE_EDGE_FALL) {
        gpioBaseAddr[IntTypeIndex] |= pinMask;
        gpioBaseAddr[IntPolIndex] &= (~pinMask);
    }
}

/*******************************************************************************
 *
 * GPIO_getInterruptType_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T GPIO_InterruptType_NO_XInt
GPIO_getInterruptType_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntTypeIndex;
    uint32_t           IntPolIndex;
    uint32_t           shiftAmt;
    uint32_t           RagValue1;
    uint32_t           RagValue2;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntTypeIndex = GPIO_INTTYPE_INDEX;
    IntPolIndex  = GPIO_INTPOL_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        shiftAmt = (pin % 32U);
    } else {
        shiftAmt = 0U;
    }
    RagValue1 = (gpioBaseAddr[IntTypeIndex] >> shiftAmt) & 0x01;
    RagValue2 = (gpioBaseAddr[IntPolIndex] >> shiftAmt) & 0x01;

    if ((RagValue1 == 0) && (RagValue2 == 0)) {
        return GPIO_INTERRUPT_TYPE_LEVEL_LOW;
    } else if ((RagValue1 == 0) && (RagValue2 == 1)) {
        return GPIO_INTERRUPT_TYPE_LEVEL_HIGH;
    } else if ((RagValue1 == 1) && (RagValue2 == 0)) {
        return GPIO_INTERRUPT_TYPE_EDGE_FALL;
    } else {
        return GPIO_INTERRUPT_TYPE_EDGE_RISE;
    }
}

/*******************************************************************************
 *
 * GPIO_enableInterrupt_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_enableInterrupt_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntMaskIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntMaskIndex = GPIO_INTMASK_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    gpioBaseAddr[IntMaskIndex] |= pinMask;
}

/*******************************************************************************
 *
 * GPIO_disableInterrupt_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_disableInterrupt_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntMaskIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntMaskIndex = GPIO_INTMASK_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    gpioBaseAddr[IntMaskIndex] &= (~pinMask);
}

/*******************************************************************************
 *
 * GPIO_getInterruptState_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T bool
GPIO_getInterruptState_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntMaskIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntMaskIndex = GPIO_INTMASK_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    return ((gpioBaseAddr[IntMaskIndex]) & pinMask);
}

/*******************************************************************************
 *
 * GPIO_clearInterrupt_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_clearInterrupt_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntClrIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntClrIndex = GPIO_INTCLR_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    gpioBaseAddr[IntClrIndex] |= pinMask;
}

/*******************************************************************************
 *
 * GPIO_readInterruptRaw_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T bool
GPIO_readInterruptRaw_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntRawIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntRawIndex = GPIO_INTRAW_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    return ((gpioBaseAddr[IntRawIndex]) & pinMask);
}

/*******************************************************************************
 *
 * GPIO_readInterruptStatus_NO_XInt
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T bool
GPIO_readInterruptStatus_NO_XInt(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           IntStatusIndex;
    uint32_t           pinMask;

    gpioBaseAddr = (uint32_t *)(GPIOINTERRUPT_BASE +
                                (pin / 32U) * GPIO_INT_REGS_STEP);

    IntStatusIndex = GPIO_INTSTATUS_INDEX;
    if (pin != 200) // Non GPOG configuration
    {
        pinMask = (uint32_t)1U << (pin % 32U);
    } else {
        pinMask = (uint32_t)1U;
    }
    return ((gpioBaseAddr[IntStatusIndex]) & pinMask);
}

#else

/*******************************************************************************
 *
 *  GPIO_setPinConfig
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setPinConfig(uint32_t pinConfig)
{
    uint32_t muxRegAddr;
    uint32_t pinMask, shiftAmt;

    muxRegAddr = (uint32_t)IO_CFG_BASE + (pinConfig >> 16);
    shiftAmt   = ((pinConfig >> 8) & (uint32_t)0xFFU);
    pinMask    = (uint32_t)0x3U << shiftAmt;

    //
    // Clear fields in MUX register first to avoid glitches
    //
    HWREG(muxRegAddr) &= ~pinMask;

    //
    // Write value into MUX register
    //
    HWREG(muxRegAddr) |= ((pinConfig & (uint32_t)0x3U) << shiftAmt);

}

/*******************************************************************************
 *
 *  GPIO_setPadConfig
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setPadConfig(uint32_t pin, uint32_t pinType)
{
    volatile uint32_t *gpioBaseAddr1;
    volatile uint32_t *gpioBaseAddr2;
    uint32_t           PudIndex;
    uint32_t           PddIndex;
    uint32_t           OutInvIndex;
    uint32_t           pinMask;

    gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
                            (pin / 32U) * (GPIO_O_GPBCTRL_3527 - GPIO_O_GPACTRL_3527));
    gpioBaseAddr2 = (uint32_t *)(IO_CFG_BASE +
                            GPIO_O_GPAPDD + (pin / 32U) * GPIO_EXT_REGS_STEP);

    PudIndex      = GPIO_GPxPUD_INDEX;
    PddIndex      = GPIO_GPxPDD_INDEX;
    OutInvIndex   = GPIO_GPxOUTINV_INDEX;
    pinMask       = (uint32_t)1U << (pin % 32U);

    //
    // Enable pull-up if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_PULLUP) != 0U) {
        gpioBaseAddr1[PudIndex] &= ~pinMask;
    } else {
        gpioBaseAddr1[PudIndex] |= pinMask;
    }

    //
    // Pull-Down if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_PULLDOWN) != 0U) {
        gpioBaseAddr2[PddIndex] &= ~pinMask;
    } else {
        gpioBaseAddr2[PddIndex] |= pinMask;
    }

    //
    // Inverted Push-Pull if necessary
    //
    if ((pinType & GPIO_PIN_TYPE_OUTPUT_INVERT) != 0U) {
        gpioBaseAddr2[OutInvIndex] |= pinMask;
    } else {
        gpioBaseAddr2[OutInvIndex] &= ~pinMask;
    }
}

/*******************************************************************************
 *
 *  GPIO_getPadConfig
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_getPadConfig(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr1;
    volatile uint32_t *gpioBaseAddr2;
    uint32_t PudIndex;
    uint32_t PddIndex;
    uint32_t OutInvIndex;

    uint32_t pinMask;
    uint32_t pinTypeRes;

    gpioBaseAddr1 = (uint32_t *)(GPIOCTRL_BASE +
                            (pin / 32U) * (GPIO_O_GPBCTRL_3527 - GPIO_O_GPACTRL_3527));
    gpioBaseAddr2 = (uint32_t *)(IO_CFG_BASE +
                            GPIO_O_GPAPDD + (pin / 32U) * GPIO_EXT_REGS_STEP);

    PudIndex      = GPIO_GPxPUD_INDEX;
    PddIndex      = GPIO_GPxPDD_INDEX;
    OutInvIndex   = GPIO_GPxOUTINV_INDEX;
    pinMask       = (uint32_t)1U << (pin % 32U);

    pinTypeRes    = GPIO_PIN_TYPE_STD;

    //
    // Get pull-up value
    //
    if ((gpioBaseAddr1[PudIndex] & pinMask) == 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_PULLUP;
    }

    //
    // Get Pull-Down value
    //
    if ((gpioBaseAddr2[PddIndex] & pinMask) != 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_PULLDOWN;
    }

    //
    // Get Inverted Push-Pull value
    //
    if ((gpioBaseAddr2[OutInvIndex] & pinMask) != 0U) {
        pinTypeRes |= GPIO_PIN_TYPE_OUTPUT_INVERT;
    }

    return (pinTypeRes);
}

/*******************************************************************************
 *
 * GPIO_setStrength : I2C GPIO : 4MA/20MA   Standard GPIO : 2MA/4MA/8MA/12MA
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setStrength(uint32_t pin, GPIO_DriveStrength StrengthLevel)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           DsIndex;

    //
    // Check the arguments.
    //
    ASSERT(GPIO_isPinValid(pin));

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    // Find the gpio_comfig_t structure corresponding to the pin
    const gpio_config_t *config = GPIO_getConfigByPin(pin);
    ASSERT(config);

    if (config->type == GPIO_TYPE_PI2C) {
        pinMask = (uint32_t)StrengthLevel >> 2;
    } else if (config->type == GPIO_TYPE_GPIO) {
        pinMask = (uint32_t)StrengthLevel & 0x3;
    }

	DsIndex = GPIO_GPxDS_INDEX + ((pin % 32U) / 16U);

    //
    // Enable driver strength
    //
    gpioBaseAddr[DsIndex] &= ~(0x3 << (2 * (pin % 16U)));
    gpioBaseAddr[DsIndex] |= (pinMask << (config->pin_S));
}

/*******************************************************************************
 *
 *  GPIO_setSchmittTrigger
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setSchmittTrigger(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           IsIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

	pinMask = (uint32_t)1U << (pin % 32U);
	IsIndex = GPIO_GPxIS_INDEX;

    if (pinType) {
        gpioBaseAddr[IsIndex] |= pinMask;
    } else {
        gpioBaseAddr[IsIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_setOutPolarityInv
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setOutPolarityInv(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           OutInvIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

	pinMask     = (uint32_t)1U << (pin % 32U);
	OutInvIndex = GPIO_GPxOUTINV_INDEX;

    if (pinType) {
        gpioBaseAddr[OutInvIndex] |= pinMask;
    } else {
        gpioBaseAddr[OutInvIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_setLoopBack
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setLoopBack(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           LoopEnIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

	pinMask     = (uint32_t)1U << (pin % 32U);
	LoopEnIndex = GPIO_GPxLOOPEN_INDEX;

    if (pinType) {
        gpioBaseAddr[LoopEnIndex] |= pinMask;
    } else {
        gpioBaseAddr[LoopEnIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_setI2CMode
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T void
GPIO_setI2CMode(uint32_t pin, bool pinType)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           pinMask;
    uint32_t           I2cModeIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if ((pin != 227) || (pin != 230)) // Non GPOH configuration
    {
        pinMask      = (uint32_t)1U << (pin % 32U);
        I2cModeIndex = GPIO_GPxPI2C_MODE_INDEX;
    } else {
        pinMask      = (uint32_t)1U << (pin % 32U);
        I2cModeIndex = (GPIO_O_GPHPI2C_MODE - GPIO_O_GPHPDD) / 4;
    }

    if (pinType) {
        gpioBaseAddr[I2cModeIndex] |= pinMask;
    } else {
        gpioBaseAddr[I2cModeIndex] &= ~(pinMask);
    }
}

/*******************************************************************************
 *
 *  GPIO_readPinOutReport
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_readPinOutReport(uint32_t pin)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           RegValue;
    uint32_t           PoutRptIndex;

    gpioBaseAddr = (uint32_t *)(GPIOEXT_BASE + (pin / 32U) * GPIO_EXT_REGS_STEP);

    if (pin == 224) // GPIOH configuration
    {
        PoutRptIndex = (GPIO_O_GPH_POUT_RPT - GPIO_O_GPHPDD) / 4;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((RegValue >> (0U)) & 0x1);
    } else // Non GPIOH configuration
    {
        PoutRptIndex = GPIO_GPxPOUT_RPT_INDEX;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((RegValue >> (pin % 32U)) & 0x1);
    }
}

/*******************************************************************************
 *
 *  GPIO_readPortOutReport
 *
 *******************************************************************************/
GS32_DRIVER_GPIO_FUNC_T uint32_t
GPIO_readPortOutReport(GPIO_Port_32BIT port)
{
    volatile uint32_t *gpioBaseAddr;
    uint32_t           RegValue;
    uint32_t           PoutRptIndex;

    gpioBaseAddr = (uint32_t *)(IO_CFG_BASE + (port * GPIO_EXT_REGS_STEP));
    if (port == GPIO_PORTH) {
        PoutRptIndex = (GPIO_O_GPH_POUT_RPT - GPIO_O_GPHPDD) / 4;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((uint32_t)(RegValue & 0x01));
    } else {
        PoutRptIndex = GPIO_GPxPOUT_RPT_INDEX;
        RegValue     = gpioBaseAddr[PoutRptIndex];
        return ((uint32_t)(RegValue & 0xFFFFFFFF));
    }
}

#endif /*!(GS32_PART_NUM==0x035 || GS32_PART_NUM==0x027)*/

#endif /*IS_GS32F00xx(0x30)*/
