/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    sysctl_subcrg_v22.h
*   @brief   do NOT include this file directly, include "sysctl.h" instead
*
*/
/*
*  commit history
*  20241218,Hezhiyuan,Modify comments
*/
#ifndef DEVICE_DRIVERLIB_SYSCTL_SUBCRG_V22_H_
#define DEVICE_DRIVERLIB_SYSCTL_SUBCRG_V22_H_

#include "gs32_version.h"

#if IS_GS32F3xx(0x22)

#ifdef __cplusplus
extern "C"{
#endif

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_sysctl_subcrg.h"

#define CRG_CFG_ADDR_SHIFT      25
#define CRG_CFG_ADDR_MASK       0x7F
#define CRG_CFG_MASK_SHIFT      5
#define CRG_CFG_MASK_MASK       0xFFFFF
#define CRG_CFG_OFF_SHIFT       0
#define CRG_CFG_OFF_MASK        0x1F

#define GS32F3xx_CRG_CFG(addr, mask, off)        ((addr) <<CRG_CFG_ADDR_SHIFT | (mask)<<CRG_CFG_MASK_SHIFT | (off))


/**
 * 
 *  The following are values that can be passed to SysCtl_enablePeripheral()
 *  and SysCtl_disablePeripheral() as the  peripheral parameter.
 * 
 */
typedef enum
{
    /*offset == 0x08*/
    SYSCTL_PERIPH_CLK_TIMER    = GS32F3xx_CRG_CFG(0x8, 0x01, 0),

    SYSCTL_PERIPH_CLK_SCI      = GS32F3xx_CRG_CFG(0x8, 0x3F, 2),
    SYSCTL_PERIPH_CLK_SCIA     = GS32F3xx_CRG_CFG(0x8, 0x01, 2),
    SYSCTL_PERIPH_CLK_SCIB     = GS32F3xx_CRG_CFG(0x8, 0x01, 3),
    SYSCTL_PERIPH_CLK_SCIC     = GS32F3xx_CRG_CFG(0x8, 0x01, 4),
    SYSCTL_PERIPH_CLK_SCID     = GS32F3xx_CRG_CFG(0x8, 0x01, 5),
    SYSCTL_PERIPH_CLK_UARTA    = GS32F3xx_CRG_CFG(0x8, 0x01, 6),
    SYSCTL_PERIPH_CLK_UARTB    = GS32F3xx_CRG_CFG(0x8, 0x01, 7),

    SYSCTL_PERIPH_CLK_I2C      = GS32F3xx_CRG_CFG(0x8, 0x07, 8),
    SYSCTL_PERIPH_CLK_I2CA     = GS32F3xx_CRG_CFG(0x8, 0x01, 8),
    SYSCTL_PERIPH_CLK_I2CB     = GS32F3xx_CRG_CFG(0x8, 0x01, 9),
    SYSCTL_PERIPH_CLK_CMI2C    = GS32F3xx_CRG_CFG(0x8, 0x01, 10),

    SYSCTL_PERIPH_CLK_PMBUS    = GS32F3xx_CRG_CFG(0x8, 0x01, 11),

    SYSCTL_PERIPH_CLK_SPI      = GS32F3xx_CRG_CFG(0x8, 0x1F, 16),
    SYSCTL_PERIPH_CLK_SPIA     = GS32F3xx_CRG_CFG(0x8, 0x01, 16),
    SYSCTL_PERIPH_CLK_SPIB     = GS32F3xx_CRG_CFG(0x8, 0x01, 17),
    SYSCTL_PERIPH_CLK_SPIC     = GS32F3xx_CRG_CFG(0x8, 0x01, 18),
    SYSCTL_PERIPH_CLK_SPID     = GS32F3xx_CRG_CFG(0x8, 0x01, 19),
    SYSCTL_PERIPH_CLK_CMSPI    = GS32F3xx_CRG_CFG(0x8, 0x01, 20),

    SYSCTL_PERIPH_CLK_CAN      = GS32F3xx_CRG_CFG(0x8, 0x07, 24),
    SYSCTL_PERIPH_CLK_CANA     = GS32F3xx_CRG_CFG(0x8, 0x01, 24),
    SYSCTL_PERIPH_CLK_CANB     = GS32F3xx_CRG_CFG(0x8, 0x01, 25),
    SYSCTL_PERIPH_CLK_CANC     = GS32F3xx_CRG_CFG(0x8, 0x01, 26),

    SYSCTL_PERIPH_CLK_LIN      = GS32F3xx_CRG_CFG(0x8, 0x03, 29),
    SYSCTL_PERIPH_CLK_LINA     = GS32F3xx_CRG_CFG(0x8, 0x01, 29),
    SYSCTL_PERIPH_CLK_LINB     = GS32F3xx_CRG_CFG(0x8, 0x01, 30),

    /*offset == 0x0C*/
    SYSCTL_PERIPH_CLK_CRC      = GS32F3xx_CRG_CFG(0xC, 0x01, 4),

    SYSCTL_PERIPH_CLK_DAC      = GS32F3xx_CRG_CFG(0xC, 0x0F, 8),
    SYSCTL_PERIPH_CLK_DACA     = GS32F3xx_CRG_CFG(0xC, 0x01, 8),
    SYSCTL_PERIPH_CLK_DACB     = GS32F3xx_CRG_CFG(0xC, 0x01, 9),
    SYSCTL_PERIPH_CLK_DACC     = GS32F3xx_CRG_CFG(0xC, 0x01, 10),
    SYSCTL_PERIPH_CLK_DACCOM   = GS32F3xx_CRG_CFG(0xC, 0x01, 11),

    SYSCTL_PERIPH_CLK_EPG      = GS32F3xx_CRG_CFG(0xC, 0x01, 12),
    SYSCTL_PERIPH_CLK_XBAR     = GS32F3xx_CRG_CFG(0xC, 0x01, 20),

    /*offset == 0x10*/
    SYSCTL_PERIPH_CLK_CLB      = GS32F3xx_CRG_CFG(0x10, 0x1FF, 0),
    SYSCTL_PERIPH_CLK_CLB1     = GS32F3xx_CRG_CFG(0x10, 0x01, 0),
    SYSCTL_PERIPH_CLK_CLB2     = GS32F3xx_CRG_CFG(0x10, 0x01, 1),
    SYSCTL_PERIPH_CLK_CLB3     = GS32F3xx_CRG_CFG(0x10, 0x01, 2),
    SYSCTL_PERIPH_CLK_CLB4     = GS32F3xx_CRG_CFG(0x10, 0x01, 3),
    SYSCTL_PERIPH_CLK_CLB5     = GS32F3xx_CRG_CFG(0x10, 0x01, 4),
    SYSCTL_PERIPH_CLK_CLB6     = GS32F3xx_CRG_CFG(0x10, 0x01, 5),
    SYSCTL_PERIPH_CLK_CLB7     = GS32F3xx_CRG_CFG(0x10, 0x01, 6),
    SYSCTL_PERIPH_CLK_CLB8     = GS32F3xx_CRG_CFG(0x10, 0x01, 7),
    SYSCTL_PERIPH_CLK_CLBCOM   = GS32F3xx_CRG_CFG(0x10, 0x01, 8),

    SYSCTL_PERIPH_CLK_SDFM     = GS32F3xx_CRG_CFG(0x10, 0x01F, 12),
    SYSCTL_PERIPH_CLK_SDFM1    = GS32F3xx_CRG_CFG(0x10, 0x01, 12),
    SYSCTL_PERIPH_CLK_SDFM2    = GS32F3xx_CRG_CFG(0x10, 0x01, 13),
    SYSCTL_PERIPH_CLK_SDFM3    = GS32F3xx_CRG_CFG(0x10, 0x01, 14),
    SYSCTL_PERIPH_CLK_SDFM4    = GS32F3xx_CRG_CFG(0x10, 0x01, 15),
    SYSCTL_PERIPH_CLK_SDFMCOM  = GS32F3xx_CRG_CFG(0x10, 0x01, 16),

    SYSCTL_PERIPH_CLK_EQEP     = GS32F3xx_CRG_CFG(0x10, 0x3F, 20),
    SYSCTL_PERIPH_CLK_EQEP1    = GS32F3xx_CRG_CFG(0x10, 0x01, 20),
    SYSCTL_PERIPH_CLK_EQEP2    = GS32F3xx_CRG_CFG(0x10, 0x01, 21),
    SYSCTL_PERIPH_CLK_EQEP3    = GS32F3xx_CRG_CFG(0x10, 0x01, 22),
    SYSCTL_PERIPH_CLK_EQEP4    = GS32F3xx_CRG_CFG(0x10, 0x01, 23),
    SYSCTL_PERIPH_CLK_EQEP5    = GS32F3xx_CRG_CFG(0x10, 0x01, 24),
    SYSCTL_PERIPH_CLK_EQEP6    = GS32F3xx_CRG_CFG(0x10, 0x01, 25),

    /*offset == 0x14*/
    SYSCTL_PERIPH_CLK_EPWM     = GS32F3xx_CRG_CFG(0x14, 0x7FFFF, 0),
    SYSCTL_PERIPH_CLK_EPWM1    = GS32F3xx_CRG_CFG(0x14, 0x1, 0),
    SYSCTL_PERIPH_CLK_EPWM2    = GS32F3xx_CRG_CFG(0x14, 0x1, 1),
    SYSCTL_PERIPH_CLK_EPWM3    = GS32F3xx_CRG_CFG(0x14, 0x1, 2),
    SYSCTL_PERIPH_CLK_EPWM4    = GS32F3xx_CRG_CFG(0x14, 0x1, 3),
    SYSCTL_PERIPH_CLK_EPWM5    = GS32F3xx_CRG_CFG(0x14, 0x1, 4),
    SYSCTL_PERIPH_CLK_EPWM6    = GS32F3xx_CRG_CFG(0x14, 0x1, 5),
    SYSCTL_PERIPH_CLK_EPWM7    = GS32F3xx_CRG_CFG(0x14, 0x1, 6),
    SYSCTL_PERIPH_CLK_EPWM8    = GS32F3xx_CRG_CFG(0x14, 0x1, 7),
    SYSCTL_PERIPH_CLK_EPWM9    = GS32F3xx_CRG_CFG(0x14, 0x1, 8),
    SYSCTL_PERIPH_CLK_EPWM10   = GS32F3xx_CRG_CFG(0x14, 0x1, 9),
    SYSCTL_PERIPH_CLK_EPWM11   = GS32F3xx_CRG_CFG(0x14, 0x1, 10),
    SYSCTL_PERIPH_CLK_EPWM12   = GS32F3xx_CRG_CFG(0x14, 0x1, 11),
    SYSCTL_PERIPH_CLK_EPWM13   = GS32F3xx_CRG_CFG(0x14, 0x1, 12),
    SYSCTL_PERIPH_CLK_EPWM14   = GS32F3xx_CRG_CFG(0x14, 0x1, 13),
    SYSCTL_PERIPH_CLK_EPWM15   = GS32F3xx_CRG_CFG(0x14, 0x1, 14),
    SYSCTL_PERIPH_CLK_EPWM16   = GS32F3xx_CRG_CFG(0x14, 0x1, 15),
    SYSCTL_PERIPH_CLK_EPWM17   = GS32F3xx_CRG_CFG(0x14, 0x1, 16),
    SYSCTL_PERIPH_CLK_EPWM18   = GS32F3xx_CRG_CFG(0x14, 0x1, 17),
    SYSCTL_PERIPH_CLK_EPWMCOM  = GS32F3xx_CRG_CFG(0x14, 0x1, 18),

    /*offset == 0x18*/
    SYSCTL_PERIPH_CLK_ECAP     = GS32F3xx_CRG_CFG(0x18, 0x7F, 0),
    SYSCTL_PERIPH_CLK_ECAP1    = GS32F3xx_CRG_CFG(0x18, 0x01, 0),
    SYSCTL_PERIPH_CLK_ECAP2    = GS32F3xx_CRG_CFG(0x18, 0x01, 1),
    SYSCTL_PERIPH_CLK_ECAP3    = GS32F3xx_CRG_CFG(0x18, 0x01, 2),
    SYSCTL_PERIPH_CLK_ECAP4    = GS32F3xx_CRG_CFG(0x18, 0x01, 3),
    SYSCTL_PERIPH_CLK_ECAP5    = GS32F3xx_CRG_CFG(0x18, 0x01, 4),
    SYSCTL_PERIPH_CLK_ECAP6    = GS32F3xx_CRG_CFG(0x18, 0x01, 5),
    SYSCTL_PERIPH_CLK_ECAP7    = GS32F3xx_CRG_CFG(0x18, 0x01, 6),

    /*offset == 0x1C*/
    SYSCTL_PERIPH_CLK_CMPSS    = GS32F3xx_CRG_CFG(0x1C, 0xFFF, 0),
    SYSCTL_PERIPH_CLK_CMPSS1   = GS32F3xx_CRG_CFG(0x1C, 0x1, 0),
    SYSCTL_PERIPH_CLK_CMPSS2   = GS32F3xx_CRG_CFG(0x1C, 0x1, 1),
    SYSCTL_PERIPH_CLK_CMPSS3   = GS32F3xx_CRG_CFG(0x1C, 0x1, 2),
    SYSCTL_PERIPH_CLK_CMPSS4   = GS32F3xx_CRG_CFG(0x1C, 0x1, 3),
    SYSCTL_PERIPH_CLK_CMPSS5   = GS32F3xx_CRG_CFG(0x1C, 0x1, 4),
    SYSCTL_PERIPH_CLK_CMPSS6   = GS32F3xx_CRG_CFG(0x1C, 0x1, 5),
    SYSCTL_PERIPH_CLK_CMPSS7   = GS32F3xx_CRG_CFG(0x1C, 0x1, 6),
    SYSCTL_PERIPH_CLK_CMPSS8   = GS32F3xx_CRG_CFG(0x1C, 0x1, 7),
    SYSCTL_PERIPH_CLK_CMPSS9   = GS32F3xx_CRG_CFG(0x1C, 0x1, 8),
    SYSCTL_PERIPH_CLK_CMPSS10  = GS32F3xx_CRG_CFG(0x1C, 0x1, 9),
    SYSCTL_PERIPH_CLK_CMPSS11  = GS32F3xx_CRG_CFG(0x1C, 0x1, 10),
    SYSCTL_PERIPH_CLK_CMPSS12  = GS32F3xx_CRG_CFG(0x1C, 0x1, 11),

    /*offset == 0x20*/
    SYSCTL_PERIPH_CLK_GPIO     = GS32F3xx_CRG_CFG(0x20, 0x3FFF, 0),
    SYSCTL_PERIPH_CLK_GPIO1    = GS32F3xx_CRG_CFG(0x20, 0x1, 0),
    SYSCTL_PERIPH_CLK_GPIO2    = GS32F3xx_CRG_CFG(0x20, 0x1, 1),
    SYSCTL_PERIPH_CLK_GPIO3    = GS32F3xx_CRG_CFG(0x20, 0x1, 2),
    SYSCTL_PERIPH_CLK_GPIO4    = GS32F3xx_CRG_CFG(0x20, 0x1, 3),
    SYSCTL_PERIPH_CLK_GPIO5    = GS32F3xx_CRG_CFG(0x20, 0x1, 4),
    SYSCTL_PERIPH_CLK_GPIO6    = GS32F3xx_CRG_CFG(0x20, 0x1, 5),
    SYSCTL_PERIPH_CLK_GPIO7    = GS32F3xx_CRG_CFG(0x20, 0x1, 6),
    SYSCTL_PERIPH_CLK_GPIO8    = GS32F3xx_CRG_CFG(0x20, 0x1, 7),
    SYSCTL_PERIPH_CLK_GPIO9    = GS32F3xx_CRG_CFG(0x20, 0x1, 8),
    SYSCTL_PERIPH_CLK_GPIO10   = GS32F3xx_CRG_CFG(0x20, 0x1, 9),
    SYSCTL_PERIPH_CLK_GPIO11   = GS32F3xx_CRG_CFG(0x20, 0x1, 10),
    SYSCTL_PERIPH_CLK_GPIO12   = GS32F3xx_CRG_CFG(0x20, 0x1, 11),
    SYSCTL_PERIPH_CLK_GPIO13   = GS32F3xx_CRG_CFG(0x20, 0x1, 12),
    SYSCTL_PERIPH_CLK_GPIO14   = GS32F3xx_CRG_CFG(0x20, 0x1, 13),

    /*offset == 0x24*/
    SYSCTL_PERIPH_CLK_ADC      = GS32F3xx_CRG_CFG(0x24, 0x1F, 0),
    SYSCTL_PERIPH_CLK_ADCA     = GS32F3xx_CRG_CFG(0x24, 0x1, 0),
    SYSCTL_PERIPH_CLK_ADCB     = GS32F3xx_CRG_CFG(0x24, 0x1, 1),
    SYSCTL_PERIPH_CLK_ADCC     = GS32F3xx_CRG_CFG(0x24, 0x1, 2),
    SYSCTL_PERIPH_CLK_ADCD     = GS32F3xx_CRG_CFG(0x24, 0x1, 3),
    SYSCTL_PERIPH_CLK_ADCCOM   = GS32F3xx_CRG_CFG(0x24, 0x1, 4),

    SYSCTL_PERIPH_CLK_TBCLKSYNC = 0xFFFFFFFF,
} SysCtl_PeripheralPCLOCKCR;

/**
 * 
 *  The following are values that can be passed to SysCtl_resetPeripheral() as
 *  the  peripheral parameter.
 * 
 */
typedef enum
{
    /*offset == 0x100*/
    SYSCTL_PERIPH_RST_TIMER    = GS32F3xx_CRG_CFG(0x0, 0x01, 0),

    SYSCTL_PERIPH_RST_SCI      = GS32F3xx_CRG_CFG(0x0, 0x3F, 2),
    SYSCTL_PERIPH_RST_SCIA     = GS32F3xx_CRG_CFG(0x0, 0x01, 2),
    SYSCTL_PERIPH_RST_SCIB     = GS32F3xx_CRG_CFG(0x0, 0x01, 3),
    SYSCTL_PERIPH_RST_SCIC     = GS32F3xx_CRG_CFG(0x0, 0x01, 4),
    SYSCTL_PERIPH_RST_SCID     = GS32F3xx_CRG_CFG(0x0, 0x01, 5),
    SYSCTL_PERIPH_RST_UARTA    = GS32F3xx_CRG_CFG(0x0, 0x01, 6),
    SYSCTL_PERIPH_RST_UARTB    = GS32F3xx_CRG_CFG(0x0, 0x01, 7),

    SYSCTL_PERIPH_RST_I2C      = GS32F3xx_CRG_CFG(0x0, 0x07, 8),
    SYSCTL_PERIPH_RST_I2CA     = GS32F3xx_CRG_CFG(0x0, 0x01, 8),
    SYSCTL_PERIPH_RST_I2CB     = GS32F3xx_CRG_CFG(0x0, 0x01, 9),
    SYSCTL_PERIPH_RST_CMI2C    = GS32F3xx_CRG_CFG(0x0, 0x01, 10),

    SYSCTL_PERIPH_RST_PMBUS    = GS32F3xx_CRG_CFG(0x0, 0x01, 11),

    SYSCTL_PERIPH_RST_SPI      = GS32F3xx_CRG_CFG(0x0, 0x1F, 16),
    SYSCTL_PERIPH_RST_SPIA     = GS32F3xx_CRG_CFG(0x0, 0x01, 16),
    SYSCTL_PERIPH_RST_SPIB     = GS32F3xx_CRG_CFG(0x0, 0x01, 17),
    SYSCTL_PERIPH_RST_SPIC     = GS32F3xx_CRG_CFG(0x0, 0x01, 18),
    SYSCTL_PERIPH_RST_SPID     = GS32F3xx_CRG_CFG(0x0, 0x01, 19),
    SYSCTL_PERIPH_RST_CMSPI    = GS32F3xx_CRG_CFG(0x0, 0x01, 20),

    SYSCTL_PERIPH_RST_CAN      = GS32F3xx_CRG_CFG(0x0, 0x07, 24),
    SYSCTL_PERIPH_RST_CANA     = GS32F3xx_CRG_CFG(0x0, 0x01, 24),
    SYSCTL_PERIPH_RST_CANB     = GS32F3xx_CRG_CFG(0x0, 0x01, 25),
    SYSCTL_PERIPH_RST_CANC     = GS32F3xx_CRG_CFG(0x0, 0x01, 26),

    SYSCTL_PERIPH_RST_LIN      = GS32F3xx_CRG_CFG(0x0, 0x03, 29),
    SYSCTL_PERIPH_RST_LINA     = GS32F3xx_CRG_CFG(0x0, 0x01, 29),
    SYSCTL_PERIPH_RST_LINB     = GS32F3xx_CRG_CFG(0x0, 0x01, 30),

    /*offset == 0x104*/
    SYSCTL_PERIPH_RST_CRC      = GS32F3xx_CRG_CFG(0x4, 0x01, 4),

    SYSCTL_PERIPH_RST_DAC      = GS32F3xx_CRG_CFG(0x4, 0x0F, 8),
    SYSCTL_PERIPH_RST_DACA     = GS32F3xx_CRG_CFG(0x4, 0x01, 8),
    SYSCTL_PERIPH_RST_DACB     = GS32F3xx_CRG_CFG(0x4, 0x01, 9),
    SYSCTL_PERIPH_RST_DACC     = GS32F3xx_CRG_CFG(0x4, 0x01, 10),
    SYSCTL_PERIPH_RST_DACCOM   = GS32F3xx_CRG_CFG(0x4, 0x01, 11),

    SYSCTL_PERIPH_RST_EPG      = GS32F3xx_CRG_CFG(0x4, 0x01, 12),
    SYSCTL_PERIPH_RST_XBAR     = GS32F3xx_CRG_CFG(0x4, 0x01, 20),

    /*offset == 0x108*/
    SYSCTL_PERIPH_RST_CLB      = GS32F3xx_CRG_CFG(0x8, 0x1FF, 0),
    SYSCTL_PERIPH_RST_CLB1     = GS32F3xx_CRG_CFG(0x8, 0x01, 0),
    SYSCTL_PERIPH_RST_CLB2     = GS32F3xx_CRG_CFG(0x8, 0x01, 1),
    SYSCTL_PERIPH_RST_CLB3     = GS32F3xx_CRG_CFG(0x8, 0x01, 2),
    SYSCTL_PERIPH_RST_CLB4     = GS32F3xx_CRG_CFG(0x8, 0x01, 3),
    SYSCTL_PERIPH_RST_CLB5     = GS32F3xx_CRG_CFG(0x8, 0x01, 4),
    SYSCTL_PERIPH_RST_CLB6     = GS32F3xx_CRG_CFG(0x8, 0x01, 5),
    SYSCTL_PERIPH_RST_CLB7     = GS32F3xx_CRG_CFG(0x8, 0x01, 6),
    SYSCTL_PERIPH_RST_CLB8     = GS32F3xx_CRG_CFG(0x8, 0x01, 7),
    SYSCTL_PERIPH_RST_CLBCOM   = GS32F3xx_CRG_CFG(0x8, 0x01, 8),

    SYSCTL_PERIPH_RST_SDFM     = GS32F3xx_CRG_CFG(0x8, 0x01F, 12),
    SYSCTL_PERIPH_RST_SDFM1    = GS32F3xx_CRG_CFG(0x8, 0x01, 12),
    SYSCTL_PERIPH_RST_SDFM2    = GS32F3xx_CRG_CFG(0x8, 0x01, 13),
    SYSCTL_PERIPH_RST_SDFM3    = GS32F3xx_CRG_CFG(0x8, 0x01, 14),
    SYSCTL_PERIPH_RST_SDFM4    = GS32F3xx_CRG_CFG(0x8, 0x01, 15),
    SYSCTL_PERIPH_RST_SDFMCOM  = GS32F3xx_CRG_CFG(0x8, 0x01, 16),

    SYSCTL_PERIPH_RST_EQEP     = GS32F3xx_CRG_CFG(0x8, 0x3F, 20),
    SYSCTL_PERIPH_RST_EQEP1    = GS32F3xx_CRG_CFG(0x8, 0x01, 20),
    SYSCTL_PERIPH_RST_EQEP2    = GS32F3xx_CRG_CFG(0x8, 0x01, 21),
    SYSCTL_PERIPH_RST_EQEP3    = GS32F3xx_CRG_CFG(0x8, 0x01, 22),
    SYSCTL_PERIPH_RST_EQEP4    = GS32F3xx_CRG_CFG(0x8, 0x01, 23),
    SYSCTL_PERIPH_RST_EQEP5    = GS32F3xx_CRG_CFG(0x8, 0x01, 24),
    SYSCTL_PERIPH_RST_EQEP6    = GS32F3xx_CRG_CFG(0x8, 0x01, 25),

    /*offset == 0x10C*/
    SYSCTL_PERIPH_RST_EPWM     = GS32F3xx_CRG_CFG(0xC, 0x7FFFF, 0),
    SYSCTL_PERIPH_RST_EPWM1    = GS32F3xx_CRG_CFG(0xC, 0x1, 0),
    SYSCTL_PERIPH_RST_EPWM2    = GS32F3xx_CRG_CFG(0xC, 0x1, 1),
    SYSCTL_PERIPH_RST_EPWM3    = GS32F3xx_CRG_CFG(0xC, 0x1, 2),
    SYSCTL_PERIPH_RST_EPWM4    = GS32F3xx_CRG_CFG(0xC, 0x1, 3),
    SYSCTL_PERIPH_RST_EPWM5    = GS32F3xx_CRG_CFG(0xC, 0x1, 4),
    SYSCTL_PERIPH_RST_EPWM6    = GS32F3xx_CRG_CFG(0xC, 0x1, 5),
    SYSCTL_PERIPH_RST_EPWM7    = GS32F3xx_CRG_CFG(0xC, 0x1, 6),
    SYSCTL_PERIPH_RST_EPWM8    = GS32F3xx_CRG_CFG(0xC, 0x1, 7),
    SYSCTL_PERIPH_RST_EPWM9    = GS32F3xx_CRG_CFG(0xC, 0x1, 8),
    SYSCTL_PERIPH_RST_EPWM10   = GS32F3xx_CRG_CFG(0xC, 0x1, 9),
    SYSCTL_PERIPH_RST_EPWM11   = GS32F3xx_CRG_CFG(0xC, 0x1, 10),
    SYSCTL_PERIPH_RST_EPWM12   = GS32F3xx_CRG_CFG(0xC, 0x1, 11),
    SYSCTL_PERIPH_RST_EPWM13   = GS32F3xx_CRG_CFG(0xC, 0x1, 12),
    SYSCTL_PERIPH_RST_EPWM14   = GS32F3xx_CRG_CFG(0xC, 0x1, 13),
    SYSCTL_PERIPH_RST_EPWM15   = GS32F3xx_CRG_CFG(0xC, 0x1, 14),
    SYSCTL_PERIPH_RST_EPWM16   = GS32F3xx_CRG_CFG(0xC, 0x1, 15),
    SYSCTL_PERIPH_RST_EPWM17   = GS32F3xx_CRG_CFG(0xC, 0x1, 16),
    SYSCTL_PERIPH_RST_EPWM18   = GS32F3xx_CRG_CFG(0xC, 0x1, 17),
    SYSCTL_PERIPH_RST_EPWMCOM  = GS32F3xx_CRG_CFG(0xC, 0x1, 18),

    /*offset == 0x110*/
    SYSCTL_PERIPH_RST_ECAP     = GS32F3xx_CRG_CFG(0x10, 0x7F, 0),
    SYSCTL_PERIPH_RST_ECAP1    = GS32F3xx_CRG_CFG(0x10, 0x01, 0),
    SYSCTL_PERIPH_RST_ECAP2    = GS32F3xx_CRG_CFG(0x10, 0x01, 1),
    SYSCTL_PERIPH_RST_ECAP3    = GS32F3xx_CRG_CFG(0x10, 0x01, 2),
    SYSCTL_PERIPH_RST_ECAP4    = GS32F3xx_CRG_CFG(0x10, 0x01, 3),
    SYSCTL_PERIPH_RST_ECAP5    = GS32F3xx_CRG_CFG(0x10, 0x01, 4),
    SYSCTL_PERIPH_RST_ECAP6    = GS32F3xx_CRG_CFG(0x10, 0x01, 5),
    SYSCTL_PERIPH_RST_ECAP7    = GS32F3xx_CRG_CFG(0x10, 0x01, 6),

    /*offset == 0x114*/
    SYSCTL_PERIPH_RST_CMPSS    = GS32F3xx_CRG_CFG(0x14, 0xFFF, 0),
    SYSCTL_PERIPH_RST_CMPSS1   = GS32F3xx_CRG_CFG(0x14, 0x1, 0),
    SYSCTL_PERIPH_RST_CMPSS2   = GS32F3xx_CRG_CFG(0x14, 0x1, 1),
    SYSCTL_PERIPH_RST_CMPSS3   = GS32F3xx_CRG_CFG(0x14, 0x1, 2),
    SYSCTL_PERIPH_RST_CMPSS4   = GS32F3xx_CRG_CFG(0x14, 0x1, 3),
    SYSCTL_PERIPH_RST_CMPSS5   = GS32F3xx_CRG_CFG(0x14, 0x1, 4),
    SYSCTL_PERIPH_RST_CMPSS6   = GS32F3xx_CRG_CFG(0x14, 0x1, 5),
    SYSCTL_PERIPH_RST_CMPSS7   = GS32F3xx_CRG_CFG(0x14, 0x1, 6),
    SYSCTL_PERIPH_RST_CMPSS8   = GS32F3xx_CRG_CFG(0x14, 0x1, 7),
    SYSCTL_PERIPH_RST_CMPSS9   = GS32F3xx_CRG_CFG(0x14, 0x1, 8),
    SYSCTL_PERIPH_RST_CMPSS10  = GS32F3xx_CRG_CFG(0x14, 0x1, 9),
    SYSCTL_PERIPH_RST_CMPSS11  = GS32F3xx_CRG_CFG(0x14, 0x1, 10),
    SYSCTL_PERIPH_RST_CMPSS12  = GS32F3xx_CRG_CFG(0x14, 0x1, 11),

    /*offset == 0x118*/
    SYSCTL_PERIPH_RST_GPIO     = GS32F3xx_CRG_CFG(0x18, 0x3FFF, 0),
    SYSCTL_PERIPH_RST_GPIO1    = GS32F3xx_CRG_CFG(0x18, 0x1, 0),
    SYSCTL_PERIPH_RST_GPIO2    = GS32F3xx_CRG_CFG(0x18, 0x1, 1),
    SYSCTL_PERIPH_RST_GPIO3    = GS32F3xx_CRG_CFG(0x18, 0x1, 2),
    SYSCTL_PERIPH_RST_GPIO4    = GS32F3xx_CRG_CFG(0x18, 0x1, 3),
    SYSCTL_PERIPH_RST_GPIO5    = GS32F3xx_CRG_CFG(0x18, 0x1, 4),
    SYSCTL_PERIPH_RST_GPIO6    = GS32F3xx_CRG_CFG(0x18, 0x1, 5),
    SYSCTL_PERIPH_RST_GPIO7    = GS32F3xx_CRG_CFG(0x18, 0x1, 6),
    SYSCTL_PERIPH_RST_GPIO8    = GS32F3xx_CRG_CFG(0x18, 0x1, 7),
    SYSCTL_PERIPH_RST_GPIO9    = GS32F3xx_CRG_CFG(0x18, 0x1, 8),
    SYSCTL_PERIPH_RST_GPIO10   = GS32F3xx_CRG_CFG(0x18, 0x1, 9),
    SYSCTL_PERIPH_RST_GPIO11   = GS32F3xx_CRG_CFG(0x18, 0x1, 10),
    SYSCTL_PERIPH_RST_GPIO12   = GS32F3xx_CRG_CFG(0x18, 0x1, 11),
    SYSCTL_PERIPH_RST_GPIO13   = GS32F3xx_CRG_CFG(0x18, 0x1, 12),
    SYSCTL_PERIPH_RST_GPIO14   = GS32F3xx_CRG_CFG(0x18, 0x1, 13),

    /*offset == 0x11C*/
    SYSCTL_PERIPH_RST_ADC      = GS32F3xx_CRG_CFG(0x1C, 0x1F, 0),
    SYSCTL_PERIPH_RST_ADCA     = GS32F3xx_CRG_CFG(0x1C, 0x1, 0),
    SYSCTL_PERIPH_RST_ADCB     = GS32F3xx_CRG_CFG(0x1C, 0x1, 1),
    SYSCTL_PERIPH_RST_ADCC     = GS32F3xx_CRG_CFG(0x1C, 0x1, 2),
    SYSCTL_PERIPH_RST_ADCD     = GS32F3xx_CRG_CFG(0x1C, 0x1, 3),
    SYSCTL_PERIPH_RST_ADCCOM   = GS32F3xx_CRG_CFG(0x1C, 0x1, 4),
} SysCtl_PeripheralSOFTPRES;


typedef enum {
    SYSCTL_PERIPH_ICG_BYPASS_ADC          = 1<<0,
    SYSCTL_PERIPH_ICG_BYPASS_APB          = 1<<3,
    SYSCTL_PERIPH_ICG_BYPASS_AHB          = 1<<4,
    SYSCTL_PERIPH_ICG_BYPASS_EPWM         = 1<<5,
    SYSCTL_PERIPH_ICG_BYPASS_ECAP         = 1<<6,
    SYSCTL_PERIPH_ICG_BYPASS_EQEP         = 1<<7,
    SYSCTL_PERIPH_ICG_BYPASS_CMPSS        = 1<<8,
    SYSCTL_PERIPH_ICG_BYPASS_CLB          = 1<<9,
    SYSCTL_PERIPH_ICG_BYPASS_XBAR         = 1<<10,
    SYSCTL_PERIPH_ICG_BYPASS_SDFM         = 1<<11,
    SYSCTL_PERIPH_ICG_BYPASS_DAC          = 1<<12,
    SYSCTL_PERIPH_ICG_BYPASS_EPG          = 1<<14,
    SYSCTL_PERIPH_ICG_BYPASS_CRC          = 1<<16,
    SYSCTL_PERIPH_ICG_BYPASS_SEMA         = 1<<17,
    SYSCTL_PERIPH_ICG_BYPASS_DSP_DMA0     = 1<<19,
    SYSCTL_PERIPH_ICG_BYPASS_DSP_DMA1     = 1<<20,
    SYSCTL_PERIPH_ICG_BYPASS_MU0          = 1<<21,
    SYSCTL_PERIPH_ICG_BYPASS_MU1          = 1<<22,
}SysCtl_PeripheralICGBYPASS;


/*
 * SUB_CRG_LOCK   lock control register   offset 0x0000
 *
 *  Register Field                Field Range
 *  sub_crg_lock                  31 : 0
 *
 * Writing the unlock value 0x5A5A5A5A unlocks all lock-protected registers, allowing writes.
 * Writing any other value locks all lock-protected registers.
 *
 */
__STATIC_INLINE void SysCtl_setSubCrgLock(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_SUB_CRG_LOCK) = 0x01;
}

__STATIC_INLINE void SysCtl_setSubCrgUnlock(void)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_SUB_CRG_LOCK) = 0x5A5A5A5A;
}

__STATIC_INLINE uint32_t SysCtl_getSubCrgLock(void)
{
    return HWREG(SUB_CRG_BASE + SUBCRG_CFG_SUB_CRG_LOCK);
}

/*
 * PARA_ICG_BYPASS   PARA_ICG_BYPASS control register 1   offset 0x0030
 *
 *  Register Field                Field Range
 *  reserved                      31:1
 *  para_icg_bypass               0
 *
 */
__STATIC_INLINE void SysCtl_setParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PARA_ICG_BYPASS) = bypass & 0x01;
}

__STATIC_INLINE uint32_t SysCtl_getParaIcgBypass(void)
{
    return HWREG(SUB_CRG_BASE + SUBCRG_CFG_PARA_ICG_BYPASS) & 0x01;
}

/*
 * PERI_ICG_BYPASS   Peripheral subsystem module PARA_ICG_BYPASS control register 2   offset 0x0030
 *
 *  Register Field             Field Range
 *  reserved                    31:23
 *  mu1_para_icg_bypass         22
 *  mu0_para_icg_bypass         21
 *  dsp_dma1_para_icg_bypass    20
 *  dsp_dma0_para_icg_bypass    19
 *  sema_para_icg_bypass        17
 *  crc_para_icg_bypass         16
 *  reserved                    15
 *  epg_para_icg_bypass         14
 *  reserved                    13
 *  dac_para_icg_bypass         12
 *  sdfm_para_icg_bypass        11
 *  xbar_para_icg_bypass        10
 *  plu_para_icg_bypass         9
 *  cmpss_para_icg_bypass       8
 *  eqep_para_icg_bypass        7
 *  ecap_para_icg_bypass        6
 *  pwm_para_icg_bypass         5
 *  ahb1_comm_para_icg_bypass   4
 *  apb_comm_para_icg_bypass    3
 *  reserved                    2:1
 *  adc_para_icg_bypass         0
 *
 */

/* --- adc --- */
__STATIC_INLINE void SysCtl_setAdcParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<0)) | (bypass & 0x01);
}

__STATIC_INLINE uint32_t SysCtl_getAdcParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 0) & 0x01;
}

/* --- apb --- */
__STATIC_INLINE void SysCtl_setApbParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<3)) | ((bypass & 0x01) << 3);
}

__STATIC_INLINE uint32_t SysCtl_getApbParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 3) & 0x01;
}

/* --- ahb --- */
__STATIC_INLINE void SysCtl_setAhbParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<4)) | ((bypass & 0x01) << 4);
}

__STATIC_INLINE uint32_t SysCtl_getAhbParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 4) & 0x01;
}

/* --- epwm --- */
__STATIC_INLINE void SysCtl_setEpwmParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<5)) | ((bypass & 0x01) << 5);
}

__STATIC_INLINE uint32_t SysCtl_getEpwmParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 5) & 0x01;
}

/* --- ecap --- */
__STATIC_INLINE void SysCtl_setEcapParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<6)) | ((bypass & 0x01) << 6);
}

__STATIC_INLINE uint32_t SysCtl_getEcapParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 6) & 0x01;
}

/* --- epep --- */
__STATIC_INLINE void SysCtl_setEqepParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<7)) | ((bypass & 0x01) << 7);
}

__STATIC_INLINE uint32_t SysCtl_getEqepParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 7) & 0x01;
}

/* --- cmpss --- */
__STATIC_INLINE void SysCtl_setCmpssParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<8)) | ((bypass & 0x01) << 8);
}

__STATIC_INLINE uint32_t SysCtl_getCmpssParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 8) & 0x01;
}

/* --- clb --- */
__STATIC_INLINE void SysCtl_setClbParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<9)) | ((bypass & 0x01) << 9);
}

__STATIC_INLINE uint32_t SysCtl_getClbParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 9) & 0x01;
}

/* --- xbar --- */
__STATIC_INLINE void SysCtl_setXbarParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<10)) | ((bypass & 0x01) << 10);
}

__STATIC_INLINE uint32_t SysCtl_getXbarParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 10) & 0x01;
}

/* --- sdfm --- */
__STATIC_INLINE void SysCtl_setSdfmParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<11)) | ((bypass & 0x01) << 11);
}

__STATIC_INLINE uint32_t SysCtl_getSdfmParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 11) & 0x01;
}

/* --- dac --- */
__STATIC_INLINE void SysCtl_setDacParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<12)) | ((bypass & 0x01) << 12);
}

__STATIC_INLINE uint32_t SysCtl_getDacParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 12) & 0x01;
}

/* --- epg --- */
__STATIC_INLINE void SysCtl_setEpgParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<14)) | ((bypass & 0x01) << 14);
}

__STATIC_INLINE uint32_t SysCtl_getEpgParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 14) & 0x01;
}

/* --- crc --- */
__STATIC_INLINE void SysCtl_setCrcParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<16)) | ((bypass & 0x01) << 16);
}

__STATIC_INLINE uint32_t SysCtl_getCrcParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 16) & 0x01;
}

/* --- sema --- */
__STATIC_INLINE void SysCtl_setSemaParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<17)) | ((bypass & 0x01) << 17);
}

__STATIC_INLINE uint32_t SysCtl_getSemaParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 17) & 0x01;
}

/* --- dsp dma0 --- */
__STATIC_INLINE void SysCtl_setDspDma0ParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<19)) | ((bypass & 0x01) << 19);
}

__STATIC_INLINE uint32_t SysCtl_getDspDma0ParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 19) & 0x01;
}

/* --- dsp dma1 --- */
__STATIC_INLINE void SysCtl_setDspDma1ParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<20)) | ((bypass & 0x01) << 20);
}

__STATIC_INLINE uint32_t SysCtl_getDspDma1ParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 20) & 0x01;
}

/* --- mu0 --- */
__STATIC_INLINE void SysCtl_setMu0ParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<21)) | ((bypass & 0x01) << 21);
}

__STATIC_INLINE uint32_t SysCtl_getMu0ParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 21) & 0x01;
}

/* --- mu1 --- */
__STATIC_INLINE void SysCtl_setMu1ParaIcgBypass(uint32_t bypass)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) =
            (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) & ~(1<<22)) | ((bypass & 0x01) << 22);
}

__STATIC_INLINE uint32_t SysCtl_getMu1ParaIcgBypass(void)
{
    return (HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) >> 22) & 0x01;
}

/**
 * @brief   Enables the clock to a peripheral.
 *
 * @param [in]   peripheral is the peripheral to enable, see type SysCtl_PeripheralPCLOCKCR.
 *
 *  Peripherals are enabled with this function.  At power-up, all peripherals
 *  are disabled; they must be enabled in order to operate or respond to
 *  register reads/writes.
 *
 *  @remarks 
 *  - Note that there should be at least 5 cycles delay between enabling the
 *    peripheral clock and accessing the peripheral registers. The delay should be
 *    added by the user if the peripheral is accessed immediately after this
 *    function call.
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheral(SysCtl_PeripheralPCLOCKCR peripheral)
{
	if(peripheral != SYSCTL_PERIPH_CLK_TBCLKSYNC)
	{
    HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK)) |=
            ((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK);
    }
    else
    {
    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) |= 1<<18;
    }
}

/**
 * @brief   Disables a peripheral.
 *
 * @param [in]   peripheral is the peripheral to disable.
 *
 *  Peripherals are disabled with this function.  Once disabled, they will not
 *  operate or respond to register reads/writes. But the peripheral registers are not reset.
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheral(SysCtl_PeripheralPCLOCKCR peripheral)
{
	if(peripheral != SYSCTL_PERIPH_CLK_TBCLKSYNC)
	{
    HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK)) &=
            ~(((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));
	}
	else
	{
	    HWREG(AHB_COMM_PARA_BASE + AHB1_COMM_PARA_TBCLKSYNC) = 0;
	}
}


/**
 * @brief  peripheral icg bypass enable.
 *
 * @param [in]   peripheral is the peripheral to enable, see type SysCtl_PeripheralPCLOCKCR.
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_enablePeripheralIcgBypass(SysCtl_PeripheralICGBYPASS peripheral)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) |= peripheral;
}

/**
 * @brief   peripheral icg bypass disable
 *
 * @param [in]   peripheral is the peripheral to disable.
 *
 *
 * @return  None
 */
__STATIC_INLINE void SysCtl_disablePeripheralIcgBypass(SysCtl_PeripheralICGBYPASS peripheral)
{
    HWREG(SUB_CRG_BASE + SUBCRG_CFG_PERI_ICG_BYPASS) &= ~peripheral;
}

/**
 * 
 *  @brief Resets a peripheral
 * 
 *  @param [in] peripheral is the peripheral to reset.
 *  bit0~bit7 is the peripheral number, bit8~bit15 is the bit position in the register
 * 
 *  This function uses the Sysctl SRST_XXXX_N registers to reset a specified peripheral.
 *  Module registers will be returned to their reset states.
 *  Writing 0 put the peripheral into reset state.
 *  Writing 1 put the peripheral into release state.
 *  After reset and release, the peripheral registers returns to default values.
 * 
 *  @return None.
 * 
 */
__STATIC_INLINE void SysCtl_resetPeripheral(SysCtl_PeripheralSOFTPRES peripheral)
{
    HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK) + 0x100) &=
            ~(((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));

    HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK) + 0x100) |=
            (((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));
}

__STATIC_INLINE void SysCtl_setPeripheralReset(SysCtl_PeripheralSOFTPRES peripheral, uint32_t reset)
{
    if (reset) {
        HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK) + 0x100) &=
                ~(((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));
    } else {
        HWREG(SUB_CRG_BASE + ((peripheral>>CRG_CFG_ADDR_SHIFT) & CRG_CFG_ADDR_MASK) + 0x100) |=
                (((peripheral >> CRG_CFG_MASK_SHIFT) & CRG_CFG_MASK_MASK) << (peripheral & CRG_CFG_OFF_MASK));
    }
}

/* --- cpuTimer --- */
__STATIC_INLINE void SysCtl_resetCpuTimer(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_TIMER);
}

/* --- sci --- */
__STATIC_INLINE void SysCtl_resetSci(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SCI);
}

__STATIC_INLINE void SysCtl_resetScia(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SCIA);
}

__STATIC_INLINE void SysCtl_resetScib(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SCIB);
}

__STATIC_INLINE void SysCtl_resetScic(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SCIC);
}

__STATIC_INLINE void SysCtl_resetScid(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SCID);
}

__STATIC_INLINE void SysCtl_resetUarta(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_UARTA);
}

__STATIC_INLINE void SysCtl_resetUartb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_UARTB);
}

/* --- i2c --- */
__STATIC_INLINE void SysCtl_resetI2c(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_I2C);
}

__STATIC_INLINE void SysCtl_resetI2ca(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_I2CA);
}

__STATIC_INLINE void SysCtl_resetI2cb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_I2CB);
}

__STATIC_INLINE void SysCtl_resetCMI2c(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMI2C);
}

/* --- pmbus --- */
__STATIC_INLINE void SysCtl_resetPmbus(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_PMBUS);
}

/* --- spi --- */
__STATIC_INLINE void SysCtl_resetSpi(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SPI);
}

__STATIC_INLINE void SysCtl_resetSpia(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SPIA);
}

__STATIC_INLINE void SysCtl_resetSpib(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SPIB);
}

__STATIC_INLINE void SysCtl_resetSpic(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SPIC);
}

__STATIC_INLINE void SysCtl_resetSpid(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SPID);
}

__STATIC_INLINE void SysCtl_resetCMSpi(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMSPI);
}

/* --- can --- */
__STATIC_INLINE void SysCtl_resetCan(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CAN);
}

__STATIC_INLINE void SysCtl_resetCana(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CANA);
}

__STATIC_INLINE void SysCtl_resetCanb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CANB);
}

__STATIC_INLINE void SysCtl_resetCanc(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CANC);
}

/* --- lin --- */
__STATIC_INLINE void SysCtl_resetLin(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_LIN);
}

__STATIC_INLINE void SysCtl_resetLina(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_LINA);
}

__STATIC_INLINE void SysCtl_resetLinb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_LINB);
}

/* --- CRC ---*/
__STATIC_INLINE void SysCtl_resetCrc(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CRC);
}

/* --- dac ---*/
__STATIC_INLINE void SysCtl_resetDac(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_DAC);
}

__STATIC_INLINE void SysCtl_resetDaca(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_DACA);
}

__STATIC_INLINE void SysCtl_resetDacb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_DACB);
}

__STATIC_INLINE void SysCtl_resetDacc(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_DACC);
}

//be careful
__STATIC_INLINE void SysCtl_resetDaccom(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_DACCOM);
}

/* --- epg ---*/
__STATIC_INLINE void SysCtl_resetEpg(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPG);
}

/* --- xbar ---*/
__STATIC_INLINE void SysCtl_resetXbar(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_XBAR);
}

/* --- Clb ---*/
__STATIC_INLINE void SysCtl_resetClb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB);
}

/* index from 1 to 8 */
__STATIC_INLINE void SysCtl_resetClbx(uint32_t index)
{
    if (index > 0 && index <= 8) {
        SysCtl_resetPeripheral((SysCtl_PeripheralSOFTPRES)GS32F3xx_CRG_CFG(0x8, 0x01, index-1));
    }
}

__STATIC_INLINE void SysCtl_resetClb1(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB1);
}

__STATIC_INLINE void SysCtl_resetClb2(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB2);
}

__STATIC_INLINE void SysCtl_resetClb3(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB3);
}

__STATIC_INLINE void SysCtl_resetClb4(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB4);
}

__STATIC_INLINE void SysCtl_resetClb5(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB5);
}

__STATIC_INLINE void SysCtl_resetClb6(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB6);
}

__STATIC_INLINE void SysCtl_resetClb7(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB7);
}

__STATIC_INLINE void SysCtl_resetClb8(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLB8);
}

//to be careful
__STATIC_INLINE void SysCtl_resetClbcom(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CLBCOM);
}

/* --- Sdfm ---*/
__STATIC_INLINE void SysCtl_resetSdfm(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SDFM);
}

__STATIC_INLINE void SysCtl_resetSdfm1(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SDFM1);
}

__STATIC_INLINE void SysCtl_resetSdfm2(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SDFM1);
}

__STATIC_INLINE void SysCtl_resetSdfm3(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SDFM3);
}

__STATIC_INLINE void SysCtl_resetSdfm4(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SDFM4);
}

__STATIC_INLINE void SysCtl_resetSdfmcom(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_SDFMCOM);
}

/* --- eqep ---*/
__STATIC_INLINE void SysCtl_resetEqep(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP);
}

/* index from 1 to 6 */
__STATIC_INLINE void SysCtl_resetEqepx(uint32_t index)
{
    if (index > 0 && index <= 6) {
        SysCtl_resetPeripheral((SysCtl_PeripheralSOFTPRES)GS32F3xx_CRG_CFG(0x8, 0x01, index-1+20));
    }
}

__STATIC_INLINE void SysCtl_resetEqep1(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP1);
}

__STATIC_INLINE void SysCtl_resetEqep2(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP2);
}

__STATIC_INLINE void SysCtl_resetEqep3(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP3);
}

__STATIC_INLINE void SysCtl_resetEqep4(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP4);
}

__STATIC_INLINE void SysCtl_resetEqep5(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP5);
}

__STATIC_INLINE void SysCtl_resetEqep6(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EQEP6);
}

/* --- epwm ---*/
__STATIC_INLINE void SysCtl_resetEpwm(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM);
}

/* index from 1 to 18 */
__STATIC_INLINE void SysCtl_resetEpwmx(uint32_t index)
{
    if (index > 0 && index <= 18) {
        SysCtl_resetPeripheral((SysCtl_PeripheralSOFTPRES)GS32F3xx_CRG_CFG(0xC, 0x1, index-1));
    }
}

__STATIC_INLINE void SysCtl_resetEpwm1(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM1);
}

__STATIC_INLINE void SysCtl_resetEpwm2(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM2);
}

__STATIC_INLINE void SysCtl_resetEpwm3(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM3);
}

__STATIC_INLINE void SysCtl_resetEpwm4(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM4);
}

__STATIC_INLINE void SysCtl_resetEpwm5(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM5);
}

__STATIC_INLINE void SysCtl_resetEpwm6(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM6);
}

__STATIC_INLINE void SysCtl_resetEpwm7(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM7);
}

__STATIC_INLINE void SysCtl_resetEpwm8(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM8);
}

__STATIC_INLINE void SysCtl_resetEpwm9(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM9);
}

__STATIC_INLINE void SysCtl_resetEpwm10(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM10);
}

__STATIC_INLINE void SysCtl_resetEpwm11(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM11);
}

__STATIC_INLINE void SysCtl_resetEpwm12(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM12);
}

__STATIC_INLINE void SysCtl_resetEpwm13(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM13);
}

__STATIC_INLINE void SysCtl_resetEpwm14(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM14);
}

__STATIC_INLINE void SysCtl_resetEpwm15(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM15);
}

__STATIC_INLINE void SysCtl_resetEpwm16(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM16);
}

__STATIC_INLINE void SysCtl_resetEpwm17(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM17);
}

__STATIC_INLINE void SysCtl_resetEpwm18(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWM18);
}

// to be careful
__STATIC_INLINE void SysCtl_resetEpwmcom(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_EPWMCOM);
}

/* --- ecap ---*/
__STATIC_INLINE void SysCtl_resetEcap(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ECAP);
}

/* index from 1 to 7 */
__STATIC_INLINE void SysCtl_resetEcapx(uint32_t index)
{
    if (index > 0 && index <= 7) {
        SysCtl_resetPeripheral((SysCtl_PeripheralSOFTPRES)GS32F3xx_CRG_CFG(0x10, 0x01, index-1));
    }
}

/* --- cmpss ---*/
__STATIC_INLINE void SysCtl_resetCmpss(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS);
}

/* index from 1 to 12 */
__STATIC_INLINE void SysCtl_resetCmpssx(uint32_t index)
{
    if (index > 0 && index <= 12) {
        SysCtl_resetPeripheral((SysCtl_PeripheralSOFTPRES)GS32F3xx_CRG_CFG(0x14, 0x1, index-1));
    }
}

__STATIC_INLINE void SysCtl_resetCmpss1(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS1);
}

__STATIC_INLINE void SysCtl_resetCmpss2(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS2);
}

__STATIC_INLINE void SysCtl_resetCmpss3(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS3);
}

__STATIC_INLINE void SysCtl_resetCmpss4(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS4);
}

__STATIC_INLINE void SysCtl_resetCmpss5(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS5);
}

__STATIC_INLINE void SysCtl_resetCmpss6(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS6);
}

__STATIC_INLINE void SysCtl_resetCmpss7(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS7);
}

__STATIC_INLINE void SysCtl_resetCmpss8(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS8);
}

__STATIC_INLINE void SysCtl_resetCmpss9(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS9);
}

__STATIC_INLINE void SysCtl_resetCmpss10(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS10);
}

__STATIC_INLINE void SysCtl_resetCmpss11(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS11);
}

__STATIC_INLINE void SysCtl_resetCmpss12(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_CMPSS12);
}

/* --- gpio ---*/
__STATIC_INLINE void SysCtl_resetGpio(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO);
}

/* group from 1 to 14 */
__STATIC_INLINE void SysCtl_resetGpioGroupx(uint32_t group)
{
    if (group > 0 && group <= 14) {
        SysCtl_resetPeripheral((SysCtl_PeripheralSOFTPRES)GS32F3xx_CRG_CFG(0x18, 0x1, group-1));
    }
}

__STATIC_INLINE void SysCtl_resetGpio1(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO1);
}

__STATIC_INLINE void SysCtl_resetGpio2(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO2);
}

__STATIC_INLINE void SysCtl_resetGpio3(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO3);
}

__STATIC_INLINE void SysCtl_resetGpio4(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO4);
}

__STATIC_INLINE void SysCtl_resetGpio5(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO5);
}

__STATIC_INLINE void SysCtl_resetGpio6(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO6);
}

__STATIC_INLINE void SysCtl_resetGpio7(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO7);
}

__STATIC_INLINE void SysCtl_resetGpio8(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO8);
}

__STATIC_INLINE void SysCtl_resetGpio9(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO9);
}

__STATIC_INLINE void SysCtl_resetGpio10(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO10);
}

__STATIC_INLINE void SysCtl_resetGpio11(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO11);
}

__STATIC_INLINE void SysCtl_resetGpio12(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO12);
}

__STATIC_INLINE void SysCtl_resetGpio13(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO13);
}

__STATIC_INLINE void SysCtl_resetGpio14(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_GPIO14);
}

/* --- adc ---*/
__STATIC_INLINE void SysCtl_resetAdc(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ADC);
}

__STATIC_INLINE void SysCtl_resetAdca(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ADCA);
}

__STATIC_INLINE void SysCtl_resetAdcb(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ADCB);
}

__STATIC_INLINE void SysCtl_resetAdcc(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ADCC);
}

__STATIC_INLINE void SysCtl_resetAdcd(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ADCD);
}

//to be careful
__STATIC_INLINE void SysCtl_resetAdccom(void)
{
    SysCtl_resetPeripheral(SYSCTL_PERIPH_RST_ADCCOM);
}

/*
 * This function is used to select the function of an analog pin.
 * There are two types of pins:
 *     type1 - analog input pin, function is selected by ASEL register in asysctl;
 *             ADC_A0~ADC_C14, CMP1, CMP3, CMP5, CMP7,
 *     type2 - AGPIO pin, function is selected by AGPIO_SEL register in sysctl_h;
 *             AGPIO13/20/21
 * The input parameter can be found in "asysctl.h" - Analog pin configuration macro definition
 * The input parameter is a 32bit integer combined of the following 3 parts:
 * bit20 - select analog pin or AGPIO pin;
 * bit16~19 is ASEL offset in asysctl or AGPIO_SEL offset in sysctl_h;
 * bit0~3 is ASEL value or AGPIO_SEL value
 * These registers base address is 0x40350200 - A0_SEL register for analog pin;
 * These registers base address is 0x40370540 - AGPIO13_SEL register for AGPIO pin;
 *
 */
__STATIC_INLINE void SysCtl_setAnalogPin(uint32_t analogPinCfg)
{
//    if((analogPinCfg & 0x01000000) == 0)
//        HWREG(ANALOGSUBSYS_BASE + ANA_CFG_O_A0_SEL + ((analogPinCfg>>16)*4)) = (analogPinCfg&0xFFFF);
//    else
//        HWREG(SYSCTL_H_BASE + SYSCTL_H_O_AGPIO13_SEL + ((analogPinCfg>>16)&0xFF)*4) = ((analogPinCfg & 0xFF) | (0x08));
}

#ifdef __cplusplus
}
#endif

#endif

#endif /* DEVICE_DRIVERLIB_SYSCTL_CRG_H_ */
