/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    HPEC_Source_Template.c
*   @brief   
*   @details
*
*   commit history
*   2024/03/12, Zhao Lei, 3.0 baudrate fraction divider
*   2024/03/24, Jason, change function getLineStatus.
*
*/

#ifdef __cplusplus
extern "C"{
#endif

#include "gs32_version.h"
#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "uart.h"

#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)

void SCI_setConfig(uint32_t base, uint32_t lspclkHz, uint32_t baud, uint32_t config)
{

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));
    ASSERT(baud != 0U);
    ASSERT((baud * 16U) <= lspclkHz);

    SCI_setBaud(base, lspclkHz, baud);

    /* Set parity, data length, and number of stop bits. */
    HWREG(base + SCI_O_LCR) = ((HWREG(base + SCI_O_LCR) &
                                 ~(SCI_CONFIG_PAR_MASK |
                                   SCI_CONFIG_STOP_MASK |
                                   SCI_CONFIG_WLEN_MASK)) | config);
}

void UART_writeCharArray(uint32_t base, const uint8_t * const array,
                         uint32_t length)
{
    uint32_t i;
    char *pArray = (char *)array;

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

         /* FIFO is enabled.
            For loop to write characters */
        for(i = 0U; i < length; i++)
        {

           /* Wait until space is available in the transmit FIFO. */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)
            while(SCI_getTxFIFOStatus(base) == SCI_TX_FIFO_16)
#else
            while(!SCI_isSpaceAvailableNonFIFO(base))
#endif
            {
            }

            /* Send a char. */
            HWREG(base + SCI_O_RBR_THR_DLL) = pArray[i];
        }
    }
    else
    {

         /* FIFO is not enabled.
            For loop to write number of characters */
        for(i = 0U; i < length; i++)
        {
            /* Wait until space is available in the transmit buffer.*/
            while(!SCI_isSpaceAvailableNonFIFO(base))
            {
            }

            /* Send a char. */
            HWREG(base + SCI_O_RBR_THR_DLL) = pArray[i];
        }
    }
}

void SCI_writeCharArray(uint32_t base, const uint16_t * const array,
                   uint32_t length)
{
    uint32_t i;
    uint16_t *pArray = (uint16_t *)array;

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

     /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

        /* FIFO is enabled.
           For loop to write number of characters */
        for(i = 0U; i < length; i++)
        {
            while(SCI_getTxFIFOStatus(base) == SCI_TX_FIFO_16)

            {
            }

             /* Send a char. */
            HWREG(base + SCI_O_RBR_THR_DLL) = pArray[i];
        }
    }
    else
    {
        /* FIFO is not enabled.
        For loop to write number of characters */
        for(i = 0U; i < length; i++)
        {
        	/* Wait until space is available in the transmit buffer.*/
            while(!SCI_isSpaceAvailableNonFIFO(base))
            {
            }

            /* Send a char. */
            HWREG(base + SCI_O_RBR_THR_DLL) = pArray[i];
        }
    }
}

void UART_readCharArray(uint32_t base, uint8_t * const array, uint32_t length)
{
    uint16_t i;
    char *pArray = (char *)array;

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    if(SCI_isFIFOEnabled(base))
    {
        /* FIFO is enabled.
         For loop to read characters */
        for(i = 0U; i < length; i++)
        {
            while(SCI_getRxFIFOStatus(base) == SCI_RX_FIFO_0)
            {
            }

            /* Return the character from the receive buffer. */
            pArray[i] = (HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RXBUF_SAR_M);
        }
    }
    else
    {
         /* FIFO is not enabled.
         For loop to read characters */
        for(i = 0U; i < length; i++)
        {
             /* Wait until a character is available in the receive buffer. */
            while(!SCI_isDataAvailable(base))
            {
            }

             /* Return the character from the receive buffer. */
            pArray[i] = (HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RXBUF_SAR_M);
        }
    }
}

void SCI_readCharArray(uint32_t base, uint16_t * const array, uint32_t length)
{
    uint16_t i;
    uint16_t *pArray = (uint16_t *)array;

    /* Check the arguments. */
    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {
        /* FIFO is enabled.For loop to read characters */
        for(i = 0U; i < length; i++)
        {
            while(SCI_getRxFIFOStatus(base) == SCI_RX_FIFO_0)
            {
            }

            /*Return the character from the receive buffer.*/
            pArray[i] = (HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RXBUF_SAR_M);
        }
    }
    else
    {
        /* FIFO is enabled.For loop to read characters */
        for(i = 0U; i < length; i++)
        {
            /* Wait until a character is available in the receive buffer. */
            while(!SCI_isDataAvailable(base))
            {
            }

            /* Return the character from the receive buffer. */
            pArray[i] = (HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RXBUF_SAR_M);
        }
    }
}

void SCI_setBaud(uint32_t base, uint32_t lspclkHz, uint32_t baud)
{
    uint32_t divider;

#if IS_GS32F00xx(0x12, 0x30) || IS_GS32F3xx(0x22)

    /* Compute the baud rate divider {ROUND TO NEAREST INTEGER} */
    divider = lspclkHz * 4 / baud;

    uint32_t intDiv = divider >> 6;
    uint32_t fractionDiv = divider & 0x3F;

    while (SCI_getSCIStatusRegister(base) & SCI_UART_STATUS_BUSY)
    {
    	if (SCI_isDataAvailable(base))
    	{
    		SCI_readCharBlockingNonFIFO(base);
    	}
    }

    /* Set the baud rate. */
    HWREG(base + SCI_O_LCR) |= SCI_LCR_DLAB;

    HWREG(base + SCI_O_IER_DLH) = (intDiv & 0xFF00U) >> 8U;
    HWREG(base + SCI_O_RBR_THR_DLL) = intDiv & 0x00FFU;
    HWREG(base + SCI_O_DLF) = fractionDiv;

    HWREG(base + SCI_O_LCR) &= ~SCI_LCR_DLAB;
#else

    /* Compute the baud rate divider {ROUND TO NEAREST INTEGER} */
    divider = ((float)((float)lspclkHz / ((float)baud * 16.0F))) + 0.5F;

    /* Set the baud rate. */
    HWREG(base + SCI_O_LCR) |= SCI_LCR_DLAB;
    HWREG(base + SCI_O_IER_DLH) = (divider & 0xFF00U) >> 8U;
    HWREG(base + SCI_O_RBR_THR_DLL) = divider & 0x00FFU;
    HWREG(base + SCI_O_LCR) &= ~SCI_LCR_DLAB;
#endif
}

 void SCI_clearInterruptFlag(uint32_t base, SCI_INT_STATUS intFlags)
{

    /* Check the arguments. */
	uint32_t value = 0;
    ASSERT(SCI_isBaseValid(base));

    if (intFlags == SCI_MODEM_STATUS)
    {
    	value = HWREG(base + SCI_O_MSR);
    }
    else if (intFlags == SCI_THR_EMPTY)
    {
        value = HWREG(base + SCI_O_IIR_FCR);
    }
    else if (intFlags == SCI_RX_DATA_AVAILABLE || intFlags == SCI_CHARACTER_TIMEOUT)
    {

    }
    else if (intFlags == SCI_RX_LINE_STATUS)
    {
    	value = SCI_getLineStatus(base);
    }
    else if (intFlags == SCI_BUSY_DETECT)
    {
    	value = SCI_getSCIStatusRegister(base);
    }
}


void SCI_setFractionalBaudRate(uint32_t base, uint32_t lspclkHz, uint32_t baud)
{
    uint32_t divider = 0;
    uint32_t div_fraction = 16;
    uint32_t dlf_value = 0;


    /* Compute the baud rate divider {ROUND TO NEAREST INTEGER} */
    divider = (lspclkHz / baud) / 16;
    dlf_value = (((float)lspclkHz / (float)baud) / 16 - divider) * div_fraction;


    /* Set the baud rate. */
    HWREG(base + SCI_O_LCR) |= SCI_LCR_DLAB;
    HWREG(base + SCI_O_IER_DLH) = (divider & 0xFF00U) >> 8U;
    HWREG(base + SCI_O_RBR_THR_DLL) = divider & 0x00FFU;
    HWREG(base + SCI_O_DLF) = dlf_value;
    HWREG(base + SCI_O_LCR) &= ~SCI_LCR_DLAB;
}

void SCI_Mode9bit_SendAddr(uint32_t base, char addr)
{
    if((HWREG(base + SCI_O_LCR_EXT) & SCI_LCR_EXT_SEND_ADDR) == SCI_LCR_EXT_ADDR_MATCH)
    {
        HWREG(base + SCI_O_TAR) = addr << 1 + 1;
    }
    else
    {
        HWREG(base + SCI_O_TAR) = addr;
        SCI_setSendAddrFlag(base);
    }

}


int SCI_Mode9bit_RecvAddr(uint32_t base, char *addr)
{
    if((HWREG(base + SCI_O_LSR) & SCI_LSR_ADDR_RCVD) != SCI_LSR_ADDR_RCVD)
        return -1;
    if((HWREG(base + SCI_O_LCR_EXT) & SCI_LCR_EXT_ADDR_MATCH) == SCI_LCR_EXT_ADDR_MATCH)
        *addr = HWREG(base + SCI_O_RAR);
    else
        *addr = (HWREG(base + SCI_O_RBR_THR_DLL)) >> 1;
    return 0;
}

void SCI_Mode9bit_writeCharArray(uint32_t base, const uint8_t * const array,
                   uint16_t length)
{
    uint16_t i;
    char *pArray = (char *)array;

    ASSERT(SCI_isBaseValid(base));

    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

        /* FIFO is enabled. For loop to write characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until space is available in the transmit FIFO. */
            while(SCI_getTxFIFOStatus(base) == 0x00)
            {
            }


            /* Send a char. */
            if((HWREG(base + SCI_O_RBR_THR_DLL) & SCI_LCR_EXT_SEND_ADDR) == SCI_LCR_EXT_SEND_ADDR)
                HWREG(base + SCI_O_TAR) = pArray[i] << 1 + 0;
            else
                HWREG(base + SCI_O_RBR_THR_DLL) = pArray[i];
        }
    }
    else
    {

        /* FIFO is enabled. For loop to write characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until space is available in the transmit buffer. */
            while(!SCI_isSpaceAvailableNonFIFO(base))
            {
            }


            /* Send a char. */
            if((HWREG(base + SCI_O_RBR_THR_DLL) & SCI_LCR_EXT_SEND_ADDR) == SCI_LCR_EXT_SEND_ADDR)
                HWREG(base + SCI_O_TAR) = pArray[i] << 1 + 0;
            else
                HWREG(base + SCI_O_RBR_THR_DLL) = pArray[i];
        }
    }
}

void SCI_Mode9bit_readCharArray(uint32_t base, uint8_t * const array, uint16_t length)
{
    uint16_t i;
    char *pArray = (char *)array;

    ASSERT(SCI_isBaseValid(base));



    /* Check if FIFO enhancement is enabled. */
    if(SCI_isFIFOEnabled(base))
    {

        /* FIFO is enabled. For loop to write characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until a character is available in the receive FIFO. */
            while(SCI_getRxFIFOStatus(base) == SCI_RX_FIFO_0)
            {
            }


            /* Return the character from the receive buffer. */
            if((HWREG(base + SCI_O_RBR_THR_DLL) & SCI_LCR_EXT_ADDR_MATCH) == SCI_LCR_EXT_ADDR_MATCH)
                pArray[i] = (HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RXBUF_SAR_M);
            else
                pArray[i] = HWREG(base + SCI_O_RBR_THR_DLL) >> 1;
            
        }
    }
    else
    {

        /* FIFO is enabled. For loop to write characters */
        for(i = 0U; i < length; i++)
        {

            /* Wait until a character is available in the receive buffer. */
            while(!SCI_isDataAvailable(base))
            {
            }


            /* Return the character from the receive buffer. */
            if((HWREG(base + SCI_O_RBR_THR_DLL) & SCI_LCR_EXT_ADDR_MATCH) == SCI_LCR_EXT_ADDR_MATCH)
                pArray[i] = (HWREG(base + SCI_O_RBR_THR_DLL) & SCI_RXBUF_SAR_M);
            else
                pArray[i] = HWREG(base + SCI_O_RBR_THR_DLL) >> 1;
        }
    } 
}


#endif /* #if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22) */

#ifdef __cplusplus
}
#endif

