/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    xbar.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

#include "gs32_version.h"
#include "xbar.h"

#if IS_GS32F00xx(0x30)

// Array mapping pins to offsets
struct xbarInput_offset xbar_InputGPIO_map[] = {
    {46, 0},   		// Pin 0   ~ 46  maps to offset 0
    {49, 1},   		// Pin 48  ~ 49  maps to offset 1
    {55, 4},   		// Pin 53  ~ 55  maps to offset 4
    {63, 6},   		// Pin 58  ~ 63  maps to offset 6
    {200, 142},		// Pin 200       maps to offset 142
    {228, 165},		// Pin 224 ~ 228 maps to offset 165
    {239, 166},		// Pin 230 ~ 239 maps to offset 166
    {242, 167},		// Pin 241 ~ 242 maps to offset 167
    {245, 168},		// Pin 244 ~ 245 maps to offset 168
    {248, 170} 		// Pin 248       maps to offset 170
};
#elif IS_GS32F00xx(0x12)

struct xbarInput_offset xbar_InputGPIO_map[] = {

#if GS32_PART_NUM==0x0049
    {38, 0},   		// Pin 0   ~ 38	 maps to offset 0
	{40, -27},   	// Pin 39  ~ 40  maps to offset -27
	{43, 0},   		// Pin 41  ~ 43  maps to offset 0
	{44, -24},   	// Pin 44  		 maps to offset -24
	{65, 0},   		// Pin 45  ~ 65  maps to offset 0
#else
	{65, 0},   		// Pin 0   ~ 65  maps to offset 0
#endif
    {254, 155},   	// Pin 224 ~ 254 maps to offset 155
};

#elif IS_GS32F3xx(0x22)

struct xbarInput_offset xbar_InputGPIO_map[] = {
    {168, 0},   	// Pin 0   ~ 168   maps to offset 0
    {242, 29},   	// Pin 168  ~ 242  maps to offset 29
};

#endif

/**
 * @brief config outxbar and plu_outxbar select signal by muxconfig
 *
 * @param base xbar and plu_outxbar baseaddress
 * @param output outxbar and plu_outxbar out
 * @param muxConfig input signal
 */
void XBAR_setOutputMuxConfig(uint32_t base, XBAR_OutputNum output, XBAR_OutputMuxConfig muxConfig)
{


    //
    // Check the arguments.
    //
    ASSERT(XBAR_isBaseValid(base));
    uint32_t shift;
    uint16_t offset;
    uint16_t mux=((uint32_t)muxConfig>>12);

    if ((mux>=0x2)&(mux<0x4))
    {
    	if(output >= 8)
    	{
    		offset = ((uint16_t)(output - 8) * 8) + 4U;
    	}
    	else
    	{
    		offset = ((uint16_t)output * 8) + 4U;
    	}
    }
    // mux32----47
    else if ((mux>=0x4)&(mux<0x6))
    {
#if IS_GS32F3xx(0x22) || IS_GS32F00xx(0x12, 0x30)
    	if(output >= 8)
    	{
    		offset =XBAR_O_OUTPUT1MUX32TO47CFG+((uint16_t)(output - 8) * 8);
    	}
    	else
    	{
    		offset =XBAR_O_OUTPUT1MUX32TO47CFG+((uint16_t)output * 8);
    	}
#endif
    } // mux48---63
    else if ((mux>=0x6)&(mux<0x8))
    {
#if IS_GS32F3xx(0x22) || IS_GS32F00xx(0x12, 0x30)
    	if(output >= 8)
    	{
    		offset = XBAR_O_OUTPUT1MUX32TO47CFG+((uint16_t)(output - 8) * 8)+4;
    	}
    	else
    	{
    		offset = XBAR_O_OUTPUT1MUX32TO47CFG+((uint16_t)output * 8)+4;
    	}
    // set output clb2 to outputxbar
    	if((muxConfig && 0xff) == 0x1 )
    	{
    		HWREG(0x4006df00 + XBAR_TOP_PLU2_OUT_SEL1) =0X1111;
    		HWREG(0x4006df00 + XBAR_TOP_PLU2_OUT_SEL2) =0X11;
    	}
#endif
    }
    else
    {
    	if(output >= 8)
    	{
            offset = (uint16_t)(output - 8) * 8;
    	}
    	else
    	{
    		offset = (uint16_t)output * 8;
    	}
    }


    shift = ((uint32_t)muxConfig >> 8U) & 0x1FU;

    HWREG(base + XBAR_O_OUTPUT1MUX0TO15CFG + offset) =
        (HWREG(base + XBAR_O_OUTPUT1MUX0TO15CFG + offset) & ~((uint32_t)0x3U << shift)) | // contain other bits unchange
        (((uint32_t)muxConfig & 0x3U) << shift);

}

/**
 * @brief config Epwm input signal
 *
 * @param trip Epwm trip input
 * @param muxConfig Epwmxbar input signal
 */
void XBAR_setEPWMMuxConfig(XBAR_TripNum trip, XBAR_EPWMMuxConfig muxConfig)
{

    uint32_t shift;
    uint16_t offset;
    uint32_t reg_addr;

    uint16_t mux=((uint32_t)muxConfig>>12);

    //Trip16~Trip31
    if (XBAR_TRIP16<=trip)
    {
        reg_addr=XBAR_EPWM_CFG_REG_BASE+EPWMXBAR_BASE_OFFSET*(uint16_t)(trip/XBAR_TRIP16);//epwmxbar2 muxconfig address
        trip=(XBAR_TripNum)(trip % XBAR_TRIP16);
    }else
    {
        reg_addr=XBAR_EPWM_CFG_REG_BASE; //epwmxbar1 muxconfig address
    }
    // mux16---31
    if ((mux>=0x2)&(mux<0x4))
    {
        offset = ((uint16_t)trip * 8) + 4U;
    }
      // mux32----47
    else if ((mux>=0x4)&(mux<0x6))
    {
        offset =XBAR_O_TRIP4MUX32TO47CFG+((uint16_t)trip * 8);
    }
    // mux48---63
    else if ((mux>=0x6)&(mux<0x8))
    {
        offset = XBAR_O_TRIP4MUX32TO47CFG+((uint16_t)trip * 8)+4;
    }
    // mux0---15
    else
    {
        offset = (uint16_t)trip * 8;
    }

    shift = ((uint32_t)muxConfig >> 8U) & 0x1FU;

    HWREG(reg_addr + offset) =
        (HWREG(reg_addr + offset) & ~((uint32_t)0x3U << shift)) |
        (((uint32_t)muxConfig & 0x3U) << shift);
}


/**
 * @brief Pluxbar config
 *
 * @param auxSignal Pluxbar input mux select
 * @param muxConfig Pluxbar input signal select
 */
void XBAR_setCLBMuxConfig(XBAR_AuxSigNum auxSignal, XBAR_CLBMuxConfig muxConfig)
{

    uint32_t shift;
    uint16_t offset;

    uint16_t mux=((uint32_t)muxConfig>>12);

    if ((mux>=0x2)&(mux<0x4))
    {
        offset = ((uint16_t)auxSignal * 8) + 4U;
    }
    // mux32----47
    else if ((mux>=0x4)&(mux<0x6))
    {
#if IS_GS32F3xx(0x22) || IS_GS32F00xx(0x12, 0x30)
        offset =XBAR_O_AUXSIG0MUX32TO47CFG+((uint16_t)auxSignal * 8);
#endif
    } // mux48---63
    else if ((mux>=0x6)&(mux<0x8))
    {
#if IS_GS32F3xx(0x22) || IS_GS32F00xx(0x12, 0x30)
        offset =XBAR_O_AUXSIG0MUX32TO47CFG+((uint16_t)auxSignal * 8)+4;
#endif
    }
    else
    {
        offset = (uint16_t)auxSignal * 8;
    }

    // Newly added signals in GS32 DSP
    if (auxSignal > 28)
    {
        if (auxSignal < 88)
        {
            offset += 96U;
        }
        else
        {
            offset += 128U;
        } 
    }
    shift = ((uint32_t)muxConfig >> 8U) & 0x1FU;

    HWREG(XBAR_CLB_CFG_REG_BASE + offset) =
        (HWREG(XBAR_CLB_CFG_REG_BASE + offset) & ~((uint32_t)0x3U << shift)) |
        (((uint32_t)muxConfig & 0x3U) << shift);
}

/**
 * @brief Get inputxbar flag
 *
 * @param inputFlag input signal flag
 * @return true
 * @return false
 */
bool XBAR_getInputFlagStatus(XBAR_InputFlag inputFlag)
{

    uint32_t offset;
    uint32_t inputMask;

    //
    // Determine flag register offset.
    //
    switch ((uint16_t)inputFlag & XBAR_INPUT_FLG_REG_M)
    {
    case XBAR_INPUT_FLG_REG_1:
        offset = XBAR_O_FLG1;
        break;

    case XBAR_INPUT_FLG_REG_2:
        offset = XBAR_O_FLG2;
        break;

    case XBAR_INPUT_FLG_REG_3:
        offset = XBAR_O_FLG3;
        break;
    case XBAR_INPUT_FLG_REG_4:
        offset = XBAR_O_FLG4;
        break;
#if IS_GS32F00xx(0x12, 0x30)
    case XBAR_INPUT_FLG_REG_5:
        offset = XBAR_O_FLG5;
        break;
    case XBAR_INPUT_FLG_REG_6:
        offset = XBAR_O_FLG6;
        break;
#endif
#if IS_GS32F00xx(0x30)
    case XBAR_INPUT_FLG_REG_7:
        offset = XBAR_O_FLG7;
        break;
    case XBAR_INPUT_FLG_REG_8:
        offset = XBAR_O_FLG8;
        break;
    case XBAR_INPUT_FLG_REG_9:
        offset = XBAR_O_FLG9;
        break; 
    case XBAR_INPUT_FLG_REG_10:
        offset = XBAR_O_FLG10;
        break; 
#endif
#if IS_GS32F3xx(0x22) || IS_GS32F00xx(0x30)
    case XBAR_INPUT_FLG_REG_11:
        offset = XBAR_O_FLG11;
        break;
    case XBAR_INPUT_FLG_REG_12:
        offset = XBAR_O_FLG12;
        break;
    case XBAR_INPUT_FLG_REG_13:
        offset = XBAR_O_FLG13;
        break;
    case XBAR_INPUT_FLG_REG_14:
        offset = XBAR_O_FLG14;
        break;
    case XBAR_INPUT_FLG_REG_15:
        offset = XBAR_O_FLG15;
        break;
    case XBAR_INPUT_FLG_REG_16:
        offset = XBAR_O_FLG16;
        break;
#endif
    default:
        //
        // This should never happen if a valid inputFlag value is used.
        //
        offset = 0U;
        break;
    }

    //
    // Get the status of the X-BAR input latch.
    //
    inputMask = (uint32_t)1U << ((uint32_t)inputFlag & XBAR_INPUT_FLG_INPUT_M);

    return ((HWREG(XBAR_BASE + offset) & inputMask) != 0U);

}

/**
 * @brief Clear inputxbar flag
 *
 * @param inputFlag input signal flag
 */
void XBAR_clearInputFlag(XBAR_InputFlag inputFlag)
{

    uint32_t offset;
    uint32_t inputMask;

    //
    // Determine flag clear register offset.
    //
    switch ((uint16_t)inputFlag & XBAR_INPUT_FLG_REG_M)
    {
    case XBAR_INPUT_FLG_REG_1:
        offset = XBAR_O_CLR1;
        break;

    case XBAR_INPUT_FLG_REG_2:
        offset = XBAR_O_CLR2;
        break;

    case XBAR_INPUT_FLG_REG_3:
        offset = XBAR_O_CLR3;
        break;
    case XBAR_INPUT_FLG_REG_4:
        offset = XBAR_O_CLR4;
        break;
#if IS_GS32F00xx(0x12, 0x30)
    case XBAR_INPUT_FLG_REG_5:
        offset = XBAR_O_CLR5;
        break;
    case XBAR_INPUT_FLG_REG_6:
        offset = XBAR_O_CLR6;
        break;
#endif
#if IS_GS32F00xx(0x30)
    case XBAR_INPUT_FLG_REG_7:
        offset = XBAR_O_CLR7;
        break;
    case XBAR_INPUT_FLG_REG_8:
        offset = XBAR_O_CLR8;
        break;
    case XBAR_INPUT_FLG_REG_9:
        offset = XBAR_O_CLR9;
        break; 
    case XBAR_INPUT_FLG_REG_10:
        offset = XBAR_O_CLR10;
        break; 
#endif
#if IS_GS32F3xx(0x22) || IS_GS32F00xx(0x30)
    case XBAR_INPUT_FLG_REG_11:
        offset = XBAR_O_CLR11;
        break;
    case XBAR_INPUT_FLG_REG_12:
        offset = XBAR_O_CLR12;
        break;
    case XBAR_INPUT_FLG_REG_13:
        offset = XBAR_O_CLR13;
        break;
    case XBAR_INPUT_FLG_REG_14:
        offset = XBAR_O_CLR14;
        break;
    case XBAR_INPUT_FLG_REG_15:
        offset = XBAR_O_CLR15;
        break;
    case XBAR_INPUT_FLG_REG_16:
        offset = XBAR_O_CLR16;
        break;
#endif
    default:
        //
        // This should never happen if a valid inputFlag value is used.
        //
        offset = 0U;
        break;
    }

    //
    // Set the bit that clears the X-BAR input latch.
    //
    inputMask = (uint32_t)1U << ((uint32_t)inputFlag & XBAR_INPUT_FLG_INPUT_M);
    HWREG(XBAR_BASE + offset) = inputMask;

}

#ifdef __cplusplus
}
#endif
