/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */


/**
*   @file    xdma.h
*   @brief   
*
*/
/*
 * commit history
 * 20240308, LYF, verify for chip 2.0
 * 20240319, LYF, Organize XDMA and DMAMUX driver version management
 * 20240322, LYF, translate source files to C source, ASCII text.
 * 20240513, LYF, code error in XDMA_clearInterrupt
 * 20240612, LYF, fix code error in XDMA_forceTrigger
 * 20240612, LYF, move the HkSelect and HardInf configuration inside the driver by force 
 * 20240617, LYF, fix XDMA_checkRamInAxi2Space issue
 * 20240705, LYF, Delate legacy code and use user defined isr function name.  
 * 20250108, LYF, add XDMA_setRWLen for special application
 */

#ifndef DEVICE_DRIVERLIB_XDMA_H_
#define DEVICE_DRIVERLIB_XDMA_H_

#ifdef __cplusplus
extern "C"{
#endif

//****************************************************************************
//!
//!                             Include Files                                  
//!
//****************************************************************************

#include "inc/hw_types.h"
#include "inc/hw_memmap.h"
#include "inc/hw_xdma.h"
#include "regs/regs_xdma.h"
#include "dmamux.h"
#include "debug.h"

//*****************************************************************************
//!
//!                           Macros & Typedefs                                
//!
//*****************************************************************************

#if (XDMAC_VERSION == 0x22)

//*******************************************************************************
//!
//! 2 X DSP heterogeneous processors.
//! Each DSP domain has an independent module named DSP_DMA and DSP_DMA1 in the
//! header file. the two DSP domains can
//! access each other's DSP_DMA modules. So, no remapping is required.
//! DSP domain 's 1st DMA Module:
//!     DSP_DMA_BASE     
//!     DSP_DMA_CH0_BASE   
//!     DSP_DMA_CH1_BASE   
//!     DSP_DMA_CH2_BASE    
//!     DSP_DMA_CH3_BASE    
//!     DSP_DMA_CH4_BASE    
//!     DSP_DMA_CH5_BASE    
//!     DSP_DMA_CH6_BASE     
//!     DSP_DMA_CH7_BASE     
//!
//! DSP domain 's 2nd DMA Module:
//!     DSP_DMA1_BASE     
//!     DSP_DMA1_CH0_BASE   
//!     DSP_DMA1_CH1_BASE   
//!     DSP_DMA1_CH2_BASE    
//!     DSP_DMA1_CH3_BASE    
//!     DSP_DMA1_CH4_BASE    
//!     DSP_DMA1_CH5_BASE    
//!     DSP_DMA1_CH6_BASE     
//!     DSP_DMA1_CH7_BASE  
//!
//*******************************************************************************
//*******************************************************************************
//!
//! The following defines the addresses. Users can use either the original names 
//! or the new names for specific purposes.
//!
//*******************************************************************************
#endif /* #if (XDMAC_VERSION == 0x22) */


#if (XDMAC_VERSION == 0x20 || XDMAC_VERSION == 0x22)

#define XDMA_Channel(base)           ((XDMA_Channel_TypeDef *)(base))
#define XDMAC(base)                  ((XDMA_CtlSta_TypeDef *)(base))   

#define XDMA_BASE_M					0xFFF

/**
 * @brief XDMA reinitializes when the transfer count is zero and waits for a trigger.
 *
 */
#define XDMA_CFG_MULTBLK_TYPE_S   0U
#define XDMA_CFG_MULTBLK_TYPE_M   0x3U

/**
 * @brief XDMA transfers Size.
 *
 */
#define XDMA_CFG_DATA_SIZE_TYPE_S     0x2U
#define XDMA_CFG_DATA_SIZE_TYPE_M     0x3CU

/**
 * @brief XDMA Transfer Wideth.
 *
 */
#define XDMA_CFG_DATA_WIDTH_TYPE_S   0x6U
#define XDMA_CFG_DATA_WIDTH_TYPE_M   0xC00U

/**
 * @brief XDMA interrupt identification.
 *
 */
#define XDMA_INT_TFR         0x01U
#define XDMA_INT_BLOCK       0x02U
#define XDMA_INT_SRCTRAN     0x04U
#define XDMA_INT_DSTTRAN     0x08U
#define XDMA_INT_ERR         0x10U

/**
 * @brief XDMA Channel interrupt signal enable mask.
 *
 */
#define XDMA_CHANNEL_INT_SIGL_H_ENABLE      (0x0FU)
#define XDMA_CHANNEL_INT_SIGL_H_DISABLE     (0x00U)
#define XDMA_CHANNEL_INT_SIGL_L_ENABLE      (0xFFFFFFFFU)
#define XDMA_CHANNEL_INT_SIGL_L_DISABLE     (0x00U)

#define XDMAX_NUM_CHANNELS           (8U)

#define XDMAREGH(base, offset)      (HWREG(base + (offset) + 0x04))
#define XDMAREGL(base, offset)      (HWREG(base + (offset)))

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/**
 * @brief DW_axi_dmac status.
 *
 * @param DW_axi_dmac status disabled.
 * @param DW_axi_dmac status enabled.
 */
typedef enum
{
    XDMA_DISABLED = 0U,
    XDMA_ENABLED = 1U
}XDMA_StatusEnabled;

/**
 * @brief DW_axi_dmac status active.
 *
 * @param DW_axi_dmac Status Inactived.
 * @param DW_axi_dmac Status Actived.
 */
typedef enum
{
    XDMA_INACTIVED = 0U,
    XDMA_ACTIVED = 1U
}XDMA_StatusActived;

/**
 * @brief DW_axi_dmac status Valid.
 *
 * @param DW_axi_dmac Status Invalid.
 * @param DW_axi_dmac Status Valid.
 */
typedef enum
{
    XDMA_INVALID = 0U,
    XDMA_VALID = 1U
}XDMA_StatusValid;

/**
 * @brief DW_axi_dmac status Transfer.
 *
 * @param DW_axi_dmac DMA/BLOCK Transfer Not completed.
 * @param DW_axi_dmac DMA/BLOCK Transfer completed.
 */
typedef enum
{
    XDMA_TFR_NOT_COMPLETED = 0U,
    XDMA_TFR_COMPLETED = 1U
}XDMA_StatusTransfer;

/**
 * @brief DW_axi_dmac Action mode.
 *
 * @param DW_axi_dmac set to be disable action.
 * @param DW_axi_dmac set to be enable action.
 */
typedef enum
{
    XDMA_DISABLE = 0U,
    XDMA_ENABLE = 1U
}XDMA_ActionEnable;

/**
 * @brief DW_axi_dmac Action active.
 *
 * @param DW_axi_dmac set to be Inactived.
 * @param DW_axi_dmac set to be Actived.
 */
typedef enum
{
    XDMA_INACTIVE = 0U,
    XDMA_ACTIVE = 1U
}XDMA_ActionActive;

/**
 * @brief DW_axi_dmac Action clear.
 *
 * @param DW_axi_dmac set to no clearn.
 * @param DW_axi_dmac set to Clear Action.
 */
typedef enum
{
    XDMA_NO_CLEAR = 0U,
    XDMA_CLEAR = 1U
}XDMA_ActionClear;

/**
 * @brief DW_axi_dmac Action Reset.
 *
 * @param DW_axi_dmac set to be no Action.
 * @param DW_axi_dmac set to Reset Action.
 */
typedef enum
{
    XDMA_NO_RST = 0U,
    XDMA_RST = 1U
}XDMA_ActionReset;

/**
 * @brief DW_axi_dmac Parity Type.
 *
 * @param XDMA_EVEN_PARITY_MODE The Even Parity check is used for registers in the Common/Channel Specific space
 * @param XDMA_ODD_PARITY_MODE  The Odd Parity check is used for registers in the Common/Channel Specific space
 */
typedef enum
{
    XDMA_EVEN_PARITY_MODE, 
    XDMA_ODD_PARITY_MODE
}XDMA_ParityType;

/**
 * @brief DW_axi_dmac EccIntfc Type.
 *
 * @param XDMA_ECC_CH_MEMIF Channel FIFO Memory Interface.
 * @param XDMA_ECC_UID_MEMIF Channel UID Memory Interface.
 * @param XDMA_AXI_MXIF AXI Manager Interface.
 * @param XDMA_RSV_ECC_INTFC Reserved
 */
typedef enum
{
    XDMA_ECC_CH_MEMIF,
    XDMA_ECC_UID_MEMIF,
    XDMA_AXI_MXIF,
    XDMA_RSV_ECC_INTFC
}XDMA_EccIntfcType;

/**
 * @brief DW_axi_dmac EccEiPoints.
 *
 * @param XDMA_EI_ADDRESS  Address Error Injection.
 * @param XDMA_EI_DATA  Data Error Injection.
 * @param XDMA_EI_ECC_CHECKBITS  ECC Checkbits Error Injection.
 */
typedef enum
{
    XDMA_EI_ADDRESS = 0U,
    XDMA_EI_DATA = 1U,
    XDMA_EI_ECC_CHECKBITS = 2U
}XDMA_EccEiPoints;

/**
 * @brief DW_axi_dmac EccEiType.
 *
 * @param XDMA_EI_SINGLE_BIT  Insert double bit error.
 * @param XDMA_EI_DOUBLE_BIT  Insert single bit error.
 */
typedef enum
{
    XDMA_EI_SINGLE_BIT = 0U,
    XDMA_EI_DOUBLE_BIT = 1U 
}XDMA_EccEiType;

/**
 * @brief DW_axi_dmac EccEiMemIntfc.
 *
 * @param XDMA_EI_FIFO_MEM  Select FIFO memory interface for error Injection.
 * @param XDMA_EI_UIM_MEM  Select UID memory interface for error Injection.
 */
typedef enum
{
    XDMA_EI_FIFO_MEM = 0U,
    XDMA_EI_UIM_MEM = 1U
}XDMA_EccEiMemIntfc;

/**
 * @brief DW_axi_dmac ShadowLliLast.
 *
 * @param XDMA_NOT_LAST_ITEM  Indicates shadowreg/LLI content is the last one.
 * @param XDMA_LAST_ITEM  Indicates shadowreg/LLI content is not the last one.
 */
typedef enum
{
    XDMA_NOT_LAST_ITEM = 0,
    XDMA_LAST_ITEM = 1
}XDMA_ShadowLliLast;

/**
 * @brief DW_axi_dmac AddressIncrement.
 *
 * @param XDMA_ADDR_INCRE  Source/Destination address incremented on every source transfer
 * @param XDMA_ADDR_NO_CHANGE  Source/Destination address is fixed.
 */
typedef enum
{
    XDMA_ADDR_INCRE = 0,
    XDMA_ADDR_NO_CHANGE = 1
}XDMA_AddressIncrement;

/**
 * @brief DW_axi_dmac MasterInfSelect.
 *
 * @param XDMA_MASTER1_INTF  Source/Destination device on Manager-2 interface layer (AXI Manager 1)
 * @param XDMA_MASTER2_INTF  Source/Destination device on Manager-1 interface layer (AXI Manager 2)
 */
typedef enum
{
    XDMA_MASTER1_INTF = 0,
    XDMA_MASTER2_INTF = 1
}XDMA_MasterInfSelect;

/**
 * @brief DW_axi_dmac ChnLockLevel.
 *
 * @param XDMA_DMA_TFR_CH_LOCK  Duration of the Channel locking is for the entire DMA transfer
 * @param XDMA_BLOCK_TFR_CH_LOCK  Duration of the Channel locking is for the current block transfer
 * @param XDMA_RSVD_TFT_CH_LOCK  Reserved
 */
typedef enum
{
    XDMA_DMA_TFR_CH_LOCK = 0,
    XDMA_BLOCK_TFR_CH_LOCK = 1,
    XDMA_RSVD_TFT_CH_LOCK = 3,
}XDMA_ChnLockLevel;

/**
 * @brief DW_axi_dmac ChnLockType.
 *
 * @param XDMA_NO_CHANNEL_LOCK  Channel is not locked during the transfers
 * @param XDMA_CHANNEL_LOCK  Channel is locked and granted exclusive access to the Manager Bus Interface
 */
typedef enum
{
    XDMA_NO_CHANNEL_LOCK = 0,
    XDMA_CHANNEL_LOCK = 1,
}XDMA_ChnLockType;

/**
 * @brief DW_axi_dmac HardwarePol.
 *
 * @param XDMA_ACTIVE_HIGH  Polarity of the Handshaking Interface used for the Destination peripheral is Active High
 * @param XDMA_ACTIVE_LOW  Polarity of the Handshaking Interface used for the Destination peripheral is Active Low
 */
typedef enum
{
    XDMA_ACTIVE_HIGH = 0,
    XDMA_ACTIVE_LOW = 1,
}XDMA_HK_HardwarePol;

/**
 * @brief DW_axi_dmac MultblkType.
 *
 * @param XDMA_CONTIGOUS  Contiguous monoblock Type used for Destination Transfer
 * @param XDMA_RELOAD  Contiguous Multiblock Type used for Destination Transfer
 * @param XDMA_SHADOW_REGISTER  Shadow Register based Multiblock Type used for Destination Transfer
 * @param XDMA_LINKED_LIST  Linked List based Multiblock Type used for Destination Transfer
 */
typedef enum
{
    XDMA_CONTIGOUS = 0,
    XDMA_RELOAD = 1,
    XDMA_SHADOW_REGISTER = 2,
    XDMA_LINKED_LIST = 3,
}XDMA_MultblkType;

/**
 * @brief DW_axi_dmac DataLeftInFIFO.
 *
 * @param XDMA_AXI Transmission response
 * @param XDMA_CHN TERMINATE
 */
typedef enum
{
    XDMA_AXI_TFRERR_RESPONSE_RECEPTION = 0,
    XDMA_CHN_TERMINATE = 1,
}XDMA_DataLeftInFIFO;

/**
 * @brief DW_axi_dmac TransTypeFlowCtrl.
 *
 * @param XDMA_TT_FC_0_M2M_DMAC  Transfer Type is memory to memory and Flow Controller is DW_axi_dmac
 * @param XDMA_TT_FC_1_M2P_DMAC  Transfer Type is memory to peripheral and Flow Controller is DW_axi_dmac
 * @param XDMA_TT_FC_2_P2M_DMAC  Transfer Type is peripheral to memory and Flow Controller is DW_axi_dmac
 * @param XDMA_TT_FC_3_P2P_DMAC  Transfer Type is peripheral to peripheral and Flow Controller is DW_axi_dmac
 * @param XDMA_TT_FC_4_P2M_P  Transfer Type is peripheral to Memory and Flow Controller is Source peripheral
 * @param XDMA_TT_FC_5_P2P_SP  Transfer Type is peripheral to peripheral and Flow Controller is Source periphera
 * @param XDMA_TT_FC_6_M2P_P  Transfer Type is memory to peripheral and Flow Controller is Destination peripheral
 * @param XDMA_TT_FC_7_P2P_DP  Transfer Type is peripheral to peripheral and Flow Controller is Destination peripheral
 */
typedef enum
{
    XDMA_TT_FC_0_M2M_DMAC = 0,
    XDMA_TT_FC_1_M2P_DMAC,
    XDMA_TT_FC_2_P2M_DMAC,
    XDMA_TT_FC_3_P2P_DMAC,
    XDMA_TT_FC_4_P2M_P,
    XDMA_TT_FC_5_P2P_SP,
    XDMA_TT_FC_6_M2P_P,
    XDMA_TT_FC_7_P2P_DP,
} XDMA_TransTypeFlowCtrl;
/**
 * @brief DW_axi_dmac BurstTransLength.
 *
 * @param XDMA_BTL_1  1 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_4  4 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_8  8 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_16  16 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_32  32 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_64  64 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_128  128 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_256  256 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_512  512 Data Item write to Destination/read from Source in the burst transaction
 * @param XDMA_BTL_1024  1024 Data Item write to Destination/read from Source in the burst transaction
 */
typedef enum
{
    XDMA_BTL_1 = 0,
    XDMA_BTL_4,
    XDMA_BTL_8,
    XDMA_BTL_16,
    XDMA_BTL_32,
    XDMA_BTL_64,
    XDMA_BTL_128,
    XDMA_BTL_256,
    XDMA_BTL_512,
    XDMA_BTL_1024,
} XDMA_BurstTransLength;

/**
 * @brief DW_axi_dmac TransferWidth.
 *
 * @param XDMA_TR_WIDTH_BYTE_1  Source/Destination Transfer Width is 8 bits
 * @param XDMA_TR_WIDTH_BYTE_2  Source/Destination Transfer Width is 16 bits
 * @param XDMA_TR_WIDTH_BYTE_4  Source/Destination Transfer Width is 32 bits
 * @param XDMA_TR_WIDTH_BYTE_8  Source/Destination Transfer Width is 64 bits
 * @param XDMA_TR_WIDTH_BYTE_16  Source/Destination Transfer Width is 128 bits
 * @param XDMA_TR_WIDTH_BYTE_32  Source/Destination Transfer Width is 256 bits
 * @param XDMA_TR_WIDTH_BYTE_64  Source/Destination Transfer Width is 512 bits
 * @param XDMA_TR_WIDTH_BYTE_128  Source/Destination Transfer Width is 1024 bits
 */
typedef enum
{
    XDMA_TR_WIDTH_BYTE_1 = 0,
    XDMA_TR_WIDTH_BYTE_2,
    XDMA_TR_WIDTH_BYTE_4,
    XDMA_TR_WIDTH_BYTE_8,
    XDMA_TR_WIDTH_BYTE_16,
    XDMA_TR_WIDTH_BYTE_32,
    XDMA_TR_WIDTH_BYTE_64,
    XDMA_TR_WIDTH_BYTE_128,
} XDMA_TransferWidth;

/**
 * @brief DW_axi_dmac HandshakingSelect.
 *
 * @param DW_axi_dmac Hardware shaking
 * @param DW_axi_dmac Software shaking
 */
typedef enum
{
    XDMA_HKS_HARDWARE = 0,
    XDMA_HKS_SOFTWARE,
} XDMA_HandshakingSelect;

typedef enum
{
    XDMA_HKS_HARD_INF_0 = 0,
    XDMA_HKS_HARD_INF_1,
    XDMA_HKS_HARD_INF_2,
    XDMA_HKS_HARD_INF_3,
    XDMA_HKS_HARD_INF_4,
    XDMA_HKS_HARD_INF_5,
    XDMA_HKS_HARD_INF_6,
    XDMA_HKS_HARD_INF_7,
    XDMA_HKS_HARD_INF_8,
    XDMA_HKS_HARD_INF_9,
    XDMA_HKS_HARD_INF_10,
    XDMA_HKS_HARD_INF_11,
    XDMA_HKS_HARD_INF_12,
    XDMA_HKS_HARD_INF_13,
    XDMA_HKS_HARD_INF_14,
    XDMA_HKS_HARD_INF_15,
} XDMA_HK_HARDWARE_INF;

typedef enum
{
    XDMA_CH_PRIORITY_0 = 0,     //lowest
    XDMA_CH_PRIORITY_1,
    XDMA_CH_PRIORITY_2,
    XDMA_CH_PRIORITY_3,
    XDMA_CH_PRIORITY_4,
    XDMA_CH_PRIORITY_5,
    XDMA_CH_PRIORITY_6,
    XDMA_CH_PRIORITY_7,         //highest
} XDMA_CHANNEL_PRIORITY;
//*****************************************************************************
//
//! Values that can be passed to XDMA_configChannel() as the
//! configure parameter.
//
//*****************************************************************************
typedef struct
{
    boolean              enableInterrupt; /* Enable/Disable interrupt mode */
    boolean				 enableAWfifo;      /* Enable/Disable xdma FIFO, default: disabled */
    uint32_t             destAddr; /* destination address */
    uint32_t             srcAddr; /* source address */
    uint32_t             blockTS; /* block transfer size */
    uint32_t			 awLen;			// Setting for ARLEN or AWLEN (Actual FIFO size -1, Max FIFO SIZE:128)
    XDMA_TransTypeFlowCtrl ttfc; /* transfer type and flow control */
    XDMA_BurstTransLength srcBtl; /* source burst transfer length */
    XDMA_BurstTransLength destBtl; /* dest burst transfer length */
    XDMA_AddressIncrement srcAddrDirect; /* source address Increment */
    XDMA_AddressIncrement destAddrDirect; /* dest address Increment */
    XDMA_TransferWidth    srcTrWidthBytes; /* src transfer width */
    XDMA_TransferWidth    destTrWidthBytes; /* dest transfer width */
    //
    // These following 4 items could not configured from user level.
    // they are re-configured by the function of XDMA_configChannel inside driver.
    //    
    XDMA_HandshakingSelect srcHkSelect; /* source handshaking select, reconfiged by driver inside. */
    XDMA_HandshakingSelect destHkSelect; /* dest handshaking select, reconfiged by driver inside. */
    XDMA_HK_HARDWARE_INF   srcHardInf; /* source hardware handshaking interface, reconfiged by driver inside. */
    XDMA_HK_HARDWARE_INF   destHardInf; /* dest hardware handshaking interface, reconfiged by driver inside. */
    XDMA_MultblkType reloadSrc; /* source address auto reload TYPE */
    XDMA_MultblkType reloadDst; /* dest address auto reload TYPE */
    DMAMUX_ReqId_Type dmaSrcReqId; /* source request trigger id */
    DMAMUX_ReqId_Type dmaDstReqId; /* dest request trigger id */
    //uint32_t srcGatherCount; /* source gather count */
    //uint32_t srcGatherInterval; /* source gather interval */
    //uint32_t destScatterCount; /* dest scatter count */
    //uint32_t destScatterInterval; /* dest scatter interval */

    XDMA_CHANNEL_PRIORITY chPriority;    /*  A priority of 7 is the highest priority, and 0
                                            is the lowest. This field must be programmed within the
                                            range 0 to DMAH_NUM_CHANNELS-1. A programmed
                                            value outside this range will cause erroneous behavior.*/
}XDMA_ConfigParams;

typedef struct
{
    union XCTL_L_REG ctl_l;
    union XCTL_H_REG ctl_h;
} XDmaCh_Ctl;

typedef struct
{
     union XCFG_L_REG cfg_l;
     union XCFG_H_REG cfg_h;
} XDmaCh_Cfg;

typedef struct
{
    XDmaCh_Ctl ctl;
    XDmaCh_Cfg cfg;
    union DMAMUX_CCR_REG  srcCcr;
    union DMAMUX_CCR_REG  destCcr;
    union DMAMUX_RGCR_REG srcRgcr;
    union DMAMUX_RGCR_REG destRgcr;
} XDmaCh_Parameters;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Declarations                      */
/* ========================================================================== */

/* None */

/**
 * \brief   add brief
 *
 * \param   parameter1   description of parameter1
 * \param   parameter2   description of parameter2
 *
 * \retval  None
 */

/**
 * @brief Checks an DMA channel base address.
 * This function determines if a DMA channel base address is valid.
 *
 * @param base specifies the DMA channel base address.
 * @return Returns \b true if the base address is valid and \b false
 * otherwise.
 */
__STATIC_INLINE boolean
XDMA_isBaseValid(uint32_t base)
{
    return( 
            (base == DMA1_BASE)
            || (base == DMA1_CH1_BASE)
            || (base == DMA1_CH2_BASE)
            || (base == DMA1_CH3_BASE)
            || (base == DMA1_CH4_BASE)
            || (base == DMA1_CH5_BASE)
            || (base == DMA1_CH6_BASE)
            || (base == DMA1_CH7_BASE)
            || (base == DMA1_CH8_BASE)
#if (XDMAC_VERSION == 0x22)
            || (base == DMA2_BASE)
            || (base == DMA2_CH1_BASE)
            || (base == DMA2_CH2_BASE)
            || (base == DMA2_CH3_BASE)
            || (base == DMA2_CH4_BASE)
            || (base == DMA2_CH5_BASE)
            || (base == DMA2_CH6_BASE)
            || (base == DMA2_CH7_BASE)
            || (base == DMA2_CH8_BASE)
#endif /* #if (XDMAC_VERSION == 0x22) */
          );

}

/**
 * @brief Converts the channel base to the corresponding DMA module base.
 * This function takes the channel base as input and returns the module base
 * where the given channel base address is located.
 *
 * @param base base is the channel base address of the DMA controller.
 * @return Returns the corresponding DMA moudle base.
 */
__STATIC_INLINE uint32_t
XDMA_convertChnBase2DmaBase(uint32_t base)
{
    uint32_t curDmacBase = 0U;
    
    switch(base)
    {
        case DMA1_BASE:
        case DMA1_CH1_BASE:
        case DMA1_CH2_BASE:
        case DMA1_CH3_BASE:
        case DMA1_CH4_BASE:
        case DMA1_CH5_BASE:
        case DMA1_CH6_BASE:
        case DMA1_CH7_BASE:
        case DMA1_CH8_BASE:
            curDmacBase = DMA1_BASE;
            break;

#if (XDMAC_VERSION == 0x22) 
        case DMA2_BASE:
        case DMA2_CH1_BASE:
        case DMA2_CH2_BASE:
        case DMA2_CH3_BASE:
        case DMA2_CH4_BASE:
        case DMA2_CH5_BASE:
        case DMA2_CH6_BASE:
        case DMA2_CH7_BASE:
        case DMA2_CH8_BASE:
            curDmacBase = DMA2_BASE;
            break;
#endif /* #if (XDMAC_VERSION == 0x22) */
        default:
            curDmacBase = 0U;
            ASSERT(0);
            break;
    }

    return curDmacBase;    
}

/**
 * @brief Converts the channel base to the corresponding DMA MUX module base.
 * This function takes the channel base as input and returns the module base
 * where the given channel base address is located.
 *
 * @param base base is the channel base address of the DMA controller.
 * @return Returns the corresponding DMA MUX Module base.
 */
__STATIC_INLINE uint32_t
XDMA_convertChnbase2DmamuxBase(uint32_t base)
{
    uint32_t curDmamuxBase = 0U;
    
    switch(base)
    {
        case DMA1_BASE:
        case DMA1_CH1_BASE:
        case DMA1_CH2_BASE:
        case DMA1_CH3_BASE:
        case DMA1_CH4_BASE:
        case DMA1_CH5_BASE:
        case DMA1_CH6_BASE:
        case DMA1_CH7_BASE:
        case DMA1_CH8_BASE:
            curDmamuxBase = DMA1_MUX_BASE;
            break;
        
#if (XDMAC_VERSION == 0x22) 
        case DMA2_BASE:
        case DMA2_CH1_BASE:
        case DMA2_CH2_BASE:
        case DMA2_CH3_BASE:
        case DMA2_CH4_BASE:
        case DMA2_CH5_BASE:
        case DMA2_CH6_BASE:
        case DMA2_CH7_BASE:
        case DMA2_CH8_BASE:
            curDmamuxBase = DMA2_MUX_BASE;
            break;
#endif /*#if (XDMAC_VERSION == 0x22)   */

        default:
            curDmamuxBase = 0U;
            ASSERT(0);
            break;
    }

    return curDmamuxBase;    
}

/**
 * @brief Converts the channel base to the DMA channel number.
 * This function takes the channel base as input and returns the module base
 * where the given channel base address is located.
 *
 * @param base base is the channel base address of the DMA controller.
 * @return Returns the corresponding DMA channel Number.
 */
__STATIC_INLINE uint32_t
XDMA_convertChnBase2ChnNum(uint32_t base)
{
    uint32_t chnnel = 0U;
    
    switch(base)
    {
        case DMA1_CH1_BASE:
        case DMA2_CH1_BASE:
            chnnel = 0;
            break;
        
        case DMA1_CH2_BASE:
        case DMA2_CH2_BASE:
            chnnel = 1;
            break;

        case DMA1_CH3_BASE:
        case DMA2_CH3_BASE:
            chnnel = 2;
            break;
        
        case DMA1_CH4_BASE:
        case DMA2_CH4_BASE:
            chnnel = 3;
            break;
        
        case DMA1_CH5_BASE:
        case DMA2_CH5_BASE:
            chnnel = 4;
            break;

        case DMA1_CH6_BASE:
        case DMA2_CH6_BASE:
            chnnel = 5;
            break;
        
        case DMA1_CH7_BASE:
        case DMA2_CH7_BASE:
            chnnel = 6;
            break;
                  
        case DMA1_CH8_BASE:
        case DMA2_CH8_BASE:
            chnnel = 7;
            break;

        default:
            chnnel = 0;
            break;
    }

    return chnnel;    
}

/**
 * @brief Initializes the DMA controller to a known state.
 * This function configures does a hard reset of the DMA controller in order
 * to put it into a known state. The function also sets the DMA to run free
 * during an emulation suspend (see the field DEBUGCTRL.FREE for more info).
 *
 * @param base base is the channel base address of the DMA controller.
 * @return None.
 */
__STATIC_INLINE void
XDMA_initController(uint32_t base)
{
    uint32_t dmacBase = 0U;
    uint32_t dmamuxBase = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments. */
    ASSERT(XDMA_isBaseValid(base));
    
    /*Covert the channel base to the Module base and the dmamux base address.*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);
    dmamuxBase = XDMA_convertChnbase2DmamuxBase(base);

    /*Point to the Controller Config register */
    hwCtl = XDMAC(dmacBase);

    /*Dmac Disabled.*/
    hwCtl->COMCFG_L.bit.EN = XDMA_DISABLE;
    
    /*Check Dmac Disable success.*/
    while (hwCtl->COMCFG_L.bit.EN != XDMA_DISABLE);

    /*Dmac Enabled.*/
    hwCtl->COMCFG_L.bit.EN = XDMA_ENABLE;

    /*Clear legacy config for dmamux module.*/
    DMAMUX_resetDmaMux(dmamuxBase);    
}


/**
 * @brief Enable the DMA controller.
 * This function configures DMA module to eanbled status.
 *
 * @param base base is the channel base address of the DMA controller.
 * @return None.
 */
__STATIC_INLINE void
XDMA_enableModule(uint32_t base)
{
    uint32_t dmacBase = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to Module base*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register */
    hwCtl = XDMAC(dmacBase);

    /* Enabled Dma Controller*/
    hwCtl->COMCFG_L.bit.EN = XDMA_ENABLE;
}

/**
 * @brief Disable the DMA controller.
 * This function configures DMA module to disabled status.
 *
 * @param base base is the channel base address of the DMA controller.
 * @return None.
 */
__STATIC_INLINE void
XDMA_disableModule(uint32_t base)
{
    uint32_t dmacBase = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to Module base*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);


    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);


    /*Disabled Dma Controller*/
    hwCtl->COMCFG_L.bit.EN = XDMA_DISABLE;
}

/**
 * @brief Trigger Soft Reset Channel
 * This function does a soft reset to place the channel into its default state
 *
 * @param base base is the channel base address of the DMA controller.
 * @return None.
 */
__STATIC_INLINE void
XDMA_triggerSoftReset(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    uint32_t tempVal = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to channel number and dmac base base*/
    channel = XDMA_convertChnBase2ChnNum(base);    
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Disabled Dmac channel*/
    tempVal = hwCtl->CHNEN_L.bit.CHN_EN | (1U << (8 +channel));
    tempVal &= (~(1U << channel) & 0xFFFF);
    hwCtl->CHNEN_L.bit.CHN_EN = tempVal;


    /*Check Dmac Disable success*/
    while (hwCtl->CHNEN_L.bit.CHN_EN & (1 << channel));

    /*Enabled Dmac channel again*/
    hwCtl->CHNEN_L.bit.CHN_EN |= (1U << (8 + channel)) | (1U << channel);

}

/**
 * @brief Enables peripherals to trigger a DMA transfer.
 * This function enables the selected peripheral trigger to start a DMA
 * transfer on the specified channel.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_enableTrigger(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief Disables peripherals from triggering a DMA transfer.
 * This function disables the selected peripheral trigger to start a DMA
 * transfer on the specified channel.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_disableTrigger(uint32_t base)
{

	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief Force a peripheral trigger to a DMA channel.
 * This function sets the peripheral trigger flag and if triggering a DMA
 * burst is enabled (see DMA_enableTrigger()), a DMA burst transfer will be
 * forced.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_forceTrigger(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Config the channel to Softeware handshaking, Both for Source and Destination.*/
    hwChannel->CFG_H.bit.HS_SEL_SRC = XDMA_HKS_SOFTWARE;
    hwChannel->CFG_H.bit.HS_SEL_DST = XDMA_HKS_SOFTWARE;

    /*Enable and Active source software transaction request for the DMA controller*/
    hwChannel->SWHSSRC_L.bit.SWHS_REQ_SRC_WE = XDMA_ENABLE;
    hwChannel->SWHSSRC_L.bit.SWHS_REQ_SRC = XDMA_ACTIVE;    

    /*Enable and Active destination software transaction request for the DMA controller*/
    hwChannel->SWHSDST_L.bit.SWHS_REQ_DST_WE = XDMA_ENABLE;
    hwChannel->SWHSDST_L.bit.SWHS_REQ_DST = XDMA_ACTIVE;
    

    /*Enable and Active source Single transaction request for the DMA controller*/
    hwChannel->SWHSSRC_L.bit.SWHS_SGLREQ_SRC_WE = XDMA_ENABLE;
    hwChannel->SWHSSRC_L.bit.SWHS_SGLREQ_SRC = XDMA_ACTIVE;

    /*Enable and Active destination single transaction request for the DMA controller*/
    hwChannel->SWHSDST_L.bit.SWHS_SGLREQ_DST_WE = XDMA_ENABLE;
    hwChannel->SWHSDST_L.bit.SWHS_SGLREQ_DST = XDMA_ACTIVE;
}

/**
 * @brief Clears a DMA channel's peripheral trigger flag.
 * This function clears the peripheral trigger flag. Normally, you would use
 * this function when initializing the DMA for the first time. The flag is
 * cleared automatically when the DMA starts the first burst of a transfer.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_clearTriggerFlag(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief Gets the status of a DMA channel's Transfer Status Flag.
 * This function returns \b true if the Transfer Status Flag is set, which
 * means a DMA transfer has begun.
 * This flag is cleared when TRANSFER_COUNT reaches zero, or when the
 * HARDRESET or SOFTRESET bit is set.
 *
 * @param base is the base address of the DMA channel control registers.
 * @return Returns \b true if the Transfer Status Flag is set. Returns \b false
 * otherwise.
 */
__STATIC_INLINE boolean
XDMA_getTransferStatusFlag(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);

    return 0;
}
/**
 * @brief Gets the status of a DMA channel's Burst Status Flag.
 * This function returns \b true if the Burst Status Flag is set, which
 * means a DMA burst has begun.
 * This flag is cleared when BURST_COUNT reaches zero, or when the
 * HARDRESET or SOFTRESET bit is set.
 *
 * @param base is the base address of the DMA channel control registers.
 * @return Returns \b true if the Transfer Status Flag is set. Returns \b false
 * otherwise.
 */
__STATIC_INLINE boolean
XDMA_getBurstStatusFlag(uint32_t base)
{
    //
    //  .
    //
	/*Empty function for current IP not support*/
    ASSERT(FALSE);

    return 0;
}

/**
 * @brief Gets the status of a DMA channel's Run Status Flag.
 * This function returns \b true if the Run Status Flag is set, which
 * means the DMA channel is enabled.
 * This flag is cleared when a transfer completes (TRANSFER_COUNT = 0) and
 * continuous mode is disabled, or when the HARDRESET, SOFTRESET, or HALT bit
 * is set.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return Returns \b true if the channel is enabled. Returns \b false
 * otherwise.
 */
__STATIC_INLINE boolean
XDMA_getRunStatusFlag(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to channel number and dmac base address*/
    channel = XDMA_convertChnBase2ChnNum(base);     
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Check the current channel is enabled or disabled and return*/
    return ((hwCtl->CHNEN_L.bit.CHN_EN & (1U << channel)) != 0U);
}

/**
 * @brief Gets the status of a DMA channel's Overflow Flag.
 * This function returns \b true if the Overflow Flag is set, which
 * means peripheral event trigger was received while Peripheral Event Trigger
 * Flag was already set.
 * This flag can be cleared by writing to ERRCLR bit, using the function
 * DMA_clearErrorFlag().
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return Returns \b true if the channel is enabled. Returns \b false
 * otherwise.
 */
__STATIC_INLINE boolean
XDMA_getOverflowFlag(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);

    return 0;
}

/**
 * @brief Gets the status of a DMA channel's peripheral trigger flag.
 * This function returns \b true if a peripheral trigger event has occurred
 * The flag is automatically cleared when the first burst transfer begins, but
 * if needed, it can be cleared using DMA_clearTriggerFlag().
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return Returns \b true if a peripheral trigger event has occurred and its
 * flag is set. Returns \b false otherwise.
 */
__STATIC_INLINE boolean
XDMA_getTriggerFlagStatus(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);

    return 0;
}

/**
 * @brief Starts a DMA channel.
 * This function starts the DMA running, typically after you have configured
 * it. It will wait for the first trigger event to start operation. To halt
 * the channel use DMA_stopChannel().
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_startChannel(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dmacBase = 0U;
    uint32_t tempVal = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to channel number and dmac base address*/
    channel = XDMA_convertChnBase2ChnNum(base);    
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Enable the channel*/
    tempVal = ((1U << (8U + channel)) | (1U << channel));
    hwCtl->CHNEN_L.bit.CHN_EN |= tempVal;

    tempVal = hwCtl->CHNEN_L.bit.CHN_EN;

}

/**
 * @brief Halts a DMA channel.
 * This function halts the DMA at its current state and any current read-write
 * access is completed. To start the channel again use DMA_startChannel().
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_stopChannel(uint32_t base)
{
    uint32_t channel = 0U;
    uint32_t dma_base=0;
    volatile uint32_t timeout=0;
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to channel number and dmac base address*/
    channel = XDMA_convertChnBase2ChnNum(base);     
    /*Stop the channel*/
    dma_base=(base&(~XDMA_BASE_M));//Retrieve the address of dma_base.

    if((HWREG(dma_base+XDMA_O_XCHNEN) & (1<<channel)) == 0)//Determine whether this channel is enabled.
    {
    	return ;
    }

    if((HWREG(base+XDMA_O_XENINTS) & (1U<<29U)) == 0)//Enable the function for judging the channel's suspended state
    {
    	return;
    }

    HWREG(dma_base+XDMA_O_XCHNEN) = (1 << (16+channel)) | (1 << (24+channel));//Suspend a certain DMA_channel

    while((HWREG(base+XDMA_O_XINTS) & (1U << 29U)) == 0 && (timeout++ < 128));//Poll the DMA_channel to ensure successful hang-up.

    HWREG(dma_base+XDMA_O_XCHNEN) = 1U << (8U +channel);//Disable DMA channel
}

/**
 * @brief Reset all channels.
 * This function is used to reset all channels.
 *
 * @param base base is the channel base address of the DMA controller.
 * @return None.
 */
__STATIC_INLINE void
XDMA_resetAllChannel(uint32_t base)
{
    uint32_t dmacBase = 0U;
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    /*Covert the channel base to the Module base and the dmamux base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);
    /*Dmac reset*/
    HWREG(dmacBase+XDMA_O_XRESET)=1U;

}

/**
 * @brief Enables a DMA channel interrupt source.
 * This function enables the indicated DMA channel interrupt source.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_enableInterrupt(uint32_t base)
{
    uint32_t dmacBase = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Enable Current Channel interrupt*/
    hwCtl->COMCFG_L.bit.INT_EN = XDMA_ENABLE;
}

/**
 * @brief Disables a DMA channel interrupt source.
 * This function disables the indicated DMA channel interrupt source.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_disableInterrupt(uint32_t base)
{
    uint32_t dmacBase = 0U;
    XDMA_CtlSta_TypeDef *hwCtl = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Disable Current Channel interrupt*/
    hwCtl->COMCFG_L.bit.INT_EN = XDMA_ENABLE;
}

/**
 * @brief Enables the DMA channel overrun interrupt.
 * This function enables the indicated DMA channel's ability to generate an
 * interrupt upon the detection of an overrun. An overrun is when a peripheral
 * event trigger is received by the DMA before a previous trigger on that
 * channel had been serviced and its flag had been cleared.
 *
 * Note that this is the same interrupt signal as the interrupt that gets
 * generated at the beginning/end of a transfer. That interrupt must first be
 * enabled using DMA_enableInterrupt() in order for the overrun interrupt to
 * be generated.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_enableOverrunInterrupt(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief Disables the DMA channel overrun interrupt.
 * This function disables the indicated DMA channel's ability to generate an
 * interrupt upon the detection of an overrun.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_disableOverrunInterrupt(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief Clears the DMA channel error flags.
 * This function clears both the DMA channel's sync error flag and its
 * overrun error flag.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @return None.
 */
__STATIC_INLINE void
XDMA_clearErrorFlag(uint32_t base)
{
    uint32_t dmacBase = 0U;
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);


    /*clear the whole dmac of the current channel 's Err interrupts*/
    hwChannel->CLRINTS_L.all = XDMA_CHX_ERR_INTS_L_M;
    hwChannel->CLRINTS_H.all = XDMA_CHX_ERR_INTS_H_M; 
}

/**
 * @brief Sets the DMA channel priority mode.
 * This function sets the channel interrupt mode. When the \e ch1IsHighPri
 * parameter is \b false, the DMA channels are serviced in round-robin mode.
 * This is the default behavior.
 * This function can only be used by channel 0, in function, the base address
 * would be translate to channel 0 base address by force.
 *
 *
 * If \b true, channel 0 will be given higher priority than the other
 * channels. This means that if a channel 0 trigger occurs, the current word
 * transfer on any other channel is completed and channel 0 is serviced for
 * the complete burst count. The lower-priority channel's interrupted transfer
 * will then resume.
 * @param base base is the base address of the DMA control registers.
 * @param ch1IsHighPri ch1IsHighPri is a flag to indicate the channel interrupt mode.
 * @return None.
 */
__STATIC_INLINE void
XDMA_setPriorityMode(uint32_t base, boolean ch0IsHighPri)
{
    uint32_t dmacBase = 0U;
    XDMA_Channel_TypeDef *hwChannel = NULL;
    
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);    

    /*Point to the first Channel register.
     *Ch0 base address is the dmac base address + 0x100.
     */
    hwChannel = XDMA_Channel(dmacBase + 0x100);

    /*Set priority for channel 0*/
    if (ch0IsHighPri)
        hwChannel->CFG_H.bit.CH_PRIOR = XDMA_CH_PRIORITY_7;
    else
        hwChannel->CFG_H.bit.CH_PRIOR = XDMA_CH_PRIORITY_0;
}



/**
 * @brief   set channel priority
 *
 * @param   base       the base address of the DMA channel control registers.
 * @param   prior      the channel priority to set .
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_setChannelPriority(uint32_t base, XDMA_CHANNEL_PRIORITY prior)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    
    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set priority for channel*/
    hwChannel->CFG_H.bit.CH_PRIOR = prior;
}

/**
 * @brief Configures the source address for the DMA channel
 * This function configures the source address of a DMA
 * channel.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param srcAddr *srcAddr is a source address.
 * @return None.
 */
__STATIC_INLINE void
XDMA_configSourceAddress(uint32_t base, uint32_t srcAddr)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Set priority for channel*/
    hwChannel = XDMA_Channel(base);

    /*Set priority for channel*/
    hwChannel->SAR_L.bit.SAR = srcAddr;
    hwChannel->SAR_H.bit.SAR = 0U;
}

/**
 * @brief Configures the destination address for the DMA channel
 * This function configures the destinaton address of a DMA
 * channel.
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param destAddr *destAddr is the destination address.
 * @return None.
 */
__STATIC_INLINE void
XDMA_configDestAddress(uint32_t base, uint32_t destAddr)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);


    /*Set destination Address*/
    hwChannel->DAR_L.bit.DAR = destAddr;
    hwChannel->DAR_H.bit.DAR = 0U;
}

/**
 * @brief Set Source Manager Select.
 * This function sets the Source Manager Select typically before the DMA starts running.
 * The default SMS (Source Manager Select) is 0 (AXI manager 1). In some devices, if the RAM or
 * Peripheral has been connected to the AXI manager 2, we need to set the SMS to 2 (AXI Manager 2).
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param srcInf srcInf is the Source Manager Select interface.
 * @return None.
 */
__STATIC_INLINE void
XDMA_setSrcManagerSelect(uint32_t base, XDMA_MasterInfSelect srcInf)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(srcInf >= XDMA_MASTER1_INTF && srcInf <= XDMA_MASTER2_INTF);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Source Manager Select*/
    hwChannel->CTL_L.bit.SMS = srcInf;
}

/**
 * @brief Set Destination Manager Select.
 * This function sets the Destination Manager Select typically before the DMA starts running.
 * The default DMS (Destination Manager Select) is 0 (AXI manager 1). In some devices, if the RAM or
 * Peripheral has been connected to the AXI manager 2, we need to set the DMS to 2 (AXI Manager 2).
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param destInf destInf is the Destination Manager Select interface.
 * @return None.
 */
__STATIC_INLINE void
XDMA_setDestManagerSelect(uint32_t base, XDMA_MasterInfSelect destInf)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(destInf >= XDMA_MASTER1_INTF && destInf <= XDMA_MASTER2_INTF);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Destination Manager Select*/
    hwChannel->CTL_L.bit.DMS = destInf;
}

/**
 * @brief Configure block transfer done
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param 0: DONE bit is deasserted at the end of block transfer
 *        1: SET the DONE bit at the end of block transfer
 * @return None.
 */
__STATIC_INLINE void
XDMA_configDone(uint32_t base, uint32_t done)
{
	/* Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief Assigns a hardware handshaking interface (0 : 15)
 *
 * @param base base is the base address of the DMA channel control registers.
 * @param dest_per dest_per is the Dest Hardware Interface
 * @param src_per src_per is the Source Hardware Interface
 * @return None.
 */
__STATIC_INLINE void
XDMA_configHardwareInterface(uint32_t base, uint32_t dest_per, uint32_t src_per)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT((dest_per >= 0U) && (dest_per <= 15U));
    ASSERT((src_per >= 0U) && (src_per <= 15U));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /* Assigns a hardware handshaking interface both for destination and source HK*/
    hwChannel->CFG_H.bit.SRC_PER = src_per;
    hwChannel->CFG_H.bit.DST_PER = dest_per;
}

/**
 * @brief   config protection control
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   protctl  protctl    Protection Control bits, 3 bits total for HPROT.
 *          CFGx.PROTCTL[1] to HPROT[1] privileged
 *          CFGx.PROTCTL[2] to HPROT[2] non-buffered
 *          CFGx.PROTCTL[3] to HPROT[3] non-cached
 *          ps: HPROT[0] is tied high
 * @return  None
 */
__STATIC_INLINE void
XDMA_configProtctl(uint32_t base, uint32_t protctl)
{
	/* Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   FIFO Mode Select
 *
 * @param   base       the base address of the DMA channel control registers.
 * @param   fifo_mode  fifo mode as bellow:
 *          0x0 (FIFO_MODE_0): Space/data available for single
 *          AHB transfer of the specified transfer width
 *          0x1 (FIFO_MODE_1): Data available is greater than or
 *          equal to half the FIFO depth for destination transfers and
 *          space available is greater than half the fifo depth for
 *          source transfers. The exceptions are at the end of a burst
 *          transaction request or at the end of a block transfer
 * @return  None
 */
__STATIC_INLINE void
XDMA_configFifoMode(uint32_t base, uint32_t fifo_mode)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   Flow Control Mode
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   fc_mode  fc mode as bellow:
 *          0x0 (FCMODE_0): Source transaction requests are
 *          serviced when they occur. Data pre-fetching is enabled
 *          0x1 (FCMODE_1): Source transaction requests are not
 *          serviced until a destination transaction request occurs. In
 *           this mode, the amount of data transferred from the source
 *           is limited so that it is guaranteed to be transferred to the
 *           destination prior to block termination by the destination.
 *           Data pre-fetching is disabled.
 * @return  None
 */
__STATIC_INLINE void
XDMA_configFcMode(uint32_t base, uint32_t fc_mode)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   enable Automatic Reload
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @retval  None
 */
__STATIC_INLINE void
XDMA_enableReload(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Source/Destination Multi Block Transfer Type as Reload*/
    hwChannel->CFG_L.bit.SRC_MULTBLK_TYPE = XDMA_RELOAD;
    hwChannel->CFG_L.bit.DST_MULTBLK_TYPE = XDMA_RELOAD;
}

/**
 * @brief   disable Automatic Reload
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @retval  None
 */
__STATIC_INLINE void
XDMA_disableReload(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Source/Destination Multi Block Transfer Type as Reload*/
    hwChannel->CFG_L.bit.SRC_MULTBLK_TYPE = XDMA_CONTIGOUS;
    hwChannel->CFG_L.bit.DST_MULTBLK_TYPE = XDMA_CONTIGOUS;
}

/**
 * @brief   config Handshaking Interface Polarity
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   src_hs_pol   Source Handshaking Interface Polarity ,0 is Active high, 1 is Active low.
 * @param   dst_hs_pol   Dest Handshaking Interface Polarity ,0 is Active high, 1 is Active low.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configHandshakingInterfacePolarity(uint32_t base, uint32_t src_hs_pol, uint32_t dst_hs_pol)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT((src_hs_pol >= 0U) && (src_hs_pol <= 1U));
    ASSERT((dst_hs_pol >= 0U) && (dst_hs_pol <= 1U));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Source/Destination Hardware Handshaking Interface Polarity*/
    hwChannel->CFG_H.bit.SRC_HWHS_POL = src_hs_pol;
    hwChannel->CFG_H.bit.DST_HWHS_POL = dst_hs_pol;    
}

/**
 * @brief   Select Software or Hardware Handshaking
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   hs_sel_src   Source Handshaking Interface select, 0 is hardware, 1 is software.
 * @param   hs_sel_dst   Dest Handshaking Interface select, 0 is hardware, 1 is software.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configHandshakingInterface(uint32_t base, uint32_t hs_sel_src, uint32_t hs_sel_dst)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT((hs_sel_src >= 0U) && (hs_sel_src <= 1U));
    ASSERT((hs_sel_dst >= 0U) && (hs_sel_dst <= 1U));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Source/Destination Software or Hardware Handshaking Select*/
    hwChannel->CFG_H.bit.HS_SEL_SRC = hs_sel_src;
    hwChannel->CFG_H.bit.HS_SEL_DST = hs_sel_dst;
}

/**
 * @brief   get Channel FIFO status
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  0 is not empty, 1 is empty
 */
__STATIC_INLINE boolean
XDMA_isFifoEmpty(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Check the total number of data left in the channel FIFO*/
    return (hwChannel->STATUS_H.bit.DATA_LEFT_IN_FIFO != 0U);
}

/**
 * @brief   channel suspend
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_suspendChannel(uint32_t base)
{
    XDMA_CtlSta_TypeDef *hwCtl = NULL;
    uint32_t dmacBase = 0U;
    uint32_t channel = 0U;
    uint32_t tempVal = 0U;
    
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to channel number and dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);
    channel = XDMA_convertChnBase2DmaBase(base);    

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Enable current channel Suspend Bit*/
    hwCtl->CHNEN_L.bit.CHN_SUSP |= (1U << (8 + channel)) | (1U << channel);
}

/**
 * @brief   channel resume
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_resumeChannel(uint32_t base)
{
    XDMA_CtlSta_TypeDef *hwCtl = NULL;
    uint32_t dmacBase = 0U;
    uint32_t channel = 0U;
    uint32_t tempVal = 0U;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Covert the channel base to channel number and dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);
    channel = XDMA_convertChnBase2DmaBase(base);    

    /*Point to the Controller Config register */
    hwCtl = XDMAC(dmacBase);

    /*Enable current channel Suspend Bit*/
    tempVal = hwCtl->CHNEN_L.bit.CHN_SUSP | (1U << (8 +channel));
    tempVal &= (~(1U << channel) & 0xFFFF);
    hwCtl->CHNEN_L.bit.CHN_SUSP = tempVal;
}

/**
 * @brief   config Block Transfer Size
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   block_ts   Block Transfer Size, 0-4095.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configBlockTs(uint32_t base, uint32_t block_ts)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(block_ts <= 0x3FFFFFF);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set BLOCK TS*/
    hwChannel->BLOCK_L.bit.BLOCK_TS = block_ts;
}

/**
 * @brief   config Transfer Type and Flow Control
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   tt_fc      Transfer Type and Flow Control.
 *          0x0 (TT_FC_0): Transfer type is Memory to Memory and
 *          Flow Controller is DW_ahb_dmac
 *          0x1 (TT_FC_1): Transfer type is Memory to Peripheral
 *          and Flow Controller is DW_ahb_dmac
 *          0x2 (TT_FC_2): Transfer type is Peripheral to Memory
 *          and Flow Controller is DW_ahb_dmac
 *          0x3 (TT_FC_3): Transfer type is Peripheral to Peripheral
 *          and Flow Controller is DW_ahb_dmac
 *          0x4 (TT_FC_4): Transfer type is Peripheral to Memory
 *          and Flow Controller is Peripheral
 *          0x5 (TT_FC_5): Transfer type is Peripheral to Peripheral
 *          and Flow Controller is Source Peripheral
 *          0x6 (TT_FC_6): Transfer type is Memory to Peripheral
 *          and Flow Controller is Peripheral
 *          0x7 (TT_FC_7): Transfer type is Peripheral to Peripheral
 *          and Flow Controller is Destination Peripheral
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configTtFc(uint32_t base, uint32_t tt_fc)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(tt_fc <= XDMA_TT_FC_M);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Block Transfer Size*/
    hwChannel->CFG_H.bit.TT_FC = tt_fc;
}

/**
 * @brief   config Burst Transaction Length
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   src_msize  source Burst Transaction Length.
 * @param   dst_msize  dest Burst Transaction Length.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configMSize(uint32_t base, uint32_t src_msize, uint32_t dst_msize)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(src_msize <= XDMA_BTL_1024);
    ASSERT(dst_msize <= XDMA_BTL_1024);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*set Source and Destination Burst Transaction Length*/
    hwChannel->CTL_L.bit.SRC_MSIZE = src_msize;
    hwChannel->CTL_L.bit.DST_MSIZE = dst_msize; 

}

/**
 * @brief   config Address Increment
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   sinc       source Address Increment.
 *          0x0 (XDMA_ADDR_INCRE): Increments the source address
 *          0x1 (XDMA_ADDR_NO_CHANGE): No change in the source address
 * @param   dinc       dest Address Increment.
 *          0x0 (XDMA_ADDR_INCRE): Increments the destination address
 *          0x1 (XDMA_ADDR_NO_CHANGE): No change in the destination address
 * @return  None
 */
__STATIC_INLINE void
XDMA_configAddrInc(uint32_t base, uint32_t sinc, uint32_t dinc)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(sinc >= XDMA_ADDR_INCRE && sinc <= XDMA_ADDR_NO_CHANGE);
    ASSERT(dinc >= XDMA_ADDR_INCRE && dinc <= XDMA_ADDR_NO_CHANGE);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Source/Destination Address Increment*/
    hwChannel->CTL_L.bit.SINC = sinc;
    hwChannel->CTL_L.bit.DINC = dinc;
}

/**
 * @brief   config Transfer Width
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   src_tr_width  source Transfer Width.
 *          0x0 (XDMA_TR_WIDTH_BYTE_1): Source transfer width is 8 bits
 *          0x1 (XDMA_TR_WIDTH_BYTE_2): Source transfer width is 16 bits
 *          0x2 (XDMA_TR_WIDTH_BYTE_4): Source transfer width is 32 bits
 *          0x3 (XDMA_TR_WIDTH_BYTE_8): Source transfer width is 64 bits
 *          0x4 (XDMA_TR_WIDTH_BYTE_16): Source transfer width is 128 bits
 *          0x5 (XDMA_TR_WIDTH_BYTE_32): Source transfer width is 256 bits
 *          0x6 (XDMA_TR_WIDTH_BYTE_64): Source transfer width is 256 bits
 *          0x7 (XDMA_TR_WIDTH_BYTE_128): Source transfer width is 256 bits
 * @param   dst_tr_width  dest Transfer Width.
 *          0x0 (XDMA_TR_WIDTH_BYTE_1): Destination transfer width is 8 bits
 *          0x1 (XDMA_TR_WIDTH_BYTE_2): Destination transfer width is 16 bits
 *          0x2 (XDMA_TR_WIDTH_BYTE_4): Destination transfer width is 32 bits
 *          0x3 (XDMA_TR_WIDTH_BYTE_8): Destination transfer width is 64 bits
 *          0x4 (XDMA_TR_WIDTH_BYTE_16): Destination transfer width is 128 bits
 *          0x5 (XDMA_TR_WIDTH_BYTE_32): Destination transfer width is 256 bits
 *          0x6 (XDMA_TR_WIDTH_BYTE_64): Destination transfer width is 256 bits
 *          0x7 (XDMA_TR_WIDTH_BYTE_128): Destination transfer width is 256 bits
 * @return  None
 */
__STATIC_INLINE void
XDMA_configTrWidth(uint32_t base, uint32_t src_tr_width, uint32_t dst_tr_width)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(src_tr_width >= XDMA_TR_WIDTH_BYTE_1 && src_tr_width <= XDMA_TR_WIDTH_BYTE_128);
    ASSERT(dst_tr_width >= XDMA_TR_WIDTH_BYTE_1 && dst_tr_width <= XDMA_TR_WIDTH_BYTE_128);


    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set Source/Destination Transfer Width*/
    hwChannel->CTL_L.bit.SRC_TR_WIDTH = src_tr_width;
    hwChannel->CTL_L.bit.DST_TR_WIDTH = dst_tr_width;
}

/**
 * @brief   Source gather enable
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_enableSrcGather(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   Source gather disable
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_disableSrcGather(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   Destination scatter enable
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_enableDstScatter(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   Destination scatter disable
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_disableDstScatter(uint32_t base)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   config source gather
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   sgi           Source Gather Interval.
 * @param   sgc           Source Gather Count. Source contiguous transfer count between successive gather boundaries.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configSrcGather(uint32_t base, uint32_t sgi, uint32_t sgc)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   config dest scatter
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   dsi           Destination Scatter Interval.
 * @param   dsc           Destination Scatter Count.Destination contiguous transfer count between successive scatter boundaries.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configDstScatter(uint32_t base, uint32_t dsi, uint32_t dsc)
{
	/*Empty function for current IP not support*/
    ASSERT(FALSE);
}

/**
 * @brief   config channel ctrl
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   ctl
 *          DONE             bit    44           Done bit.
 *          BLOCK_TS         bit    43:32        Block Transfer Size.
 *          TT_FC            bit    22:20        Transfer Type and Flow Control.
 *          DST_SCATTER_EN   bit    18           Destination scatter enable.
 *          SRC_GATHER_EN    bit    17           Source gather enable.
 *          SRC_MSIZE        bit    16:14        Source Burst Transaction Length.
 *          DEST_MSIZE       bit    13:11        Destination Burst Transaction Length.
 *          SINC             bit    10:9         Source Address Increment.
 *          DINC             bit    8:7          Destination Address Increment.
 *          SRC_TR_WIDTH     bit    6:4          Source Transfer Width.
 *          DST_TR_WIDTH     bit    3:1          Destination Transfer Width.
 *          INT_EN           bit    0            Interrupt Enable Bit.
 * @return  None
 */
__STATIC_INLINE void
XDMA_configChannelCTL(uint32_t base, uint64_t ctl)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    
    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set CTL configuration*/
    hwChannel->CTL_H.all = ((ctl >> 32) & 0xffffffff);
    hwChannel->CTL_L.all = (ctl & 0xffffffff);
}

__STATIC_INLINE void
XDMA_configChannelBlockTS(uint32_t base, uint32_t blockTS)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set CTL configuration*/
    hwChannel->CTL_H.all = (hwChannel->CTL_H.all & 0xfffff000) | (blockTS & 0x00000fff);
}

/**
 * @brief   config channel ctl
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   ctl           the pointer of ctl value struct
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_Ch_setCtl(uint32_t base, XDmaCh_Ctl *ctl)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    ASSERT(ctl != NULL);

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set CTL configuration*/
    hwChannel->CTL_H.all = ctl->ctl_h.all;
    hwChannel->CTL_L.all = ctl->ctl_l.all;
}

/**
 * @brief   config channel cfg
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   cfg
 *          DEST_PER           bit    46:43        Assigns a Destination hardware interface.
 *          SRC_PER            bit    42:39        Assigns a Source Hardware Interface.
 *          PROTCTL            bit    36:34        Protection Control bits used to drive the AHB HPROT[3:1] bus.
 *          FIFO_MODE          bit    33           FIFO Mode Select.
 *          FCMODE             bit    32           Flow Control Mode.
 *          RELOAD_DST         bit    31           Automatic Destination Reload.
 *          RELOAD_SRC         bit    30           Automatic Source Reload.
 *          SRC_HS_POL         bit    19           Source Handshaking Interface Polarity
 *          DST_HS_POL         bit    18           Destination Handshaking Interface Polarity.
 *          HS_SEL_SRC         bit    11           Source Software or Hardware Handshaking Select.
 *          HS_SEL_DST         bit    10           Destination Software or Hardware Handshaking Select.
 *          CH_SUSP            bit    8            Channel Suspend.
 *          CH_PRIOR           bit    7:5          Channel Priority.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_configChannelCFG(uint32_t base, uint64_t cfg)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set CFG configuration*/
    hwChannel->CFG_H.all = ((cfg >> 32) & 0xffffffff);
    hwChannel->CFG_L.all = (cfg & 0xffffffff);
}

/**
 * @brief   config channel cfg
 *
 * @param   base base is  the base address of the DMA channel control registers.
 * @param   cfg           the pointer of cfg value struct
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_Ch_setCfg(uint32_t base, XDmaCh_Cfg *cfg)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set CFG configuration*/
    hwChannel->CFG_H.all = cfg->cfg_h.all;
    hwChannel->CFG_L.all = cfg->cfg_l.all;
}

/**
 * @brief   get channel ctrl
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @return  ctl value
 *          DONE             bit    44           Done bit.
 *          BLOCK_TS         bit    43:32        Block Transfer Size.
 *          TT_FC            bit    22:20        Transfer Type and Flow Control.
 *          DST_SCATTER_EN   bit    18           Destination scatter enable.
 *          SRC_GATHER_EN    bit    17           Source gather enable.
 *          SRC_MSIZE        bit    16:14        Source Burst Transaction Length.
 *          DEST_MSIZE       bit    13:11        Destination Burst Transaction Length.
 *          SINC             bit    10:9         Source Address Increment.
 *          DINC             bit    8:7          Destination Address Increment.
 *          SRC_TR_WIDTH     bit    6:4          Source Transfer Width.
 *          DST_TR_WIDTH     bit    3:1          Destination Transfer Width.
 *          INT_EN           bit    0            Interrupt Enable Bit.
 */
__STATIC_INLINE uint64_t
XDMA_getChannelCTL(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*return CTL configuration*/
    return ((uint64_t)hwChannel->CTL_H.all << 32) + ((uint64_t)hwChannel->CTL_L.all);
}

/**
 * @brief   get channel ctl
 *
 * @param   base base is  the base address of the DMA channel control registers.
 * @return  ctl           value of channel ctl
 *
 */
__STATIC_INLINE XDmaCh_Ctl
XDMA_Ch_getCtl(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    XDmaCh_Ctl ctl = {0};    
    
    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);
    
    /*get CTL configuration*/
    ctl.ctl_h.all = hwChannel->CTL_H.all;
    ctl.ctl_l.all = hwChannel->CTL_L.all;

    return ctl;
}

/**
 * @brief   get channel cfg
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @return  cfg value
 *          DEST_PER           bit    46:43        Assigns a Destination hardware interface.
 *          SRC_PER            bit    42:39        Assigns a Source Hardware Interface.
 *          PROTCTL            bit    36:34        Protection Control bits used to drive the AHB HPROT[3:1] bus.
 *          FIFO_MODE          bit    33           FIFO Mode Select.
 *          FCMODE             bit    32           Flow Control Mode.
 *          RELOAD_DST         bit    31           Automatic Destination Reload.
 *          RELOAD_SRC         bit    30           Automatic Source Reload.
 *          SRC_HS_POL         bit    19           Source Handshaking Interface Polarity
 *          DST_HS_POL         bit    18           Destination Handshaking Interface Polarity.
 *          HS_SEL_SRC         bit    11           Source Software or Hardware Handshaking Select.
 *          HS_SEL_DST         bit    10           Destination Software or Hardware Handshaking Select.
 *          FIFO_EMPTY         bit    9            Channel FIFO status.
 *          CH_SUSP            bit    8            Channel Suspend.
 *          CH_PRIOR           bit    7:5          Channel Priority.
 *
 */
__STATIC_INLINE uint64_t
XDMA_getChannelCFG(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*return CFG configuration*/
    return ((uint64_t)hwChannel->CFG_H.all << 32) + (uint64_t)hwChannel->CFG_L.all;
}

/**
 * @brief   get channel cfg
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @return  cfg           value of channel cfg
 *
 */
__STATIC_INLINE XDmaCh_Cfg
XDMA_Ch_getCfg(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    XDmaCh_Cfg cfg = {0};

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*get CFG configuration*/
    cfg.cfg_h.all = hwChannel->CTL_H.all;
    cfg.cfg_l.all = hwChannel->CTL_L.all;

    return cfg;
}

/**
 * @brief   Get interrupt status for all channels.
 * This function retrieves the interrupt status for all channels.
 *
 * @param   base base is the base address of the DMA channel control register
 *                 or the base address of the XDMAC control register.
 * @return  Return The interrupt status for all channels.
 *
 */
__STATIC_INLINE uint32_t
XDMA_getChnsInterruptStatus(uint32_t base)
{
    XDMA_CtlSta_TypeDef *hwCtl = NULL;
    uint8_t status = 0U;
    uint32_t dmacBase = 0U;
    uint32_t channel = 0U;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));
    /*Convert the channel base to DMA base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);

    /*Point to the Controller Config register*/
    hwCtl = XDMAC(dmacBase);

    /*Return the interrupt status for the current channel*/
    return (hwCtl->CHNINTS_L.bit.CHN_INTS);
}

/**
 * @brief   get Interrupt Status
 *
 * @param   base base   the base address of the DMA channel control registers.
 *
 * @return  status like bellow
 *          XDMA_INT_TFR
 *          XDMA_INT_BLOCK
 *          XDMA_INT_SRCTRAN
 *          XDMA_INT_DSTTRAN
 *          XDMA_INT_ERR
 */
__STATIC_INLINE uint8_t
XDMA_getInterruptStatus(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    uint8_t status = 0U;
    uint32_t dmacBase = 0U;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));


    /*Covert the channel base to channel number and dmac base address*/
    dmacBase = XDMA_convertChnBase2DmaBase(base);


    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);


    /*Check Tfr interrupt status*/
    if (hwChannel->INTS_L.bit.DMA_TFR_DONE == XDMA_TFR_COMPLETED)
        status |= XDMA_INT_TFR;
    

    /*Check Block interrupt status*/
    if (hwChannel->INTS_L.bit.BLOCK_TFR_DONE == XDMA_TFR_COMPLETED)
        status |= XDMA_INT_BLOCK;

    /*Check SrcTran interrupt status*/
    if(hwChannel->INTS_L.bit.SRC_TRANSCOMP == XDMA_TFR_COMPLETED)
        status |= XDMA_INT_SRCTRAN;

    /*Check DstTran interrupt status*/
    if (hwChannel->INTS_L.bit.DST_TRANSCOMP == XDMA_TFR_COMPLETED)
        status |= XDMA_INT_DSTTRAN;

    /*Check Err interrupt status*/
    if (((hwChannel->INTS_L.all & 0xFFFFFFE0) != 0) || ((hwChannel->INTS_H.all & 0xF) != 0))
        status |= XDMA_INT_ERR;

    return status;

}

/**
 * @brief   mask Interrupt
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   mask   interrupt to disable, as bellow
 *          XDMA_INT_TFR
 *          XDMA_INT_BLOCK
 *          XDMA_INT_SRCTRAN
 *          XDMA_INT_DSTTRAN
 *          XDMA_INT_ERR
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_maskInterrupt(uint32_t base, uint8_t mask)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);


    /*Check Err interrupt mask, if mask contain Err, set mask and enable*/
    if (mask & XDMA_INT_ERR)
    {
        hwChannel->ENINTS_L.all &= 0x1F;  
        hwChannel->ENINTS_H.all = 0x00;
    }

    /*Check Tfr interrupt mask, if mask contain Tfr, set mask and enable*/
    if (mask & XDMA_INT_TFR)
    {
        hwChannel->ENINTS_L.bit.DMA_TFR_DONE_EN = XDMA_DISABLE;
    }

    /*Check Block interrupt mask, if mask contain Block, set mask and enable*/
    if (mask & XDMA_INT_BLOCK)
    {
        hwChannel->ENINTS_L.bit.BLOCK_TFR_DONE_EN = XDMA_DISABLE;    
    }

    /*Check SrcTran interrupt mask, if mask contain SrcTran, set mask and enable*/
    if (mask & XDMA_INT_SRCTRAN)
    {
        hwChannel->ENINTS_L.bit.SRC_TRANSCOMP_EN = XDMA_DISABLE;
    }


    /*Check DstTran interrupt mask, if mask contain DstTran, set mask and enable */
    if (mask & XDMA_INT_DSTTRAN)
    {
        hwChannel->ENINTS_L.bit.DST_TRANSCOMP_EN = XDMA_DISABLE;
    }
 
}

/**
 * @brief   unmask Interrupt
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   mask   interrupt to enable, as bellow
 *          XDMA_INT_TFR
 *          XDMA_INT_BLOCK
 *          XDMA_INT_SRCTRAN
 *          XDMA_INT_DSTTRAN
 *          XDMA_INT_ERR
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_unMaskInterrupt(uint32_t base, uint8_t mask)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /*Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /*Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    hwChannel->ENINTS_L.all = 0x20000000U;      /* un-mask suspend for flag check when call XDMA_stopChannel() */
    hwChannel->ENINTS_H.all = 0x0U;
    /*Check Err interrupt mask, if mask contain Err, set un-mask and enable*/
    if (mask & XDMA_INT_ERR)
    {
        hwChannel->ENINTS_L.all |= 0xFFFFFFE0U;
        hwChannel->ENINTS_H.all |= 0x0FU;
    }    

    /*Check Tfr interrupt mask, if mask contain Tfr, set un-mask and enable*/
    if (mask & XDMA_INT_TFR)
    {
        hwChannel->ENINTS_L.bit.DMA_TFR_DONE_EN = XDMA_ENABLE;
    }

    /*Check Block interrupt mask, if mask contain Block, set un-mask and enable*/
    if (mask & XDMA_INT_BLOCK)
    {
    	/*Enable Block TFR DONE interrupt*/
        hwChannel->CTL_H.bit.IOC_BlkTfr = XDMA_ENABLE;
        hwChannel->ENINTS_L.bit.BLOCK_TFR_DONE_EN = XDMA_ENABLE;
    }

    /*Check SrcTran interrupt mask, if mask contain SrcTran, set un-mask and enable*/
    if (mask & XDMA_INT_SRCTRAN)
    {
        hwChannel->ENINTS_L.bit.SRC_TRANSCOMP_EN = XDMA_ENABLE;
    }
    /*Check DstTran interrupt mask, if mask contain DstTran, set un-mask and enable */
    if (mask & XDMA_INT_DSTTRAN)
    {
        hwChannel->ENINTS_L.bit.DST_TRANSCOMP_EN = XDMA_ENABLE;
    }
}

/**
 * @brief   clear Interrupt
 *
 * @param   base base is the base address of the DMA channel control registers.
 * @param   mask   interrupt to clear, as bellow
 *          XDMA_INT_TFR
 *          XDMA_INT_BLOCK
 *          XDMA_INT_SRCTRAN
 *          XDMA_INT_DSTTRAN
 *          XDMA_INT_ERR
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_clearInterrupt(uint32_t base, uint8_t clear)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    union XCLRINTS_L_REG clrInts_L;
    union XCLRINTS_H_REG clrInts_H;
    
    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Note: for  XCLRINTS is a Write only register, the driver can't access it by bit field directly*/
    clrInts_L.all = 0U;
    clrInts_H.all = 0U;

    /* Check Err interrupt clear mask, if the clear mask contain Err, clear all Err status*/
    if (clear & XDMA_INT_ERR)
    {
        clrInts_L.all = XDMA_CHX_ERR_INTS_L_M;
        clrInts_H.all = XDMA_CHX_ERR_INTS_H_M;
    }

    /*Check Tfr interrupt clear mask, if the clear mask contain Tfr, clear all TFR interrupt status*/
    if (clear & XDMA_INT_TFR)
        clrInts_L.bit.DMA_TFR_DONE_CLR |= XDMA_CLEAR;

    /*  Check Block interrupt clear mask, if the clear mask contain Block, clear all Block interrupt status*/
    if (clear & XDMA_INT_BLOCK)
        clrInts_L.bit.BLOCK_TFR_DONE_CLR |= XDMA_CLEAR;
    /*Check SrcTran interrupt clear mask, if the clear mask contain SrcTran, clear all SrcTran interrupt status*/
    if (clear & XDMA_INT_SRCTRAN)
        clrInts_L.bit.SRC_TRANSCOMP_CLR |= XDMA_CLEAR;

    /*Check DstTran interrupt clear mask, if the clear mask contain DstTran, clear all DstTran status*/
    if (clear & XDMA_INT_DSTTRAN)
        clrInts_L.bit.DST_TRANSCOMP_CLR |= XDMA_CLEAR;

    /* Write back the Ints clear register*/
    hwChannel->CLRINTS_L.all = clrInts_L.all;
    hwChannel->CLRINTS_H.all = clrInts_H.all;
}

/**
 * @brief   Source Software Transaction Request
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_ReqSrcSoftwareTransaction(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /*Set source software transaction request enable and active */
    hwChannel->SWHSSRC_L.bit.SWHS_REQ_SRC_WE = XDMA_ENABLE;
    hwChannel->SWHSSRC_L.bit.SWHS_REQ_SRC = XDMA_ACTIVE;
}

/**
 * @brief   @Destination Software Transaction Request
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_ReqDstSoftwareTransaction(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    /* Set Destination software transactionrequest enable and active */
    hwChannel->SWHSDST_L.bit.SWHS_REQ_DST_WE = XDMA_ENABLE;
    hwChannel->SWHSDST_L.bit.SWHS_REQ_DST = XDMA_ACTIVE;
}

/**
 * @brief   Source Software Sigle Transaction Request
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_ReqSglSrcSoftwareTransaction(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    
    /* Set Source single transaction request enable and active*/
    hwChannel->SWHSSRC_L.bit.SWHS_SGLREQ_SRC_WE = XDMA_ENABLE;
    hwChannel->SWHSSRC_L.bit.SWHS_SGLREQ_SRC = XDMA_ACTIVE;
}

/**
 * @brief   Destination Software Sigle Transaction Request
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_ReqSglDstSoftwareTransaction(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register */
    hwChannel = XDMA_Channel(base);


    /* Set Destination single transaction request enable and active */
    hwChannel->SWHSDST_L.bit.SWHS_SGLREQ_DST_WE = XDMA_ENABLE;
    hwChannel->SWHSDST_L.bit.SWHS_SGLREQ_DST = XDMA_ACTIVE;
}

/**
 * @brief   Last Source Software Transaction Request
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_ReqlstSrcSoftwareTransaction(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);


    /* Set Source Last transaction request enable and active */
    hwChannel->SWHSSRC_L.bit.SWHS_LST_SRC_WE = XDMA_ENABLE;
    hwChannel->SWHSSRC_L.bit.SWHS_LST_SRC = XDMA_ACTIVE;    
}

/**
 * @brief   Last Destination Software Transaction Request
 *
 * @param   base base is the base address of the DMA channel control registers.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_ReqlstDstSoftwareTransaction(uint32_t base)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);


    /* Set Destination Last transaction request enable and active */
    hwChannel->SWHSDST_L.bit.SWHS_LST_DST_WE = XDMA_ENABLE;
    hwChannel->SWHSDST_L.bit.SWHS_LST_DST = XDMA_ACTIVE;
}

/**
 * @brief   Configure ALEN or AWLEN FIFO size.
 *  This function configures to set AR_LEN and AW_LEN FIFO size for XDMA.
 *
 * @param   base base is the base address of the DMA channel control register.
 * @param  transfParams is a pointer to the configuration parameters.
 *		   Refer to struct #DMA_ConfigParams.
 *
 * @return  None
 */
__STATIC_INLINE void
XDMA_setRWLen(uint32_t base, const XDMA_ConfigParams *transfParams)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;
    uint32_t rwLen = transfParams->awLen;


    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);

    if (!transfParams->enableAWfifo)
    {
    	/* Destination Burst Length Enable*/
        hwChannel->CTL_H.bit.AWLEN_EN = XDMA_DISABLE;

        /* Destination Burst Length*/
        hwChannel->CTL_H.bit.AWLEN = 0;

        /* Source Burst Length Enable*/
        hwChannel->CTL_H.bit.ARLEN_EN = XDMA_DISABLE;

        /* Destination Burst Length*/
        hwChannel->CTL_H.bit.ARLEN = 0;
    }
    else
    {
    	/* check the max FIFO length support*/
    	rwLen = (rwLen > 256)? 256 : rwLen;
    	rwLen = (rwLen == 0)? 0:rwLen-1;

    	/* Destination Burst Length Enable*/
        hwChannel->CTL_H.bit.AWLEN_EN = XDMA_ENABLE;

        /* Destination Burst Length*/
        hwChannel->CTL_H.bit.AWLEN = rwLen;

        /* Source Burst Length Enable*/
        hwChannel->CTL_H.bit.ARLEN_EN = XDMA_ENABLE;

        /* Destination Burst Length*/
        hwChannel->CTL_H.bit.ARLEN = rwLen;
    }
}

#if (XDMAC_VERSION == 0x20)
#define ITCM_RAM_BASE           (ITCM_BASE)
#define ITCM_RAM_LENGTH         (0x80000U)
#define DTCM_RAM_BASE           (DTCM_BASE)
#define DTCM_RAM_LENGTH         (0x80000U)
#define ILM_RAM_BASE            (ILM_BASE)
#define ILM_RAM_LENGTH          (0x80000U)
#define DLM_RAM_BASE            (DLM_BASE)
#define DLM_RAM_LENGTH          (0x80000U)

//*****************************************************************************
//
//! Check if the current RAM buffer is in ITCM or DTCM memory space.
//!
//! \param addr is the address of the current RAM buffer.
//! \param len is the length of the current RAM buffer.
//!
//! This function checks whether the current memory buffer is in the ITCM or DTCM
//! memory space. In Device 20, ITCM and DTCM require selecting AXI2 as the 
//! source/Destination Manager for XDMA. Additionally, in Device 20, ITCM or DTCM
//! is always not used d cache. If using other memory space (used cache), 
//! the application needs to flush the cache manually in Device 20.
//! If the RAM buffer is in ITCM or DTCM, it returns true; otherwise, it returns false.
//!
//! \return Returns \b true if in ITCM or DTCM memory space. Returns \b false otherwise.
//
//*****************************************************************************
__STATIC_INLINE bool 
XDMA_checkRamInAxi2Space(uint32_t addr, uint32_t len)
{
    if (((addr >= ITCM_RAM_BASE) && (addr < (ITCM_RAM_BASE + ITCM_RAM_LENGTH))) 
        || ((addr >= DTCM_RAM_BASE) && (addr < (DTCM_RAM_BASE + DTCM_RAM_LENGTH))) 
        || ((addr >= ILM_RAM_BASE) && (addr < (ILM_RAM_BASE + ILM_RAM_LENGTH)))
        || ((addr >= DLM_RAM_BASE) && (addr < (DLM_RAM_BASE + DLM_RAM_LENGTH)))        
       )
    {
        if (((addr + len) < (ITCM_RAM_BASE + ITCM_RAM_LENGTH))
             || ((addr + len) < (DTCM_RAM_BASE + DTCM_RAM_LENGTH))
             || ((addr + len) < (ILM_RAM_BASE + ILM_RAM_LENGTH)) 
             || ((addr + len) < (DLM_RAM_BASE + DLM_RAM_LENGTH))
           )
        {
            return true;
        }
        else
        {
            ASSERT(false);  // Not a valid address and length.
        }
    }
    else
    {
        return false;
    }
}

//*****************************************************************************
//
//! Do extra configuration related to the device and AXI bus structure.
//!
//! \param base is the base address of the DMA channel control register.
//! \param transfParams is a pointer to the configuration parameters.
//!        Refer to struct #DMA_ConfigParams.
//!
//! This function configures SMS and DMS for the current channel.
//! If the current channel transfers data from or to ITCM or DTCM, the channel
//! needs to select AXI2 as Manager.
//! If the current channel transfers data from or to other RAM, the channel
//! needs to select AXI1 as Manager.
//!
//! \return None.
//
//*****************************************************************************
__STATIC_INLINE void 
XDMA_doExtraConfOfChips(uint32_t base, const XDMA_ConfigParams *transfParams)
{
    //
    // Check if the source address is in ITCM/DTCM/ILM/DLM or not.
    // For ITCM/DTCM/ILM/DLM RAM, XDMA should set the SMS to AXI Manager 2, and there is
    // no need to clear the cache.
    // For other RAM, XDMA should set the SMS to AXI Manager 1. If the RAM run in system,
    // the program need to clear cache before start channel.
    //
    if (XDMA_checkRamInAxi2Space(transfParams->srcAddr, 
                transfParams->blockTS * (1U << transfParams->srcTrWidthBytes)))
    {
        //
        //  XDMA needs to set AXI2 as the source manager select when reading from ITCM/DTCM.
        //
        XDMA_setSrcManagerSelect(base, XDMA_MASTER2_INTF);
    }
    else
    {
  
        //
        //  XDMA needs to set AXI1 as the source manager select when reading from other RAM.
        //
        XDMA_setSrcManagerSelect(base, XDMA_MASTER1_INTF);       
    }

    
    //
    // Check if the destination address is in ITCM/DTCM/ILM/DLM or not.
    // For ITCM/DTCM/ILM/DLM RAM, XDMA should set the SMS to AXI Manager 2, and there is
    // no need to clear the cache.
    // For other RAM, XDMA should set the DMS to AXI Manager 1 
    //
    if (XDMA_checkRamInAxi2Space(transfParams->destAddr, 
                transfParams->blockTS * (1U << transfParams->destTrWidthBytes)))
    {
        //
        //  XDMA needs to set AXI2 as the destination manager select when writing to ITCM/DTCM.
        //
    
        XDMA_setDestManagerSelect(base, XDMA_MASTER2_INTF);
    }
    else
    {
    
        //
        //  XDMA needs to set AXI1 as the destination manager select when writing to other RAM.
        //   
        XDMA_setDestManagerSelect(base, XDMA_MASTER1_INTF);
    }
}

#elif (XDMAC_VERSION == 0x22)

#define RAM_BASE                (AXI_SRAM0_BASE)
#define RAM_LENGTH              (0x40000U)
/**
 * @brief Check if the current RAM buffer is in RAM memory space.
 * This function checks whether the current memory buffer is in RAM memory space.
 * For devices 21 or 22, XDMA should enable the features AW_CACHE and AR_CACHE when
 * reading or writing RAM space with cache. XDMA can automatically flush the cache
 * when transferred.
 * If the RAM buffer is in RAM memory space with cache, it returns true;
 * otherwise, it returns false.
 *
 * @param addr addr is the address of the current RAM buffer.
 * @param len len is the length of the current RAM buffer.
 * @return Returns \b true if in RAM memory space with cache. Returns \b false
 * otherwise.
 */
__STATIC_INLINE bool 
XDMA_checkRamSpace(uint32_t addr, uint32_t len)
{
    if ((addr >= RAM_BASE) && (addr < (RAM_BASE + RAM_LENGTH)))
    {
        if ((addr + len) < (RAM_BASE + RAM_LENGTH)) 
            return true;
        else             
            ASSERT(false);  /* Not a valid address and length*/

    }
    else
    {
        return false;
    }
}

/**
 * @brief Do extra configuration related to the device and AXI bus structure.
 * This function configures to enable AR_CACHE and AW_CACHE feature for XDMA.
 * If the current channel transfers data from RAM with cache, the channel needs
 * to enable the AR_CACHE feature.
 * If the current channel transfers data to RAM with cache, the channel needs
 * to enable the AW_CACHE feature.
 *
 * @param base base is the base address of the DMA channel control register.
 * @param transfParams transfParams is a pointer to the configuration parameters.
 * 		  Refer to struct #DMA_ConfigParams.
 * @return None.
 */
__STATIC_INLINE void 
XDMA_doExtraConfOfChips(uint32_t base, const XDMA_ConfigParams *transfParams)
{
    XDMA_Channel_TypeDef *hwChannel = NULL;

    /* Check the arguments*/
    ASSERT(XDMA_isBaseValid(base));    

    /* Point to the Channel register*/
    hwChannel = XDMA_Channel(base);  
    
    /* Handle cache case for RAM from (0x1020 0000 to 0x1024 0000, total 256KB).
     * For RAM in Device 21 and 22, AXI_DMA needs to enable AXI AW_CACHE feature
     * when writing to RAM. For RAM in Device 21 and 22, AXI_DMA needs to enable
     * AXI SR_CACHE feature when reading from RAM.
     * */
    switch(transfParams->ttfc)
    {
        case XDMA_TT_FC_2_P2M_DMAC:
        case XDMA_TT_FC_4_P2M_P:
            if (XDMA_checkRamSpace(transfParams->destAddr, 
                        transfParams->blockTS * (1U << transfParams->destTrWidthBytes)))
            {
                hwChannel->CTL_L.bit.AW_CACHE = 0xF;
            }
            break;

        case XDMA_TT_FC_1_M2P_DMAC:
        case XDMA_TT_FC_6_M2P_P:
            if (XDMA_checkRamSpace(transfParams->srcAddr, 
                        transfParams->blockTS * (1U << transfParams->srcTrWidthBytes)))
            {
                hwChannel->CTL_L.bit.AR_CACHE = 0xF;
            }
            break;

        case XDMA_TT_FC_0_M2M_DMAC:
            if (XDMA_checkRamSpace(transfParams->destAddr, 
                         transfParams->blockTS * (1U << transfParams->destTrWidthBytes)))
            {
                hwChannel->CTL_L.bit.AW_CACHE = 0xF;
            }
            if (XDMA_checkRamSpace(transfParams->srcAddr, 
                        transfParams->blockTS * (1U << transfParams->srcTrWidthBytes)))
            {
                hwChannel->CTL_L.bit.AR_CACHE = 0xF;
            }
            break;

        default:
            break;
    }
}

#else 
__STATIC_INLINE void 
XDMA_doExtraConfOfChips(uint32_t base, const XDMA_ConfigParams *transfParams)
{
    //Not implement yet, and then we will add it in the future
}
#endif


/**
 * @brief Setup DMA to transfer data on the specified channel.
 * This function configures the DMA transfer on the specified channel.
 *
 * @param base is Base address of the DMA channel control register
 * @param *transfParams configuration parameter
 * 						Refer struct #DMA_ConfigParams
 *
 * @return None.
 */
extern void
XDMA_configChannel(uint32_t base, const XDMA_ConfigParams *transfParams);


/**
 * @brief Clear DMA setting from previous ransfer data on the specified channel.
 * This function configures the DMA transfer on the specified channel.
 *
 * @param base is Base address of the DMA channel control register
 * @param *transfParams configuration parameter
 * 						Refer struct #DMA_ConfigParams
 * @return None.
 */
extern void 
XDMA_DeConfChannel(uint32_t base, const XDMA_ConfigParams *transfParams);

/**
 * @brief   config channel parameters
 *
 * @param   base is the base address of the DMA channel control registers.
 * @retval  chParams      the parameters of channel config inclue:
 *                        channel ctl, channel cfg,dmamux ccr and dmamux rgcr
 *
 */
extern void
XDmaChan_config(uint32_t base, const XDmaCh_Parameters *chParams);

/**
 * @brief Configures the DMA channel
 * This function configures the source and destination addresses of a DMA
 * channel. The parameters are pointers to the data to be transferred.
 *
 * @param base is the base address of the DMA channel control registers.
 * @param *destAddr is the destination address.
 * @param *srcAddr is a source address.
 * @return return None.
 */
extern void
XDMA_configAddresses(uint32_t base, uint32_t destAddr, uint32_t srcAddr);

/**
 * @brief Configures the DMA channel's burst settings.
 * This function configures the size of each burst and the address step size.
 *
 * The \e size parameter is the number of words that will be transferred
 * during a single burst. Possible amounts range from 1 word to 32 words.
 *
 * The \e srcStep and \e destStep parameters is not support
 *
 * \note Note that regardless of what data size (configured by
 * DMA_configMode()) is used, parameters are in terms of 16-bits words.
 *		Note that DMA_configBurst must be called before DMA_configTransfer
 *
 * @param base is the base address of the DMA channel control registers.
 * @param size is the number of words transferred per burst.
 * @param srcStep is the amount to increment or decrement the source address
 * after each word of a burst.
 * @param destStep is the amount to increment or decrement the destination
 * address after each word of a burst.
 * @return return None.
 */
extern void XDMA_configBurst(uint32_t base, uint16_t size, int16_t srcStep,
                            int16_t destStep);

/**
 * @brief Configures the DMA channel's transfer settings.
 * This function configures the transfer size and the address step that is
 * made after each burst.
 *
 * The \e transferSize parameter is the number of bursts per transfer. If DMA
 * channel interrupts are enabled, they will occur after this number of bursts
 * have completed. The maximum number of bursts is 65536.
 *
 * The \e srcStep and \e destStep parameters is not support
 *
 * \note Note that regardless of what data size (configured by
 * DMA_configMode()) is used, parameters are in terms of 16-bits words.
 *		 Note that DMA_configTransfer must be called after DMA_configBurst
 *
 * @param base is the base address of the DMA channel control registers.
 * @param transferSize is the number of bursts per transfer.
 * @param srcStep is the amount to increment or decrement the source address
 * after each burst of a transfer unless a wrap occurs.
 * @param destStep is the amount to increment or decrement the destination
 * address after each burst of a transfer unless a wrap occurs.
 *
 * @return return None.
 */
extern void
XDMA_configTransfer(uint32_t base, uint32_t transferSize, int16_t srcStep,
                   int16_t destStep);

/**
 * @brief Configures the DMA channel's wrap settings.
 * This function configures the DMA channel's wrap settings.
 *
 * The \e srcWrapSize and \e destWrapSize parameters are the number of bursts
 * that are to be transferred before their respective addresses are wrapped.
 * The maximum wrap size is 65536 bursts.
 *
 * The \e srcStep and \e destStep parameters specify the address step that
 * that are to be transferred before their respective addresses are wrapped.
 * The maximum wrap size is 65536 bursts.
 *
 * The \e srcStep and \e destStep parameters specify the address step that
 * should be added to the source and destination addresses when the wrap
 * occurs.Only signed values from -4096 to 4095 are valid.
 *
 * \note Note that regardless of what data size (configured by
 * DMA_configMode()) is used, parameters are in terms of 16-bits words.
 *
 * @param base is the base address of the DMA channel control registers.
 * @param srcWrapSize is the number of bursts to be transferred before a wrap
 * of the source address occurs.
 * @param srcStep is the amount to increment or decrement the source address
 * after each burst of a transfer unless a wrap occurs.
 * @param destStep is the amount to increment or decrement the destination
 * address after each burst of a transfer unless a wrap occurs.
 *
 * @return return None.
 */
extern void
XDMA_configWrap(uint32_t base, uint32_t srcWrapSize, int16_t srcStep,
               uint32_t destWrapSize, int16_t destStep);

/**
 * @brief Configures the DMA channel trigger and mode.
 * This function configures the DMA channel's trigger and mode.
 *
 * The \e trigger parameter is the interrupt source that will trigger the
 * start of a DMA transfer.
 *
 * The \e config parameter is the logical OR of the following values:
 * - \b DMA_CFG_ONESHOT_DISABLE or \b DMA_CFG_ONESHOT_ENABLE. If enabled,
 *   the subsequent burst transfers occur without additional event triggers
 *   after the first event trigger. If disabled, only one burst transfer is
 *   performed per event trigger.
 *
 * - \b DMA_CFG_CONTINUOUS_DISABLE or \b DMA_CFG_CONTINUOUS_ENABLE. If enabled
 *  the DMA reinitializes when the transfer count is zero and waits for the
 *  next interrupt event trigger. If disabled, the DMA stops and clears the
 *  run status bit.
 *
 * - \b DMA_CFG_SIZE_16BIT or \b DMA_CFG_SIZE_32BIT. This setting selects
 * whether the databus width is 16 or 32 bits.
 *
 * @param base is the base address of the DMA channel control registers.
 * @param trigger is the interrupt source that triggers a DMA transfer.
 * @param config is a bit field of several configuration selections.
 *
 * @return return None.
 */
extern void
XDMA_configMode(uint32_t base, DMAMUX_TrigId_Type trigger, uint32_t config);


#endif /* #if (XDMAC_VERSION == 0x20 || XDMAC_VERSION == 0x22) */

#ifdef __cplusplus
}
#endif

#endif /* DEVICE_DRIVERLIB_XDMA_H_ */
