/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
 
 /**
*   @file    device.h
*   @brief   
*
*   commit history:
*   2024/04/28, ZhaoLei, rename Device_initDSP
*   2024/04/30, ZhaoLei, add autotest api
*   2025/01/14, ZhaoLei, move LOG_SCI_BASE to board_cfg.h
*/

#ifndef GS32_DRIVER_INIT_DEVICE_H_
#define GS32_DRIVER_INIT_DEVICE_H_

#include "gs32_version.h"

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#if IS_GS32F00xx()
  #include "device_gs32f00xx/gs32f00xx.h"
#elif IS_GS32F3xx()
  #include "device_gs32f3xx/gs32f3xx.h"
#else
  #error Device type must be defined.
#endif
#include "driverlib.h"
#include "system_dsp.h"
#include "board_cfg.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */


/* move clock configuration and MACRO into board_cfg.h */


/******************************************************************************
 *
 * @brief DEVICE_DELAY_US()
 * Macro to call SysCtl_delay() to achieve a delay in microseconds. The macro
 * will convert the desired delay in microseconds to the count value expected
 * by the function. \b x is the number of microseconds to delay.
 *
 ******************************************************************************/

#if IS_GS32F00xx(0x12)
#define DEVICE_DELAY_US(x)   SysCtl_delay((DEVICE_SYSCLK_FREQ/(1000000)) * (x) / 6)
#elif IS_GS32F00xx(0x30)
#define DEVICE_DELAY_US(x)   SysCtl_delay((DEVICE_SYSCLK_FREQ/(1000000)) * (x) / 5)
#elif IS_GS32F3xx()
#define DEVICE_DELAY_US(x)   SysCtl_delay((DEVICE_SYSCLK_FREQ/(1000000)) * (x) / 7)
#endif

#define DELAY_US(x)          DEVICE_DELAY_US(x)

#ifndef LOG_SCI_BASE
#if IS_GS32F00xx()
#define LOG_SCI_BASE         SCIA_BASE
#elif IS_GS32F3xx()
#define LOG_SCI_BASE         SCIB_BASE
#endif
#endif
/* ========================================================================== */
/*    Defines, Globals, and Header Includes related to Flash Support          */
/* ========================================================================== */
#ifdef _FLASH
#include <stddef.h>

extern uint16_t RamfuncsLoadStart;
extern uint16_t RamfuncsLoadEnd;
extern uint16_t RamfuncsLoadSize;
extern uint16_t RamfuncsRunStart;
extern uint16_t RamfuncsRunEnd;
extern uint16_t RamfuncsRunSize;

extern uint16_t isrcodefuncsLoadStart;
extern uint16_t isrcodefuncsLoadEnd;
extern uint16_t isrcodefuncsLoadSize;
extern uint16_t isrcodefuncsRunStart;
extern uint16_t isrcodefuncsRunEnd;
extern uint16_t isrcodefuncsRunSize;

#define DEVICE_FLASH_WAITSTATES 3

#endif

/* ========================================================================== */
/*                         Global Functions Prototypes                        */
/* ========================================================================== */
extern void Device_init(void);
extern void Device_enableAllPeripherals(void);
extern void Device_initGPIO(void);
extern void __error__(const char *filename, uint32_t line);
extern void UartPrint_init(uint32_t SCIBase, uint32_t baud);

#ifdef __RUNNING_AUTOTEST__
extern void testFinished(int32_t exitCode, char *fmt, ...);
#else
#define testFinished(exitCode, fmt, ...)
#endif

#ifdef __cplusplus
}
#endif


#endif /* GS32_DRIVER_INIT_DEVICE_H_ */
//
// End of file
//
