/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex03_temp_sensor.c
 *
 * @Title ADC get temperature sensor.
 *
 * This example sets up the ePWM to periodically trigger the ADC.
 * The ADC converts the internal connection to the temperature sensor,
 * which is then interpreted as a temperature by calling the
 * ADC_getTemperatureC() function.
 *
 * @Watch Variables:
 * - sensorSample - The raw reading from the temperature sensor
 * - sensorTemp   - The interpretation of the sensor sample as
 * 					a temperature in degrees Celsius.
 */

/*
 * Include Files
 */
#include "adc_ex03_board.h"

/*
 * Global Variables
 */
volatile uint16_t sensorSample;
volatile float sensorTemp;

/*
 * Local Function Prototypes
 */
void initEPWM(void);

/* Main */
void main(void)
{
    /* Initialize device clock and peripherals */
    Device_init();

    /*
     * Board Initialization
     * - Configure and power up ADCA.
     * - Configure ADC's SOC0 to be triggered by ePWM1.
     * - Enable ADC interrupt
     */
    Board_init();

    /* Set up the ePWM */
    initEPWM();

    /* Enable Global Interrupt (INTM) and realtime interrupt (DBGM) */
    EINT;


    /* Start ePWM1, enabling SOCA and putting the counter in up-count mode */
    EPWM_enableADCTrigger(EPWM1_BASE, EPWM_SOC_A);
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_UP);
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
	ASysCtl_enableTemperatureSensor();

	/* Loop indefinitely */
    while(1);
}

/* initEPWM - Function to configure ePWM1 to generate the SOC. */
void initEPWM(void)
{
    /* Disable SOCA */
    EPWM_disableADCTrigger(EPWM1_BASE, EPWM_SOC_A);

    /* Configure the SOC to occur on the first up-count event */
    EPWM_setADCTriggerSource(EPWM1_BASE, EPWM_SOC_A, EPWM_SOC_TBCTR_U_CMPA);
    EPWM_setADCTriggerEventPrescale(EPWM1_BASE, EPWM_SOC_A, 1);

    /*
     * Set the compare A value to 1000 and the period to 1999
     * Assuming ePWM clock is 100MHz, this would give 50kHz sampling
     * 50MHz ePWM clock would give 25kHz sampling, etc.
     * The sample rate can also be modulated by changing the ePWM period
     * The sample rate can also be modulated by changing the ePWM period
     */
    EPWM_setCounterCompareValue(EPWM1_BASE, EPWM_COUNTER_COMPARE_A, 1000);
    EPWM_setTimeBasePeriod(EPWM1_BASE, 1999);

    /* Set the local ePWM module clock divider to /1 */
    EPWM_setClockPrescaler(EPWM1_BASE,
                           EPWM_CLOCK_DIVIDER_1,
                           EPWM_HSCLOCK_DIVIDER_1);

    /* Freeze the counter */
    EPWM_setTimeBaseCounterMode(EPWM1_BASE, EPWM_COUNTER_MODE_STOP_FREEZE);

}

/* adcC1ISR - ADC C Interrupt 1 ISR */
void adcC1ISR(void)
{
    /* Read the raw result */
    sensorSample = ADC_readResult(myADC0_RESULT_BASE, ADC_SOC_NUMBER0);

    /* Convert the result to a temperature in degrees C */
    sensorTemp = ADC_getTemperatureC(sensorSample, 3.3f);

    /* Clear the interrupt flag */
    ADC_clearInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1);

    /* Check if overflow has occurred */
    if(true == ADC_getInterruptOverflowStatus(myADC0_BASE, ADC_INT_NUMBER1))
    {
        ADC_clearInterruptOverflowStatus(myADC0_BASE, ADC_INT_NUMBER1);
        ADC_clearInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1);
    }

}

