
/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex4_soc_software_sync.c
 *
 * @Tile ADC synchronous software triggering.
 *
 * This example converts some voltages on ADCA and ADCC using input 5 of the
 * input X-BAR as a software force. Input 5 is triggered by toggling GPIO0,
 * but any spare GPIO could be used. This method will ensure that both ADCs
 * start converting at exactly the same time.
 *
 * @External Connections:
 * - A2, A3, C2, C3 pins should be connected to signals to convert
 *
 * @Watch Variables:
 * - myADC0Result0 : a digital representation of the voltage on pin A2
 * - myADC0Result1 : a digital representation of the voltage on pin A3
 * - myADC1Result0 : a digital representation of the voltage on pin C2
 * - myADC1Result1 : a digital representation of the voltage on pin C3
 *
 */

#ifdef __cplusplus
extern "C"
{
#endif

/*
 * Include Files
 */
#include "adc_ex04_board.h"

/*
 * Global Variables
 */
__SYM_RETAIN uint16_t myADC0Result0;
__SYM_RETAIN uint16_t myADC0Result1;
__SYM_RETAIN uint16_t myADC1Result0;
__SYM_RETAIN uint16_t myADC1Result1;

void main(void)
{
    /* Initialize device clock and peripherals */
    Device_init();

    /*
     * Board Initialization
     * - Write ADC configurations and power up the ADC for both ADC A and ADC C.
     * - Setup GPIO 0 to trigger input XBAR line 5.  GPIO0 is used
     *   as an example, but any spare GPIO could be used. The
     *   physical GPIO pin should be allowed to float, since the
     *   code configures it as an output and changes the value.
     * - Setup ADC acquisition window and compare values.
     */
    Board_init();

    /* Enable global Interrupts*/
    EINT;

    /* Take conversions indefinitely in loop */
    do
    {
        /* Convert, wait for completion, and store results */

        /* Toggle GPIO0 in software.
         * This will cause a trigger to both ADCs via input XBAR, line 5.
         */
    	/* Set pin */
		GPIO_writePin(0U, 1U);
		/* Clear pin */
		GPIO_writePin(0U, 0U);

		/*
		 * Wait for ADCA to complete, then acknowledge the flag.
		 * Since both ADCs are running synchronously,it isn't necessary
		 * to wait for completion notification from both ADCs.
		 */
        while(ADC_getInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1) == 0U);
        ADC_clearInterruptStatus(myADC0_BASE, ADC_INT_NUMBER1);

        /* Store results */
        myADC0Result0 = ADC_readResult(myADC0_RESULT_BASE, ADC_SOC_NUMBER0);
        myADC0Result1 = ADC_readResult(myADC0_RESULT_BASE, ADC_SOC_NUMBER1);
        myADC1Result0 = ADC_readResult(myADC1_RESULT_BASE, ADC_SOC_NUMBER0);
        myADC1Result1 = ADC_readResult(myADC1_RESULT_BASE, ADC_SOC_NUMBER1);

    }
    while(1);
}

#ifdef __cplusplus
}
#endif
