/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex07_ppb_offset_bitfield.c
 *
 * @Title ADC offset adjust using post-processing block
 *
 * ADC offset adjust using post-processing block.
 * Some SOCs have automatic offset adjustment applied by the post-processing
 * block.
 * After the program runs,the memory will contain ADC & post-processing
 * block(PPB) results.
 *
 * @External Connections:
 * - A2, C2 pins should be connected to signals to convert.
 *
 * @Watch Variables
 * - myADC0Result	 - a digital representation of the voltage on pin A2.
 * - myADC0PPBResult - a digital representation of the voltage  on pin A2,
 * 					   minus 100 LSBs of automatically added offset.
 * - myADC1Result 	 - a digital representation of the voltage on pin C2.
 * - myADC1PPBResult - a digital representation of the voltage  on pin C2
 * 					   plus 100 LSBs of automatically added offset
 *
 */

/*
 * Include Files
 */
#include "adc_ex07_board.h"

/*
 * Global Variables
 */
uint16_t myADC0Result;
uint16_t myADC0PPBResult;
uint16_t myADC1Result;
uint16_t myADC1PPBResult;

/* Main */
int main(void)
{
	/*
	 * Initialize device clock and peripherals
	 */
    Device_init();

    /*
     * Board Initialization
     * - Configure the ADCs and power them up
     * - Setup the ADCs for software conversions
     * - Setup PPB offset correction.
     *      conversion on channel A will subtract 100.
     *      conversion on channel C will add 100.
     */
    Board_init();


    /* Enable global Interrupts */
	EINT;

	/* Take conversions indefinitely in loop */
    do
    {
    	/*
    	 * Convert, wait for completion, and store results.
    	 * Start conversions immediately via software, ADCA
    	 */
    	AdcaRegs.ADCSOCFRC1.bit.SOC0 = 1;
    	AdcaRegs.ADCSOCFRC1.bit.SOC1 = 1;

        /*
         * Wait for ADCA to complete, then acknowledge flag
         */
    	while(AdcaRegs.ADCINTFLG.bit.ADCINT1 == false)
    	{

        }
    	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

        /*
         * Start conversions immediately via software, ADCC
         */
    	AdccRegs.ADCSOCFRC1.bit.SOC0 = 1;
    	AdccRegs.ADCSOCFRC1.bit.SOC1 = 1;

        /*
         * Wait for ADCC to complete, then acknowledge flag
         */
    	while(AdccRegs.ADCINTFLG.bit.ADCINT1 == false)
    	{

        }
    	AdccRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

        /*
         * Store results
         */
    	myADC0Result = AdcaResultRegs.ADCRESULT0;
    	myADC0PPBResult = AdcaResultRegs.ADCPPB1RESULT.bit.RESULT;
    	myADC1Result = AdccResultRegs.ADCRESULT0;
    	myADC1PPBResult = AdccResultRegs.ADCPPB1RESULT.bit.RESULT;

    }
    while(1);
}
