/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
/*
 * @File adc_ex14_ppb_pwm_trip_bitfield.c
 *
 * @Title ADC limits check and PWM trip in case of out of bound input.
 *
 * This example demonstrates EPWM tripping through ADC limit detection PPB  block.
 * ADCAINT1 is configured to periodically trigger the ADCA channel 2 post initial
 * software forced trigger.
 * The limit detection post-processing block(PPB) is configured and if the ADC
 * results are outside of the defined range, the post-processing block will
 * generate an ADCxEVTy event.
 * This event is configured as EPWM trip source through configuring EPWM
 * XBAR and corresponding EPWM's trip zone and digital compare sub-modules.
 *   The example showcases
 *    - one-shot
 *    - cycle-by-cycle
 *    - and direct tripping of PWMs
 *   through ADCAEVT1 source via Digital compare submodule.
 *
 *   The default limits are 0LSBs and 3600LSBs. With VREFHI set to 3.3V, the
 *   PPB will generate a trip event if the input voltage goes above about
 *   2.9V.
 *
 * @External Connections:
 * - A2 should be connected to a signal to convert.
 * - Observe the following signals on an oscilloscope
 * 	- ePWM1(GPIO0 - GPIO1)
 * 	- ePWM2(GPIO2 - GPIO3)
 * 	- ePWM3(GPIO4 - GPIO5)
 *
 * @Watch Variables
 *  - adcA2Results - digital representation of the voltage on pin A2.
 *
 */

/*
 * Include Files
 */
#include "device.h"

/*
 * Macros & Typedefs
 */
/*
 * EPWM1 -> myEPWM1 Pinmux
 */
/*
 * EPWM1_A - GPIO Settings
 */
#define GPIO_PIN_EPWM1_A 0
#define myEPWM1_EPWMA_GPIO 0
#define myEPWM1_EPWMA_PIN_CONFIG GPIO_0_EPWM1_A

/*
 * EPWM1_B - GPIO Settings
 */
#define GPIO_PIN_EPWM1_B 1
#define myEPWM1_EPWMB_GPIO 1
#define myEPWM1_EPWMB_PIN_CONFIG GPIO_1_EPWM1_B

/*
 * EPWM2 -> myEPWM2 Pinmux
 */
/*
 * EPWM2_A - GPIO Settings
 */
#define GPIO_PIN_EPWM2_A 2
#define myEPWM2_EPWMA_GPIO 2
#define myEPWM2_EPWMA_PIN_CONFIG GPIO_2_EPWM2_A
/*
 * EPWM2_B - GPIO Settings
 */
#define GPIO_PIN_EPWM2_B 3
#define myEPWM2_EPWMB_GPIO 3
#define myEPWM2_EPWMB_PIN_CONFIG GPIO_3_EPWM2_B

/*
 * EPWM3 -> myEPWM3 Pinmux
 */
/*
 * EPWM3_A - GPIO Settings
 */
#define GPIO_PIN_EPWM3_A 4
#define myEPWM3_EPWMA_GPIO 4
#define myEPWM3_EPWMA_PIN_CONFIG GPIO_4_EPWM3_A
/*
 * EPWM3_B - GPIO Settings
 */
#define GPIO_PIN_EPWM3_B 5
#define myEPWM3_EPWMB_GPIO 5
#define myEPWM3_EPWMB_PIN_CONFIG GPIO_5_EPWM3_B

#define RESULTS_BUFFER_SIZE 512U

/*
 * Global Constants
 */
/* Index into result buffer */
volatile uint16_t indexA = 0;
/* Flag to indicate buffer is full */
volatile uint16_t bufferFull;
/* ADC result buffer */
uint16_t adcA2Results[RESULTS_BUFFER_SIZE];

/*
 * Characteristics of PWMs to be tripped
 * EPWM1 - frequency -> 10kHz, dutyA -> 50%, dutyB -> 50%, ePWM1B - inverted
 * EPWM2 - frequency -> 2.5kHz, dutyA -> 20%, dutyB -> 50%, ePWM2B - inverted
 * EPWM3 - frequency -> 5kHz, dutyA -> 70%, dutyB -> 50%, ePWM3B - inverted
 */
EPWM_SignalParams pwmSignal1 =
    {10000, 0.5f, 0.5f, true, DEVICE_SYSCLK_FREQ,
     EPWM_COUNTER_MODE_UP_DOWN, EPWM_CLOCK_DIVIDER_1,
     EPWM_HSCLOCK_DIVIDER_1};

EPWM_SignalParams pwmSignal2 =
    {25000, 0.2f, 0.5f, true, DEVICE_SYSCLK_FREQ,
     EPWM_COUNTER_MODE_UP_DOWN, EPWM_CLOCK_DIVIDER_1,
     EPWM_HSCLOCK_DIVIDER_1};

EPWM_SignalParams pwmSignal3 =
    {5000, 0.7f, 0.5f, true, DEVICE_SYSCLK_FREQ,
     EPWM_COUNTER_MODE_UP_DOWN, EPWM_CLOCK_DIVIDER_1,
     EPWM_HSCLOCK_DIVIDER_1};


/*
 * Local Function Prototypes
 */
/*
 * ADC configuration related APIs
 */
void configureADC(uint32_t adcBase);
void configureADCSOC(uint32_t adcBase, uint16_t channel);
void configureLimitDetectionPPB(uint16_t soc, uint16_t limitHigh,
                                uint16_t limitLow);
void ADCA_1INT_IRQHandler();

/*
 * EPWM configuration related APIs
 */
void configureOSHTripSignal(uint32_t epwmBase);
void configureCBCTripSignal(uint32_t epwmBase);
void configureDirectTripSignal(uint32_t epwmBase);
void initEPWMGPIO(void);

/* Main */
int main(void)
{
	/* Initialize device clock and peripherals */
    Device_init();
#if IS_GS32F3xx(0x22)
    /* bypass dll */
    EPWM_setHrpwmDllCfg0(PREEPWM_BASE,0x1);
#endif

    /* For this case just init GPIO pins for ePWM1, ePWM2, ePWM3 */
    initEPWMGPIO();

    /* Enable internal reference on ADCs */
    ADC_setVREF(ADCA_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCB_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);
    ADC_setVREF(ADCC_BASE, ADC_REFERENCE_INTERNAL, ADC_REFERENCE_3_3V);

    /* Configure the ADC and power it up */
    configureADC(ADCA_BASE);

    /* Configuring ePWM modules for desired frequency and duty */
    EPWM_configureSignal(EPWM1_BASE, &pwmSignal1);
    EPWM_configureSignal(EPWM2_BASE, &pwmSignal2);
    EPWM_configureSignal(EPWM3_BASE, &pwmSignal3);

    /* Configure ADCEVTx as one-shot trip signal for ePWM1 */
    configureOSHTripSignal(EPWM1_BASE);

    /* Configure ADCEVTx as cycle-by-cycle trip signal for ePWM2 */
    configureCBCTripSignal(EPWM2_BASE);

    /* Configure ADCEVTx as direct trip signal for ePWM3 */
    configureDirectTripSignal(EPWM3_BASE);

    /* Setup the ADC for ADCAINT1 triggered conversions on channel 2 */
    configureADCSOC(ADCA_BASE, 2U);

    /*
     * Configure ADC post-processing limits
     * SOC0 will generate an interrupt if conversion is above or below limits
     */
    configureLimitDetectionPPB(0U, 3600U, 1000U);

    /* Enable ADC interrupt */
    Interrupt_register(INT_ADCA1,ADCA_1INT_IRQHandler);
    Interrupt_enable(INT_ADCA1);

    /* Enable global Interrupts */
    EINT;  // Enable Global interrupt INTM

    /*
     * Start ePWM:
     * Enable sync and clock to PWM
     */
    SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

    /* Trigger ADC once through software */
    AdcaRegs.ADCSOCFRC1.bit.SOC0 = 1;

    /* IDLE loop. Just sit and loop forever (optional): */
    do
    {

    }
    while(1);
}

void ADCA_1INT_IRQHandler()
{
    /* Add the latest result to the buffer */
    adcA2Results[indexA++] = AdcaResultRegs.ADCRESULT0;

    /* Set the bufferFull flag if the buffer is full */
    if (RESULTS_BUFFER_SIZE <= indexA)
    {
        indexA = 0;
    }

    /* Clear the interrupt flag */
    AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;

    /* Check if overflow has occurred */
    if(true == AdcaRegs.ADCINTOVF.bit.ADCINT1)
    {
    	AdcaRegs.ADCINTOVFCLR.bit.ADCINT1 = 1;
    	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
    }
}

/*
 * configureOSHTripSignal - Configure ADCAEVT1 as one-shot trip signal for
 *                          desired PWM instance
 */
void configureOSHTripSignal(uint32_t epwmBase)
{
	/* Configure trip 7 input to be triggered by ADCAEVT1 */
	XBAR_setEPWMMuxConfig(XBAR_TRIP7, XBAR_EPWM_MUX00_ADCAEVT1);

    /* Enable mux 0 */
    XBAR_enableEPWMMux(XBAR_TRIP7, XBAR_MUX00);

    /* Select Trip 7 input as input to DC module */
    EPWM_selectDigitalCompareTripInput(epwmBase, EPWM_DC_TRIP_TRIPIN7,
                                       EPWM_DC_TYPE_DCAH);

    /* DCAEVT1 is generated when DCAH is high. */
    EPWM_setTripZoneDigitalCompareEventCondition(epwmBase, EPWM_TZ_DC_OUTPUT_A1,
                                                 EPWM_TZ_EVENT_DCXH_HIGH);

    /* DCAEVT1 uses the unfiltered version of DCAEVT1 */
    EPWM_setDigitalCompareEventSource(epwmBase, EPWM_DC_MODULE_A,
                                      EPWM_DC_EVENT_1, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);

    /* DCAEVT1 is asynchronous */
    EPWM_setDigitalCompareEventSyncMode(epwmBase, EPWM_DC_MODULE_A,
                                        EPWM_DC_EVENT_1, EPWM_DC_EVENT_INPUT_NOT_SYNCED);

    /* Enable Trip zone signals */
    EPWM_enableTripZoneSignals(epwmBase, EPWM_TZ_SIGNAL_DCAEVT1);

    /*
     * Action on DCAEVT1
     * Force the EPWMxA LOW and EPWMxB HIGH
     */
    EPWM_setTripZoneAction(epwmBase,
                           EPWM_TZ_ACTION_EVENT_TZA,
                           EPWM_TZ_ACTION_LOW);

    EPWM_setTripZoneAction(epwmBase,
                           EPWM_TZ_ACTION_EVENT_TZB,
                           EPWM_TZ_ACTION_HIGH);

    /* Clear any spurious trip zone flags before enabling the interrupt */
    EPWM_clearTripZoneFlag(epwmBase, EPWM_TZ_FLAG_DCAEVT1);

    EPWM_clearOneShotTripZoneFlag(epwmBase, EPWM_TZ_OST_FLAG_DCAEVT1);

    /* Enable TZ interrupt */
    EPWM_enableTripZoneInterrupt(epwmBase, EPWM_TZ_INTERRUPT_OST);
}

/*
 * configureCBCT/*pSignal - Configure ADCAEVT1 as cycle-by-cycle trip signal
 *                          for desired PWM instance
 */
void configureCBCTripSignal(uint32_t epwmBase)
{
    /* Configure trip 8 input to be triggered by ADCAEVT1 */
    XBAR_setEPWMMuxConfig(XBAR_TRIP8, XBAR_EPWM_MUX00_ADCAEVT1);

    /* Enable mux 0 */
    XBAR_enableEPWMMux(XBAR_TRIP8, XBAR_MUX00);

    /* Select Trip 8 input as input to DC module */
    EPWM_selectDigitalCompareTripInput(epwmBase, EPWM_DC_TRIP_TRIPIN8,
                                       EPWM_DC_TYPE_DCAH);

    /* DCAEVT2 is generated when DCAH is high. */
    EPWM_setTripZoneDigitalCompareEventCondition(epwmBase, EPWM_TZ_DC_OUTPUT_A2,
                                                 EPWM_TZ_EVENT_DCXH_HIGH);

    /* DCAEVT2 uses the unfiltered version of DCAEVT2 */
    EPWM_setDigitalCompareEventSource(epwmBase, EPWM_DC_MODULE_A,
                                      EPWM_DC_EVENT_2, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);

    /* DCAEVT2 is asynchronous */
    EPWM_setDigitalCompareEventSyncMode(epwmBase, EPWM_DC_MODULE_A,
                                        EPWM_DC_EVENT_2, EPWM_DC_EVENT_INPUT_NOT_SYNCED);

    /* Enable Trip zone signals */
    EPWM_enableTripZoneSignals(epwmBase, EPWM_TZ_SIGNAL_DCAEVT2);

    /* Configure CBC clear event */
    EPWM_selectCycleByCycleTripZoneClearEvent(epwmBase,
                                              EPWM_TZ_CBC_PULSE_CLR_CNTR_ZERO_PERIOD);
    /*
     * Action on DCAEVT2
     * Force the EPWMxA/EPWMxB LOW
     */
    EPWM_setTripZoneAction(epwmBase, EPWM_TZ_ACTION_EVENT_TZA,
                           EPWM_TZ_ACTION_LOW);

    EPWM_setTripZoneAction(epwmBase, EPWM_TZ_ACTION_EVENT_TZB,
                           EPWM_TZ_ACTION_HIGH);

    /* Clear any spurious trip zone flags before enabling the interrupt */
    EPWM_clearTripZoneFlag(epwmBase, EPWM_TZ_FLAG_DCAEVT2);

    EPWM_clearCycleByCycleTripZoneFlag(epwmBase, EPWM_TZ_CBC_FLAG_DCAEVT2);

    /*enable TZ interrupt */
    EPWM_enableTripZoneInterrupt(epwmBase, EPWM_TZ_INTERRUPT_CBC);
}

/*
/* configureDirectTripSignal - Configure ADCAEVT1 as direct trip signal for
 *                             desired PWM instance
 */
void configureDirectTripSignal(uint32_t epwmBase)
{
    /* Configure trip 9 input to be triggered by ADCAEVT1 */
    XBAR_setEPWMMuxConfig(XBAR_TRIP9, XBAR_EPWM_MUX00_ADCAEVT1);

    /* Enable mux 0 */
    XBAR_enableEPWMMux(XBAR_TRIP9, XBAR_MUX00);

    /* Selec/*Trip 9 input as input to DC module */
    EPWM_selectDigitalCompareTripInput(epwmBase, EPWM_DC_TRIP_TRIPIN9,
                                       EPWM_DC_TYPE_DCAH);
    EPWM_selectDigitalCompareTripInput(epwmBase, EPWM_DC_TRIP_TRIPIN9,
                                       EPWM_DC_TYPE_DCBH);

    /* DCAEVT1 is generated when DCAH is high. DCBEVT2 is generated when DCBH
     * is high
     */
    EPWM_setTripZoneDigitalCompareEventCondition(epwmBase, EPWM_TZ_DC_OUTPUT_A1,
                                                 EPWM_TZ_EVENT_DCXH_HIGH);
    EPWM_setTripZoneDigitalCompareEventCondition(epwmBase, EPWM_TZ_DC_OUTPUT_B2,
                                                 EPWM_TZ_EVENT_DCXH_HIGH);

    /* DCAEVT1/DCBEVT2 uses the unfiltered version of DCAEVT1/DCBEVT2 */
    EPWM_setDigitalCompareEventSource(epwmBase, EPWM_DC_MODULE_A,
                                      EPWM_DC_EVENT_1, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);
    EPWM_setDigitalCompareEventSource(epwmBase, EPWM_DC_MODULE_B,
                                      EPWM_DC_EVENT_2, EPWM_DC_EVENT_SOURCE_ORIG_SIGNAL);

    /* DCAEVT1/ DCBEVT2 is asynchronous */
    EPWM_setDigitalCompareEventSyncMode(epwmBase, EPWM_DC_MODULE_A,
                                        EPWM_DC_EVENT_1, EPWM_DC_EVENT_INPUT_NOT_SYNCED);
    EPWM_setDigitalCompareEventSyncMode(epwmBase, EPWM_DC_MODULE_B,
                                        EPWM_DC_EVENT_2, EPWM_DC_EVENT_INPUT_NOT_SYNCED);

    /* Action on TZA/TZB
     * Force the EPWMxA LOW and EPWMxB HIGH
     */
    EPWM_setTripZoneAction(epwmBase,
                           EPWM_TZ_ACTION_EVENT_DCAEVT1,
                           EPWM_TZ_ACTION_LOW);

    EPWM_setTripZoneAction(epwmBase,
                           EPWM_TZ_ACTION_EVENT_DCBEVT2,
                           EPWM_TZ_ACTION_HIGH);

    /* Clear any spurious trip zone flags before enabling the interrupt */
    EPWM_clearTripZoneFlag(epwmBase, (EPWM_TZ_INTERRUPT_DCAEVT1 |
                                      EPWM_TZ_INTERRUPT_DCBEVT2));

    /* Enable TZ interrupt */
    EPWM_enableTripZoneInterrupt(epwmBase, (EPWM_TZ_INTERRUPT_DCAEVT1 |
                                            EPWM_TZ_INTERRUPT_DCBEVT2));
}

/*
/* configureADC - Write ADC configurations and power up the ADC for both
 *                ADC A and ADC B
 */
void configureADC(uint32_t adcBase)
{
    /* Set ADCCLK divider to /4 */
    AdcaRegs.ADCCTL2.bit.PRESCALE = 6;

    /* Set pulse positions to late */
    AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1;

    /* Power up the ADCs and then delay for 1 ms */
    AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1;

    /* Delay for 1ms to allow ADC time to power up */
    DEVICE_DELAY_US(1000);
}

/*
/* configureLimitDetectionPPB - Configure high and low limits for ADCPPB
 */
void configureLimitDetectionPPB(uint16_t soc, uint16_t limitHigh,
                                uint16_t limitLow)
{
    /* Associate PPB1 with soc */
	AdcaRegs.ADCPPB1CONFIG.bit.CONFIG = soc;

    /* Set high and low limits */
	AdcaRegs.ADCPPB1TRIPLO.bit.LIMITLO = limitLow;
	AdcaRegs.ADCPPB1TRIPHI = limitHigh;

    /* Enable high and low limit PPB events */
	AdcaRegs.ADCEVTSEL.bit.PPB1TRIPHI = 0;
	AdcaRegs.ADCEVTSEL.bit.PPB1TRIPLO = 0;

    /* Configure CBCEN */
    /*if the event condition is no longer present automatically clear the ADCEVTSTAT*/
	AdcaRegs.ADCPPB1CONFIG.bit.CBCEN = 1;
}

/*
 * configureADCSOC - Setup ADC EPWM channel and trigger settings
 */
void configureADCSOC(uint32_t adcBase, uint16_t channel)
{
    uint16_t acqps;

    /* Determine minimum acquisition window (in SYSCLKS) */
    /* 8 SYSCLKS */
    acqps = 8;
    /*
     * - NOTE: A longer sampling window will be required if the ADC driving
     *   source is less than ideal (an ideal source would be a high bandwidth
     *   op-amp with a small series resistance). See TI application report
     *   SPRACT6 for guidance on ADC driver design.
     */

    /* Select the channels to convert and end of conversion flag */
    /* ADCA */
	AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = 0;
	AdcaRegs.ADCSOC0CTL.bit.CHSEL = channel;
	AdcaRegs.ADCSOC0CTL.bit.ACQPS = acqps;

    /* Configure ADCINT1 as SOC0 trigger */
	AdcaRegs.ADCINTSOCSEL1.bit.SOC0 = 1;

    /* Enable continuous mode */
	AdcaRegs.ADCINTSEL1N2.bit.INT1CONT = 1;

    /* Configure source as EOC1, clear & enable the interrupt */
	AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = 0;
	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1;
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
}

/*
 * initEPWMGPIO - Configure ePWM1-ePWM3 GPIO
 */
void initEPWMGPIO(void)
{
	/* PinMux for modules assigned to CPU1 */

	/* EPWM1 -> myEPWM1 Pinmux */
	GPIO_setPinConfig(myEPWM1_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM1_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM1_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(myEPWM1_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM1_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM1_EPWMB_GPIO, GPIO_QUAL_SYNC);

	/* EPWM2 -> myEPWM2 Pinmux */
	GPIO_setPinConfig(myEPWM2_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM2_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM2_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(myEPWM2_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM2_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM2_EPWMB_GPIO, GPIO_QUAL_SYNC);

	/* EPWM3 -> myEPWM3 Pinmux */
	GPIO_setPinConfig(myEPWM3_EPWMA_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM3_EPWMA_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM3_EPWMA_GPIO, GPIO_QUAL_SYNC);

	GPIO_setPinConfig(myEPWM3_EPWMB_PIN_CONFIG);
	GPIO_setPadConfig(myEPWM3_EPWMB_GPIO, GPIO_PIN_TYPE_STD);
	GPIO_setQualificationMode(myEPWM3_EPWMB_GPIO, GPIO_QUAL_SYNC);

}

/*END OF FILE*/
