/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#ifdef __cplusplus
extern "C"{
#endif

/*
 * Include Files
 */
#include "adc_ex16_board.h"

/*
 * @brief Board Configurations
 *
 * Initializes the rest of the modules.
 * Call this function in your application if you wish to do all module
 * initialization.
 * If you wish to not use some of the initializations, instead of the
 * Board_init use the individual Module_inits.
 */
void Board_init()
{
	ADC_init();
	INTERRUPT_init();

}

/*
 * ADC Configurations
 */
void ADC_init()
{
	myADC0_init();
}

void myADC0_init(){

	/*
	 * @brief ADC Initialization:
	 * 		  Write ADC configurations and power up the ADC.
	 */
	/* Select ADC Reference mode and Reference voltage */
	ADC_setVREF(myADC0_BASE,ADC_REFERENCE_INTERNAL,ADC_REFERENCE_3_3V);

	/* Set trim */
	AdcaRegs.ADCINLTRIM1 = ADC_TRIM_VALUE;

	/* Configures the analog-to-digital converter module prescaler. */
	AdcaRegs.ADCCTL2.bit.PRESCALE = 6;

	/* Sets the timing of the end-of-conversion pulse */
	AdcaRegs.ADCCTL1.bit.INTPULSEPOS = 1;

	/* Powers up the ADCA core. */
	AdcaRegs.ADCCTL1.bit.ADCPWDNZ = 1;

	/* Delay for 1ms to allow ADCA time to power up */
	DEVICE_DELAY_US(5000);

	/*
	 * @brief SOC Configuration:
	 * 		  Setup ADC EPWM channel and trigger settings.
	 */
	/* Disables SOC burst mode. */
	AdcaRegs.ADCBURSTCTL.bit.BURST_EN = 0;

	/* Sets the priority mode of the SOCs. */
	AdcaRegs.ADCSOCPRICTL.bit.SOCPRIORITY = 0;

	/*
	 * Start of Conversion 0 Configuration
	 *
	 *
	 * Configures Repeater Modules
	 *
	 *
	 * 		Repeater module 1 parameters
	 *      base: 			myADC0_BASE
	 * 		RepInstance: 	ADC_TRIGGER_REPEATER1
	 *      ADC_RepeaterConfig : {
	 *      mode			: ADC_REPMODE_OVERSAMPLING
	 * 	  	Trigger			: ADC_TRIGGER_EPWM1_SOCA
	 *	  	SyncInput		: ADC_SYNCIN_DISABLE
	 *	  	Count			: 3
	 *	  	phase			: 0
	 *	  	spread			: 0
	 */
	AdcaRegs.REP1CTL.bit.TRIGGER = 5;
	AdcaRegs.REP1CTL.bit.MODE = 0;
	AdcaRegs.REP1CTL.bit.SYNCINSEL = 0;
	AdcaRegs.REP1N.bit.NSEL = 3;
	AdcaRegs.REP1PHASE.bit.PHASE = 0;
	AdcaRegs.REP1SPREAD.bit.SPREAD = 0;

	/*
	 * @brief Start of Conversion 0 Configuration.
	 *
	 * Configures a SOC in the ADC and its interrupt SOC trigger.
	 * 			SOC number		: 0
	 * 			Trigger			: ADC_TRIGGER_REPEATER1
	 * 			Channel			: ADC_CH_ADCIN0
	 * 			Sample Window	: 15 SYSCLK cycles
	 * 			Interrupt Trigger: ADC_INT_SOC_TRIGGER_NONE
	 */
	AdcaRegs.ADCSOC0CTL.bit.TRIGSEL = 40;
	AdcaRegs.ADCSOC0CTL.bit.CHSEL = 0;
	AdcaRegs.ADCSOC0CTL.bit.ACQPS = 50U;
	AdcaRegs.ADCINTSOCSEL1.bit.SOC0 = 0;

	/*
	 * @brief ADC Interrupt Initialization:
	 *
	 * ADC Interrupt 1 Configuration
	 * 		Source	: ADC_INT_TRIGGER_EOC0
	 * 		Interrupt Source: enabled
	 * 		Continuous Mode	: disabled
	 */
	AdcaRegs.ADCINTSEL1N2.bit.INT1SEL = 0;
	AdcaRegs.ADCINTSEL1N2.bit.INT1E = 1;
	AdcaRegs.ADCINTFLGCLR.bit.ADCINT1 = 1;
	AdcaRegs.ADCINTSEL1N2.bit.INT1CONT = 0;

	/*
	 * PPB Configuration: Configure high and low limits detection for ADCPPB
	 *
	 * Post Processing Block 1 Configuration
	 * 		Configures a post-processing block (PPB) in the ADC.
	 * 		PPB Number				: 1
	 * 		SOC/EOC number			: 0
	 * 		Calibration Offset		: 0
	 * 		Reference Offset		: 0
	 * 		Two's Complement		: Disabled
	 * 		Trip High Limit			: 0
	 * 		Trip Low Limit			: 0
	 * 		Clear PPB Event Flags	: Disabled
	 * 		Accumulation Limit		: 1
	 * 		SyncInput Source		: ADC_SYNCIN_DISABLE
	 * 		Comparator Source		: ADC_PPB_COMPSOURCE_RESULT
	 * 		Right Shift				: 0
	 * 		Absolute value				: false
	 */
	AdcaRegs.ADCPPB1CONFIG.bit.CONFIG = 0;
	AdcaRegs.ADCEVTSEL.bit.PPB1TRIPHI = 0;
	AdcaRegs.ADCEVTSEL.bit.PPB1TRIPLO = 0;
	AdcaRegs.ADCEVTSEL.bit.PPB1ZERO = 0;
	AdcaRegs.ADCEVTINTSEL.bit.PPB1TRIPHI = 0;
	AdcaRegs.ADCEVTINTSEL.bit.PPB1TRIPLO = 0;
	AdcaRegs.ADCEVTINTSEL.bit.PPB1ZERO = 0;
	AdcaRegs.ADCPPB1OFFCAL.bit.OFFCAL = 100;
	AdcaRegs.ADCPPB1OFFREF = 0;
	AdcaRegs.ADCPPB1CONFIG.bit.TWOSCOMPEN = 0;
	AdcaRegs.ADCPPB1TRIPLO.bit.LIMITLO = 0;
	AdcaRegs.ADCPPB1TRIPHI = 0;
	AdcaRegs.ADCPPB1CONFIG.bit.CBCEN = 0;
	AdcaRegs.ADCPPB1CONFIG2.bit.COMPSEL = 0;
	AdcaRegs.ADCPPB1CONFIG2.bit.SHIFT = 0;
	AdcaRegs.ADCPPB1CONFIG.bit.ABSEN = 0;
}

/*
 * INTERRUPT Configurations
 */
void INTERRUPT_init()
{
	/* Interrupt Setings for INT_myADC0_1 */
	Interrupt_register(INT_ADCA1, &adcA1ISR);
	Interrupt_enable(INT_ADCA1);
}

#ifdef __cplusplus

#endif

