/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
* @file    can_ex2_polling_and_int.c
* @brief
* @details
*  1) Use HAL_CAN driver to handle the CAN message with virtual MailBox
*     This is also called Message Object.
*  2) Setup mailbox_1/3/5 as Tx type, CAN ID is 0x15555505, 0x15555506, 0x15555507;
*     Setup mailbox_2/4/6 as Rx type, acceptance CAN ID is 0x15555525, 0x15555526, 0x15555527;
*     Setup mailbox_8 as Rx type, acceptance CAN ID is 0x15555528, mask is 0x00000000, means accept all IDs
*  3) Send message objects in the mailbox_1/3/5;
*     Receive message objects through mailbox_2/4/6;
*  4) Call the specified API to update information from CAN registers/RxFIFO to HAL_CAN mailboxes.
*  5) For CAN message transmission, just call HAL_CAN_sendMessage;
*     For CAN message reception, need call HAL_CAN_updateRegsAndMB and then call HAL_CAN_readMessage.
*  6) User can set the MACRO "HAL_CAN_POLLING_MODE" to select Polling mode or Interrupt mode
*     Polling Mode: disable CAN Tx/Rx interrupt and call the API "HAL_CAN_updateRegsAndMB" in background task or in periodic task;
*     Interupt Mode: enable CAN Tx/Rx interrupt and call the API "HAL_CAN_updateRegsAndMB" in CAN Tx/Rx ISR.
*  7) Note: the poll interval is set by the last line calling DEVICE_DELAY_US();
*  8) Use external device to send CAN messages with ID=0x15555525~0x15555529;
*     The variable can_rx_received_cnt[1/3/5/7] indicates the received messages by mailbox_2/4/6/8.
*
*  Note:
*  Max 16 HW Rx filters IDs are available in CAN module
*  User can setup more Rx Mailbox than 16, but need handle the ID filter separately .*
*/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "hal_can_ex2_polling_and_int.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/* Use MACRO to replace the API function names. */
#define CAN_setupMessageObject		HAL_CAN_setupMessageObject
#define CAN_sendMessage				HAL_CAN_sendMessage_8bit

/*
 * 1 - select Polling Mode
 * 0 - select Interrupt Mode
 */
#define HAL_CAN_POLLING_MODE   0

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
uint8_t tx_buff[8]={0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88};
uint8_t CanaRxData[4][8];

volatile uint32_t dlog_can_runtime[4];

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* CAN Tx failure count for each Tx mailbox */
volatile uint32_t can_tx_fail_cnt[8];
volatile uint32_t can_tx_status;
/* CAN received messages count for each Rx mailbox */
volatile uint32_t can_rx_received_cnt[8];
volatile uint32_t can_rx_status;

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief    Setup some mailboxes for Tx/Rx messages
 *
 * @details  Setup mailbox_1/3/5 as Tx type,
 *                 CAN ID is 0x15555505, 0x15555506, 0x15555507;
*            Setup mailbox_2/4/6 as Rx type,
*                  acceptance CAN ID is 0x15555525, 0x15555526, 0x15555527;
*            Setup mailbox_8 as Rx type,
*                  acceptance CAN ID is 0x15555528, mask is 0x00000000, means accept all IDs
 */
static void CANA_initMessageObjects(void)
{
	/* mailbox_1, Tx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_1, 0x15555505, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_2, Rx, filter+mask enable, interrupt enable
	and Rx message filter configuration is also included in the following API. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_2, 0x15555525, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_3, Tx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_3, 0x15555506, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_4, Rx, filter+mask enable, interrupt enable
	and Rx message ID filter configuration is also included in the following API. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_4, 0x15555526, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_5, Tx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_5, 0x15555507, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_6, Rx, filter+mask enable, interrupt enable
	and Rx message filter configuration is also included in the following API. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_6, 0x15555527, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_8, Rx, filter+mask enable, interrupt enable
	and Rx message filter configuration is also included in the following API.
	Mask is 0x00000000, means all bits don't care, this mailbox accepts all messages IDs. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_8, 0x15555528, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x00000000,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);
}

/*
 * @brief    save received CAN messages into global variables
 * @details  check the interrupt pending variable INT0ID;
 *           save can messages into global buffers;
 *           increment the received count.
 *           CAN messages received by MailBox_2 are stored into CanaRxData[0]
 *           CAN messages received by MailBox_4 are stored into CanaRxData[1]
 *           CAN messages received by MailBox_6 are stored into CanaRxData[2]
 */
void hal_can_read_msg(void)
{
	uint32_t u32Int0Id;
	uint32_t msgObjId;
	HAL_CAN_MAIL_BOX *pTxMB;
	HAL_CAN_MAIL_BOX *pRxMB;

	/* Option #1 of reading message: check the Interrupt Pending bit
	HAL_CAN_R[0].CAN_INT is not updated automatically, need call the following API to get INT0ID.
	INT0ID is the mailbox number that has Interrupt Pending, mailbox number is 1~32. */
	u32Int0Id = HAL_CAN_getINT0ID(CAN_BASE);
	if(u32Int0Id > 0) {
		/* note: message object ID is from 0~31. */
		msgObjId = u32Int0Id - 1;
		can_rx_received_cnt[msgObjId]++;

		/* get the pointer to the virtual mailbox. */
		pRxMB = HAL_CAN_getMBbyObjID(CAN_BASE, msgObjId);

		/* user can access any variables/flags in the mailbox structure. */
		if(pRxMB->ID.bit.Dir == CAN_MSG_TYPE_RECEIVE) {
			/* the data in Mailbox can be read by software */
			*(uint32_t *)(&CanaRxData[msgObjId/2][0]) = pRxMB->Data[0].all;
			*(uint32_t *)(&CanaRxData[msgObjId/2][4]) = pRxMB->Data[1].all;

			/* must release the MailBox, it means the mailbox has been read by software. */
			/* NewData bit and IntPnd bit is cleared in this API function. */
			HAL_CAN_releaseMB(CAN_BASE, msgObjId);
		}
	}
}

#if HAL_CAN_POLLING_MODE==0
/*
 * @brief    CAN ISR which need be registered into IRQ vector table.
 */
void hal_can_isr_user(void)
{
	/* clear CPU cycle counter */
	CPU_clearCycleCnt();
	/* read CAN registers and RxFIFO */
	HAL_CAN_updateRegsAndMB(CAN_BASE);
	/* Save CAN messages from virtual MB to user data buffer */
	hal_can_read_msg();
	/* read CPU cycle counter */
	dlog_can_runtime[2] = CPU_getCycleCnt();
}
#endif

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief    hal_can and CAN module initialization
 * @details  configure CAN 500Kbps, bit time 20Tq
 *           start CAN module.
 */
void hal_can_init(void)
{
	/* initialize HAL_CAN structure variables */
	HAL_CAN_initCfg(CAN_BASE);

	/* put CAN into reset mode, and disable all CAN interrupt sources. */
	CAN_initModule(CAN_BASE);

	/* Configure CAN speed to 500Kbps based on 40MHz clock source, with each bit time 20Tq */
	CAN_setBitRate(CAN_BASE, 40000000, 500000, 20);

	/* configure Filter_0 as disabled to avoid accepting unwanted CAN IDs */
	CAN_disableMsgFilter(CAN_BASE, 0);

	/* Configure the Virtual mailbox when CAN in RESET mode */
	CANA_initMessageObjects();

	/* put CAN into normal mode.
	 * The virtual mailbox can be changed during CAN normal mode,
	 * but the message filter/mask need be configured during CAN RESET mode. */
	CAN_startModule(CAN_BASE);

	/* Enable CAN Rx interrupt, Tx interrupt, Error interrupt;
	 * Need enable these interrupts to see corresponding interrupts flags */
	CAN_enableInterrupt(CAN_BASE, CAN_RTIE_RIE_M | CAN_RTIE_TSIE_M | CAN_RTIE_EIE_M);

#if HAL_CAN_POLLING_MODE==0
	/* enable and register CAN interrupt in ECLIC (interrupt controller) */
	Interrupt_register(CAN_IRQ_NUM, hal_can_isr_user);
	Interrupt_enable(CAN_IRQ_NUM);
#endif
}

/*
 * @brief    Run hal_can periodic Tx/Rx task.
 */
void hal_can_run(void)
{
	uint32_t u32Int0Id;
	uint32_t msgObjId;
	HAL_CAN_MAIL_BOX *pTxMB;
	HAL_CAN_MAIL_BOX *pRxMB;
	uint64_t *pu64 = (uint64_t *)&tx_buff[0];

	/* update registers and mailboxes, call this function periodically or in CAN ISR */
#if HAL_CAN_POLLING_MODE==1
	CPU_clearCycleCnt();
	HAL_CAN_updateRegsAndMB(CAN_BASE);
	dlog_can_runtime[0] = CPU_getCycleCnt();
#endif

	/* Method #1 for sending message: load mailbox and send mailbox
	 note: the loadMailBox change the mailbox ID, so it's partial setupMessageObject. */
	HAL_CAN_loadMailBoxDataOnly_8bit(CAN_BASE, can_mailbox_1, &tx_buff[0], 8);
	can_tx_status = HAL_CAN_sendMailBox(CAN_BASE, can_mailbox_1);
	if(can_tx_status)
		(*pu64)++;
	else {
		can_tx_fail_cnt[0]++;
	}

	/* Method #2 for sending message: update mailbox by pointer pMB and send out mailbox
	get the pointer to the specified mailbox (message object 0). */
	pTxMB = HAL_CAN_getMBbyObjID(CAN_BASE, can_mailbox_3);

	/* fill in Tx data from tx_buff. */
	pTxMB->Data[0].all = *(uint32_t *)(&tx_buff[0]);
	pTxMB->Data[1].all = *(uint32_t *)(&tx_buff[4]);

	/* mark the mailbox as newdata ready for Tx. */
	pTxMB->Ctrl.bit.NewDat = 1;

	/* send out mailbox. */
	can_tx_status = HAL_CAN_sendMailBox(CAN_BASE, can_mailbox_3);
	if(can_tx_status)
		(*pu64)++;
	else {
		can_tx_fail_cnt[2]++;
	}

	/* Method #3 for sending message: call sendMessage API directly
	note: need take care of what type of data array is used, uint8/uint16/uint32. */
	can_tx_status = CAN_sendMessage(CAN_BASE, can_mailbox_5, 8, tx_buff);
	if(can_tx_status)
		(*pu64)++;
	else {
		can_tx_fail_cnt[4]++;
	}

#if HAL_CAN_POLLING_MODE==1
	CPU_clearCycleCnt();
	hal_can_read_msg();
	dlog_can_runtime[1] = CPU_getCycleCnt();
#endif

	/* Delay some time to perform next round CAN Msg Transmission. */
	DEVICE_DELAY_US(10000);
}
