/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
* @file    hal_can_ex3_32_mb.c
* @brief
* @details
*  0) Use MACRO "HAL_CAN_ENABLE_64_MB" (defined in Project Settings) to test 64 Mailboxes;
*     Use MACRO "HAL_CAN_POLLING_MODE" to select interrupt mode or polling mode.
*     Use external device or CAN analyzer to generate the following CAN2.0 or CANFD messages:
*         EXT ID 0x10000000 ~ 0x10000009;
*         STD ID 0x7F0 ~ 0x7F7;
*         EXT ID 0x1000FF00;
*         STD ID 0x720 ~ 0x737;
*  1) Use HAL_CAN driver to handle 32 or 64 mailboxes
*  2) Setup mailbox_1~10 as Rx type, acceptance CAN ID is 0x10000000~0x10000009;
*     Setup mailbox_11~20 as Tx type, CAN ID is 0x10010000~0x10010009;
*     Setup mailbox_21~28 as Rx type, acceptance CAN ID is 0x7F0~0x7F7;
*     Setup mailbox_29~31 as Tx type, CAN ID is 0x1F0~0x3F0
*     Setup mailbox_32    as Rx type, acceptance CAN ID 0x1000FF00
*     Setup mailbox_33~56 as Rx Type, acceptance CAN STD ID 0x720 ~ 0x737
*     Setup mailbox_57~61 as Tx Type, CAN ID is STD ID 0x138~0x13D
*  3) Call the specified API to update information from CAN registers/RxFIFO to HAL_CAN mailboxes.
*  4) Demonstrates several methods of sending CAN messages.
*     For CAN message reception, need call HAL_CAN_updateRegsAndMB and then call HAL_CAN_readMessage.
*  5) User can set the MACRO "HAL_CAN_POLLING_MODE" to select Polling mode or Interrupt mode
*     Polling Mode: disable CAN Tx/Rx interrupt and call the API "HAL_CAN_updateRegsAndMB" in background task or in periodic task;
*     Interupt Mode: enable CAN Tx/Rx interrupt and call the API "HAL_CAN_updateRegsAndMB" in CAN Tx/Rx ISR.
*  6) Note: the poll interval is set by the last line calling DEVICE_DELAY_US();
*  7) Use external device to send CAN messages with ID matching the Rx Mailboxes
*     Check variable can_rx_received_cnt[] to see how many messages received by corresponding mailbox.
*     Check variable can_tx_fail_cnt[] to see which mailbox failed to send messages.
*
*  Note:
*  Max 16 HW Rx filters IDs are available in CAN module;
*  User can setup more Rx Mailbox than 16, but need handle the ID filter separately;
*  The example allows all CAN ID be accepted by filter_0 if there are more than 15 HW CAN ID filters needed.
*/

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */
#include "device.h"
#include "hal_can_ex4_fd.h"
#include "string.h"


/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */
/* Use MACRO to replace the API function names. */
#define CAN_setupMessageObject		HAL_CAN_setupMessageObject
#define CAN_sendMessage				HAL_CAN_sendMessage_8bit

/*
 * 1 - select Polling Mode
 * 0 - select Interrupt Mode
 */
#define HAL_CAN_POLLING_MODE   0

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */
uint8_t tx_u8buff[64]={
	0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88,
	0x21, 0x32, 0x43, 0x54, 0x65, 0x76, 0x87, 0x98,
	0x31, 0x42, 0x53, 0x64, 0x75, 0x86, 0x97, 0xA8,
	0x41, 0x52, 0x63, 0x74, 0x85, 0x96, 0xA7, 0xB8,
	0x51, 0x62, 0x73, 0x84, 0x95, 0xA6, 0xB7, 0xC8,
	0x61, 0x72, 0x83, 0x94, 0xA5, 0xB6, 0xC7, 0xD8,
	0x71, 0x82, 0x93, 0xA4, 0xB5, 0xC6, 0xD7, 0xE8,
	0x81, 0x92, 0xA3, 0xB4, 0xC5, 0xD6, 0xE7, 0xF8,
};
uint16_t tx_u16buff[8]={0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77, 0x88};
uint8_t CanaRxData[HAL_TOTAL_MB_NUMS][8];

volatile uint32_t CanaFdRxData[16];
volatile uint32_t dlog_canfd_rxcnt;

volatile uint32_t dlog_can_runtime[4];

volatile uint32_t dlog_can_isrtime[256];
volatile uint32_t dlog_can_isridx;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* CAN Tx failure count for each Tx mailbox */
volatile uint32_t can_tx_fail_cnt[HAL_TOTAL_MB_NUMS];
volatile uint32_t can_tx_status;
/* CAN received messages count for each Rx mailbox */
volatile uint32_t can_rx_received_cnt[HAL_TOTAL_MB_NUMS];
volatile uint32_t can_rx_status;

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */
/*
 * @brief    Setup some mailboxes for Tx/Rx messages
 */
static void CANA_initMessageObjects(void)
{
	/**************************************************************************
	 * Mailbox_1~10 setup as Rx, EXT ID
	 **************************************************************************/
	/* mailbox_1, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_1, 0x10000000, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_2, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_2, 0x10000001, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_3, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_3, 0x10000002, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_4, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_4, 0x10000003, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_5, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_5, 0x10000004, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_6, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_6, 0x10000005, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_7, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_7, 0x10000006, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_8, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_8, 0x10000007, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						CAN_DATA_LENGTH_64);

	/* mailbox_9, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_9, 0x10000008, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						CAN_DATA_LENGTH_64);

	/* mailbox_10, Rx, filter+mask enable, interrupt enable */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_10, 0x10000009, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						CAN_DATA_LENGTH_64);

	/**************************************************************************
	 * Mailbox_11~20 setup as Tx, EXT ID
	 **************************************************************************/
	/* mailbox_11, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_11, 0x10010000, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_12, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_12, 0x10010001, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_13, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_13, 0x10010002, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_14, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_14, 0x10010003, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_15, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_15, 0x10010004, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_16, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_16, 0x10010005, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_17, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_17, 0x10010006, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_18, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_18, 0x10010007, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS | HAL_CAN_MSG_ENABLE_FDF | HAL_CAN_MSG_ENABLE_BRS,
						CAN_DATA_LENGTH_64);

	/* mailbox_19, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_19, 0x10010008, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS | HAL_CAN_MSG_ENABLE_FDF | HAL_CAN_MSG_ENABLE_BRS,
						CAN_DATA_LENGTH_64);

	/* mailbox_20, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_20, 0x10010009, CAN_MSG_FRAME_EXT,
			            CAN_MSG_OBJ_TYPE_TX, 0x1FFFFFFF,
						HAL_CAN_MSG_OBJ_NO_FLAGS | HAL_CAN_MSG_ENABLE_FDF | HAL_CAN_MSG_ENABLE_BRS,
						CAN_DATA_LENGTH_64);

	/**************************************************************************
	 * Mailbox_21~28 setup as Rx, STD ID
	 **************************************************************************/
	/* mailbox_21, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_21, 0x7F0, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_22, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_22, 0x7F1, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_23, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_23, 0x7F2, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_24, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_24, 0x7F3, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_25, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_25, 0x7F4, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_26, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_26, 0x7F5, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_27, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_27, 0x7F6, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/* mailbox_28, Rx, filter+mask enable, interrupt enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_28, 0x7F7, CAN_MSG_FRAME_STD,
						CAN_MSG_OBJ_TYPE_RX, 0x7FF,
						CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_USE_EXT_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

	/**************************************************************************
	 * Mailbox_29~31 setup as Tx, STD ID
	 **************************************************************************/
	/* mailbox_29, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_29, 0x1F0, CAN_MSG_FRAME_STD,
			            CAN_MSG_OBJ_TYPE_TX, 0x7FF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_30, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_30, 0x2F0, CAN_MSG_FRAME_STD,
			            CAN_MSG_OBJ_TYPE_TX, 0x7FF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/* mailbox_31, Tx, interrupt not enable. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_31, 0x3F0, CAN_MSG_FRAME_STD,
			            CAN_MSG_OBJ_TYPE_TX, 0x7FF,
						HAL_CAN_MSG_OBJ_NO_FLAGS,
						8);

	/**************************************************************************
	 * Mailbox_32 setup as Rx, STD ID or EXT ID
	 **************************************************************************/
	/* mailbox_32, Rx, filter+mask enable, interrupt enable
	Mask is 0, means all bits don't care, this mailbox accepts all messages IDs. */
	CAN_setupMessageObject(CAN_BASE, can_mailbox_32, 0x1000FF00, CAN_MSG_FRAME_EXT,
						CAN_MSG_OBJ_TYPE_RX, 0x1FFFFFFF,
						CAN_MSG_OBJ_USE_ID_FILTER | CAN_MSG_OBJ_RX_INT_ENABLE,
						8);

#if HAL_CAN_ENABLE_64_MB
	/**************************************************************************
	 * Mailbox_33~56 setup as Rx, STD ID 0x700~
	 **************************************************************************/
	for(uint32_t objID=can_mailbox_33; objID<=can_mailbox_56; objID++)
	{
		CAN_setupMessageObject(CAN_BASE, objID, 0x700+objID, CAN_MSG_FRAME_STD,
							CAN_MSG_OBJ_TYPE_RX, 0x7FF,
							CAN_MSG_OBJ_USE_ID_FILTER |	CAN_MSG_OBJ_RX_INT_ENABLE,
							8);
	}

	/**************************************************************************
	 * Mailbox_57~61 setup as Tx, STD ID 0x100~
	 **************************************************************************/
	for(uint32_t objID=can_mailbox_57; objID<=can_mailbox_61; objID++)
	{
		CAN_setupMessageObject(CAN_BASE, objID, 0x100+objID, CAN_MSG_FRAME_STD,
				            CAN_MSG_OBJ_TYPE_TX, 0x7FF,
							HAL_CAN_MSG_OBJ_NO_FLAGS | HAL_CAN_MSG_ENABLE_FDF | HAL_CAN_MSG_ENABLE_BRS,
							8);
	}
#endif

}

/*
 * @brief    save received CAN messages into global variables
 * @details  check the interrupt pending variable INT0ID;
 *           save can messages into global buffers;
 *           increment the received count.
 *           CAN messages received by MailBox_2 are stored into CanaRxData[0]
 *           CAN messages received by MailBox_4 are stored into CanaRxData[1]
 *           CAN messages received by MailBox_6 are stored into CanaRxData[2]
 */
void hal_can_read_msg(void)
{
	uint32_t u32Int0Id;
	uint32_t msgObjId;
	HAL_CAN_MAIL_BOX *pTxMB;
	HAL_CAN_MAIL_BOX *pRxMB;

	/* Option #1 of reading message: check the Interrupt Pending bit
	HAL_CAN_R[0].CAN_INT is not updated automatically, need call the following API to get INT0ID.
	INT0ID is the mailbox number that has Interrupt Pending, mailbox number is 1~32. */
	u32Int0Id = HAL_CAN_getINT0ID(CAN_BASE);
	if(u32Int0Id > 0) {
		/* note: message object ID is from 0~31. */
		msgObjId = u32Int0Id - 1;
		can_rx_received_cnt[msgObjId]++;

		/* get the pointer to the virtual mailbox. */
		pRxMB = HAL_CAN_getMBbyObjID(CAN_BASE, msgObjId);

		/* user can access any variables/flags in the mailbox structure. */
		if(pRxMB->ID.bit.Dir == CAN_MSG_TYPE_RECEIVE) {
			/* If a CANFD message is received. */
			if(pRxMB->Ctrl.bit.FDF)
			{
				uint32_t nLength = CAN_getMessageLength(pRxMB->Ctrl.bit.DLC);
				memcpy((void *)&CanaFdRxData[0], (void *)&pRxMB->Data[0], nLength);
				dlog_canfd_rxcnt++;
			}
			else /* If a CAN2.0 message received */
			{
				/* the data in Mailbox can be read by software */
				*(uint32_t *)(&CanaRxData[msgObjId/2][0]) = pRxMB->Data[0].all;
				*(uint32_t *)(&CanaRxData[msgObjId/2][4]) = pRxMB->Data[1].all;
			}

			/* must release the MailBox, it means the mailbox has been read by software. */
			/* NewData bit and IntPnd bit is cleared in this API function. */
			HAL_CAN_releaseMB(CAN_BASE, msgObjId);
		}
	}
}

#if HAL_CAN_POLLING_MODE==0
/*
 * @brief    CAN ISR which need be registered into IRQ vector table.
 */
void hal_can_isr_user(void)
{
	/* clear CPU cycle counter */
	CPU_clearCycleCnt();
	/* read CAN registers and RxFIFO */
	HAL_CAN_updateRegsAndMB(CAN_BASE);
	/* Save CAN messages from virtual MB to user data buffer */
	hal_can_read_msg();
	/* read CPU cycle counter */
	dlog_can_runtime[2] = CPU_getCycleCnt();
	dlog_can_isrtime[dlog_can_isridx++] = CPU_getCycleCnt();
	dlog_can_isridx &=0x00FF;
}
#endif

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/*
 * @brief    hal_can and CAN module initialization
 * @details  configure CAN 500Kbps, bit time 20Tq
 *           start CAN module.
 */
void hal_can_init(void)
{
	/* initialize HAL_CAN structure variables */
	HAL_CAN_initCfg(CAN_BASE);

	/* put CAN into reset mode, and disable all CAN interrupt sources. */
	CAN_initModule(CAN_BASE);

	/* enable CANFD mode */
	CAN_setFDMode(CAN_BASE, true);
	CAN_SetCanFdIsoMode(CAN_BASE, CAN_FD_ISO_11898_1);

	/* Configure CAN speed to 500Kbps based on 40MHz clock source, with each bit time 20Tq */
	//CAN_setBitRate(CAN_BASE, 40000000, 500000, 20);
	CAN_setBitRateSamplePoint(CAN_BASE, 40000000, 500 * 1000, 0.8);

	//CAN_setDataBitRate(CAN_BASE, 40000000, 2000000, 20);
	CAN_setDataBitRateSamplePoint(CAN_BASE, 40000000, 2U * 1000 * 1000, 0.8);

	/* Configure Filter_0 to accept all IDs in case more than 15 ID filters are needed */
	CAN_setMsgFilter(CAN_BASE, 0, 0x0, 0x1FFFFFFF, CAN_IDTYPE_STD_OR_EXT);
	/* configure Filter_0 as disabled to avoid accepting unwanted CAN IDs */
	CAN_disableMsgFilter(CAN_BASE, 0);

	/* Configure the Virtual mailbox when CAN in RESET mode */
	CANA_initMessageObjects();

	/* put CAN into normal mode.
	 * The virtual mailbox can be changed during CAN normal mode,
	 * but the message filter/mask need be configured during CAN RESET mode. */
	CAN_startModule(CAN_BASE);

	/* Enable CAN Rx interrupt, Tx interrupt, Error interrupt;
	 * Need enable these interrupts to see corresponding interrupts flags */
	CAN_enableInterrupt(CAN_BASE, CAN_RTIE_RIE_M | CAN_RTIE_TSIE_M | CAN_RTIE_EIE_M);

#if HAL_CAN_POLLING_MODE==0
	/* enable and register CAN interrupt in ECLIC (interrupt controller) */
	Interrupt_register(CAN_IRQ_NUM, hal_can_isr_user);
	Interrupt_enable(CAN_IRQ_NUM);
#endif
}

/*
 * @brief    Run hal_can periodic Tx/Rx task.
 */
void hal_can_run(void)
{
	uint32_t u32Int0Id;
	uint32_t msgObjId;
	HAL_CAN_MAIL_BOX *pTxMB;
	HAL_CAN_MAIL_BOX *pRxMB;
	uint64_t *pu64 = (uint64_t *)&tx_u8buff[0];

	/* update registers and mailboxes, call this function in periodic task or in CAN ISR */
#if HAL_CAN_POLLING_MODE==1
	CPU_clearCycleCnt();
	HAL_CAN_updateRegsAndMB(CAN_BASE);
	dlog_can_runtime[0] = CPU_getCycleCnt();
#endif

	/**************************************************************************
	 * Method #1 for sending message: load mailbox and send mailbox
	 * note: the loadMailBox change the mailbox ID, so it's partial setupMessageObject.
	 **************************************************************************/
	msgObjId = can_mailbox_11;
	HAL_CAN_loadMailBoxDataOnly_8bit(CAN_BASE, msgObjId, &tx_u8buff[0], 8);
	can_tx_status = HAL_CAN_sendMailBox(CAN_BASE, msgObjId);
	if(can_tx_status)
		(*pu64)++;
	else
		can_tx_fail_cnt[msgObjId]++;

	/**************************************************************************
	 * Method #2 for sending message: update mailbox by pointer pMB and send out mailbox
	 * get the pointer to the specified mailbox (message object 0).
	 ***************************************************************************/
	msgObjId = can_mailbox_12;
	pTxMB = HAL_CAN_getMBbyObjID(CAN_BASE, msgObjId);

	/* fill in Tx data from tx_u8buff. */
	pTxMB->Data[0].all = *(uint32_t *)(&tx_u8buff[0]);
	pTxMB->Data[1].all = *(uint32_t *)(&tx_u8buff[4]);

	/* mark the mailbox as newdata ready for Tx. */
	pTxMB->Ctrl.bit.NewDat = 1;

	/* send out mailbox. */
	can_tx_status = HAL_CAN_sendMailBox(CAN_BASE, msgObjId);
	if(can_tx_status)
		(*pu64)++;
	else
		can_tx_fail_cnt[msgObjId]++;

	/**************************************************************************
	 * Method #3 for sending message: call sendMessage API directly
	 * note: need take care of what type of data array is used, uint8/uint16/uint32.
	**************************************************************************/
	msgObjId = can_mailbox_13;
	can_tx_status = HAL_CAN_sendMessage(CAN_BASE, msgObjId, 8, tx_u16buff);
	if(can_tx_status) {
		tx_u16buff[0]++;
		tx_u16buff[0] &= 0x00FF;
	}
	else
		can_tx_fail_cnt[msgObjId]++;

	/**************************************************************************
	 * send mailbox_14~20
	 **************************************************************************/
	for(msgObjId = can_mailbox_14; msgObjId <= can_mailbox_17; msgObjId++)
	{
		can_tx_status = HAL_CAN_sendMessage_8bit(CAN_BASE, msgObjId, 8, tx_u8buff);
		if(can_tx_status)
			(*pu64)++;
		else
			can_tx_fail_cnt[msgObjId]++;
	}
	/********************** CANFD Frames transmission ************************/
	for(msgObjId = can_mailbox_18; msgObjId <= can_mailbox_20; msgObjId++)
	{
		can_tx_status = HAL_CAN_sendMessage_8bit(CAN_BASE, msgObjId, CAN_DATA_LENGTH_64, tx_u8buff);
		if(can_tx_status)
			(*pu64)++;
		else
			can_tx_fail_cnt[msgObjId]++;
	}

	/* Delay some time to avoid TxFIFO Full */
	DEVICE_DELAY_US(5000);

	/**************************************************************************
	 * send mailbox_29~31
	 **************************************************************************/
	for(msgObjId = can_mailbox_29; msgObjId <= can_mailbox_31; msgObjId++)
	{
		can_tx_status = HAL_CAN_sendMessage_8bit(CAN_BASE, msgObjId, CAN_DATA_LENGTH_8, tx_u8buff);
		if(can_tx_status)
			(*pu64)++;
		else
			can_tx_fail_cnt[msgObjId]++;
	}

#if HAL_CAN_ENABLE_64_MB
	/**************************************************************************
	 * send mailbox_57~61
	 **************************************************************************/
	for(msgObjId = can_mailbox_57; msgObjId <= can_mailbox_61; msgObjId++)
	{
		can_tx_status = HAL_CAN_sendMessage_8bit(CAN_BASE, msgObjId, CAN_DATA_LENGTH_16, tx_u8buff);
		if(can_tx_status)
			(*pu64)++;
		else
			can_tx_fail_cnt[msgObjId]++;
	}
#endif


#if HAL_CAN_POLLING_MODE==1
	/* call this function to read messages saved in Rx Mailboxes */
	hal_can_read_msg();
#endif

	/* Delay some time to perform next round CAN Msg Transmission. */
	DEVICE_DELAY_US(10000);
}
