/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    cputimer_ex2_sync_with_epwm.c
*   @brief   CPUtimer is configured as same period with EPWM
*   @details check the variable in liveview: dlog_epwm1_ts, dlog_epwm2_ts.
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "device.h"
#include "board_cfg.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* CPUTimer IRQ period is 100uS */
#define CPUTIMER0_PERIOD  ((DEVICE_APBCLK_FREQ/10000)-1)
/* EPWM TBCTR period is 100uS */
#define EPWM1_PERIOD      ((DEVICE_AHBCLK_FREQ/10000)-1)

/* CPUTimer IRQ period is 100uS */
#define CPUTIMER1_PERIOD  (DEVICE_APBCLK_FREQ/10000)
/* EPWM TBCTR period is 100uS */
#define EPWM2_PERIOD      (DEVICE_AHBCLK_FREQ/10000)

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

uint32_t cputimer0_int_cnt;
uint32_t dlog_epwm1_ts[32];
uint32_t dlog_epwm1_idx;

uint32_t cputimer1_int_cnt;
uint32_t dlog_epwm2_ts[32];
uint32_t dlog_epwm2_idx;

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */
/******************************************************************************
 * EPWM configurations
 * 1)Counter Up mode, prescaler is div by 2;
 * 2)EPWM period is EPWM1_PERIOD = 1000
 * 3)when Counter==Zero, generate EPWM1 Interrupt;
 ******************************************************************************/
void EPWM_initChannel(uint32_t base, uint32_t nPeriod, EPWM_TimeBaseCountMode tb_mode)
{
	EPWM_setClockPrescaler(base, EPWM_CLOCK_DIVIDER_1, EPWM_HSCLOCK_DIVIDER_1);
	EPWM_setTimeBasePeriod(base, nPeriod);
	EPWM_setTimeBaseCounter(base, 0);
	EPWM_setTimeBaseCounterMode(base, tb_mode);
//	EPWM_setInterruptEventCount(base, 1);
//	EPWM_setInterruptSource(base, EPWM_INT_TBCTR_ZERO);
//	EPWM_enableInterrupt(base);
}


/**
 * \brief   CPU Timer0жϴ
 * \param   none
 * \retval  None
 */
__INTERRUPT void Cputimer0_IRQHandler(void)
{
    /* CPU Timer0 жϱ־λ */
    CPUTimer_clearOverflowFlag(CPUTIMER0_BASE);
    cputimer0_int_cnt++;

    /* ȡEPWM TBCTRֵӦһȶֵ */
    dlog_epwm1_ts[dlog_epwm1_idx] = EPWM_getTimeBaseCounterValue(EPWM1_BASE);
    dlog_epwm1_idx++;
    dlog_epwm1_idx &= (32-1);
}

/**
 * \brief   CPU Timer0жϴ
 * \param   none
 * \retval  None
 */
__INTERRUPT void Cputimer1_IRQHandler(void)
{
    /* CPU Timer0 жϱ־λ */
    CPUTimer_clearOverflowFlag(CPUTIMER1_BASE);
    cputimer1_int_cnt++;

    /* ȡEPWM TBCTRֵӦһȶֵ */
    dlog_epwm2_ts[dlog_epwm2_idx] = EPWM_getTimeBaseCounterValue(EPWM2_BASE);
    dlog_epwm2_idx++;
    dlog_epwm2_idx &= (32-1);
}

/**
  * \brief  Main program
  * \param  None
  * \retval None
  */
int main(void)
{
    Device_init();

    /* ָGPIOΪGPIOģʽ */
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);

    /*
     * Configure CPUTimer0 synchronized with EPWM1
     */
    /* ʼEPWM1Counter upģʽڵCPUTimer0 */
    EPWM_initChannel(EPWM1_BASE, EPWM1_PERIOD, EPWM_COUNTER_MODE_UP);

    /* ʼCpuTimer0Թ̶вж */
    /* CPUTimer0 is started within this function */
    CPUTimer_init(CPUTIMER0_BASE, CPUTIMER0_PERIOD);
    /* עCPUTimerжϴ */
    Interrupt_register(INT_TIMER0, Cputimer0_IRQHandler);
    /* ϵͳжϿʹCpuTimer0ж */
    Interrupt_enable(INT_TIMER0);

    /*
     * Configure CPUTimer1 synchronized with EPWM2
     */
    /* ʼEPWM2Counter upģʽڵCPUTimer1 */
    EPWM_initChannel(EPWM2_BASE, EPWM2_PERIOD, EPWM_COUNTER_MODE_UP);

    /* ʼCpuTimer1Թ̶вж */
    CPUTimer_init(CPUTIMER1_BASE, CPUTIMER1_PERIOD);
    /* עCPUTimerжϴ */
    Interrupt_register(INT_TIMER1, Cputimer1_IRQHandler);
    /* ϵͳжϿʹCpuTimer1ж */
    Interrupt_enable(INT_TIMER1);

    /* start EPWM TB counter */
    SysCtl_enablePeripheral_TBCLKSYNC();

    __enable_irq();

    /* We should never get here as control is now taken by the scheduler */
    for(;;);
}

#ifdef __cplusplus
}
#endif

