/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file     cputimer_ex4_dma.c
*   @brief    Configure CPUTimer0 to trigger a DMA channel to control a GPIO output
*   @details  Change "gpio_outdata" value to generate various waveforms on the GPIO output
*             View these variables for debug purpose:
*             gpio_outdata
*             dlog_dma_ch1_isr_cnt;
*             dlog_dma_ch1_src_addr
*             dlog_dma_ch1_dest_addr
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "stdio.h"
#include "device.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

#if IS_GS32F00xx(0x30)
#define GPIO_BASE_ADDR_STEP 		0x1000
#endif



/* CPUTimer0 period is 100mS */
#define CPUTIMER0_PERIOD  (DEVICE_APBCLK_FREQ/10)


#if IS_GS32F00xx(0x12) ||IS_GS32F3xx(0x22)

#define DMAx_BASE					DMA1_BASE
#define DMA_CHx_BASE				DMA1_CH1_BASE

/* The DMA controlled GPIO data register address */
#define MY_GPIO_DATA_REG_ADDR		((GPIODATA_BASE + GPIO_MASKLOWBYTE + (GPIO_PIN_LED1 / 16U) * GPIO_BASE_ADDR_STEP + 4*(0x01<<(GPIO_PIN_LED1%16))))
#define GPIO_REG_VAL   				((uint32_t)(1<<(GPIO_PIN_LED1%16)))

#elif IS_GS32F00xx(0x30)

#define DMAx_BASE					DMA_BASE
#define DMA_CHx_BASE				DMA_CH1_BASE
/* The DMA controlled GPIO data register address */
#define MY_GPIO_DATA_REG_ADDR		(&GpioDataRegs.GPATOGGLE)
#define GPIO_REG_VAL   				(1<<((GPIO_PIN_LED1)%32))

#endif


/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

uint32_t dlog_dma_ch1_isr_cnt;
uint32_t dlog_dma_ch1_src_addr;
uint32_t dlog_dma_ch1_dest_addr;

uint32_t gpio_outdata[8] = {
	0x0000, 	GPIO_REG_VAL, 	0x0000, 	GPIO_REG_VAL,
	0x0000, 	GPIO_REG_VAL, 	0x0000, 	GPIO_REG_VAL
};

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

/*
 * @brief   DMA Channel1 Block Complete ISR
 */
__INTERRUPT void Dma_Ch1_ISR(void)
{
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
	DMA_clearBlockInterruptStatus(DMA_CHx_BASE);
	__DSB();
	dlog_dma_ch1_isr_cnt++;

	/* record the current DMA CH1 Source Address Register value for debug purpose */
	dlog_dma_ch1_src_addr = HWREG(DMA_CHx_BASE + DMA_O_SAR);
	/* record the current DMA CH1 Dest Address Register value for debug purpose */
	dlog_dma_ch1_dest_addr = HWREG(DMA_CHx_BASE + DMA_O_DAR);

	if(DMA_getInterruptStatus(DMA_CHx_BASE)&DMA_INT_ERR)
		while(1);

	/* re-configure source address */
	DMA_configSourceAddress(DMA_CHx_BASE, (uint32_t)(&gpio_outdata[0]));
	/* restart DMA Channel */
	DMA_startChannel(DMA_CHx_BASE);

#elif IS_GS32F3xx(0x22) || IS_GS32F3xx(0x23)
	XDMA_clearInterrupt(DMA_CHx_BASE, XDMA_INT_BLOCK);
	__DSB();
	dlog_dma_ch1_isr_cnt++;

	/* record the current DMA CH1 Source Address Register value for debug purpose */
	dlog_dma_ch1_src_addr = HWREG(DMA_CHx_BASE + XDMA_O_XSAR);
	/* record the current DMA CH1 Dest Address Register value for debug purpose */
	dlog_dma_ch1_dest_addr = HWREG(DMA_CHx_BASE + XDMA_O_XDAR);

	if(XDMA_getInterruptStatus(DMA_CHx_BASE)&XDMA_INT_ERR)
		while(1);

	/* re-configure source address */
	XDMA_configSourceAddress(DMA_CHx_BASE, (uint32_t)(&gpio_outdata[0]));
	/* restart DMA Channel */
	XDMA_startChannel(DMA_CHx_BASE);
#endif
	GPIO_togglePin(GPIO_PIN_LED2);
}
/**
 * @brief   Configure CPUTimer as periodic mode
 * @param   base   CPUTimer base address
 * @param   period CPUTimer period
 * @retval  None
 */
void CpuTimer_initPeriodMode(uint32_t base, uint32_t period)
{
	CPUTimer_stopTimer(base);
	 /* CPUTimer clock prescaler 1 */
#if IS_GS32F00xx(0x12) || IS_GS32F3xx(0x22)    
    CPUTimer_setPreScaler(base, CPUTIMER_CLOCK_PRESCALER_1);
#else
    CPUTimer_setPreScaler(base, 1);
#endif
	/* CPUTimer size 32bit */
    CPUTimer_setTimerSize(base, CPUTIMER_TIMERSIZE_32BIT);
	/* set CPUTimer as periodic mode */
	CPUTimer_setTimerMode(base, CPUTIMER_TIMERMODE_PERIODIC);
	/* set CPUTimer period */
	CPUTimer_setPeriod(base, period);
	/* Enable CPUTimer interrupt */
	CPUTimer_enableInterrupt(base);
	/* clear interrupt flag */
	CPUTimer_clearOverflowFlag(base);
}

/*
 * @brief  Configure DMA Channel_1 to triggered by CPUTimer0 to control GPIO LED1
 */
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
/* This is for GS32F00xx DMA initialization */
void Dma_initChanForCputimer(void)
{   
    uint32_t status = 0;
    uint32_t i;
    DMA_ConfigParams dmaCfg = {0};

    /* DMA global enable */
    DMA_initController(DMAx_BASE);
    
    /* Stop DMA channel first in case it's still running */
    DMA_stopChannel(DMA_CHx_BASE);
    dmaCfg.enableInterrupt = 1;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_Trig_ap_timer0;
    dmaCfg.dmaSrcReqId = 0;//
    dmaCfg.srcAddr = (uint32_t)(&gpio_outdata[0]);
	dmaCfg.destAddr = (uint32_t)MY_GPIO_DATA_REG_ADDR;
    dmaCfg.blockTS = 8;
    dmaCfg.ttfc    = DMA_TT_FC_1_M2P_DMAC;
    dmaCfg.srcBtl  = DMA_BTL_1;
    dmaCfg.destBtl = DMA_BTL_1;
    dmaCfg.srcAddrDirect = DMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = DMA_ADDR_NO_CHANGE;
    dmaCfg.srcTrWidthBytes = DMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= DMA_TR_WIDTH_BYTE_4;
    /* auto reload source address if needed */
    //dmaCfg.reloadSrc = 1;

    DMA_configChannel(DMA_CHx_BASE, &dmaCfg);
    /* Enable Block Complete Interrupt and Error Interrupt */
    DMA_clearInterrupt(DMA_CHx_BASE, DMA_INT_BLOCK | DMA_INT_ERR);
    DMA_unMaskInterrupt(DMA_CHx_BASE, DMA_INT_BLOCK | DMA_INT_ERR);
}
#elif IS_GS32F3xx(0x22)
/* This is for GS32F00xx XDMA(AXI DMA) initialization */
void Dma_initChanForCputimer(void)
{
    uint32_t status = 0;
    uint32_t i;
    XDMA_ConfigParams dmaCfg = {0};

    /* DMA global enable */
    XDMA_initController(DMAx_BASE);

    /* Stop DMA channel first in case it's still running */
    XDMA_stopChannel(DMA_CHx_BASE);
    dmaCfg.enableInterrupt = 1;
    dmaCfg.dmaDstReqId = DMAMUX_ReqId_Trig_ap_timer0;
    dmaCfg.dmaSrcReqId = 0;
    dmaCfg.srcAddr = (uint32_t)(&gpio_outdata[0]);
	dmaCfg.destAddr = (uint32_t)MY_GPIO_DATA_REG_ADDR;
    dmaCfg.blockTS = 8;
    dmaCfg.ttfc    = XDMA_TT_FC_1_M2P_DMAC;
    dmaCfg.srcBtl  = XDMA_BTL_1;
    dmaCfg.destBtl = XDMA_BTL_1;
    dmaCfg.srcAddrDirect = XDMA_ADDR_INCRE;
    dmaCfg.destAddrDirect = XDMA_ADDR_NO_CHANGE;
    dmaCfg.srcTrWidthBytes = XDMA_TR_WIDTH_BYTE_4;
    dmaCfg.destTrWidthBytes= XDMA_TR_WIDTH_BYTE_4;
    /*
     * Must disable DMA fifo mode
     * so that each DMA requested data is transferred to dest immediately.
     * Otherwise the data is hold inside the DMA internal FIFO and were transferred when block completed.
     */
    dmaCfg.enableAWfifo = 0;
    /* auto reload source address if needed */
    //dmaCfg.reloadSrc = 1;

    /* write DMA configuration into corresponding registers */
    XDMA_configChannel(DMA_CHx_BASE, &dmaCfg);

    /* Enable Block Complete Interrupt and Error Interrupt */
    XDMA_clearInterrupt(DMA_CHx_BASE, XDMA_INT_BLOCK | XDMA_INT_ERR);
    XDMA_unMaskInterrupt(DMA_CHx_BASE, XDMA_INT_BLOCK | XDMA_INT_ERR);
}
#endif

 /**
 * \brief   main function to demonstrate DMA controlled GPIO toggling
 * \param   none
 * \retval  None
 */
void main(void)
{
    Device_init();

    /* Configure GPIO output to driver LED1 */
    GPIO_setPinConfig(GPIO_CFG_LED1);
    GPIO_setAnalogMode(GPIO_PIN_LED1, GPIO_ANALOG_DISABLED);
    GPIO_setDirectionMode(GPIO_PIN_LED1, GPIO_DIR_MODE_OUT);

    GPIO_writePin(GPIO_PIN_LED1, 1);

    /* Configure GPIO output to driver LED2 */
	GPIO_setPinConfig(GPIO_CFG_LED2);
	GPIO_setAnalogMode(GPIO_PIN_LED2, GPIO_ANALOG_DISABLED);
	GPIO_setDirectionMode(GPIO_PIN_LED2, GPIO_DIR_MODE_OUT);

    /* Configure CPUTimer0 to generate timeout flag every 100mS */
    CpuTimer_initPeriodMode(CPUTIMER0_BASE, CPUTIMER0_PERIOD);

    /* DMA channel_1 initialization as triggered by CPUTimer0 */
    Dma_initChanForCputimer();
#if IS_GS32F00xx(0x12) || IS_GS32F00xx(0x30)
    /* DMA channel start */
    DMA_startChannel(DMA_CHx_BASE);
#elif IS_GS32F3xx(0x22)
    /* DMA channel start */
    XDMA_startChannel(DMA_CHx_BASE);
#endif
    /* Register and enable DMA CH1 interrupt */
    Interrupt_enable(INT_DMA1_CH1);
    Interrupt_register(INT_DMA1_CH1, Dma_Ch1_ISR);

    __enable_irq();

    CPUTimer_startTimer(CPUTIMER0_BASE);

    while(1);
}

#ifdef __cplusplus
}
#endif

