/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include <stdlib.h>
#include "device.h"
#include "printf.h"
#include "log.h"
#include "board_cfg.h"

/*
 * Calculate CRC16 for 1KB input data array.
 * dlog_crc_ts[0]:
 *   CPU execution cycles for CRC16 by software
 *   =11307 cycles, 22 cycles per 16bit data
 * dlog_crc_ts[1]:
 *   CPU instructions count for CRC16 by software
 *   =9234 instructions, 18 instructions per 16bit data
 * dlog_crc_ts[2]:
 *   CPU execution cycles for CRC16 by HW
 *   =5672 cycles, 11 cycles per 16bit data
 * dlog_crc_ts[3]:
 *   CPU instructions count for CRC16 by HW
 *   =2071 instructions, 4 instructions per 16bit data
 */
uint32_t dlog_crc_ts[4];

/*
 * crc_result[0]: crc16 result by software, 0xECFE
 * crc_result[1]: crc16 result by hardware, 0xECFE
 */
uint16_t crc_result[4];

/*
 * CRC input data array.
 */
uint8_t crc_input[1024];

/*
 * CRC16 lookup table for CRC-16/MODBUS:
 * poly=0x8005, init=0xFFFF, RevIn=ture, RevOut=ture, XorOut=0;
 */
const uint16_t crc_table[256] = {
		0x0000, 0xc0c1, 0xc181, 0x0140, 0xc301, 0x03c0, 0x0280, 0xc241,
		0xc601, 0x06c0, 0x0780, 0xc741, 0x0500, 0xc5c1, 0xc481, 0x0440,
		0xcc01, 0x0cc0, 0x0d80, 0xcd41, 0x0f00, 0xcfc1, 0xce81, 0x0e40,
		0x0a00, 0xcac1, 0xcb81, 0x0b40, 0xc901, 0x09c0, 0x0880, 0xc841,
		0xd801, 0x18c0, 0x1980, 0xd941, 0x1b00, 0xdbc1, 0xda81, 0x1a40,
		0x1e00, 0xdec1, 0xdf81, 0x1f40, 0xdd01, 0x1dc0, 0x1c80, 0xdc41,
		0x1400, 0xd4c1, 0xd581, 0x1540, 0xd701, 0x17c0, 0x1680, 0xd641,
		0xd201, 0x12c0, 0x1380, 0xd341, 0x1100, 0xd1c1, 0xd081, 0x1040,
		0xf001, 0x30c0, 0x3180, 0xf141, 0x3300, 0xf3c1, 0xf281, 0x3240,
		0x3600, 0xf6c1, 0xf781, 0x3740, 0xf501, 0x35c0, 0x3480, 0xf441,
		0x3c00, 0xfcc1, 0xfd81, 0x3d40, 0xff01, 0x3fc0, 0x3e80, 0xfe41,
		0xfa01, 0x3ac0, 0x3b80, 0xfb41, 0x3900, 0xf9c1, 0xf881, 0x3840,
		0x2800, 0xe8c1, 0xe981, 0x2940, 0xeb01, 0x2bc0, 0x2a80, 0xea41,
		0xee01, 0x2ec0, 0x2f80, 0xef41, 0x2d00, 0xedc1, 0xec81, 0x2c40,
		0xe401, 0x24c0, 0x2580, 0xe541, 0x2700, 0xe7c1, 0xe681, 0x2640,
		0x2200, 0xe2c1, 0xe381, 0x2340, 0xe101, 0x21c0, 0x2080, 0xe041,
		0xa001, 0x60c0, 0x6180, 0xa141, 0x6300, 0xa3c1, 0xa281, 0x6240,
		0x6600, 0xa6c1, 0xa781, 0x6740, 0xa501, 0x65c0, 0x6480, 0xa441,
		0x6c00, 0xacc1, 0xad81, 0x6d40, 0xaf01, 0x6fc0, 0x6e80, 0xae41,
		0xaa01, 0x6ac0, 0x6b80, 0xab41, 0x6900, 0xa9c1, 0xa881, 0x6840,
		0x7800, 0xb8c1, 0xb981, 0x7940, 0xbb01, 0x7bc0, 0x7a80, 0xba41,
		0xbe01, 0x7ec0, 0x7f80, 0xbf41, 0x7d00, 0xbdc1, 0xbc81, 0x7c40,
		0xb401, 0x74c0, 0x7580, 0xb541, 0x7700, 0xb7c1, 0xb681, 0x7640,
		0x7200, 0xb2c1, 0xb381, 0x7340, 0xb101, 0x71c0, 0x7080, 0xb041,
		0x5000, 0x90c1, 0x9181, 0x5140, 0x9301, 0x53c0, 0x5280, 0x9241,
		0x9601, 0x56c0, 0x5780, 0x9741, 0x5500, 0x95c1, 0x9481, 0x5440,
		0x9c01, 0x5cc0, 0x5d80, 0x9d41, 0x5f00, 0x9fc1, 0x9e81, 0x5e40,
		0x5a00, 0x9ac1, 0x9b81, 0x5b40, 0x9901, 0x59c0, 0x5880, 0x9841,
		0x8801, 0x48c0, 0x4980, 0x8941, 0x4b00, 0x8bc1, 0x8a81, 0x4a40,
		0x4e00, 0x8ec1, 0x8f81, 0x4f40, 0x8d01, 0x4dc0, 0x4c80, 0x8c41,
		0x4400, 0x84c1, 0x8581, 0x4540, 0x8701, 0x47c0, 0x4680, 0x8641,
		0x8201, 0x42c0, 0x4380, 0x8341, 0x4100, 0x81c1, 0x8081, 0x4040
};

/*
 * \brief      Perform CRC16 on input data array with software lookup table
 * \param[in]  input_array, pointer to input data array
 * \param[in]  input_length, length of input data array
 * \return     CRC16 result of the input data array
 *
 * \details    There are 9 instructions in one loop
 *             1Kbyte data needs 1024 loops = 9236 CPU cycles
 */
uint16_t calculate_crc16(uint8_t *input_array, uint16_t input_length)
{
	uint16_t crctmp = 0xFFFF;
	uint8_t crc_index;
    for (uint16_t i = 0; i < input_length; i++)
    {
        crc_index = (crctmp ^ input_array[i]) & 0xff;
        crctmp = (crc_table[crc_index] ^ (crctmp >> 8)) & 0xFFFF;
    }
    return crctmp;
}

/*
 * \brief    Perform CRC16 on input data array with CRC HW.
 * \param[in]  pBufIn, pointer to input data array
 * \param[in]  len, length of input data array
 * \return     CRC16 result of the input data array
 */
uint16_t CRC_calcCrc16Test(uint8_t *pBufIn, uint32_t len)
{
	uint32_t in_rev_cfg;
	uint32_t out_rev_cfg;
	uint32_t seed_value;
	uint32_t poly_size;
	uint32_t poly_value;
	uint32_t data_len;
	uint32_t rslt_xor_out;

	uint16_t *pDataIn;

	uint16_t ret;

	/* CRC hardware initialization */
	poly_size = CRC_POLYSIZE_16BIT;
	poly_value = 0x8005;
	seed_value = 0xFFFF;
	in_rev_cfg = CRC_REVIN_BYTEREV;
	out_rev_cfg = CRC_REVOUT_ON;
	CRC_setConfig(CRC_BASE, poly_size, poly_value, in_rev_cfg, out_rev_cfg, seed_value);
	data_len = len/2;
	rslt_xor_out = 0x0000;

	/* swap input data byte order, so as to process the data in 16bit width. */
	pDataIn = (uint16_t*)(pBufIn);
	for(uint32_t i=0; i<data_len; i++)
	{
		*pDataIn = __gs_bswap16(*pDataIn);
		pDataIn++;
	}

	/* CRC calculation by CRC HW. */
	dlog_crc_ts[2] = CPU_getCycleCnt();
	dlog_crc_ts[3] = CPU_getInstrCnt();
	ret = CRC_calcResult16(CRC_BASE, (uint16_t*)(pBufIn), data_len) ^ rslt_xor_out;
	dlog_crc_ts[2] = CPU_getCycleCnt()-dlog_crc_ts[2];
	dlog_crc_ts[3] = CPU_getInstrCnt()-dlog_crc_ts[3];

	return ret;
}

/*
 * \brief    main function
 * \details  Perform CRC16 by software lookup table. Result is in crc_result[0].
 *           Perform CRC16 by CRC hardware. Result is in crc_result[1].
 *           Measure the CPU execution time for above methods.
 */
int main(void)
{
    uint32_t oldTicks;
    uint32_t tickCnt;

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello CRC benchmark!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);
    log_info("Code @ 0x%08X, Data @ 0x%08X.\r\n", (uint32_t)main, (uint32_t)&oldTicks);

    __enable_irq();

    /* CRC input data initialization */
    for(uint32_t i=0; i<sizeof(crc_input); i++)
    	crc_input[i] = (uint8_t)i;

    /*
     * CRC16 by software lookup table
     * dlog_crc_ts[0], CPU cycles for 1KB CRC16 calculation;
     * dlog_crc_ts[1], CPU instructions for 1KB CRC16 calculation;
     * crc_result[0], CRC result for CRC-16/MODBUS (poly=0x8005, init=0xFFFF, RevIn, RevOut)
     */
    dlog_crc_ts[0] = CPU_getCycleCnt();
    dlog_crc_ts[1] = CPU_getInstrCnt();
    crc_result[0] = calculate_crc16(crc_input, sizeof(crc_input));
    dlog_crc_ts[0] = CPU_getCycleCnt()-dlog_crc_ts[0];
    dlog_crc_ts[1] = CPU_getInstrCnt()-dlog_crc_ts[1];

    /*
     * CRC16 by CRC Hardware
     * dlog_crc_ts[2], CPU cycles for 1KB CRC16 calculation;
     * dlog_crc_ts[3], CPU instructions for 1KB CRC16 calculation;
     * crc_result[1], CRC result for CRC-16/MODBUS (poly=0x8005, init=0xFFFF, RevIn, RevOut)
     */
    crc_result[1] = CRC_calcCrc16Test(crc_input, sizeof(crc_input));


    while (1) {
    }

    return 0;
}

