/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

/**
*   @file    hrcap_ex01_calibration.c
*   @brief
*   @details
*
*/

#ifdef __cplusplus
extern "C"{
#endif

/* ========================================================================== */
/*                             Include Files                                  */
/* ========================================================================== */

#include "hrcap_ex01_calibration.h"
#include "printf.h"
#include "hrcap_ex01_board.h"

/* ========================================================================== */
/*                           Macros & Typedefs                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Structures and Enums                               */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Constants                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Local Variables                                 */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Constants                                */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                            Global Variables                                */
/* ========================================================================== */

volatile uint32_t cap1Count;
volatile uint32_t cap2Count;
volatile uint32_t cap3Count;
volatile uint32_t cap4Count;

uint32_t absCountOn1, absCountOn2;
uint32_t absCountOff1, absCountOff2;
uint32_t absCountPeriod1, absCountPeriod2;

float32_t onTime1 = 0, onTime2 = 0;
float32_t offTime1 = 0, offTime2 = 0;
float32_t period1 = 0, period2 = 0;

uint32_t ecapIntCount;
uint16_t hrcapIntCount = 0;
uint16_t ecapIntCalCount = 0;
uint16_t calStatus = 0;

HRCAPCAL_CalResultObj hrcapCalResult;

uint64_t totalCount = 0;
float32_t inputFreqMHz = 0;

/* ========================================================================== */
/*                          Local Function Prototypes                         */
/* ========================================================================== */

void myEPWM0_init(void);
void EPWM_init(void);

/* ========================================================================== */
/*                          Local Function Definitions                        */
/* ========================================================================== */

/* None */

/* ========================================================================== */
/*                         Global Functions Definitions                       */
/* ========================================================================== */

void main(void){

	Device_init();

	Device_initGPIO();

	Interrupt_initModule();

	Interrupt_initVectorTable();

	// Board initialization
    //
	Board_init();

	EPWM_init();

	ecapIntCount = 0;
	cap2Count = 0U;
	cap3Count = 0U;
	cap4Count = 0U;

    // Loop forever. Suspend or place breakpoints to observe the buffers.
    while(1)
    {
        //
        // Wait for a value to be captured by the HRCAP
        //
        if (period1 != 0)
        {
            //
            // Convert from nS to MHz
            //
            inputFreqMHz = 1 / (period1 / 1000);
        }
    }
}

// EPWM Configurations
void EPWM_init()
{
	myEPWM0_init();
}

// EPWM Configurations for EPWM1~6
void myEPWM0_init()
{
	// Disable sync(Freeze clock to PWM as well)
	//
	SysCtl_disablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);

	// Configure ePWM
	// epwm1
	EPWM_disablePhaseShiftLoad(myEPWM1_BASE);
	EPWM_setTimeBaseCounterMode(myEPWM1_BASE, EPWM_COUNTER_MODE_UP_DOWN);
	EPWM_setTimeBasePeriod(myEPWM1_BASE, PWM_PRD);
	EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_A,PWM_CMP_MAX); // 默认在ctr=0进行影子加载
	EPWM_setCounterCompareValue(myEPWM1_BASE, EPWM_COUNTER_COMPARE_B,PWM_CMP_MIN); // 默认在ctr=0进行影子加载
	EPWM_setPhaseShift(myEPWM1_BASE, 0U);
	EPWM_setActionQualifierAction(myEPWM1_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_HIGH,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_UP_CMPA);
	EPWM_setActionQualifierAction(myEPWM1_BASE,
								  EPWM_AQ_OUTPUT_A,
								  EPWM_AQ_OUTPUT_LOW,
								  EPWM_AQ_OUTPUT_ON_TIMEBASE_DOWN_CMPB);
	EPWM_setClockPrescaler(myEPWM1_BASE,
						   EPWM_CLOCK_DIVIDER_1,
						   EPWM_HSCLOCK_DIVIDER_1);

	// Enable sync and clock to PWM
	//
	SysCtl_enablePeripheral(SYSCTL_PERIPH_CLK_TBCLKSYNC);
}

void Hrcap_ex01_IrqHandler1(void)
{
    ecapIntCalCount++;
    if(ecapIntCount++ > 10)
    {
        ecapIntCount = 0;

        //
        // Get the raw time stamps
        //
        cap1Count = ECAP_getEventTimeStamp(myECAP0_BASE, ECAP_EVENT_1);
        cap2Count = ECAP_getEventTimeStamp(myECAP0_BASE, ECAP_EVENT_2);
        cap3Count = ECAP_getEventTimeStamp(myECAP0_BASE, ECAP_EVENT_3);
        cap4Count = ECAP_getEventTimeStamp(myECAP0_BASE, ECAP_EVENT_4);

        absCountOn1 = cap2Count - cap1Count;
        absCountOff1 = cap3Count - cap2Count;
        absCountPeriod1 = cap3Count - cap1Count;

        absCountOn2 = cap4Count - cap3Count;
        absCountOff2 = cap3Count - cap2Count;
        absCountPeriod2 = cap4Count - cap2Count;

        //
        // Convert counts to nanoseconds using the scale factor
        //
        onTime1 = HRCAP_convertEventTimeStampNanoseconds(absCountOn1,
                                                   hrcapCalResult.scaleFactor);
        offTime1 = HRCAP_convertEventTimeStampNanoseconds(absCountOff1,
                                                   hrcapCalResult.scaleFactor);
        period1 = HRCAP_convertEventTimeStampNanoseconds(absCountPeriod1,
                                                   hrcapCalResult.scaleFactor);

        onTime2 = HRCAP_convertEventTimeStampNanoseconds(absCountOn2,
                                                   hrcapCalResult.scaleFactor);
        offTime2 = HRCAP_convertEventTimeStampNanoseconds(absCountOff2,
                                                   hrcapCalResult.scaleFactor);
        period2 = HRCAP_convertEventTimeStampNanoseconds(absCountPeriod2,
                                                   hrcapCalResult.scaleFactor);

        totalCount++;
    }

    ECAP_clearInterrupt(myECAP0_BASE, ECAP_ISR_SOURCE_CAPTURE_EVENT_4);
    ECAP_clearGlobalInterrupt(myECAP0_BASE);
    ECAP_reArm(myECAP0_BASE);
    ECAP_resetCounters(myECAP0_BASE);

}

void Hrcap_ex01_IrqHandler2(void)
{
	hrcapIntCount++;

	    //
	    // Get calibration interrupt sources
	    //
	    calStatus = HRCAP_getCalibrationFlags(myECAP0_HR_BASE);

	    //
	    // Get calibration clock counts
	    //
	    hrcapCalResult.hrclkCount = HRCAP_getCalibrationClockPeriod(myECAP0_HR_BASE,
	                                            HRCAP_CALIBRATION_CLOCK_HRCLK);
	    hrcapCalResult.sysclkcount = HRCAP_getCalibrationClockPeriod(myECAP0_HR_BASE,
	                                            HRCAP_CALIBRATION_CLOCK_SYSCLK);

	    //
	    // The following options are possible
	    //   - HRCALCAL_STATUS_DONE_ISR
	    //   - HRCALCAL_STATUS_DONE_PERIOD_OVERFLOW_ISR
	    //   - Software forced generated interrupt
	    //
	    if(HRCALCAL_STATUS_DONE_ISR == calStatus)
	    {
	        //
	        // Calculate scale factor
	        //
	        hrcapCalResult.scaleFactor = HRCAP_getScaleFactor(myECAP0_HR_BASE);
	    }
	    else if (HRCALCAL_STATUS_DONE_PERIOD_OVERFLOW_ISR == calStatus)
	    {
	        //
	        // Calibration done with an overflow. Determine which counter has
	        // overflowed
	        //
	        if(hrcapCalResult.hrclkCount > hrcapCalResult.sysclkcount)
	        {
	            //
	            // HRCLK has overflowed
	            //
	            hrcapCalResult.scaleFactor = hrcapCalResult.sysclkcount *
	                                         HRCAPCAL_INV_OVERFLOW;
	        }
	        else if(hrcapCalResult.hrclkCount < hrcapCalResult.sysclkcount)
	        {
	            //
	            // SYSCLK has overflowed
	            //
	            hrcapCalResult.scaleFactor = HRCAPCAL_OVERFLOW /
	                                         hrcapCalResult.hrclkCount;
	        }
	        else
	        {
	            //
	            // Both SYSCLK and HRCLK have overflowed
	            //
	            hrcapCalResult.scaleFactor = 1.0f;
	        }
	    }
	    else
	    {
	        //
	        // Software generated interrupt
	        //
	    }

	    //
	    // Clear the interrupts
	    //
	    HRCAP_clearCalibrationFlags(myECAP0_HR_BASE, calStatus);

}


#ifdef __cplusplus
}
#endif
