/*
 *   Copyright (c) Gejian Semiconductors 2023
 *   All rights reserved.
 *
 *  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 *  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 *  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
 *  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
 *  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 *  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 *  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 *  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 *  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 *  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */
#include <string.h>
#include "stdlib.h"
#include "device.h"
#include "driverlib.h"
#include "printf.h"
#include "board_cfg.h"
#include "log.h"
#include "fft.h"
#include "gs32_math.h"


#define FFT_TW_REAL_BASE_ADDR		(0x4008C400)
#define FFT_TW_IMAG_BASE_ADDR    	(0x4008CC00)
#define FFT_WIN_COEF_BASE_ADDR   	(0x4008D400)

#define FFT_DATA_BASE_ADDR       	(0x40088400)
#define FFT_RESULT_BASE_ADDR     	(0x40088400)

#define FFT_POINTS               	2048
#define signal_frequency 			20				// źƵ20hz
#define sample_frequency 			4096			// 4096hzũҪڵźƵʵ2رԭʼźеϢ

extern const uint32_t tw_imag[512];
extern const uint32_t tw_real[512];
extern const uint32_t win_coef[1024];


float fft_result[FFT_POINTS*2];

volatile float sample_data[sample_frequency];

void CreateSinArray(float Amplitude,float Displacement, uint16_t samples)		// 
{

    float increment = (2*3.141592653589793) / samples;                 // x
    float outputData = 0;                                          // ż

    for (uint16_t t = 0; t < samples; t++)
    {
        // Ҳʽy=Asin(x+)+b
        outputData = Amplitude * __sin(signal_frequency * increment * t)+ Displacement + __sin(0.5*signal_frequency * increment * t);
        sample_data[t] = outputData;
    }
}

void fft_init(int trigByDMA)
{
    HWREG(SYSCTL_BASE + 0x404) = 1;			// fft_ram_mode

    FFT_SET_TYPE(FFT_BASE, FFT_TYPE_REAL);
    FFT_SET_LEN(FFT_BASE, REAL_FFT_LEN_2048);

}

void fft_load_factor_by_cpu(void)
{
	uint32_t i;

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_REAL_BASE_ADDR)[i] = tw_real[i];
	}

	for (i=0; i<512; i+=1) {
		((volatile uint32_t *)FFT_TW_IMAG_BASE_ADDR)[i] = tw_imag[i];
	}

	for (i=0; i<1024; i+=1) {
		((volatile uint32_t *)FFT_WIN_COEF_BASE_ADDR)[i] = win_coef[i];
	}
}

void fft_load_data_by_cpu(void)
{
    for (uint32_t i=0; i<sizeof(sample_data)/sizeof(float); i+=1) {
        ((volatile float *)FFT_DATA_BASE_ADDR)[i] = (float)(sample_data[i]);	// ݷdata_ram˴Ϊ
    }
}

void fft_dump_result_by_cpu(void)
{
    for (uint32_t i=0; i<sizeof(fft_result)/sizeof(uint32_t); i+=1) {
        fft_result[i] = abs(((volatile float *)FFT_RESULT_BASE_ADDR)[i]);
    }
}

void fft_show_result(float *result, uint32_t points)
{
    log_debug("------- dump fft result start ----------\r\n");

    for (uint32_t i=0; i<points; i+=2) {
        log_debug("0x%08X:%08X %08X\n", FFT_RESULT_BASE_ADDR + i*4, result[i], result[i+1]);
    }

    log_debug("------- dump fft result end ----------\r\n");
}


void fft_dump_reg(void)
{
    log_debug("------- dump fft register start ----------\n");
    log_debug("           CFG_0x%02X: 0x%08X\n", FFT_O_FFT_CFG, HWREG(FFT_BASE + FFT_O_FFT_CFG));
    log_debug("         START_0x%02X: 0x%08X\n", FFT_O_FFT_START, HWREG(FFT_BASE + FFT_O_FFT_START));
    log_debug("DSPDMATRIG_CFG_0x%02X: 0x%08X\n", FFT_O_FFT_DSPDMATRIG_CFG, HWREG(FFT_BASE + FFT_O_FFT_DSPDMATRIG_CFG));
    log_debug("  AUTOSTART_EN_0x%02X: 0x%08X\n", FFT_O_FFT_AUTOSTART_EN, HWREG(FFT_BASE + FFT_O_FFT_AUTOSTART_EN));
    log_debug("        STATUS_0x%02X: 0x%08X\n", FFT_O_FFT_STATUS, HWREG(FFT_BASE + FFT_O_FFT_STATUS));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_FFT_INT_RAW, HWREG(FFT_BASE + FFT_O_FFT_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_FFT_INT_MASK, HWREG(FFT_BASE + FFT_O_FFT_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FLG, HWREG(FFT_BASE + FFT_O_FFT_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_FFT_INT_FRC, HWREG(FFT_BASE + FFT_O_FFT_INT_FRC));
    log_debug("       INT_RAW_0x%02X: 0x%08X\n", FFT_O_WIN_INT_RAW, HWREG(FFT_BASE + FFT_O_WIN_INT_RAW));
    log_debug("      INT_MASK_0x%02X: 0x%08X\n", FFT_O_WIN_INT_MASK, HWREG(FFT_BASE + FFT_O_WIN_INT_MASK));
    log_debug("       INT_FLG_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FLG, HWREG(FFT_BASE + FFT_O_WIN_INT_FLG));
    log_debug("       INT_FRC_0x%02X: 0x%08X\n", FFT_O_WIN_INT_FRC, HWREG(FFT_BASE + FFT_O_WIN_INT_FRC));
}


int main(void)
{

    Device_init();

    UartPrint_init(LOG_SCI_BASE, 115200);

    log_set_level(LOG_DEBUG);

    log_info("Hello DSP FFT!\r\n");
    log_info("Core running @ %d MHz.\r\n", DEVICE_SYSCLK_FREQ / 1000000);


    log_debug("------- %s start ----------\r\n", __FUNCTION__);

    CreateSinArray(2, 0.5, sample_frequency);		// f(x)=2sin(20x)+sin(10x)+0.5

    fft_dump_reg();		// ӡĴϢ

    fft_init(0);

    fft_load_factor_by_cpu();						// תдram
    fft_load_data_by_cpu();							// a_li_lrdata_ram˴Ϊ

    memset(fft_result, 0x00, sizeof(fft_result));	// Ϊfft_resultٿռ

    log_debug("start windowing ...\r\n");
    FFT_WIN_SOFT_START(FFT_BASE);					// Ӵ
    fft_dump_reg();									// һδӡĴϢ
    while(FFT_IS_WIN_BUSY(FFT_BASE));				// жǷڼӴ
    log_debug("windowing finished.\r\n");

    log_debug("start calculating ...\r\n");
    FFT_SOFT_START(FFT_BASE);						//fft
    fft_dump_reg();									//ӡĴϢ
    while(FFT_IS_BUSY(FFT_BASE));
    log_debug("fft calculate finished.\r\n");

    FFT_CLEAR_WIN_INTERRUPT(FFT_BASE);					// ж
    FFT_CLEAR_INTERRUPT_FLAG(FFT_BASE, FFT_INT_CLR_OVF_INT_CLR | FFT_INT_CLR_DONE_INT_CLR);

    fft_dump_result_by_cpu();						// ת͵飬ͨdsp live view鿴fft_result

    /*
     * fft_resultУ±0ΪֱֱźŵƫDisplacementýֱķֵ,displacement = fft_result[0]/(FFT_POINTS*2)/256
     * ԭʼźŵƵΪֵߴsignal_frequency=(n-1)*sample_frequency/(FFT_POINTS*2)nΪ±꣬FFT_POINTS֮Գ2ΪfftΪʵ㣬Ϊó2
     * ƵʵķֵAmplitude=fft_result[n]/(FFT_POINTS*2)/128
     */

    fft_dump_reg();									// ӡĴ

    fft_show_result(fft_result, FFT_POINTS*2);

    log_debug("------- %s end ----------\r\n", __FUNCTION__);

    for(;;);

    return 0;
}

